﻿param(
  [string]$CsvPath,
  [int]$MaxColWidth = 60,
  [int]$MinColWidth = 6,
  [switch]$DeleteCsv
)

# === Excelファイルが開いていないか確認 ===
function Test-XlsxFileOpen([string]$filePath) {
    if (-not (Test-Path $filePath)) { return $false }
    try {
        $stream = [System.IO.File]::Open($filePath, 'Open', 'ReadWrite', 'None')
        $stream.Close()
        return $false
    } catch {
        return $true
    }
}

# === 出力ファイル設定 ===
$xlsxPath = [System.IO.Path]::ChangeExtension($CsvPath, "xlsx")
if (Test-XlsxFileOpen $xlsxPath) {
    Write-Host "[ERROR] Excelファイルが開かれています。閉じてから再実行してください。"
    exit 1
}

# === CSV読み込み ===
try {
    $data = Import-Csv -Path $CsvPath -Delimiter ',' -Encoding UTF8
} catch {
    Write-Host "[ERROR] CSV読み込み失敗: $($_.Exception.Message)"
    exit 1
}
if (-not $data -or $data.Count -eq 0) {
    Write-Host "[WARN] CSVにデータがありません。"
    exit 0
}

# === 不要列削除 ===
$dropCols = @('患者名','報告者名','家族説明','患者説明')
foreach ($rec in $data) {
    foreach ($col in $dropCols) {
        if ($rec.PSObject.Properties.Name -contains $col) {
            $rec.PSObject.Properties.Remove($col)
        }
    }
}

# === others_* を統合して「その他」列にまとめる ===
$otherCols = @()
foreach ($p in $data[0].PSObject.Properties.Name) {
    if ($p -match '^others_') { $otherCols += $p }
}

if ($otherCols.Count -gt 0) {
    foreach ($rec in $data) {
        $merged = ""
        foreach ($c in $otherCols) {
            $val = $rec.$c
            if (-not [string]::IsNullOrWhiteSpace($val)) {
                $merged += "$val`n"
            }
        }
        $merged = $merged.Trim()
        Add-Member -InputObject $rec -NotePropertyName "その他" -NotePropertyValue $merged -Force
    }

    foreach ($rec in $data) {
        foreach ($c in $otherCols) {
            if ($rec.PSObject.Properties.Name -contains $c) {
                $rec.PSObject.Properties.Remove($c)
            }
        }
    }
}

# === Excel起動（安全モード） ===
try {
    # 既存インスタンスを優先使用
    $excel = [Runtime.InteropServices.Marshal]::GetActiveObject("Excel.Application")
    Write-Host "[INFO] 既存のExcelインスタンスを使用します。"
} catch {
    # 新規起動
    $excel = New-Object -ComObject Excel.Application
    Write-Host "[INFO] 新しいExcelインスタンスを起動しました。"
}

$excel.Visible = $false
$excel.DisplayAlerts = $false

# Workbooks.Add() が固まることがあるため、再試行ロジック付き
$retry = 0
while (-not $wb -and $retry -lt 3) {
    try {
        $wb = $excel.Workbooks.Add([Type]::Missing)
    } catch {
        Write-Host "[WARN] Excelブック作成失敗 ($retry)。再試行中..."
        Start-Sleep -Seconds 1
        $retry++
    }
}

if (-not $wb) {
    Write-Host "[ERROR] Excelブックを作成できませんでした。Excelを閉じて再実行してください。"
    $excel.Quit()
    exit 1
}

Write-Host "[INFO] Excelブック作成成功。"

# === Dataシート作成 ===
$shData = $wb.Sheets.Add()
$shData.Name = "Data"
$shData.Tab.ColorIndex = 34  # 青タブ

# === データ書き込み（ジャグ配列方式・完全安定版） ===
$headers = $data[0].PSObject.Properties.Name
$colCount = $headers.Count
$rowCount = $data.Count

# ヘッダ書き込み
for ($c = 0; $c -lt $colCount; $c++) {
    $shData.Cells.Item(1, $c + 1).Value2 = $headers[$c]
}

# 本文データ（ジャグ配列で作成）
$values = @()
foreach ($rowObj in $data) {
    $rowArr = @()
    foreach ($h in $headers) {
        $v = $rowObj.$h
        if ($null -eq $v) { $v = "" }
        $rowArr += [string]$v
    }
    $values += ,$rowArr
}

# Excelに一括書き込み（ジャグ配列方式）
for ($r = 0; $r -lt $rowCount; $r++) {
    for ($c = 0; $c -lt $colCount; $c++) {
        $shData.Cells.Item($r + 2, $c + 1).Value2 = $values[$r][$c]
    }
}


# === 上3行を確保してボタンスペースを作る ===
$shData.Rows("1:3").Insert() | Out-Null

# === 影響度レベルヘッダ調整 ===
$lastCol = $shData.Cells.Item(4, $shData.Columns.Count).End(-4159).Column
for($c=1; $c -le $lastCol; $c++){
    $h = [string]$shData.Cells.Item(4,$c).Text
    if($h -eq '影響度レベル'){
        $shData.Cells.Item(4,$c).Value2 = "影響度`nレベル"
        break
    }
}

# === 列幅・整形設定 ===
$longTextCols = @('発生内容','要因','その他')
$fixedWidthMap = @{
  '影響度レベル' = 2.5;
  '性別' = 3;
  '年齢' = 4;
  '発生日' = 12;
  '発生場所' = 20;   # 👈 ナースステーション対応で拡大
  '職種' = 10;
  '関係性' = 10;
  '経験年数' = 10;
}

$used = $shData.UsedRange
$used.WrapText = $true

for ($c=1; $c -le $lastCol; $c++) {
    $colObj = $shData.Columns.Item($c)
    $head = [string]$shData.Cells.Item(4,$c).Text

    if ($longTextCols -contains $head) {
        $colObj.WrapText = $true
        $colObj.HorizontalAlignment = -4131
        switch ($head) {
            '発生内容' { $colObj.ColumnWidth = 45 }
            '要因'     { $colObj.ColumnWidth = 40 }
            'その他'   { $colObj.ColumnWidth = 40 }
        }
    } elseif ($head -eq '発生日') {
        $colObj.WrapText = $false
        $colObj.HorizontalAlignment = -4108
        $colObj.ColumnWidth = 12
    } else {
        $colObj.HorizontalAlignment = -4108
        if ($fixedWidthMap.ContainsKey($head)) {
            $colObj.ColumnWidth = [double]$fixedWidthMap[$head]
        } else {
            $null = $colObj.AutoFit()
        }
    }
}

$shData.Range($shData.Cells.Item(4,1), $shData.Cells.Item(4,$lastCol)).Font.Bold = $true
$shData.Rows.AutoFit() | Out-Null

# === 統計シート作成 ===
$shSummary = $wb.Sheets.Add([Type]::Missing, $shData)
$shSummary.Name = "インシデント・アクシデント統計"

# === 集計用関数（タイトル行1行化）===
function New-CountTable {
    param($Sheet,$Title,$FieldName,$StartRow,$StartCol,$TopN)
    $Sheet.Cells.Item($StartRow,$StartCol).Value2="$Title"
    $Sheet.Cells.Item($StartRow,$StartCol).Font.Bold=$true
    $Sheet.Cells.Item($StartRow,$StartCol+1).Value2="件数"
    if(-not $headerToCol.ContainsKey($FieldName)){return}
    $col=$headerToCol[$FieldName]
    $counts=@{}
    for($r=5;$r -le $lastRowData;$r++){
        $txt=[string]$shData.Cells.Item($r,$col).Text
        if([string]::IsNullOrWhiteSpace($txt)){continue}
        foreach($item in ($txt -split "`r?`n")){
            $s=$item.Trim()
            if($s){if(-not $counts.ContainsKey($s)){$counts[$s]=0};$counts[$s]++}
        }
    }
    $data=$counts.GetEnumerator()|Sort-Object Value -Descending
    if($TopN -gt 0){$data=$data|Select-Object -First $TopN}
    $rOut=$StartRow+1
    foreach($kv in $data){
        $Sheet.Cells.Item($rOut,$StartCol).Value2=$kv.Key
        $Sheet.Cells.Item($rOut,$StartCol+1).Value2=[double]$kv.Value
        $rOut++
    }
    return @{StartRow=$StartRow;EndRow=$rOut-1;StartCol=$StartCol}
}

# === データ列対応 ===
$lastRowData = $shData.Cells.Item($shData.Rows.Count,1).End(-4162).Row
$headerToCol = @{}
for($c=1; $c -le $lastCol; $c++){
    $name = [string]$shData.Cells.Item(4,$c).Text
    if($name){ $headerToCol[$name]=$c }
}

$row0=2
$tblWhere=New-CountTable $shSummary "発生場所" "発生場所" $row0 1 0
$tblCont =New-CountTable $shSummary "発生内容TOP3" "発生内容" $row0 5 3
$tblCause=New-CountTable $shSummary "要因" "要因" $row0 9 0

# === Summaryシート列幅調整 ===
if ($shSummary -ne $null) {
    Write-Host "[INFO] Summaryシートの列幅を最適化中..."

    # まずすべての列を自動調整（折り返し）
    $shSummary.Cells.EntireColumn.AutoFit()
    $shSummary.Cells.WrapText = $true

    # カラム幅個別指定（見えやすさ優先）
    # A列: 発生場所、B列: 件数
    $shSummary.Columns("A").ColumnWidth = 22
    $shSummary.Columns("B").ColumnWidth = 9

    # C〜D: 発生内容TOP3
    $shSummary.Columns("C").ColumnWidth = 25
    $shSummary.Columns("D").ColumnWidth = 9

    # E〜F: 要因＋件数
    $shSummary.Columns("E").ColumnWidth = 25
    $shSummary.Columns("F").ColumnWidth = 9

    # 中央寄せ（数字列）
    foreach ($col in "B","D","F") {
        $shSummary.Columns($col).HorizontalAlignment = -4108  # Center
    }

    Write-Host "[INFO] Summaryシート列幅調整完了。"
}


# === グラフ配置 ===
$W=420;$H=300;$GAP=40
$chartTop=($tblWhere.EndRow,$tblCont.EndRow,$tblCause.EndRow|Measure-Object -Maximum).Maximum+2
$anchor=$shSummary.Range("A$chartTop")

function Add-Pie {param($Sheet,$range,$left,$top,$title)
    $co=$Sheet.ChartObjects().Add($left,$top,$W,$H)
    $ch=$co.Chart;$ch.ChartType=5;$ch.SetSourceData($range)
    $ch.HasTitle=$true;$ch.ChartTitle.Text=$title
}

$range1 = $shSummary.Range(
    $shSummary.Cells.Item($tblWhere.StartRow+1, $tblWhere.StartCol),
    $shSummary.Cells.Item($tblWhere.EndRow, $tblWhere.StartCol+1)
)
Add-Pie $shSummary $range1 $anchor.Left $anchor.Top "発生場所（件数）"

$range2 = $shSummary.Range(
    $shSummary.Cells.Item($tblCont.StartRow+1, $tblCont.StartCol),
    $shSummary.Cells.Item($tblCont.EndRow, $tblCont.StartCol+1)
)
Add-Pie $shSummary $range2 ($anchor.Left+$W+$GAP) $anchor.Top "発生内容 TOP3"

$range3 = $shSummary.Range(
    $shSummary.Cells.Item($tblCause.StartRow+1, $tblCause.StartCol),
    $shSummary.Cells.Item($tblCause.EndRow, $tblCause.StartCol+1)
)
$co3 = $shSummary.ChartObjects().Add($anchor.Left,$anchor.Top+$H+$GAP,($W*2+$GAP),320)
$ch3 = $co3.Chart;$ch3.ChartType=51;$ch3.SetSourceData($range3)
$ch3.HasTitle=$true;$ch3.ChartTitle.Text="要因（件数）"

# === タイトル・ボタン ===
$shSummary.Rows("1:1").Insert()|Out-Null
$shSummary.Cells.Item(1,1).Value2="★ インシデント・アクシデント統計（自動集計レポート） ★"
$shSummary.Range("A1:L1").Merge()
$shSummary.Range("A1").HorizontalAlignment=-4108
$shSummary.Range("A1").Font.Size=16
$shSummary.Range("A1").Font.Bold=$true
$shSummary.Range("A1").Interior.ColorIndex=36

function Add-NavigationButton {
    param($Sheet,$Text,$LinkTarget,$Left,$Top,$Width,$Height,$ColorIndex)
    if(-not $Width){$Width=110};if(-not $Height){$Height=28};if(-not $ColorIndex){$ColorIndex=34}
    $shape=$Sheet.Shapes.AddShape(5,$Left,$Top,$Width,$Height)
    $shape.TextFrame.Characters().Text=$Text
    $shape.Fill.ForeColor.SchemeColor=$ColorIndex
    $shape.Line.Visible=$true
    $shape.Line.ForeColor.SchemeColor=23
    $shape.TextFrame.HorizontalAlignment=-4108
    $shape.TextFrame.VerticalAlignment=-4108
    $shape.TextFrame.Characters().Font.ColorIndex=2
    $shape.TextFrame.Characters().Font.Bold=$true
    $shape.OnAction=$null
    $null=$Sheet.Hyperlinks.Add($shape,"",$LinkTarget,"",$Text)
    return $shape
}

# ✅ ボタン配置修正（右上）
$null=Add-NavigationButton -Sheet $shSummary -Text "▶ Dataへ" `
    -LinkTarget "Data!A1" -Left 760 -Top 5 -ColorIndex 33
$null=Add-NavigationButton -Sheet $shData -Text "▶ 統計へ" `
    -LinkTarget "インシデント・アクシデント統計!A1" -Left 350 -Top 5 -ColorIndex 17

# === 保存 ===
try{$wb.Windows(1).DisplayWorkbookTabs=$true}catch{}
$shSummary.Move($wb.Sheets.Item(1))
$shSummary.Activate()
$wb.SaveAs($xlsxPath,51)
$wb.Close($false)
$excel.Quit()

# === COM解放 ===
[void][Runtime.InteropServices.Marshal]::ReleaseComObject($shData)
[void][Runtime.InteropServices.Marshal]::ReleaseComObject($shSummary)
[void][Runtime.InteropServices.Marshal]::ReleaseComObject($wb)
[void][Runtime.InteropServices.Marshal]::ReleaseComObject($excel)
[GC]::Collect(); [GC]::WaitForPendingFinalizers()

Write-Host "OK: $xlsxPath を作成（Hope4.1：右上ボタン・統計1行化・完全安定版）"

if($DeleteCsv){
  try{
    Remove-Item -LiteralPath $CsvPath -Force -ErrorAction Stop
    Write-Host "削除しました: $CsvPath"
  }catch{
    Write-Host "WARN: CSV削除できず: $($_.Exception.Message)"
  }
}
