﻿param(
  [string]$Directory=".",
  [string]$Out="",
  [switch]$Utf8,
  [string]$Include="",
  [string]$Exclude=""
)

# UTF-8(BOM)でコンソール/外部プロセスを統一
[Console]::OutputEncoding = [System.Text.UTF8Encoding]::new()
$OutputEncoding            = [System.Text.UTF8Encoding]::new()


# ===== 列名マッピング（日本語統一） =====
$HeaderAliases = @{
 'risk1'='影響度レベル';
  'pt_name'='患者名'; 'patient_name'='患者名'; 
  'pt_id'='患者ID';   'patient_id'='患者ID';  'pt_age'='年齢';
 'sex'='性別';
  'your_name'='報告者名'; 'reporter_name'='報告者名'; 
"dementia"="認知症";
"change"="状態変化";
"exp"="経験年数";
"info_family"="家族説明";
"info_pt"="患者説明";
"relation"="関係性";

# 場所系
 'where'='発生場所';

# 職種
'occupation'='職種'; 

'G0'='移動手段';
  
  # 日付系
  'occur_date'='発生日'; 'when_date'='発生日'; 'when_ymd'='発生日'; 
  'when_year'='発生年'; 'when_month'='発生月'; 'when_day'='発生日_日';
}

# 先頭固定列（_id/_sourceは出さない）
$OrderFirst = @('影響度レベル','患者名','性別','年齢','認知症' ,'発生日','発生場所','発生内容','移動手段','要因','患者ID','報告者名','職種','関係性','経験年数')
$OrderLast  = @()

# help/説明などの除外（名前で判定）
$NameExcludeRegex = @(
  [regex]'(?i)(^|_)help($|_)'
)

# 常時除外（日時・入力系など）
$AlwaysDropNameRegex = @(
  [regex]'(?i)^(CalendarHead|CalendarMonth|CalendarYear)$',
 [regex]'(?i)^FR_00000_Calendar$', 
  [regex]'(?i)^input_',
  [regex]'(?i)^repo_',
  [regex]'(?i)^when_(year|month|day|day_of_week|hour|minutes)$',
  [regex]'(?i)^hyouka_box$'
)

# チェック／ラジオ系だけ許可（G*, CB* など）＋必要ホワイトリスト
$CheckFieldNameRegex = @(
  [regex]'^(?i)G\d+$',
  [regex]'^(?i)CB\d+(_\d+)?$',
  [regex]'(?i)^(sex|relation|dementia|exp|change|info_family|info_pt|others_\d+)$'
[regex]'(?i)^(occupation|職種)$' 
)

# === PDF () リテラル → 文字列（\ddd オクタル対応） ===
function Decode-PdfLiteral([string]$inner){
  $bytes = New-Object System.Collections.Generic.List[byte]
  $i=0; $len=$inner.Length
  while($i -lt $len){
    $ch = $inner[$i]
    if($ch -eq '\'){
      if($i+1 -ge $len){break}
      $i++; $n=$inner[$i]
      switch($n){
        'n' {$bytes.Add(0x0A); $i++; continue}
        'r' {$bytes.Add(0x0D); $i++; continue}
        't' {$bytes.Add(0x09); $i++; continue}
        'b' {$bytes.Add(0x08); $i++; continue}
        'f' {$bytes.Add(0x0C); $i++; continue}
        '(' {$bytes.Add([byte][char]'('); $i++; continue}
        ')' {$bytes.Add([byte][char]')'); $i++; continue}
        '\' {$bytes.Add([byte][char]'\'); $i++; continue}
        default{
          if($n -eq "`r" -or $n -eq "`n"){ if($n -eq "`r" -and $i+1 -lt $len -and $inner[$i+1]-eq "`n"){ $i++ } $i++; continue }
          if($n -match '[0-7]'){
            $oct=""+$n
            if($i+1 -lt $len -and $inner[$i+1] -match '[0-7]'){ $i++; $oct+=$inner[$i] }
            if($i+1 -lt $len -and $inner[$i+1] -match '[0-7]'){ $i++; $oct+=$inner[$i] }
            $bytes.Add([byte][Convert]::ToInt32($oct,8)); $i++; continue
          }
          $bytes.Add([byte][char]$n); $i++; continue
        }
      }
    } else {
      $bytes.Add([byte][char]$ch); $i++
    }
  }
  $buf=$bytes.ToArray()
  if($buf.Length -ge 2 -and $buf[0]-eq 0xFE -and $buf[1]-eq 0xFF){ return [Text.Encoding]::BigEndianUnicode.GetString($buf,2,$buf.Length-2) }
  elseif($buf.Length -ge 2 -and $buf[0]-eq 0xFF -and $buf[1]-eq 0xFE){ return [Text.Encoding]::Unicode.GetString($buf,2,$buf.Length-2) }
  else { try { return [Text.Encoding]::GetEncoding(932).GetString($buf) } catch { return [Text.Encoding]::GetEncoding(28591).GetString($buf) } }
}

# === FDF（辞書境界で /T と /V を結び付け）===
function Parse-Fdf($path){
  $bytes = [IO.File]::ReadAllBytes($path)
  $txt   = [Text.Encoding]::GetEncoding(28591).GetString($bytes)

  $dictRe = [regex]'<<[\s\S]*?>>'                               # フィールド辞書
  $nameRe = [regex]'/T\s*\((.*?)\)'                              # /T (name)
  $valRe  = [regex]'/V\s*(\((?:[^\\\)]|\\.)*\)|<[0-9A-Fa-f\s]+>|/[^\s<>\[\]\(\)/%]+)'  # /V (...)

  $result=@{}
  foreach($dm in $dictRe.Matches($txt)){
    $s=$dm.Value

    $nm=$nameRe.Match($s)
    if(-not $nm.Success){ continue }
    $name=$nm.Groups[1].Value

    $vm=$valRe.Match($s)
    if(-not $vm.Success){ continue }
    $raw=$vm.Groups[1].Value

    $val=""
    if($raw.StartsWith('(')){
      $inner=$raw.Substring(1,$raw.Length-2)
      $val = Decode-PdfLiteral $inner
    } elseif($raw.StartsWith('<')){
      $h=($raw -replace '[^0-9A-Fa-f]','')
      $bytes2 = New-Object byte[] ($h.Length/2)
      for($i=0;$i -lt $bytes2.Length;$i++){ $bytes2[$i]=[Convert]::ToByte($h.Substring($i*2,2),16) }
      if($h.Length -ge 4 -and $h.Substring(0,4).ToUpper()-eq 'FEFF'){
        $val=[Text.Encoding]::BigEndianUnicode.GetString($bytes2,2,$bytes2.Length-2)
      } elseif($h.Length -ge 4 -and $h.Substring(0,4).ToUpper()-eq 'FFFE'){
        $val=[Text.Encoding]::Unicode.GetString($bytes2,2,$bytes2.Length-2)
      } else {
        try { $val=[Text.Encoding]::GetEncoding(932).GetString($bytes2) }
        catch { $val=[Text.Encoding]::ASCII.GetString($bytes2) }
      }
    } elseif($raw.StartsWith('/')){
      $val=$raw.Substring(1)   # /Yes, /On, /ExportName
    } else {
      $val=$raw
    }

    $result[$name]=$val
  }
  return $result
}

# === XFDF ===
function Parse-Xfdf($path){
  $bytes=[IO.File]::ReadAllBytes($path)
  $head=[Text.Encoding]::ASCII.GetString($bytes,0,[Math]::Min(200,$bytes.Length))
  $encName='utf-8'; if($head -match 'encoding="([^"]+)"'){ $encName=$matches[1] }
  $enc=[Text.Encoding]::GetEncoding($encName); $text=$enc.GetString($bytes)
  $xml=New-Object System.Xml.XmlDocument; $xml.LoadXml($text)
  $nodes=$xml.SelectNodes("//*[local-name()='field']"); $result=@{}
  foreach($n in $nodes){
    $name=$n.Attributes['name'].Value; $vals=@()
    foreach($c in $n.ChildNodes){ if($c.LocalName -eq 'value'){ $vals+=$c.InnerText } }
    $result[$name]=($vals -join '; ')
  }
  $result
}

# === 値の判定：チェック済みか？（出力値も返す） ===
function Get-CheckedOutValue([string]$v){
  if([string]::IsNullOrWhiteSpace($v)){ return @{ include=$false; out='' } }
  $t=$v.Trim(); $l=$t.ToLower()

  # OFF系（未選択・なし 等）は除外
  $off = @('off','no','0','false','未選択','未該当','なし','未記入','空白','いいえ','未チェック','該当なし')
  foreach($w in $off){ if($l -eq $w){ return @{ include=$false; out='' } } }

  # ON系は「有」に正規化
  $on = @('on','yes','1','true','有','あり','該当','該当あり','選択','チェック','オン','チェック済','はい')
  foreach($w in $on){ if($l -eq $w){ return @{ include=$true; out='有' } } }

  # ラジオ等：短いラベルはそのまま採用
  if(($t.Length -le 40) -and ($t -notmatch "[\r\n]")){ return @{ include=$true; out=$t } }

  return @{ include=$false; out='' }
}

# === 対象フォルダ解決 ===
$here=Split-Path -Parent $MyInvocation.MyCommand.Path
if($Directory -eq "."){ $Directory=$here }
$Directory=[IO.Path]::GetFullPath($Directory)

Write-Host "[INFO] Target: ""$Directory"""

$files=Get-ChildItem -Path $Directory -File | Where-Object { $_.Extension -match '^\.(fdf|xfdf)$' }
if(-not $files){ Write-Host "FDF/XFDF が見つかりません: $Directory"; exit }

$records=@()
[hashtable]$allCols=@{}

foreach($fi in ($files | Sort-Object Name)){
  # 種別判定
  $bytes=[IO.File]::ReadAllBytes($fi.FullName)
  $head=[Text.Encoding]::ASCII.GetString($bytes,0,[Math]::Min(512,$bytes.Length))
  $isFdf=($head -match '%FDF-|/FDF'); $isXfdf=($head.ToLower().Contains('<xfdf'))
  $recMap=if($isFdf -and -not $isXfdf){ Parse-Fdf $fi.FullName } else { Parse-Xfdf $fi.FullName }

  # 1件分
  $rowMap=@{}

  # ---- 発生日補完用 一時変数（when_*は出力しないが拾っておく）----
  $tmpYear = $null; $tmpMonth = $null; $tmpDay = $null

  foreach($origName in $recMap.Keys){
    # help/説明など除外
    $skipByName=$false; foreach($rx in $NameExcludeRegex){ if($rx.IsMatch($origName)){ $skipByName=$true; break } }
    if($skipByName){ continue }

    # 列名マッピング
    $alias = $origName
    if($HeaderAliases.ContainsKey($origName)){ $alias = $HeaderAliases[$origName] }
    $val = $recMap[$origName]

    # 発生年/月/日は補完用に保持（出力はしない）
    if($alias -eq '発生年'){ $tmpYear = $val; continue }
    if($alias -eq '発生月'){ $tmpMonth = $val; continue }
    if($alias -eq '発生日_日'){ $tmpDay = $val; continue }

    # 基礎4＋場所は無条件で採用
    if($OrderFirst -contains $alias){
      $rowMap[$alias]=$val; $allCols[$alias]=$true; continue
    }

    # 常時除外（origName/aliasの両方を見る）
    $drop=$false
    foreach($rx in $AlwaysDropNameRegex){ if($rx.IsMatch($origName) -or $rx.IsMatch($alias)){ $drop=$true; break } }
    if($drop){ continue }

    # チェック／ラジオ系のみ対象
    $okName=$false
    foreach($rx in $CheckFieldNameRegex){ if($rx.IsMatch($origName)){ $okName=$true; break } }
    if(-not $okName){ continue }

    # 値が「有」or短いラベルのときだけ採用
    $chk = Get-CheckedOutValue $val
    if($chk['include']){
      $rowMap[$alias]=$chk['out']; $allCols[$alias]=$true
    }
  }

  # 発生日が空なら、年/月/日から補完
  if((-not $rowMap.ContainsKey('発生日')) -or [string]::IsNullOrWhiteSpace($rowMap['発生日'])){
    if($tmpYear -and $tmpMonth -and $tmpDay){
      $rowMap['発生日'] = "{0}年{1}月{2}日" -f $tmpYear, $tmpMonth, $tmpDay
      $allCols['発生日'] = $true
    }
  }
  # === G系（発生内容）まとめ ===
  $gItems = @()
  foreach($k in $rowMap.Keys | Where-Object { $_ -match '^G\d+' }){
    $v = $rowMap[$k]
    if($v -and $v -ne '') { $gItems += $v }
  }
  if($gItems.Count -gt 0){
    $rowMap['発生内容'] = ($gItems -join  "`r`n")
    $allCols['発生内容'] = $true
  }

  # === CB系（発生・発見の要因）まとめ ===
  $cbItems = @()
  foreach($k in $rowMap.Keys | Where-Object { $_ -match '^CB\d' }){
    $v = $rowMap[$k]
    if($v -and $v -ne '') { $cbItems += $v }
  }
  if($cbItems.Count -gt 0){
    $rowMap['要因'] = ($cbItems -join "`r`n")
    $allCols['要因'] = $true
  }

  $records+=,$rowMap
}

if(-not $records){ Write-Host "抽出対象（チェック済み＋基礎列）がありません。フィールド名の見直しが必要かも。"; exit }

# 列順（_id/_source なし）＋念のためCalendar系を最終フィルタ
$allKeys=@($allCols.Keys)
$mid=$allKeys | Where-Object {
  ($OrderFirst -notcontains $_) -and ($OrderLast -notcontains $_) -and ($_ -ne '_id') -and ($_ -ne '_source')
} | Sort-Object
$cols=$OrderFirst + $mid + $OrderLast

# Calendar系・G系・CB系を最終的に除外
$cols = $cols | Where-Object {
    $_ -notmatch '(?i)^(CalendarHead|CalendarMonth|CalendarYear|G\d+|CB\d+(_\d+)?)$'
}


# 出力先
if($Out -ne ""){ $outFileName=$Out } else { if($Utf8){ $outFileName="fdf_merged_utf8.csv" } else { $outFileName="fdf_merged_sjis.csv" } }
$outPath=Join-Path $Directory $outFileName
$encodingObj=if($Utf8){ New-Object System.Text.UTF8Encoding($true) } else { [Text.Encoding]::GetEncoding(932) }

# === 書き出し（ロック回避：使用中なら別名で保存） ===
function New-StreamWriterSafe([string]$path, $enc){
  try{ return New-Object System.IO.StreamWriter($path, $false, $enc) }
  catch [System.IO.IOException]{ return $null }
}
$sw = New-StreamWriterSafe $outPath $encodingObj
if($sw -eq $null){
  $dir=[IO.Path]::GetDirectoryName($outPath)
  $name=[IO.Path]::GetFileNameWithoutExtension($outPath)
  $ext=[IO.Path]::GetExtension($outPath)
  $ts=Get-Date -Format 'yyyyMMdd-HHmmss'
  $outPath=[IO.Path]::Combine($dir, ($name + "_" + $ts + $ext))
  Write-Host "WARN: 出力先が使用中のため別名で保存します → $outPath"
  $sw = New-Object System.IO.StreamWriter($outPath, $false, $encodingObj)
}

try{
  # ヘッダ
  $sw.WriteLine( (($cols | ForEach-Object { '"'+($_ -replace '"','""')+'"' }) -join ',') )
  foreach($row in $records){
    $csvRow=@()
    foreach($c in $cols){
      $v = ""
      if($row.ContainsKey($c)){ $v = $row[$c] }
      if($null -eq $v){ $v = "" }
      $csvRow += ('"'+( [string]$v -replace '"','""' )+'"')
    }
    $sw.WriteLine( ($csvRow -join ',') )
  }
}
finally{
  if($sw){ $sw.Close() }
}

Write-Host ("OK: {0} 件 → {1}  (encoding = {2})" -f $records.Count, $outPath, $encodingObj.WebName)
