/*
 *  Title: DaiJa_V5 ( Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2024.2.11
 *  Copyright: 2020 - 2025
 */
package quan;

import java.util.Random;
import util.DJ;
import util.comp.CompMat;
import util.comp.CompVec;
import util.comp.Complex;

/**
 * <p> 表　題: Class: QuBasis</p>
 * <p> 説　明: 量子ビットの基底</p>
 * <p> 標準基底、正規直交基底</p>
 * <p> 基底変換</p>
 * <p> </p>
 * <p> 単位行列</p>
 * <p> 座標回転</p>
 * <p> </p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024</p>
 * <p> 作成日: 2024.02.13</p>
 */
public class QuBasis {
  
  public static final int BASIS_0 = 0; // 基底のインデックス＿０
  public static final int BASIS_1 = 1; // 基底のインデックス＿１
  public static final int BASIS_2 = 2; // 基底のインデックス＿２
  public static final int BASIS_3 = 3; // 基底のインデックス＿３  
  private CompMat basisMat; // 基底、複素正方行列
  
  private int[] qbCounter; // 基底軸毎のカウント値の配列
  private QuBit[] selectedQuBits; // 選ばれた基底軸の配列
  private int classicBit; // 古典ビット(0,1, ,n-1)
  private int[] clBitArray; // 古典ビットの配列
  
  private final long randomSeed = 314; // 疑似乱数初期値
  private final Random random = new Random(randomSeed); // 疑似乱数 
  
  /**
   * 標準基底を生成する（量子ビット数を指定する）
   * @param numOfQuBit int // 量子ビット数
   */
  public QuBasis(int numOfQuBit) {
    int order = numOfProbAmp(numOfQuBit);
    basisMat = new CompMat(order); // 単位複素数行列（正方行列）
  }
  
  /**
   * 基底を生成する（確率振幅行列を指定する）
   * @param basisMat CompMat // 基底行列（確率振幅行列）
   */
  public QuBasis(CompMat basisMat) {
    this.basisMat = basisMat;
  }
  
  /**
   * 基底（２次）を生成する（回転角度を指定する）
   * @param theta int // 角度
   */
  public QuBasis(long theta) {
    this.basisMat = new QuBasis((double)theta * Math.PI / 180.0).getBasisMat();
  }
    
  /**
   * 基底（２次）を生成する（回転角度を指定する）
   * @param theta double // 回転角θ[radian]
   */
  public QuBasis(double theta) {
    basisMat = QuBasis.rotate(theta, 0.0, 0.0); 
  }
    
  /**
   * 基底（２次）を生成する（回転角度を指定する）
   * @param theta double // 回転角θ[radian]
   * @param lamda double // 回転角λ[radian]
   * @param phi double // 回転角φ[radian]
   */
  public QuBasis(double theta, double lamda, double phi) {
    basisMat = QuBasis.rotate(theta, lamda, phi);
  }
    
  /** デバッグ用に基底の個数を任意に指定できる
   * 基底を生成する（基底の個数を指定する）
   * @param order int // 基底の個数
   * @return quBasis QuBasis // 基底
   */
  public static QuBasis makeQuBasis(int order) {
    CompMat basisMat = new CompMat(order, order); // 複素数正方行列
    QuBasis quBasis = new QuBasis(basisMat);
    return quBasis;
  }
  
  /**
   * 基底の次数を得る
   * @return order int // 量子ゲートの次数
   */
  public int getOrder() {
    return basisMat.rowNumber();
  }
  
  /**
   * 基底を得る
   * @return basisMat CompMat // 基底
   */
  public CompMat getBasisMat() {
    return basisMat;
  }

  /**
   * 列番号で指定された基底の列ベクトルを得る
   * @param basisIndex int // 基底から取得する列ベクトルの列番号
   * @return basisVec CompVec // 基底から取得された列ベクトル
   */
  public CompVec getBasisVec(int basisIndex) {
    int row = basisMat.rowNumber(); // 基底の行数
    CompVec basisVec = new CompVec(row);
    for (int i = 0; i < row; i++) {
      basisVec.setComp(i, basisMat.getComp(i, basisIndex));
    }    
    return basisVec;
  }
  
  /**
   * 列番号で指定された基底の列ベクトルをコピーする
   * @param basisIndex int // 基底からコピーする列ベクトルの列番号
   * @return basisVec CompVec // 基底からコピーされた列ベクトル
   */
  public CompVec copyBasisVec(int basisIndex) {
    int row = basisMat.rowNumber(); // 基底の行数
    CompVec basisVec = new CompVec(row);
    for (int i = 0; i < row; i++) {
      basisVec.setComp(i, basisMat.copyComp(i, basisIndex));
    }    
    return basisVec;
  }
  
  /** 
   * 指定した軸の基底から取り出す
   * ※列がコピーされる
   * @param basisIndex int // 行番号
   * @return vasisQuBit QuBit // 基底
   */
  public QuBit getBasisQuBit(int basisIndex) {
    int row = basisMat.rowNumber(); // 基底の行数
    QuBit basisQuBit = new QuBit(QuBit.calcNumOfQuBit(row));
    for (int i = 0; i < row; i++) {
      basisQuBit.setProbAmp(i, basisMat.copyComp(i, basisIndex));
    }    
    return basisQuBit;
  }
  
  /**
   * 基底の要素を設定する
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @param element Complex // 要素
   */
  public void setElement(int row, int col, Complex element) {
    basisMat.setComp(row, col, element);
  }
  
  /**
   * 基底を設定する
   * @param basisMat CompMat // 基底の行列
   */
  public void setBasisMat(CompMat basisMat) {
    this.basisMat = basisMat;
  }
  
  /**
   * 量子ビット数から確率振幅数を求める（２のべき乗）
   * @param numQuBit int // 量子ビット数
   * @return numProbAmp int // 確率振幅数
   */
  public static int numOfProbAmp(int numQuBit) {
    int numProbAmp = 1;
    for(int i = 1; i <= numQuBit; i++) {
      numProbAmp *= 2; // ２のべき乗
    }
    return numProbAmp;
  } 
  
    
  // 単位行列 ---------------------------------------------
  /**
   * 単位実数行列の生成
   * @param order int // 行列の次数
   * @return ientity double[][] // 単位実数行列（正方行列）
   */
  public static double[][] makeRealIdentity(int order) {
    double[][] ientity = new double[order][order]; 
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        ientity[i][j] = 0.0;
      }
      ientity[i][i] = 1.0;
    }
    return ientity;
  }
  
  /**
   * 標準基底行列（複素数Complex[][]）を生成する
   * @param order int // 次数
   * @return stdBasis Complex[][] // 標準単位基底行列（正方行列）
   */
  public static Complex[][] standardBasisComplex(int order) {
    Complex[][] stdBasis = new Complex[order][order];
    
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        stdBasis[i][j] = new Complex(0.0, 0.0);
      }
      stdBasis[i][i] = new Complex(1.0, 0.0);
    }
    return stdBasis;
  }
  
  /**
   * 標準基底行列（複素数CompMat）を生成する
   * @param order int // 行列の次数
   * @return ientity CompMat // 単位複素数行列（正方行列）
   */
  public static CompMat standardBasis(int order) {
    CompMat basis = new CompMat(order, order);
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        basis.setComp(i, j, 0.0, 0.0);
      }
      basis.setComp(i, i, 1.0, 0.0);
    }
    return basis;
  }
  
  /**
   * 基底行列の共役を得る
   * @return _conjugate QuBasis // 共役転置な基底行列
   */
  public QuBasis conjugate() {
    int row = basisMat.rowNumber();
    // int col = basisMat.colNumber();
    QuBasis conjugate = makeQuBasis(row); // 共役転置な基底行列
    CompMat _conjugate = conjugate.basisMat;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < row; j++) {
        // _conjugate.setComp(j, i, basisMat.copyComp(i, j).conjugate());
        _conjugate.setComp(j, i, basisMat.getComp(i, j).conjugate());
      }
    }
    return conjugate;
  }
  

  // 座標回転 ---------------------------------------------
  /**
   * 回転行列（２次）を生成する
   * @param theta double // 回転角θ[radian]
   * @param lamda double // 回転角λ[radian]
   * @param phi double // 回転角φ[radian]
   * @return rotor　Complex[][] // 回転行列
   */
  public static Complex[][] rotate2(double theta, double lamda, double phi) {
    Complex[][] rotor = Complex.complexMatrix(2, 2);
    double cosTheta = Math.cos(theta);
    double sinTheta = Math.sin(theta);
    
    double p = 6.123233995736766E-17;
    double m = -p;
    if (( sinTheta <= p ) && (sinTheta >= m )) sinTheta = 0.0;
    if (( cosTheta <= p ) && (cosTheta >= m )) cosTheta = 0.0;
    
    rotor[0][0].setReal(cosTheta);
    
    rotor[0][1].setReal(-sinTheta);
    rotor[0][1].mul(Complex.euler(lamda));
    
    rotor[1][0].setReal(+sinTheta);
    rotor[1][0].mul(Complex.euler(phi));
    
    rotor[1][1].setReal(cosTheta);
    rotor[1][1].mul(Complex.euler(lamda + phi));
    return rotor;
  }
    
  /**
   * 回転行列（２次）を生成する
   * @param theta double // 回転角θ[radian]
   * @param lamda double // 回転角λ[radian]
   * @param phi double // 回転角φ[radian]
   * @return rotorMat CompMat // 回転行列
   */
  public static CompMat rotate(double theta, double lamda, double phi) {
    CompMat rotorMat = new CompMat(2, 2);
    Complex[][] rotor = rotorMat.getMatrix();
    double cosTheta = Math.cos(theta);
    double sinTheta = Math.sin(theta);
    
    double p = 6.123233995736766E-17;
    double m = -p;
    if (( sinTheta <= p ) && (sinTheta >= m )) sinTheta = 0.0;
    if (( cosTheta <= p ) && (cosTheta >= m )) cosTheta = 0.0;
    
    // 回転した座標を元の座標系で表した値
    // 元の座標系（標準基底)を絶対座標）として回転座標（基底）を表す
    rotor[0][0].setReal(cosTheta);
    
    rotor[0][1].setReal(-sinTheta);
    rotor[0][1].mul(Complex.euler(lamda));
    
    rotor[1][0].setReal(+sinTheta);
    rotor[1][0].mul(Complex.euler(phi));
    
    rotor[1][1].setReal(cosTheta);
    rotor[1][1].mul(Complex.euler(lamda + phi));

    return rotorMat;
  }

  /**
   * Gram-Schmidt の直交化法
   * @param referenceVectors Complex[][] // 参照ベクトル
   * @return orthonormalBasis QuBasis // 正規直交基底
   */
  public static QuBasis gramSchmidt(Complex[][] referenceVectors) {
    int row = referenceVectors.length;
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "QuBasis.gramSchmidt(Complex[][] referenceVectors)" + "\n"
          + " row length (=" + row + ") is less than one.");
      return null;
    }
    int col = referenceVectors[0].length;
    if (row != col) {
      DJ.print("***** ERROR ***** "
          + "QuBasis.gramSchmidt(Complex[][] referenceVectors)" + "\n"
          + " row and col length is not same.\n"
          + " row length is " + row + ", \n"
          + " col length is " + col + ".");
      return null;
    }
        
    QuBasis orthonormalBasis = makeQuBasis(row);
    Complex[][] u = orthonormalBasis.basisMat.getMatrix();

    for (int j = 0; j < col; j++) {
      u[0][j] = referenceVectors[0][j].copyComplex(); // u'0 = v0
    }
    Complex norm = Complex.norm(u[0]); // |v0| = norm(u'0)
    for (int j = 0; j < col; j ++) {
      u[0][j].div(norm); // u0 = u'0/|u'0|
    }
    
    for (int i = 1; i < row; i++) {
      Complex[] _u = new Complex[row]; // 
      for (int j = 0; j < col; j++) {
        _u[j] = referenceVectors[i][j].copyComplex(); // v'[i] = v[i]
      }
      
      for (int k = 0; k < i; k++) {
        Complex innPro = Complex.innerProduct(u[k], referenceVectors[i]); // v[i-1]・u[i]
        // Gram-Schmidt の直交化法で、内積の順序を変える場合
        // Complex innPro = Complex.innerProduct(referenceVectors[i], u[k]);
        
        for (int j = 0; j < col; j++) {
          u[i][j] = u[k][j].product(innPro); // (v[i-1]・u[i])u[i][k]
          _u[j].sub(u[i][j]); // u'[i] = v[i] - (v[i-1]・u[i])u[i][k]
        }
      }
      
      norm = Complex.norm(_u); // |u'[i]|
      for (int j = 0; j < col; j++) {
        u[i][j] = _u[j];
        u[i][j].div(norm); // u[i] = u'[i] / |u'[i]|
      }
    }

    return orthonormalBasis;
  }
  
  /**
   * 正規直交基底のチェック
   * ※チェック対象の名称を引数のラベルで受け取る
   * @param label String // チェック対象の名称　例：v2
   */
  public void checkOrthonormal(String label) {
    Complex[][] orthonormalArrays = basisMat.copyMatrix();
    DJ.printF(label, orthonormalArrays); // 
    
    DJ._print("行列式|" + label + "|がゼロでなければベクトルは独立");
    Complex determinant = Complex.determinant(orthonormalArrays);
    DJ.print_("|" + label + "|", determinant); // 
    DJ.printF("  |" + label + "|", determinant); // 
    
    DJ._print("行の内積・ノルムが１ならばその基底は正規化されている");
    Complex[] uArray;
    Complex innPro;
    Complex norm;
    int length = orthonormalArrays.length;
    for (int i = 0; i < length; i++) {
      DJ.print("" + i +"行目の内積とノルム");
      uArray = orthonormalArrays[i];
      innPro = Complex.innerProduct(uArray);
      // DJ.print_("innPro(" + label + "[" + i +"])", innPro); // 1.0+0.0i
      DJ.printF_("  innPro(" + label + "[" + i +"])", innPro); // 1+0i
      norm = Complex.norm(uArray);
      // DJ.print_("norm(" + label + "[" + i +"])", norm); // 1.0+0.0i
      DJ.printF("  norm(" + label + "[" + i +"])", norm); // 1+0i
    }

    DJ._print("行と行の内積がゼロならば２つの基底は直交している");
    Complex[] uArray_i;
    Complex[] uArray_k;
    for (int i = 0; i < length; i++) {
      uArray_i = orthonormalArrays[i];
      for (int k = i + 1; k < length; k++) {
        DJ.print_("" + i +"行目と" + k +"行目の内積");
        uArray_k = orthonormalArrays[k];
        innPro = Complex.innerProduct(uArray_i, uArray_k);
        DJ.print_(" ", innPro); // 
        DJ.printF("  ", innPro); // 
      }
    }
  
  }
  
  /**
   * 正規直交基底のチェック
   * @param referenceVectors Complex[][] // 参照ベクトル
   * ※"参照ベクトルuの行の内積・ノルム、および行間の内積もチェックする
   */
  public static void checkOrthonormal(Complex[][] referenceVectors) {
    // DJ._print("Orthonormal check"); // 
    
    // DJ.print("v", referenceVectors); // 
    DJ.printF("v", referenceVectors); // 
    DJ.print("行列式|v|がゼロならばベクトルは独立ではない");
    Complex determinant = Complex.determinant(referenceVectors);
    DJ.print_("|v|", determinant); // 
    DJ.printF("  |v|", determinant); // 
    
    DJ._print("参照ベクトルの各行の内積・ノルムが１ならばベクトルは正規化されている");
    Complex[] v;
    Complex innPro;
    Complex norm;
    int length = referenceVectors.length;
    for (int i = 0; i < length; i++) {
      DJ.print("参照ベクトルの" + i +"行目の内積とノルム");
      v = referenceVectors[i];
      innPro = Complex.innerProduct(v);
      DJ.print_("innPro(v[" + i +"])", innPro); // 1.0+0.0i
      DJ.printF("  innPro(v[" + i +"])", innPro); // 1+0i
      norm = Complex.norm(v);
      DJ.print_("norm(v[" + i +"])", norm); // 1.0+0.0i
      DJ.printF("  norm(v[" + i +"])", norm); // 1+0i
    }
    
    DJ._print("参照ベクトルの行と行の内積がゼロならば２つの基底は直交している");
    Complex[] u_i;
    Complex[] u_k;
    for (int i = 0; i < length; i++) {
      u_i = referenceVectors[i];
      for (int k = i + 1; k < length; k++) {
        DJ.print_("参照ベクトルの" + i +"行目と" + k +"行目の内積");
        u_k = referenceVectors[k];
        innPro = Complex.innerProduct(u_i, u_k);
        DJ.print_(" ", innPro); // 
        DJ.printF("  ", innPro); // 
      }
    }
    
    QuBasis orthonormalVec = QuBasis.gramSchmidt(referenceVectors);
    
    // DJ._print("u", orthonormalVec); // 
    DJ.printF("u", orthonormalVec); // 
    Complex[][] orthonormalArrays = orthonormalVec.getBasisMat().getMatrix();
    DJ.print("行列式|u|がゼロならばベクトルは独立ではない");
    determinant = Complex.determinant(orthonormalArrays);
    DJ.print_("|u|", determinant); // 
    DJ.printF("  |u|", determinant); // 
    
    DJ._print("行の内積・ノルムが１ならばその基底は正規化されている");
    Complex[] u;
    length = referenceVectors.length;
    for (int i = 0; i < length; i++) {
      DJ.print("導出基底の" + i +"行目の内積とノルム");
      u = orthonormalArrays[i];
      innPro = Complex.innerProduct(u);
      DJ.print_("innPro(u[" + i +"])", innPro); // 1.0+0.0i
      DJ.printF("  innPro(u[" + i +"])", innPro); // 1+0i
      norm = Complex.norm(u);
      DJ.print_("norm(u[" + i +"])", norm); // 1.0+0.0i
      DJ.printF("  norm(u[" + i +"])", norm); // 1+0i
    }

    DJ._print("行と行の内積がゼロならば２つの基底は直交している");
    for (int i = 0; i < length; i++) {
      u_i = orthonormalArrays[i];
      for (int k = i + 1; k < length; k++) {
        DJ.print_("導出基底の" + i +"行目と" + k +"行目の内積");
        u_k = orthonormalArrays[k];
        innPro = Complex.innerProduct(u_i, u_k);
        DJ.print_(" ", innPro); // 
        DJ.printF("  ", innPro); // 
      }
    }

  }  
  
  /**
   * 正規直交基底のチェック（正規直交化処理を含む）
   */
  public void checkOrthonormal() {
    Complex[][] referenceVectors = basisMat.getMatrix();
    
    DJ.print("v", referenceVectors); // 
    DJ.printF("v", referenceVectors); // 
    DJ.print("行列式|v|がゼロならばベクトルは独立ではない");
    Complex determinant = Complex.determinant(referenceVectors);
    DJ.print_("|v|", determinant); // 
    DJ.printF("  |v|", determinant); // 
    
    QuBasis orthonormalVec = QuBasis.gramSchmidt(referenceVectors);
    
    DJ._print("u", orthonormalVec); // 
    DJ.printF("u", orthonormalVec); // 
    Complex[][] orthonormalArrays = orthonormalVec.getBasisMat().getMatrix();
    DJ.print("行列式|u|がゼロならばベクトルは独立ではない");
    determinant = Complex.determinant(orthonormalArrays);
    DJ.print_("|u|", determinant); // 
    DJ.printF("  |u|", determinant); // 
    
    DJ._print("行の内積・ノルムが１ならばその基底は正規化されている");
    Complex[] u;
    Complex innPro;
    Complex norm;
    int length = referenceVectors.length;
    for (int i = 0; i < length; i++) {
      DJ.print("" + i +"行目の内積とノルム");
      u = orthonormalArrays[i];
      innPro = Complex.innerProduct(u);
      DJ.print_("innPro(u[" + i +"])", innPro); // 1.0+0.0i
      DJ.printF("  innPro(u[" + i +"])", innPro); // 1+0i
      norm = Complex.norm(u);
      DJ.print_("norm(u[" + i +"])", norm); // 1.0+0.0i
      DJ.printF("  norm(u[" + i +"])", norm); // 1+0i
    }

    DJ._print("行と行の内積がゼロならば２つの基底は直交している");
    Complex[] u_i;
    Complex[] u_k;
    for (int i = 0; i < length; i++) {
      u_i = orthonormalArrays[i];
      for (int k = i + 1; k < length; k++) {
        DJ.print_("" + i +"行目と" + k +"行目の内積");
        u_k = orthonormalArrays[k];
        innPro = Complex.innerProduct(u_i, u_k);
        DJ.print_(" ", innPro); // 
        DJ.printF("  ", innPro); // 

      }
    }

  }
  

// 測定 -------------------------------------------------
  
  /**
   * 各基底軸の選択された回数を得る
   * @return qbCounter int[] // 基底毎のカウント値
   */
  public int[] getQBCounter() {
    return qbCounter;
  }
  
  /**
   * 選択された基底軸を得る
   * @return selectedQuBits QuBit[] // 選択された基底
   */
  public QuBit[] getSelectedQuBits() {
    return selectedQuBits;
  }
  
  /**
   * 選択された古典ビットを得る
   * @return qbCounter int[] // 選択された古典ビット
   */
  public int[] getClassicBit() {
    return clBitArray;
  }
  
  /**
   * 測定を試行回数繰り返し、基底毎のカウント値を得る
   * @param targetQuBit QuBit // 被測定量子ビット
   * @param trialNumber int // 試行回数
   * @return count int[] // 基底毎のカウント値
   */
  public int[] countQuBasis(QuBit targetQuBit, int trialNumber) {
    int numOfBasis = basisMat.colNumber();
    qbCounter = new int[numOfBasis];
    selectedQuBits = new QuBit[trialNumber]; // 選択された基底
    clBitArray = new int[trialNumber]; // 古典ビット配列
    for (int i = 0; i < trialNumber; i++) {
      selectedQuBits[i] = this.measurement(targetQuBit);
      clBitArray[i] = classicBit;
    }
    
    return qbCounter;
  }
  
  /**
   * 基底による測定で遷移した量子ビットを出力する
   * ※　遷移する基底は乱数で選択される
   * 　　基底が選択される確率は確率振幅の自乗で与えられる
   * @param targetQuBit QuBit // 被測定量子ビット
   * @return 
   */
  public QuBit measurement(QuBit targetQuBit) {
    // DJ.printF("$ basisMat", basisMat);
    // DJ.printF("$ targetQuBit", targetQuBit);
    int numOfProbAmp = targetQuBit.getProbAmpLength(); // 確率振幅の個数
    QuBit paQuBit = this.basisChang(targetQuBit); // 基底変換後の確率振幅
    // DJ.printF("$ paQuBit", paQuBit);

    double paAccum = 0; // 確率振幅の自乗の累積値
    int resultQubitNum = QuBit.calcNumOfQuBit(basisMat.rowNumber());
    QuBit resultQubit = new QuBit(resultQubitNum); // 出力する量子ビット
    // DJ.printF("$ resultQubit", resultQubit);
    boolean errorCheck = false;
    double randomNum = random.nextDouble(); // 疑似乱数
    
    if(qbCounter == null) qbCounter = new int[numOfProbAmp]; // 基底カウンタ
    
    for (int i = 0; i < numOfProbAmp; i++) {
      // DJ.print("$ i = ", i);
      paAccum = paAccum + paQuBit.getProbAmp(i).square(); // 自乗の累積
      // DJ.print("$ paAccum = ", paAccum);
      // DJ.print("$ randomNum = ", randomNum);
      if (randomNum <= paAccum) {
        for (int k = 0; k < numOfProbAmp; k++) {
          resultQubit.setProbAmp(k, basisMat.copyComp(k, i));
        }
        // DJ.printF("$ resultQubit", resultQubit);
        qbCounter[i]++; // 基底カウンタを更新
        classicBit = i; // 古典ビット(0,1, ,n-1)
        errorCheck = true;
        break;
      }
    }
    // 確率の累積が乱数を超えないエラーのチェック
    if (!errorCheck) {
      DJ.print("***** ERROR ***** "
          + "QuBasis.innerProduct(QuBit operand)" + "\n"
          + " basisMat row number is less than one.");
      return null;
    }
        
    return resultQubit;
  }
  
  /**
   * 基底変換した量子ビットの確率振幅を求める（１量子ビット）
   *  基底行列basisMatの転置と量子ビット（確率振幅）の内積を求める
   *  target QuBasis // 被演算数複素数基底
   * @param operand QuBit // 演算数量子ビット
   * @return resultQB QuBit // 測定結果の量子ビット（確率振幅）
   */
  public QuBit basisChang(QuBit operand) {
    int row = basisMat.rowNumber(); // 基底、複素正方行列
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "QuBasis.basisChang(QuBit operand)" + "\n"
          + " basisMat row number is less than one.");
      return null;
    }
    int col = basisMat.colNumber(); // 基底、複素正方行列
    if (col%2 != 0) {
      DJ.print("***** ERROR ***** "
          + "QuBasis.basisChang(QuBit operand)" + "\n"
          + " basisMat column number is (" + col
          + ") is not even number.");
      return null;
    }
    
    int probAmpLength = operand.getProbAmpLength(); // 確率振幅数
    if (row != probAmpLength) {
      DJ.print("***** ERROR ***** "
          + "QuBasis.basisChang(QuBit operand)" + "\n"
          + " Element numbers are not same.\n"
          + " basisMat row number is " + row + ", \n"
          + " ProbAmp length is " + probAmpLength + ".");
      return null;
    }

    // 被演算数複素数行列は共役行列に置き換える
    CompMat _basisMat = basisMat.conjugate();
    int numOfQuBit = QuBit.calcNumOfQuBit(row);
    QuBit resultQB = new QuBit(numOfQuBit);
    CompVec _product = resultQB.getProbAmp();
    for (int i = 0; i < col; i++) {
      for (int k = 0; k < row; k++) {
        Complex conj = _basisMat.getComp(i, k);
        conj.mul(operand.getProbAmp(k));
        _product.getComp(i).add(conj);
      }
    }
    return resultQB;
  }
  
  /** // 使われていないかも？？？
   * 基底変換した量子ビットの確率振幅を求める（複数量子ビット）
   * 基底行列basisMatの転置と量子ビット（確率振幅）行列の内積を求める
   * @param quBitMat CompMat // 量子ビット（確率振幅）行列
   * @return resultQBMat CompMat // 量子ビット（確率振幅）行列
   */
  public CompMat basisChang(CompMat quBitMat) {
    int row = basisMat.rowNumber(); // 基底、複素正方行列
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "quBasis.basisChang(CompMat quBitMat)" + "\n"
          + " basisMat row number is less than one.");
      return null;
    }
    int col = basisMat.colNumber(); // 基底、複素正方行列
    if (col != row) {
      DJ.print("***** ERROR ***** "
          + "quBasis.basisChang(CompMat quBitMat)" + "\n"
          + " basisMat column number is (" + col
          + ") not equal to row number.");
      return null;
    }
    
    int numOfQuBit = quBitMat.rowNumber(); // 量子ビット行列に含まれる量子ビット数
    if (!QuBit.checkPowerOfTwo(numOfQuBit)) { // ２のべき乗か？
      DJ.print("***** ERROR ***** "
          + "quBasis.basisChang(CompMat quBitMat)" + "\n"
          + " number of QuBits is not power of two.");
      return null;
    }
    
    int order = quBitMat.colNumber(); // 量子ビットの確率振幅数
    if (order != row) {
      DJ.print("***** ERROR ***** "
          + "quBasis.basisChang(CompMat quBitMat)" + "\n"
          + " order of QuBits (" + order
          + ") is not equal to QuBasis(" + row + ")");
      return null;
    }
    
    // 測定用基底行列は共役転置行列に置き換える
    CompMat _basisMat = basisMat.conjugate();
    CompMat resultQBMat = new CompMat(col, numOfQuBit);
    
    for (int i = 0; i < col; i++) {
      for (int j = 0; j < numOfQuBit; j++) {
        for (int k = 0; k < order; k++) {
          Complex conj = _basisMat.copyComp(i, k);
          conj.mul(quBitMat.getComp(k, j));
          Complex val = resultQBMat.getComp(i, j);
          val.add(conj);
        }
      }
    }
    return resultQBMat;
  }
  
  /**
   * 基底同士の相互作用により基底を生成する
   * @param operandBasis QuBasis // 相互作用させる基底
   * @return interactedGate QuGate // 相互作用による基底
   */
  public QuBasis interactQuBasis(QuBasis operandBasis) {
    int row0 = basisMat.rowNumber();
    int col0 = basisMat.colNumber();
    CompMat operandMat = operandBasis.basisMat;
    int row1 = operandMat.rowNumber();
    int col1 = operandMat.colNumber();
    
    int row = row0 * row1;
    int numOfQuBit = QuBit.calcNumOfQuBit(row);
    QuBasis interactedBasis = new QuBasis(numOfQuBit);
    for (int i0 = 0; i0 < row0; i0++) {
      for (int j0 = 0; j0 < col0; j0++) {
        
        for (int i1 = 0; i1 < row1; i1++) {
          for (int j1 = 0; j1 < col1; j1++) {
            int i = i0 * row1 + i1;            
            int j = j0 * col1 + j1;
            Complex element = basisMat.copyComp(i0, j0);
            element.mul(operandMat.getComp(i1, j1));
            interactedBasis.setElement(i, j, element);
          }
        }
        
      }
    }
    
    return interactedBasis;
  }
  
  
  // ==================================================================
  // 全体を中括弧で括る（行列と同じ書式）
  // 基底ごとに鍵括弧で括り、コンマで区切り、改行する
  // 基底ごとにコンマで区切る
  // 基底の要素はコンマで区切らない（ベクトルと同じ書式）
  
  /**  CompMatと同じ
   * 連続する空白文字列を得る
   * @param size int // 空白文字の個数
   * @return space.toString String // 空白文字列
   */
  private String getSpace(int size) {
    StringBuilder space = new StringBuilder(); // 表示文字列
    for (int i = 0; i < size; i++) {
      space.append(" "); // 空白を追加
    }
    return space.toString();
  }
  

  /**
   * 1.基底の文字列（行列表示）を得る
   * @return String // 複素数行列の文字列
   */
  @Override
  public String toString() {
    int row = basisMat.rowNumber(); // 行数 (列内の要素数
    int col = basisMat.colNumber(); // 列数（列内の要素数
    int maxWide[] = new int[col]; // 数値文字列の長さの最大長 (最小例：-0+0i)
    for (int j = 0; j < col; j++) maxWide[j] = 5; // 少数の桁数
    
    // 数値文字列の抽出
    String[][] buf = new String[row][col]; // 各行列要素の文字列
    int[][] wides = new int[row][col]; // 各行列要素の文字数　→　余白数
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        // DJ.print_("i,j=", i);  DJ.print_(", ", j);
        String text = basisMat.getComp(i, j).toString(); // 各行列要素の数値文字列化

        if (basisMat.getReal(i, j) >= 0) { // 実部が正
          buf[i][j] = " " + text; // 数値文字列の先頭に空白を追加して格納
          wides[i][j] = text.length() + 1; // 数値文字列の長さ
        } else {
          buf[i][j] = text; // 数値文字列を格納
          wides[i][j] = text.length(); // 数値文字列の長さ
        }
        // DJ.print(",  text=", text);
        // DJ.print_("wides[" + i + "][" + j + "]=", wides[i][j]);

        if (wides[i][j] > maxWide[j]) { // 数値文字列の長さを最大長と比較
          maxWide[j] = wides[i][j]; // 数値文字列の最大長を更新
        }
        // DJ.print(",  maxWide[" + j + "]=", maxWide[j]);
      }
    }
    
    // 数値文字列に枠を追加して格納
    StringBuilder sb = new StringBuilder();
    int index;
    
    String filler = "│";
    int gap = filler.length();
    int spaces = maxWide[0];
    for (int j = 1; j < col; j++) {
      spaces = spaces + gap + 1 + maxWide[j]; 
    }
    sb.append("┌").append(getSpace(spaces)).append("┐\n"); // 最上行
    
    for (int i = 0; i < row; i++) { // 中間行
      sb.append("│"); // 左枠
      for (int j = 0; j < col; j++) { // 行内の数値文字列
        sb.append(buf[i][j]); // 数値
        sb.append(getSpace(maxWide[j] - wides[i][j])).append(filler); // 余白と区切り枠
      }
      index = sb.length();
      sb.replace(index - gap, index, "│\n"); // 右枠
    }
//    sb.append("└").append(getSpace(spaces)).append("┘\n"); // 最下行
    sb.append("└").append(getSpace(spaces)).append("┘"); // 最下行
    return sb.toString();
  }

  
  /** CompMatを修正
   * 2.複素数行列の文字列（行列表示、枠・書式付き）を得る
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toFormat() {
    int row = basisMat.rowNumber(); // 行の長さ＝行内の要素数（列数）
    int col = basisMat.colNumber(); // 列の長さ＝列内の要素数（行数）
    int maxWide[] = new int[col]; // 数値文字列の長さの最大長 (最小例：-0+0i)
    for (int j = 0; j < col; j++) maxWide[j] = 5;
    
    // 数値文字列の抽出
    String[][] buf = new String[row][col]; // 文字列バッファ
    int[][] wides = new int[row][col]; // 余白数
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        // DJ.print_("i,j=", i);  DJ.print_(", ", j);
        String text = basisMat.getComp(i, j).toFormat();

        if (basisMat.getReal(i, j) >= 0) { // 実部が正
          buf[i][j] = " " + text;
          wides[i][j] = text.length() + 1; // 数値文字列の長さ
        } else {
          buf[i][j] = text;
          wides[i][j] = text.length(); // 数値文字列の長さ
        }
        // DJ.print(",  text=", text);
        // DJ.print_("wides[" + i + "][" + j + "]=", wides[i][j]);

        if (wides[i][j] > maxWide[j]) {
          maxWide[j] = wides[i][j]; // wideを更新
        }
        // DJ.print(",  maxWide[" + j + "]=", maxWide[j]);
      }
    }
    
    StringBuilder sb = new StringBuilder("F\n");
    int index;
    
    String filler = "│";
    int gap = filler.length();
    int spaces = maxWide[0];
    for (int j = 1; j < col; j++) {
      spaces = spaces + gap + 1 + maxWide[j]; 
    }
    sb.append("┌").append(getSpace(spaces)).append("┐\n");
    
    for (int i = 0; i < row; i++) {
      sb.append("│");
      for (int j = 0; j < col; j++) {
        sb.append(buf[i][j]); // 数値
        sb.append(getSpace(maxWide[j] - wides[i][j])).append(filler); // 余白
      }
      index = sb.length();
      sb.replace(index - gap, index, "│\n");
    }
    sb.append("└").append(getSpace(spaces)).append("┘");
    return sb.toString();
  }
  
  
  /**
   * 3.基底の文字列（行列表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toString(String label) {
    String str = this.toString();
    
    return label + "\n"  + str;
  }
  

  /**  CompMatと同じ
   * 4.基底の文字列（行列表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toFormat(String label) {
    String str = this.toFormat();
    
    return label + ":" + str;
  }
  
} // QuBasis Class

// EOF

