/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2025
 */
package quan;

import java.util.ArrayList;
import util.comp.Complex;
import util.DJ;
import util.comp.CompMat;
import util.comp.CompVec;

/**
 * <p> 表　題: Class: QuBit</p>
 * <p> 説　明: 量子ビット</p>
 * <p> 
 * <p> １個の量子ビットは２個の確率振幅(probability amplitude)で表される
 * <p> 多量子ビットの確率振幅の個数は２の量子ビット数べき乗となる
 * <p> 相互作用の無い多量子ビットは複数の量子ビットで表される
 * <p> 
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024</p>
 * <p> 作成日: 2024.02.13</p>
 */
public class QuBit {
  
  public static final int PA0 = 0; // 先頭の確率振幅のインデックス
  public static final int PA1 = 1; // ２番目の確率振幅のインデックス
  
  public static final String STD0_QUBIT = "std0QuBit"; // 標準量子ビット[1 0]'ラベル
  public static final String STD1_QUBIT = "std1QuBit"; // 標準量子ビット[0 1]'ラベル
  
  public static final boolean H_BIT = false;  // 行（横方向）表示
  public static final boolean V_BIT = true; // 列（縦方向）表示
  
  public static boolean P2_FLAG = true; // ２のべき乗
  public static boolean N2_FLAG = false; // ２のべき乗ではない
  
  private CompVec probAmp; // 確率振幅
  
  /**
   * 単位量子ビットの生成
   * 単位量子ビット|e0>=[1+0i 0+0i]' ('は転置)
   * 
   */
  public QuBit() {
    probAmp = new CompVec(2); // probAmpの要素は全て0+0i
    probAmp.setReal(0, 1.0); // probAmp[0]=1+0i
  }
  
  /**
   * 回転量子ビットの生成
   * @param radian double // 回転角度[ラジアン]
   */
  public QuBit(double radian) {
    double lamda = 0.0;
    double phi = 0.0;
    CompMat rotatedMat = QuBasis.rotate(radian, lamda, phi);
    // DJ.printF("rotatedMat", rotatedMat);
    // 回転行列の1列目を抽出し、確率振幅に代入する
    probAmp = new CompVec(2); // probAmpの要素は全て0+0i
    probAmp.setComp(PA0, rotatedMat.getComp(PA0, PA0));
    probAmp.setComp(PA1, rotatedMat.getComp(PA1, PA0));
    
  }
  
  /** 
   * 単位量子ビットの生成
   * 最初の確率振幅のみ指定する
   * 2番目の確率振幅は自動生成される（正の確率振幅が採用される）
   * @param pa0 Complex // 確率振幅
   */
  public QuBit(Complex pa0) {
    probAmp = new CompVec(2);
    probAmp.setComp(PA0, pa0);
    probAmp.setComp(PA1, QuBit.getOtherPA(pa0));
  }
  
  
  /** 
   * 単位量子ビットの生成
   * @param pa0 Complex // 確率振幅
   * @param pa1 Complex // 確率振幅
   */
  public QuBit(Complex pa0, Complex pa1) {
    probAmp = new CompVec(2);
    probAmp.setComp(PA0, pa0);
    probAmp.setComp(PA1, pa1);
  }
  
  /**
   * 標準量子ビットを生成する（１量子ビット）
   * @param quBitLabel String // 生成する標準量子ビットのラベル
   */    
  public QuBit(String quBitLabel) {
    switch (quBitLabel) {
      case STD0_QUBIT: // 標準量子ビット[1 0]'
        probAmp = new CompVec(2, QuBit.PA0); // 確率振幅[1 0]'
        break;
      case STD1_QUBIT: // 標準量子ビット[0 1]'
        probAmp = new CompVec(2, QuBit.PA1); // 確率振幅[0 1]'
        break;
      default: // エラー
        DJ.print("***** ERROR ***** "
            + "QuBit.QuBit(String quBitLabel)" + "\n"
            + " Argument is (" + quBitLabel + ") is wrong.");
        break;
    }
  }

  /**
   * 多量子ビットの生成
   * @param numOfQuBits int // 量子ビットの個数
   */
  public QuBit(int numOfQuBits) {
    int numOfProbAmps = QuBit.calcNumOfProbAmp(numOfQuBits); // 確率振幅数
    probAmp = new CompVec(numOfProbAmps);
  }
  
  /**
   * 標準量子ビットを生成する（多量子ビット）
   * インデックスで指定された確率振幅のみを１とする量子ビット
   * @param numOfQuBit int // 量子ビット数：２のべき乗
   * @param index int // 実部を１とする確率振幅のインデックス
   */
  public QuBit(int numOfQuBit, int index) {
    int numOfProbAmp = QuBit.calcNumOfProbAmp(numOfQuBit); // 確率振幅数
    if (index >= numOfProbAmp) {
      DJ.print("***** ERROR ***** "
          + "QuBit.unitQuBit(int, int)" + "\n"
          + "index (" + index + ") is greater than equal to"
          + "Number of ProbAmp (" + numOfProbAmp + ").");
    }
    
    probAmp = new CompVec(numOfProbAmp, index);
  }
  
  /**
   * 量子ビットの生成（コピー）
   * @param quBit QuBit // コピーされる量子ビット
   */
  public QuBit(QuBit quBit) {
      probAmp = quBit.copyProbAmpVec();  
  }
  
  /**
   * 量子ビットの生成
   * @param probAmpVec CompVec // 確率振幅の配列
   */
  public QuBit(CompVec probAmpVec) {
    int numOfProbAmps = probAmpVec.length();
    if (numOfProbAmps <= 0) {
      DJ.print("***** ERROR ***** "
          + "QuBit.QuBit(CompVec probAmpVec)" + "\n"
          + " Number of ProbAmps (" + numOfProbAmps
          + ") is less than 1.");
    }
    else if (!checkPowerOfTwo(numOfProbAmps)) { // 確率振幅の個数は２のべき乗
      DJ.print("***** ERROR ***** "
          + "QuBit.QuBit(CompVec probAmpVec)" + "\n"
          + " Number of ProbAmps (" + numOfProbAmps
          + ") is not power of 2.");
    }
    else {
      probAmp = probAmpVec.copyCompVec();
    }
  }
  
  /**
   * 複素数配列
   * 　確率振幅を与える複素数配列の長さが２でなければ、
   * 　確率振幅はデフォルト値（1,0）に初期化される。
   * @param probAmpArray Complex[] // 確率振幅を与える複素数配列
   */
  public QuBit(Complex[] probAmpArray) {
    int numOfProbAmps = probAmpArray.length;
    if (numOfProbAmps%2 != 0) {
      DJ.print("***** ERROR ***** "
          + "QuBit.QuBit(Complex[] probAmp2)" + "\n"
          + " Number of QuBits (" + numOfProbAmps
          + ") is not even number.");
    }
    else {
      probAmp = new CompVec(numOfProbAmps);
      probAmp.copyArray(probAmpArray);
    }
  }
  
  /**
   * 量子ビットリストで量子ビットを生成する
   * @param quBitList ArrayList＜QuBit＞ // 量子ビットリスト
   */
  public QuBit(ArrayList<QuBit> quBitList) {
    int numOfQuBits = quBitList.size(); // 量子ビット数
    int numOfProbAmps = QuBit.calcNumOfProbAmp(numOfQuBits); // 確率振幅数
    
    probAmp = new CompVec(numOfProbAmps);
    for (int i = 0; i < numOfQuBits; i++) {
      int index = i * 2;
      probAmp.setComp(index, quBitList.get(i).copyProbAmp(QuBit.PA0));
      probAmp.setComp(index + 1, quBitList.get(i).copyProbAmp(QuBit.PA1));
    }
  }
  
  // ------------------------------------------------------
  /**
   * 確率振幅を反転した量子ビットを返す
   * @return reverseQB QuBit // 確率振幅を反転した量子ビット
   */
  public QuBit reverseQuBit(){
    QuBit reverseQB = new QuBit(
        probAmp.copyComp(QuBit.PA1),
        probAmp.copyComp(QuBit.PA0));
    return reverseQB;
  }
  
  /**
   * インデックスで指定された確率振幅のみを１とする量子ビット
   * @param numOfQuBit int // 量子ビット数：２のべき乗
   * @param index int // 要素を１とする確率振幅のインデックス
   * @return aQuBit QuBit // 
   */
  public static QuBit unitQuBit(int numOfQuBit, int index) {
    int numOfProbAmp = calcNumOfProbAmp(numOfQuBit);
    if (index >= numOfProbAmp) {
      DJ.print("***** ERROR ***** "
          + "QuBit.unitQuBit(int, int)" + "\n"
          + "index (" + index + ") is greater than equal to"
          + "Number of ProbAmp (" + numOfProbAmp + ").");
    }
    
    QuBit aQuBit = new QuBit(numOfQuBit);
    aQuBit.probAmp.setComp(index, 1.0, 0.0); // インデックス、実部、虚部
    
    return aQuBit; // 量子ビット
  }
  
  /**
   * 標準量子ビットの配列を生成する
   * @param numOfQuBit int // 量子ビット数
   * @return stdQuBitArray QuBit[] // 標準量子ビットの配列
   */
  public static QuBit[] standardQuBit(int numOfQuBit) {
    int order = QuBit.calcNumOfProbAmp(numOfQuBit); // 確率振幅の個数
    QuBit[] stdQuBitArray = new QuBit[order];
    for (int i = 0; i < order; i++) {
      stdQuBitArray[i] = new QuBit(numOfQuBit);
      stdQuBitArray[i].probAmp.setComp(i, 1.0, 0.0); // インデックス、実部、虚部
    }
    
    return stdQuBitArray;
  }
  
  /**
   * 回転した量子ビットを生成（角度で指定）
   * @param angle double // 回転角度
   * @return aQuBit QuBit // 量子ビット
   */
  public static QuBit rotateDegreeQuBit(double angle) {
    double pai = Math.PI;
    double theta = pai * angle / 180.0; // radian

    return rotateRadianQuBit(theta);
  }
  
  /**
   * 回転した量子ビットを生成（ラジアンで指定）
   * @param theta double // 
   * @return aQuBit QuBit // 量子ビット
   */
  public static QuBit rotateRadianQuBit(double theta) {
    double cosTheta = Math.cos(theta);
    double sinTheta = Math.sin(theta);

    // sin、cosの値がゼロ近い場合、ゼロにする    
//    double p = 6.123233995736767E-17; // 後尾の桁ｎ値が異なる
    double p = 6.123233995736766E-17;
    double m = -p;
    if (( sinTheta <= p ) && (sinTheta >= m )) sinTheta = 0.0;
    if (( cosTheta <= p ) && (cosTheta >= m )) cosTheta = 0.0;
    
    Complex pa0 = new Complex(cosTheta, 0.0);
    Complex pa1 = new Complex(sinTheta, 0.0);
    QuBit aQuBit = new QuBit(pa0, pa1);
    
    return aQuBit;
  }
  
  // ------------------------------------------------------
  
  /**
   * 確率振幅の個数が２のべき乗であることをチェックする
   * ※ 0:2^0=1 を除く
   * @param numOfProbAmps
   * @return 
   */
  public static boolean checkPowerOfTwo(int numOfProbAmps) {
    boolean result = N2_FLAG; // false; ２のべき乗ではない
    int operand = numOfProbAmps;
       
    if (operand < 2) {}
    else if ((operand & (operand - 1)) == 0) result = P2_FLAG;

    return result;
  }
  
  
  /**
   * 量子ビットの個数を得る
   * @return numOfQuBits int // 量子ビットの個数
   */
  public int getNumOfQuBit() {
    int numOfProbAmp = probAmp.length();
    int numOfQuBit = calcNumOfQuBit(numOfProbAmp);
    return numOfQuBit;
  }
  
  /**
   * 確率振幅配列の長さを得る
   * @return probAmp int // 確率振幅配列の長さ
   */
  public int getProbAmpLength() {
    return probAmp.length();
  }
  
  /**
   * 確率振幅を得る
   * @param index int // インデックス
   * @return probAmp.getComp(index) Complex // 確率振幅
   */
  public Complex getProbAmp(int index) {
    return probAmp.getComp(index);
  }
  
  /**
   * 確率振幅を設定する
   * @param index int // インデックス
   * @param probAmp Complex // 確率振幅
   */
  public void setProbAmp(int index, Complex probAmp) {
    this.probAmp.setComp(index, probAmp);
  }
  
  /**
   * 確率振幅のコピーを得る
   * @param index int // インデックス
   * @return probAmp[index] Complex // 確率振幅
   */
  public Complex copyProbAmp(int index) {
    return probAmp.copyComp(index);
  }
  
  /**
   * 確率振幅配列を得る
   * @return probAmp CompVec // 確率振幅配列
   */
  public CompVec getProbAmp() {
    return probAmp;
  }
  
  /**
   * 確率振幅配列のコピーを得る
   * @return _probAmp CompVec // 確率振幅配列
   */
  public CompVec copyProbAmpVec() {
    CompVec _probAmp = probAmp.copyCompVec();
    return _probAmp;
  }
  
  /**
   * 確率振幅配列を得る
   * @return probAmpArray Complex[] // 確率振幅配列
   */
  public Complex[] getProbAmpArray() {
    return probAmp.getArray();
  }
  
  /**
   * 確率振幅配列を得る
   * @return _probAmp Complex[] // 確率振幅配列
   */
  public Complex[] copyProbArray() {
    Complex[] _probAmp = probAmp.copyArray();
    return _probAmp;
  }
  
  public void setProbAmp(CompVec probAmps) {
    Complex[] complexArray = probAmps.copyArray();
    probAmp.copyArray(complexArray);
  }
  
  /**
   * 単位多量子ビットのリストの生成
   * @param numOfQuBits int // 量子ビットの個数
   * @return aQuBits ArrayList< QuBit> // 量子ビットのリスト
   */
  public static ArrayList<QuBit> QuBits(int numOfQuBits) {
    ArrayList<QuBit> aQuBits = new ArrayList<>();
    for (int i = 0; i < numOfQuBits; i++) {
      QuBit aQuBit = new QuBit();
      aQuBits.add(aQuBit);
    }
    return aQuBits;
  }
  
  
  // 量子ビットの演算 -------------------------------------
  
  /**
   * 量子ビットの対応する確率振幅毎の和を求める
   * @param operand QuBit // 量子ビット
   */
  public void sum(QuBit operand) {
    probAmp.sum(operand.getProbAmp());
  }
  
  /**
   * 量子ビットに実数を掛ける
   * @param value double // 実数
   */
  public void product(double value) {
    probAmp.product(value);
  }
  
  /**
   * 量子ビットに複素数を掛ける
   * @param value double // 複素数
   */
  public void product(Complex value) {
    probAmp.product(value);
  }
  
  
// 量子ビットを表す正規直交基底の確率振幅を求める
// 正規直交基底で量子ビットを表す場合の確率振幅を求める

// 無意味な処理：量子ビットはもともと正規直交基底で表されているので
// 使用されていない
  /**
   * 量子ビットを正規直交基底で表す場合の確率振幅を求める
   * @return  probAmpVec CompVec // 確率振幅
   */
  public CompVec basisChang() {
    int paLength = probAmp.length(); // 対象量子ビットの確率振幅の長さ
    int numOfQuBit = calcNumOfQuBit(paLength); // 量子ビットの個数    
    QuBasis stdBasis = new QuBasis(numOfQuBit); // 標準基底
    
    CompMat basisMat = stdBasis.getBasisMat();
    // int row = basisMat.rowNumber(); // 基底ベクトルの長さ（要素数）
    
    // 測定用標準基底行列は共役転置行列に置き換える
    CompVec probAmpVec = new CompVec(paLength);
    
    for (int i = 0; i < paLength; i++) {
      for (int k = 0; k < paLength; k++) {
        Complex basis = basisMat.copyComp(i, k);
        basis.mul(probAmp.getComp(k));
        Complex val = probAmpVec.getComp(i);
        val.add(basis);
      }
    }
        
    return probAmpVec;
  }

  
  /**
   * 量子ビットを所与の測定用直交基底で表す確率振幅を求める
   * ※測定用基底行列は共役転置行列に置き換えられる
   * @param operandBasis QuBasis // 所与の測定用直交基底
   * @return probAmpVec CompVec // 確率振幅
   */
  public CompVec basisChang(QuBasis operandBasis) {
    int length = probAmp.length(); // 対象量子ビットの確率振幅の長さ
    if (length < 1) {
      DJ.print("***** ERROR ***** "
          + "QuBit.basisChang(QuBasis operandBasis)" + "\n"
          + " Qubit length(=" + length + ") is less than one.");
      return null;
    }
    
    CompMat basisMat = operandBasis.getBasisMat();
    int row = basisMat.rowNumber(); // 基底ベクトルの長さ（要素数）
    if (row != length) {
      DJ.print("***** ERROR ***** "
          + "quBasis.basisChang(QuBasis operandBasis)" + "\n"
          + " basisMat row number(=" + row + ") is " + "\n"
          + "not equal to length of Qubit(=" + length + ") .");
      return null;
    }
    
    // 測定用基底行列は共役転置行列に置き換える
    CompMat _basisMat = basisMat.conjugate();
    CompVec probAmpVec = new CompVec(row);
    
    for (int i = 0; i < row; i++) {
      for (int k = 0; k < length; k++) {
        Complex basis = _basisMat.copyComp(i, k);
        basis.mul(probAmp.getComp(k));
        Complex val = probAmpVec.getComp(i);
        val.add(basis);
      }
    }
        
    return probAmpVec;
  }
  
  
  /**
   * 一方の確率振幅から他方の確率振幅を得る（+/- の一方の解のみ）
   * @param oneOfProbAmp Complex // 一方の確率振幅
   * @return otherPA Complex // 他方の確率振幅
   */
  public static Complex getOtherPA(Complex oneOfProbAmp) {
    Complex conjyuPA = oneOfProbAmp.conjugate(); // 共役複素数
    conjyuPA.mul(oneOfProbAmp); // 一方の確率振幅の自乗
    Complex comp1 = new Complex(1.0, 0.0);
    comp1.sub(conjyuPA); // 1 - 一方の確率振幅の自乗
    Complex otherPA = comp1.sqrt(); // 他方の確率振幅
    return otherPA;
  }
  
  /**
   * 量子ビットを量子ゲートに適用する
   * ※量子ビットには結果が上書きされる
   * @param gate QuGate // 量子ゲート
   */
  public void apply(QuGate gate) {
    int paLength = probAmp.length(); // 確率振幅の長さ
    int gateOrder = gate.getOrder();
    
    if (paLength != gateOrder) {
      DJ.print("***** ERROR ***** "
          + "QuBit.apply(QuGate gate)" + "\n"
          + " QuBit's probability-amplitude length (" + paLength
          + ") is not equal to Gate order (" + gateOrder + ").");
      probAmp = null;
      return;
    }
    
    Complex[] _qbVec = probAmp.getArray();
    Complex[][] _gateMat = gate.getElement();
    
    Complex[] result = Complex.product(_gateMat, _qbVec);
    probAmp.copyArray(result);
  }
  
  /**
   * もつれ状態を判定する（判定結果のみ）
   * @return result boolean // 判定 false：もつれていない、true：もつれている
   */
  public boolean entangleCheck() {
    boolean judgment = false; // もつれていない
    int probAmpLength = probAmp.length(); // 確率振幅数
    int numOfQuBit = probAmpLength / 2; // 量子ビット数
    
    Complex front = probAmp.copyComp(0);
    Complex rear = probAmp.getComp(probAmpLength -1);
    front.mul(rear);
    Complex _front;
    for (int i = 1; i < numOfQuBit; i++) {
      _front = probAmp.copyComp(i);
      rear = probAmp.getComp(probAmpLength - i -1);
      _front.mul(rear);
      
      Complex def = _front.copyComplex();
      def.sub(front);
      if (def.abs() > 1.0E-15) {
        judgment = true; // もつれている
      }
    }
    
    return judgment;
  }
  
  
  /**
   * もつれ状態を判定する（判定中の詳細情報を表示する）
   * @return result boolean // 判定 false：もつれていない、true：もつれている
   */
  public boolean entangleDetails() {
    boolean judgment = false; // もつれていない
    int probAmpLength = probAmp.length(); // 確率振幅数
    int numOfQuBit = probAmpLength / 2; // 量子ビット数
    
    Complex front = probAmp.copyComp(0);
    Complex rear = probAmp.getComp(probAmpLength -1);
    front.mul(rear);
    DJ.print("$ (" + 0 + ") product = ", front);
    
    Complex _front;
    for (int i = 1; i < numOfQuBit; i++) {
      _front = probAmp.copyComp(i);
      rear = probAmp.getComp(probAmpLength - i -1);
      _front.mul(rear);
      DJ.print("$ (" + i + ") product = ", _front);
      
      Complex def = _front.copyComplex();
      def.sub(front);
      DJ.print("$ difference = ", def.abs());
      if (def.abs() > 1.0E-15) {
        judgment = true; // もつれている
      }
      DJ.print("$ judgment = ", Boolean.toString(judgment));
    }
    
    return judgment;
  }
  
  /**
   * もつれ判定結果の文字列を得る
   * @param judgment boolean // もつれ判定結果 true:もつれている
   * @return diagnosis String // 判定結果を表す文字列
   */
  public static String entangleStatus(boolean judgment) {
    String diagnosis;
    if (judgment) diagnosis = "Entangled"; // もつれている
    else          diagnosis = "Not entangle"; // もつれていない
    return diagnosis;
  }
  
  
  // 共役・内積 -------------------------------------------
  
  /** 使用されていない
   * 共役な量子ビットを得る
   * @return _conjugate CompVec // 共役な複素数ベクタ
   */
  public QuBit conjugate() {
    int length = probAmp.length();
    QuBit _conjugate = new QuBit(length);
    for (int i = 0; i < length; i++) {
      _conjugate.setProbAmp(i, probAmp.getComp(i).conjugate());
    }
    return _conjugate;
  }

  /**
   * 量子ビットの内積を求める
   * @param operand QuBit // 量子ビット
   * @return _innerProduct Complex // 内積、複素数
   */
  public Complex innerProduct(QuBit operand) {
    int length = probAmp.length();
    CompVec operandVec = operand.getProbAmp();
    int operandLength = operandVec.length();
    if (length != operandLength) {
      DJ.print("***** ERROR ***** "
          + "Complex.innerProduct(CompVec)" + "\n"
          + " vecter length are not same."
          + " target vecter length is " + length + ", "
          + " operand length is " + operandLength + ".");
      return null;
    }

    // 被演算数複素数配列を共役配列に置き換える
    CompVec _conjugate = probAmp.conjugate();
    Complex _innerProduct = new Complex(0.0, 0.0);
    for (int i = 0; i < length; i++) {
      Complex conj = _conjugate.getComp(i);
      conj.mul(operandVec.getComp(i));
      _innerProduct.add(conj);
    }
    return _innerProduct;
  }
   
  /**
   * 量子ビットの自身との内積を求める
   * @return _innerProduct Complex // 内積、複素数
   */
  public Complex innerProduct() {
    int length = probAmp.length();
    // 被演算数複素数配列を共役配列に置き換える
    CompVec _conjugate = probAmp.conjugate();
    Complex _innerProduct = new Complex(0.0, 0.0);
    for (int i = 0; i < length; i++) {
      Complex conj = _conjugate.getComp(i);
      conj.mul(probAmp.getComp(i));
      _innerProduct.add(conj);
    }
    return _innerProduct;
  }
  
  
  // 相互作用 --------------------------------------------
  
  /**
   * 確率振幅の個数から量子ビットの個数を求める
   * @param numOfProbAmp int // 確率振幅の個数
   * @return numOfQuBit int // 量子ビットの個数
   */
  public static int calcNumOfQuBit(int numOfProbAmp) {
    int numOfQuBit = (int)(Math.log(numOfProbAmp) / Math.log(2));
    return numOfQuBit;
  }
  
  /**
   * 量子ビット数から確率振幅数を求める（２のべき乗）
   * @param numOfQuBit int // 量子ビット数
   * @return numOfProbAmp int // 確率振幅数
   */
  public static int calcNumOfProbAmp(int numOfQuBit) {
    int numOfProbAmp = 1;
    for(int i = 1; i <= numOfQuBit; i++) {
      numOfProbAmp *= 2; // ２のべき乗
    }
    return numOfProbAmp;
  } 
  
  // Depreciated
  /** 
   * 相互作用した２量子ビットを得る
   * @param probAmpArray Complex[] // 量確率振幅の配列
   * @return Complex[] // 確率振幅の配列
   */
  public Complex[] interactQuBit2(Complex[] probAmpArray) {
    Complex[] probAmpArray4 = Complex.complexArray(2 * 2); // 確率振幅
    for ( int i = 0; i < 2; i++) {
      for ( int j = 0; j < 2; j++) {
        probAmpArray4[i * 2 + j].setComplex(
            probAmp.getReal(i), probAmp.getImag(i));
        probAmpArray4[i * 2 + j].mul(probAmpArray[j]);
      }
    }
    return probAmpArray4;
  }
  
  /**
   * 相互作用した２量子ビットを得る
   * @param quBit QuBit // 相互作用させる２量子ビット
   * @return quBit2 QuBit // ２量子ビット
   */
  public QuBit interactQuBit2(QuBit quBit) { 
    CompVec _probAmpVec = quBit.getProbAmp(); // 確率振幅
    int numOfQuBit = 2;
    QuBit quBit2 = new QuBit(numOfQuBit); // 2量子ビット
    CompVec probAmp4Vec = quBit2.probAmp;
    for ( int i = 0; i < 2; i++) {
      for ( int j = 0; j < 2; j++) {
        Complex comp = probAmp4Vec.getComp(i * 2 + j);
        comp.setComplex(probAmp.getReal(i), probAmp.getImag(i));
        comp.mul(_probAmpVec.getComp(j));
      }
    }
    return quBit2;
  }
  
  /** 
   * 相互作用した多量子ビットを得る
   * @param probAmpVec CompVec // 確率振幅配列
   * @return interactedquBit QuBit // 相互作用した多量子ビット
   */
  public static QuBit interactQuBit(CompVec probAmpVec) {
    Complex[] probAmpArray = probAmpVec.getArray();
    Complex[] probAmp4Array = QuBit.interactQuBit(probAmpArray);
    QuBit interactedquBit = new QuBit(probAmp4Array);
    return interactedquBit;
  }
  
  /**
   * 多量子ビットの相互作用
   * @param quBit QuBit // 相互作用させる多量子ビット
   * @return quBit2 QuBit // ２量子ビット
   */
  public QuBit interactQuBit(QuBit quBit) {
    int numOfQuBit = this.getNumOfQuBit(); // 量子ビット数
    int numOfPa = this.getProbAmpLength(); // 確率振幅の個数
    
    int numOfQuBitOp = quBit.getNumOfQuBit(); // 量子ビット数
    int numOfPaOp = quBit.getProbAmpLength(); // 確率振幅の個数
    
    int _numOfQuBit = numOfQuBit + numOfQuBitOp; // 量子ビット数
    int _numOfPa = numOfPa * numOfPaOp; // 確率振幅の個数
    
    if (_numOfQuBit != QuBit.calcNumOfQuBit(_numOfPa)) {
      DJ.print("***** ERROR ***** "
          + "QuBit.interactQuBit(QuBit quBit)" + "\n"
          + " Number of QuBit (" + _numOfQuBit
          + ") is not equal to Number of QuBit ("
          + QuBit.calcNumOfQuBit(_numOfPa));
      return null;
    }
    
    QuBit _quBit = new QuBit(_numOfQuBit); // 量子ビット
    CompVec _probAmpVec = _quBit.getProbAmp(); // 確率振幅
    
    CompVec probAmpVecOp = quBit.getProbAmp(); // 確率振幅
    
    for ( int i = 0; i < numOfPa; i++) {
      for ( int j = 0; j < numOfPaOp; j++) {
        Complex comp = _probAmpVec.getComp(i * numOfPaOp + j);
        comp.setComplex(probAmp.getReal(i), probAmp.getImag(i));
        comp.mul(probAmpVecOp.getComp(j));
      }
    }
    return _quBit;
  }
  
  /**
   * 多量子ビットの相互作用
   * @param probAmpArray Complex[] // 多量子ビットの確率振幅配列
   * @return quBitInteracted Complex[] // 
   */
  public static Complex[] interactQuBit(Complex[] probAmpArray) {
    int paLength = probAmpArray.length; // 確率振幅の配列の長さ
    if (paLength < 4) {
      DJ.print("***** ERROR ***** "
          + "QuBit.interactQuBit(Complex[] probAmpArray)" + "\n"
          + " Number of probability amplitudes (" + paLength
          + ") is less than 4.");
    }
    if (paLength%2 != 0) {
      DJ.print("***** ERROR ***** "
          + "QuBit.interactQuBit(Complex[] probAmpArray)" + "\n"
          + " Number of probability amplitudes (" + paLength
          + ") is not even number.");
      return null;
    }

    int resultSize = QuBit.calcNumOfProbAmp(paLength / 2);
    Complex[] interactedQB = new Complex[resultSize];
    interactedQB[resultSize - 2] = probAmpArray[paLength - 2];
    interactedQB[resultSize - 1] = probAmpArray[paLength - 1]; // 最後尾の確率振幅
    
    int paTempSize = 1; // 暫定確率振幅の長さ
    for(int i = 0; i < paLength - 2; i += 2) { // 確率振幅数-2の繰り返し
      paTempSize *= 2; // 暫定確率振幅の長さはべき乗
      
      int paId = paLength - 4 - i; // aLength-2-1-1-i
      int index = resultSize - paTempSize;
      
      // 前半の積
      for (int k = 0; k < paTempSize ; k++) { // 確率振幅の長さの繰り返し
        interactedQB[index - paTempSize + k] = 
            probAmpArray[paId].
                product(interactedQB[index + k]);
      }          
      // 後半の積
      for (int k = 0; k < paTempSize; k++) { // 確率振幅の長さの繰り返し
        interactedQB[index + k] = 
            probAmpArray[paId + 1].copyComplex().
                product(interactedQB[index + k]);
      }
    }
    return interactedQB;
  }
  
  
  // 使用されていない
  /** 他のメソッドと呼び出しのシグネチャが同じになるので、名称を変更
   * 多量子ビットの相互作用(確率振幅配列 ← 量子ビットのリスト)
   * @param quBits ArrayList< QuBit> // 量子ビットのリスト)
   * @return _probAmps Complex[] // 確率振幅の配列
   */  
  public static Complex[] _interactQuBit_(ArrayList<QuBit> quBits) {
    int numOfQuBit = quBits.size(); // 量子ビット数
    int numOfProbAmp = calcNumOfProbAmp(numOfQuBit); // 確率振幅の配列の長さ
    Complex[] _probAmps = new Complex[numOfProbAmp]; // 確率振幅の配列
    
    QuBit _quBitE = quBits.get(numOfQuBit - 1); // 最後尾の量子ビット
    _probAmps[PA0] = _quBitE.copyProbAmp(PA0);
    _probAmps[PA1] = _quBitE.copyProbAmp(PA1);
    
    int paLength = 1; // 確率振幅の長さ
    for(int i = numOfQuBit - 2; i >= 0; i--) { // 量子ビット数-1の繰り返し
      paLength *= 2; // 確率振幅の長さはべき乗
      for (int k = 0; k < paLength; k++) { // 確率振幅の長さの繰り返し
        QuBit _quBit = quBits.get(i); // i番目の量子ビット
        // 後半の積
        Complex _pa1 = _quBit.copyProbAmp(PA1);
        _pa1.mul(_probAmps[k]);
        _probAmps[paLength + k] = _pa1;
        // 前半の積
        Complex _pa0 = _quBit.copyProbAmp(PA0);
        _pa0.mul(_probAmps[k]);
        _probAmps[k] = _pa0;
      }
    }
    
    return _probAmps; // 確率振幅の配列
  }
  
  /**
   * 多量子ビットの相互作用(量子ビット ← 量子ビットのリスト)
   * @param quBits ArrayList< QuBit> // 量子ビットのリスト)
   * @return interactedQb QuBit // 量子ビット
   */  
  public static QuBit interactQuBit(ArrayList<QuBit> quBits) {
    int numOfQuBit = quBits.size(); // 量子ビット数
    int numOfProbAmp = calcNumOfProbAmp(numOfQuBit); // 確率振幅の配列の長さ
    QuBit interactedQb = new QuBit(QuBit.calcNumOfQuBit(numOfProbAmp));
    
    CompVec _probAmps = interactedQb.probAmp; // 確率振幅の配列
    
    QuBit _quBitE = quBits.get(numOfQuBit - 1); // 最後尾の量子ビット
    _probAmps.setComp(PA0, _quBitE.copyProbAmp(PA0));
    _probAmps.setComp(PA1, _quBitE.copyProbAmp(PA1));
    
    int paLength = 1; // 確率振幅の長さ
    for(int i = numOfQuBit - 2; i >= 0; i--) { // 量子ビット数-1の繰り返し
      paLength *= 2; // 確率振幅の長さはべき乗
      for (int k = 0; k < paLength; k++) { // 確率振幅の長さの繰り返し
        QuBit _quBit = quBits.get(i); // i番目の量子ビット
        // 後半の積
        Complex _pa1 = _quBit.copyProbAmp(PA1);
        _pa1.mul(_probAmps.getComp(k));
        _probAmps.setComp(paLength + k, _pa1);
        // 前半の積
        Complex _pa0 = _quBit.copyProbAmp(PA0);
        _pa0.mul(_probAmps.getComp(k));
        _probAmps.setComp(k, _pa0);
      }
    }
    
    return interactedQb;
  }
  
  
  // -----------------------------------------------------
  // 量子ビットを測定する（測定ゲート）
  
  // 多量子ビットの各量子ビットが標準基底に遷移する確率を得る
  // メソッドのフォーム：量子ビット数N
  
  // 答えのフォーム：N量子ビットの第k量子ビットが0, 1となる確率
  // Pk_0:[√(1/N+0i), √(1/N+0i), ..... , √(1/N+0i)]   
  // Pk_1:[√(1/N+0i), √(1/N+0i), ..... , √(1/N+0i)]   
  
  /**
   * 多量子ビットから量子ビットごとの遷移確率を収集する
   * @return probArray double[][] // 量子ビット毎の遷移確率のベクトル、２行
   */
  public double[][] getTransProb() {
    int paLength = probAmp.length(); // 確率振幅の個数
    int numOfQuBit = calcNumOfQuBit(paLength); // 量子ビットの個数    
    double probArray[][] = new double[2][numOfQuBit]; // index = 0:[1 0]', 1:[0 1]'
    
    int k = paLength / 2; // k:連続して処理する個数
    for (int i = 0; i < numOfQuBit; i++) { // i:量子ビットの桁（昇順、012...）
      double c0 = 0.0;
      double c1 = 0.0;
      int j = 0; // 確率振幅のインデックス
      do {
        for (int m = 0; m < k; m++) {
          c0 += probAmp.getComp(j).square();
          j++;
        }
        for (int m = 0; m < k; m++) {
          c1 += probAmp.getComp(j).square();
          j++;
        }
      } while(j < paLength);
      k /= 2;
      probArray[0][i] = c0;
      probArray[1][i] = c1;     
    }
    
    return probArray;
  }
  
  /**
   * 多量子ビットから量子ビットごとの遷移確率を収集する
   * ※　実行結果は上のgetTransProb()と同じ
   * @return probArray double[][] // 量子ビット毎の遷移確率のベクトル、２行
   */
  public double[][] extract() {
    int paLength = probAmp.length(); // 確率振幅の個数
    int numOfQuBit = calcNumOfQuBit(paLength); // 量子ビットの個数    
    double probArray[][] = new double[2][numOfQuBit]; // index = 0:[1 0]', 1:[0 1]'
    
    for (int i = 0; i < numOfQuBit; i++) { // i:量子ビットの桁（昇順、012...）
      int k = (int)Math.pow(2, numOfQuBit - 1 - i); // k:連続して処理する個数
      boolean flag = true;
      int count = 0;
      double c0 = 0.0;
      double c1 = 0.0;
      for (int j = 0; j < paLength; j++) {
        Complex _probAmp = probAmp.getComp(j);
        if (flag) c0 += _probAmp.square();
        else      c1 += _probAmp.square();
        count++;
        if (count >= k) {
          count = 0;
          flag = !flag;
        }
      }
      probArray[0][i] = c0;
      probArray[1][i] = c1;
    }
    
    return probArray;
  }
  
  
  // ------------------------------------------------------
  // 量子ビットの比較
  public boolean equals(QuBit qb) {
    
    if (this == qb) return true; // アドレスが同じ

    int thisLength = probAmp.length();
    int length = qb.getProbAmpLength();
    
    if (length != thisLength) return false; // 確率振幅数が違う

    Complex[] thisProbAmp = this.getProbAmpArray();
    Complex[] _probAmp = qb.getProbAmpArray();
    for (int i = 0; i < thisLength; i++) {
      if (!thisProbAmp[i].equals(_probAmp[i])) {
        return false; // 値が違う
      }
    }
    
    return true; // 値が等しい
  }  
  

  
  // ==================================================================
  
  /** CompVecクラスより移植
   * 連続する空白文字列を得る
   * @param size int // 空白文字の個数
   * @return space.toString String // 空白文字列
   */
  private static String getSpace(int size) {
    StringBuilder space = new StringBuilder(); // 表示文字列
    for (int i = 0; i < size; i++) {
      space.append(" "); // 空白を追加
    }
    return space.toString();
  }
  
  /**
   * 1.量子ビットの文字列（行表示）を得る
   * @return String // 複素数の文字列
   */
  @Override
  public String toString() {
    int length = probAmp.length();
    StringBuilder sb = new StringBuilder("[");
    for (int i = 0; i < length; i += 2) {
      sb.append(probAmp.getComp(i).toString());
      sb.append(", ");
      sb.append(probAmp.getComp(i+1).toString());
      sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 2.量子ビットの文字列（行・列選択表示）を得る
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public String toString(boolean orient) {
    int length = probAmp.length();
    StringBuilder sb;
    if (orient) sb = new StringBuilder("\n["); // 列（縦）ベクトル
    else        sb = new StringBuilder(" ["); // 行（横）ベクトル
    for (int i = 0; i < length; i += 2) {
      sb.append(probAmp.getComp(i).toString());
      sb.append(", ");
      sb.append(probAmp.getComp(i+1).toString());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /** CompVecクラスより移植
   * 3.量子ビットの文字列（行表示、枠・書式付き）を得る
   * @return sb.toString() // 複素数の文字列
   */
  public String toFormat() {
    int order = probAmp.length(); // ベクトルの次数
    int maxWide = 5; // 数値文字列の長さの最大長 (最小例：-0+0i)
    
    // 数値文字列の抽出
    String[] buf = new String[order]; // 文字列バッファ
    int[] wides = new int[order]; // 余白数
    for (int i = 0; i < order; i++) {
      // DJ._print_("i=", i);
      String text = probAmp.getComp(i).toFormat();
      
      int tempWide;
      if (probAmp.getReal(i) >= 0) { // 実部が正
        buf[i] = " " + text;
        tempWide = text.length() + 1; // 数値文字列の長さ
      }
      else {
        buf[i] = text;
        tempWide = text.length(); // 数値文字列の長さ
      }
      // DJ.print(",  text=", text);
      
      wides[i] = tempWide;
      if (tempWide > maxWide) maxWide = tempWide; // wideを更新
      // DJ.print_("tempWide=", tempWide);
      // DJ.print(",  maxWide=", maxWide);
      
    }
    // DJ._print("last maxWide=", maxWide);
    // DJ._print("text=");
    // for (int i = 0; i < order; i++) {
    //   DJ.print_(" ", i); DJ.print(" ", buf[i]);
    // }
    
    // 出力文字列の生成
    StringBuilder sb; // 表示文字列
    sb = new StringBuilder("F \n"); // 列（縦）ベクトル
    sb.append("┌").append(getSpace(maxWide)).append("┐\n");
    for (int i = 0; i < order; i++) {
      sb.append("│");
      sb.append(buf[i]); // 数値
      sb.append(getSpace(maxWide - wides[i])); // 余白
      sb.append("│");
      sb.append("\n"); // 列（縦）ベクトル
    }
    sb.append("└").append(getSpace(maxWide)).append("┘");
    return sb.toString();
  }
  
  /**
   * 4.量子ビットの文字列（行・列選択表示、書式付き）を得る
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public String toFormat(boolean orient) {
    int length = probAmp.length();
    StringBuilder sb;
    if (orient) sb = new StringBuilder("F\n["); // 列（縦）ベクトル
    else        sb = new StringBuilder("F ["); // 行（横）ベクトル
    for (int i = 0; i < length; i += 2) {
      sb.append(probAmp.getComp(i).toFormat());
      sb.append(", ");
      sb.append(probAmp.getComp(i+1).toFormat());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 5.量子ビットの（行表示、ラベル付き）文字列を得る
   * @param label String // 表示用ラベル
   * @return sb String // 複素数の文字列
   */
  public String toString(String label) {
    int length = probAmp.length();
    StringBuilder sb = new StringBuilder(label + ":[");
    for (int i = 0; i < length; i += 2) {
      sb.append(probAmp.getComp(i).toString());
      sb.append(", ");
      sb.append(probAmp.getComp(i+1).toString());
      sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 6.量子ビットの（行・列選択表示、ラベル付き）文字列を得る
   * @param label String // 表示用ラベル
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return sb String // 複素数の文字列
   */
  public String toString(String label, boolean orient) {
    int length = probAmp.length();
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":\n["); // 列（縦）ベクトル
    else        sb = new StringBuilder(label + ":["); // 行（横）ベクトル
    for (int i = 0; i < length; i += 2) {
      sb.append(probAmp.getComp(i).toString());
      sb.append(", ");
      sb.append(probAmp.getComp(i+1).toString());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 7.量子ビットの文字列（列表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @return String // 複素数の文字列
   */
  public String toFormat(String label) {
    String str = this.toFormat();
    
    return label + ":" + str;
  }
  
  /**
   * 8.量子ビットの文字列を得る（行・列選択表示、ラベル付き、書式付き）
   * @param label String // 表示用ラベル
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public String toFormat(String label, boolean orient) {
    int length = probAmp.length();
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":F\n["); // 列（縦）ベクトル
    else        sb = new StringBuilder(label + ":F ["); // 行（横）ベクトル
    for (int i = 0; i < length; i += 2) {
      sb.append(probAmp.getComp(i).toFormat());
      sb.append(", ");
      sb.append(probAmp.getComp(i+1).toFormat());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  
  // ------------------------------------------------------
  // 量子ビット（行列）の表示
  // 量子ビット行列の各要素に格納された量子ビットを列（縦）ベクトルで表示
  
  /**
   * 量子ビット行列の文字列を得る（行（縦）表示、書式付き）
   * Depreciated ※ 量子ビット配列を転置して表示する
   * @param quBitMap CompMat // 量子ビット行列
   * @return sb.toString() String // 複素数行列の文字列
   */
  public static String toFormat(CompMat quBitMap) {
    int col = quBitMap.rowNumber(); // 転置するため行と列を入れ替える
    int row = quBitMap.rowNumber();
    int maxWide[] = new int[col]; // 数値文字列の長さの最大長 (最小例：-0+0i)
    for (int j = 0; j < col; j++) maxWide[j] = 5;
    
    // 数値文字列の抽出
    String[][] buf = new String[row][col]; // 文字列バッファ
    int[][] wides = new int[row][col]; // 余白数
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        // DJ.print_("i,j=", i);  DJ.print_(", ", j);
        String text = quBitMap.getComp(i, j).toFormat();

        if (quBitMap.getReal(i, j) >= 0) { // 実部が正
          buf[i][j] = " " + text;
          wides[i][j] = text.length() + 1; // 数値文字列の長さ
        } else {
          buf[i][j] = text;
          wides[i][j] = text.length(); // 数値文字列の長さ
        }
        // DJ.print(",  text=", text);
        // DJ.print_("wides[" + i + "][" + j + "]=", wides[i][j]);

        if (wides[i][j] > maxWide[j]) {
          maxWide[j] = wides[i][j]; // wideを更新
        }
        // DJ.print(",  maxWide[" + j + "]=", maxWide[j]);
      }
    }
    
    // 書式に従って数値を配置し枠で囲む
    StringBuilder sb = new StringBuilder("F\n");
    int index;
    
    String filler = "：";
    int gap = filler.length();
    int spaces = maxWide[0];
    for (int j = 1; j < col; j++) {
      spaces = spaces + gap + 1 + maxWide[j]; 
    }
    sb.append("┌").append(getSpace(spaces)).append("┐\n");
    
    for (int i = 0; i < row; i++) {
      sb.append("│");
      for (int j = 0; j < col; j++) {
        sb.append(buf[i][j]); // 数値
        sb.append(getSpace(maxWide[j] - wides[i][j])).append(filler); // 余白
      }
      index = sb.length();
      sb.replace(index - gap, index, "│\n");
    }
    sb.append("└").append(getSpace(spaces)).append("┘");
    return sb.toString();
  }
  
  /**
   * 量子ビット行列の文字列を得る（行（縦）表示、ラベル付き、書式付き）
   * @param label String // 表示用ラベル
   * @param quBitMap CompMat // 量子ビット行列
   * @return sb.toString() String // 複素数行列の文字列
   */
  public static String toFormat(String label, CompMat quBitMap) {
    String str = QuBit.toFormat(quBitMap);
    
    return label + ":" + str;
  }

  
  // ------------------------------------------------------
  // 量子ビット（配列）の表示
  // 量子ビット配列の各要素に格納された量子ビットを列（縦）ベクトルで表示
  
  /**
   * 量子ビット配列の文字列を得る（行（縦）表示、書式付き）
   * @param quBitArray QuBit[] // 量子ビット配列
   * @return sb.toString() String // 複素数行列の文字列
   */
  public static String toFormat(QuBit[] quBitArray) {
    int col = quBitArray.length;
    int row = quBitArray[0].getProbAmpLength();
    int maxWide[] = new int[col]; // 数値文字列の長さの最大長 (最小例：-0+0i)
    for (int j = 0; j < col; j++) maxWide[j] = 5;
    
    // 量子ビット配列を転置した行列にコピー
    CompMat probAmpMat = new CompMat(row, col);
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        Complex element = quBitArray[j].copyProbAmp(i);
        probAmpMat.setComp(i, j, element);
      }
    }
    // DJ.printF("probAmpMat", probAmpMat);
    
    // 数値文字列の抽出
    String[][] buf = new String[row][col]; // 文字列バッファ
    int[][] wides = new int[row][col]; // 余白数
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        // DJ.print_("i,j=", i);  DJ.print_(", ", j);
        String text = probAmpMat.getComp(i, j).toFormat();

        if (probAmpMat.getReal(i, j) >= 0) { // 実部が正
          buf[i][j] = " " + text;
          wides[i][j] = text.length() + 1; // 数値文字列の長さ
        } else {
          buf[i][j] = text;
          wides[i][j] = text.length(); // 数値文字列の長さ
        }
        // DJ.print(",  text=", text);
        // DJ.print_("wides[" + i + "][" + j + "]=", wides[i][j]);

        if (wides[i][j] > maxWide[j]) {
          maxWide[j] = wides[i][j]; // wideを更新
        }
        // DJ.print(",  maxWide[" + j + "]=", maxWide[j]);
      }
    }
    
    // 書式に従って数値を配置し枠で囲む
    StringBuilder sb = new StringBuilder("F\n");
    int index;
    
    String filler = "￤";
    int gap = filler.length();
    int spaces = maxWide[0];
    for (int j = 1; j < col; j++) {
      spaces = spaces + gap + 1 + maxWide[j]; 
    }
    sb.append("┌").append(getSpace(spaces)).append("┐\n");
    
    for (int i = 0; i < row; i++) {
      sb.append("│");
      for (int j = 0; j < col; j++) {
        sb.append(buf[i][j]); // 数値
        sb.append(getSpace(maxWide[j] - wides[i][j])).append(filler); // 余白
      }
      index = sb.length();
      sb.replace(index - gap, index, "│\n");
    }
//    sb.append("└").append(getSpace(spaces)).append("┘\n");
    sb.append("└").append(getSpace(spaces)).append("┘");
    return sb.toString();
   }

  /**
   * 量子ビット配列の文字列を得る（行（縦）表示、ラベル付き、書式付き）
   * @param label String // 表示用ラベル
   * @param quBitArray QuBit[] // 量子ビット配列
   * @return sb.toString() String // 複素数行列の文字列
   */
  public static String toFormat(String label, QuBit[] quBitArray) {
    String str = QuBit.toFormat(quBitArray);
    
    return label + ":" + str;
  }
  
  
  // ------------------------------------------------------
  // 量子ビット（リスト）の表示
  // 全体を鍵括弧で括る
  // 量子ビットごとに確率振幅を丸括弧で括り、各確率振幅をコンマで区切る
  // 量子ビットごとに（丸括弧ごとに）コンマで区切る
  //
  // public static final boolean H_BIT = false;  // 行（横方向）表示
  // public static final boolean V_BIT = true; // 列（縦方向）表示

  /**
   * 1.多量子ビット（リスト）の文字列（行表示）を得る
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @return String // 多量子ビットの文字列
   */
  public static String toString(ArrayList<QuBit> quBitsList) {
    int order = quBitsList.size();
    StringBuilder sb = new StringBuilder(" ");
    for (int i = 0; i < order; i++) {
      sb.append("(").append(i).append("):");
      sb.append(quBitsList.get(i).toString());
      sb.append(", ");
    }
    int index = sb.length();
    sb.delete(index - 2, index);
    return sb.toString();
  }
  
  /**
   * 2.多量子ビット（リスト）の文字列（行・列選択表示）を得る
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @param orient boolean // 量子ビットの表示方向 true：列（縦）false：行（横）
   * @return String // 多量子ビットの文字列
   */
  public static String toString(ArrayList<QuBit> quBitsList, boolean orient) {
    int order = quBitsList.size();
    StringBuilder sb;
    if (orient) sb = new StringBuilder("\n");
    else        sb = new StringBuilder();
    for (int i = 0; i < order; i++) {
      sb.append("(").append(i).append("):");
      sb.append(quBitsList.get(i).toString());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.delete(index - 2, index);
    return sb.toString();
  }
  
  /**
   * 3.多量子ビット（リスト）の文字列（行表示、書式付き）を得る
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @return String // 多量子ビットの文字列
   */
  public static String toFormat(ArrayList<QuBit> quBitsList) {
    int order = quBitsList.size();
    StringBuilder sb = new StringBuilder(" ");
    for (int i = 0; i < order; i++) {
      sb.append("(").append(i).append("):");
      sb.append(quBitsList.get(i).toFormat());
      sb.append(", ");
    }
    int index = sb.length();
    sb.delete(index - 2, index);
    return sb.toString();
  }
  
  /**
   * 4.多量子ビット（リスト）の文字列（行・列選択表示、書式付き）を得る
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @param orient boolean // 量子ビットの表示方向 true：列（縦）false：行（横）
   * @return String // 多量子ビットの文字列
   */
  public static String toFormat(ArrayList<QuBit> quBitsList, boolean orient) {
    int order = quBitsList.size();
    StringBuilder sb;
    if (orient) sb = new StringBuilder("\n");
    else        sb = new StringBuilder(" ");
    for (int i = 0; i < order; i++) {
      sb.append("(").append(i).append("):");
      sb.append(quBitsList.get(i).toFormat());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.delete(index - 2, index);
    return sb.toString();
  }
  
  /**
   * 5.多量子ビット（リスト、ラベル付き）の文字列（行表示）を得る
   * @param label String // ラベル
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @return String // 多量子ビットの文字列
   */
  public static String toString(String label, ArrayList<QuBit> quBitsList) {
      int order = quBitsList.size();
      StringBuilder sb = new StringBuilder(label + " ");
      for (int i = 0; i < order; i++) {
        sb.append("(").append(i).append("):");
        sb.append(quBitsList.get(i).toString());
        sb.append(", ");
      }
      int index = sb.length();
      sb.delete(index - 2, index );
      return sb.toString();
  }
  
  /**
   * 6.多量子ビット（リスト）の文字列（行・列選択表示、ラベル付き）を得る
   * @param label String // ラベル
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @param orient boolean // 量子ビットの表示方向 true：列（縦）false：行（横）
   * @return String // 多量子ビットの文字列
   */
  public static String toString(
      String label, ArrayList<QuBit> quBitsList, boolean orient) {
    int order = quBitsList.size();
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + "\n");
    else        sb = new StringBuilder(label + " ");
    for (int i = 0; i < order; i++) {
      sb.append("(").append(i).append("):");
      sb.append(quBitsList.get(i).toString());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.delete(index - 2, index);
    return sb.toString();
  }
  
  /**
   * 7.多量子ビット（リスト）の文字列（行表示、ラベル付き、書式付き）を得る
   * @param label String // ラベル
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @return String // 多量子ビットの文字列
   */
  public static String toFormat(String label, ArrayList<QuBit> quBitsList) {
      int order = quBitsList.size();
      StringBuilder sb = new StringBuilder(label + " ");
      for (int i = 0; i < order; i++) {
        sb.append("(").append(i).append("):");
        sb.append(quBitsList.get(i).toFormat());
        sb.append(", ");
      }
      int index = sb.length();
      sb.delete(index - 2, index );
      return sb.toString();
  }
  
  /**
   * 8.多量子ビット（リスト）の文字列（行・列選択表示、ラベル付き、書式付き）を得る
   * @param label String // ラベル
   * @param quBitsList ArrayList< QuBit> // 多量子ビットのリスト
   * @param orient boolean // 量子ビットの表示方向 true：列（縦）false：行（横）
   * @return String // 多量子ビットの文字列
   */
  public static String toFormat(
      String label, ArrayList<QuBit> quBitsList, boolean orient) {
    int order = quBitsList.size();
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + "\n");
    else        sb = new StringBuilder(label + " ");
    for (int i = 0; i < order; i++) {
      sb.append("(").append(i).append("):");
      sb.append(quBitsList.get(i).toFormat());
      if (orient) sb.append(",\n"); // 列（縦）ベクトル
      else        sb.append(", "); // 行（横）ベクトル
    }
    int index = sb.length();
    sb.delete(index - 2, index);
    return sb.toString();
  }
  
} // QuBit Class

// EOF


