/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2021.4.4
 *  Copyright: 2020, 2021
 */
package task;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Random;
import java.util.StringTokenizer;

import view.SomViewer;
import util.DJ;
import util.LogEditor;
import util.TimeStamp;
import view.SomViewerLauncher;

/**
 * <p> 表　題: Class: SomLinear</p>
 * <p> 説　明: 一次元自己組織化マップ</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2007-2021</p>
 * <p> 作成日: 2021.06.22</p>
 */
public class SomLinear extends Task {

  // パラメータ選択肢
  // 学習方法
  public static final int LEARN_ONLINE = 0; // 逐次学習（線形関数）
  public static final int LEARN_BATCH = 1; // 一括学習（指数関数）
  // 初期化タイプ
  public static final int INITIAL_SHUFFLE = 0; // 撹拌による初期化
  public static final int INITIAL_LOAD = 1; // ファイル読込による初期化
  // 攪拌方法
  public static final int RANDOM_SHUFFLE = 0; // 乱数撹拌
  public static final int LINEAR_SHUFFLE = 1; // 線形撹拌
  // 乱数初期値
  public static final long DEFAULT_RANDOM_SEED = 314159265358979L;
   // 計量値
  public static final int QUANTITY_OFF = 0; // 計算しない
  public static final int QUANTITY_LOW = 1; // 計算する、ログ出力少
  public static final int QUANTITY_MID = 2; // 計算する、ログ出力中
  public static final int QUANTITY_ALL = 3; // 計算する、全てログ出力
  
  // 学習・表示パラメータ
  int epoch = 400; // エポックの回数
  int interval = 1; //50; // 経過表示間隔
  
  // SOMパラメータ
  public double learningRate = 0.1; // 学習係数 0.05～0.2
  public double learningRadius = 10.0; // 初期学習半径 dimOfCodeX / 2
  public double lastRadius = 1.0; // 最終学習半径
  
  // 因子ベクタ（学習用入力データ）
  public String factorFileName = "FactorFile";// 入力データのディフォルト・ファイル名
  public int numOfTrial = 0; // 試行因子ベクタ数（デフォルト値）

  // 参照ベクタ（出力層のユニット数）
  public int dimOfCodeX = 20; // Ｘ軸格子数
  public int dimOfCodeY = 20; // Ｙ軸格子数
  
  // Ｕマップ（隣接する参照ベクタの差分を等高線で表示）
  public int uMapXSize = 400; // ＵマップＸ軸表示幅[ドット]
  public int uMapYSize = 400; // ＵマップＹ軸表示幅[ドット]
  public int numOfContour = 10; // Ｕマップ等高線数（デフォルト値）
  public int baseContour = 10; // Ｕマップ基準等高線（デフォルト値）
  public double borderContour = 0.2; // Ｕマップ境界等高線

  // 評価、誤差
  public int evaluationType = 0; // 評価方法（デフォルト値）
  public double quantumCoeff = 1.0; // 量子化誤差表示係数
  public double disorderCoef = 1.0; // 擾乱量表示係数
  
  // 変数
  public int learningType = LEARN_ONLINE; // 学習方法（デフォルト値）
  public int initialType = INITIAL_SHUFFLE; // 初期化タイプ（デフォルト値）
  public int shuffleMethod = RANDOM_SHUFFLE; // 撹拌方法（デフォルト値）
  public int quantityFlag = QUANTITY_OFF; // 計量値の計算（デフォルト値）
  private double quantity = 0.0; // 計量値
  private double lowestQuantity = Double.MAX_VALUE; // 最小計量値
  private int lowestQuantityId = 0; // 最小計量値のエポック回数
  private final long randomSeed = DEFAULT_RANDOM_SEED; // 乱数初期値（デフォルト値）
  private int index; // 入力データのインデックス
  private long numberOfLearning = 500; // 総学習回数（デフォルト値）
  
  // 因子ベクタ
  private String titleLine = "#SOM Facter File : " ; // タイトル・コメント行
  private String[] factorNames; // 因子名
  private int dimOfFactor; // 因子ベクタの次数
  private int numOfFactor; // 因子ベクタ数
  private double[][] factorVectors; // 因子ベクタ配列[numOfFactor][dimOfFactor]
  private String[] factorLabels; // 因子ラベル配列[numOfFactor]
  private double[] minElement; // 因子ベクタ配列の各因子ごとの最小値
  private double[] maxElement; // 因子ベクタ配列の各因子ごとの最大値
  private double[][] trialVectors; // 試行用因子ベクタ配列[trialNum][dimOfFactor]
  private String[] trialLabels; // 試行用因子ラベル配列[trialNum]
  private ArrayList<Integer> indexList; // 因子ベクタ用ランダム・インデックス・リスト
  
  // 参照ベクタ
  private int numOfCodeVector; // 参照ベクタの個数[dimOfCodeX*dimOfCodeY]
  private double[][] codeVectorArray; // 参照ベクタ配列[numOfCodeVector][dimOfFactor]
  
  // Ｕマップ
  private int dimOfUX; // ＵマップのＸ軸格子数[dimOfCodeX*2+1]
  private int dimOfUY; // ＵマップのＸ軸格子数[dimOfCodeY*2+1]
  private double[][] uMatrix; // Ｕマップ（Ｕ行列）[dimOfUX][dimOfUY]
  
  // ラベル・マップ
  private  String[][] labelMap; // ラベル・マップ（ラベル行列）[Ｘ軸格子数＊Ｙ軸格子数]
  private  String[][] lowestLabelMap; // 最小計量値時のラベル・マップ
  
  // 誤差
  private double quantumError; // 量子化誤差
  private double disorderError; // Ｕマップの擾乱量
  private int oldNumOfUpper; // 前回の閾値以上の格子点の数（擾乱量の算出）
  
  // バッチ補助変数（参照ベクタの重みの算出に用いる）
  private double[][] wN; // 重みの分子
  private double[][] wD; // 重みの分母
    
  // オブジェクト
  private Random randamGenerator; // 乱数発生器
  private SomViewerLauncher somViewerLauncher; // ＳＯＭビューワ・ランチャ
  private SomViewer somViewer; // SOMマップ・ビューワ  

 /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    //#    // パターン・ビューワ・ランチャ
    //    patternViewerFlag = true; // パターン・ビューワを表示する
    //    patternData0 = new double[3][dataNum]; // 入力x、出力y、正解t（仮の書式）
    //    patternViewerLauncher = DJ.pattern(
    //        PatternViewer.PATTERN_SPRAY, patternData0, "SelfOrganizingMap");
    
    // グラフ・ビューワ・ランチャ
    graphShift = 1; // 0; // -2; // グラフの縦軸を移動
    
    if(quantityFlag == QUANTITY_OFF) { // 計量値：計算しない
      graphData = new double[2];
      dataName = new String[2]; // データ名
      dataName[0] = "uMapDisorder"; // Ｕマップの擾乱量
      dataName[1] = "quantumError"; // 量子化誤差
    }
    else {
      graphData = new double[3];
      dataName = new String[3]; // データ名
      dataName[0] = "uMapDisorder"; // Ｕマップの擾乱量
      dataName[1] = "quantumError"; // 量子化誤差
      dataName[2] = "Quantity"; // 計量値
    }
    
    graphViewerLauncher = DJ.graph(epoch, interval, dataName, graphData);
    
    // SOMビューワ・ランチャ
    somViewerLauncher = DJ.launchSomViewer(
            SomViewer.SOM_GENERAL, "SelfOrganizingMap");
    
    somLinear(); // タスク本体の呼び出し
  }
  
  /**
   * 自己組織化マップ
   */
  public void somLinear() {
    DJ._print("SomLinear.somLinear() ==================");
    
    DJ._print(" ##### 入力データの読み込み #####");
    DJ.print(" 入力データのファイル名：", factorFileName);
    if (!loadFactor(factorFileName)) { // 入力データ・ファイルの読込
      DJ.print("***** ERROR ***** SomLinear.somLinear()"
               + " Failed to load Factor Data file.");
      return;
    }
    
    DJ._print(" ##### SOMの準備 #####");
    numOfCodeVector = dimOfCodeX * dimOfCodeY; // 参照ベクタの個数[Ｘ軸格子数＊Ｙ軸格子数]
    codeVectorArray = new double[numOfCodeVector][dimOfFactor]; // 参照ベクタ配列[参照ベクタの個数][因子ベクタの次数]

    // バッチ補助変数（参照ベクタの重みの算出に用いる）
    wN = new double[numOfCodeVector][dimOfFactor]; // 重みの分子
    wD = new double[numOfCodeVector][dimOfFactor]; // 重みの分母
    dimOfUX = dimOfCodeX * 2; // ＵマップのＸ軸格子数
    dimOfUY = dimOfCodeY * 2; // ＵマップのＸ軸格子数
    
    uMatrix = new double[dimOfUX][dimOfUY];  // Ｕマップ（Ｕ行列）
    labelMap = new String[dimOfCodeX][dimOfCodeY]; // ラベル・マップ（ラベル行列）
    lowestLabelMap = new String[dimOfCodeX][dimOfCodeY]; // 最小計量値時のラベル・マップ
    // DJ._print("・総学習回数　＝　エポック回数　×　エポック毎の学習データ数");
   numberOfLearning = epoch * numOfFactor; // 総学習回数
    
    DJ._print("・各要因ごとの最小値と最大値を求める");
    minElement = new double[dimOfFactor]; // ベクタ配列の各要素ごとの最小値
    maxElement = new double[dimOfFactor]; // ベクタ配列の各要素ごとの最大値
    getMinMaxOfVector();   
    DJ.print("Minimum of Factor", minElement); // 各因子の最小値
    DJ.print("Maximum of Factor", maxElement); // 各因子の最大値
    
    DJ._print("・参照ベクタを生成"); // リセット
    if (initialType == INITIAL_SHUFFLE) { // 撹拌による初期化
      switch (shuffleMethod) {
        default:
        case RANDOM_SHUFFLE: // 乱数で初期化
          randamGenerator = new Random(randomSeed); // 乱数発生器
          for (int id = 0; id < numOfCodeVector; id++) {
            codeVectorArray[id] = getRandomShuffledCode(minElement, maxElement);
          }
          break;
        case LINEAR_SHUFFLE: // 線形撹拌（未定義）
          break;
      }
    } // INITIAL_SHUFFLE
//    else { // ファイル読込による初期化 INITIAL_LOAD
//      // 未実装
//    }
    
    // DJ._print("GraphViewerへの参照を得る");
    graphViewer = graphViewerLauncher.getGraphViewer();
    if (graphViewer != null)
      graphViewer.shiftGraphAxis(graphShift); // グラフの縦軸をシフトする   
    
    // DJ._print("SomViewerへの参照を得る");
    somViewer = somViewerLauncher.getSomViewer();
    if (somViewer == null) {
      DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
          + " SomViewer is null.");
      return;
    }

    // SOMビューワにコメントを設定する
    somViewer.appendComment(titleLine);
    // SOMビューワにプロパティを設定する
    somViewer.setProperty(numOfTrial, evaluationType,
          numOfContour, baseContour,  borderContour, 
          uMapXSize, uMapYSize);
    // SOMビューワにパラメータを設定する
    somViewer.setParameter(dimOfFactor, dimOfCodeX, dimOfCodeY,
            factorNames, numOfFactor, factorVectors, factorLabels);
    // SOMへの参照をあらかじめSomViewerへ渡しておく
    somViewer.setMap(codeVectorArray, uMatrix, labelMap);
    // SOMビューワを表示する
    somViewer.setVisible(true);
    
    oldNumOfUpper = 0; // 前回の閾値以上の格子点の数
    quantumError = 0.0; // 量子化誤差
    quantity = 0.0; // 計量値
    lowestQuantity = Double.MAX_VALUE; // 最小計量値
    lowestQuantityId = 0; // 最小計量値のエポック回数

    double disorderShift = dimOfFactor * numOfCodeVector / disorderCoef; // 擾乱量のスケーリング係数
    double quantumShift = dimOfFactor / quantumCoeff * 100.0; // 量子化誤差のスケーリング係数
    

    DJ._print("・パラメータ");
    DJ.print_("　学習方法：learningType=", learningType);
    DJ.print(",  0:逐次学習(線形関数), 1:一括学習(線形関数), 2:一括学習(指数関数)");
    
    DJ.print_("　エポック回数：epoch=", epoch);
    DJ.print(", 表示間隔：interval=", interval);
    
    DJ.print("　エポック毎の学習データ数（因子ベクタ数）：numOfFactor=", numOfFactor);
    
    DJ.print_("　学習係数：learningRate=", learningRate);
    DJ.print(", 学習半径：learningRadius=", learningRadius);
    
    DJ.print_("　評価方法：evaluationType=", evaluationType);
    DJ.print(", 量子化誤差表示係数：quantumCoeff=", quantumCoeff);
    
    DJ._print("・入力データ");
    DJ.print("　入力データ・ファイル : ", titleLine); // ファイルのタイトル行
    DJ.print_("　因子ベクタ次数：dimOfFactor=", dimOfFactor);
    DJ.print("　因子ベクタ数：numOfFactor=", numOfFactor);
    DJ.print("　因子名リスト：factorNames=", DJ.stringArrayToString(factorNames));
    DJ.print("　因子ベクタ配列：factorVectors", factorVectors);
    DJ.print("　試行用因子ベクタ数：numOfTrial=", numOfTrial);
    DJ.print("　試行用因子ベクタ配列：trialVectors", trialVectors);
    DJ.print("　因子ベクタのラベル：factorLabels=", DJ.stringArrayToString(factorLabels));
    DJ.print("　試行用因子ベクタのラベル：trialLabels=", DJ.stringArrayToString(trialLabels));
    
    DJ._print("・参照ベクタ");
    DJ.print_("　Ｘ軸格子数：dimOfCodeX=", dimOfCodeX);
    DJ.print_(", Ｙ軸格子数：dimOfCodeY=", dimOfCodeY);
    DJ.print(", 参照ベクタ数：numOfCodeVector=", numOfCodeVector);
    
    DJ._print("・Ｕマップ");
    DJ.print_("　Ｘ軸格子数：dimOfUX=", dimOfUX);
    DJ.print_(", Ｙ軸格子数：dimOfUY=", dimOfUY);
    DJ.print_(", Ｘ軸表示幅：uMapXSize=", uMapXSize);
    DJ.print(", Ｙ軸表示幅：uMapYSize=", uMapYSize);
    
    DJ.print_("　等高線数：numOfContour=", numOfContour);
    DJ.print_(", 基準等高線：baseContour=", baseContour);
    DJ.print(", 境界等高線：borderContour=", borderContour);

    
    DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1)
              | (i == epoch); //経過表示フラグ
      
      // ニューラルネット(参照ベクタ)の学習
      switch (learningType) {
        default:
        case LEARN_ONLINE: online(i); // 逐次学習（線形関数）
          break;
        case LEARN_BATCH: batch(i); // 一括学習（指数関数）
          break;
      }

      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

      // DJ._print(" ##### １エポックの実行結果 #####");
      // マップを表示する
      if (intervalFlag) { // 経過表示インターバル
        // Ｕマップ求める。
        calculateUMatrix(); // Ｕマップ（Ｕ行列）       
        // DJ.print("U Matrix", uMatrix); // Ｕマップ
        // ラベル・マップを更新する
        makeLabelMap();
        // 試行因子ベクタのラベルを追加する
        appendLabelMap();

        int iCount = i * numOfFactor; // 現在までの学習回数
        somViewer.updateMap(iCount); // マップを更新する
      } // interval
      
      // DJ._print("　Ｕマップの擾乱量を保存する"); // 擾乱量はエポック毎に算出
      disorderError = getUmapDisorder(uMatrix) / disorderShift; // スケーリング
      // DJ._print("　量子化誤差を保存する"); // 量子化誤差はエポック毎に算出
      double qErr = quantumError / quantumShift; // 量子化誤差のスケーリング
      quantumError = 0.0; // 量子化誤差
      
      // グラフを表示する
      if (intervalFlag) { // 経過表示インターバル
        // グラフ表示用データを代入
        graphData[0] = disorderError; // Ｕマップの擾乱量
        graphData[1] = qErr; // 量子化誤差
        
        if(quantityFlag > QUANTITY_OFF) { // 計量値を計算する
          calculateQuantity(i); // 計量値を求める
          graphData[2] = quantity * 0.002; // 計量値（スケーリングして渡す）
        }
        
        updateGraph(i); // グラフを更新する
        
        //# DJ.print_(" lapTime = ", lapTime); DJ.print("[msec]");
        
        // DJ._print("・学習済み参照ベクタ");
        // DJ.print(" codeVectorArray", codeVectorArray);
        DJ.print_("epoch=" + i + " ");
        
        if(quantityFlag > QUANTITY_MID) { // 計量値を計算する
          for (int k = 0; k < 5; k++) {
            DJ.print_("U" + k + "=[" + String.format("%.3f", codeVectorArray[k][0])
                  + " " + String.format("%.3f", codeVectorArray[k][1]) + "] ");
          }
        }
        DJ.print("");
        
        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            wait(SLEEP_TIME); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          }
          catch (InterruptedException e) {
            DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
               + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval

      // 実行処理の中断
      if (abortFlag) {
        DJ._print("##### Abort action requested #####");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch ---------------------------------------");
    }
    DJ._print(" End of all epoch ---------------");
    
    DJ._print("・因子名");
    DJ.print(" factorNames=", DJ.stringArrayToString(factorNames));
    
    DJ._print("・学習済み参照ベクタ");
    DJ.print(" codeVectorArray", codeVectorArray);
    
//#
//    DJ._print("・ラベルマップ[Ｘ軸格子数＊Ｙ軸格子数]");
//    DJ.print(" labelMap[" + labelMap.length + "] = ");
//    for (int ix = 0; ix < labelMap.length; ix++) 
//        DJ.print(" labelMap = ", DJ.stringArrayToString(labelMap[ix]));
    
    DJ._print(" Last epoch = ", epoch);
    
    DJ.print("　最終誤差と計算量");
    if (quantityFlag > QUANTITY_OFF) { // 計量値を計算する
      // DJ._print("・最小計量値時のラベルマップ[Ｘ軸格子数＊Ｙ軸格子数]");
      DJ.print(" lowestLabelMap(" + lowestQuantityId + ")：");
      for (int ix = 0; ix < lowestLabelMap.length; ix++) {
        DJ.print(" labelMap = ", DJ.stringArrayToString(lowestLabelMap[ix]));
      }

      DJ.print_(dataName[2] + " = ", quantity); // 計量値
      DJ.print(", lowestQuantity(" + lowestQuantityId // 最小計算値のエポック回数
              + ") = ", lowestQuantity); // 計算値の最小値
    }
    if (graphViewer != null) {
      DJ.print_(dataName[0] + " = ", graphData[0]); // Ｕマップの擾乱量
      DJ.print(", " + dataName[1] + " = ", graphData[1]); // 量子化誤差
    }
    
    // 実行時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    
    DJ.print_("・タスク終了日時：", TimeStamp.getTimeFormated());
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print(", タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");

  } // somLinear()
 
  // ---------------------------------------------------------------------------
  /**
   * 乱数で撹拌された参照ベクタを生成する。
   * @param min double[] // 各因子の最小値の配列
   * @param max double[] // 各因子の最大値の配列
   * @return aCodeVector double[] // 参照ベクタ
   */
  double[] getRandomShuffledCode(double[] min, double[] max) {
    double[] aCodeVector = new double[dimOfFactor]; // 因子ベクタ
    for  (int i = 0; i < dimOfFactor; i++) {
      aCodeVector[i] = (max[i] - min[i]) * randamGenerator.nextDouble() + min[i];
    }
    return aCodeVector;
  }
  
  /**
   * 入力データの読み込み
   * （１）String titleLine:入力データ・ファイルのタイトルとコメント
   * （２）String[] factorName:因子名配列
   * （３）double[][] factorVectors:因子ベクタ配列
   * （４）String[] dataLabel:因子ベクタ・ラベル配列
   * （５）double[][] trialVectors:試行用因子ベクタ配列
   * （６）String[] trialLabels:試行用因子ベクタ・ラベル配列
   * @param fileName String // 入力データ・ファイル名
   * @return resultFlag boolean // 読み込みの成否
   */
  private boolean loadFactor(String fileName) {
    ArrayList<String> stringList = LogEditor.getText(fileName);
    if (stringList == null) {
      DJ.print("***** ERROR ***** SomLinear.loadFactor()"
               + " Failed to load Factor-Vector file.");
      return false;
    }
    
    DJ.print(" （１）String comment:入力データ・ファイルのタイトルとコメント");
    titleLine = stringList.get(0);
    // DJ.print("Title and comment of Factor Files : ", titleLine); // タイトルとコメント
    
    DJ.print(" （２）String[] factorName:因子名配列");
    String dataNameLine = stringList.get(1); // 因子名行
    // DJ.print("Facter Name Line : ", dataNameLine); // 因子名行
    
    StringTokenizer dataNameToken = new StringTokenizer(dataNameLine, ",");
    ArrayList<String> factorNameList = new ArrayList<>(); // 因子名リスト
    while (dataNameToken.hasMoreTokens()) {
      factorNameList.add(dataNameToken.nextToken()); // 因子名を追加
    }
    // DJ.printList("Name of Factors : ", factorNameList); // 因子名リスト
    
    dimOfFactor = factorNameList.size(); // 因子ベクタの次数
    if (dimOfFactor < 1) {
      DJ.print("***** ERROR ***** SomLinear.loadFactor()"
               + " Dimension of factorNameList < 1.");
      return false;
    }
    factorNames = new String[dimOfFactor];
    for (int i = 0; i < dimOfFactor; i++) {
      factorNames[i] = factorNameList.get(i); // 因子名の格納
    }
    
    DJ.print(" （３）double[][] factorVectors:因子ベクタ配列");
    DJ.print(" （４）String[] factorLabels:因子ベクタ・ラベル配列");
    //  因子ベクタ数をチェック
    int numOfLines = stringList.size(); // 行数
    numOfFactor = numOfLines - 2 - numOfTrial; // 因子ベクタ数
    if (numOfFactor < 1) { // 因子ベクタ数
      DJ.print("***** ERROR ***** SomLinear.loadFactor()"
               + " Number of Facter Vector is less than 1.");
      return false;
    }
    factorVectors = new double[numOfFactor][dimOfFactor]; // 因子ベクタ配列[因子ベクタ数][因子数]
    factorLabels = new String[numOfFactor] ; // 因子ラベル配列［ベクタ数］
    indexList = DJ.permutationRandom(numOfFactor); // 因子ベクタ用ランダム・インデックス・リスト
    
    DJ.print(" （５）double[][] trialVectors:試行用因子ベクタ配列");
    DJ.print(" （６）String[] trialLabels:試行用因子ベクタ・ラベル配列");
    if (numOfTrial < 1) {
      trialVectors = new double[1][dimOfFactor]; // 試行用因子ベクタ配列[試行ベクタ数][因子数]
      trialLabels = new String[1] ; // 試行用因子ラベル配列［試行ベクタ数］
    }
    else {
      trialVectors = new double[numOfTrial][dimOfFactor]; // 試行用因子ベクタ配列[試行ベクタ数][因子数]
      trialLabels = new String[numOfTrial] ; // 試行用因子ラベル配列［試行ベクタ数］
    }
    
    int rowNum = numOfLines - 2; // 因子ベクタ数の総数（残りの行数）
    for (int j = 0; j < rowNum; j++) { // 因子ベクタの取出し
      String dataLine = stringList.get(j + 2);
      StringTokenizer dataToken = new StringTokenizer(dataLine, ",");
      
//$
//      //#     int randumIndex = indexList.get(j); // ランダムにシャッフルしたインデックス
//      int randumIndex = j; // ランダムにしないインデックス
      
      // 因子の取出し
      for (int i = 0; i < dimOfFactor; i++) {
        if (dataToken.hasMoreTokens()) {
          double value = Double.parseDouble(dataToken.nextToken().trim());
          if (j < numOfFactor) factorVectors[j][i] = value;
          else           trialVectors[j - numOfFactor][i] = value;
        }
        else { // データが無い場合はゼロを代入
          if (j < numOfFactor) factorVectors[j][i] = 0.0;
          else           trialVectors[j - numOfFactor][i] = 0.0;
        }
      } // for (i < dimOfFactor)[因子数]
      
      // ラベルの取出し
      if (dataToken.hasMoreTokens()) { // ラベルが有るか。
        String label = dataToken.nextToken();
        if (j < numOfFactor) factorLabels[j] = label;
        else           trialLabels[j - numOfFactor] = label;
      }
      else { // ラベルが無い場合はnullを代入
        if (j < numOfFactor) factorLabels[j] = "";
        else           trialLabels[j - numOfFactor] = "";
      }
    } // for (j < rowNum)
    
    return true;
  }
  
  /**
   * 因子ベクタの各要素ごとの最小値と最大値を求める
   */
  private void getMinMaxOfVector() {
    // 初期値を設定
    for (int i = 0; i < dimOfFactor; i++) {
      minElement[i] = +Double.MAX_VALUE; // 正の最大値で初期化
      maxElement[i] = -Double.MAX_VALUE; // 負の最大値で初期化
    }
    // 最小値と最大値を求める
    for (int j = 0; j < numOfFactor; j++) {
      for (int i = 0; i < dimOfFactor; i++) {
        if (factorVectors[j][i] < minElement[i]) minElement[i] = factorVectors[j][i] ;
        if (factorVectors[j][i] > maxElement[i]) maxElement[i] = factorVectors[j][i] ;
      }
    }
  }

  /**
   * 最近似ユニット（BMU:Best Match Unit）のインデックスを得る
   * @param factorVector double[] // 因子ベクタ（入力）
   * @return bestMatchIndex int // 最近似ユニットのインデックス
   */
  public int findWinner(double[] factorVector) {
    double delta; // ベクタの要素ごとの差分
    double aNorm; // ノルム（差分の平方和）
    double leastNorm = Double.MAX_VALUE; // ノルムの最少値
    int bestMatchIndex = -1; // 最近似ユニットのインデックス
    
    // 全ての参照ベクタに対して因子ベクタとのノルムを求め、比較する。
    for (int i = 0; i < numOfCodeVector; i++) { // 参照ベクタの個数
      aNorm = 0.0; // ノルム（差分の平方和）
      for (int j = 0; j < dimOfFactor; j++) { // 因子ベクタの次元
        // 参照ベクタと因子ベクタのノルムを計算する。
        delta = codeVectorArray[i][j] - factorVector[j]; // 因子ごとの差分
        aNorm = aNorm + delta * delta; // 差分の平方和
        if (aNorm > leastNorm) break; // 平方和が大きければループから抜ける。
      } // j
      if (aNorm < leastNorm) { // 求めたノルムがノルムの最少値よりも小さい
        leastNorm = aNorm; // ノルムを更新
        bestMatchIndex = i; // 最近似ユニットのインデックスを変更
      }
    } // i
    quantumError = quantumError + Math.sqrt(leastNorm); // 量子化誤差を更新
    
    return bestMatchIndex;
  }
  
  /**
   * 近傍の協調処理を行う（逐次学習）
   * @param factorVector double[] // 因子ベクタ（入力）
   * @param bestMatchIndex int // 最近似ユニットのインデックス
   * @param iCount int // 現在までに実行したエポック回数
   */
  private void adaptNeighbour(
          double[] factorVector, int bestMatchIndex, int iCount) {
    double remainder = 1.0 - ((double)iCount / (double)numberOfLearning); // 残実行回数比率
    // 学習半径は線形的に R→1 へ収束する。
    double radius = 1.0 + (learningRadius - 1.0) * remainder; // 実効学習半径
    // 学習係数
    // 最近似ユニットのインデックス
    int bestX = bestMatchIndex % dimOfCodeX;
    int bestY = bestMatchIndex / dimOfCodeX;
    // 最近似ユニットとの距離が学習半径内のグリッドを求め、
    for (int id = 0; id < numOfCodeVector; id++) { // 参照ベクタの個数
      int tempX = id % dimOfCodeX;
      int tempY = id / dimOfCodeX;
      // 実効学習半径以内ならば協調処理を行う
      double distance = getGridDistance(bestX, bestY, tempX, tempY);
      if (distance <= radius) {
        double rate = learningRate * (1.0 - distance / radius) * remainder; // 実効学習係数
	    adaptCode(id, factorVector, rate); // 参照ベクタ近傍関数
      }
    }
  }

  // 参照ベクタ近傍関数
  // 参照ベクタcodeVectorの値を因子ベクタfactorVectorの値に近づける。
  private void adaptCode(int id, double[] factorVector, double rate) {
    double[] codeVector = codeVectorArray[id]; // 参照ベクタ
    for (int i = 0; i < dimOfFactor; i++) { // 因子ベクタの次数
      codeVectorArray[id][i] = codeVector[i]
              + rate * (factorVector[i] - codeVector[i]);
    }
  }
  
  /**
   * 参照ベクタを逐次学習（線形関数）
   * @param iEpoch int // 現在のエポック回数
   */
  private void online(int iEpoch) {
    
    Collections.shuffle(indexList); // 学習用データのインデックスをシャッフル
    // 全データを学習
    for (int j = 0; j < numOfFactor; j++) { // 学習データ数（因子ベクタ数）
      // DJ._print(" Learning loop started. ------------------------------");
      // DJ.print_(" i=" + iEpoch + ", j=" + j);

      //$ index = j; // 入力データ用の非ランダム・インデックス
      index = indexList.get(j); // ランダム・インデックス
      // DJ.print(", index = " + index );

      // DJ._print(" 最近似参照ベクタを見つける");
      int bestMatchIndex = findWinner(factorVectors[index]);

      // 近傍の協調処理を行う。
      int currentCount = iEpoch * numOfFactor + j; // 現在までの学習回数
      adaptNeighbour(factorVectors[index], bestMatchIndex, currentCount);
      
      // SOMの学習を解析するためにマップを表示する
      if(quantityFlag >= QUANTITY_ALL) { // 計量値を計算, LOG出力中
        // Ｕマップ求める。
        calculateUMatrix(); // Ｕマップ（Ｕ行列）       
        // ラベル・マップを更新する
        makeLabelMap();
        // 試行因子ベクタのラベルを追加する
        appendLabelMap();
        somViewer.updateMap(currentCount); // マップを更新する
      }
      
      //# DJ.print_(" lapTime = ", lapTime); DJ.print("[msec]");
      if(quantityFlag  >= QUANTITY_ALL) { // 計量値を計算, LOG出力中
        // DJ._print(" BNU, 因子ベクタ");
        DJ._print_(" Number of trial = " + currentCount);
        DJ.print__(", Factor(" + j + ")", factorVectors[index]);
        DJ.print(", BMU index=", bestMatchIndex);
      }
      
      if(quantityFlag >= QUANTITY_ALL) { // 計量値を計算する, LOG出力少
        calculateQuantity(currentCount); // 計量値を求める
        DJ.print_(" quantity=", quantity);
        DJ.print(", lowestQuantity(" // 最小計量値のエポック回数
             + lowestQuantityId + ")", lowestQuantity); // 計算値の最小値
      }
      
      // DJ._print(" 参照ベクタ");
      if(quantityFlag >= QUANTITY_ALL) { // 計量値を計算, 全てログ出力
        DJ.print(" 参照ベクタ", codeVectorArray);
      }
      
      if(quantityFlag >= QUANTITY_ALL) { // 計量値を計算, LOG出力中
        DJ.print(" ラベルマップ");
        for (int ix = 0; ix < labelMap.length; ix++) 
            DJ.print(" " + ix + ":  ", DJ.stringArrayToString(labelMap[ix]));
      }
      
//      if(quantityFlag >= QUANTITY_LOW) // 計量値を計算する, LOG出力少
//          DJ.print_("|");
      
//      DJ.print_(", " + dataName[0] + " = ", graphData[0]);
//      if (quantityFlag == 0) { // 計量値, 0:計算しない, 1:計算する
//        DJ.print(", " + dataName[1] + " = ", graphData[1]);
//      } else { // 計量値, 1:計算する
//        DJ.print_(", " + dataName[1] + " = ", graphData[1]);
//        DJ.print(", " + dataName[2] + " = ", graphData[2]);
//      }

      // スレッドの休止（実行速度の調整および経過表示のため）
      synchronized (this) {
        try {
          // DJ.print("Ｅnter to wait(sleepTime)");
          wait(1); // タイムアウト付きで待機状態
          // DJ.print("Resume from wait(sleepTime)");
          if (pauseFlag) {
            wait(); // 休止状態
          }
        } catch (InterruptedException e) {
          DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                  + " Exception occur in wait(sleepTime):" + e.toString());
        }
      } // synchronized()
        
      // 実行処理の中断
      if (abortFlag) {
        if(quantityFlag >= QUANTITY_LOW) // 計量値を計算, LOG出力中
            DJ.print(" ;");
        DJ._print("##### Abort action requested #####");
        epoch = iEpoch; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
        return;
      }
      
      // DJ._print(" End of one data -----------------");
    } // 全データを学習
    // DJ._print(" End of one epoch ---------------");

//    if(quantityFlag >= QUANTITY_LOW) { // 計量値を計算, LOG出力中
    if(quantityFlag >= QUANTITY_ALL) { // 計量値を計算, LOG出力中
      DJ.print_(" ;"); // =======================================================
    }
  }

  /**
   * 参照ベクタの学習を一括学習（指数関数）
   * @param iEpoch int // 現在のエポック回数
   */
  private void batch(int iEpoch) {
    int bestMatchIndex; // 最近似ユニットのインデックス
    // バッチ補助変数をゼロ・クリア
    for (int id = 0; id < numOfCodeVector; id++) { // 参照ベクタの個数
      for (int k = 0; k < dimOfFactor; k++) { // 因子ベクタの次数
            wN[id][k] = 0.0; wD[id][k] = 0.0;
      }
    }
    
    double tau = 2.5; // 時定数
    double et = learningRadius * Math.exp(-tau * (double)iEpoch / (double)epoch); // 仮学習半径
    double radius = Math.max(et, lastRadius); // 実効学習半径
   
    for (int j = 0; j < numOfFactor; j++) { // 因子ベクタ数
      bestMatchIndex = findWinner(factorVectors[j]); // 最近似参照ベクタ
      // 全ての参照ベクタでBMUとの距離、近傍関数、重みの分子と分母を求める
      for (int id = 0; id < numOfCodeVector; id++) { // 参照ベクタの個数
        double sr = 2.0 * radius * radius;
        double norm = getGridNorm(bestMatchIndex, id);
        double nFunc = Math.exp(-norm / sr);
        for (int k = 0; k < dimOfFactor; k++) { // 因子ベクタの次数
          wN[id][k] = wN[id][k] + nFunc * factorVectors[j][k];
          wD[id][k] = wD[id][k] + nFunc;
        }
      } // 参照ベクタ
    } // 因子ベクタ数
    
    // 参照ベクタの重みを更新
    for (int id = 0; id < numOfCodeVector; id++) { // 参照ベクタの個数
      for (int k = 0; k < dimOfFactor; k++) { // 因子ベクタの次数
        if ((wD[id][k] > 0.0) || (wD[id][k] < 0.0)) {
          codeVectorArray[id][k] = wN[id][k] / wD[id][k];
        }
      }
    }

  } // batch()

  // 格子間の距離を得る。（ベクタ間のノルムではない）
  private double getGridDistance(int bestX, int bestY, int tempX, int tempY) {
    double distance;
    int delta = bestX - tempX;
    distance = delta * delta;
    delta = bestY - tempY;
    distance = distance + delta * delta;
    distance = Math.sqrt(distance);
    return distance;
  }

  // 格子間のノルムを得る。（ベクタ間のノルムではない）
  private double getGridNorm(int bestIndex, int tempIndex) {
    int bestX = bestIndex % dimOfCodeX; // BMUのX座標
    int bestY = bestIndex / dimOfCodeX; // BMUのY座標
    int tempX = tempIndex % dimOfCodeX; // 参照ベクタのX座標
    int tempY = tempIndex / dimOfCodeX;
    
    double norm;
    int delta = bestX - tempX;
    norm = delta * delta;
    delta = bestY - tempY;
    norm = norm + delta * delta;
    return norm;
  }

  /**
   * 参照ベクタ行列からＵマップを求める
   */
  private void calculateUMatrix() {
    // 下、右、右下との差分の二乗和の平方根をＵマップに代入
    for (int i = 0; i < dimOfCodeX; i++) { // i:マップＸ軸
      for (int j = 0; j < dimOfCodeY; j++) {   // j:マップＹ軸
        int id =  dimOfCodeX * j + i;
        // 下側ユニットとの差分（下端の行を除く）
        if (i < dimOfCodeX - 1) {
          double dx = 0;
    	   for (int k = 0; k < dimOfFactor; k++) { // k:参照ベクタの次数
            double delta = codeVectorArray[id][k]
                         - codeVectorArray[id + 1][k];
            dx = dx + delta * delta;
          }
          uMatrix[ 2 * i + 1][2 * j] = Math.sqrt(dx);
        }
        // 右側ユニットとの差分（右端の列を除く）
        if (j < dimOfCodeY - 1) {
          double dy = 0;
          for (int k = 0; k < dimOfFactor; k++) { // k:参照ベクタの次数
            double delta = codeVectorArray[id][k]
                         - codeVectorArray[id + dimOfCodeX][k];
            dy = dy + delta * delta;
          }
          uMatrix[2 * i][2 * j + 1] = Math.sqrt(dy);
        }
        // 右下ユニットとの差分（下端の行と右端の列を除く）
        if ((j < dimOfCodeY - 1) && (i < dimOfCodeX - 1)) {
          double dz1 = 0;
          double dz2 = 0;
          for (int k = 0; k < dimOfFactor; k++) { // k:参照ベクタの次数
            double delta = codeVectorArray[id][k]
                         - codeVectorArray[id + dimOfCodeX + 1][k];
            dz1 = dz1 + delta * delta;
            delta = codeVectorArray[id + dimOfCodeX][k]
                  - codeVectorArray[id + 1][k];
            dz2 = dz2 + delta * delta;
          }
          double dz = (Math.sqrt(dz1 / 2.0) + Math.sqrt(dz2 / 2.0)) / 2.0;
          uMatrix[2 * i + 1][2 * j + 1] = dz;
        }
      } // j
    } // i
    
    // 上下左右の値から、最大と最小を除いた値の平均をＵマップに代入
    for (int i = 0; i< dimOfUX; i+= 2) {
      for (int j = 0; j < dimOfUY; j+= 2) {
        // 上端の行と左端の列を除く
        if ( (i > 0) && (j > 0) ) {
          double[] medtable = new double[4];
          medtable[0] = uMatrix[i-1][j];
          medtable[1] = uMatrix[i+1][j];
          medtable[2] = uMatrix[i][j-1];
          medtable[3] = uMatrix[i][j+1];
          Arrays.sort(medtable);
          /* Actually mean of two median values */
          uMatrix[i][j]=(medtable[1]+medtable[2])/2.0;
        } 
        // 左端の列（上端を除く）
        else if ( (j == 0) && (i > 0) ) {
          double[] medtable = new double[3];
          medtable[0] = uMatrix[i-1][j];
          medtable[1] = uMatrix[i+1][j];
          medtable[2] = uMatrix[i][j+1];
          Arrays.sort(medtable);
          uMatrix[i][j] = medtable[1];
        }
        // 上端の行（左端を除く）
        else if ( (i == 0) && (j > 0) ) {
          double[] medtable = new double[3];
          medtable[0] = uMatrix[i+1][j];
          medtable[1] = uMatrix[i][j-1];
          medtable[2] = uMatrix[i][j+1];
          Arrays.sort(medtable);
          uMatrix[i][j] = medtable[1];
        }
        // 左上の隅のみ
        else if ((i == 0) && (j == 0)) {
          uMatrix[i][j] = (uMatrix[i+1][j] + uMatrix[i][j+1]) / 2.0;
        }
      } // for (j < uyDim; j+= 2)
    } // for (i< uxDim; i+= 2)
  } // calculateUMatrix()

  /**
   * Ｕマップの擾乱量を求める
   * @param uMatrix double[][] // Ｕマップ
   * @return diff double // Ｕマップの擾乱量
   */
  public double getUmapDisorder(double[][] uMatrix) {
    double min = +Double.MAX_VALUE; // 正の最大値で初期化
    double max = -Double.MAX_VALUE; // 負の最大値で初期化
    int numOfX = uMatrix.length;
    int numOfY = uMatrix[0].length;
    
    // Ｕマップの最小値と最大値を求める
    for (int i = 0; i < numOfX; i++) {
      for (int j = 0; j < numOfY; j++) {
        if (uMatrix[i][j] < min) min = uMatrix[i][j];
        if (uMatrix[i][j] > max) max = uMatrix[i][j];
      }
    }
    // 閾値以上の格子点の数を求める
    // 干上がると下位が増え、閾値以上となる格子点が減るため
    int numOfUpper = 0; // 閾値以上の格子点の数
    double threshold = min + (max - min) * 0.5; // 中間値を閾値とする 0.05～0.5
    for (int i = 0; i < numOfX; i++) {
      for (int j = 0; j < numOfY; j++) {
        if (uMatrix[i][j] >= threshold) numOfUpper = numOfUpper + 1;
      }
    }
    // 擾乱量を求める
    double diff = (double)(numOfUpper - oldNumOfUpper); // 変動分
    double disorder = diff * diff; // 擾乱量（変動分の自乗）
    oldNumOfUpper = numOfUpper; // 閾値以上の格子点の数を保存
    
    return disorder;
  }
  
  /**
   * ラベル・マップを更新する
   */
  public void makeLabelMap() {
    // ラベル・マップを空白でリセット
    for (int i = 0; i < dimOfCodeX; i++){
      for (int j = 0; j < dimOfCodeY; j++){
        labelMap[i][j] = ""; // 空白
      }
    }
    
    // 最近似ユニットにラベルを張り付ける
    for (int id = 0; id < numOfFactor; id++) {
      // 最近似ユニットを見つける。
      int bestMatchIndex = findWinner(factorVectors[id]);
      // ラベルを貼り付ける。
      int i = bestMatchIndex % dimOfCodeX;
      int j = bestMatchIndex / dimOfCodeX;
      
      if (labelMap[i][j].equals("")) {
        labelMap[i][j] = factorLabels[id]; // 先頭記号は無し
      }
      else {
        labelMap[i][j] = labelMap[i][j] + factorLabels[id]; // 多重記号は無し
      }
    } // numOfFactor
  } // makeLabelMap()
  
  /**
   * ラベル・マップに試行因子ベクタのラベルを追加する
   */
  public void appendLabelMap() {
    
    if (numOfTrial < 1) return;
    
    // 試行因子ベクタの最近似ユニットにラベルを張り付ける
    for (int id = 0; id < numOfTrial; id++) {
      // 最近似ユニットを見つける。
      int bestMatchIndex = findWinner(trialVectors[id]);
      // ラベルを貼り付ける。
      int i = bestMatchIndex % dimOfCodeX;
      int j = bestMatchIndex / dimOfCodeX;
      if (labelMap[i][j].equals("")) {
        labelMap[i][j] = "★" + trialLabels[id]; // 先頭記号は★
      }
      else {
        labelMap[i][j] = labelMap[i][j] + ":" + "★" + trialLabels[id];
      }
    } // numOfFactor
  } // appendLabelMap()
  
  /**
   * 計量値を求める
   * @param currentCount int // 現在までのエポック回数
   */
  public final void calculateQuantity(int currentCount) {
    double sum = 0.0;
    double xS = 0.0, yS = 0.0, xE = 0.0, yE = 0.0;
    String labelS = "", labelE = ""; // 始点、終点
    int id; // インデックス
    boolean findFlag; // true:終点候補が見つかった、false:見つからない
    int labelCount = 0; // ラベル数のカウンタ
    
    // 最初の始点候補を見つける
    for (id = 0; id < labelMap.length; id++) {
      labelS = labelMap[id][0]; // 始点
      if (!labelS.equals("")) {
        labelCount = labelCount + 1;
        break;
      }
    } // i
    if (labelS.equals("")) {
      DJ.print("***** WARNING ***** SelfOrganizingMap.calculateQuantity()"
               + " Failed to find first factor label.");
      quantity = -1.0; // 計量値
      return;
    }
    id = id + 1; // インデックスを更新
    
    // 残りの候補を見つける
    while (id < labelMap.length - 1) {
      findFlag = false; // 終点候補が見つからない（リセット）
      for (; id < labelMap.length; id++) {
        labelE = labelMap[id][0]; // 終点
        if (!labelE.equals("")) {
          labelCount = labelCount + 1;
          findFlag = true; // 終点候補が見つかった
          break;
        }
      } // id
    
      if (!findFlag) break; // 終点候補が見つからなければ、ブレークして終了
      
      for (int k = 0; k < numOfFactor; k++) {
        if (labelS.equals(factorLabels[k])) {
          xS = factorVectors[k][0]; // 因子ベクタ配列[numOfFactor][dimOfFactor]
          yS = factorVectors[k][1];
          break;
        }
      }
      for (int k = 0; k < numOfFactor; k++) {
        if (labelE.equals(factorLabels[k])) {
          xE = factorVectors[k][0];
          yE = factorVectors[k][1];
          break;
        }
      }
      double val = (xE - xS) * (xE - xS) + (yE - yS) * (yE - yS);
      sum = sum + Math.sqrt(val);
      
      id = id + 1;
      labelS = labelE; // 始点を更新
    } // while

    quantity = sum; // * 0.002; // 計量値（スケーリング無し）
    
    if (quantity < 0.0000001) {
      DJ.print("***** WARNING ***** SelfOrganizingMap.calculateQuantity()"
               + " quantity is too small. quantity = " + quantity);
      quantity = -1.0; // 計量値
      return;
    }
    
    // 計算値の最小値
    if (quantity < lowestQuantity) {
      if (labelCount == numOfFactor) {
        lowestQuantity = quantity; // 計算値の最小値を更新
        lowestQuantityId = currentCount; // 最小計量値のエポック回数
        // 最小計量値時のラベルマップ[Ｘ軸格子数＊Ｙ軸格子数]
        System.arraycopy(labelMap, 0, lowestLabelMap, 0, lowestLabelMap.length);
      }
    }
    
  } // calculateQuantity()
  
} // SomLinear クラス

// EOF
