/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2021.4.4
 *  Copyright: 2020, 2021
 */
package task;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Random;
import java.util.StringTokenizer;

import view.SomViewer;
import util.DJ;
import util.LogEditor;
import util.TimeStamp;
import view.SomViewerLauncher;

/**
 * <p> 表　題: Class: SomPlayer</p>
 * <p> 説　明: 学習済みの自己組織化マップを読込んで表示する</p>
 * <p> 　　　  自己組織化マップの学習は行われない</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2021</p>
 * <p> 作成日: 2021.05.30</p>
 */
public class SomPlayer extends Task {

  // 学習・表示パラメータ
  int epoch = 1; // エポックの回数
  int interval = 1; //50; // 経過表示間隔
  
  // SOMパラメータ
  public double learningRate = 0.1; // 学習係数 0.05～0.2
  public double learningRadius = 10.0; // 初期学習半径 dimOfCodeX / 2
  public double lastRadius = 1.0; // 最終学習半径
  
  // 因子ベクタ（学習用入力データ）
  public String factorFileName = "FactorFile";// 因子データのディフォルト・ファイル名
  public int numOfTrial = 0; // 試行因子ベクタ数（デフォルト値）

  // ニューラルネットワーク（参照ベクタの配列）
  public int dimOfCodeX = 20; // Ｘ軸格子数
  public int dimOfCodeY = 20; // Ｙ軸格子数
  
  // Ｕマップ（隣接する参照ベクタの差分を等高線で表示）
  public int uMapXSize = 400; // ＵマップＸ軸表示幅[ドット]
  public int uMapYSize = 400; // ＵマップＹ軸表示幅[ドット]
  public int numOfContour = 10; // Ｕマップ等高線数（デフォルト値）
  public int baseContour = 10; // Ｕマップ基準等高線（デフォルト値）
  public double borderContour = 0.2; // Ｕマップ境界等高線

  // 評価、誤差
  public int evaluationType = 0; // 評価方法（デフォルト値）
  public double quantumCoeff = 1.0; // 量子化誤差表示係数
  public double disorderCoef = 1.0; // 擾乱量表示係数
  
  // 変数
  public int learningType; // 学習方法
  public long initialType; // 初期化タイプ
  public int shuffleMethod; // 撹拌方法
  
  // 因子ベクタ
  private String titleLine = "#SOM Data : " ; // タイトル・コメント行
  private String trialTitleLine = "#SOM Trial Data : " ; // 試行用タイトル・コメント行
  private String[] factorNames; // 因子名
  private int dimOfFactor; // 因子ベクタの次数
  private int numOfFactor; // 因子ベクタ数
  private double[][] factorVectors; // 因子ベクタ配列[numOfFactor][dimOfFactor]
  private String[] factorLabels; // 因子ラベル配列[numOfFactor]
  private double[][] trialVectors; // 試行用因子ベクタ配列[trialNum][dimOfFactor]
  private String[] trialLabels; // 試行用因子ラベル配列[trialNum]
  
  // 参照ベクタ
  private int numOfCodeVector; // 参照ベクタの個数[dimOfCodeX*dimOfCodeY]
  private double[][] codeVectorArray; // 参照ベクタ配列[numOfCodeVector][dimOfFactor]
  
  // Ｕマップ
  private int dimOfUX; // ＵマップのＸ軸格子数[dimOfCodeX*2+1]
  private int dimOfUY; // ＵマップのＸ軸格子数[dimOfCodeY*2+1]
  private double[][] uMatrix; // Ｕマップ（Ｕ行列）[dimOfUX][dimOfUY]
  
  // ラベル・マップ
  private  String[][] labelMap; // ラベル・マップ（ラベル行列）[Ｘ軸格子数＊Ｙ軸格子数]
  
  // 誤差
  private double quantumError; // 量子化誤差
  
  // バッチ補助変数（参照ベクタの重みの算出に用いる）
  double[][] wN; // 重みの分子
  double[][] wD; // 重みの分母
    
  // オブジェクト
  private Random randamGenerator; // 乱数発生器
  private SomViewerLauncher somViewerLauncher; // ＳＯＭビューワ・ランチャ
  private SomViewer somViewer; // SOMマップ・ビューワ  
  
//$ 使用しない
//  public SomPlayer() {
//    super();
//    
//    DJ._print(" ############################################################");
//    DJ.print(" ##");
//    DJ.print(" ##      ・SOMタスクの実行を開始するには、             ");
//    DJ.print(" ##      コンソールの「Run/Pause」ボタンをクリックしてください。 ");
//    DJ.print(" ##");
//    DJ.print(" ############################################################");
//  }
  
 /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // SOMビューワ・ランチャ
    somViewerLauncher = DJ.launchSomViewer(
            SomViewer.SOM_GENERAL, "SelfOrganizingMap");
    
    somPlayer(); // タスク本体の呼び出し
  }
  
  /**
   * 学習済みの自己組織化マップを読込んで表示する
   * 試行用因子データを読込んでラベルを表示する
   */
  public void somPlayer() {
    DJ._print("SomPlayer.somPlayer() ==================");
    
    // DJ._print("SomViewerへの参照を得る");
    somViewer = somViewerLauncher.getSomViewer();
    if (somViewer == null) {
      DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
          + " SomViewer is null.");
      return;
    }

//    DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1)
              | (i == epoch); //経過表示フラグ
      
      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

      // グラフは表示しない
      if (intervalFlag) { // 経過表示インターバル
        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            wait(SLEEP_TIME); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          }
          catch (InterruptedException e) {
            DJ.print(" ***** ERROR ***** " + getClass().getName() + "\n"
               + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval

      // 実行処理の中断
      if (abortFlag) {
        DJ._print(" ##### Abort action requested #####");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch ---------------------------------------");
    }
//    DJ._print(" End of all epoch ---------------");

//    DJ._print(" ##### 実行の一時停止 #####");
//    DJ.print(" 実行再開にはコンソールのRunボタンをクリックしてください。");
    
    DJ._print(" ############################################################");
    DJ.print(" ##");
    DJ.print(" ##      ・学習済みのSOMコードを読込むには、             ");
    DJ.print(" ##      コンソールの「Run/Pause」ボタンをクリックして、 ");
    DJ.print(" ##      表示されたファイル・ダイアログで                ");
    DJ.print(" ##      SOMコード・ファイル(SomCode_xxxxx.dat)を選択し ");
    DJ.print(" ##      「開く」ボタンをクリックしてください。          ");
    DJ.print(" ##");
    DJ.print(" ############################################################");

//#
//    // スレッドの休止（デバッグ用一時停止のため）
//    pauseFlag = true;
//    synchronized (this) {
//      try { if (pauseFlag) wait(); // 休止状態
//      } catch (InterruptedException e) {
//        DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
//                + " Exception occur in wait(sleepTime):" + e.toString());
//      }
//    } // synchronized()
    
    DJ._print(" ##### SOMコードの読み込み #####");
    String somFileName = "SomCode"; // ファイル名の接頭ワード
    DJ.print(" SOMコード・ファイル名：", somFileName);
    if (!loadSom(somFileName)) { // SOMコード・ファイルの読込
      DJ.print("***** ERROR ***** SomPlayer.somPlayer()"
               + " Failed to load SOM Code file.");
      return;
    }
    
//$
//    // 読込んだデータを表示
//    DJ._print("・因子名");
//    DJ.print(" factorNames=", DJ.stringArrayToString(factorNames));
//    DJ._print("・学習済み参照ベクタ");
//    DJ.print(" codeVectorArray = ", codeVectorArray);
//    DJ._print("・ラベルマップ[Ｘ軸格子数＊Ｙ軸格子数]");
//    DJ.print(" labelMap[" + labelMap.length + "] = ");
//    for (int ix = 0; ix < labelMap.length; ix++) 
//        DJ.print(" labelMap = ", DJ.stringArrayToString(labelMap[ix]));
    
    
    // Ｕマップ（Ｕ行列）を求める。
    dimOfUX = dimOfCodeX * 2; // ＵマップのＸ軸格子数をリセット
    dimOfUY = dimOfCodeY * 2; // ＵマップのＸ軸格子数をリセット
    uMatrix = new double[dimOfUX][dimOfUY];  // Ｕマップをリセット
    calculateUMatrix(); // Ｕマップ（Ｕ行列）       
    // DJ.print("U Matrix", uMatrix); // Ｕマップ
    
    // SOMビューワにコメントを設定する
    somViewer.setComment(titleLine);
    
    // SOMビューワにプロパティを設定する
    somViewer.setProperty(numOfTrial, evaluationType,
          numOfContour, baseContour,  borderContour, 
          uMapXSize, uMapYSize);
    // SOMビューワにパラメータを設定する
    somViewer.setParameter(dimOfFactor, dimOfCodeX, dimOfCodeY,
            factorNames, numOfFactor, factorVectors, factorLabels);
    // SOMへの参照をあらかじめSomViewerへ渡しておく
    somViewer.setMap(codeVectorArray, uMatrix, labelMap);
//    // SOMビューワを表示する
//    somViewer.setVisible(true);
    
    int iCount = epoch * numOfFactor; // 現在までの学習回数
    somViewer.updateMap(iCount); // マップを更新する
    
    // SOMビューワを表示する
    somViewer.setVisible(true);
    
    
    DJ._print(" ############################################################");
    DJ.print(" ##");
    DJ.print(" ##      ・試行用因子ベクタを読込むには、             ");
    DJ.print(" ##      コンソールの「Run/Pause」ボタンをクリックして、 ");
    DJ.print(" ##      表示されたファイル・ダイアログで                ");
    DJ.print(" ##      試行用因子ベクタ・ファイル(Factor_xxxx.csv)を選択し、");
    DJ.print(" ##      「開く」ボタンをクリックしてください。          ");
    DJ.print(" ##");
    DJ.print(" ############################################################");


    // スレッドの休止（デバッグ用一時停止のため）
    pauseFlag = true;
    synchronized (this) {
      try { if (pauseFlag) wait(); // 休止状態
      } catch (InterruptedException e) {
        DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                + " Exception occur in wait(sleepTime):" + e.toString());
      }
    } // synchronized()
    
    DJ._print(" ##### 試行用因子ベクタの読み込み #####");
    String trialFileName = "Factor"; // ファイル名の接頭ワード
    DJ.print(" 試行用因子ベクタ・ファイル名：", trialFileName);
    if (!loadTrialFactor(trialFileName)) { // 試行用因子ベクタ・ファイルの読込
      DJ.print("***** ERROR ***** SomPlayer.somPlayer()"
               + " Failed to load Trial Factor Vector file.");
      return;
    }
    
    // 試行因子ベクタのラベルを追加する
    appendLabelMap();
    
    somViewer.updateMap(iCount); // マップを更新する
    
    // SOMビューワを表示する
    somViewer.setVisible(true);
    
    // 実行時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    
    DJ.print_("・タスク終了日時：", TimeStamp.getTimeFormated());
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print(", タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");

  } // somPlayer()
 
  // ---------------------------------------------------------------------------
  /**
   * 乱数で撹拌された参照ベクタを生成する。
   * @param min double[] // 各因子の最小値の配列
   * @param max double[] // 各因子の最大値の配列
   * @return aCodeVector double[] // 参照ベクタ
   */
  double[] getRandomShuffledCode(double[] min, double[] max) {
    double[] aCodeVector = new double[dimOfFactor]; // 因子ベクタ
    for  (int i = 0; i < dimOfFactor; i++) {
      aCodeVector[i] = (max[i] - min[i]) * randamGenerator.nextDouble() + min[i];
    }
    return aCodeVector;
  }
  
  /**
   * 試行用因子データの読み込み
   * 
   * @param fileName String // 試行用因子データ・ファイル名
   * @return resultFlag boolean // 読み込みの成否
   */
  private boolean loadTrialFactor(String fileName) {
    ArrayList<String> stringList = LogEditor.loadText(fileName);
    if (stringList == null) {
      DJ.print("***** ERROR ***** SomPlayer.loadTrialFactor()"
               + " Failed to load Trial Factor-Vector file.");
      return false;
    }
    int numOfLines = stringList.size(); // 行数
    if (numOfLines < 3) {
      DJ.print("***** ERROR ***** SomPlayer.loadTrialFactor()"
               + " Num of lines in Factor-Vector file less than 3.");
      return false;
    }
    
    DJ.print(" （１）String comment:試行用因子データ・ファイルのタイトルとコメント");
    trialTitleLine = stringList.get(0); // タイトルとコメント
    DJ.print("Title and comment of Trial Factor Files : ", trialTitleLine);
    
    DJ.print(" （２）String[] factorName:因子名配列");
    String dataNameLine = stringList.get(1); // 因子名行
    // DJ.print("Facter Name Line : ", dataNameLine); // 因子名行
    
    StringTokenizer dataNameToken = new StringTokenizer(dataNameLine, ",");
    ArrayList<String> factorNameList = new ArrayList<>(); // 因子名リスト
    while (dataNameToken.hasMoreTokens()) {
      factorNameList.add(dataNameToken.nextToken()); // 因子名を追加
    }
    DJ.printList("Name of Factors : ", factorNameList); // 因子名リスト
    
    if (factorNameList.size() != dimOfFactor) { // 因子ベクタの次数
      DJ.print("***** ERROR ***** SomPlayer.loadTrialFactor()"
               + " Dimension of factorNameList != dimOfFactor.");
      return false;
    }
//    factorNames = new String[dimOfFactor];
//    for (int i = 0; i < dimOfFactor; i++) {
//      factorNames[i] = factorNameList.get(i); // 因子名の格納
//    }
    
    DJ.print(" （３）double[][] trialVectors:試行用因子ベクタ配列");
    DJ.print(" （４）String[] trialLabels:試行用因子ベクタ・ラベル配列");
    //  因子ベクタ数をチェック
    numOfTrial = numOfLines - 2; // 試行用因子ベクタ数
    if (numOfTrial < 1) { // 試行用因子ベクタ数
      DJ.print("***** ERROR ***** SomPlayer.loadTrialFactor()"
               + " Number of Trial Facter Vector is less than 1.");
      return false;
    }
    trialVectors = new double[numOfTrial][dimOfFactor]; // 試行用因子ベクタ配列[試行ベクタ数][因子数]
    trialLabels = new String[numOfTrial] ; // 試行用因子ラベル配列［試行ベクタ数］
    
    int rowNum = numOfLines - 2; // 因子ベクタ数の総数（残りの行数）
    for (int j = 0; j < rowNum; j++) { // 因子ベクタの取出し
      String dataLine = stringList.get(j + 2);
      StringTokenizer dataToken = new StringTokenizer(dataLine, ",");
    
      // 因子の取出し
      for (int i = 0; i < dimOfFactor; i++) {
        if (dataToken.hasMoreTokens()) {
          double value = Double.parseDouble(dataToken.nextToken().trim());
          trialVectors[j][i] = value;
        }
        else { // データが無い場合はゼロを代入
          trialVectors[j][i] = 0.0;
        }
      } // dimOfFactor
    
      // ラベルの取出し
      if (dataToken.hasMoreTokens()) { // ラベルが有るか。
        String label = dataToken.nextToken();
        trialLabels[j] = label;
      }
      else { // ラベルが無い場合はnullを代入
        trialLabels[j] = "";
      }
      
    } // rowNum
    
    DJ.print("　試行用因子ベクタ数：numOfTrial=", numOfTrial);
    DJ.print("　試行用因子ベクタ配列：trialVectors", trialVectors);
    DJ.print("　試行用因子ベクタのラベル：trialLabels=",
            DJ.stringArrayToString(trialLabels));
    
    return true;
  }
  
  /**
   * SOMファイルの読み込み
   * 
   * （１）String titleLine:タイトル,因子種別,因子数,因子ベクタ数
   * （２）Property: // プロパティ
   * （３）"Factor Vector: Dimension of factor = "
   *       int dimOfFactor:因子ベクタの次元
   * （４）"Factor Names: String[] factorNames = " 因子名配列
   * （５）double[][] factorVectors; // 因子ベクタ配列[個数][次元]
   * （６）
   * （７）"Code Vectors: Number of codeVectors = "
   * （８）int numOfCodeVector:参照ベクタの個数
   * （９）double[][] codeVectorArray; // 参照ベクタ配列[個数][次元]
   * （10）"Row number of Label Map = "
   * （11）labelMap.length; // 行数
   * （12）"Collumn number of Label Map = "
   * （13）labelMap[0].length; // 列数
   * （14）"Label Map = "
   * （15）String[][] labelMap; // ラベル・マップ（ラベル行列）
   * 
   * @param fileName String // 仮（ディフォルト）ファイル名
   * @return resultFlag boolean // 実行結果の成否フラグ
   */
  private boolean loadSom(String fileName) {
    // ファイル・ダイアログを表示し、指定（選択、入力）されたファイルを読込む
    ArrayList<String> stringList = LogEditor.loadText(fileName);
    if (stringList == null) {
      DJ.print("***** ERROR ***** SomPlayer.loadSom()"
               + " Failed to load SOM Code file.");
      return false;
    }
    
    int id = 0; // ライン・インデックス
    DJ.print(" （１）String titleLine:タイトル,因子種別,因子数,因子ベクタ数");
    titleLine = stringList.get(id);
    DJ.print("Title of SOM Files", titleLine); // SOMファイルのタイトル
    
    DJ.print(" （２）Property // プロパティ");
    stringList.get(++id); // Property:
    String tag = stringList.get(++id); // Number Of Trial =
    numOfTrial = Integer.valueOf(stringList.get(++id));
    DJ.print(tag, numOfTrial); // 試行因子ベクタ数
    
    tag = stringList.get(++id); // Evaluation Type = 
    evaluationType = Integer.valueOf(stringList.get(++id));
    DJ.print(tag, evaluationType); // 評価方法
    
    tag = stringList.get(++id); // Numbe Of Contour = 
    numOfContour = Integer.valueOf(stringList.get(++id));
    DJ.print(tag, numOfContour); // Ｕマップ等高線数
    
    tag = stringList.get(++id); // Base Contour = 
    baseContour = Integer.valueOf(stringList.get(++id));
    DJ.print(tag, baseContour); // Ｕマップ等高線数
    
    tag = stringList.get(++id); // Border Contour = 
    borderContour = Double.valueOf(stringList.get(++id));
    DJ.print(tag, borderContour); // Ｕマップ境界等高線
    
    tag = stringList.get(++id); // U-Map X-Axis Size = 
    uMapXSize = Integer.valueOf(stringList.get(++id));
    DJ.print(tag, uMapXSize); // ＵマップＸ軸表示幅[ドット]
    
    tag = stringList.get(++id); // U-Map Y-Axis Size = 
    uMapYSize = Integer.valueOf(stringList.get(++id));
    DJ.print(tag, uMapYSize); // ＵマップＹ軸表示幅[ドット]
    
    DJ.print(" （３）String[] factorName; // 因子名");
    stringList.get(++id); // Factor Vector:
    tag = stringList.get(++id); // Dimension of factor =
    dimOfFactor = Integer.valueOf(stringList.get(++id));
    DJ.print(tag, dimOfFactor); // 因子ベクタの次数
    
    DJ.print(" （４）String[] factorNames;"); // 因子名配列
    stringList.get(++id); // Factor Name:
    tag = stringList.get(++id); // factorNames =
    String favtorNameLine = stringList.get(++id); // 因子名行
    StringTokenizer factorNameToken = new StringTokenizer(favtorNameLine, ",");
    ArrayList<String> factorNameList = new ArrayList<>(); // 因子名リスト
    while (factorNameToken.hasMoreTokens()) {
      factorNameList.add(factorNameToken.nextToken()); // 因子名を追加
    }
    dimOfFactor = factorNameList.size(); // 因子ベクタの次数
    if (dimOfFactor < 1) {
      DJ.print("***** ERROR ***** SomPlayer.loadSom()"
               + " Dimension of factorNameList < 1.");
//      return false;
    }
    factorNames = new String[dimOfFactor];
    for (int i = 0; i < dimOfFactor; i++) {
      factorNames[i] = factorNameList.get(i); // 因子名の格納
    }
    DJ.print(tag, DJ.stringArrayToString(factorNames)); // 因子名配列
    
    DJ.print(" （５）double[][] factorVectors; // 因子ベクタ配列[個数][次元]");
    tag = stringList.get(++id); // factorVectors =
    numOfFactor = Integer.valueOf(stringList.get(++id)); // 因子ベクタの個数
    factorVectors = new double[numOfFactor][dimOfFactor]; // 参照ベクタ配列[因子ベクタの個数][因子ベクタの次数]
    for (int j = 0; j < numOfFactor; j++) { // 因子ベクタベクタの取出し
      String factorLine = stringList.get(++id);
      StringTokenizer factorToken = new StringTokenizer(factorLine, ",");
      // 因子ベクタの取出し
      for (int i = 0; i < dimOfFactor; i++) {
        if (factorToken.hasMoreTokens()) {
          factorVectors[j][i] = 
                  Double.parseDouble(factorToken.nextToken().trim());
        }
      } // dimOfFactor
    } // numOfFactor
    DJ.print(tag.replace("=", ""), factorVectors);
    
    DJ.print(" （６）String[] factorLabels; // 因子ラベル配列[個数]");
    stringList.get(++id); // Factor Label:
    tag = stringList.get(++id); // factorLabels = 
    int numOfFactorLabels = Integer.valueOf(stringList.get(++id));
    DJ.print(tag + " ", numOfFactorLabels); // 因子ラベルの個数
    
    String factorLabelLine = stringList.get(++id); // 因子ラベル行
    StringTokenizer factorLabelToken = new StringTokenizer(factorLabelLine, ",");
    ArrayList<String> factorLabelList = new ArrayList<>(); // 因子ラベル・リスト
    while (factorLabelToken.hasMoreTokens()) {
      factorLabelList.add(factorLabelToken.nextToken()); //  因子ラベルを追加
    }
    if (factorLabelList.size() != numOfFactorLabels) { // 因子ラベルの個数
      DJ.print("***** ERROR ***** SomPlayer.loadSom()"
               + " Size of factorLabelList != numOfFactorLabels.");
      return false;
    }
    if (factorLabelList.size() != numOfFactor) { // 因子ラベルの個数を因子数と比較
      DJ.print("***** ERROR ***** SomPlayer.loadSom()"
               + " Size of factorLabelList != numOfFactor.");
      return false;
    }
    factorLabels = new String[numOfFactor];
    for (int i = 0; i < numOfFactor; i++) {
      factorLabels[i] = factorLabelList.get(i); // 因子ラベルの格納
    }
    DJ.print(tag, DJ.stringArrayToString(factorLabels)); // 因子ラベル配列

    DJ.print(" （７）、（８）int numOfCodeVectore; // 参照ベクタの個数");
    stringList.get(++id); // Code Vectors:
    tag = stringList.get(++id); // Number of codeVectors =
    numOfCodeVector = Integer.valueOf(stringList.get(++id));
    DJ.print(tag + " ", numOfCodeVector); // 参照ベクタの個数
    
    DJ.print(" （９）double[][] codeVectorArray; // 参照ベクタ配列[個数][次元]");
    tag = stringList.get(++id); // codeVectors =
    codeVectorArray = new double[numOfCodeVector][dimOfFactor]; // 参照ベクタ配列[参照ベクタの個数][因子ベクタの次数]
    for (int j = 0; j < numOfCodeVector; j++) { // 参照ベクタの取出し
      String codeLine = stringList.get(++id);
      StringTokenizer codeToken = new StringTokenizer(codeLine, ",");
      // 参照ベクタの取出し
      for (int i = 0; i < dimOfFactor; i++) {
        if (codeToken.hasMoreTokens()) {
          codeVectorArray[j][i] = 
                  Double.parseDouble(codeToken.nextToken().trim());
        }
      } // dimOfFactor
    } // numOfCodeVector
    DJ.print(tag.replace("=", ""), codeVectorArray);
    
    DJ.print(" （10）、（11）int labelMap.length; // ラベル・マップの行数");
    stringList.get(++id); // Label Map:
    tag = stringList.get(++id); // Row number of Label Map =
    dimOfCodeX = Integer.valueOf(stringList.get(++id)); // SOMのX軸格子数
    DJ.print(tag + " : ", dimOfCodeX); // ラベル・マップの行数
    DJ.print(" （12）、（13）int labelMap[0].length; // ラベル・マップの列数");
    tag = stringList.get(++id); // Collumn number of Label Map = 
    dimOfCodeY = Integer.valueOf(stringList.get(++id)); // SOMのY軸格子数
    DJ.print(tag + " : ", dimOfCodeY); // ラベル・マップの列数
    DJ.print(" labelMap[" + dimOfCodeX + "][" + dimOfCodeY + "] = ");
    
    DJ.print(" （14）、（15）String[][] labelMap; // ラベル・マップ（ラベル行列）");
    tag = stringList.get(++id); // Label Map =
    if ( numOfCodeVector != dimOfCodeX * dimOfCodeY) {
      DJ._print("***** WARNING ***** SomPlayer.loadSom()"
               + " Number of labels is diferent from number of CodeVectors.");
      DJ.print_(" Number of labels", numOfCodeVector);
      DJ.print(" Number of CodeVectors", dimOfCodeX * dimOfCodeY);
    }
    labelMap = new String[dimOfCodeX][dimOfCodeY]; // ラベル・マップ（ラベル行列）
    // ラベル・マップ（ラベル行列）の取出し
    for (int j = 0; j < dimOfCodeX; j++) {
      String labelLine = stringList.get(++id);
      StringTokenizer labelToken = new StringTokenizer(labelLine, ",");
      for (int i = 0; i < dimOfCodeY; i++) {
        if (labelToken.hasMoreTokens()) {
          labelMap[j][i] = labelToken.nextToken().trim(); // ラベル・マップ
        }
      } // dimOfFactor
    } // dimOfCodeX

    DJ.print(tag);
    for (int j = 0; j < dimOfCodeX; j++) {
        DJ.print(" " + j + " ", DJ.stringArrayToString(labelMap[j]));
    }
    
    return true;
  }
  
  /**
   * 最近似ユニット（BMU:Best Match Unit）のインデックスを得る
   * @param factorVector double[] // 因子ベクタ（入力）
   * @return bestMatchIndex int // 最近似ユニットのインデックス
   */
  public int findWinner(double[] factorVector) {
    double delta; // ベクタの要素ごとの差分
    double aNorm; // ノルム（差分の平方和）
    double leastNorm = Double.MAX_VALUE; // ノルムの最少値
    int bestMatchIndex = -1; // 最近似ユニットのインデックス
    
    // 全ての参照ベクタに対して因子ベクタとのノルムを求め、比較する。
    for (int i = 0; i < numOfCodeVector; i++) { // 参照ベクタの個数
      aNorm = 0.0; // ノルム（差分の平方和）
      for (int j = 0; j < dimOfFactor; j++) { // 因子ベクタの次元
        // 参照ベクタと因子ベクタのノルムを計算する。
        delta = codeVectorArray[i][j] - factorVector[j]; // 因子ごとの差分
        aNorm = aNorm + delta * delta; // 差分の平方和
        if (aNorm > leastNorm) break; // 平方和が大きければループから抜ける。
      } // j
      if (aNorm < leastNorm) { // 求めたノルムがノルムの最少値よりも小さい
        leastNorm = aNorm; // ノルムを更新
        bestMatchIndex = i; // 最近似ユニットのインデックスを変更
      }
    } // i
    quantumError = quantumError + Math.sqrt(leastNorm); // 量子化誤差を更新
    
    return bestMatchIndex;
  }
  
  /**
   * 参照ベクタ行列からＵマップを求める
   */
  private void calculateUMatrix() {
    // 下、右、右下との差分の二乗和の平方根をＵマップに代入
    for (int j = 0; j < dimOfCodeY; j++) {   // j:マップＹ軸
      for (int i = 0; i < dimOfCodeX; i++) { // i:マップＸ軸
        int id =  dimOfCodeX * j + i;
        // 最後の列以外
        if (i < dimOfCodeX - 1) {
          double dx = 0;
    	   for (int k = 0; k < dimOfFactor; k++) { // k:参照ベクタの次数
            double delta = codeVectorArray[id][k]
                         - codeVectorArray[id + 1][k];
            dx = dx + delta * delta;
          }
          uMatrix[ 2 * i + 1][2 * j] = Math.sqrt(dx);
        }
        // 右側ユニットとの差分（右端の列を除く）
        if (j < dimOfCodeY - 1) {
          double dy = 0;
          for (int k = 0; k < dimOfFactor; k++) { // k:参照ベクタの次数
            double delta = codeVectorArray[id][k]
                         - codeVectorArray[id + dimOfCodeX][k];
            dy = dy + delta * delta;
          }
          uMatrix[2 * i][2 * j + 1] = Math.sqrt(dy);
        }
        // 右下ユニットとの差分（下端の行と右端の列を除く）
        if ((j < dimOfCodeY - 1) && (i < dimOfCodeX - 1)) {
          double dz1 = 0;
          double dz2 = 0;
          for (int k = 0; k < dimOfFactor; k++) { // k:参照ベクタの次数
            double delta = codeVectorArray[id][k]
                         - codeVectorArray[id + dimOfCodeX + 1][k];
            dz1 = dz1 + delta * delta;
            delta = codeVectorArray[id + dimOfCodeX][k]
                  - codeVectorArray[id + 1][k];
            dz2 = dz2 + delta * delta;
          }
          double dz = (Math.sqrt(dz1 / 2.0) + Math.sqrt(dz2 / 2.0)) / 2.0;
          uMatrix[2 * i + 1][2 * j + 1] = dz;
        }
      }
    }
    
    // 上下左右の値から、最大と最小を除いた値の平均をＵマップに代入
    for (int j = 0; j < dimOfUY; j+= 2) {
      for (int i = 0; i< dimOfUX; i+= 2) {
        // 上端の行と左端の列を除く
        if ( (i > 0) && (j > 0) ) {
          double[] medtable = new double[4];
          medtable[0] = uMatrix[i-1][j];
          medtable[1] = uMatrix[i+1][j];
          medtable[2] = uMatrix[i][j-1];
          medtable[3] = uMatrix[i][j+1];
          Arrays.sort(medtable);
          /* Actually mean of two median values */
          uMatrix[i][j]=(medtable[1]+medtable[2])/2.0;
        } 
        // 左端の列（上端を除く）
        else if ( (j == 0) && (i > 0) ) {
          double[] medtable = new double[3];
          medtable[0] = uMatrix[i-1][j];
          medtable[1] = uMatrix[i+1][j];
          medtable[2] = uMatrix[i][j+1];
          Arrays.sort(medtable);
          uMatrix[i][j] = medtable[1];
        }
        // 上端の行（左端を除く）
        else if ( (i == 0) && (j > 0) ) {
          double[] medtable = new double[3];
          medtable[0] = uMatrix[i+1][j];
          medtable[1] = uMatrix[i][j-1];
          medtable[2] = uMatrix[i][j+1];
          Arrays.sort(medtable);
          uMatrix[i][j] = medtable[1];
        }
        // 左上の隅のみ
        else if ((i == 0) && (j == 0)) {
          uMatrix[i][j] = (uMatrix[i+1][j] + uMatrix[i][j+1]) / 2.0;
        }
      } // for (i< uxDim; i+= 2)
    } // for (j < uyDim; j+= 2)
  } // calculateUMatrix()

  /**
   * ラベル・マップを更新する
   */
  public void makeLabelMap() {
    // ラベル・マップを空白でリセット
    for (int i = 0; i < dimOfCodeX; i++){
      for (int j = 0; j < dimOfCodeY; j++){
        labelMap[i][j] = ""; // 空白
      }
    }
    
    // 最近似ユニットにラベルを張り付ける
    for (int id = 0; id < numOfFactor; id++) {
      // 最近似ユニットを見つける。
      int bestMatchIndex = findWinner(factorVectors[id]);
      // ラベルを貼り付ける。
      int i = bestMatchIndex % dimOfCodeX;
      int j = bestMatchIndex / dimOfCodeY;
      if (labelMap[i][j].equals("")) {
        labelMap[i][j] = factorLabels[id]; // 先頭記号は無し
      }
      else {
        labelMap[i][j] = labelMap[i][j] + factorLabels[id]; // 多重記号は無し
      }
    } // numOfFactor
  } // makeLabelMap()
  
  /**
   * ラベル・マップに試行因子ベクタのラベルを追加する
   */
  public void appendLabelMap() {
    
    if (numOfTrial < 1) {
      DJ._print("***** WARNING ***** SomPlayer.appendLabelMap()"
               + " Number of labels is less than 1.");
      return;
    }
    
    // 試行因子ベクタの最近似ユニットにラベルを張り付ける
    for (int id = 0; id < numOfTrial; id++) {
      // 最近似ユニットを見つける。
      int bestMatchIndex = findWinner(trialVectors[id]);
      // ラベルを貼り付ける。
      int i = bestMatchIndex % dimOfCodeX;
      int j = bestMatchIndex / dimOfCodeY;
      if (labelMap[i][j].equals("")) {
        labelMap[i][j] = "◎" + trialLabels[id]; // 先頭記号は◎
      }
      else {
        labelMap[i][j] = labelMap[i][j] + ":" + "◎" + trialLabels[id];
      }
    } // numOfFactor
  } // appendLabelMap()
  
} // SomPlayer クラス

// EOF