/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva))
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2020.7.1
 *  Copyright: 2020 - 2025
 */
package task;

import util.DJ;
import util.LogEditor;
import view.GraphViewer;
import view.GraphViewerLauncher;
import view.PatternViewer;
import view.PatternViewerLauncher;

/**
 * <p> 表　題: Class: Task</p>
 * <p> 説　明: 実行するプログラム</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019 ～ 2025</p>
 * <p> 作成日: 2019.1.22</p>
 * <p> 更新日: 2025.6.29</p>
 */
public abstract class Task implements Runnable {
  
  protected static final int SLEEP_TIME = 1; //5; //50; //1000; // ゼロは設定不可

  protected boolean pauseFlag = false; // 休止フラグ true:休止 false:実行
  protected boolean abortFlag = false; // 終了フラグ true:終了 false:継続
  
  public boolean stepMode = false; // ステップ送り実行モード
  
  // パターン・ビューワ
  protected boolean patternViewerFlag; // true:表示、false:非表示
  protected PatternViewerLauncher patternViewerLauncher; // ランチャ
  protected PatternViewer patternViewer; // パターン・ビューワ（※名称変更不可）
  protected double[][] patternData0; // パターン・ビューワー・データ
  protected double[][] patternData1; // パターン・ビューワー・データ
  // グラフ・ビューワ
  protected int graphShift = 0; // グラフの縦軸を移動
  protected GraphViewerLauncher graphViewerLauncher; // ランチャ
  protected GraphViewer graphViewer; // グラフ・ビューワ（※名称変更不可）
  protected double[] graphData; // グラフ・ビューワ・データ
  protected String[] dataName; // グラフ・データ名（誤差名）

  // 実行時間計測用変数
  protected boolean intervalFlag = false; // 経過表示フラグ
  protected long beginTime = 0; // 処理開始時刻
  protected long finishTime = 0; // 処理終了時刻
  protected long startTime = 0; // 実行開始時刻
  protected long endTime = 0; // 実行終了時刻
  protected double lapTime = 0; // エポック実行時間
  protected double totalTime = 0; // 実行時間
  
  /**
   * タスクを開始する
   */
  public void startTask() {
    Thread taskThread = createThread(this);// タスク実行する新規スレッド
    startThread(taskThread);
  }

  /**
   * タスクを休止する
   */
  public void pauseTask() {
    if (pauseFlag) {
      pauseFlag = false;
      synchronized(this) {
        this.notify();
      }
    }
    else  {
      pauseFlag = true;
    }
  }
  
  /**
   * タスクを終了する
   */
  public void abortTask() {
    if (pauseFlag) {
      pauseFlag = false;
      synchronized(this) {
        this.notify();
      }
    }
    abortFlag = true;
  }
  
  /**
   * タスク実行用スレッドの生成
   * @param task Task // タスク
   * @return taskThread Thread // タスク実行用スレッド
   */
  public Thread createThread(Task task) {
	  Thread taskThread = new Thread(task); // タスク実行用スレッド
   	taskThread.setPriority(Thread.NORM_PRIORITY);
    return taskThread;
  }

  /**
   * スレッドの実行開始
   * @param thread Thread // スレッド
   * @return resultFlag boolean // true:実行開始, false:実行できない
   */
  public boolean startThread(Thread thread) {
    if (thread == null) {
      LogEditor.print("LogFrame.startThread():There are no taskThread.");
      LogEditor.print("***** ERROR ***** 'taskThread' is not started.");
      return false;
    }
    else {
      thread.start(); // メソッドrun()を実行
      return true;
    }
  }

  /**
   * 新規のスレッドでタスクを実行
   */
  @Override
  public void run() {
    DJ._print("Task is started!");
    DJ.print("===============================================================");
    
    runTask(); // タスク呼び出し用メソッド
  }

  /**
   * タスク呼び出し用メソッド
   * ※コンソールでのタスク生成時にニューラルネットワークが指定された場合は、
   * 　そのタスクでrunTask（）をオーバーライドして、直接呼び出す。
   */
  public void runTask() {
    DJ._print("Task is started!");
    DJ.print("===============================================================");

    DJ._print("Task is running.");
    
    DJ._print("End of Task.");
    DJ.print("===============================================================");
  }
  
  /**
   * ステップ送り実行モードを反転する
   */
  public void stepTask() {
    // stepMode = !stepMode; // ステップ送り実行モードを反転する
    
    if (stepMode) {
      stepMode = false;
      DJ._print("### Task step mode OFF. ###");
    }
    else {
      stepMode = true;
      DJ._print("### Task step mode ON. ###");
    }
    
  }
  
  
  /**
   * ビューワの表示を更新
   * @param i int // エポック
   */
  // void updateViewer(int i) {
  protected void updateViewer(int i) {
    updatePattern(); // パターンの表示を更新
    updateGraph(i); // グラフの表示を更新
  }
  
  /**
   * グラフの表示を更新
   * @param i int // エポック
   */
  // void updateGraph(int i) {
  protected void updateGraph(int i) {
    // グラフの更新
    if (graphViewerLauncher == null) {
      DJ._print("GraphViewerLauncher is null.");
    }
    else{
      if (graphViewer == null) {
        graphViewer = graphViewerLauncher.getGraphViewer();
        if (graphViewer != null) {
          graphViewer.shiftGraphAxis(graphShift); // グラフの縦軸をシフトする
        }
      }
      if (graphViewer != null) {
        // グラフ表示用データを代入
        graphViewer.updateGraph(i, graphData);
      }
    }
  }
  
  /**
   * パターンの表示を更新
   */
  // void updatePattern() {
  protected void updatePattern() {
    if (patternViewer == null) {
      patternViewer = patternViewerLauncher.getPatternViewer();
    }
    if (patternViewer != null) {
      patternViewer.setVisible(patternViewerFlag); // 表示する／しない
      if (patternData1 == null) patternViewer.updatePattern(patternData0);
      else patternViewer.updatePattern(patternData0, patternData1);
    }
  }
  
  /**
   * 外部（コンソール）からグラフ・ビューワの表示を更新
   */
  public void showGraph() {
    if (graphViewer == null) updateGraph(0); // グラフ・ビューワを起動
    if (graphViewer != null) { // パターン・ビューワは存在するか？
      if (graphViewer.isShowing()) {
        graphViewer.setVisible(false); // false:表示しない
      }
      else {
//$        graphViewer.updatePattern(i, graphData);
        graphViewer.setVisible(true); // true:表示する
      }
    }
    else {
      DJ._print("GraphViewer is null.");
    }
  }
  
  /**
   * 外部（コンソール）からパターン・ビューワの表示を更新
   */
  public void showPattern() {
    if (patternViewer != null) { // パターン・ビューワは存在するか？
      if (patternViewer.isShowing()) {
        patternViewer.setVisible(false); // false:表示しない
      }
      else {
//$        if (patternData1 == null) patternViewer.updatePattern(patternData0);
//$        else patternViewer.updatePattern(patternData0, patternData1);
        patternViewer.setVisible(true); // true:表示する
      }
    }
    else {
      DJ._print("PatternViewer is null.");
    }
  }
  
  /**
   * タスクのステップ実行
   * タスクの途中経過観察のため一時停止する
   */
  public void stepMode() {
    // スレッドの休止
    pauseFlag = true;
    synchronized (this) {
      try { if (pauseFlag) wait(); // 休止状態
      } catch (InterruptedException e) {
        DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                + " Exception occur in wait(sleepTime):" + e.toString());
      }
    } // synchronized()
  }  
  
} // End of Task

// EOF
