/*
 *  Title: DaiJa_V5 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2022
 */
package task.control;

import active.Activator;
import data.Control;
import java.util.ArrayList;
import java.util.Collections;
import layer.ControlLayer;
import layer.Layer;
import task.Task;
import util.DJ;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: ReverseTuner  </p>
 * <p> 説　明: 制御対象の特性を学習したニューラルネットワークによる制御
 *             力操作量を変化させて制御対象に入力し、
 *             制御対象のデータ（力操作量、加速度、速度、位置）を収集する。
 *             収集した制御対象のデータを用いて、ニューラルネットワークを学習し、
 *             学習したニューラルネットワークで制御対象を制御する。
 *             Inverser4をベースに作成した。</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2022</p>
 * <p> 作成日: 2022.08.30</p>
 */
public class ReverseTuner extends Task {
  
  static final int TWO_LAYER = 2; // ２層
  static final int THREE_LAYER = 3; // ３層（中間層×２）
  int layerNum = THREE_LAYER; // 層数
  
  // 学習・試行制御
  int epoch = 1000; // エポックの回数
  int batchNum = 20; // 10; // バッチ数
  int interval = 20; //1; // 経過表示間隔
  
  // 学習パラメータ
  double initialCoef = 0.8; //0.2; // 重みとバイアスの初期値係数
  double eta = 0.3; //0.5; // 学習係数
  int dataNum = 1000; // １秒間のデータ数　1[s]/1[ms] = 1000 [個]

  // 制御系と制御対象のパラメータ
  int speedGain = 1280; //2048 // 比例ゲイン（速度ループゲイン）
  int integralGain = 0; // 積分ゲイン
  double inertia = 1.0; // 慣性質量［kgm^2/sec^2］
  double friction = 2.0; // 粘性摩擦[kg/sec]

  // 各層のノード数
  int inNodeNum = 2; // 入力層のノード数
  int midNodeNum = 32; //6; // 中間層のノード数
  int outNodeNum = 1; // 出力層のノード数
  
  // レイヤー構成
  private ControlLayer middleLayer0; // 中間層０
  private ControlLayer middleLayer1; // 中間層１
  private ControlLayer outputLayer; // 出力層
  
  // 誤差変数
  double squareError; // 二乗誤差
  double meanError; // 二乗誤差データの平均
  double aveError; // エポック毎の誤差の平均値
  
  // 制御対象データの収集
  static final int ST0 = 0; // 初期状態
  static final int ST1 = 1; // 状態
  static final int ST2 = 2; // 状態
  static final int ST3 = 3; // 状態
  static final int ST4 = 4; // 状態
  static final int ST5 = 5; // 状態
//  static final int ST6 = 6; // 状態
//  static final int ST7 = 7; // 状態
  static final int STE = 10; // 最終状態
  int state = ST0; //ST2; //ST7; //ST0; // 始動時の状態
  
  int routeNum = 42; //44; //42; //21; // データ収集経路数 44以下
  int sampleNum = dataNum * routeNum; // 収集データ数
  double accLmt = 0.005; //0.025; //0.1; //0.5, 1.2; // 加速度制限値
  double spdLmt = 0.01; //0.005; //0.1; //0.5; // 速度制限値
  double dfCef = 0.000125; //0.004; //0.002; // 力操作量増減定数

  
  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; // パターン・ビューワを表示する
    patternData0 = new double[10][sampleNum]; // ID,出力(力),速度,正解(速度所望値)
    patternData1 = new double[10][sampleNum]; // ID,力操作量,加速度,所望速度
    for (int patternNum = 1; patternNum < 10; patternNum++) {
      for (int k = 0; k < sampleNum; k++) {
        patternData0[patternNum][k] = 10.0; // 表示領域外に初期化
        patternData1[patternNum][k] = 10.0; // 表示領域外に初期化
      }
    }
    patternViewerLauncher = DJ.pattern( // 判定パターンとデータ
//        PatternViewer.PATTERN_TUNER, patternData0, "ReverseTuner:t-a,v,p",
        PatternViewer.PATTERN_SPRAY, patternData0, "ReverseTuner:a-v",
//        PatternViewer.PATTERN_TUNER, patternData1,  "ReverseTuner:a-v");
        PatternViewer.PATTERN_SPRAY, patternData1,  "ReverseTuner:v-p");
    
    // グラフ・ビューワ・ランチャを得る
    graphShift = 2; //7; // グラフの縦軸を移動
    graphData = new double[2];
    dataName = new String[2]; // データ名
    dataName[0] = "squareError"; // 学習時の二乗誤差
    dataName[1] = "aveError"; // 学習時の平均二乗誤差
    graphViewerLauncher = DJ.graph(epoch, interval, dataName, graphData);
    
    reverseTuner(); // タスク本体の呼び出し
  }
  
  /** 
   * 収集した制御対象のデータでニューラルネットワークを学習し、制御する。
   */
  public void reverseTuner() {
    DJ._print("DaiJa_V5, reverseTuner() ==========================");
    
    DJ._print("・パラメータ");
    DJ.print_(" 重みとバイアスの初期値係数:initialCoef=", initialCoef);
    DJ.print_(", 学習係数:eta=",eta);
    DJ.print(", エポックの回数:epoch=",epoch);
    
    DJ.print_(" バッチ数:batchNum=",batchNum);
    DJ.print_(", 経過表示間隔:interval=",interval);
    DJ.print(", １秒間のデータ数：dataNum=", dataNum);
    
    DJ.print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・各層のノード数");
    DJ.print_(" 入力層ノード数:inNodeNum=",inNodeNum);
    DJ.print_(", 中間層ノード数:midNodeNum=",midNodeNum);
    DJ.print(", 出力層ノード数:outNodeNum=",outNodeNum);
    
    DJ._print("・データ収集用パラメータ");
    DJ.print_(" データ収集経路数:routeNum=", routeNum);
    DJ.print(" 収集データ数:sampleNum=", sampleNum);
    
    DJ.print_(" 加速度制限値:accLmt=",accLmt);
    DJ.print_(", 速度制限値:spdLmt=",spdLmt);
    DJ.print(", 力操作量増減定数:dfCef=",dfCef);
    
    DJ._print("・制御系と制御対象（負荷）を設定");
    Control control = new Control(); // 制御系インスタンスを生成
    control.setParameter(0, 0, inertia, friction); // パラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl(); // 制御系と制御対象をリセット
    
    DJ._print("　入力データと正解データを定義する");
    double[][] inputData = new double[inNodeNum][sampleNum]; // 入力データ
    double[] correctData = new double[sampleNum]; // 正解データ
    DJ._print("　学習用データを定義する");
    double[] forceData = new double[sampleNum]; // 力操作量データ
    double[] accelData = new double[sampleNum]; // 加速度データ
    double[] speedData = new double[sampleNum]; // 速度データ
    double[] positionData = new double[sampleNum]; // 位置データ
    
    double inc = 1.0 / (double)dataNum; // クロックの増分
    double force = 0.0; // 力操作量
    double various; // 経路ごとの制約係数
//    double acc = 0.0; // 理論加速度
//    double spd = 0.0; // 理論速度
//    double psn = 0.0; // 理論位置
    
//    double accel; // 応答加速度
//    double speed; // 応答速度
//    double position; // 応答位置
    double oldAccel = 0.0; // 旧応答加速度
    double oldSpeed = 0.0; // 旧応答速度
    double oldPosition = 0.0; // 旧応答位置

    double df = dfCef; // 力操作量の更新量
    double f = dfCef; // 力操作量の初期値
    double clock; // 時間軸表示用
    
    int id; // 通算インデックス
        
    DJ._print("・制御対象を駆動し、学習用のデータを収集する");
    DJ.print_("状態遷移：status=", state);
    for (int j = 0; j < routeNum; j++) { // 経路パターン（全ての経路）
        clock = 0.0;
      
      for (int k = 0; k < dataNum; k++) { // 経路の繰り返し（データ1000個分）
        id = j * dataNum + k;
        
        
        
        // シーケンス操作を組み込む（状態遷移）
        switch (state) {
          case ST0: // 初期状態
//            if (oldAccel > accLmt) { // 加速度を制限
//              f = -df;
//              df = df * 1.1;
//              accLmt = accLmt * 1.05;
//              spdLmt = spdLmt * 1.05;
//              state = ST1; // 状態１
//              DJ.print_(" ,", state);
//            }
            if (oldSpeed > spdLmt) { // 速度を制限
              f = -df;
              df = df * 1.2; //1.15;
              accLmt = accLmt * 1.2; //05;
              spdLmt = spdLmt * 1.6; //1.25; //2; //05;
              state = ST1; // 状態１
              DJ.print_(" ,", state);
            }
            if (j > 5) state = ST2; // 状態２

            break;
          case ST1: // 状態１
//            if (oldAccel < -accLmt) { // 加速度を制限
//              f = df;
//              df = df * 1.1;
//              accLmt = accLmt * 1.05;
//              spdLmt = spdLmt * 1.05;
//              state = ST0; // 状態０
//              DJ.print_(" ,", state);
//            } 
            if (oldSpeed < -spdLmt) { // 速度を制限
              f = df;
              df = df * 1.2; //1.15;
              accLmt = accLmt * 1.2;
              spdLmt = spdLmt * 1.6; //1.25; //2;
              state = ST0; // 状態０
              DJ.print_(" ,", state);
            }
            if (j > 5) state = ST3; // 状態３
            break;
            
            
          case ST2: // 初期状態
//            if (oldAccel > accLmt) { // 加速度を制限
//              f = -df;
//              df = df * 1.1;
//              accLmt = accLmt * 1.05;
//              spdLmt = spdLmt * 1.05;
//              state = ST3; // 状態３
//              DJ.print_(" ,", state);
//            }
            if (oldSpeed > spdLmt) { // 速度を制限
              f = -df;
              df = df * 1.12; //1.08; //1.05; //15;
//              accLmt = accLmt * 1.2; //05;
              spdLmt = spdLmt + 0.03; //0.025; //0.018; // * 1.1; //2; //05;
              state = ST3; // 状態３
              DJ.print_(" ,", state);
            }
            break;
          case ST3: // 状態１
//            if (oldAccel < -accLmt) { // 加速度を制限
//              f = df;
//              df = df * 1.1;
//              accLmt = accLmt * 1.05;
//              spdLmt = spdLmt * 1.05;
//              state = ST2; // 状態２
//              DJ.print_(" ,", state);
//            } 
            if (oldSpeed < -spdLmt) { // 速度を制限
              f = df;
              df = df * 1.12; //1.08; //1.05; //15;
//              accLmt = accLmt * 1.2;
              spdLmt = spdLmt + 0.03; //0.025; //0.018; // * 1.1; //2;
              state = ST2; // 状態２
              DJ.print_(" ,", state);
            }
            break;
            
            
            
            
//          case ST2: // 状態２
//            if (oldSpeed > spdLmt) { // 正側速度を制限
//              state = ST4; // 状態４
//              DJ.print_(" ,", state);
//              break;
//            }
//            various = (double)(j + 1) / 21; //  +0.0 ～ +1.0
//            if (oldSpeed > spdLmt * various) { // 正側速度を制限
//              f = -df;
//              state = ST3; // 状態３
//              DJ.print_(" ,", state);
//            }
//            break;
//          case ST3: // 状態３
//            if (oldSpeed < -spdLmt) { // 負側速度を制限
//              state = ST5; // 状態５
//              DJ.print_(" ,", state);
//              break;
//            }
//            various = (double)(j + 1) / 21; //  +0.0 ～ +1.0
//            if (oldSpeed < -spdLmt * various) { // 負側速度を制限
//              f = df;
//              state = ST2; // 状態２
//              DJ.print_(" ,", state);
//            }
//            break;
//          case ST4: // 状態４
//            various = 1.0 - (double)(j - 21) / 21; //  +0.0 ～ +1.0
//            if (oldSpeed > spdLmt * various + 0.03) { // 正側速度を制限
//              f = -df;
//              state = ST5; // 状態５
//              DJ.print_(" ,", state);
//            }
//            break;
//          case ST5: // 状態５
//            various = 1.0 - (double)(j - 21) / 21; //  +0.0 ～ +1.0
//            if (oldSpeed < -spdLmt * various - 0.03) { // 負側速度を制限
//              f = df;
//              state = ST4; // 状態４
//              DJ.print_(" ,", state);
//            }
//            break;
            
//          case ST6: // 状態６
//            various = (double)(j + 1) / 21; //  +0.0 ～ +1.0
//            if (oldSpeed > spdLmt * various) { // 正側速度を制限
////            if ((oldSpeed > spdLmt * various) || // 正側速度を制限
////                (oldAccel > accLmt * various + 0.1)) { // 加速度を制限                    
//              f = +df;
//              state = ST7; // 状態７
//              DJ.print_(" ,", state);
//            }
////            if (oldSpeed > spdLmt) { // 正側速度を制限
////              state = ST4; // 状態４
////              DJ.print_(" ,", state);
////            }
//            break;
//          case ST7: // 状態７
//            various = (double)(j + 1) / 21; //  +0.0 ～ +1.0
//            if (oldSpeed < -spdLmt * various) { // 負側速度を制限
//              f = -df;
//              state = ST6; // 状態６
//              DJ.print_(" ,", state);
//            }
////            if (oldSpeed < -spdLmt) { // 正側速度を制限
////              state = ST5; // 状態５
////              DJ.print_(" ,", state);
////            }
//            break;
            
          case STE: // 最終状態
          default:
            DJ.print_(" ,", state);
//            break;
            return;
        }
    
        force = force + f;


        // DJ._print("・制御対象から入力（力操作量）と出力（応答）を収集する");
//        acc = force; // 加速度参照値（M=1[kg]を仮定）
//        spd = spd + acc / (double)dataNum; // 速度参照値
//        inputData[j][k] = spd;
//        psn = psn + spd / (double)dataNum; // 位置参照値
//        correctData[j][k] = psn; 
                
        // 制御対象を１回実行する
        forceData[id] = force / Control.DAC_FACTOR; // 力操作量
        control.setForce(forceData[id]); // 力の操作量を制御対象に出力
        control.execute(); // 制御対象の実行
        
        // 実行結果を読み出す
        accelData[id] = control.getAcceleration() * 1000.0; // 応答加速度[m/sec^2]
        oldAccel = accelData[id]; // 旧応答加速度
        speedData[id] = control.getSpeed(); // 応答速度[m/sec]
        oldSpeed = speedData[id];
        positionData[id] = control.getPosition(); // 応答位置[m]
        oldPosition = positionData[id];
        
        inputData[0][id] = accelData[id]; // 応答加速度
        inputData[1][id] = speedData[id]; // 応答速度
        correctData[id] = forceData[id]; // 力操作量
        
        
        patternData0[0][id] = accelData[id]; // 所望の加速度 // 2頁目横軸
//        patternData0[1][id] = 10.0; // 非表示
//        patternData0[2][id] = 10.0; // spd; // 所望の速度
//        patternData0[3][id] = 10.0; // force; // 所望の力操作量（指令値）
//        patternData0[4][id] = 10.0; //  control.getAcceleration() * 2000.0; // 応答加速度
        patternData0[5][id] = speedData[id]; // 応答速度
//        patternData0[6][id] = 10.0; // positionData[j][k] * 4.0; // 応答位置
//        patternData0[7][id] = 10.0; // acc * 2.0; // 所望の加速度
//        patternData0[8][id] = 10.0; // 応答位置
///        patternData0[9][id] = positionData[id]; // 応答位置
        
//        patternData1[0][id] = clock; // 時間軸用クロック
//        patternData1[1][id] = force;// 所望の力操作量（指令値）
        patternData1[0][id] = speedData[id]; // 応答速度
//        patternData1[1][id] = 10.0; // 非表示
//        patternData1[2][id] = 10.0; // 非表示
//        patternData1[3][id] = 10.0; // 非表示
//        patternData1[4][id] = 10.0; // 非表示
//        patternData1[5][id] = 10.0; // 非表示
        if (j < routeNum / 2)
          patternData1[6][id] = positionData[id]; // 所望の位置
        else
          patternData1[6][id] = 10.0; // 非表示（必須）
//        patternData1[7][id] = 10.0; // 非表示
//        patternData1[8][id] = 10.0; // 非表示
        if (j < routeNum / 2)
          patternData1[9][id] = 10.0; // 非表示（必須）
        else
          patternData1[9][id] = positionData[id]; // 所望の位置

        clock = clock + inc; //クロックを更新
      } // k
      
    } // j
    DJ._print("最終状態:Last status=", state);
    
    updatePattern(); // パターン表示を更新
    
    DJ._print("・収集データ");
    //# DJ.print("・応答加速度：inputData0=", accelData); //  応答加速度
    //# DJ.print("・応答速度：inputData1=", speedData); // 応答速度
    //# DJ.print("・力操作量：correctData=", forceData); // 力操作量
    

    DJ._print(" ############################################################");
    DJ.print(" ##");
    DJ.print(" ##      ・制御対象の逆関数の学習を開始するには、             ");
    DJ.print(" ##      コンソールの「Run/Pause」ボタンをクリックしてください。");
    DJ.print(" ##");
    DJ.print(" ############################################################");

    pauseFlag = true;
    synchronized (this) {
      try { if (pauseFlag) wait(); // 休止状態
      } catch (InterruptedException e) {
        DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                + " Exception occur in wait(sleepTime):" + e.toString());
      }
    } // synchronized()
    
    DJ._print(" ##### 制御対象の逆関数の学習を開始 #####");
    DJ.print(" 順序をランダムにシャッフルしたインデックスのリスト");
    ArrayList<Integer> indexList = DJ.permutationRandom(sampleNum); 
    //# DJ.printList("・ランダム・インデックス：indexList=", indexList); //  応答加速度

    DJ._print("・ニューラルネットの各層の初期化");
    middleLayer0 = new ControlLayer(inNodeNum, midNodeNum);
//$    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
//$    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.ADA_GRAD);
    if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
      middleLayer0.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
    else // ２層：中間１層を飛ばして中間０層と出力層でデータを受け渡しする
      middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
    double[] mid0X = middleLayer0.getX();
    double[] mid0Y = middleLayer0.getY();
    // double[] mid0C = middleLayer0.getC();
    // double[] mid0E = middleLayer0.getE();
    // double[] mid0dEdX = middleLayer0.getdEdX();
    //# double[] mid0dEdY = middleLayer0.getdEdY(); // 参照を代入（注１）

    double[] mid1Y;
    double[] mid1dEdX;
    middleLayer1 = new ControlLayer(midNodeNum, midNodeNum);
//$    middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
//$    middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.ADA_GRAD);
    if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
      middleLayer1.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
    else // ２層：中間１層を飛ばして中間０層と出力層でデータを受け渡しする
      middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
    //# double[] mid1X = middleLayer1.getX(); // 参照を代入（注３）
    mid1Y = middleLayer1.getY();
    // double[] mid1C = middleLayer1.getC();
    // double[] mid1E = middleLayer1.getE();
    mid1dEdX = middleLayer1.getdEdX();
    //# double[] mid1dEdY = middleLayer1.getdEdY(); // 参照を代入（注２）

    outputLayer = new ControlLayer(midNodeNum, outNodeNum);
//    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.SGD);
    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.ADA_GRAD);
    //# double[] outX = outputLayer.getX(); // 参照を代入（注４）
    double[] outY = outputLayer.getY();
    double[] outC = outputLayer.getC();
    double[] outE = outputLayer.getE();
    double[] outdEdX = outputLayer.getdEdX();
    // double[] outdEdY = outputLayer.getdEdY();

    double[] outS = new double[outC.length]; // 応答速度

//$    middleLayer0.setdEdY(mid1dEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
//$    middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
//$    middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
//$    outputLayer.setX(mid1Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
//$    
//$    middleLayer0.setdEdY(outdEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
//$    // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
//$    // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
//$    outputLayer.setX(mid0Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    
    if (layerNum == THREE_LAYER) { // ３層：中間１層を使用する
      middleLayer0.setdEdY(mid1dEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
      middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
      middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
      outputLayer.setX(mid1Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    }
    else { // ２層：中間１層を飛ばして中間０層と出力層でデータを受け渡しする
      middleLayer0.setdEdY(outdEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
      //# middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
      //# middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
      outputLayer.setX(mid0Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    }
    
    // DJ._print("PatternViewerへの参照を得る");
    patternViewer = patternViewerLauncher.getPatternViewer();
    // DJ._print("GraphViewerへの参照を得る");
    graphViewer = graphViewerLauncher.getGraphViewer();
    if (graphViewer != null) graphViewer.shiftGraphAxis(graphShift); // グラフの縦軸をシフトする
    
    DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | 
              (i == epoch); //経過表示フラグ

      // DJ._print("・学習用データのインデックスをシャッフル");
      Collections.shuffle(indexList);
      
      // DJ.print("・学習誤差と試行誤差の初期化");
      squareError = 0.0; // 二乗誤差
      meanError = 0.0; // 二乗誤差データの平均
    
//$      // 制御をリセット
//$      control.resetControl();

        clock = 0.0;

      // 全データを学習
//#      for (int j = 0; j < dataNum; j++) {
      for (int j = 0; j < sampleNum; j++) {
        // DJ._print("・エポックの実行ループ開始 ----------------------------");
        // DJ.print(" エポック:i=" + i + ", 学習データ:j=" + j);
        
        int index = indexList.get(j); // ランダム・インデックス
//##        int index = j; // 非ランダム・インデックス
        //# DJ._print(" 学習用データID：learnIndex=", index);
        
        // DJ._print(" ##### 前向き伝播の呼び出し #####");
        // DJ._print("・入力データを取り出し");
//**/        mid0X[0] = control.getAcceleration() * 100.0; // 入力データ（応答加速度）
//**/        mid0X[1] = correctData[index]; // 正解データ（所望速度）
//**/        mid0X[1] = control.getSpeed(); // 正解データ（応答速度）
        mid0X[0] = inputData[0][index]; // 入力データ（応答加速度）
        mid0X[1] = inputData[1][index]; // 入力データ（応答速度）
        
        middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
        if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
          middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
        outputLayer.forward(); // 出力層の順伝播処理を呼び出し
        
//$        // DJ._print(" 制御対象の実行（力操作量　--->　応答速度）");
//        // DJ._print(" 推定された力操作量を制御対象（負荷）に与える");
//        control.setForce(outY[0] / Control.DAC_FACTOR); // 力の操作量
//        control.execute(); // 制御対象の実行
////**/        outS[0] = control.getSpeed(); // 応答速度;
///**/        outS[0] = control.getAcceleration() * 100.0; // 応答加速度;
        
        // DJ._print(" ##### 後向き伝播の呼び出し #####");
//**/        outC[0] = correctData[index]; // 正解データを取出し（所望速度）
//**/        outC[0] = inputData[index] * 100.0; // 正解データを取出し（所望加速度）
        
        outS[0] = outY[0]; // 力操作量;
        outC[0] = correctData[index]; // 正解データを取出し（力操作量）
        
        //# outputLayer.backward(outC); // 出力層の逆伝播処理を呼び出し
        outputLayer.backward(outC, outS); // 出力層の逆伝播処理を呼び出し（逆関数の学習時）
        if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
          middleLayer1.backward(); // 中間層１の逆伝播処理を呼び出し
        middleLayer0.backward(); // 中間層０の逆伝播処理を呼び出し
        
        // DJ._print("・バッチ処理");
        if ((j % batchNum) == 0 ) {
          // DJ._print(" ##### 更新の呼び出し #####");
          middleLayer0.update(); // 中間層０の更新処理を呼び出し
          if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
            middleLayer1.update(); // 中間層１の更新処理を呼び出し
          outputLayer.update(); // 出力層の更新処理を呼び出し
        }

        // DJ._print(" ##### 学習時の誤差の保存 #####");
        if (intervalFlag) { // 実行時間に影響を与える
          // 二乗誤差の算出
          squareError = outE[0] * outE[0] / 2.0;
          meanError = meanError + squareError;
        
          // パターン表示用データの保存
          patternData0[0][j] = clock; // 時間軸用クロック
          patternData0[1][j] = outC[0] * 0.002; //0.06; // 正解データ（力操作量)
          patternData0[2][j] = outS[0] * 0.002; //0.06; // 推定値（力操作量）
          patternData0[3][j] = 10.0; // ダミー
          patternData0[4][j] = 10.0; // outS[0] * 0.002; // 推定値（力操作量）
          patternData0[5][j] = 10.0; // ダミー
          patternData0[6][j] = 10.0; // ダミーpositionData[j]; // 所望位置
          patternData0[7][j] = 10.0; // outC[0] * 0.002; // 正解データ（力操作量)
          patternData0[8][j] = 10.0; // ダミー
          patternData0[9][j] = 10.0; // ダミー
          
          patternData1[0][j] = clock; // 時間軸用クロック
          patternData1[1][j] = 10.0; // ダミー
          patternData1[2][j] = 10.0; // ダミー 
          patternData1[3][j] = 10.0; // ダミー
          patternData1[4][j] = accelData[j]; // 所望加速度
          patternData1[5][j] = 10.0; // ダミー
          patternData1[6][j] = 10.0; // ダミー
          patternData1[7][j] = 10.0; // ダミー
          patternData1[8][j] = speedData[j]; // 所望速度
          patternData1[9][j] = 10.0; // ダミー

          clock = clock + inc; //クロックを更新
          if (clock > 1.0) clock = 0.0;
        }
        
        // DJ._print(" End of one data ---------------");
      } // 全データを学習
      // DJ._print(" End of one epoch ---------------");

      
      // DJ._print("　エポック毎の誤差の平均値を保存する");
//#      aveError = meanError / dataNum ;
      aveError = meanError / sampleNum ;
      
      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

       // 経過表示インターバル
      if (intervalFlag) { // 実行時間に影響を与える
        // DJ._print(" ##### １エポックの実行結果 #####");
        DJ.print_(" i=" + i);
        DJ.print_(", squareError = ", squareError);
        DJ.print(", aveError = ", aveError);

        // グラフ表示用データを代入
        graphData[0] = squareError;
        graphData[1] = aveError;
        updateViewer(i); // ビューワの表示を更新
        
        DJ.print_(", lapTime = ", lapTime); DJ.print("[msec]");
        
        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            wait(SLEEP_TIME); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
    
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      // DJ._print(" End of one epoch ---------------------------------------");
    }
    DJ._print(" End of all epoch --------------------------------------------");

    
//    DJ._print(" ##### End of debugging #####");
//
//    if(true) return; // $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$
    
    
    DJ._print("・エポック実行回数");
    DJ.print(" Last epoch = ", epoch);
    DJ.print_("・最終誤差: ");
    for (int k = 0; k < graphData.length; k++) {
      DJ.print_("  " + dataName[k] + "=" + graphData[k]);
    }
    DJ.print("");
    

    DJ._print(" ############################################################");
    DJ.print(" ##");
    DJ.print(" ##      ・学習結果を検証するには、             ");
    DJ.print(" ##      コンソールの「Run/Pause」ボタンをクリックしてください。");
    DJ.print(" ##");
    DJ.print(" ############################################################");

    pauseFlag = true;
    synchronized (this) {
      try { if (pauseFlag) wait(); // 休止状態
      } catch (InterruptedException e) {
        DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                + " Exception occur in wait(sleepTime):" + e.toString());
      }
    } // synchronized()
    
    
    DJ._print("・検証：学習した逆関数で制御対象を駆動し、応答速度を比較する");
    DJ._print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・検証：学習した逆関数で制御対象を駆動し、応答速度を比較する");
    control.setParameter(0, 0, inertia, friction); // パラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl();
    
    for (int k = 0; k < 10; k++) {
      for (int j = 0; j < sampleNum; j++) {
        patternData0[k][j] = 10.0; // ダミー
        patternData1[k][j] = 10.0; // ダミー
      }
    }
    
    int extra = 500; // 時間軸の追加表示
//    dataNum = dataNum + extra;
    double[] accelCmd = new double[dataNum + extra]; // 加速度目標値
    double[] speedCmd = new double[dataNum + extra]; // 速度目標値
    double[] positionCmd = new double[dataNum + extra]; // 位置目標値
    
    accelCmd[0] = 0.0;
    speedCmd[0] = 0.0;
    positionCmd[0] = 0.0;
    clock = 0.0;
    patternData0[0][0] = clock;
    patternData1[0][0] = clock;
    
    double spd = 0.0; // 速度目標値生成用補助変数
    double psn = 0.0; // 位置目標値生成用補助変数

//    int latency = 1; // 遅延時間, 1,2
//    for (int j = latency; j < dataNum + extra; j++) {
//          clock = clock + inc; //クロックを更新
//          
//      // 速度経路（加速度一定）
//      if (clock < 0.1) {
//        accelCmd[j] =  0.0; // 加速度目標値
//        speedCmd[j] = 0.0; // 速度目標値
//        positionCmd[j] = 0.0; // 位置目標値
//      }
//      else if (clock < 0.7) {
//        speedCmd[j] = (clock - 0.1) * 0.2; // 速度目標値
//        accelCmd[j] = (speedCmd[j] - speedCmd[j - 1]) * dataNum; // 加速度目標値
//        positionCmd[j] = positionCmd[j - 1] + speedCmd[j] / dataNum ;
//      }
//      else {
//        speedCmd[j] = 0.6 * 0.2; // 速度目標値
//        accelCmd[j] = (speedCmd[j] - speedCmd[j - 1]) * dataNum; // 加速度目標値
//        positionCmd[j] = positionCmd[j - 1] + speedCmd[j] / dataNum ;
//      }
//    } // for j loop
      
//    int latency = 1; // 遅延時間, 1,2
//    for (int j = latency; j < dataNum + extra; j++) {
//          clock = clock + inc; //クロックを更新
//      // 台形状の速度を与える場合
//      if (clock < 0.3)  spd = spd + 0.0001;
//      else if (clock < 0.7) spd = spd + 0.0;
//      else if (clock < 1.0) spd = spd - 0.0001;
//      else spd = 0.0;
//      speedCmd[j] = spd;
//      accelCmd[j] = (speedCmd[j] - speedCmd[j - 1]) * dataNum; // 加速度目標値
//      positionCmd[j] = positionCmd[j - 1] + speedCmd[j] / dataNum ;
//    } // for j loop

    int latency = 0; // 遅延時間
    for (int j = latency; j < dataNum + extra; j++) {
          clock = clock + inc; //クロックを更新

      // 加速度目標値（三角波）を生成
      if (clock < 0.5) 
        accelCmd[j] = (+1.0 - Math.cos(4.0 * Math.PI * clock)) * 0.1; //2.0;
      else 
        accelCmd[j] = (-1.0 + Math.cos(4.0 * Math.PI * clock)) * 0.1; //2.0;
      // 速度目標値（所望速度）を生成
      spd = spd + accelCmd[j] / dataNum; // 1000.0;
      speedCmd[j] = spd;
      // 位置目標値を生成
      psn = psn + speedCmd[j] / dataNum; // 1000.0;
      positionCmd[j] = psn;
      
    } // for j loop
      
    clock = 0.0;
    for (int j = latency; j < dataNum + extra; j++) {
          clock = clock + inc; //クロックを更新
      
//      correctData[j] = speedCmd[j] * 0.1; // 所望速度
          
      mid0X[0] = accelCmd[j-latency] * 0.1; // 0.05; // 加速度目標値
      mid0X[1] = speedCmd[j-latency] * 0.1; // 0.05; // 所望速度
      middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
      if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
        middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
      outputLayer.forward(); // 出力層の順伝播処理を呼び出し
      
      force = outY[0] / Control.DAC_FACTOR; // ＮＮで算出された力操作量
      control.setForce(force * 1.0); //0.985); // 0.92; // 力の操作量を制御対象に出力
      control.execute(); // 制御対象の実行

      patternData0[0][j] = clock; // クロック
      patternData0[1][j] = 10.0; // ダミー
      patternData0[2][j] = outY[0] * 0.2;// ＮＮで算出された力操作量
      patternData0[3][j] = 10.0; // ダミー
      patternData0[4][j] = control.getAcceleration() * 200.0;//50.0; // 応答加速度
      patternData0[5][j] = control.getSpeed() * 0.2; //0.05; // 応答速度
      patternData0[6][j] = control.getPosition() * 0.8; //0.05; // 応答位置
      patternData0[7][j] = accelCmd[j] * 4.0; // 加速度目標値
      patternData0[8][j] = speedCmd[j] * 4.0; // 所望の加速度
      patternData0[9][j] = positionCmd[j] * 16.0; //8.0; // 位置目標値
      
      patternData1[0][j] = clock; // クロック
      patternData1[1][j] = 10.0; // ダミー
      patternData1[2][j] = outY[0] * 0.1;// ＮＮで算出された力操作量
      patternData1[3][j] = 10.0; // ダミー
      patternData1[4][j] = control.getAcceleration() * 100.0;//50.0; // 応答加速度
      patternData1[5][j] = control.getSpeed() * 0.1; //0.05; // 応答速度
      patternData1[6][j] = 10.0; // control.getPosition() * 0.4; //0.05; // 応答位置
      patternData1[7][j] = accelCmd[j] * 2.0; // 加速度目標値
      patternData1[8][j] = speedCmd[j] * 2.0; // 所望の加速度
      patternData1[9][j] = 10.0; // positionCmd[j] * 4.0; // 位置目標値
    }

    updatePattern(); // パターン表示を更新
    
    DJ._print_("・検証結果  ");
    DJ.print("収集データ数: sampleNum=", sampleNum);
//    DJ.print("力操作量の推定値: f~", patternData1[1]);
//    DJ.print("応答速度: v", patternData1[5]);
    
    // 処理時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print_("・タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
    DJ.print(", タスク終了日時：", TimeStamp.getTimeFormated());
    
    DJ._print("##### ReverseTuner 終了 #####");


  } // reverseTuner()

} // ReverseTuner Class

// End of file
