/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package task.control;

import active.Activator;
import data.Control;
import java.util.ArrayList;
import java.util.Collections;
import layer.ControlLayer;
import layer.Layer;
import task.Task;
import util.DJ;
import util.LogEditor;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: TargetLearner</p>
 * <p> 説　明: 制御対象を学習し、入力する力操作量を変えて検証する。
 *      力操作量と速度を入力し、加速度を出力する。速度と位置も求める。
 *      推定加速度と応答加速度の誤差で学習する。
 *      台形、Ｓ字、折線状の力操作量を用いる。</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.08.17</p>
 */
public class TargetLearner extends Task {

  // 学習・試行制御
  int epoch = 1000; // エポックの回数
  int batchNum = 1; // 10; // バッチ数
  int interval = 10; // 経過表示間隔
  
  // 学習パラメータ
  double initialCoef = 0.5; // 重みとバイアスの初期値係数
  double eta = 0.2; // 学習係数
  int dataNum = 1000; // エポック毎の学習データ数

  // 制御系と制御対象のパラメータ
  Control control; // 制御系と制御対象
  int speedGain = 1280; //2048 // 比例ゲイン（速度ループゲイン）
  int integralGain = 0; // 積分ゲイン
  double inertia = 1.0; // 慣性質量［kgm^2/sec^2］
  double friction = 2.0; // 粘性摩擦[kg/sec]

  // 各層のノード数
  int inNodeNum = 2; // 入力層のノード数
  int midNodeNum = 8; // 中間層のノード数
  int outNodeNum = 1; // 出力層のノード数
  
  // レイヤー構成
  private ControlLayer middleLayer0; // 中間層０
  //#  private ControlLayer middleLayer1; // 中間層１
  private ControlLayer outputLayer; // 出力層
  
  // 誤差変数
  double squareError; // 二乗誤差
  double meanError; // 二乗誤差データの平均
  double aveError; // エポック毎の誤差の平均値
        
  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; // パターン・ビューワを表示する
    patternData0 = new double[10][dataNum]; // ID,力,加速度,速度,位置
    patternData1 = new double[10][dataNum]; // ID,力操作量,加速度,所望速度
    patternViewerLauncher = DJ.pattern( // 判定パターンとデータ
        PatternViewer.PATTERN_TUNER, patternData0, "TargetLearn:Learning",
        PatternViewer.PATTERN_TUNER, patternData1, "TargetLearn:Trial");
    
    // グラフ・ビューワ・ランチャを得る
    graphShift = 8; // グラフの縦軸を移動
    graphData = new double[2];
    dataName = new String[2]; // データ名
    dataName[0] = "squareError"; // 学習時の二乗誤差
    dataName[1] = "aveError"; // 学習時の平均二乗誤差
    graphViewerLauncher = DJ.graph(epoch, interval, dataName, graphData);
    
    targetLearner(); // タスク本体の呼び出し
  }
  
  /** 
   *  制御対象を学習し、入力する力操作量を変えて検証する。
   */
  public void targetLearner() {
    DJ._print("DaiJa_V3, TargetLearn.targetLearner() ========================");
    
    DJ._print("・パラメータ");
    DJ.print_(" 重みとバイアスの初期値係数:initialCoef=", initialCoef);
    DJ.print_(", 学習係数:eta=",eta);
    DJ.print(", エポックの回数:epoch=",epoch);
    
    DJ.print_(" バッチ数:batchNum=",batchNum);
    //#    DJ.print_(", ドロップ・アウト率:dropOutRate=",dropOutRate);
    DJ.print_(", 経過表示間隔:interval=",interval);
    DJ.print(", 学習データ数：dataNum=", dataNum);
    
    DJ.print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・各層のノード数");
    DJ.print_(" 入力層ノード数:inNodeNum=",inNodeNum);
    DJ.print_(", 中間層ノード数:midNodeNum=",midNodeNum);
    DJ.print(", 出力層ノード数:outNodeNum=",outNodeNum);
    
    DJ._print("・制御系と制御対象（負荷）を設定");
    control = new Control();
    control.setParameter(0, 0, inertia, friction); // 制御対象のパラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl();
    
    // DJ._print("入力データと教師データを作成");
    double[] inputData = new double[dataNum]; // 入力データ
    double[] correctData = new double[dataNum]; // 正解データ
    //    double[] acceleration = new double[dataNum]; // 加速度データ
    
// ファイルから実行データを読込まず、制御対象から直接収集する場合
//    DJ._print("・制御対象の実行データを収集");
//    double[][] sampleData = new double[5][dataNum]; // 実行データ収集
//    DJ.print_(", 行数：" + sampleData.length + "（時点,操作量,応答）");
//    DJ.print(", 列数：", sampleData[0].length + "（選点）");
//    
//    DJ._print("・入力データと教師データを作成");
//    double f = 0.0; // 力操作量
//    double inc = 1.0 / (double)dataNum;
//    double clock = -inc;
//    for (int j = 0; j < dataNum; j++) {
//      clock = clock + inc;
///*
//      // （１）力操作量を与える（台形状）
//      int cycle = dataNum / 3;
//      if (j < cycle) f = f + 2.0; // 入力データ（力の操作量）
//      else if (j < cycle * 2) f = f + 1.0; 
//      else f = f - 1.05;
//*/ /*
//      // （２）力操作量を与える(Ｓ字状)
//      f = (-2 * clock + 3 )* clock * clock; // 力の操作量
//      f = f * 500.0; // スケーリング
//*/ /**/
//      // （３）力操作量を与える（折線ジグザグ状）
//      int cycle = dataNum / 5;
//      if (j < cycle) f = f + 2.5; // 入力データ（力の操作量）
//      else if (j < cycle * 2) f = f - 3.0; 
//      else if (j < cycle * 3) f = f + 1.5; 
//      else if (j < cycle * 4) f = f - 2.0; 
//      else f = f + 0.75;
///**/
//      sampleData[0][j] = clock;
//      sampleData[1][j] = f; // 力操作量
//      control.setForce(sampleData[1][j]);
//      control.execute();
//      sampleData[2][j] = control.getAcceleration(); // 応答加速度
//      sampleData[3][j] = control.getSpeed(); // 応答速度
//      sampleData[4][j] = control.getPosition(); // 応答位置
//      
//      inputData[j] = clock; // 入力データ
///*a*/      correctData[j] = sampleData[2][j] * 100.0; // 応答加速度(正解データ)
////*a*/      correctData[j] = sampleData[2][j] * 10.0; // 応答加速度(正解データ)
/////*v*/      correctData[j] = sampleData[3][j]; // 応答速度(正解データ)
////*v*/      correctData[j] = sampleData[3][j] * 0.1; // 応答速度(正解データ)
//      // 正解データが小さいほど収束が速い
//      
//      patternData0[0][j] = clock; // パターン表示（0頁目）の時間軸
//      patternData1[0][j] = clock; // パターン表示（1頁目）の時間軸
//    }
    
    DJ._print("・収集データををファイルから読込");
    DJ.print(" DaiJa/resorce/V3_SampleData.dat, 5[種類]×1000[点]");
    double[][] sampleData = LogEditor.loadData("DaiJa-V4_SampleData.dat");
    DJ.print_(", 行数：" + sampleData.length + "（時点,操作量,応答）");
    DJ.print(", 列数：", sampleData[0].length + "（データ数）");
    
    DJ.print_(" クロック: ");
    DJ.print("clock", sampleData[0]);
    DJ.print_(" 力操作量: ");
    DJ.print("fm", sampleData[1]);
    DJ.print_(" 応答加速度: ");
    DJ.print("a", sampleData[2]);
    DJ.print_(" 応答速度: ");
    DJ.print("v", sampleData[3]);
    DJ.print_(" 応答位置: ");
    DJ.print("p", sampleData[4]);
    
    DJ._print("・入力データと教師データを作成");
//    double f = 0.0; // 力操作量
    double inc = 1.0 / (double)dataNum;
    double clock = -inc;
    for (int j = 0; j < dataNum; j++) {
      inputData[j] = sampleData[0][j]; // 入力データ
/*a*/      correctData[j] = sampleData[2][j] * 100.0; // 応答加速度(正解データ)
//*a*/      correctData[j] = sampleData[2][j] * 10.0; // 応答加速度(正解データ)
///*v*/      correctData[j] = sampleData[3][j]; // 応答速度(正解データ)
//*v*/      correctData[j] = sampleData[3][j] * 0.1; // 応答速度(正解データ)
      // 正解データが小さいほど収束が速い
      
      patternData0[0][j] = sampleData[0][j]; // パターン表示（0頁目）の時間軸
      patternData1[0][j] = sampleData[0][j]; // パターン表示（1頁目）の時間軸
    }
    
    // DJ.print(" 順序をランダムにシャッフルしたインデックスのリスト");
/*sh*/    ArrayList<Integer> indexList = DJ.permutationRandom(dataNum);
    // DJ.print("inputData", inputData);
    
    DJ._print("・ニューラルネットの各層の初期化");
    middleLayer0 = new ControlLayer(inNodeNum, midNodeNum);
/**/    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
//**/    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.ADA_GRAD);
    double[] mid0X = middleLayer0.getX();
    double[] mid0Y = middleLayer0.getY();
    // double[] mid0C = middleLayer0.getC();
    // double[] mid0E = middleLayer0.getE();
    // double[] mid0dEdX = middleLayer0.getdEdX();
    //# double[] mid0dEdY = middleLayer0.getdEdY(); // 参照を代入（注１）

    // middleLayer1 = new ControlLayer(midNodeNum, midNodeNum);
    // middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
    ////  middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.ADA_GRAD);
    //# double[] mid1X = middleLayer1.getX(); // 参照を代入（注３）
    // double[] mid1Y = middleLayer1.getY();
    // double[] mid1C = middleLayer1.getC();
    // double[] mid1E = middleLayer1.getE();
    // double[] mid1dEdX = middleLayer1.getdEdX();
    //# double[] mid1dEdY = middleLayer1.getdEdY(); // 参照を代入（注２）

    outputLayer = new ControlLayer(midNodeNum, outNodeNum);
/**/    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.SGD);
//**/    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.ADA_GRAD);
    //# double[] outX = outputLayer.getX(); // 参照を代入（注４）
    double[] outY = outputLayer.getY();
    double[] outC = outputLayer.getC();
    double[] outE = outputLayer.getE();
    double[] outdEdX = outputLayer.getdEdX();
    // double[] outdEdY = outputLayer.getdEdY();

    //#    double[] outS = new double[outC.length]; // 正解（所望速度）

    // middleLayer0.setdEdY(mid1dEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
    // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
    // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
    // outputLayer.setX(mid1Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    
    middleLayer0.setdEdY(outdEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
    // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
    // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
    outputLayer.setX(mid0Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    
    // DJ._print("PatternViewerへの参照を得る");
    patternViewer = patternViewerLauncher.getPatternViewer();
    // DJ._print("GraphViewerへの参照を得る");
    graphViewer = graphViewerLauncher.getGraphViewer();
    if (graphViewer != null) graphViewer.shiftGraphAxis(graphShift); // グラフの縦軸をシフトする
    
    DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | 
              (i == epoch); //経過表示フラグ

      // DJ._print("・学習用データのインデックスをシャッフル");
/*sh*/      Collections.shuffle(indexList);
      
      // DJ.print("・学習誤差と試行誤差の初期化");
      squareError = 0.0; // 二乗誤差
      meanError = 0.0; // 二乗誤差データの平均
      
      // 推定値
      double acceleration[] = new double[dataNum]; // 推定加速度
      double speed[] = new double[dataNum]; // 推定速度
      double position[] = new double[dataNum]; // 推定位置
    
      // 制御をリセット
      control.resetControl();
      
      // 全データを学習
      for (int j = 0; j < dataNum; j++) {
        // DJ._print("・エポックの実行ループ開始 ----------------------------");
        // DJ.print(" エポック:i=" + i + ", 学習データ:j=" + j);
        
        // DJ._print(" ##### 前向き伝播の呼び出し #####");
/*sh*/        int index = indexList.get(j); // ランダム・インデックス
//*ns*/        int index = j; // 非ランダム・インデックス
        // DJ._print(" 学習用データID：learnIndex=", learnIndex);
        mid0X[0] = sampleData[1][index] * 0.002; // 入力データ（力操作量）
        mid0X[1] = sampleData[3][index]; // 入力データ（速度）
        //#        mid0X[2] = patternData1[4][index]; // 入力データ（位置）
        middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
        //#        middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
        outputLayer.forward(); // 出力層の順伝播処理を呼び出し
        
        // DJ._print(" ##### 後向き伝播の呼び出し #####");
        outC[0] = correctData[index]; // 正解データ（応答加速度）
        //#        outC[1] = patternData1[3][index]; // 正解データ（応答速度）
        //#        outC[2] = patternData1[4][index]; // 正解データ（応答位置）
        outputLayer.backward(outC); // 出力層の逆伝播処理を呼び出し
        //#        outputLayer.backward(outC, outS); // 出力層の逆伝播処理を呼び出し
        //#        middleLayer1.backward(); // 中間層１の逆伝播処理を呼び出し
        middleLayer0.backward(); // 中間層０の逆伝播処理を呼び出し
        
        // DJ._print("・バッチ処理");
        if ((j % batchNum) == 0 ) {
          // DJ._print(" ##### 更新の呼び出し #####");
          middleLayer0.update(); // 中間層０の更新処理を呼び出し
          //#          middleLayer1.update(); // 中間層１の更新処理を呼び出し
          outputLayer.update(); // 出力層の更新処理を呼び出し
        }
        
/*a*/        acceleration[index] = outY[0]; // 推定加速度を保存
//*v*/        speed[index] = outY[0]; // 推定速度を保存
        
        // DJ._print(" ##### 学習時の誤差の保存 #####");
        if (intervalFlag) { // 実行時間に影響を与える
          // 二乗誤差の算出
          squareError = outE[0] * outE[0] / 2.0;
          meanError = meanError + squareError;
          
          // パターン表示用データの保存
          patternData0[1][index] = sampleData[1][index] * 0.001; // 力操作量
//*a*/          patternData0[2][index] = acceleration[index]; // 加速度(推定値)・赤
          patternData0[3][index] = sampleData[2][index] * 100.0; // 応答加速度・黒
          patternData0[4][index] = sampleData[3][index]; // 応答速度
          patternData0[5][index] = sampleData[4][index] * 2.0; // 応答位置
          
          
          patternData0[6][index] = -10.0; // ダミー
          patternData0[7][index] = -10.0; // ダミー
//          patternData0[8][index] = -10.0; // 速度（推定値）
//          patternData0[9][index] = -10.0; // 位置（推定値）
        }
        
        // DJ._print(" End of one data ---------------");
      } // 全データを学習
      // DJ._print(" End of one epoch ---------------");
      
      // DJ._print("　エポック毎の誤差の平均値を保存する");
      aveError = meanError / dataNum ;
      
      // 速度と位置の疑似推定値を求め、パターン表示用データに保存
      for (int j = 0; j < dataNum; j++) {
        // DJ._print("速度、位置の推定値を求める");
/*a*/        if (j > 0) speed[j] = speed[j - 1] + acceleration[j] / (double)dataNum;
        
//*a*/        patternData0[2][j] = acceleration[j] * 10.0; // 推定加速度・赤
/*a*/        patternData0[2][j] = acceleration[j]; // 推定加速度・赤
//*v*/        if (j > 0) acceleration[j] = speed[j] - speed[j - 1];
//*v*/        patternData0[2][j] = acceleration[j] * 1000.0; // 推定加速度・赤

//*a*/        patternData0[8][j] = speed[j] * 100.0; // 推定速度
/*a*/        patternData0[8][j] = speed[j] * 10.0; // 推定速度
//*v*/        patternData0[8][j] = speed[j] * 10.0; //1.0; // 推定速度
        
        if (j > 0) position[j] = position[j - 1] + speed[j] / (double)dataNum;
//*a*/        patternData0[9][j] = position[j] * 200.0; // 推定位置
/*a*/        patternData0[9][j] = position[j] * 20.0; // 推定位置
//*v*/        patternData0[9][j] = position[j] * 20.0; //2.0; // 推定位置
      }

      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

       // 経過表示インターバル
      if (intervalFlag) { // 実行時間に影響を与える
        // DJ._print(" ##### １エポックの実行結果 #####");
        DJ.print_(" i=" + i);
        DJ.print_(", squareError = ", squareError);
        DJ.print(", aveError = ", aveError);

        // グラフ表示用データを代入
        graphData[0] = squareError;
        graphData[1] = aveError;
        updateViewer(i); // ビューワの表示を更新
        
        DJ.print_(", lapTime = ", lapTime); DJ.print("[msec]");
        
        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            wait(SLEEP_TIME); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
    
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      // DJ._print(" End of one epoch ---------------------------------------");
    }
    DJ._print(" End of all epoch --------------------------------------------");
    
    DJ._print("・エポック実行回数");
    DJ.print(" Last epoch = ", epoch);
    DJ.print_("・最終誤差: ");
    for (int k = 0; k < graphData.length; k++) {
      DJ.print_("  " + dataName[k] + "=" + graphData[k]);
    }
    DJ.print("");
    
    DJ._print("・学習した重みとバイアスをファイルに保存する");
    // 諸元の書き込み
    String[] element = new String[1]; //  NNの諸元
    element[0] = "TargetLearner:" + TimeStamp.getTimeFormated(); // 表題
    // 中間層
    double[][] mid0W = middleLayer0.getW(); // 中間０層の重み
    double[][] mid0B = new double[1][];
    mid0B[0] = middleLayer0.getB(); // 中間０層のバイアス
    // 出力層
    double[][] outW = outputLayer.getW(); // 出力層の重み
    double[][] outB = new double[1][];
    outB[0] = outputLayer.getB(); // 出力層のバイアス
    // リストに集約
    ArrayList<double[][]> coeffList = new ArrayList<>(); // 重みとバイアス
    coeffList.add(mid0W);
    coeffList.add(mid0B);
    coeffList.add(outW);
    coeffList.add(outB);
    // ファイルに保存
    saveData(element, coeffList, "DaiJa-V4_Emulator.dat");
    
    DJ._print("・中間０層の重み: ", mid0W);
    DJ.print("・中間０層のバイアス: ", mid0B);
    DJ.print("・出力層の重み: ", outW);
    DJ.print("・出力層のバイアス: ", outB);
    
    DJ._print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
//    DJ._print("・収集データ");
//    DJ._print("・収集データ数: ", dataNum);
//    DJ.print_(" クロック: ");
//    DJ.print("clock", sampleData[0]);
//    DJ.print_(" 力操作量: ");
//    DJ.print("fm", sampleData[1]);
//    DJ.print_(" 応答加速度: ");
//    DJ.print("a", sampleData[2]);
//    DJ.print_(" 応答速度: ");
//    DJ.print("v", sampleData[3]);
//    DJ.print_(" 応答位置: ");
//    DJ.print("p", sampleData[4]);
    
    DJ._print("・制御対象を学習したNNを検証する");
    control.setParameter(0, 0, inertia, friction); // 制御対象のパラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl();
    
//    double[][] trialData = new double[7][dataNum]; // 力操作量と試行結果
    
    middleLayer0.clear();
    outputLayer.clear();
     
//    double force = 0.0; // 力操作量
    double acceleration = 0.0; // 推定加速度
    double speed = 0.0; // （推定速度
    double position = 0.0; // 推定位置
    clock = -inc;
    for (int j = 0; j < dataNum; j++) {
      clock = clock + inc;
/*
      // （１）力操作量を与える（台形状）
      int cycle = dataNum / 3;
      if (j < cycle) force = force + 1.0; // 入力データ（力の操作量）
      else if (j < cycle * 2) force = force + 0.5; 
      else force = force - 0.525;
*/ /*
      // （２）力操作量を与える（Ｓ字状）
      force = (-2.0 * clock + 3.0 ) * clock * clock; // 力操作量（S字）
      force = force * 500.0; // スケーリング
*/ /*
      // （３）力操作量を与える（折線ジグザグ状）
      int cycle = dataNum / 5;
      if (j < cycle) force = force + 2.5; // 入力データ（力の操作量）
      else if (j < cycle * 2) force = force - 3.0; 
      else if (j < cycle * 3) force = force + 1.5; 
      else if (j < cycle * 4) force = force - 2.0; 
      else force = force + 0.75;
*/      

      // DJ._print("制御対象を実行し、応答を得る");
//      trialData[0][j] = force; // 力操作量
//      control.setForce(trialData[0][j]); // 力操作量
//      control.execute();
//      trialData[1][j] = control.getAcceleration(); // 応答加速度
//      trialData[2][j] = control.getSpeed(); // 応答速度
//      trialData[3][j] = control.getPosition(); // 応答位置
      
      // DJ._print("（２）学習したNfwdNNに力操作量を与た場合の応答速度");
//      mid0X[0] = trialData[0][j] * 0.002; // 力操作量を入力

      mid0X[0] = sampleData[1][j] * 0.002; // 力操作量を入力

      // 仮の速度と位置の疑似推定値を求める
//*a*/      double acceleration_ = outY[0]; // 推定加速度
//*a*/      double speed_ = speed + acceleration_ / dataNum; // 推定速度

//*v*/      double speed_ = outY[0]; // 推定速度
//*v*/          double acceleration_;
/////*v*/     if (j > 1) acceleration_ = speed_ - patternData1[8][j - 2] / 10.0;
      
//**/       mid0X[1] = speed_ * 0.1; // 推定速度を入力
//**/ //        if (j > 0) mid0X[1] = patternData1[8][j - 1] * 0.1;
//**/      mid0X[1] = trialData[2][j]; // 応答速度を入力
//**/      mid0X[1] = speed_ * 10.0; // 仮の推定速度
//**/      mid0X[1] = trialData[2][j]; // 応答速度を入力

/**/      mid0X[1] = sampleData[3][j]; // 応答速度を入力
      
      //#      mid0X[2] = outY[2]; // 位置(推定値) // 入力データ（位置）
      middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
      //#        middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
      outputLayer.forward(); // 出力層の順伝播処理を呼び出し
      
     // 加速度を推定した場合 
/*a*/      acceleration = outY[0]; // 推定加速度
/*a*/      speed = speed + acceleration / dataNum; // 推定速度
/*a*/      position = position + speed / dataNum; // 推定位置
     // 速度を推定した場合 
//*v*/      speed = outY[0]; // 推定速度
//*v*/      if (j > 0) acceleration = speed - patternData1[8][j - 1] * 0.10; /// 1.0; // 推定加速度
//*v*/      position = position + speed / (double)dataNum; // 推定位置
      
      patternData1[1][j] = sampleData[1][j] * 0.001; /// 200.0; // 力操作量
      patternData1[2][j] = -10.0; // ダミー
      patternData1[3][j] = -10.0; // ダミー 
      
//      patternData1[4][j] = trialData[1][j] * 100.0; //2000.0; // 応答加速度
//      patternData1[5][j] = trialData[2][j]; // 応答速度
//      patternData1[6][j] = trialData[3][j] * 2.0; //10.0; // 応答位置
      patternData1[4][j] = sampleData[2][j] * 100.0; //2000.0; // 応答加速度
      patternData1[5][j] = sampleData[3][j]; // 応答速度
      patternData1[6][j] = sampleData[4][j] * 2.0; //10.0; // 応答位置
      
/*a*/      patternData1[7][j] = acceleration * 1.0; // 加速度(推定値)
//*v*/      patternData1[7][j] = acceleration * 1000.0; // 加速度(推定値)

/*a*/        patternData1[8][j] = speed * 10.0; // 推定速度
//*v*/        patternData1[8][j] = speed * 10.0; //1.0; // 推定速度

/*a*/        patternData1[9][j] = position * 20.0; // 推定位置
//*v*/        patternData1[9][j] = position * 20.0; // 推定位置
    }
    
    // パターンの更新
    updatePattern();

    // 処理時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print_("・タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
    DJ.print(", タスク終了日時：", TimeStamp.getTimeFormated());
    
    DJ._print("##### TargetLearner 終了 #####");
  } // targetLearner()
  
  // ---------------------------------------------------------------------------
  // データ・ファイルへ保存

  /**
   * テキストデータの保存（テキスト配列、データ・リスト）
   * @param text String[] // テキスト配列
   * @param dataList ArrayList<double[][]> // データ・リスト
   * @param fileName String // ファイル名
   */
  public static void saveData(String[] text, ArrayList<double[][]> dataList, String fileName) {
    ArrayList<String> buf = new ArrayList<>();
    // 文字列をバッファに追加
    int textLength = text.length;
    buf.add(textLength + "\r\n"); // テキスト配列の要素数
    if (textLength > 0) {
      buf.add(text[0]);
      for (int i = 1; i < text.length; i++) {
        buf.add("," + text[i]);
      }
      buf.add("\r\n"); // 復帰改行
    }
    // データをバッファに追加
    int dataSize = dataList.size();
    buf.add(dataSize + "\r\n"); // データ・リストの要素数
    if (dataSize > 0) {
      for (int k = 0; k < dataSize; k++) {
        double[][] data = dataList.get(k);
        buf.add(data.length + "\r\n"); // データ配列の要素数
        for (int i = 0; i < data.length; i++) {
          StringBuilder str = new StringBuilder();
          str.append(data[i][0]);
          for (int j = 1; j < data[0].length; j++) {
            str.append(",").append(data[i][j]); // データをコンマで分離
          }
          buf.add(str.toString() + "\r\n");
        }
      } // k
    }
    LogEditor.saveText(buf, fileName); // ファイルに保存
  }
  
} // TargetLearner Class

// End of file
