/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package task.control;

import active.Activator;
import data.Control;
import java.util.ArrayList;
import layer.ControlLayer;
import layer.Layer;
import task.Task;
import util.DJ;
import util.LogEditor;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: TargetSampler</p>
 * <p> 説　明: 制御対象の実行データ（Clock,f,a,v,p）を収集する。
 * 　　制御対象に折れ線・台形・Ｓ字などの力操作量を与える。
 *     収集データしたファイルに保存する。
 *     ※ＮＮの学習は行わない。</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.08.14</p>
 * <p> 更新日: 2021.07.07</p>
 */
public class TargetSampler extends Task {

  // 学習・試行制御
  int epoch = 1; // エポックの回数
  int batchNum = 1; // バッチ数
  int interval = 1; //1; // 経過表示間隔
  
  // 学習パラメータ
  double initialCoef = 0.01; // 重みとバイアスの初期値係数
  double eta = 0.01; // 学習係数
  int dataNum = 1000; // エポック毎の学習データ数　1000[個]

  // 制御系と制御対象のパラメータ
  int speedGain = 1280; //2048 // 比例ゲイン（速度ループゲイン）
  int integralGain = 0; // 積分ゲイン
  double inertia = 1.0; // 慣性質量［kgm^2/sec^2］
  double friction = 2.0; // 粘性摩擦[kg/sec]

  // 各層のノード数
  int inNodeNum = 1; // 入力層のノード数
  int midNodeNum = 1; // 中間層のノード数
  int outNodeNum = 1; // 出力層のノード数
  
  // レイヤー構成
  private ControlLayer middleLayer0; // 中間層０
  //#  private ControlLayer middleLayer1; // 中間層１
  private ControlLayer outputLayer; // 出力層
  
  // 収集データを保存するファイルの名称
  private static final String FILE_NAME = "DaiJa-V4_SampledData.dat";
  
  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; // パターン・ビューワを表示する
    patternData0 = new double[5][dataNum]; // ID,力,加速度,速度,位置
    patternViewerLauncher = DJ.pattern( // 判定パターンとデータ
        PatternViewer.PATTERN_TUNER, patternData0, "TargetSampler0");
    
    targetSampler(); // タスク本体の呼び出し
  }
  
  /**
   * 制御対象の実行データ（Clock,f,a,v,p）を収集する。
   * 　　制御対象に折れ線・台形・Ｓ字などの力操作量を与える。
   *     収集データをファイルに保存する。
   */
  public void targetSampler() {
    DJ._print("DaiJa_V3, TargetSampler.targetSampler() ======================");
    
    DJ._print("・パラメータ");
//    DJ.print_(" 重みとバイアスの初期値係数:initialCoef=", initialCoef);
//    DJ.print_(", 学習係数:eta=",eta);
    DJ.print(", エポックの回数:epoch=",epoch);
    
//    DJ.print_(", 経過表示間隔:interval=",interval);
    DJ.print(", 学習データ数：dataNum=", dataNum);
    
    DJ.print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・制御系と制御対象（負荷）を設定");
    Control control = new Control();
    control.setParameter(0, 0, inertia, friction); // 制御対象のパラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl(); // 制御系をリセット
    
    DJ._print("・制御対象の実行データを収集");
    double[][] sampleData = new double[5][dataNum]; // 実行データ収集
    double f = 0.0; // 力操作量
    double inc = 1.0 / (double)dataNum;
    double clock = -inc;
    for (int j = 0; j < dataNum; j++) {
      clock = clock + inc;
      sampleData[0][j] = (double)clock;
/*      
      // （１）台形状の力操作量を与える場合
      int cycle = dataNum / 3;
      if (j < cycle) f = f + 3.0; //4.0; // 入力データ（力の操作量）
      else if (j < cycle * 2) f = f + 0.0;
      else f = f - 3.0;
*//*
      // （２）Ｓ字カーブ状の力操作量を与える場合
      f = (-2 * clock + 3 )* clock * clock; // 力の操作量
      f = f * 2000.0; // スケーリング
*//*
      // （３）折れ線（ジグザグ状）の力操作量を与える場合
      int cycle = dataNum / 5;
      if (j < cycle) f = f + 2.5; // 入力データ（力の操作量）
      else if (j < cycle * 2) f = f - 3.0; 
      else if (j < cycle * 3) f = f + 1.5; 
      else if (j < cycle * 4) f = f - 2.0; 
      else f = f + 0.75;
*/      
      // （３）折れ線（１周期分）の力操作量を与える場合
      int cycle = dataNum / 8;
      double fInc = 80.0; // 力操作量の増分（１サンプリング当たり）
      
      if      (j > cycle * 8) f = 0.0;
      else if (j > cycle * 7) f = f - fInc; // 入力データ（力の操作量）
      else if (j > cycle * 5) f = f + fInc;
      else if (j > cycle * 4) f = f - fInc; 
      else if (j > cycle * 3) f = f + fInc; // 入力データ（力の操作量）
      else if (j > cycle)     f = f - fInc;
      else if (j > 0)         f = f + fInc; 
      
      
     
      sampleData[1][j] = f; // 力操作量
      control.setForce(f);
      control.execute();
      sampleData[2][j] = control.getAcceleration(); // 応答加速度
      sampleData[3][j] = control.getSpeed(); // 応答速度
      sampleData[4][j] = control.getPosition(); // 応答位置
    }
    int dataNum_ = sampleData[0].length; // 生成した学習データ数
    if (dataNum_ != dataNum) {
      DJ.print("***** ERROR ***** TargetSampler.targetSampler()" + "\n"
              + " Size of sampleData(" + dataNum_ +") is not equal to "
              + dataNum + "."); // データ数が正しくない
      return;
    }
    
    // 学習用データをパターン・ビューワで表示
    for (int j = 0; j < dataNum; j++) {
      patternData0[0][j] = sampleData[0][j]; // Clock
      // patternData0[1][j] = sampleData[1][j] * Control.DAC_FACTOR * 0.2;
      patternData0[1][j] = sampleData[1][j] * 0.001; // 0.0005 // 力操作量
      patternData0[2][j] = sampleData[2][j] * 200.0; //100.0; // 応答加速度
      patternData0[3][j] = sampleData[3][j] * 5.0; //1.0; // 応答速度
      patternData0[4][j] = sampleData[4][j] * 100.0; //2.0; // 応答位置
    }
    
    DJ._print("・学習対象（制御系と制御対象）の実行準備");
    control.setParameter(10, 10, inertia, friction); // パラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    
    DJ._print(" 学習用見本データ：sampleData");
    DJ.print_(", 行数：" + sampleData.length + "（時点,操作量,応答）");
    DJ.print(", 列数：", sampleData[0].length + "（選点）");
    
    DJ._print("・入力データと教師データを作成");
    double[] inputData = new double[dataNum]; // 入力データ
    double[] correctData = new double[dataNum]; // 正解データ
    
    clock = -inc;
    for (int j = 0; j < dataNum; j++) {
      clock = clock + inc;
      inputData[j] = clock; // 入力データ
      correctData[j] = sampleData[2][j] * 10.0; // 正解データ（所望速度）
    }

    // DJ.print(" 順序をランダムにシャッフルしたインデックスのリスト");
    //#    ArrayList<Integer> indexList = DJ.permutationRandom(dataNum);
    //#    DJ.print("inputData", inputData);
    
    DJ._print("・ニューラルネットの各層の初期化");
    middleLayer0 = new ControlLayer(inNodeNum, midNodeNum);
    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
//    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.ADA_GRAD);
    double[] mid0X = middleLayer0.getX();
    double[] mid0Y = middleLayer0.getY();
    // double[] mid0C = middleLayer0.getC();
    // double[] mid0E = middleLayer0.getE();
    // double[] mid0dEdX = middleLayer0.getdEdX();
    //# double[] mid0dEdY = middleLayer0.getdEdY(); // 参照を代入（注１）

    // middleLayer1 = new ControlLayer(midNodeNum, midNodeNum);
    // middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
    //# double[] mid1X = middleLayer1.getX(); // 参照を代入（注３）
    // double[] mid1Y = middleLayer1.getY();
    // double[] mid1C = middleLayer1.getC();
    // double[] mid1E = middleLayer1.getE();
    // double[] mid1dEdX = middleLayer1.getdEdX();
    //# double[] mid1dEdY = middleLayer1.getdEdY(); // 参照を代入（注２）

    outputLayer = new ControlLayer(midNodeNum, outNodeNum);
    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.SGD);
    // outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.ADA_GRAD);
    //# double[] outX = outputLayer.getX(); // 参照を代入（注４）
    double[] outY = outputLayer.getY();
    double[] outC = outputLayer.getC();
    double[] outE = outputLayer.getE();
    double[] outdEdX = outputLayer.getdEdX();
    // double[] outdEdY = outputLayer.getdEdY();

    double[] outS = new double[outC.length]; // 正解（所望速度）

    // middleLayer0.setdEdY(mid1dEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
    // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
    // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
    // outputLayer.setX(mid1Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    
    middleLayer0.setdEdY(outdEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
    // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
    // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
    outputLayer.setX(mid0Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    
    // DJ._print("PatternViewerへの参照を得る");
    patternViewer = patternViewerLauncher.getPatternViewer();
    
    // DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | 
              (i == epoch); //経過表示フラグ

      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

       // 経過表示インターバル
      if (intervalFlag) { // 実行時間に影響を与える

        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            // wait(SLEEP_TIME); // タイムアウト付きで待機状態
            wait(100); // パターン・ビューワの起動を待つ
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
    
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch ---------------------------------------");
    }
    DJ._print(" End of all epoch --------------------------------------------");
    
    // パターンの更新
    updatePattern();
    
    DJ._print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・収集データ");
    DJ.print(" 収集データ数: ", dataNum);
    DJ.print_(" クロック: ");
    DJ.print("clock", sampleData[0]);
    DJ.print_(" 力操作量: ");
    DJ.print("fm", sampleData[1]);
    DJ.print_(" 応答加速度: ");
    DJ.print("a", sampleData[2]);
    DJ.print_(" 応答速度: ");
    DJ.print("v", sampleData[3]);
    DJ.print_(" 応答位置: ");
    DJ.print("p", sampleData[4]);
    
    DJ._print_("・収集データをファイルに保存する");
//$    saveData(sampleData, "DaiJa-V4_SampleData.dat");
    saveData(sampleData, FILE_NAME);
    // 処理時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    finishTime = System.currentTimeMillis(); // タスク終了時刻
    DJ.print_("・タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
    DJ.print(", タスク終了日時：", TimeStamp.getTimeFormated());
    
    DJ._print("##### TargetSampler 終了 #####");
  } // targetSampler()
    
  // ---------------------------------------------------------------------------
  // データ・ファイルへ保存
  
  /**
   * テキストデータの保存
   * @param sampleData double[][] // 保存する文字列
   * @param fileName String // ファイル名
   */
  public static void saveData(double[][] sampleData, String fileName) {
    int row = sampleData.length;
    ArrayList<String> buf = new ArrayList<>(row);
    for (int i = 0; i < row; i++) {
      StringBuilder str = new StringBuilder();
      str.append(sampleData[i][0]);
      for (int j = 1; j < sampleData[0].length; j++) {
        str.append(",").append(sampleData[i][j]);
      }
      buf.add(str.toString() + "\r\n");
    }
    LogEditor.saveText(buf, fileName); // ファイルに保存
  }
    
} // TargetSampler Class

// End of file
