/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.1
 *  @since 2022.8.18
 *  Copyright: 2020 - 2025
 */
package task.qu;

import quan.QuBasis;
import quan.QuBit;
import quan.QuGate;
import task.Task;
import util.DJ;
import util.TimeStamp;
import util.comp.CompVec;
import util.comp.Complex;
import view.PatternViewer;

/**
 * <p> 表　題: Class: BellsInequality</p>
 * <p> 説　明: ベルの不等式</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024</p>
 * <p> 作成日: 2024.08.17</p>
 */
public class BellsInequality extends Task {
  
  // 学習・試行制御
  int epoch = 6; // エポックの回数
  int interval = 1; //10; // 経過表示間隔
  
  // 量子計算パラメータ
  int delta = 30; // 5; //10; //15; // 基底角度の増分（度）
  
  // パターン・ビューワ
  int dataNum = 200; // 表示データ数

  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; //true; // false:非表示
    patternData0 = new double[5][dataNum]; // 附番,左辺,右辺,右辺第０項,右辺第１項
    patternData1 = new double[5][dataNum]; // ID,
    patternViewerLauncher = DJ.pattern(  // 判定パターン、データ、タイトル
        PatternViewer.PATTERN_TUNER, patternData0, "Bell's Inequality",
        PatternViewer.PATTERN_TUNER, patternData1, "Bell's Inequality");
    
    this.stepMode = true; // タスクのステップ実行モード
    
    bellsInequality(); // タスク本体の呼び出し
  }
  
  /** 
   * ベルの不等式
   * （１）測定用基底角度θを０度、３０度、６０度に固定した場合
   * （２）測定用基底角度θを変化させて、確率をグラフ表示する場合
   */
  public void bellsInequality() {
    DJ._print("BellsInequality.bellsInequality() ========================");
    DJ.print("■ベルの不等式：Bell's Inequality");
    DJ.print("  局所実在論を仮定して得られるベルの不等式は、\n"
           + "  実験により成立しないことが証明されました。\n"
           + "  本タスクでは、量子計算によりベルの不等式が\n"
           + "  破られることを示します。");
    
    DJ._print("・ベルの実験\n"
      + "　織姫Vegaと彦星Altairは互いの測定結果が相手に伝わらないように、\n"
      + "　十分に離れた距離（天の川の両岸、およそ１５光年）を隔てて、\n"
      + "　もつれた２個の量子ビットのそれぞれ一方を測定します。\n"
      + "　測定に使う基底は３個(b0, b1, b2)有りますが、織姫は基底b0とb2から、\n"
      + "　彦星は基底b1とb2から測定ごとにどちらかをランダムに選んで使います。\n"
      + "　先に織姫が測定します。量子ビットは測定に使った基底に従って遷移します。\n"
      + "　織姫の測定により、もつれている彦星の量子ビットも織姫の量子ビットと\n"
      + "　同じ量子状態に遷移します。その量子ビットを彦星が後から測定します。\n"
      + "　選んだ基底のどの基底に遷移するかの確率が定まります。\n");
    
    DJ._print("・局所存在論（局所性と存在性）を仮定すると次の不等式が成立します。\n"
      + "　Bellの不等式：P(b0_0, b1_1) <= P(b0_0, b2_1) + P(b2_0, b1_1)\n"
      + "　ここで、P()は発生確率を表します。末尾の_0と_1は、それぞれ、\n"
      + "　測定結果が選んだ基底の第０基底に遷移した場合を表し、\n"
      + "　測定結果が選んだ基底の第０基底以外に遷移した場合を表します。\n\n"
        
      + "　P(b0_0, b1_1)は織姫が基底b0で測定した結果が第０基底に遷移し、\n"
      + "　　　彦星がb1で測定した結果が第１基底に遷移した場合の確率です。\n\n"
        
      + "　P(b0_0, b2_1)は織姫が基底b0で測定した結果が第０基底に遷移し、\n"
      + "　　　彦星がb2で測定した結果が第１基底に遷移した場合の確率です。\n\n"
        
      + "　P(b2_0, b1_1)は織姫が基底b2で測定した結果が第０基底に遷移し、\n"
      + "　　　彦星がb1で測定した結果が第１基底に遷移した場合の確率です。");
    
    DJ._print("・局所存在論では結果は測定する前から実在し、離れた場所での測定\n"
      + "　結果は互いに他方には影響しない（相対論的に光の早さ以上の速さでは\n"
      + "　情報は伝わらない）と考えて、確率は以下の集合に置き換えられます。");
    
    DJ._print("・測定結果を集合（ベン図：Venn Diagram）で表します。\n"
      + "　ベン図は基底b0,b1,b2による全ての測定結果を含んでいます。");
        
    DJ._print(
        "  左辺　　　P(b0_0, b1_1)は集合(b0_0,b1_1,b2_0)+(b0_0,b1_1,b2_1)\n"
      + "  右辺第１項P(b0_0, b2_1)は集合(b0_0,b1_0,b2_1)+(b0_0,b1_1,b2_1)\n"
      + "  右辺第２項P(b2_0, b1_1)は集合(b0_0,b1_1,b2_0)+(b0_1,b1_1,b2_0)");

    DJ._print(
        "    ┌───────────────────────────┐     \n"
      + "    │      　            (b0_0,b1_1,b2_1)                  │     \n"
      + "    │                    左辺・右辺第１項                  │     \n"
      + "    │      　          ┌─────────────────┼─┐ \n"
      + "┌─┼─────────┼────────┐                │　│ \n"
      + "│  │      　          │                │                │  │ \n"
      + "│  │(b0_0,b1_0,b2_1)  │(b0_0,b1_0,b2_0)│(b0_0,b1_1,b2_0)│  │ \n"
      + "│  │   右辺第１項     │　              │左辺・右辺第２項│  │ \n"
      + "│  └─────────┼────────┼────────┘  │ \n"
      + "│                      │                │  (b0_1,b1_1,b2_0)  │ \n"
      + "│  (b0_1,b1_0,b2_1)    │(b0_1,b1_0,b2_0)│    右辺第２項      │ \n"
      + "│                      └────────┼──────────┘ \n"
      + "└────────────────────┘                       ");
        
    DJ._print("・以上のように、左辺の２個の要素は全て右辺の集合に\n"
      + "  含まれるので、Bellの不等式が導出されます。\n\n"
        
      + "  P(b0_0, b1_1) <= P(b0_0, b2_1) + P(b2_0, b1_1)");
    
    
    
    DJ.printS(3, "BellsInequality(0)", this); // Pause Task --------------------
    DJ.print("Code(0)タスク実行パラメータ");
    DJ._print("・タスク実行パラメータ");
    DJ.print("  エポックの回数：epoch = ",epoch); // epoch = 70
    DJ.print("  経過表示間隔：interval = ",interval); // interval = 1
    // DJ.print("  表示データ数：dataNum = ", dataNum);
    
    DJ.print("・量子計算のパラメータ");
    DJ.print("  基底角度の増分（度）：delta = ",delta); // delta = 1
    
    
    
    DJ.printS(1, "BellsInequality(1)", this); // Pause Task ----------------
    DJ.print("Code(1)測定用基底の角度を固定した場合");
    
    DJ._print("Code(1-1)①測定用基底角度を0度、60度、30度に固定");
    DJ.print("・織姫と彦星は以下の３個の基底(b0, b1, b2)から、\n"
      + "  それぞれ２個の基底(b0, b2)と(b1, b2)を使って測定します。\n"
      + "                                       \n"
      + "  b0 : θ = 0π  &  π/2 =  0 &  90deg \n"
      + "  b1 : θ = π/3 & 5π/6 = 60 & 150deg \n"
      + "  b2 : θ = π/6 & 2π/3 = 30 & 120deg \n"
      + "                                       \n"
      + "　先に織姫が測定し、測定により遷移した量子ビットを\n"
      + "  彦星が後から測定します。\n");
    
    DJ.printS(2, "BellsInequality(1-1)②", this); // Pause Task ----------------
    DJ.print("Code(1-1)②ベルの実験の量子回路");
    DJ._print("・ベルの実験回路はHゲート(Hadamard Gate)と、\n"
      + "　CXゲート(C-Notゲート)で構成されます。\n"
      + "　HとCXゲートにより織姫と彦星の量子ビットqb0とqb1はもつれます。\n"
      + "　織姫が第０量子ビットqb0をM0で測定すると、測定に使用した基底に\n"
      + "　従って遷移します。織姫の測定と同時に彦星の量子ビットも遷移し、\n"
      + "　織姫と彦星の量子ビットqb0とqb1のもつれも解消します。その後で、\n"
      + "　彦星は選択した基底を使い第１量子ビットqb1をM1で測定します。\n"
      + "              ┌-┐  ┌-┐  ┌─┐          \n"
      + "qb0 = [1 0]'─┤H├─┤C├─┤M0│          \n"
      + "              ├-┤  │|│  ├─┤  ┌─┐  \n"
      + "qb1 = [1 0]'─┤I├─┤X├─┤I ├─┤M1│  \n"
      + "              └-┘  └-┘  └─┘  └─┘  ");

    
    DJ.printS(1, "BellsInequality(1-1)③", this); // Pause Task ----------------
    DJ.print("Code(1-1)③入力量子ビットの生成");
    DJ.print("・織姫と彦星の量子ビットqb0=[1 0]'とqb1=[1 0]'を生成します。");
    QuBit qb0 = new QuBit();
    DJ.printF("qb0", qb0);
    // qb0:F
    // ┌     ┐     
    // │ 1+0i│ 
    // │ 0+0i│     
    // └     ┘     
    QuBit qb1 = new QuBit();
    DJ.printF("qb1", qb1);
    // qb1:F 
    // ┌     ┐     
    // │ 1+0i│
    // │ 0+0i│     
    // └     ┘     

    DJ.print("・二個の量子ビットqb0=qb1=[1 0]'を相互作用させ、\n"
           + "  ２量子ビットqb0_1=[1 0 0 0]'を生成します。");
    QuBit qb0_1 = qb0.interactQuBit(qb1);
    DJ.printF("qb0_1", qb0_1);
    // qb0_1:F 
    // ┌     ┐ ┌ ┐ ┌ ┐
    // │ 1+0i│=│1│?│1│
    // │ 0+0i│ │0│ │0│
    // │ 0+0i│ └ ┘ └ ┘
    // │ 0+0i│
    // └     ┘
    
    
    DJ.printS(2, "BellsInequality(1-2)", this); // Pause Task ------------------
    DJ.print("Code(1-2)量子ゲートの生成");
    
    DJ._print("Code(1-2)①Hゲート（Hadamardゲート）の生成");
    DJ.print("Hゲート \n"
    +"          ┌      ┐ \n"
    +" H = 1/√2│ 1   1│ \n"
    +"          │ 1  -1│ \n"
    +"          └      ┘ ");
    QuGate hadamardGate = new QuGate(QuGate.H_GATE);
    // DJ.print(hadamardGate.toFormat("hadamardGate")); // Hゲート（多桁表示）
    DJ.printF("hadamardGate", hadamardGate); // Hゲート
    // hadamardGate:F
    // ┌                        ┐
    // │ 0.70711+0i   0.70711+0i│
    // │ 0.70711+0i  -0.70711+0i│
    // └                        ┘
    
    DJ.printS(1, "BellsInequality(1-2)②", this); // Pause Task ----------------
    DJ.print("Code(1-2)②Iゲート（恒等ゲート）の生成");
    DJ.print("Iゲート \n"
    +"     ┌      ┐ \n"
    +" I = │ 1   0│ \n"
    +"     │ 0   1│ \n"
    +"     └      ┘ ");
    QuGate iGate = new QuGate(QuGate.I_GATE);
    DJ.printF("iGate", iGate); // Ｉゲート
    // iGate:F
    // ┌            ┐
    // │ 1+0i   0+0i│
    // │ 0+0i   1+0i│
    // └            ┘    

    DJ.printS(1, "BellsInequality(1-2)③", this); // Pause Task ----------------
    DJ.print("Code(1-2)③HゲートとIゲートを相互作用させ、"
            + "２量子ゲートを生成");
    DJ.print("H_Iゲート:H?I \n"
    +"            ┌     ┐ \n"
    +" H?I = 1/√2│ I  I│ \n"
    +"            │ I -I│ \n"
    +"            └     ┘ ");
    QuGate h_iGate = hadamardGate.interactQuGate(iGate);
    DJ.printF("h_iGate", h_iGate); // H_Iゲート
    // h_iGate:F
    // ┌                                                  ┐
    // │ 0.70711+0i   0+0i         0.70711+0i   0+0i      │
    // │ 0+0i         0.70711+0i   0+0i         0.70711+0i│
    // │ 0.70711+0i   0+0i        -0.70711+0i   -0+0i     │
    // │ 0+0i         0.70711+0i   -0+0i       -0.70711+0i│
    // └                                                  ┘
    
    DJ.printS(1, "BellsInequality(1-2)④", this); // Pause Task ----------------
    DJ.print("Code(1-2)④CXゲート（C-Notゲート）の生成");
    DJ.print("CXゲート \n"
    +"      ┌              ┐ \n"
    +" CX = │ 1   0   0   0│ \n"
    +"      │ 0   1   0   0│ \n"
    +"      │ 0   0   0   1│ \n"
    +"      │ 0   0   1   0│ \n"
    +"      └              ┘ ");
    
    DJ.print("・CXゲートの生成には、量子ビット数、および\n"
        + "ターゲット量子ビット、制御量子ビット配列を指定します。");
    DJ.print(" QuGate cxGate = new QuGate("
            +" QuGate.CX_GATE, numOfBits, target, controlArray)");
    int numOfBits = 2; // 量子ビット数
    int target = 1; // ターゲット量子ビット
    DJ.print_("numOfBits = " + numOfBits + ", target = " + target);
    int[] controlArray = {0}; // 制御量子ビット配列
    DJ.print(", controlArray", controlArray);
    DJ.print("controlArray", controlArray);
    QuGate cxGate = new QuGate(
        QuGate.CX_GATE, numOfBits, target, controlArray); // ＣＸゲート
    DJ.printF("cxGate", cxGate); // CXゲート
    // cxGate:F
    // ┌                          ┐
    // │ 1+0i   0+0i   0+0i   0+0i│
    // │ 0+0i   1+0i   0+0i   0+0i│
    // │ 0+0i   0+0i   0+0i   1+0i│
    // │ 0+0i   0+0i   1+0i   0+0i│
    // └                          ┘
    
    
    DJ.printS(2, "BellsInequality(1-3)", this); // Pause Task ------------------
    DJ.print("Code(1-3)測定対象となる量子ビットtargetQuBitの生成");

    DJ._print("Code(1-3)①入力量子ビットをHゲートで重ね合わせ状態に");
    QuBit qbH01 = h_iGate.apply(qb0_1);
    DJ.printF("qbH01", qbH01);
    // qbH01:F 
    // ┌           ┐
    // │ 0.70711+0i│
    // │ 0+0i      │
    // │ 0.70711+0i│
    // │ 0+0i      │
    // └           ┘
    
    DJ.printS(1, "BellsInequality(1-3)②", this); // Pause Task ----------------
    DJ.print("Code(1-3)②入力量子ビットをCXゲートでもつれ状態に");
    DJ.print("・量子ビットqbH01を量子ゲートcxGateに適用し、\n"
           + "  測定対象のもつれた量子ビットtargetQuBitを生成します。");
    DJ.print("targetQuBit = cxGate.apply(qbH01");
    QuBit targetQuBit = cxGate.apply(qbH01);
    DJ.printF("targetQuBit", targetQuBit);
    // targetQuBit:F 
    // ┌           ┐
    // │ 0.70711+0i│←────────┐
    // │ 0+0i      │←┐(1) product   │(0) product
    // │ 0+0i      │←┘     = 0.0    │    ≒ 0.5
    // │ 0.70711+0i│←────────┘
    // └           ┘

    DJ.printS(1, "BellsInequality(1-3)③", this); // Pause Task ----------------
    DJ.print("Code(1-3)③測定対象の量子ビットのもつれ状態を判定");
    // boolean checked = targetQuBit.entangleCheck(); // 判定結果のみを表示
    boolean checked = targetQuBit.entangleDetails(); // 判定中の詳細情報を表示
    // $ (0) product = :0.4999999999999999+0.0i
    // $ (1) product = :0.0+0.0i
    // $ difference = 0.4999999999999999　←　(0)と(1)に差が有ればもつれている
    // $ judgment = true
    DJ._print("もつれ状態：", Boolean.toString(checked));
    // もつれ状態：true ←　もつれている

    
    DJ.printS(2, "BellsInequality(1-4)", this); // Pause Task ------------------
    DJ.print("Code(1-4)測定用の３種類の基底b0,b1,b2を生成");
    double theta = Math.PI / 6.0; // 基底の基準となる回転角度（30度）
    int numOfQuBit = 1; // 量子ビット数
    QuBasis eQuBasis = new QuBasis(numOfQuBit); // 恒等基底e（標準基底）
    DJ.printF("eQuBasis", eQuBasis); // 恒等基底e
    // eQuBasis:F
    // ┌            ┐
    // │ 1+0i│ 0+0i│
    // │ 0+0i│ 1+0i│
    // └            ┘
    
    
    DJ._print("Code(1-4)①基底b0を生成（回転角を0度で指定）");
    QuBasis b0QuBasis = new QuBasis(numOfQuBit); // 基底b0（標準基底）
    DJ.print("・基底b0QuBasisと恒等基底eQuBasisを相互作用させ、\n"
           + "２量子ビット基底b0_eQuBasisを生成");
    QuBasis b0_eQuBasis = b0QuBasis.interactQuBasis(eQuBasis);
    DJ.printF("b0_eQuBasis", b0_eQuBasis);
    // b0_eQuBasis:F
    // ┌                          ┐
    // │ 1+0i│ 0+0i│ 0+0i│ 0+0i│
    // │ 0+0i│ 1+0i│ 0+0i│ 0+0i│
    // │ 0+0i│ 0+0i│ 1+0i│ 0+0i│
    // │ 0+0i│ 0+0i│ 0+0i│ 1+0i│
    // └                          ┘
    
    DJ.printS(1, "BellsInequality(1-4)②", this); // Pause Task ----------------
    DJ.print("Code(1-4)②基底b1を生成（回転角を60度で指定）");
    QuBasis b1QuBasis = new QuBasis(theta * 2); // 角度を2θで指定
    DJ.printF("b1QuBasis", b1QuBasis);
    // b1QuBasis:F
    // ┌                        ┐
    // │ 0.5+0i    │-0.86603+0i│
    // │ 0.86603+0i│ 0.5+0i    │
    // └                        ┘
    
    DJ.print("・基底b1QuBasisと恒等基底eQuBasisを相互作用させ、\n"
           + "  ２量子ビット基底b1_eQuBasisを生成します。");
    QuBasis b1_eQuBasis = b1QuBasis.interactQuBasis(eQuBasis);
    DJ.printF("b1_eQuBasis", b1_eQuBasis);
    // b1_eQuBasis:F
    //  ┌                                                  ┐
    //  │ 0.5+0i    │ 0+0i      │-0.86603+0i│ -0+0i     │
    //  │ 0+0i      │ 0.5+0i    │ -0+0i     │-0.86603+0i│
    //  │ 0.86603+0i│ 0+0i      │ 0.5+0i    │ 0+0i      │
    //  │ 0+0i      │ 0.86603+0i│ 0+0i      │ 0.5+0i    │
    //  └                                                  ┘
    DJ.print("・織姫は測定に基底（偏光板）b0とb2を使用するので、\n"
           + "  この２量子ビット基底b1_eQuBasisは使われません。");
    
    
    DJ.printS(1, "BellsInequality(1-4)③", this); // Pause Task ----------------
    DJ.print("Code(1-4)③基底b2を生成（回転角を30度で指定）");
    QuBasis b2QuBasis = new QuBasis(theta); // 角度をθで指定
    DJ.printF("b2QuBasis", b2QuBasis);
    // b2QuBasis:F
    // ┌                        ┐
    // │ 0.86603+0i│-0.5+0i    │
    // │ 0.5+0i    │ 0.86603+0i│
    // └                        ┘

    DJ.print("・基底b2QuBasisと恒等基底eQuBasisを相互作用させ、\n"
           + "  ２量子ビット基底b2_eQuBasisを生成します。");
    QuBasis b2_eQuBasis = b2QuBasis.interactQuBasis(eQuBasis);
    DJ.printF("b2_eQuBasis", b2_eQuBasis);
    // b2_eQuBasis:F
    //  ┌                                                  ┐
    //  │ 0.86603+0i│ 0+0i      │-0.5+0i    │ -0+0i     │
    //  │ 0+0i      │ 0.86603+0i│ -0+0i     │-0.5+0i    │
    //  │ 0.5+0i    │ 0+0i      │ 0.86603+0i│ 0+0i      │
    //  │ 0+0i      │ 0.5+0i    │ 0+0i      │ 0.86603+0i│
    //  └                                                  ┘
    
        
    DJ.printS(2, "BellsInequality(1-5)", this); // Pause Task ------------------
    DJ.print("Code(1-5)ベルの不等式の左辺の確率 P(b0_0, b1_1)");
    
    DJ.print("Code(1-5)①織姫が対象量子ビットを基底b0_eで測定");
    
    DJ._print("・複数回測定し、各基底軸の遷移回数と古典ビットを求めます。");
    int tryalNo = 100; // 試行回数
    DJ.print("　countQuBasisルーチンの内部で試行回数回測定されます。");
    int[] count = b0_eQuBasis.countQuBasis(targetQuBit, tryalNo); 
    DJ._print("・各基底軸への遷移回数を表示します。");
    DJ.print("count = ", count); // 各基底軸に遷移した回数
    // count =[48, 0, 0, 52]
    
    DJ.printS(1, "BellsInequality(1-5)②", this); // Pause Task ----------------
    DJ._print("Code(1-5)②測定により量子ビットが遷移した基底軸");
    DJ.print("・各測定で遷移した基底軸を表示します。");
    QuBit[] selectedQBArray = b0_eQuBasis.getSelectedQuBits();
    DJ.print(QuBit.toFormat("selectedQBArray", selectedQBArray));
    // selectedQBArray:F
    // ┌                                            ┐
    // │ 0+0i￤ 1+0i￤ 1+0i￤ 0+0i￤ 0+0i ..... 0+0i│
    // │ 0+0i￤ 0+0i￤ 0+0i￤ 0+0i￤ 0+0i ..... 0+0i│
    // │ 0+0i￤ 0+0i￤ 0+0i￤ 0+0i￤ 0+0i ..... 0+0i│
    // │ 1+0i￤ 0+0i￤ 0+0i￤ 1+0i￤ 1+0i ..... 1+0i│
    // └                                            ┘    
    DJ._print("・遷移した基底軸に対応する古典ビットを表示します。\n"
            + "　２量子ビットなので古典ビットは0,1,2,3になります。");
    int[] classicBitArray = b0_eQuBasis.getClassicBit(); // １０進数
    DJ.print("classicBitArray = ", classicBitArray);
    // classicBitArray = [3, 0, 0, 3, 3, ..... , 3]
    
    DJ.printS(1, "BellsInequality(1-5)③", this); // Pause Task ----------------
    DJ._print("Code(1-5)③量子ビットが各基底軸に遷移する確率");
    QuBit qbb0e = b0_eQuBasis.basisChang(targetQuBit);
    DJ.printF("qbb0e", qbb0e);
    // qbb0e:F 
    // ┌           ┐        ┌ ┐ ┌ ┐          ┌ ┐ ┌ ┐ ┌ ┐ ┌ ┐
    // │ 0.70711+0i│= 1/√2(│1│+│0│) = 1/√2(│1│?│1│+│0│?│0│)
    // │ 0+0i      │        │0│ │0│          │0│ │0│ │1│ │1│
    // │ 0+0i      │        │0│ │0│          └ ┘ └ ┘ └ ┘ └ ┘
    // │ 0.70711+0i│        │0│ │1│
    // └           ┘        └ ┘ └ ┘
    DJ.print("　基底b0_eの第０,１項、に遷移する確率は共に(1/√2)^2=1/2です。");
    
    DJ._print("・全量子ビットの遷移確率を求めます。");
    double[][] tpb0e = qbb0e.getTransProb();
    DJ.printF("tpb0e", tpb0e);
    //  tpb0e=
    //  [[0.5, 0.5] ←[1 0]'への遷移確率
    //  [0.5, 0.5]  ←[0 1]'への遷移確率
    //  ] ↑   ↑
    //    │   └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("　基底b0_eの第０項に遷移する確率P(b0_0)は1/2です。");
    
    DJ.printS(1, "BellsInequality(1-5)④", this); // Pause Task ----------------
    DJ._print("Code(1-5)④織姫の測定により遷移した量子ビット");
    DJ.print("・遷移した量子ビットqbb0e_0を基底b0から取り出します。\n"
            + "　取り出す基底軸は吸収軸である第０基底軸なので、\n"
            + "　インデックスQuBasis.BASIS_0で指定します。");
    QuBit qbb0e_0 = b0QuBasis.getBasisQuBit(QuBasis.BASIS_0);
    DJ.printF("qbb0e_0", qbb0e_0);
    // qbb0e_0:F
    // ┌     ┐     
    // │ 1+0i│= [1 0]'
    // │ 0+0i│
    // └     ┘
    
    DJ.printS(1, "BellsInequality(1-5)⑤", this); // Pause Task ----------------
    DJ._print("Code(1-5)⑤遷移した量子ビットを彦星が基底b1で測定");
    DJ.print("・複数回測定し、各基底軸への遷移回数と古典ビットを求めます。");
    // tryalNo = 100; // 試行回数：設定済み
    DJ.print("tryalNo = ", tryalNo);
    count = b1QuBasis.countQuBasis(qbb0e_0, tryalNo); // 繰り返し測定
    DJ.print("count = ", count);
    // count =[33, 67] // 試行1000回の場合[248, 752]
    
    DJ._print("・各測定で遷移した基底を表示します。");
    selectedQBArray = b1QuBasis.getSelectedQuBits();
    DJ.print(QuBit.toFormat("selectedQBArray", selectedQBArray));
    // selectedQBArray:F
    // ┌                                                                   ┐
    // │-0.86603+0i￤-0.86603+0i￤ 0.5+0i    ￤-0.86603+0i ..... 0.5+0i    │
    // │ 0.5+0i    ￤ 0.5+0i    ￤ 0.86603+0i￤ 0.5+0i     ..... 0.86603+0i│
    // └                                                                   ┘ 
    
    DJ._print("・遷移した基底軸に対応する古典ビットを表示します。\n"
            + "　１量子ビットなので古典ビットは0,1です。");
    classicBitArray = b1QuBasis.getClassicBit();
    DJ.print("classicBitArray = ", classicBitArray);
    // classicBitArray = [1, 1, 0, 1, 1, 1, 0, 1, 1, 1, ..... , 0]
    
    DJ.printS(1, "BellsInequality(1-5)⑥", this); // Pause Task ----------------
    DJ._print("Code(1-5)⑥基底b1の第１基底軸への遷移確率");
    QuBit qbb0eb1 = b1QuBasis.basisChang(qbb0e_0);
    DJ.printF("qbb0eb1", qbb0eb1);
    // qbb0eb1:F
    // ┌           ┐ ┌      ┐
    // │ 0.5+0i    │=│ 1/2  │・・・第０確率振幅 → 確率(1/2)^2=1/4
    // │-0.86603+0i│ │-√3/2│・・・第１確率振幅 → 確率(-√3/2)^2=3/4
    // └           ┘ └      ┘
    DJ.print("　彦星の測定では第１基底軸へ遷移する確率なので、\n"
           + "　第１確率振幅の自乗が遷移確率になります。");
    
    DJ._print("・各軸への遷移確率を求めます。");
    double[][] tpb0eb1 = qbb0eb1.getTransProb();
    DJ.printF("tpb0eb1", tpb0eb1);
    //  tpb0eb1=
    //  [[0.25] ←[1 0]'への遷移確率
    //  [0.75]  ←[0 1]'への遷移確率 // P(b1_1)
    //  ] ↑  
    //    └ 第０量子ビットの遷移確率
    
    DJ.print("　第１確率振幅より遷移確率はP(b1_1) = (-√3/2)^2 = 3/4");
    DJ.print("　左辺P(b0_0, b1_1) = P(b0_0)*P(b1_1) = 1/2 * 3/4 = 3/8");

    
    DJ.printS(1, "BellsInequality(1-5)⑦", this); // Pause Task ----------------
    DJ._print("Code(1-5)⑦織姫と彦星が同時に測定する場合");
    
    DJ.print("・基底b0QuBasisと基底b1QuBasisを相互作用させ、\n"
           + "２量子ビット基底b0_b1QuBasisを生成");
    QuBasis b0_b1QuBasis = b0QuBasis.interactQuBasis(b1QuBasis);
    DJ.printF("b0_b1QuBasis", b0_b1QuBasis);
    //b0_1QuBasis:F
    //┌                                                  ┐
    //│ 0.5+0i    │-0.86603+0i│ 0+0i      │ -0+0i     │
    //│ 0.86603+0i│ 0.5+0i    │ 0+0i      │ 0+0i      │
    //│ 0+0i      │ -0+0i     │ 0.5+0i    │-0.86603+0i│
    //│ 0+0i      │ 0+0i      │ 0.86603+0i│ 0.5+0i    │
    //└                                                  ┘

    DJ._print("・基底b0_b1の各基底軸に遷移する確率");
    QuBit qbb0_b1 = b0_b1QuBasis.basisChang(targetQuBit);
    DJ.printF("qbb0_b1", qbb0_b1);
    // qbb0_b1:F 
    //┌           ┐         ┌ ┐        ┌ ┐       ┌ ┐
    //│ 0.35355+0i│= 1/√2 (│1│? (  1/2│1│- √3/2│0│)
    //│-0.61237+0i│         │0│        │0│       │1│
    //│ 0.61237+0i│         └ ┘        └ ┘       └ ┘
    //│ 0.35355+0i│         ┌ ┐        ┌ ┐       ┌ ┐ 
    //└           ┘        +│0│? (√3/2│1│+   1/2│0│))
    //                        │1│        │0│       │1│
    //                        └ ┘        └ ┘       └ ┘
    DJ.print("  織姫の量子ビットが[1 0]'へ遷移する確率は、\n"
           + "  量子ビットqbb0_b1の第０項あるいは１項に\n"
           + "  遷移する確率の和は(1/√2)^2=1/2です。");
    
    DJ._print("・全量子ビットの遷移確率を求めます。");
    double[][] tpb0b1 = qbb0_b1.getTransProb();
    DJ.printF("tpb0b1", tpb0b1);
    //  tpb0b1=
    //  [[0.5, 0.5] ←[1 0]'への遷移確率
    //  [0.5, 0.5]  ←[0 1]'への遷移確率
    //  ] ↑   ↑
    //    │   └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("　基底b0_b1の第０項に遷移する確率P(0_0)は1/2です。");
    
    DJ._print("・左辺の確率P(b0_0, b1_1) は、第１確率振幅の自乗");
    Complex qbb0b1pa1 = qbb0_b1.getProbAmp(QuBit.PA1); // 第１確率振幅
    Complex tpb0b1pa1 = qbb0b1pa1.product(qbb0b1pa1); // 自乗
    DJ.printF("P(b0_0, b1_1)", tpb0b1pa1); // 3/8
    // P(b0_0, b1_1):F 0.375-0i = 3/8
    
    DJ._print("・基底b0とb1で同時に測定した場合の遷移確率P(b0, b1)");
    CompVec b0_b1CompVec = qbb0_b1.copyProbAmpVec(); // 確率振幅
    b0_b1CompVec.product(b0_b1CompVec); // 確率振幅を自乗
    DJ.printF("P(b0, b1)", b0_b1CompVec);
    // P(b0, b1):F 
    //┌         ┐
    //│ 0.125+0i│
    //│ 0.375-0i│← P(b0_0, b1_1) = 3/8
    //│ 0.375+0i│
    //│ 0.125+0i│
    //└         ┘
    DJ.print("　左辺P(b0_0, b1_1) = 3/8");

    
    DJ.printS(2, "BellsInequality(1-6)", this); // Pause Task ------------------
    DJ._print("Code(1-6)ベルの不等式の右辺第０項の確率P(b0_0, b2_1)");
    DJ.print("・織姫が対象量子ビットtargetQuBitを基底b0_eで測定し、\n"
           + "　もつれ状態が解消した単独の量子ビットqbb0e_0が\n"
           + "　得られるまでの過程は(1-5)の④までと同様です。");
    
    DJ.print("Code(1-6)①遷移した量子ビットを彦星が基底b2で測定");
    DJ._print("・彦星が量子ビットqbb0e_0を基底b2QuBasisで測定します。");
    DJ.print("　量子ビットqbb0e_0を基底b2QuBasisで複数回測定し、\n"
            + "　各基底軸への遷移回数と古典ビットを求めます。");
    // tryalNo = 100; // 試行回数：設定済み
    DJ.print("tryalNo = ", tryalNo);
    count = b2QuBasis.countQuBasis(qbb0e_0, tryalNo); // 繰り返し測定
    DJ.print("count = ", count);
    // count =[75, 25] // 試行1000回の場合[735, 265]
    
    DJ.printS(1, "BellsInequality(1-6)②", this); // Pause Task ----------------
    DJ.print("Code(1-6)②彦星の測定で遷移した基底");
    DJ._print("・各測定で遷移した基底を表示します。");
    selectedQBArray = b2QuBasis.getSelectedQuBits();
    DJ.print(QuBit.toFormat("selectedQBArray", selectedQBArray));
    // selectedQBArray:F
    // ┌                                                                   ┐
    // │-0.5+0i    ￤ 0.86603+0i￤ 0.86603+0i￤ 0.86603+0i ..... 0.86603+0i│
    // │ 0.86603+0i￤ 0.5+0i    ￤ 0.5+0i    ￤ 0.5+0i     ..... 0.5+0i    │
    // └                                                                   ┘    
    DJ._print("・遷移した基底軸に対応する古典ビットを表示します。\n"
            + "　１量子ビットなので古典ビットは0,1です。");
    classicBitArray = b2QuBasis.getClassicBit();
    DJ.print("classicBitArray = ", classicBitArray);
    // classicBitArray = [1, 0, 0, 0, 1, 1, 0, 0, 1, ..... , 0]
    
    DJ.printS(1, "BellsInequality(1-6)③", this); // Pause Task ----------------
    DJ.print("Code(1-6)③基底b2の第１基底軸への遷移確率");
    
    QuBit qbb0eb2 = b2QuBasis.basisChang(qbb0e_0);
    DJ.printF("qbb0eb2", qbb0eb2);
    // qbb0eb2:F
    // ┌           ┐ ┌      ┐
    // │ 0.86603+0i│=│ √3/2│
    // │-0.5+0i    │ │-1/2  │・・・確率(-1/2)^2 = 1/4
    // └           ┘ └      ┘
    DJ.print("　彦星の測定では透過軸方向へ遷移する確率なので、\n"
           + "　第１基底に遷移する場合の確率になります。");
    DJ.print("　第１確率振幅より遷移確率はP(b2_1) = (-1/2)^2 = 1/4");

    DJ._print("・各軸への遷移確率を求めます。");
    double[][] tpb0eb2 = qbb0eb2.getTransProb();
    DJ.printF("tpb0eb2", tpb0eb2);
    //  tpb0eb2=
    //  [[0.75] ←[1 0]'への遷移確率
    //  [0.25]  ←[0 1]'への遷移確率 // P(b2_1)
    //  ] ↑  
    //    └ 第０量子ビットの遷移確率
    
    DJ.print("　第１要素より遷移確率はP(b2_1) = 1/4");
    DJ.print("　右辺P(b0_0, b2_1) = P(b0_0)*P(b2_1) = 1/2*1/4 = 1/8");
    
    
    DJ.printS(1, "BellsInequality(1-6)④", this); // Pause Task ----------------
    DJ._print("Code(1-6)④織姫と彦星が同時に測定する場合");
    
    DJ.print("・基底b0とb2を相互作用させ、２量子基底b0_2を生成");
    QuBasis b0_b2QuBasis = b0QuBasis.interactQuBasis(b2QuBasis);
    DJ.printF("b0_b2QuBasis", b0_b2QuBasis);
    // b0_b2QuBasis:F
    // ┌                                                  ┐
    // │ 0.86603+0i│-0.5+0i    │ 0+0i      │ -0+0i     │
    // │ 0.5+0i    │ 0.86603+0i│ 0+0i      │ 0+0i      │
    // │ 0+0i      │ -0+0i     │ 0.86603+0i│-0.5+0i    │
    // │ 0+0i      │ 0+0i      │ 0.5+0i    │ 0.86603+0i│
    // └                                                  ┘

    DJ._print("・基底b0_2の各基底軸に遷移する確率");
    QuBit qbb0_b2 = b0_b2QuBasis.basisChang(targetQuBit);
    DJ.printF("qbb0_b2", qbb0_b2);
    // qbb0_b2:F 
    // ┌           ┐         ┌ ┐        ┌ ┐       ┌ ┐
    // │ 0.61237+0i│= 1/√2 (│1│? (√3/2│1│-   1/2│0│)
    // │-0.35355+0i│         │0│        │0│       │1│
    // │ 0.35355+0i│         └ ┘        └ ┘       └ ┘
    // │ 0.61237+0i│         ┌ ┐        ┌ ┐       ┌ ┐ 
    // └           ┘        +│0│? (  1/2│1│+ √3/2│0│))
    //                         │1│        │0│       │1│
    //                         └ ┘        └ ┘       └ ┘
    DJ.print("  織姫の量子ビットが[1 0]'へ遷移する確率は、\n"
           + "  量子ビットqbb0_b2の第０項あるいは１項に\n"
           + "  遷移する確率は共に(1/√2)^2=1/2です。");
    
    DJ._print("・全量子ビットの遷移確率を求めます。");
    double[][] tpb0b2 = qbb0_b2.getTransProb();
    DJ.printF("tpb0b2", tpb0b2);
    //  tpb0b2=
    //  [[0.5, 0.5] ←[1 0]'への遷移確率
    //  [0.5, 0.5]  ←[0 1]'への遷移確率
    //  ] ↑   ↑
    //    │   └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("　基底b0_b2の第０項に遷移する確率P(0_0)は1/2です。");
    
    DJ._print("・右辺P(b0_0, b2_1) の確率は、qbb0_b2の第１確率振幅の自乗");
    Complex b0b2pa1 = qbb0_b2.getProbAmp(QuBit.PA1);
    Complex tpb0b2pa1 = b0b2pa1.product(b0b2pa1);
    DJ.printF("P(b0_0, b2_1)", tpb0b2pa1); // 1/8
    // P(b0_0, b2_1):F 0.125-0i
    
    DJ._print("・基底b0とb2で同時に測定した場合の、\n"
            + "  各基底への遷移確率P(b0, b2) を求めます。");
    CompVec b0_b2CompVec = qbb0_b2.copyProbAmpVec(); // 確率振幅
    b0_b2CompVec.product(b0_b2CompVec); // 確率振幅を自乗
    DJ.printF("P(b0, b2)", b0_b2CompVec);
    // P(b0, b2):F 
    //┌         ┐
    //│ 0.375+0i│
    //│ 0.125-0i│← P(b0_0, b2_1) = 1/8
    //│ 0.125+0i│
    //│ 0.375+0i│
    //└         ┘
    DJ.print("　右辺第０項P(b0_0, b2_1) = 1/8");

    
    DJ.printS(2, "BellsInequality(1-7)", this); // Pause Task ------------------
    DJ._print("Code(1-7)ベルの不等式の右辺第１項の確率P(b2_0, b1_1)");

    DJ._print("Code(1-7)①織姫が測定対象量子ビットを基底b2_eで測定");
    DJ.printF("targetQuBit", targetQuBit); // 再表示
    // targetQuBit:F 
    // ┌           ┐
    // │ 0.70711+0i│←────────┐
    // │ 0+0i      │←┐(1) product   │(0) product
    // │ 0+0i      │←┘     = 0.0    │     = 0.5
    // │ 0.70711+0i│←────────┘
    // └           ┘
    DJ.printF("b2_eQuBasis", b2_eQuBasis); // 再表示
    // b2_eQuBasis:F
    //  ┌                                                  ┐
    //  │ 0.86603+0i│ 0+0i      │-0.5+0i    │ -0+0i     │
    //  │ 0+0i      │ 0.86603+0i│ -0+0i     │-0.5+0i    │
    //  │ 0.5+0i    │ 0+0i      │ 0.86603+0i│ 0+0i      │
    //  │ 0+0i      │ 0.5+0i    │ 0+0i      │ 0.86603+0i│
    //  └                                                  ┘
    
    QuBit qbb2e = b2_eQuBasis.basisChang(targetQuBit);
    DJ.printF("qbb2e", qbb2e);
    // qbb2e:F 
    //  ┌           ┐             ┌ ┐ ┌ ┐           ┌ ┐ ┌ ┐
    //  │ 0.61237+0i│=(√3/(2√2)(│1│+│0│)+1/(2√2)(│0│-│0│)
    //  │ 0.35355+0i│             │0│ │0│           │1│ │0│
    //  │-0.35355+0i│             │0│ │0│           │0│ │1│
    //  │ 0.61237+0i│             │0│ │1│           │0│ │0│
    //  └           ┘             └ ┘ └ ┘           └ ┘ └ ┘ 
     DJ.print("・織姫の測定で織姫の量子ビットが０になる、すなわち\n"
            + "  基底b2の第０基底軸か第１基底軸に遷移する確率は、\n"
            + "  (√3/(2√2))^2 + (1/2√2)^2 = 3/8 +1/8 = 1/2 です。");
    
    DJ.printS(1, "BellsInequality(1-7)②", this); // Pause Task ----------------
    DJ._print("Code(1-7)②測定で量子ビットが各基底軸に遷移する確率");
    double[][] tpb2e0 = qbb2e.getTransProb();
    DJ.printF("tpb2e0", tpb2e0);
    //  tpb2e0=
    //  [[0.5, 0.5] ←[1 0]'への遷移確率
    //  [0.5, 0.5]  ←[0 1]'への遷移確率
    //  ] ↑   ↑
    //    │   └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率    
     DJ._print("基底b2の第０基底軸に遷移する確率P(b2_0)は0.5です。");
   
    
    DJ.printS(1, "BellsInequality(1-7)③", this); // Pause Task ----------------
    DJ.print("Code(1-7)③織姫の測定により遷移した量子ビット");
    DJ._print("・織姫の測定で基底b2の第０基底軸に遷移した\n"
            + "　量子ビットqbb2e_0をb2QuBasisから取り出します。");
    
    DJ.printF("b2QuBasis", b2QuBasis); // 再表示
    // b2QuBasis:F
    // ┌                        ┐
    // │ 0.86603+0i│-0.5+0i    │
    // │ 0.5+0i    │ 0.86603+0i│
    // └                        ┘

    DJ._print("・取り出す基底軸は吸収軸である第０基底軸なので、\n"
            + "　インデックスQuBasis.BASIS_0で指定します。");
    QuBit qbb2e_0 = b2QuBasis.getBasisQuBit(QuBasis.BASIS_0);
    DJ.printF("qbb2e_0", qbb2e_0);
    // qbb2e_0:F
    // ┌           ┐ ┌       ┐
    // │ 0.86603+0i│=│√(3)/2│
    // │ 0.5+0i    │ │1/2    │
    // └           ┘ └       ┘


    DJ.printS(1, "BellsInequality(1-7)④", this); // Pause Task ----------------
    DJ._print("Code(1-7)④遷移した量子ビットを彦星が基底b1で測定");
    DJ.printF("b1QuBasis", b1QuBasis); // 再表示
    // b1QuBasis:F
    // ┌                        ┐
    // │ 0.5+0i    │-0.86603+0i│
    // │ 0.86603+0i│ 0.5+0i    │
    // └                        ┘

    QuBit qbb2eb1 = b1QuBasis.basisChang(qbb2e_0);
    DJ.printF("qbb2eb1", qbb2eb1);
    // qbb2eb1:F
    // ┌           ┐ ┌      ┐
    // │ 0.86603+0i│=│ √3/2│
    // │-0.5+0i    │ │  -1/2│・・・確率(-1/2)^2=1/4
    // └           ┘ └      ┘
    DJ.print("　彦星の測定では透過軸方向へ遷移する確率なので、\n"
           + "　第１基底に遷移する場合の確率になります。");
    DJ.print("　第１確率振幅より遷移確率はP(b1_1) = (-1/2)^2 = 1/4");

    
    DJ.printS(1, "BellsInequality(1-7)⑤", this); // Pause Task ----------------
    DJ._print("Code(1-7)⑤基底b1の第１基底軸への遷移確率");
    double[][] tpb2eb1 = qbb2eb1.getTransProb();
    DJ.printF("tpb2eb1", tpb2eb1);
    //  tpb2eb1=
    //  [[0.75] ←[1 0]'への遷移確率
    //  [0.25]  ←[0 1]'への遷移確率
    //  ] ↑  
    //    └ 第０量子ビットの遷移確率
    DJ.print("　第１軸への遷移確率はP(b1_1) = 1/4");
    DJ.print("　よって、右辺第１項P(b2_0, b1_1) = 1/2 * 1/4 = 1/8");
    
    DJ._print("・以上によりBellの不等式が導出されました。\n\n"
      + "　P(b0_0, b1_1) <= P(b0_0, b2_1) + P(b2_0, b1_1)");
    
    
    DJ.printS(2, "BellsInequality(1-7)⑥", this); // Pause Task ------------------
    DJ._print("Code(1-7)⑥織姫と彦星が同時に測定する場合");
    
    DJ.print("・基底b2とb1を相互作用させ、２量子基底b2_b1を生成");
    QuBasis b2_b1QuBasis = b2QuBasis.interactQuBasis(b1QuBasis);
    DJ.printF("b2_b1QuBasis", b2_b1QuBasis);
    // b2_b1QuBasis:F
    //┌                                                  ┐
    //│ 0.43301+0i│-0.75+0i   │-0.25+0i   │ 0.43301-0i│
    //│ 0.75+0i   │ 0.43301+0i│-0.43301+0i│-0.25+0i   │
    //│ 0.25+0i   │-0.43301+0i│ 0.43301+0i│-0.75+0i   │
    //│ 0.43301+0i│ 0.25+0i   │ 0.75+0i   │ 0.43301+0i│
    //└                                                  ┘
    
    // b2 ? b1
    //  ┌                ┐ ┌                ┐
    // =│ √3/2 │ -1/2  │?│ 1/2   │-√3/2 │
    //  │ 1/2   │ √3/2 │ │ √3/2 │ 1/2   │
    //  └                ┘ └                ┘
    //  ┌                                 ┐
    // =│ √3/4 │-3/4   │-1/4   │ √3/4│
    //  │ 3/4   │ √3/4 │-√3/4 │-1/4  │
    //  │ 1/4   │-√3/4 │ √3/4 │-3/4  │
    //  │ √3/4 │ 1/4   │ 3/4   │ √3/4│
    //  └                                 ┘
    
    DJ._print("・基底b2_b1の各基底軸に遷移する確率");
    QuBit qbb2_b1 = b2_b1QuBasis.basisChang(targetQuBit);
    DJ.printF("qbb2_b1", qbb2_b1);
    // qbb2_b1:F 
    //┌           ┐         ┌ ┐        ┌ ┐       ┌ ┐
    //│ 0.61237+0i│= 1/√2 (│1│? (  1/2│1│- √3/2│0│)
    //│-0.35355+0i│         │0│        │0│       │1│
    //│ 0.35355+0i│         └ ┘        └ ┘       └ ┘
    //│ 0.61237+0i│         ┌ ┐        ┌ ┐       ┌ ┐ 
    //└           ┘        +│0│? (√3/2│1│+   1/2│0│))
    //                        │1│        │0│       │1│
    //                        └ ┘        └ ┘       └ ┘
    DJ.print("  織姫の量子ビットが[1 0]'へ遷移する確率は、\n"
           + "  量子ビットqbb2_b1の第０項あるいは１項に\n"
           + "  遷移する確率は(1/√2)^2=1/2です。");
    
    DJ._print("・全量子ビットの遷移確率を求めます。");
    double[][] tpb2b1 = qbb2_b1.getTransProb();
    DJ.printF("tpb2b1", tpb2b1);
    //  tpb2b1=
    //  [[0.5, 0.5] ←[1 0]'への遷移確率
    //  [0.5, 0.5]  ←[0 1]'への遷移確率
    //  ] ↑   ↑
    //    │   └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("　基底b2_b1の第０項に遷移する確率P(2_0)は1/2です。");
    
    DJ._print("・右辺の確率P(b2_0, b1_1) は、第１確率振幅の自乗");
    Complex qbb2b1pa1 = qbb2_b1.getProbAmp(QuBit.PA1);
    Complex tpb2b1pa1 = qbb2b1pa1.product(qbb2b1pa1);
    DJ.printF("P(b2_0, b1_1)", tpb2b1pa1); // 1/8
    // P(b2_0, b1_1):F 0.125-0i
    
    DJ._print("・基底b2とb1で同時に測定した場合の、\n"
            + "  各基底への遷移確率P(b2, b1) を求めます。");
    CompVec b2_b1CompVec = qbb2_b1.copyProbAmpVec(); // 確率振幅
    b2_b1CompVec.product(b2_b1CompVec); // 確率振幅を自乗
    DJ.printF("P(b2, b1)", b2_b1CompVec);
    // P(b2, b1):F 
    //┌         ┐
    //│ 0.375+0i│
    //│ 0.125-0i│← P(b2_0, b1_1) = 1/8
    //│ 0.125+0i│
    //│ 0.375+0i│
    //└         ┘
    
    DJ.print("　右辺第１項P(b2_0, b1_1) = 1/8");
    

    
    DJ.printS(2, "BellsInequality(1-8)", this); // Pause Task ------------------
    DJ.print("Code(1-8)測定結果のまとめ");
    // DJ.printF_("  theta[deg]=", 180.0 * theta / Math.PI); // theta[deg]=:F  30
    // DJ.printF(", theta[rad]=", theta); // theta[rad]=:F  0.5236

    DJ.print("Code(1-8)①各辺の確率");
    double tpb0e_0 = tpb0e[0][0]; // 織姫測定分の確率 P(0_0)
    double tpb0eb1_1 = tpb0eb1[1][0]; // 彦星測定分の確率 P(1_1)
    double tpb0e_0b1_1 = tpb0e_0 * tpb0eb1_1; // 左辺確率 0.5 * 0.75 = 0.375
    DJ.printF("・左辺の確率 P(b0_0, b1_1)=", tpb0e_0b1_1);

    double tpb0eb2_1 = tpb0eb2[1][0]; // 彦星測定分の確率 P(2_1)
    double tpb0e_0b2_1 = tpb0e_0 * tpb0eb2_1; // 右辺第０項確率 0.5 * 0.25 = 0.125 
    DJ.printF("・右辺第０項確率 P(b0_0, b2_1)=", tpb0e_0b2_1); // 0.125

    double tpb2e0_0 = tpb2e0[0][0]; // 織姫測定分の確率
    double tpb2eb1_1 = tpb2eb1[1][0];
    double tpb2_0b1_1 = tpb2e0_0 * tpb2eb1_1; // 右辺第１項確率 0.5 * 0.25 = 0.125
    DJ.printF("・右辺第１項確率 P(b2_0, b1_1)=", tpb2_0b1_1); // 0.125

    DJ._print("・測定結果");
    // 左辺の確率 P(b0_0, b1_1)
    DJ.printF_("　左辺の確率 (", tpb0e_0b1_1); // 0.375
    // 右辺の確率 P(b0_0, b2_1)+P(b2_0, b1_1)
    DJ.printF_(") >= 右辺の確率(", tpb0e_0b2_1 + tpb2_0b1_1); // 0.25
    DJ.print(")");
    
    DJ._print("・このように、基底の回転角度が３０度と６０度の場合は、\n"
        + "  ベルの不等式が成立しません。\n"
        + "  実際の実験結果は量子論的に導かれた結果と一致し、\n"
        + "  局所実在論的に導かれる結果にはなりません。");

    DJ.print("・その理由は局所実在論での仮定（局所性と実在性）が誤っている\n"
        + "  為と考えられます。おそらく、相対論的な局所性は満たされている\n"
        + "  でしょうから、実在性に誤りがあるのではないかと推測されます。");
      
      
    DJ.printS(1, "BellsInequality(1-8)②", this); // Pause Task ----------------
    DJ._print("Code(1-8)②計算過程の確認");
    DJ._print(
        "    ┌───────────────────────────┐     \n"
      + "    │      　            (b0_0,b1_1,b2_1)                  │     \n"
      + "    │                    左辺・右辺第１項                  │     \n"
      + "    │      　          ┌─────────────────┼─┐ \n"
      + "┌─┼─────────┼────────┐                │　│ \n"
      + "│  │      　          │                │                │  │ \n"
      + "│  │(b0_0,b1_0,b2_1)  │(b0_0,b1_0,b2_0)│(b0_0,b1_1,b2_0)│  │ \n"
      + "│  │   右辺第１項     │　              │左辺・右辺第２項│  │ \n"
      + "│  └─────────┼────────┼────────┘  │ \n"
      + "│                      │                │  (b0_1,b1_1,b2_0)  │ \n"
      + "│  (b0_1,b1_0,b2_1)    │(b0_1,b1_0,b2_0)│    右辺第２項      │ \n"
      + "│                      └────────┼──────────┘ \n"
      + "└────────────────────┘                       ");
    
    DJ._print("・計算過程の変数");
    DJ.printF("　tpb0e_0", tpb0e_0); // P(0_0,0_1) = 0.5   織姫測定分の確率
    DJ.printF("　tpb0eb1_1", tpb0eb1_1); // P(0_0,1_1) = 0.75  彦星測定分の確率
    DJ.printF("　tpb0eb2_1", tpb0eb2_1); // P(0_0,2_1) = 0.25  彦星測定分の確率
    DJ.printF("　tpb2e0_0", tpb2e0_0); // P(2_0,2_1) = 0.5   織姫測定分の確率
    DJ.printF("　tpb2eb1_1", tpb2eb1_1); // P(2_0,1_1) = 0.25  彦星測定分の確率


    
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(3, "BellsInequality(2)", this); // Pause Task --------------------
    DJ._print("Code(2)測定用基底角度θを変化させて、確率をグラフ表示");
    
    // DJ._print("PatternViewerへの参照を得る"); 
    patternViewer = patternViewerLauncher.getPatternViewer();
    
    DJ._print("・グラフ表示用変数の準備します。");
    int angle; // 基底の回転角度
    // int delta = 1; // 基底の回転角度の変化分
    // theta = Math.PI * angle / 180.0;
    int iMax = (int)(90 / delta) + 1; // 計算の繰り返し回数
    if (epoch > iMax) epoch = iMax;
    DJ.print("　エポックの回数：epoch = ",epoch); // 70
    DJ.print("　計算の繰り返し回数：iMax = ",iMax); // 91
    DJ.print("　経過表示間隔：interval = ",interval); // 1
    
    DJ.print("・基底b0は変化しないのでグラフ表示中は生成しません。");
    // b0QuBasis = new QuBasis(numOfQuBit); // 基底b0（標準基底）
    DJ.printF("b0QuBasis", b0QuBasis);
    
    DJ._print("・基底b0と恒等基底eを相互作用させ、"
            + "  ２量子ビット基底b0_eを生成します。");
    DJ.print("  基底b0_eを 使うのは２量子ビットを測定する織姫だけです。");
    // b0_eQuBasis = b0QuBasis.interactQuBasis(eQuBasis);
    DJ.printF("b0_eQuBasis", b0_eQuBasis);
    
    
    DJ.printS(0, "BellsInequality(2-1)", this); // Pause Task --------------------
    DJ._print("Code(2-1)確率グラフの描画ループ開始");
    for (int i = 0; i <= epoch; i++) {
      
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | 
                     (i == epoch); // 経過表示フラグ
      
      DJ.print_("===== 角度θを変えて計算：i=", i);
      angle = i * delta;
      DJ.print_(", angle[deg]=", angle);
      theta = Math.PI * angle / 180.0;
      DJ.printF_(", theta[rad]=", theta);
      DJ.print(" =====");
      
      
      // ― BellsInequality ― Code(2-2) ――――――
      // DJ._print("Code(2-2)測定用基底軸の更新");
      
      // DJ._print("・基底b0は変化しないので生成しません。");
      // b0QuBasis = new QuBasis(numOfQuBit); // 基底b0（標準基底）
      // DJ.printF("b0QuBasis", b0QuBasis);
      // b0_eQuBasis = b0QuBasis.interactQuBasis(eQuBasis);
      // DJ.printF("b0_eQuBasis", b0_eQuBasis);
      
      // DJ._print("・基底b1を生成します。（回転角を2θで指定）");
      b1QuBasis = new QuBasis(theta * 2); // 角度を2θで指定
      // DJ.printF("b1QuBasis", b1QuBasis);
      // DJ.print("・２量子ビット基底b1_eQuBasisを生成します。");
      // DJ.print("・この基底b1_eQuBasisは使われません。");
      // b1_eQuBasis = b1QuBasis.interactQuBasis(eQuBasis);
      // // DJ.printF("b1_eQuBasis", b1_eQuBasis);
      
      // DJ._print("・基底b2を生成します。（回転角をθで指定）");
      b2QuBasis = new QuBasis(theta); // 角度をθで指定
      // DJ.printF("b2QuBasis", b2QuBasis);
      // DJ.print("・２量子ビット基底b2_eQuBasisを生成します。");
      b2_eQuBasis = b2QuBasis.interactQuBasis(eQuBasis);
      // DJ.printF("b2_eQuBasis", b2_eQuBasis);
      
      
      // ― BellsInequality ― Code(2-3) ――――――
      // DJ._print("Code(2-3)測定と遷移確率の算出");
      
      // DJ._print("・左辺P(b0_0, b1_1)が選択される確率");
      // QuBit qbb0e_0 = b0QuBasis.getBasisQuBit(QuBasis.BASIS_0); // 一定
      // qbb0e = b0_eQuBasis.basisChang(targetQuBit); // 一定
      // DJ.printF("qbb0e", qbb0e;
      // DJ.print("　彦星の測定では測定結果が選んだ基底と一致しない場合の\n"
      //       + "　確率ですので、第１基底に遷移する場合の確率になります。");
      
      // DJ._print("右辺第０項P(b0_0, b2_1)が選択される確率");
      qbb2e_0 = b2QuBasis.getBasisQuBit(QuBasis.BASIS_0);
      // DJ.printF("qbb0eb2", qbb0eb2);
      
      // DJ._print("・右辺第１項P(b2_0, b1_1)が選択される確率");
      // DJ._print(" 織姫が量子ビットtargetQuBitを基底b2_eQuBasisで測定");
      qbb2e = b2_eQuBasis.basisChang(targetQuBit);
      // DJ.printF("qbb2e", qbb2e);
      
      // DJ._print(" 彦星が量子ビットqbb2e_0を基底b1（b1QuBasis）で測定");
      qbb2eb1 = b1QuBasis.basisChang(qbb2e_0);
      // DJ.printF("qbb2eb1", qbb2eb1);
      
      
      // ― BellsInequality ― Code(2-4) ――――――
      // DJ._print("Code(2-4)遷移確率の算出（続き）");
      
      // 左辺 P(b0_0, b1_1)
      tpb0e = qbb0e.getTransProb(); // P(b0_0)
      tpb0e_0 = tpb0e[0][0];
      
      qbb0e_0 = b0QuBasis.getBasisQuBit(QuBasis.BASIS_0);
      qbb0eb1 = b1QuBasis.basisChang(qbb0e_0);
      tpb0eb1 = qbb0eb1.getTransProb();
      
      tpb0eb1_1 = tpb0eb1[1][0]; // 彦星測定分の確率 P(1_1)
      tpb0e_0b1_1 = tpb0e_0 * tpb0eb1_1; // 左辺確率
      // DJ._printF("・左辺確率 P(b0_0, b1_1)=", tpb0e_0b1_1);

      // 右辺第０項 P(b0_0, b2_1)
      qbb0eb2 = b2QuBasis.basisChang(qbb0e_0);
      tpb0eb2 = qbb0eb2.getTransProb();
      
      tpb0eb2_1 = tpb0eb2[1][0]; // 彦星測定分の確率 P(2_1)
      tpb0e_0b2_1 = tpb0e_0 * tpb0eb2_1; // 右辺第０項確率 
      // DJ.printF("・右辺第０項確率 P(b0_0, b2_1)=", tpb0e_0b2_1);

      // 右辺第１項 P(b2_0, b1_1)
      tpb2e0 = qbb2e.getTransProb();
      tpb2e0_0 = tpb2e0[0][0]; // 織姫測定分の確率
      tpb2eb1 = qbb2eb1.getTransProb(); // 遷移確率
      tpb2eb1_1 = tpb2eb1[1][0]; // 彦星測定分の確率
      tpb2_0b1_1 = tpb2e0_0 * tpb2eb1_1; // 右辺第１項確率
      // DJ.printF("・右辺第１項確率 P(b2_0, b1_1)=", tpb2_0b1_1);

      // 左辺の確率 P(b0_0, b1_1)
      DJ.printF_("　左辺の確率 (", tpb0e_0b1_1);
      // 右辺の確率 P(b2_0, b1_1)+P(b2_0, b1_1)
      double tpb0b2_b2b1 = tpb0e_0b2_1 + tpb2_0b1_1; // 右辺の確率 
      if (tpb0e_0b1_1 > tpb0b2_b2b1) DJ.printF_(") > (", tpb0b2_b2b1);
      else                          DJ.printF_(") <= (", tpb0b2_b2b1);
      DJ.print(") 右辺の確率");      


      // ― BellsInequality ― Code(2-5) ――――――
      // DJ._print("Code(2-5)グラフ描画データの設定");

      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

       // 経過表示インターバル
      if (intervalFlag) { // 実行時間に影響を与える
        
        // 角度と確率をパターン・ビューワに渡す
        patternData0[0][i] = theta; // 測定用基底b1,b2の回転角度
        patternData0[1][i] = tpb0e_0b1_1; // 左辺の確率（青線）
        patternData0[2][i] = tpb0b2_b2b1; // 右辺の確率（赤線）
        patternData0[3][i] = 10.0; // ダミー
        patternData0[4][i] = 10.0; // ダミー

        patternData1[0][i] = theta;
        patternData1[1][i] = tpb0e_0b1_1; // 左辺の確率（青線）
        patternData1[2][i] = tpb0b2_b2b1; // 右辺の確率（赤線）
        patternData1[3][i] = tpb0e_0b2_1; // 右辺第０項の確率（黒線）
        patternData1[4][i] = tpb2_0b1_1; // 右辺第１項の確率（青細線）

        // updateViewer(i);
        updatePattern(); // パターン表示を更新する

        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            // wait(SLEEP_TIME); // タイムアウト付きで待機状態
            wait(100); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
    
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch ---------------------------------------");
      String id = Integer.toString(i + 1);
      DJ.printS(0, "BellsInequality(2-" + id + ")", this); // Pause Task ------------------
    } // i
    // DJ._print(" End of all epoch -----------------------------------------");

    DJ._print("グラフ表示終了");
    
    // 処理時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print_("・タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
    DJ.print(", タスク終了日時：", TimeStamp.getTimeFormated());
    

    DJ._print("量子計算の完了");
    DJ.print("BellsInequality.bellsInequality() =============================");
  } // bellsInequality()

} // BellsInequality Class

// End of file

