/*
 *  Title: DaiJa_V4 ( Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package util;

/**
 * <p> 表　題: Class: Normal</p>
 * <p> 説　明: 正規化（標準化）と非正規化（逆変換）</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.07.02</p>
 */
public class Normal {

  // 逆変換のために保存
  private double average = 0.0; // 正規化に用いた平均
  private double stdDev = 1.0; // 正規化に用いた標準偏差

  /**
   * ２次の配列の全要素の平均を得る
   * @param original double[][] // ２次の配列
   * @return average_ double // 全要素の平均値
   */
  public double average(double[][] original) {
    double sum = 0.0; // 合計
    double average_; // 平均
    
    // 合計を求める
    int row = original.length;
    int col = original[0].length;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        sum = sum + original[i][j];
      }
    }
      
    // 平均を求める
    int elementNumber = row * col;
    if (elementNumber <= 0) { // ゼロ割をチェック
        LogEditor.print("***** ERROR ***** Normal.avarage(): ");
        LogEditor.print("Element number of array <= 0.");
        average_ =  Double.NaN;
    }
    else {
      average_ = sum / (double)elementNumber;
    }
      
    return average_;
  } // average()
   
  /**
   * ２次の配列の全要素の標準偏差を得る
   * @param original double[][] // ２次の配列
   * @return stdDev_ double // 全要素の標準偏差
   */
  public double stdDev(double[][] original) {
    double stdDev_; // 標準偏差
    double average_ = average(original); // 全要素の平均

    // 平均値を引いた値を平方し、その合計を求める
    double sum = 0.0;  
    int row = original.length;
    int col = original[0].length;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        double diff = original[i][j] - average_;
        sum = sum + diff * diff;
      }
    }
      
    // 標準偏差を求める
    int elementNumber = row * col;
    if (elementNumber <= 0) { // ゼロ割をチェック
      LogEditor.print("***** ERROR ***** Normal.avarage(): ");
      LogEditor.print("Element number of array <= 0.");
      stdDev_ = Double.NaN;
    }
    else {
      stdDev_ = Math.sqrt(sum / elementNumber);
    }
    
    return stdDev_;
  } // stdDev()
    
  /**
   * ２次の配列の全要素を正規化
   * 計算式：normalizedArray = normalize(original)
   * @param original double[][] // 正規化された２次の配列
   * @return normalizedArray Double[][] // 全要素で正規化されたダブル・テンソル
   */
  public double[][] normalize(double[][] original) {
    average = average(original); // 全要素の平均
    stdDev = stdDev(original); // 全要素の標準偏差
    if (stdDev <= Double.MIN_VALUE) { // ゼロ割を回避
      LogEditor.print("***** ERROR ***** DJ.normalize(): ");
      LogEditor.print("Standard Deviation is neary Zero.");
      return null;
    }
    
    int row = original.length;
    int col = original[0].length;
    double[][] normalizedArray = new double[row][col];
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        normalizedArray[i][j] = (original[i][j] - average) / stdDev;
      } // col
    } // row
    return normalizedArray;
  } // normalize()
  
  /**
   * 正規化された領域の値を元の領域の値に戻す
   * @param normalizedValue double // 正規化された領域の値
   * @return originalValue double元の領域の値
   */
  public double reverse(double normalizedValue) {
    double originalValue = normalizedValue * stdDev + average;
    return originalValue;
  }
  
  /**
   * 正規化された配列を元の配列に戻す
   * @param normalizedArray double[][] // 正規化された領域の配列
   * @return originalArrayy double[][] // 元の領域の配列
   */
  public double[][] reverse(double[][] normalizedArray) {
    int row = normalizedArray.length;
    int col = normalizedArray[0].length;
    double[][] originalArray = new double[row][col]; 
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        originalArray[i][j] = normalizedArray[i][j] * stdDev + average;
      } // col
    } // row
    
    return originalArray;
  }
  
} // Normal Class

// EOF

