/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2024
 */
package util.comp;

import java.util.ArrayList;
import quan.QuBit;
import util.DJ;

/**
 * <p> 表　題: Class: CompVec</p>
 * <p> 説　明: 複素数配列</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024</p>
 * <p> 作成日: 2024.03.04</p>
 */
public class CompVec {
  
  public static final boolean H_VEC = false;  // 行（横）ベクトル
  public static final boolean V_VEC = true; // 列（縦）ベクトル
  
  private Complex[] complexArray; // 複素数配列
  
  /**
   * デフォルト（要素0個）の複素数配列を生成する
   */
  public CompVec() {
//    complexArray = new Complex[1];
//    complexArray[0] = new Complex();
  }
  
  /**
   * 引数で与えられた長さの複素数配列を生成する
   * @param length int // 複素数配列の長さ
   */
  public CompVec(int length) {
    complexArray = new Complex[length];
    
    for (int i = 0; i < length; i++) {
      complexArray[i] = new Complex();
    }
  }
  
  /**
   * 引数で与えられた長さと実数で複素数配列を生成する
   * @param length int // 複素数配列の長さ
   * @param real double // 実数部の値
   * @param image double // 虚数部の値
   */
  public CompVec(int length, double real, double image) {
    complexArray = new Complex[length];
    
    for (int i = 0; i < length; i++) {
      complexArray[i] = new Complex(real, image);
    }
  }
  
  /**
   * 引数で与えられた長さと複素数で複素数配列を生成する
   * @param length int // 複素数配列の長さ
   * @param complex Complex // 引数で与えられた複素数
   */
  public CompVec(int length, Complex complex) {
    complexArray = new Complex[length];
    
    for (int i = 0; i < length; i++) {
      complexArray[i] = complex.copyComplex();
    }
  }
  
  /**
   * 引数で与えられた長さと複素数で複素数配列を生成する
   * @param length int // 複素数配列の長さ
   * @param index int // 実部を1.0にする要素のインデックス
   */
  public CompVec(int length, int index) {
    complexArray = Complex.complexArray(length);
    complexArray[index].setComplex(1.0, 0.0);
  }
  
  /**
   * 実数の配列で複素数配列を生成する
   * @param dataArray double[] // 実数の配列[real, image, , , ]
   */
  public CompVec(double[] dataArray) {
    int dataLength = dataArray.length;
    if (dataLength%2 != 0) {
      DJ.print("***** ERROR ***** "
          + "CompVec.CompVec(double[] dataArray)" + "\n"
          + " Number of data (" + dataLength
          + ") is not even number.");
    }
    
    int compLength = dataLength / 2;
    complexArray = new Complex[compLength];
    for (int i = 0; i < compLength; i++) {
      int i2 = i*2;
      complexArray[i] = new Complex(dataArray[i2], dataArray[i2 + 1]);
    }
  }

  /**
   * 複素数配列で複素数配列を生成する
   * @param complexArray Complex[] // 複素数配列[comp, comp, , , ]
   */
  public CompVec(Complex[] complexArray) {
    int compLength = complexArray.length;
    this.complexArray = new Complex[compLength];
    for (int i = 0; i < compLength; i++) {
      this.complexArray[i] = complexArray[i].copyComplex();
    }
  }
  
  /**
   * 量子ビットリストで複素数配列を生成する
   * @param quBitList ArrayList＜QuBit＞ // 量子ビットリスト
   */
  public CompVec(ArrayList<QuBit> quBitList) {
    int qbListLength = quBitList.size();
    int compLength = qbListLength * 2;
//    if (compLength != (qbListLength * 2)) {
//      DJ.print("***** ERROR ***** "
//          + "CompVec.CompVec(ArrayList<QuBit> quBitList)" + "\n"
//          + " Number of QuBits (" + compLength
//          + ") is not equal to quBitList_size(" + qbListLength + "*2.");
//    }
    complexArray = new Complex[compLength];
    for (int i = 0; i < qbListLength; i++) {
      QuBit quBit = quBitList.get(i);
      complexArray[i * 2] = quBit.copyProbAmp(0);
      complexArray[i * 2 + 1] = quBit.copyProbAmp(1);
    }    
  }
  
  
  
  

  // ----------------------------------------------------
  /**
   * 複素数配列の要素の実部を得る
   * @param index int // 要素のインデックス
   * @return real Double // 実部
   */
  public double getReal(int index) {
    return complexArray[index].getReal();
  }

  /**
   * 複素数配列の要素の実部を設定する
   * @param index int // 要素のインデックス
   * @param real Double // 実部
   */
  public void setReal(int index, double real) {
    complexArray[index].setReal(real);
  }

  /**
   * 複素数配列の要素の虚部を得る
   * @param index int // 要素のインデックス
   * @return imag Double // 虚部
   */
  public double getImag(int index) {
    return complexArray[index].getImag();
  }

  /**
   * 複素数配列の要素の虚部を設定する
   * @param index int // 要素のインデックス
   * @param imag Double // 虚部
   */
  public void setImag(int index, Double imag) {
    complexArray[index].setImag(imag);
  }

  /**
   * 複素数配列の要素を得る
   * @param index int // 要素のインデックス
   * @return complex Complex //  要素（規複素数数)
   */
  public Complex getComp(int index) {
    return complexArray[index];
  }

  /**
   * 複素数配列の要素を新規複素数で得る
   * @param index int // 要素のインデックス
   * @return complex Complex //  要素（規複素数数)
   */
  public Complex copyComp(int index) {
    return complexArray[index].copyComplex();
  }

  /**
   * 複素数配列の要素を複素数のプリミティブで設定する
   * @param index int // 要素のインデックス
   * @param complex Complex // 要素（規複素数数)
   */
  public void setComp(int index, Complex complex) {
    complexArray[index].setComplex(complex);
  }

  /**
   * 複素数配列の要素をプリミティブで設定する
   * @param index int // 要素のインデックス
   * @param real Double // 実部
   * @param imag Double // 虚部
   */
  public void setComp(int index, double real, double imag) {
    complexArray[index].setComplex(real, imag);
  }
  
  /**
   * 複素数配列を複素数配列で得る
   * @return complexArray Complex[] // 複素数配列
   */
  public Complex[] getArray() {
    return complexArray;
  }
  
  /**
   * 複素数配列を新規複素数配列クラスで得る
   * @return _compVec CompVec // 新規複素数配列
   */
  public CompVec copyCompVec() {
    int length = complexArray.length;
    CompVec _compVec = new CompVec(length);
    for (int i = 0; i < length; i++) {
      _compVec.setComp(i, complexArray[i]);
    }
    return _compVec;
  }

  /**
   * 複素数配列を新規複素数配列で得る
   * @return complexArray Complex[] // 複素数配列
   */
  public Complex[] copyArray() {
    int length = complexArray.length;
    CompVec _compVec = new CompVec(length);
    for (int i = 0; i < length; i++) {
      _compVec.setComp(i, complexArray[i]);
    }
    return _compVec.complexArray;
  }
  
  /**
   * 複素数配列を複素数配列のプリミティブで設定する（）
   * @param operandArray complex[] // 複素数配列
   */
  public void setArray(Complex[] operandArray) {
    complexArray = operandArray;
  }
  
  /**
   * 複素数配列を複素数配列のプリミティブで設定する（）
   * @param operandArray complex[] // 複素数配列
   */
  public void copyArray(Complex[] operandArray) {
    int length = complexArray.length;
    int operandLength = operandArray.length;
    if (length != operandLength) {
      DJ.print("***** ERROR ***** "
          + "CompVec.product(vector)" + "\n"
          + " Array length are not same.\n"
          + " target array length is " + length + ", \n"
          + " operand array length is " + operandLength + ".");
      return;
    }
    for (int i = 0; i < length; i++) {
      complexArray[i].setComplex(operandArray[i]);
    }
  }
  
  
  
  // ------------------------------------------------------
  /**
   * 複素数配列の長さを得る
   * @return length int // 複素数配列の長さ
   */
  public int length() {
    return complexArray.length;
  }
  
  
  
  // ------------------------------------------------------
  
  /**
   * 複素数配列と複素数配列の
   * 対応する要素毎の和を求める
   * @param operand CompVec // 演算数複素数配列
   */
  public void sum(CompVec operand) {
    int length = complexArray.length;
    if (length < 1) {
      DJ.print("***** ERROR ***** "
          + "CompVec.sum(CompVec)" + "\n"
          + " target vector length (=" + length
          + ") is less than one.");
      return;
    }
    int operandLength = operand.length();
    if (length != operandLength) {
      DJ.print("***** ERROR ***** "
          + "CompVec.sum(vector)" + "\n"
          + " Vecter length are not same.\n"
          + " target vector length is " + length + ", \n"
          + " operand vecter length is " + operandLength + ".");
      return;
    }

    Complex[] operandArray = operand.getArray();
    for (int i = 0; i < length; i++) {
      complexArray[i].add(operandArray[i]);
    }
  }
  
  /**
   * 複素数配列の実数倍
   * @param real double // 乗算実数
   */
  public void product(double real) {
    int length = complexArray.length;
    for (int i = 0; i < length; i++) {
      complexArray[i].mul(real);
    }
  }
  
  /**
   * 複素数配列の複素数倍
   * @param comp Complex // 乗算複素
   */
   public void product(Complex comp) {
    int length = complexArray.length;
    for (int i = 0; i < length; i++) {
      complexArray[i].mul(comp);
    }
   }
  
  /**
   * 複素数配列と複素数配列の
   * 対応する要素毎の積を求める
   * @param operand CompVec // 演算数複素数配列
   */
  public void product(CompVec operand) {
    int length = complexArray.length;
    if (length < 1) {
      DJ.print("***** ERROR ***** "
          + "CompVec.product(CompVec)" + "\n"
          + " target vector length (=" + length
          + ") is less than one.");
      return;
    }
    int operandLength = operand.length();
    if (length != operandLength) {
      DJ.print("***** ERROR ***** "
          + "CompVec.product(vector)" + "\n"
          + " Vecter length are not same.\n"
          + " target vector length is " + length + ", \n"
          + " operand vecter length is " + operandLength + ".");
      return;
    }

    Complex[] operandArray = operand.getArray();
    for (int i = 0; i < length; i++) {
      complexArray[i].mul(operandArray[i]);
    }
  }
  
  
  
  
  // ------------------------------------------------------
  
  /**
   * 複素数配列の共役を得る
   * @return _conjugate CompVec // 共役な複素数ベクタ
   */
  public CompVec conjugate() {
    int length = complexArray.length;
    CompVec _conjugate = new CompVec(length);
    for (int i = 0; i < length; i++) {
      _conjugate.setComp(i, complexArray[i].conjugate());
    }
    return _conjugate;
  }

  /**
   * 複素数配列の内積を求める
   * @param operand Complex[] // 演算数複素数ベクトル
   * @return _innerProduct Complex // 内積、複素数
   */
  public Complex innerProduct(CompVec operand) {
    int length = complexArray.length;
    int operandLength = operand.length();
    if (length != operandLength) {
      DJ.print("***** ERROR ***** "
          + "Complex.innerProduct(CompVec)" + "\n"
          + " vecter length are not same."
          + " target vecter length is " + length + ", "
          + " operand length is " + operandLength + ".");
      return null;
    }

    // 被演算数複素数配列を共役配列に置き換える
    CompVec _conjugate = this.conjugate();
    Complex _innerProduct = new Complex(0.0, 0.0);
    for (int i = 0; i < operandLength; i++) {
      Complex conj = _conjugate.getComp(i);
      conj.mul(operand.getComp(i));
      _innerProduct.add(conj);
    }
    return _innerProduct;
  }
   
  /**
   * 複素数配列の自身との内積を求める
   * @return _innerProduct Complex // 内積、複素数
   */
  public Complex innerProduct() {
    int length = complexArray.length;

    // 被演算数複素数配列を共役配列に置き換える
    CompVec _conjugate = this.conjugate();
    Complex _innerProduct = new Complex(0.0, 0.0);
    for (int i = 0; i < length; i++) {
      Complex conj = _conjugate.getComp(i);
      conj.mul(this.getComp(i));
      _innerProduct.add(conj);
    }
    return _innerProduct;
  }
   
  
  
  
  /**
   * 複素ベクトルと複素行列の内積
   * @param operand
   * @return 
   */
  public CompVec innerProduct(CompMat operand) {
    int length = complexArray.length;
    if (length < 1) {
      DJ.print("***** ERROR ***** "
          + "CompVec.innerProduct(CompMat)" + "\n"
          + " target vector length (=" + length
          + ") is less than one.");
      return null;
    }
    CompVec conjuVec = this.conjugate(); // 演算ベクトルの共役ベクトル
    
    int row = operand.rowNumber(); // 演算行列の行数（列の要素数）
    if (length != row) {
      DJ.print("***** ERROR ***** "
          + "CompVec.innerProduct(CompMat)" + "\n"
          + " Vecter length is not equal to row number of Matrix.\n"
          + " target vector length is " + length + ", \n"
          + " operand Matrix row number is " + row + ".");
      return null;
    }
  
    int col = operand.colNumber();
    CompVec resultVec = new CompVec(col);
    Complex[] resultArray = resultVec.complexArray;
    for (int j = 0; j < col; j++) {
      
//      Complex[] operandArray = operand.getRow(j).complexArray;
      CompVec aCompVec = operand.copyColumn(j); // 第ｊ行を取出し
      for (int k = 0; k < length; k++) {
        Complex aComplex = conjuVec.copyComp(k);
        aComplex.mul(operand.getComp(k, j));
        resultVec.getComp(j).add(aComplex);
      }
    }
    
    return resultVec;
  }
  
  
  
  //   private Complex[] complexArray; // 複素数配列


  // ------------------------------------------------------
  // 複素ベクトルの比較
  public boolean equals(CompVec cv){
    
    if (this == cv) return true; // アドレスが同じ

    int thisLength = this.length();
    if (thisLength != cv.length()) return false; // 長さが違う
    
    for (int i = 0; i < thisLength; i++ ) {
      Complex thisComp = complexArray[i];
      if (!thisComp.equals(cv.complexArray[i])) {
        return false; // 値が違う
      }
    }

    return true; // 値が同じ
  }

  
  
  // ==================================================================
  // CompVecクラスでの複素数配列の表示は数学的表示に従う
  // 要素の区切りにカンマを付けない
  // 列（縦）表示では各要素を[]で括る（枠付き表示を除く）
  //
  // public static final boolean H_VEC = false;  // 行（横）ベクトル
  // public static final boolean V_VEC = true; // 列（縦）ベクトル
  
  /**
   * 連続する空白文字列を得る
   * @param size int // 空白文字の個数
   * @return space.toString String // 空白文字列
   */
  private String getSpace(int size) {
    StringBuilder space = new StringBuilder(); // 表示文字列
    for (int i = 0; i < size; i++) {
      space.append(" "); // 空白を追加
    }
    return space.toString();
  }
  
  /**
   * 1.複素数配列の文字列（行表示）を得る
   * @return String // 複素数の文字列
   */
  @Override
  public String toString() {
    int order = complexArray.length;
    StringBuilder sb = new StringBuilder("[");
    for (int i = 0; i < order; i++) {
      sb.append(complexArray[i].toString());
      sb.append("  ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 2.複素数配列の文字列（行・列選択表示）を得る
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public String toString(boolean orient) {
    int order = complexArray.length;
    StringBuilder sb = new StringBuilder("[");
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toString());
      if (orient) sb.append("]\n"); // 列（縦）ベクトル
      else        sb.append("  "); // 行（横）ベクトル
    }
    int index = sb.length();
    if (orient) sb.replace(index - 1, index, "]");
    else        sb.replace(index - 2, index, "]");
    return sb.toString();
  }

//  /**
//   * 3.複素数配列の文字列（行表示、書式付き）を得る
//   * @return String // 複素数の文字列
//   */
//  public String toFormatOld() {
//    int order = complexArray.length;
//    StringBuilder sb = new StringBuilder("F [");
//    for (int i = 0; i < order; i++) {
//      sb.append(complexArray[i].toFormat());
//      sb.append("  ");
//    }
//    int index = sb.length();
//    sb.replace(index - 2, index, "]");
//    return sb.toString();
//  }

  /**
   * 3.複素数ベクトルの文字列（列（縦）表示、枠・書式付き）を得る
   * @return sb.toString() // 複素数の文字列
   */
  public String toFormat() {
    int order = complexArray.length; // ベクトルの次数
    int maxWide = 5; // 数値文字列の長さの最大長 (最小例：-0+0i)
    
    // 数値文字列の抽出
    String[] buf = new String[order]; // 文字列バッファ
    int[] wides = new int[order]; // 余白数
    for (int i = 0; i < order; i++) {
      // DJ._print_("i=", i);
      String text = complexArray[i].toFormat();
      
      int tempWide;
      if (complexArray[i].getReal() >= 0) { // 実部が正
        buf[i] = " " + text;
        tempWide = text.length() + 1; // 数値文字列の長さ
      }
      else {
        buf[i] = text;
        tempWide = text.length(); // 数値文字列の長さ
      }
      // DJ.print(",  text=", text);
      
      wides[i] = tempWide;
      if (tempWide > maxWide) maxWide = tempWide; // wideを更新
      // DJ.print_("tempWide=", tempWide);
      // DJ.print(",  maxWide=", maxWide);
      
    }
    // DJ._print("last maxWide=", maxWide);
    // DJ._print("text=");
    // for (int i = 0; i < order; i++) {
    //   DJ.print_(" ", i); DJ.print(" ", buf[i]);
    // }
    
    // 出力文字列の生成
    StringBuilder sb; // 表示文字列
    sb = new StringBuilder("F \n"); // 列（縦）ベクトル
    sb.append("┌").append(getSpace(maxWide)).append("┐\n");
    for (int i = 0; i < order; i++) {
      sb.append("│");
      sb.append(buf[i]); // 数値
      sb.append(getSpace(maxWide - wides[i])); // 余白
      sb.append("│");
      sb.append("\n"); // 列（縦）ベクトル
    }
//    int index = sb.length();
//    sb.replace(index - 1, index, "");
//    sb.append("└").append(getSpace(maxWide)).append("┘\n");
    sb.append("└").append(getSpace(maxWide)).append("┘");
    return sb.toString();
  }
  
  /**
   * 4.複素数配列の文字列（行・列選択表示、書式付き）を得る
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public String toFormat(boolean orient) {
    int order = complexArray.length;
//    StringBuilder sb = new StringBuilder("F[");
    StringBuilder sb;
    if (orient) sb = new StringBuilder("F \n["); // 列（縦）ベクトル
    else        sb = new StringBuilder("F ["); // 行（横）ベクトル
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toFormat());
      if (orient) sb.append("]\n"); // 列（縦）ベクトル
      else        sb.append("  "); // 行（横）ベクトル
    }
    int index = sb.length();
    if (orient) sb.replace(index - 1, index, "]");
    else        sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 5.複素数配列の文字列（行表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @return sb String // 複素数の文字列
   */
  public String toString(String label) {
    int order = complexArray.length;
    StringBuilder sb = new StringBuilder(label + ":[");
    for (int i = 0; i < order; i++) {
      sb.append(complexArray[i].toString());
      sb.append("  ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 6.複素数配列の文字列（行・列選択表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return sb String // 複素数の文字列
   */
  public String toString(String label, boolean orient) {
    int order = complexArray.length;
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":\n["); // 列（縦）ベクトル
    else        sb = new StringBuilder(label + ":["); // 行（横）ベクトル
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toString());
//      sb.append("]\n");
      if (orient) sb.append("]\n");
      else        sb.append("  ");
    }
    int index = sb.length();
    if (orient) sb.replace(index - 1, index, "]");
    else        sb.replace(index - 2, index, "]");
    return sb.toString();
  }

//  /**
//   * 7.複素数配列の文字列（行表示、ラベル付き、書式付き）を得る
//   * @param label String // 表示用ラベル
//   * @return String // 複素数の文字列
//   */
//  public String toFormatOld(String label) {
//    int order = complexArray.length;
//    StringBuilder sb = new StringBuilder(label + ":F [");
//    for (int i = 0; i < order; i++) {
//      sb.append(complexArray[i].toFormat());
//      sb.append("  ");
//    }
//    int index = sb.length();
//    sb.replace(index - 2, index, "]");
//    return sb.toString();
//  }

  /**
   * 7.複素数配列の文字列（行表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @return String // 複素数の文字列
   */
  public String toFormat(String label) {
    String str = this.toFormat();
    
    return label + ":" + str;
  }
  
  /**
   * 8.複素数配列の文字列（行・列選択表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @param orient boolean // ベクトルの表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public String toFormat(String label, boolean orient) {
    int order = complexArray.length;
//    StringBuilder sb = new StringBuilder(label + ":F\n[");
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":F\n["); // 列（縦）ベクトル
    else        sb = new StringBuilder(label + ":F ["); // 行（横）ベクトル
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toFormat());
      if (orient) sb.append("]\n");
      else        sb.append("  ");
    }
    int index = sb.length();
    if (orient) sb.replace(index - 1, index, "]");
    else sb.replace(index - 2, index, "]");
    return sb.toString();
  }

   
  
  
  // ----------------------------------------------------


  
  


  
  
  
  // ----------------------------------------------------
  // CompVecクラスでの複素数配列の表示は数学的表示に従う
  // 要素の区切りにカンマを付けない
  // 列（縦）表示では各要素を[]で括る
  
  /**
   * 1.複素数配列の文字列（行表示）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toString(Complex[] complexArray) {
    int length = complexArray.length;
    StringBuilder sb = new StringBuilder("[");
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toString());
      sb.append("  ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 2.複素数配列の文字列（行・列選択表示）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public static String toString(Complex[] complexArray, boolean orient) {
    int order = complexArray.length;
    StringBuilder sb = new StringBuilder("[");
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toString());
      if (orient) sb.append("]\n"); // 列（縦）配列
      else        sb.append("  "); // 行（横）配列
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 3.複素数配列の文字列（行表示、書式付き）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toFormat(Complex[] complexArray) {
    int order = complexArray.length;
    StringBuilder sb = new StringBuilder("F[");
    for (int i = 0; i < order; i++) {
      sb.append(complexArray[i].toFormat());
      sb.append("  ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 4.複素数配列の文字列（行・列選択表示、書式付き）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public static String toFormat(Complex[] complexArray, boolean orient) {
    int order = complexArray.length;
    StringBuilder sb;
    if (orient) sb = new StringBuilder("F \n["); // 列（縦）配列
    else        sb = new StringBuilder("F ["); // 行（横）配列
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toFormat());
      if (orient) sb.append("]\n"); // 列（縦）配列
      else        sb.append("  "); // 行（横）配列
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 5.複素数配列の文字列（行表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toString(String label, Complex[] complexArray) {
    int order = complexArray.length;
    StringBuilder sb = new StringBuilder(label + ":[");
    for (int i = 0; i < order; i++) {
      sb.append(complexArray[i].toString());
      sb.append("  ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 6.複素数配列の文字列（行・列選択表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public static String toString(String label, Complex[] complexArray, boolean orient) {
    int order = complexArray.length;
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":\n["); // 列（縦）配列
    else        sb = new StringBuilder(label + ":["); // 行（横）配列
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toString());
      if (orient) sb.append("]\n");
      else        sb.append("  ");
    }
    int index = sb.length();
    if (orient) sb.replace(index - 1, index, "]");
    else        sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 7.複素数配列の文字列（行表示、ラベル付き、書式付き）を得る
   *
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toFormat(String label, Complex[] complexArray) {
    int length = complexArray.length;
    StringBuilder sb = new StringBuilder(label + ":F [");
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toFormat());
      sb.append("  ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 8.複素数配列の文字列（行・列選択表示、ラベル付き、書式付き）を得る
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toFormat(
      String label, Complex[] complexArray, boolean orient) {
    int order = complexArray.length;
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":F\n["); // 列（縦）配列
    else        sb = new StringBuilder(label + ":F ["); // 行（横）配列
    for (int i = 0; i < order; i++) {
      if (orient) sb.append("[");
      sb.append(complexArray[i].toFormat());
      if (orient) sb.append("]\n");
      else        sb.append("  ");
    }
    int index = sb.length();
    if (orient) sb.replace(index - 1, index, "]");
    else        sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
} // CompVec Class

// End of file

