/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2024
 */
package util.comp;

// import java.math.RoundingMode;
import java.text.DecimalFormat;
import quan.QuBit;
import util.DJ;

/**
 * <p> 表　題: Class: Complex</p>
 * <p> 説　明: 複素数</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024</p>
 * <p> 作成日: 2024.02.14</p>
 */
public class Complex {

//  public static final Complex COMP0 = new Complex(0.0, 0.0); // 複素数0+0i
//  public static final Complex COMP1 = new Complex(1.0, 0.0); // 複素数1+0i
  
  public static final boolean H_COMP = false; // 行（横）ベクトル
  public static final boolean V_COMP = true; // 列（縦）ベクトル
    
  private double real;
  private double imag;

  /**
   * デフォルト（ゼロ）の複素数を生成する
   */
  public Complex() {
    this.real = 0.0;
    this.imag = 0.0;
  }
    
  /**
   * 複素数を生成する
   * @param real double // 実部
   * @param imag double // 虚部
   */
  public Complex(double real, double imag) {
    this.real = real;
    this.imag = imag;
  }
  
  /**
   * 複素数配列を生成する
   * @param length int // 配列の長さ
   * @return compAry Complex[] // 複素数配列
   */
  public static Complex[] complexArray(int length) {
    Complex[] compAry = new Complex[length];
    for (int i = 0; i < length; i++) {
      compAry[i] = new Complex();
    }
    return compAry;
  }
  
  /**
   * 複素数行列を生成する
   * @param row int // 行の個数（列の長さ）
   * @param col int // 列の個数（行の長さ）
   * @return compMat Complex[][] // 複素数行列
   */
  public static Complex[][] complexMatrix(int row, int col) {
    Complex[][] compMat = new Complex[row][col];
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        compMat[i][j] = new Complex();
      }
    }
    return compMat;
  }

  /**
   * 単位複素数行列（正方行列）を生成する
   * @param order int // 行列の次数
   * @return identity Complex[][] // 単位複素数行列（正方行列）
   */
  public static Complex[][] identityMatrix(int order) {
    Complex[][] identity = new Complex[order][order];
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        identity[i][j] = new Complex();
      }
      identity[i][i] = new Complex(1.0, 0.0);
    }
    return identity;
  }
  
  /**
   * 実部を得る
   * @return real double // 実部
   */
  public double getReal() {
    return real;
  }

  /**
   * 虚部を得る
   * @return imag double // 虚部
   */
  public double getImag() {
    return imag;
  }

  /**
   * 実部を設定する
   *
   * @param real double // 実部
   */
  public void setReal(double real) {
    this.real = real;
  }

  /**
   * 虚部を設定する
   * @param imag double // 虚部
   */
  public void setImag(double imag) {
    this.imag = imag;
  }

// このメソッドは不要、そのまま引用、代入すればよい
//  /**
//   * 複素数を得る（新規複素数）
//   * @return complex Complex // 新規複素数
//   */
//  public Complex getComplex() {
//    return new Complex(real, imag);
//  }
  
  /**
   * コピーを新規複素数で生成する
   * @return complex Complex // 新規複素数
   */
  public Complex copyComplex() {
    return new Complex(real, imag);
  }
  
  /**
   * 複素数を設定する
   * @param real double // 実部
   * @param imag double // 虚部
   */
  public void setComplex(double real, double imag) {
    this.real = real;
    this.imag = imag;
  }

  /**
   * 複素数を複素数のプリミティブで設定する
   * @param complex Complex // 複素数
   */
  public void setComplex(Complex complex) {
    this.real = complex.getReal();
    this.imag = complex.getImag();
  }
  
  // ----------------------------------------------------
  /**
   * 複素数に実数を足す
   * @param real dobule // 加数
   */
  public void add(double real) {
    this.real += real;
  }
  
  /**
   * 複素数から実数を減じる
   * @param real dobule // 減数
   */
  public void sub(double real) {
    this.real -= real;
  }
  
  /**
   * 複素数に実数を掛ける
   * @param real dobule // 乗数
   */
  public void mul(double real) {
    this.real *= real;
    this.imag *= real;
  }
  
  /**
   * 複素数を実数で割る
   * @param real double // 序数
   */
  public void div(double real) {
    this.real /= real;
    this.imag /= real;
  }
  
  /**
   * 実数を複素数で割る
   * @param real double // 被序数 divided
   * @param comp Complex // 序数 divide
   * @return 
   */
  public static Complex div(double real, Complex comp) {
    double deno = comp.real * comp.real + comp.imag * comp.imag;
    double _real = real * comp.real / deno;
    double _imag = real * comp.imag / deno;
    return new Complex(_real, _imag);
  }
  
  // ----------------------------------------------------
  /**
   * 複素数を加える
   * @param comp Complex // 加える複素数
   */
  public void add(Complex comp) {
    real += comp.real;
    imag += comp.imag;
  }

  /**
   * 複素数を減じる
   * @param comp Complex // 減じる複素数
   */
  public void sub(Complex comp) {
    real -= comp.real;
    imag -= comp.imag;
  }

  /**
   * 複素数を掛ける
   * @param comp Complex // 掛ける複素数
   */
  public void mul(Complex comp) {
    double _real = real * comp.real - imag * comp.imag;
    this.imag = real * comp.imag + imag * comp.real;
    this.real = _real;
  }

  /**
   * 複素数で割る
   * @param comp Complex // 割る複素数
   */
  public void div(Complex comp) {
    double denominator = comp.real * comp.real + comp.imag * comp.imag;
    double _real = (real * comp.real + imag * comp.imag) / denominator;
    imag = (imag * comp.real - real * comp.imag) / denominator;
    this.real = _real;
  }

  // ----------------------------------------------------
  /**
   * １次元複素配列を複製する
   * @param compArray Complex[] // 複製される１次元複素配列
   * @return resultArray Complex[] // 複製された１次元複素配列
   */
  public static Complex[] copyComplex(Complex[] compArray) {
    int length = compArray.length;
    Complex[] resultArray = new Complex[length];
    for (int i = 0; i < length; i++) {
      resultArray[i] = new Complex(compArray[i].real, compArray[i].imag);
    }
    return resultArray;
  }
  
  /**
   * ２次元複素配列を複製する
   * @param compArray Complex[][] // 複製される２次元複素配列
   * @return resultArray Complex[][] // 複製された２次元複素配列
   */
  public static Complex[][] copyComplex(Complex[][] compArray) {
    int row = compArray.length;
    int col = compArray[0].length;
    Complex[][] resultArray = new Complex[row][col];
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        resultArray[i][j] = new Complex(
            compArray[i][j].real, compArray[i][j].imag);
      }
    }
    return resultArray;
  }
  
  /**
   * 複素数の複素数倍
   * @param comp Complex // 被乗算複素数
   * @return _productComplex // 複素数倍された複素数
   */
   public Complex product(Complex comp) {
    double _real = real * comp.real - imag * comp.imag;
    double _imag = real * comp.imag + imag * comp.real;
    Complex _product = new Complex(_real, _imag);
    return _product;
   }
  
  /**
   * １次元複素数配列の実数倍
   * ※引数の内容を書き換える
   * @param target Complex[] // 被乗算複素数配列
   * @param real double // 乗算実数
   */
  public static void product(Complex[] target, double real) {
    int targetLength = target.length;
    for (int i = 0; i < targetLength; i++) {
      target[i].mul(real);
    }
  }
  
  /**
   * １次元複素数配列の複素数倍
   * ※引数の内容を書き換える
   * @param target Complex[] // 被乗算複素数配列
   * @param comp Complex // 乗算複素
   */
   public static void product(Complex[] target, Complex comp) {
    int targetLength = target.length;
    for (int i = 0; i < targetLength; i++) {
      target[i].mul(comp);
    }
   }
  
  /**
   * ２次元複素数配列の実数倍
   * ※引数の内容を書き換える
   * @param target Complex[][] // 被乗算複素数行列
   * @param real  double // 乗算実数
   */
  public static void product(Complex[][] target, double real) {
    int targetLength = target.length;
    int lowLength = target[0].length;
    for (int i = 0; i < targetLength; i++) {
      for (int j = 0; j < lowLength; j++) {
        target[i][j].mul(real);
      }
    }
  }
  
  /**
   * ２次元複素数配列の複素数倍
   * ※引数の内容を書き換える
   * @param target Complex[][] // 被乗算複素数行列
   * @param comp Complex // 乗算複素数
   */
  public static void product(Complex[][] target, Complex comp) {
    int targetLength = target.length;
    int lowLength = target[0].length;
    for (int i = 0; i < targetLength; i++) {
      for (int j = 0; j < lowLength; j++) {
        target[i][j].mul(comp);
      }
    }
  }
    
  /**
   * ２次元複素数配列と１次元複素数配列の積を求める
   * ※共役転置は行わない
   * @param target Complex[][] // 被演算数２次元複素数配列
   * @param operand Complex[] // 演算数複素数ベクトル
   * @return _products Complex[] // 複素数ベクトル
   */
  public static Complex[] product(Complex[][] target, Complex[] operand) {
    int row = target.length;
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "Complex.product(Complex[][] target, Complex[] operand)" + "\n"
          + " tatrget row number (=" + row
          + ") is less than one.");
      return null;
    }
    int col = target[0].length;
    int operandRow = operand.length;
    if (row != operandRow) {
      DJ.print("***** ERROR ***** "
          + "Complex.product(Complex[][] target, Complex[] operand)" + "\n"
          + " Element number are not same.\n"
          + " target row number is " + row + ", \n"
          + " operand row number is " + operandRow + ".");
      return null;
    }

    Complex[] _products = new Complex[col];
    for (int i = 0; i < col; i++) {
      _products[i] = new Complex(0.0, 0.0);
      for (int k = 0; k < row; k++) {
//        Complex conj = 
//            target[k][i].copyComplex(); // 転置のみ
//            target[k][i].copyComplex().conjugate(); // 共役転置
        Complex element = target[i][k].copyComplex();
        element.mul(operand[k]);
        _products[i].add(element);
      }
    }
    return _products;
  }
  
//  /** デバッグ済み
//   * ２次元複素数配列と１次元複素数配列の内積を求める
//   * @param target Complex[][] // 被演算数２次元複素数配列
//   * @param operand Complex[] // 演算数複素数ベクトル
//   * @return _products Complex[] // 複素数ベクトル
//   */
//  public static Complex[] innerProduct(Complex[][] target, Complex[] operand) {
//    int row = target.length;
//    if (row < 1) {
//      DJ.print("***** ERROR ***** "
//          + "Complex.product(Complex[][] target, Complex[] operand)" + "\n"
//          + " tatrget row number (=" + row
//          + ") is less than one.");
//      return null;
//    }
//    int col = target[0].length;
//    int operandRow = operand.length;
//    if (row != operandRow) {
//      DJ.print("***** ERROR ***** "
//          + "Complex.product(Complex[][] target, Complex[] operand)" + "\n"
//          + " Element number are not same.\n"
//          + " target row number is " + row + ", \n"
//          + " operand row number is " + operandRow + ".");
//      return null;
//    }
//
//    Complex[] _products = new Complex[col];
//    for (int i = 0; i < col; i++) {
//      _products[i] = new Complex(0.0, 0.0);
//      for (int k = 0; k < row; k++) {
//        Complex conj = 
////            target[k][i].copyComplex(); // 転置のみ
//            target[k][i].copyComplex().conjugate(); // 共役転置
//        conj.mul(operand[k]);
//        _products[i].add(conj);
//      }
//    }
//    return _products;
//  }

  /**
   * ２次元複素数配列と２次元複素数配列の積を求める
   * ※共役転置は行わない
   * @param target Complex[][] // 被演算数２次元複素数配列
   * @param operand Complex[] // 演算数複素数ベクトル
   * @return _products Complex[] // 複素数ベクトル
   */
  public static Complex[][] product(Complex[][] target, Complex[][] operand) {
    int row = target.length;
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "Complex.product(Complex[][] target, Complex[][] operand)" + "\n"
          + " target row number (=" + row
          + ") is less than one.");
      return null;
    }
    int col = target[0].length;
    int operandRow = operand.length;
    if (row != operandRow) {
      DJ.print("***** ERROR ***** "
          + "Complex.product(Complex[][] target, Complex[][] operand)" + "\n"
          + " Element number are not same.\n"
          + " target row number is " + row + ", \n"
          + " operand row number is " + operandRow + ".");
      return null;
    }

    int operandCol = operand[0].length;
    Complex[][] _products = new Complex[col][operandCol];
    for (int i = 0; i < col; i++) {
      for (int j = 0; j < operandCol; j++) {
        _products[i][j] = new Complex(0.0, 0.0);
        for (int k = 0; k < row; k++) {
          // Complex conj = target[k][i].copyComplex(); // 転置のみ
          // Complex conj = target[k][i].copyComplex().conjugate(); // 共役転置
          Complex conj = target[i][k].copyComplex(); // 転置のみ
          conj.mul(operand[k][j]);
          _products[i][j].add(conj);
        }
      }
    }
    return _products;
  }
  
//  /**
//   * ２次元複素数配列と２次元複素数配列の積を求める
//   * ※被演算数行列は共役転置行列に置き換えられる
//   * @param target Complex[][] // 被演算数２次元複素数配列
//   * @param operand Complex[] // 演算数複素数ベクトル
//   * @return _products Complex[] // 複素数ベクトル
//   */
//  public static Complex[][] innerProduct(Complex[][] target, Complex[][] operand) {
//    int row = target.length;
//    if (row < 1) {
//      DJ.print("***** ERROR ***** "
//          + "Complex.product(Complex[][] target, Complex[][] operand)" + "\n"
//          + " target row number (=" + row
//          + ") is less than one.");
//      return null;
//    }
//    int col = target[0].length;
//    int operandRow = operand.length;
//    if (row != operandRow) {
//      DJ.print("***** ERROR ***** "
//          + "Complex.product(Complex[][] target, Complex[][] operand)" + "\n"
//          + " Element number are not same.\n"
//          + " target row number is " + row + ", \n"
//          + " operand row number is " + operandRow + ".");
//      return null;
//    }
//
//    int operandCol = operand[0].length;
//    Complex[][] _products = new Complex[col][operandCol];
//    for (int i = 0; i < col; i++) {
//      for (int j = 0; j < operandCol; j++) {
//        _products[i][j] = new Complex(0.0, 0.0);
//        for (int k = 0; k < row; k++) {
//          // Complex conj = target[k][i].copyComplex(); // 転置のみ
//          Complex conj = target[k][i].copyComplex().conjugate(); // 共役転置
//          conj.mul(operand[k][j]);
//          _products[i][j].add(conj);
//        }
//      }
//    }
//    return _products;
//  }
  

  // ----------------------------------------------------
  /**
   * 複素数の共役を得る
   * @return Complex // 共役複素数
   */
  public Complex conjugate() {
    return new Complex(real, -imag);
  }

  /** CompVecクラスに移植
   * １次元複素数配列の共役を得る
   * @param target Complex[] // 複素数ベクタ
   * @return _conjugate Complex[] // 共役な複素数ベクタ
   */
  public static Complex[] conjugate(Complex[] target) {
    int length = target.length;
    Complex[] _conjugate = new Complex[length];
    for (int i = 0; i < length; i++) {
      _conjugate[i] = target[i].conjugate();
    }
    return _conjugate;
  }

  /** CompMatクラスに移植
   * ２次元複素数行列の共役を得る
   * @param target Complex[][] // 複素数行列
   * @return _conjugate Complex[][] // 共役な複素数行列
   */
  public static Complex[][] conjugate(Complex[][] target) {
    int row = target.length;
    int col = target[0].length;
    Complex[][] _conjugate = new Complex[row][col];
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
//        _conjugate[i][k] = target[i][k].conjugate();
        _conjugate[j][i] = target[i][j].conjugate();
      }
    }
    return _conjugate;
  }

  /**
   * 複素数の自乗を得る
   * @return square double // 自乗
   */
  public double square() {
    return real * real + imag * imag;
  }
  
  
  /**
   * 複素数の絶対値を得る
   * @return double // 複素数の絶対値
   */
  public double abs() {
    return Math.sqrt(real * real + imag * imag);
  }

  /**
   * 複素数の偏角を得る
   * @return double // 複素数の偏角
   */
  public double arg() {
    return Math.atan(imag / real);
  }

  // ----------------------------------------------------
  /** CompVecクラスに移植
   * １次元複素数配列の内積を求める
   * @param target Complex[] // 被演算数複素数ベクトル
   * @param operand Complex[] // 演算数複素数ベクトル
   * @return _innerProduct Complex // 複素数
   */
  public static Complex innerProduct(Complex[] target, Complex[] operand) {
    int targetLength = target.length;
    int operandLength = operand.length;
    if (targetLength != operandLength) {
      DJ.print("***** ERROR ***** " + "Complex.innerProduct()" + "\n"
          + " Vecter length is not same."
          + " multiplicand vecter length is " + targetLength + ", "
          + " multiplier vecter length is " + operandLength + ".");
      return null;
    }

    // 被演算数複素数ベクトルは共役ベクトルに置き換える
    Complex[] _conjugate = Complex.conjugate(target);
    Complex _innerProduct = new Complex(0.0, 0.0);
    for (int i = 0; i < targetLength; i++) {
      _conjugate[i].mul(operand[i]);
      _innerProduct.add(_conjugate[i]);
    }
    return _innerProduct;
  }

  /**
   * １次元複素数配列の自己内積を求める
   * @param target Complex[] // 被演算数複素数ベクトル
   * @return _innerProduct Complex // 複素数
   */
  public static Complex innerProduct(Complex[] target) {
    int targetLength = target.length;

    // 被演算数複素数ベクトルは共役ベクトルに置き換える
    Complex[] _conjugate = Complex.conjugate(target);
    Complex _innerProduct = new Complex(0.0, 0.0);
    for (int i = 0; i < targetLength; i++) {
      _conjugate[i].mul(target[i]);
      _innerProduct.add(_conjugate[i]);
    }
    return _innerProduct;
  }

  
  /**
   * ２次元複素数配列と１次元複素数配列の内積を求める
   * @param target Complex[][] // 被演算数複素数行列
   * @param operand Complex[] // 演算数複素数ベクトル
   * @return _innerProducts Complex[] // 複素数ベクトル
   */
  public static Complex[] innerProduct(Complex[][] target, Complex[] operand) {
    int targetLength = target.length;
    if (targetLength < 1) {
      DJ.print("***** ERROR ***** "
          + "Complex.innerProduct(Matrix, vector)" + "\n"
          + " Number of tatrget length is less than one.");
      return null;
    }
    int rowLength = target[0].length;
    int operandLength = operand.length;
    if (rowLength != operandLength) {
      DJ.print("***** ERROR ***** "
          + "Complex.innerProduct(vector, vector)" + "\n"
          + " Vecter length is not same.\n"
          + " target matrix row length is " + rowLength + ", \n"
          + " operand vecter length is " + operandLength + ".");
      return null;
    }

    // 被演算数複素数行列は共役行列に置き換える
    Complex[][] _conjugate = Complex.conjugate(target);
    Complex[] _innerProducts = new Complex[targetLength];
    for (int i = 0; i < targetLength; i++) {
      Complex _innerProduct = new Complex(0.0, 0.0);
      for (int j = 0; j < operandLength; j++) {
        _conjugate[i][j].mul(operand[j]);
        _innerProduct.add(_conjugate[i][j]);
      }
      _innerProducts[i] = _innerProduct;
    }
    return _innerProducts;
  }
  
  /**
   * ２次元複素数配列と２次元複素数配列の積を求める
   * ※被演算数行列は共役転置行列に置き換えられる
   * @param target Complex[][] // 被演算数２次元複素数配列
   * @param operand Complex[] // 演算数複素数ベクトル
   * @return _products Complex[] // 複素数ベクトル
   */
  public static Complex[][] innerProduct(Complex[][] target, Complex[][] operand) {
    int row = target.length;
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "Complex.product(Complex[][] target, Complex[][] operand)" + "\n"
          + " target row number (=" + row
          + ") is less than one.");
      return null;
    }
    int col = target[0].length;
    int operandRow = operand.length;
    if (row != operandRow) {
      DJ.print("***** ERROR ***** "
          + "Complex.product(Complex[][] target, Complex[][] operand)" + "\n"
          + " Element number are not same.\n"
          + " target row number is " + row + ", \n"
          + " operand row number is " + operandRow + ".");
      return null;
    }

    int operandCol = operand[0].length;
    Complex[][] _products = new Complex[col][operandCol];
    for (int i = 0; i < col; i++) {
      for (int j = 0; j < operandCol; j++) {
        _products[i][j] = new Complex(0.0, 0.0);
        for (int k = 0; k < row; k++) {
          // Complex conj = target[k][i].copyComplex(); // 転置のみ
          Complex conj = target[k][i].copyComplex().conjugate(); // 共役転置
          conj.mul(operand[k][j]);
          _products[i][j].add(conj);
        }
      }
    }
    return _products;
  }
  
  /**
   * １次元複素数配列のノルムを求める
   * @param target Complex[] // 被演算数複素数ベクトル
   * @return _innerProduct Complex // 複素数
   */
  public static Complex norm(Complex[] target) {
    Complex _innerProduct = innerProduct(target);
    Complex norm = _innerProduct.sqrt();
    return norm;
  }
  
  // ----------------------------------------------------
    /**
     * 複素数の平方根を得る（+/- の一方の解のみ）
     * @return sqrt　Complex // 平方根
     */
    public Complex sqrt() {
      Complex sqrt = new Complex();
      if ((imag >= -Double.MIN_VALUE) && (imag <= Double.MIN_VALUE)) {
        if (real >= 0.0) {
          sqrt.setReal(Math.sqrt(real));
        }
        else {
          sqrt.setImag(Math.sqrt(-real));
        }
      }
      else {
        double aa = real * real;
        double bb = imag * imag;
        double rtab = Math.sqrt(aa + bb);
        double _real = Math.sqrt(0.5 * (real + rtab));
        double _imag = Math.sqrt(0.5 * (-real + rtab));
        sqrt.setReal(_real);
        
        if (imag >= 0.0) {
          sqrt.setImag(_imag);
        } else {
          sqrt.setImag(-_imag);
        }
      }
      
      return sqrt;
    }
  
  /**
   * オイラーの公式 exp(φi) = cosφ + sinφi, i=√(-1)
   * @param phi double // 角度φ[ラジアン]
   * @return Complex // 複素数 exp(φi)
   */
  public static Complex euler(double phi) {
    // double pi = Math.PI;
    double sinPi = Math.sin(phi);
    double cosPi = Math.cos(phi);

    // 微小な値を強制的にゼロにする
    double p = 6.123233995736766E-17;
//    double p = 6.123233995736767E-17;
    double m = -p;
    if (( sinPi <= p ) && (sinPi >= m )) sinPi = 0.0;
    if (( cosPi <= p ) && (cosPi >= m )) cosPi = 0.0;
        
    return new Complex(cosPi, sinPi);
  }
  
  
  /**
   * 行列式を求める
   * @param target Complex[][] // 複素行列（ベクトルの集合）
   * @return det Complex // 行列式の値
   */
  public static Complex determinant(Complex[][] target) {
    Complex r;
    Complex temp; // 行の交換用
    Complex det = new Complex(1.0, 0.0);
    
    int row = target.length;
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "Complex.determinant(Complex[][] target)_0" + "\n"
          + " target row length is " + row + ", "
          + " Order of tatrget is less than one.");
      return null;
    }
    int col = target[0].length;
    if (row != col) {
      DJ.print("***** ERROR ***** "
          + "Complex.determinant(Complex[][] target)_1" + "\n"
          + " target is not square matrix.\n"
          + " target row length is " + row + ", \n"
          + " target col length is " + col + ".");
      return null;
    }
    
    Complex[][] _target = new Complex[row][col];
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        _target[i][j] = target[i][j].copyComplex();
      }
    }
    
    for (int i = 0; i < col -1; i++) {
      // 対角成分がゼロならば、ゼロでない行と交換する
      if (_target[i][i].abs() == 0) {
        int k;
        for (k = i + 1; k < row; k++) {
          if (_target[k][i].abs() != 0) {
            break; // 対角成分がゼロでない
          }
        } 
        if (k < row) {
          // 行を交換する
          for (int j = 0; j < row; j++) {
            temp = _target[i][j];
            _target[i][j] = _target[k][j];
            _target[k][j] = temp;
          }
          det.mul(-1.0);
        }
      }
      for (int k = i + 1; k < col; k++) {
        r = _target[k][i].copyComplex();
        r.div(_target[i][i]);
        for (int j = i; j < col; j++) {
          temp = r.product(_target[i][j]);
          _target[k][j].sub(temp);
        }
      } 
      det.mul(_target[i][i]);
    }
    det.mul(_target[col - 1][col - 1]);
    
    return det;
  }

    
  
  
  
  // ------------------------------------------------------
  // 複素数の比較
  public boolean equals(Complex c){
    
    if (this == c) return true; // アドレスが同じ

    if (this.real != c.real) return false;
    else if (this.imag != c.imag) return false;

    return true; // 値が同じ
  }
    
  
  
  
  // ==================================================================
  /**
   * 複素数の文字列を得る
   * @return String // 複素数の文字列
   */
  @Override
  public String toString() {
//    if (imag >= 0) {
//      return real + "+" + imag + "i";
//    } else {
//      return real + "" + imag + "i";
//    }
    
    String realString = String.valueOf(real);
    String imageString = String.valueOf(imag);
    if (imageString.contains("E-")) {
      return realString + "+" + imageString + "i";
    } else if (imageString.contains("-")) {
      return realString + imageString + "i";
    } else {
      return realString + "+" + imageString + "i";
    }
  }

  /**
   * 複素数の文字列（ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @return sb String // 複素数の文字列
   */
  public String toString(String label) {
    StringBuilder sb = new StringBuilder(label + ":");
    sb.append(this.toString());
    return sb.toString();
  }

  /**
   * 複素数の文字列を得る（書式付き：小数点以下５桁）
   * @return String // 複素数の文字列
   */
  public String toFormat() {
    DecimalFormat fmt = new DecimalFormat("##.#####");
    // fmt.setRoundingMode(RoundingMode.FLOOR); // 丸め
    // fmt.setRoundingMode(RoundingMode.DOWN); // 切り捨て
    
//    if (imag >= 0) {
//      return fmt.format(real) + "+" + fmt.format(imag) + "i";
//    } else {
//      return fmt.format(real) + "" + fmt.format(imag) + "i";
//    }
    
    String realString = fmt.format(real);
    String imageString = fmt.format(imag);
    if (imageString.contains("-")) {
      return realString + imageString + "i";
    } else {
      return realString + "+" + imageString + "i";
    }
  }

  /**
   * 複素数の文字列（ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @return sb String // 複素数の文字列
   */
  public String toFormat(String label) {
    StringBuilder sb = new StringBuilder(label + ":F ");
    sb.append(this.toFormat());
    return sb.toString();
  }

  
  // ----------------------------------------------------
  // Complexクラスでの複素数配列の表示はプログラミング的配列表示に従う
  // 要素の区切りにカンマを付ける
  // 列（縦）表示では各要素を[]で括くらない
  
  /**
   * 1.複素数配列の文字列（行表示）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toString(Complex[] complexArray) {
    int length = complexArray.length;
    StringBuilder sb = new StringBuilder("[");
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toString());
      sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 2.複素数配列の文字列（行・列選択表示）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public static String toString(Complex[] complexArray, boolean orient) {
    int length = complexArray.length;
    StringBuilder sb = new StringBuilder("[");
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toString());
      if (orient) sb.append(",\n"); // 列（縦）配列
      else        sb.append(", "); // 行（横）配列
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 3.複素数配列の文字列（行表示、書式付き）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toFormat(Complex[] complexArray) {
    int length = complexArray.length;
    StringBuilder sb = new StringBuilder("F[");
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toFormat());
      sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 4.複素数配列の文字列（行・列選択表示、書式付き）を得る
   * @param complexArray Complex[] // 複素数の配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public static String toFormat(Complex[] complexArray, boolean orient) {
    int length = complexArray.length;
//    StringBuilder sb = new StringBuilder("F[");
    StringBuilder sb;
    if (orient) sb = new StringBuilder("F \n["); // 列（縦）配列
    else        sb = new StringBuilder("F ["); // 行（横）配列
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toFormat());
      if (orient) sb.append(",\n"); // 列（縦）配列
      else        sb.append(", "); // 行（横）配列
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  
  /**
   * 5.複素数配列の文字列（行表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toString(String label, Complex[] complexArray) {
    int length = complexArray.length;
    StringBuilder sb = new StringBuilder(label + ":[");
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toString());
      sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 6.複素数配列の文字列（行・列選択表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public static String toString(String label, Complex[] complexArray, boolean orient) {
    int length = complexArray.length;
//    StringBuilder sb = new StringBuilder(label + ":\n[");
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":\n["); // 列（縦）配列
    else        sb = new StringBuilder(label + ":["); // 行（横）配列
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toString());
      if (orient) sb.append(",\n");
      else        sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 7.複素数配列の文字列（行表示、ラベル付き、書式付き）を得る
   *
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @return String // 複素数の文字列
   */
  public static String toFormat(String label, Complex[] complexArray) {
    int length = complexArray.length;
    StringBuilder sb = new StringBuilder(label + ":F [");
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toFormat());
      sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }

  /**
   * 8.複素数配列の文字列（行・列選択表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[] // 複素数の配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @return String // 複素数の文字列
   */
  public static String toFormat(String label, Complex[] complexArray, boolean orient) {
    int length = complexArray.length;
    StringBuilder sb;
    if (orient) sb = new StringBuilder(label + ":F\n["); // 列（縦）配列
    else        sb = new StringBuilder(label + ":F ["); // 行（横）配列
    for (int i = 0; i < length; i++) {
      sb.append(complexArray[i].toFormat());
      if (orient) sb.append(",\n");
      else        sb.append(", ");
    }
    int index = sb.length();
    sb.replace(index - 2, index, "]");
    return sb.toString();
  }
  

  // ------------------------------------------------------
  // Complexクラスでの複素数行列の表示はプログラミング的行列表示に従う
  // 要素の区切りにカンマを付ける
  // 行（横）表示では行全体を[]で括り、各要素を[]で括くらない
  
  /**
   * 1.複素数行列の文字列（行列表示）を得る
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toString(Complex[][] complexArray) {
    int colLength = complexArray.length; // 行数（列の長さ）
    int order = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder("[");
    int index;
    for (int i = 0; i < colLength; i++) {
      sb.append("[");
      for (int j = 0; j < order; j++) {
        sb.append(complexArray[i][j].toString());
        sb.append(", ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

  /**
   * 2.複素数行列の文字列（行列表示、書式付き）を得る
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toFormat(Complex[][] complexArray) {
    int colLength = complexArray.length; // 行数（列の長さ）
    int order = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder("F\n[");
    int index;
    for (int i = 0; i < colLength; i++) {
      sb.append("[");
      for (int j = 0; j < order; j++) {
        sb.append(complexArray[i][j].toFormat());
        sb.append(", ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }
  
  /**
   * 3.複素数行列の文字列（行列表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toString(String label, Complex[][] complexArray) {
    int colLength = complexArray.length; // 行数（列の長さ）
    int order = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder(label + ":\n[");
    int index;
    for (int i = 0; i < colLength; i++) {
      sb.append("[");
      for (int j = 0; j < order; j++) {
        sb.append(complexArray[i][j].toString());
        sb.append(", ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

  /**
   * 4.複素数行列の文字列（行列表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toFormat(String label, Complex[][] complexArray) {
    int colLength = complexArray.length; // 行数（列の長さ）
    int order = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder(label + ":F\n[");
    int index;
    for (int i = 0; i < colLength; i++) {
      sb.append("[");
      for (int j = 0; j < order; j++) {
        sb.append(complexArray[i][j].toFormat());
        sb.append(", ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

} // Complex Class

// End of file
