/*
 *  Title: DaiJa_V5 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2022
 */
package view;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;

/**
 * <p> 表　題: Class: PatternSub</p>
 * <p> 説　明: パターン・ビューワーの補助クラス</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.10.16</p>
 * <p> 修正日: 2022.09.17</p>
 */
public class PatternSub {

  // カラー・パレットを生成
  private static final int MAX_COLOR = 11;
  public static final int RED = 0;
  public static final int YELLOW = 1;
  public static final int GREEN = 2;
  public static final int BLUE = 3;
  public static final int PURPLE = 4;
  public static final int PINK = 5;
  public static final int ORANGE = 6;
  public static final int DARK_GREEN = 7;
  public static final int LIGHT_GREEN = 8;
  public static final int DARK_BLUE = 9;
  public static final int BLACK = 10;
  private static Color[] colorPalet; // カラーパレット

  static {
    setColorPalet();
  }

  // フレーム（座標表示）用指定色
  public static final Color CARBON = Color.getHSBColor(0.0F, 1.0F, 0.0F); // 黒色
  public static final Color UMBER = Color.getHSBColor(0.113F, 1.0F, 1.0F); // アンバー色
  public static final Color BEIGE = Color.getHSBColor(0.13F, 0.3F, 0.9F); // 淡褐色
  public static final Color BROWN = Color.getHSBColor(0.13F, 0.5F, 0.4F); // 褐色

  // グラフ描画用変数
  private Box screen; // 画面サイズ(int h, w)

  public String axisNameX = "x_axis"; // Ｘ軸名称
  public String axisNameY = "y_axis"; // Ｙ軸名称

  public int framSpace = 5; //10; // フレームの内側の余白
  public int frameDouble = framSpace * 2;
  Pict axesSpace = new Pict(40, 10, 10, 25); // 表示枠の外側の余白

  public int axisSpace = 5; // 表示枠の補助余白
  public int axisTwice = axisSpace * 2; // 表示枠の補助余白２倍
  public int axisNumX = 6; //10; //4; // Ｘ軸補助線の本数
  public int axisNumY = 6; //10; //4; // Ｙ軸補助線の本数
  public float step; // グラフ座標軸の表示間隔

  Tetra axes; // グラフ枠の設定(float l, r, b, t;)
  Rect aide; // 補助表示枠(int x, y, w, h;)
  Rect board; // 表示枠(int x, y, w, h;)

  // グラフ描画用パラメータ
  private Quad param; // 座標変換用パラメータ(xMag,xShift,yMag,yShift)

  // 生成と初期化
  PatternSub() {
    // グラフ枠の設定
    axes = Tetra.tetra(); // l, r, b, t;
    axes.setTetra(-1.5F, 1.5F, -1.2F, 1.2F);
    step = 0.5F; //0.2F;
  }

  // グラフ描画用パラメータ（座標変換用パラメータ）の更新
  public void updateParameter() {
    float xMag = (float) board.w / (float) (axes.r - axes.l);
    float xShift = -(float) axes.l * xMag + (float) board.x;
    float yMag = -(float) board.h / (axes.t - axes.b);
    float yShift = -axes.b * yMag + (float) (board.y + board.h);
    param = Quad.quad(xMag, xShift, yMag, yShift);
  }

  // 座標変換
  Dot getDot(Dim vol) {
    int e = (int) (vol.x * param.m + param.n);
    int f = (int) (vol.y * param.s + param.t);
    return Dot.dot(e, f);
  }

  // カラーパレットの設定
  private static void setColorPalet() {
    colorPalet = new Color[MAX_COLOR]; // カラーパレット
    float aHue = 0.0F;
    float aMag = 0.9F; //1.0F;
    float aDim = 0.9F; //1.0F;
    for (int i = 0; i < MAX_COLOR; i++) {
      aHue = 2.0F * (float) i / (float) MAX_COLOR;
      if (aHue > 1.0F) {
        aHue = aHue - 1.0F;
        aMag = 0.6F;
        aDim = 0.8F;
      }
      if (i == 5) {
        aMag = 0.6F;
      }
      if (i == 13) {
        aMag = 0.3F;
      }
      colorPalet[i] = Color.getHSBColor(aHue, aMag, aDim);
    }
    colorPalet[MAX_COLOR - 1] = Color.getHSBColor(0.0F, 0.0F, 0.0F);
  }

  // カラーパレットから色を得る
  public Color getPaletColor(int id) {
    return colorPalet[id];
  }

  // スクリーンを用意する
  public void prepareScreen(Graphics g, Dimension size) {
    screen = Box.box(size.height, size.width);

    // 背景塗りつぶし
    g.setColor(Color.getHSBColor(0.14F, 0.1F, 0.95F)); // 薄いベージュ
    g.fillRect(0, 0, screen.w, screen.h);
  }

  // 表示枠の外側の余白の設定
  public void setAxesSpace(int l, int r, int t, int b) {
    axesSpace.l = l;
    axesSpace.r = r;
    axesSpace.t = t;
    axesSpace.b = b;
  }

  // フレームを表示する
  public void drawFrame(Graphics g) { // 画面サイズ(int h, w)
    // 補助表示枠(int x, y, w, h;)（非表示）
    aide = Rect.rect(framSpace, framSpace,
            screen.w - frameDouble, screen.h - frameDouble);

    // 表示枠(int x, y, w, h;)
    board = Rect.rect(aide.x + axesSpace.l, aide.y + axesSpace.t,
            aide.w - axesSpace.l - axesSpace.r, aide.h - axesSpace.t - axesSpace.b);

    // グラフ描画用パラメータの更新
    updateParameter();

    Dot dotS;
    Dot dotE;
    float sX, sY, eX, eY;
    float axis;

    // 縦軸座標値と水平補助線
    // 縦軸の水平補助線　原点(y=0)
    sX = axes.l;
    sY = 0.0F;
    eX = axes.r;
    eY = sY;
    dotS = getDot(Dim.dim(sX, sY));  // 座標変換
    dotE = getDot(Dim.dim(eX, eY));
    g.setColor(BEIGE); // 淡褐色
    g.drawLine(dotS.e - axisSpace, dotS.f, dotE.e, dotE.f); // 原点(y=0)
    // 縦軸の原点座標値　原点(y=0)
    g.setColor(BROWN); // 褐色
    g.drawString(String.format("%.1f", axis = 0.0F),
            dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);

    // 縦軸の上端座標値
    g.setColor(BROWN); // 褐色
    dotS = getDot(Dim.dim(axes.l, axes.t));  // 座標変換
    g.drawString(String.format("%.1f", axis = axes.t),
            dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);
    dotS = getDot(Dim.dim(axes.l, axes.b));  // 座標変換
    g.drawString(String.format("%.1f", axis = axes.b),
            dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);

    for (int j = 1; j < axisNumY / 2; j++) { // 横軸
      // 縦軸の水平補助線　正側(y>0)
      sX = axes.l;
      sY = step * j;
      eX = axes.r;
      eY = sY;
      dotS = getDot(Dim.dim(sX, sY));  // 座標変換
      dotE = getDot(Dim.dim(eX, eY));
      g.setColor(BEIGE); // 淡褐色
      g.drawLine(dotS.e - axisSpace, dotS.f, dotE.e, dotE.f);
      // 縦軸軸の座標値　正側(y>0)
      g.setColor(BROWN); // 褐色
      g.drawString(String.format("%.1f", axis = step * j),
              dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);

      // 縦軸の水平補助線　負側(y<0)
      dotS = getDot(Dim.dim(sX, -sY));  // 座標変換
      dotE = getDot(Dim.dim(eX, -eY));
      g.setColor(BEIGE); // 淡褐色
      g.drawLine(dotS.e - axisSpace, dotS.f, dotE.e, dotE.f);
      // 縦軸軸の座標値　負側(y<0)
      g.setColor(BROWN); // 褐色
      g.drawString(String.format("%.1f", axis = -step * j),
              dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);
    }

    // 横軸座標値と垂直補助線
    for (int j = 1; j < axisNumX; j++) { // 縦軸
      // 横軸の垂直補助線
      sX = axes.l + (axes.r - axes.l) * j / axisNumX;
      sY = axes.t;
      eX = sX;
      eY = axes.b;
      dotS = getDot(Dim.dim(sX, sY));  // 座標変換
      dotE = getDot(Dim.dim(eX, eY));
      g.setColor(BEIGE); // 淡褐色
      g.drawLine(dotS.e, dotS.f, dotE.e, dotE.f + axisSpace);
      // 横軸の座標値
      axis = axes.l * (axisNumX - j) * 2.0F / axisNumX + axes.r;
      g.setColor(BROWN); // 褐色
      g.drawString(String.format("%.1f", axis),
              dotS.e, dotE.f + axesSpace.b - axisTwice);
    }
    // 左右端の座標値
    g.setColor(BROWN); // 褐色
    dotE = getDot(Dim.dim(axes.l, axes.b)); // 左端の座標値
    g.drawString(String.format("%.1f", axes.l),
            dotE.e, dotE.f + axesSpace.b - axisTwice);
    dotE = getDot(Dim.dim(axes.r, axes.b)); // 右端の座標値
    g.drawString(String.format("%.1f", axes.r),
            dotE.e - axisTwice, dotE.f + axesSpace.b - axisTwice);

    // 表示枠を描画
    g.setColor(UMBER); // アンバー色
    g.drawRect(board.x, board.y, board.w, board.h); // 表示枠の描画

    // 座標軸名称を描画
    g.setColor(CARBON); // 黒色
    g.drawString(axisNameX, board.x + board.w / 2 - 33, // Ｘ軸名称
            board.y + board.h + axesSpace.b);
    g.drawString(axisNameY, board.x - axesSpace.l, // Ｙ軸名称
            board.y + board.h / 2 + axisSpace * 4);
  }

  // 制御用のフレーム（座標表示）を表示する
  public void drawServoFrame(Graphics g) { // 画面サイズ(int h, w)
    // 補助表示枠(int x, y, w, h;)（非表示）
    aide = Rect.rect(framSpace, framSpace,
            screen.w - frameDouble, screen.h - frameDouble);

    // 表示枠(int x, y, w, h;)
    board = Rect.rect(aide.x + axesSpace.l, aide.y + axesSpace.t,
            aide.w - axesSpace.l - axesSpace.r, aide.h - axesSpace.t - axesSpace.b);

    axes.setTetra(-0.2F, 1.2F, -0.2F, 1.5F); // l, r, b, t;
    step = 0.5F; //0.2F;
    updateParameter(); // グラフ描画用パラメータ（座標変換用パラメータ）の更新

    Dot dotS; // 描画開始位置
    Dot dotE; // 描画終了位置
    float sX, sY, eX, eY;
    float axis;

    // 縦軸座標値と水平補助線
    // 縦軸の水平補助線　原点(y=0)
    sX = axes.l;
    sY = 0.0F;
    eX = axes.r;
    eY = sY;
    dotS = getDot(Dim.dim(sX, sY));  // 座標変換
    dotE = getDot(Dim.dim(eX, eY));
    g.setColor(BEIGE); // 淡褐色
    g.drawLine(dotS.e - axisSpace, dotS.f, dotE.e, dotE.f); // 原点(y=0)
    // 縦軸の原点座標値　原点(y=0)
    g.setColor(BROWN); // 褐色
    g.drawString(String.format("%.1f", axis = 0.0F),
            dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);

    // 縦軸の上端座標値
    g.setColor(BROWN); // 褐色
    dotS = getDot(Dim.dim(axes.l, axes.t));  // 座標変換
    g.drawString(String.format("%.1f", axis = axes.t),
            dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);
    dotS = getDot(Dim.dim(axes.l, axes.b));  // 座標変換
    g.drawString(String.format("%.1f", axis = axes.b),
            dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);

    for (int j = 1; j < axisNumY / 2; j++) { // 横軸
      // 縦軸の水平補助線　正側(y>0)
      sX = axes.l;
      sY = step * j;
      eX = axes.r;
      eY = sY;
      dotS = getDot(Dim.dim(sX, sY));  // 座標変換
      dotE = getDot(Dim.dim(eX, eY));
      g.setColor(BEIGE); // 淡褐色
      g.drawLine(dotS.e - axisSpace, dotS.f, dotE.e, dotE.f);
      // 縦軸軸の座標値　正側(y>0)
      g.setColor(BROWN); // 褐色
      g.drawString(String.format("%.1f", axis = step * j),
              dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);

//      // 縦軸の水平補助線　負側(y<0)
//      dotS = getDot(Dim.dim(sX, -sY));  // 座標変換
//      dotE = getDot(Dim.dim(eX, -eY));
//      g.setColor(BEIGE); // 淡褐色
//      g.drawLine(dotS.e - axisSpace, dotS.f, dotE.e, dotE.f);
//      // 縦軸軸の座標値　負側(y<0)
//      g.setColor(BROWN); // 褐色
//      g.drawString(String.format("%.1f", axis = -step * j),
//              dotS.e - axesSpace.l + axisTwice, dotS.f + axisSpace);
    }

    // 横軸座標値と垂直補助線
    for (int j = 0; j < axisNumX / 2; j++) { // 縦軸
      // 横軸の垂直補助線
      sX = step * j;
      sY = axes.t;
      eX = sX;
      eY = axes.b;
      dotS = getDot(Dim.dim(sX, sY));  // 座標変換
      dotE = getDot(Dim.dim(eX, eY));
      g.setColor(BEIGE); // 淡褐色
      g.drawLine(dotS.e, dotS.f, dotE.e, dotE.f + axisSpace);
      // 横軸の座標値
      axis = step * j;
      g.setColor(BROWN); // 褐色
      g.drawString(String.format("%.1f", axis),
              dotS.e, dotE.f + axesSpace.b - axisTwice);
    }
    // 左右端の座標値
    g.setColor(BROWN); // 褐色
    dotE = getDot(Dim.dim(axes.l, axes.b)); // 左端の座標値
    g.drawString(String.format("%.1f", axes.l),
            dotE.e, dotE.f + axesSpace.b - axisTwice);
    dotE = getDot(Dim.dim(axes.r, axes.b)); // 右端の座標値
    g.drawString(String.format("%.1f", axes.r),
            dotE.e - axisTwice, dotE.f + axesSpace.b - axisTwice);

    // 表示枠を描画
    g.setColor(UMBER); // アンバー色
    g.drawRect(board.x, board.y, board.w, board.h); // 表示枠の描画

    // 座標軸名称を描画
    g.setColor(CARBON); // 黒色
    g.drawString(axisNameX, board.x + board.w / 2 - 33, // Ｘ軸名称
            board.y + board.h + axesSpace.b);
    g.drawString(axisNameY, board.x - axesSpace.l, // Ｙ軸名称
            //            board.y + board.h / 2 + axisSpace * 4);
            board.y + board.h / 2 + axisSpace * 2);
  }

  // 制御パターンを描画
  public void drawServo(Graphics g, float[][][] matrix) {
    int col = matrix[0][0].length;
    float criteria = 0.99F; //0.6F; //0.5F; // 判定値（出力が整数でない場合用）

    for (int m = 0; m < matrix.length; m++) {
      // グラフの描画
      for (int i = 0; i < col; i++) {
        float x = matrix[m][0][i];
        float y = matrix[m][1][i];
        float t = matrix[m][2][i];
        Dim yVol = Dim.dim(x, y);
        Dim tVol = Dim.dim(x, t);
        // 座標変換
        Dot yDot = getDot(yVol);
        Dot tDot = getDot(tVol);

        // 正解値
        drawSmallDot(g, tDot, 10 - m % 5); // 小十字

        // 出力値
        drawSmallCross(g, yDot, m % 5); // 小Ｘ字
      } // i
    } // m
  } // drawServo

  // 調節パターンを描画
  public void drawTuner(Graphics g, float[][][] matrix) {
    int col = matrix[0][0].length;
    int row =  matrix[0].length;
    float t = 0.0F;
    
    for (int m = 0; m < matrix.length; m++) {
      // グラフの描画
      for (int i = 0; i < col; i++) {
        float x = matrix[m][0][i]; // 入力
        float y = matrix[m][1][i]; // 出力（力操作量）
        float r = matrix[m][2][i]; // 応答（速度）
        if (row > 3) {
          t = matrix[m][3][i]; // 正解（速度所望値）
          Dim yVol = Dim.dim(x, y);
          Dim rVol = Dim.dim(x, r);
          Dim tVol = Dim.dim(x, t);
          // 座標変換
          Dot yDot = getDot(yVol);
          Dot rDot = getDot(rVol);
          Dot tDot = getDot(tVol);

          // 正解（速度所望値）
          if (row > 3) {
            drawSmallDot(g, tDot, BLACK); // 小十字
          }
          // 出力（力操作量）
          drawSmallDot(g, yDot, DARK_BLUE); // 小十字

          // 正解（速度所望値）
          float e = Math.abs(r - t); // 応答（速度）- 正解（速度所望値） 
/**/          if (e < 0.001) {
//**/          if (e < 0.0058) {
//**/          if (e < 0.00005) {
            drawSmallDiamond(g, rDot, GREEN); // 小菱形
          } else {
            drawSmallCross(g, rDot, RED); // 小Ｘ字
          }

          if (row > 4) {
            float z = 0.0F;
            Dim zvol = Dim.dim(x, z);
            Dot zDot = getDot(zvol);
            for (int k = 4; k < row; k++) {
              z = matrix[m][k][i]; // 正解（速度所望値）
              zvol = Dim.dim(x, z);
              zDot = getDot(zvol);
              if (k < 7) drawDot(g, zDot, BLUE); // 小十字
              else        drawDot(g, zDot, PINK); // 小十字
            }
          }
        }
      } // i

    } // m

  } // drawTuner

  // 散布パターンを描画
  public void drawSpray(Graphics g, float[][][] matrix) {
    int col = matrix[0][0].length;
    int row =  matrix[0].length; // Ver.5 2022.9.17

    // グラフの描画
    for (int i = 0; i < col; i++) {
      float x = matrix[0][0][i];
      float y = matrix[0][1][i];
      float t = matrix[0][2][i];
      Dim yVol = Dim.dim(x, y);
      Dim tVol = Dim.dim(x, t);
      // 座標変換
      Dot yDot = getDot(yVol);
      Dot tDot = getDot(tVol);

      // 正解値
//      drawSmallDot(g, tDot, DARK_BLUE); // 小十字
      drawDot(g, tDot, DARK_BLUE); // 点

      // 出力値
      float e = Math.abs(y - t);
      if (e < 0.05) { // 0.05 // 0.002
        drawDiamond(g, yDot, GREEN); // 菱形
//        drawDot(g, yDot, GREEN); // 点
      } else {
        drawCross(g, yDot, RED); // Ｘ字
//        drawDot(g, yDot, RED); // 点
      }
      
      // Ver.5 2022.9.17
      for (int m = 0; m < matrix.length; m++) {
        if (row > 4) {
          float z = 0.0F;
          Dim zvol = Dim.dim(x, z);
          Dot zDot = getDot(zvol);
          for (int k = 4; k < row; k++) {
            z = matrix[m][k][i]; // 正解（速度所望値）
            zvol = Dim.dim(x, z);
            zDot = getDot(zvol);
            if (k < 7) {
              drawDot(g, zDot, BLUE); // 点
            } else {
              drawDot(g, zDot, PINK); // 点
            }
          }
        }
      }
      
      
    } // i

  } // drawSpray

  // 論理パターンを描画
  public void drawLogic(Graphics g, float[][][] matrix) {
    int col = matrix[0][0].length;
    float criteria = 0.99F; //0.6F; //0.5F; // 判定値（出力が整数でない場合用）

    // グラフの描画
    for (int i = 0; i < col; i++) {
      float x0 = matrix[0][0][i];
      float x1 = matrix[0][1][i];
      float y = matrix[0][2][i];
      float t = matrix[0][3][i];

      // 入力の組み合わせに対応する位置にマークを表示する
      Dim xPos; // マーク表示位置
      xPos = Dim.dim(x0, x1);

      // 座標変換
      Dot yDot = getDot(xPos);

      // 推定値の判定結果により表示するマークの色や形状を変える
//      if (y == t) { // 出力が整数、もしくは厳密に0.0と1.0になる場合
      if ((t > criteria) && (y > criteria) || (t < criteria) && (y < criteria)) {
        // 正しく推定されている
        drawDiamond(g, yDot, GREEN); // 菱形
      } else {
        drawBigCross(g, yDot, RED); // Ｘ字
      }
    } // i

  } // drawLogic

  // 分類パターンを描画
  public void drawGroup(Graphics g, float[][][] matrix) {
    int lng = matrix.length;
    int row = matrix[0].length;
    int col = matrix[0][0].length;

    // グラフの描画
    for (int i = 0; i < col; i++) {
      float x0 = matrix[0][0][i];
      float y0 = matrix[0][1][i];
      float y = matrix[0][2][i];
      float t = matrix[0][3][i];
      Dim tVol = Dim.dim(y0, t);
      // 座標変換
      Dot y0Dot = getDot(Dim.dim(x0 / 2, y0 / 2));
//      Dot y1Dot = getDot(Dim.dim(x1/2, y1/2));
      Dot tDot = getDot(tVol);

//      if (y > 0.9 ){ // yが0.5以上・・・正しく推定されている
      if (y > 0.7) { // yが0.5以上・・・正しく推定されている
//      if (y > 0.5 ){ // yが0.5以上・・・正しく推定されている
        switch ((int) t) {
          case 0:
            drawBigCross(g, y0Dot, GREEN); // Ｘ字
//            drawCross(g, y1Dot, BLUE); // Ｘ字
            break;
          case 1:
            drawDiamond(g, y0Dot, GREEN); // 菱形
            break;
          default:
            drawGreek(g, y0Dot, GREEN); // 十字
            break;
        }
      } else {
        switch ((int) t) {
          case 0:
            drawBigCross(g, y0Dot, RED); // Ｘ字
            break;
          case 1:
            drawDiamond(g, y0Dot, RED); // 菱形
            break;
          default:
            drawGreek(g, y0Dot, RED); // 十字
            break;
        }
      }
    }

  } // drawLogic

  // 判定パターンを描画
  public void drawJudge(Graphics g, float[][][] matrix) {
    int lng = matrix.length;
    int row = matrix[0].length;
    int col = matrix[0][0].length;

    // グラフの描画
    for (int i = 0; i < lng; i++) {
      for (int j = 0; j < row; j++) {
        float y = matrix[i][j][0]; // 出力
        float t = matrix[i][j][1]; // 正解
        //　スケーリング
        float yVal = y * 2.0F - 1.0F;
        float xVal = (float) j / 5.0F - 0.9F;
        // 座標変換
        Dot yDot = getDot(Dim.dim(xVal, yVal));

        if (t > 0.5F) { // 正解
          if (y > 0.5F) {
            drawDiamond(g, yDot, GREEN); // 菱形
          } else {
            drawDiamond(g, yDot, RED); // 菱形
          }
        } else {
          if (y < 0.5F) {
            drawBigCross(g, yDot, GREEN); // Ｘ字
          } else {
            drawBigCross(g, yDot, RED); // Ｘ字
          }
        }

      } // j, row
    } // i, lng

  } // drawJudge

  // 点
  private void drawDot(Graphics g, Dot dot, int color) {
    int mg = 1; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x, y, x, y);
//    g.drawLine(x, y, x, y);
  }

  // 小十字
  private void drawSmallDot(Graphics g, Dot dot, int color) {
    int mg = 1; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y, x + mg, y);
    g.drawLine(x, y + mg, x, y - mg);
  }

  // 小Ｘ字
  private void drawSmallCross(Graphics g, Dot dot, int color) {
    int mg = 1; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y - mg, x + mg, y + mg);
    g.drawLine(x - mg, y + mg, x + mg, y - mg);
  }

  // Ｘ字
  private void drawCross(Graphics g, Dot dot, int color) {
    int mg = 2; //1; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y - mg, x + mg, y + mg);
    g.drawLine(x - mg, y + mg, x + mg, y - mg);
  }

  // 大Ｘ字
  private void drawBigCross(Graphics g, Dot dot, int color) {
    int mg = 4; // 2; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y - mg, x + mg, y + mg);
    g.drawLine(x - mg, y + mg, x + mg, y - mg);
  }

  // 十字
  private void drawGreek(Graphics g, Dot dot, int color) {
    int mg = 4; //2; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y, x + mg, y);
    g.drawLine(x, y + mg, x, y - mg);
  }

  // 小菱形
  private void drawSmallDiamond(Graphics g, Dot dot, int color) {
    int mg = 2; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y, x, y + mg);
    g.drawLine(x - mg, y, x, y - mg);
    g.drawLine(x, y + mg, x + mg, y);
    g.drawLine(x, y - mg, x + mg, y);
  }

  // 菱形
  private void drawDiamond(Graphics g, Dot dot, int color) {
    int mg = 4; //2; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y, x, y + mg);
    g.drawLine(x - mg, y, x, y - mg);
    g.drawLine(x, y + mg, x + mg, y);
    g.drawLine(x, y - mg, x + mg, y);
  }

  // 四角
  private void drawSquare(Graphics g, Dot dot, int color) {
    int mg = 3; //2; // マークの大きさ
    g.setColor(colorPalet[color]);
    int x = dot.e;
    int y = dot.f;
    g.drawLine(x - mg, y + mg, x + mg, y + mg);
    g.drawLine(x - mg, y - mg, x + mg, y - mg);
    g.drawLine(x - mg, y + mg, x - mg, y - mg);
    g.drawLine(x + mg, y + mg, x + mg, y - mg);
  }

} // PatternSub

// EOF
