/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package view;

import java.awt.BorderLayout;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import util.DJ;
import util.TimeStamp;

/**
 * <p> 表　題: Class: PatternViewer</p>
 * <p> 説　明: ２次元パターンを表示・保存・読出しする</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.8.27</p>
 */
//public class PatternViewer extends javax.swing.JFrame {
public class PatternViewer extends Viewer {
  
  public static final String BOUNDS_NAME = "patternBounds";
  
  // 描画パターン・タイプ
  public static final int PATTERN_PLAIN = 0; // 無地
  public static final int PATTERN_CHECK = 1; // 千鳥格子
  public static final int PATTERN_SPRAY = 2; // 散布
  public static final int PATTERN_LOGIC = 3; // 論理
  public static final int PATTERN_GROUP = 4; // 分類
  public static final int PATTERN_JUDGE = 5; // 判定
  public static final int PATTERN_TUNER = 6; // 調節
  public static final int PATTERN_SERVO = 7; // 制御
  
  PatternDrawer drawer0; // パターン描画用クラス：Canvasのサブクラス
  PatternDrawer drawer1; // パターン描画用クラス：Canvasのサブクラス
  
  /**
   * ２次元パターンを表示する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternData0 double[][] // パターン描画用データの配列
   * @param patternComment0 String // パターン描画用コメント
   */
  public PatternViewer(
          int patternType0, double[][] patternData0, String patternComment0) {
    initComponents();
    initialize(); // 初期化
    
     // パターン描画用クラス０
    drawer0 = new PatternDrawer(patternType0, patternData0);
    drawer0.setPage_Comment(0, patternComment0); // 頁番号、コメント
    verifyCard.add(drawer0, BorderLayout.CENTER);
    // パターン描画用クラス１
    drawer1 = new PatternDrawer(PATTERN_PLAIN, patternData0);
    drawer1.setPage_Comment(1, patternComment0); // 頁番号、コメント
    imageCard.add(drawer1, BorderLayout.CENTER);
  }
  
  /**
   * ２次元パターンを表示する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternData0 double[][] // パターン描画用の配列
   * @param patternComment0 String // コメント
   * @param patternType1 int // 描画パターン・タイプ
   * @param patternData1 double[][] // パターン描画用の配列
   * @param patternComment1 String // コメント
   */
  public PatternViewer(
          int patternType0, double[][] patternData0, String patternComment0,
          int patternType1, double[][] patternData1, String patternComment1) {
    initComponents();
    initialize(); // 初期化

    // ドローワをイン・パネルに載せる
    drawer0 = new PatternDrawer(patternType0, patternData0); // パターン描画用クラス
    drawer1 = new PatternDrawer(patternType1, patternData1); // パターン描画用クラス
    drawer0.setPage_Comment(0, patternComment0); // 頁番号、コメント
    drawer1.setPage_Comment(1, patternComment1); // 頁番号、コメント
    verifyCard.add(drawer0, BorderLayout.CENTER);
    imageCard.add(drawer1, BorderLayout.CENTER); // ダミーのドローワ
  }
  
  /**
   * ３次元パターンを表示する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternData0 double[][][] // パターン描画用の配列
   * @param patternComment0 String // コメント
   * @param patternType1 int // 描画パターン・タイプ
   * @param patternData1 double[][] // パターン描画用のリスト
   * @param patternComment1 String // コメント
   */
  public PatternViewer(
          int patternType0, double[][][] patternData0, String patternComment0,
          int patternType1, double[][] patternData1, String patternComment1) {
    initComponents();
    initialize(); // 初期化

    // ドローワをイン・パネルに載せる
    drawer0 = new PatternDrawer(patternType0, patternData0); // パターン描画用クラス
    drawer1 = new PatternDrawer(patternType1, patternData1); // パターン描画用クラス
    drawer0.setPage_Comment(0, patternComment0); // 頁番号、コメント
    drawer1.setPage_Comment(1, patternComment1); // 頁番号、コメント
    verifyCard.add(drawer0, BorderLayout.CENTER);
    imageCard.add(drawer1, BorderLayout.CENTER); // ダミーのドローワ
  }
  
  /**
   * ３次元パターンを表示する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternData0 double[][][] // パターン描画用の配列
   * @param patternComment0 String // コメント
   * @param patternType1 int // 描画パターン・タイプ
   * @param patternData1 double[][][] // パターン描画用のリスト
   * @param patternComment1 String // コメント
   */
  public PatternViewer(
          int patternType0, double[][][] patternData0, String patternComment0,
          int patternType1, double[][][] patternData1, String patternComment1) {
    initComponents();
    initialize(); // 初期化

    // ドローワをイン・パネルに載せる
    drawer0 = new PatternDrawer(patternType0, patternData0); // パターン描画用クラス
    drawer1 = new PatternDrawer(patternType1, patternData1); // パターン描画用クラス
    drawer0.setPage_Comment(0, patternComment0); // 頁番号、コメント
    drawer1.setPage_Comment(1, patternComment1); // 頁番号、コメント
    verifyCard.add(drawer0, BorderLayout.CENTER);
    imageCard.add(drawer1, BorderLayout.CENTER); // ダミーのドローワ
  }
  
  // 表示用フレームの領域を設定
  public final void setBounds() {
    PropertyViewer propertyViewer = DJ.getPropertyViewer();
    Rectangle rect = propertyViewer.getBoundsValue("patternBounds");
    if (rect != null) {
      setBounds(rect);
    }
    else { // ディフォルト値を設定
      setBounds(610, 410, 400, 400);
    }
  }
  
  // 頁とタイトルを設定
  public void setTitle(String title0, String title1) {
    if (drawer0 != null) drawer0.setPage_Comment(0, title0);
    if (drawer1 != null) drawer1.setPage_Comment(1, title1);
  }
  
  /**
   * パターンを描画する配列を受け取り、ドロワーにそのまま渡す
   * @param data double[][] // パターン描画用データ
   */
  public void updatePattern(double[][] data) {
    drawer0.setMatrix(data); // ドロワーに渡す
  } // updatePattern()
  
  /**
   * パターンを描画するテンソルを受け取り、ドロワーにそのまま渡す渡す
   * @param data0 double[][] // パターン描画用データの配列
   * @param data1 double[][] // パターン描画用データの配列
   */
  public void updatePattern(double[][] data0, double[][] data1) {
    drawer0.setMatrix(data0); // ドロワーに渡す
    drawer1.setMatrix(data1); // ドロワーに渡す
  } // updatePattern()
  
  /**
   * パターンを描画するテンソルを受け取り、ドロワーにそのまま渡す渡す
   * @param data0 double[][][] // パターン描画用データの配列
   * @param data1 double[][] // パターン描画用データのリスト
   */
  public void updatePattern(double[][][] data0, double[][] data1) {
    drawer0.setMatrix(data0); // ドロワーに渡す
    drawer1.setMatrix(data1); // ドロワーに渡す
  } // updatePattern()
  
  /**
   * パターンを描画するテンソルを受け取り、ドロワーにそのまま渡す渡す
   * @param data0 double[][][] // パターン描画用データの配列
   * @param data1 double[][][] // パターン描画用データのリスト
   */
  public void updatePattern(double[][][] data0, double[][][] data1) {
    drawer0.setMatrix(data0); // ドロワーに渡す
    drawer1.setMatrix(data1); // ドロワーに渡す
  } // updatePattern()
  
  /**
   * This method is called from within the constructor to initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is always
   * regenerated by the Form Editor.
   */
  @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    mainPanel = new javax.swing.JPanel();
    verifyCard = new javax.swing.JPanel();
    imageCard = new javax.swing.JPanel();
    controlPanel = new javax.swing.JPanel();
    valueButton = new javax.swing.JButton();
    cardButton = new javax.swing.JButton();

    setTitle("DaiJa : Pattern viewer");
    setFocusable(false);

    mainPanel.setAutoscrolls(true);
    mainPanel.setLayout(null);
    cardLayout = new java.awt.CardLayout();
    mainPanel.setLayout(cardLayout);

    verifyCard.setLayout(new java.awt.BorderLayout());
    mainPanel.add(verifyCard);
    verifyCard.setBounds(0, 0, 400, 350);

    imageCard.setLayout(new java.awt.BorderLayout());
    mainPanel.add(imageCard);
    imageCard.setBounds(0, 0, 400, 350);

    getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);

    controlPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
    controlPanel.setPreferredSize(new java.awt.Dimension(400, 24));
    controlPanel.setRequestFocusEnabled(false);

    valueButton.setText("Value");
    valueButton.setPreferredSize(new java.awt.Dimension(72, 18));

    cardButton.setText("Next Page");
    cardButton.setPreferredSize(new java.awt.Dimension(84, 18));
    cardButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        cardButtonActionPerformed(evt);
      }
    });

    javax.swing.GroupLayout controlPanelLayout = new javax.swing.GroupLayout(controlPanel);
    controlPanel.setLayout(controlPanelLayout);
    controlPanelLayout.setHorizontalGroup(
      controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(controlPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addComponent(valueButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addGap(87, 87, 87)
        .addComponent(cardButton, javax.swing.GroupLayout.PREFERRED_SIZE, 95, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addContainerGap(136, Short.MAX_VALUE))
    );
    controlPanelLayout.setVerticalGroup(
      controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(controlPanelLayout.createSequentialGroup()
        .addGroup(controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(valueButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(cardButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addGap(0, 2, Short.MAX_VALUE))
    );

    getContentPane().add(controlPanel, java.awt.BorderLayout.SOUTH);

    pack();
  }// </editor-fold>//GEN-END:initComponents

  private void cardButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cardButtonActionPerformed
    cardLayout.next(mainPanel);
  }//GEN-LAST:event_cardButtonActionPerformed

  // 初期化
  private void initialize() {
    // ボタンにリスナーを登録
    PatternViewer_button_actionAdapter buttonAdapter =
        new PatternViewer_button_actionAdapter(this);
    valueButton.addActionListener(buttonAdapter);
    
    setBounds(); // 表示用フレームの領域を設定
  } // initialize()
  
  void button_actionPerformed(ActionEvent e) {
    String cmd = e.getActionCommand();
    switch (cmd) {
      case "Value": // 数値表示
        drawer0.writeValueFlag = !drawer0.writeValueFlag;
        drawer0.repaint();
        
        drawer1.writeValueFlag = !drawer1.writeValueFlag;
        drawer1.repaint();
        break;

      default:
        break;
    }
  } // button_actionPerformed()
  
  class PatternViewer_button_actionAdapter
       implements java.awt.event.ActionListener {
    PatternViewer adaptee;

    PatternViewer_button_actionAdapter(PatternViewer adaptee) {
      this.adaptee = adaptee;
    }
    
    @Override
    public void actionPerformed(ActionEvent e) {
      adaptee.button_actionPerformed(e);
    }
  }

// 使用しない
//  /**
//   * @param args the command line arguments
//   */
//  public static void main(String args[]) {
//    /* Set the Nimbus look and feel */
////    //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
////    /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
////         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
////     */
////    try {
////      for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
////        if ("Nimbus".equals(info.getName())) {
////          javax.swing.UIManager.setLookAndFeel(info.getClassName());
////          break;
////        }
////      }
////    } catch (ClassNotFoundException ex) {
////      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
////    } catch (InstantiationException ex) {
////      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
////    } catch (IllegalAccessException ex) {
////      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
////    } catch (javax.swing.UnsupportedLookAndFeelException ex) {
////      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
////    }
////    //</editor-fold>
//
//    /* Create and display the form */
//    java.awt.EventQueue.invokeLater(new Runnable() {
//      @Override
//      public void run() {
//        new PatternViewer(
//            PATTERN_PLAIN, DoubleTensor.DT.tensor()).setVisible(true);
//      }
//    });
//  }

// =============================================================================
  
/**
 * <p> 表　題: Class: PatternDrawer</p>
 * <p> 説　明: ２次元パターンを描画する</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.8.27</p>
 */
public class PatternDrawer extends Canvas {
  
  int patternType; // 描画パターン・タイプ
//  DoubleTensor patternData; // パターン描画用データ
  double[][] patternData2; // パターン描画用データ
  double[][][] patternData3; // パターン描画用データ
  
  float[][] matrix2;
  float[][][] matrix3; // パターン描画用行列（３次元）
  boolean writeValueFlag = false; //true; // 数値表示　true:表示有り　false:表示無し
  
  PatternSub ps; // パターン・ビューワーの補助クラス
  
  int page; // 頁番号
  String comment = "Pattern Name"; // コメント

  // パターン・ドローワーの生成と初期化
//  PatternDrawer(int patternType, Tensor patternData) {
  PatternDrawer(int patternType, double[][] patternData) {
    this.patternType = patternType; // 描画パターン・タイプ
    ps = new PatternSub(); // パターン・ビューワーの補助クラス
    ps.getPaletColor(0);
  }
  
  PatternDrawer(int patternType, double[][][] patternData) {
    this.patternType = patternType; // 描画パターン・タイプ
    ps = new PatternSub(); // パターン・ビューワーの補助クラス
    ps.getPaletColor(0);
  }
  
  void setPage(int page) {
    this.page = page;
  }
  
  void setComment(String comment) {
    this.comment = comment;
  }
  
  void setPage_Comment(int page, String comment) {
    this.page = page;
    this.comment = comment;
  }
  
  // パターンの描画
  @Override
  public void paint(Graphics g) {
    // 画面のサイズを取得
    Dimension size = this.getSize();
    // スクリーンを用意する
    ps.prepareScreen(g, size);
    
    if ((patternData2 == null) && (patternData3 == null)) {
      return; // パターン描画用データが設定されていない
    }
    
    // タイトル行を表示
    drawPage(g);
    drawComment(g);
    drawDate(g);
    
    // 描画パターンを選択
    switch (patternType) {
      case PATTERN_CHECK: // 格子（碁盤状）
        checkered(g); // 格子パターンを表示
        break;
      case PATTERN_SPRAY: // 散布
        spray(g);  // 散布パターンを表示
        break;
      case PATTERN_LOGIC: // 論理
        logic(g);  // 論理パターンを表示
        break;
      case PATTERN_GROUP: // 分類
        group(g);  // 分類パターンを表示
        break;
      case PATTERN_JUDGE: // 判定
        judge(g);  // 判定パターンを表示
        break;
      case PATTERN_TUNER: // 調節
        tuner(g);  // 判定パターンを表示
        break;
      case PATTERN_SERVO: // 制御
        servo(g);  // 判定パターンを表示
        break;
      case PATTERN_PLAIN: // 無地
      default:
//        g.setColor(Color.GREEN);
//        g.drawString("PATTERN_PLAIN", 160, 15);
//        drawComment(g, "PATTERN_PLAIN");
        return;
    }
    
    if (writeValueFlag) drawText(g); // パターン描画用行列を数値で表示
  }
  
  // 制御パターンを表示
  void servo(Graphics g) {
    // パターン・データ用の配列を生成する
    makeMatrix();

    ps.setAxesSpace(40, 10, 10, 25); // 表示枠の外側の余白 (int l, r, t, b;)
    ps.drawServoFrame(g); // フレームを表示する
    ps.drawServo(g, matrix3); // 制御パターンを描画
  } // Spray()
  
  // 調節パターンを表示
  void tuner(Graphics g) {
    // パターン・データ用の配列を生成する
    makeMatrix();

    ps.setAxesSpace(40, 10, 10, 25); // 表示枠の外側の余白 (int l, r, t, b;)
    ps.drawServoFrame(g); // フレームを表示する
    ps.drawTuner(g, matrix3); // 制御パターンを描画
  } // Spray()
  
  // 判定パターンを表示
  void judge(Graphics g) {
    makeMatrix(); // パターン・データ用の配列を生成する
    ps.setAxesSpace(40, 10, 10, 25); // 表示枠の外側の余白 (int l, r, t, b;)
    ps.drawFrame(g); // 判定フレームを表示する（仮）
    ps.drawJudge(g, matrix3); // 判定パターンを描画（仮）
  } // Judge()
  
  // 分類パターンを表示
  void group(Graphics g) {
    makeMatrix(); // パターン・データ用の配列を生成する
    ps.setAxesSpace(40, 10, 10, 25); // 表示枠の外側の余白 (int l, r, t, b;)
    ps.drawFrame(g); // フレームを表示する
    ps.drawGroup(g, matrix3); // 散布パターンを描画
  } // Group()
  
  // 論理パターンを表示
  void logic(Graphics g) {
    makeMatrix(); // パターン・データ用の配列を生成する
    ps.setAxesSpace(40, 10, 10, 25); // 表示枠の外側の余白 (int l, r, t, b;)
    ps.drawFrame(g); // フレームを表示する
//    ps.updateParameter(); // グラフ描画用パラメータの更新 drawFrame()から呼ばれる
    ps.drawLogic(g, matrix3); // 散布パターンを描画
  } // Logic()
  
  // 散布パターンを表示
  void spray(Graphics g) {
    // パターン・データ用の配列を生成する
    makeMatrix();

    ps.setAxesSpace(40, 10, 10, 25); // 表示枠の外側の余白 (int l, r, t, b;)
    ps.drawFrame(g); // フレームを表示する
    ps.drawSpray(g, matrix3); // 散布パターンを描画
  } // Spray()
  
  // パターンを碁盤状に表示
  void checkered(Graphics g) {
    // パターン・データ用の配列を生成する
    getEmptyMatrix(); // 空の配列を用意する
    scaleMatrix(); // パターン・データをスケーリングして配列に代入する
    
    ps.framSpace = 30;
    
    // 表示領域全体の幅と高さ
    Dimension size = this.getSize();
    Box board = Box.box(size.height - ps.framSpace, size.width - ps.framSpace);
    
    // パターン表示位置の調整
    int nPat = matrix3.length; // 表示するパターンの個数
    if (nPat < 2) nPat = 2;
    int xFill = ps.framSpace / 2;
    int yFill = ps.framSpace / 2;
    if (matrix3.length == 1) { yFill = 140; }
    else if (matrix3.length >= 10) { xFill = 220; yFill = 10;}
    // 行列の縦横の要素数
    Num slot = Num.num(matrix3[0][0].length, matrix3[0].length);
    // 格子の高さと横幅
    Box pane = Box.box(board.h / slot.r / nPat * 2 -2,
                     board.w / slot.c / nPat * 2 -2);
    // 格子全体の高さと横幅
    Box frame =Box.box(pane.h * slot.r * 2, pane.w * slot.c * 2);
        
    int frameTop = ps.framSpace / 2 + 2;
    int frameBottom = board.h - frame.h - frameTop;
    int frameLeft = ps.framSpace / 2;
    int frameRight = board.w - frame.w - frameLeft;
    
    for (int k = 0; k < matrix3.length; k++) {
      int xShift = (k % 2) * (frame.w / 2 + ps.framSpace / 3); // +220
      int yShift = (k / 2) * (frame.h / 2 + ps.framSpace / 3);
      for (int j = 0; j < slot.c; j++) {
        for (int i = 0; i < slot.r; i++) {
          int col = frameTop + pane.h * j + yShift;
          int row = frameLeft + pane.w * i + xShift;
          g.setColor(Color.getHSBColor(0.0F, 0.0F, 1.0F - matrix3[k][j][i]));
          g.fillRect(row, col, pane.w, pane.h);
        } // i
      } // j
    } // k
    
  } // checkered()
  
  // 頁番号を表示する
  void drawPage(Graphics g) {
    g.setColor(Color.BLACK);
    g.drawString("P." + Integer.toString(page) + ",", 40, 15);
  }
  
  // コメントを表示する
  void drawComment(Graphics g) {
    g.setColor(Color.BLACK);
    g.drawString(comment, 70, 15);
  }
  void drawComment(Graphics g, String comment) {
    g.setColor(Color.BLACK);
    g.drawString(comment, 70, 15);
  }
  
  // 日時表示
  void drawDate(Graphics g) {
    g.setColor(Color.LIGHT_GRAY);
    g.drawString(TimeStamp.getTimeFormated(), 260, 15);
  }
  
  // パターン描画用行列を数値で表示
  void drawText(Graphics g) {
    // 数値表示
    int lng = matrix3.length;
    int row = matrix3[0].length;
    int col = matrix3[0][0].length;
    
    g.setColor(Color.RED);
    for (int k = 0; k < lng; k++) {
      int yShift = k * (16 * row + 20);
      for (int i = 0; i < row; i++) {
        for (int j = 0; j < col; j++) {
          int x = 50 + j * 90;
          int y = 35 + i * 16 + yShift;

          g.drawString(String.valueOf(matrix3[k][i][j]), x, y);
//          g.drawString(String.format("%.5f", matrix[k][i][j]), x, y);
        }
      }
    }
  }
  
  // パターン・データ用の配列を生成し、パターン・データを代入する
  private void makeMatrix() {
    if (matrix3 == null) {
      if (patternData2 != null) {
        int row = patternData2.length;
        int col = patternData2[0].length;
        matrix3 = new float[1][row][col];
      }
      else if (patternData3 != null) {
        int lng = patternData3.length;
        int row = patternData3[0].length;
        int col = patternData3[0][0].length;
        matrix3 = new float[lng][row][col];
      }
    }
    
    if (patternData2 != null) {
      int row = patternData2.length;
      int col = patternData2[0].length;
      for (int i = 0; i < row; i++) {
        for (int j = 0; j < col; j++) {
          matrix3[0][i][j] = (float)patternData2[i][j];
        } // j
      } // i
    }
    else if (patternData3 != null) {
        int lng = patternData3.length;
        int row = patternData3[0].length;
        int col = patternData3[0][0].length;
        for (int k = 0; k < lng; k++) {
        for (int i = 0; i < row; i++) {
          for (int j = 0; j < col; j++) {
            matrix3[k][i][j] = 
                (float)patternData3[k][i][j];
          } // j
        } // i
      } // k
    }
  } // makeMatrix()
  
  // パターン・データ用の空の配列を生成する
  private void getEmptyMatrix() {
    if (matrix3 == null) {
      if (patternData2 != null) {
        int row = patternData2.length;
        int col = patternData2[0].length;
        matrix2 = new float[row][col];
        matrix3 = new float[1][row][col];
      }
      else if (patternData3 != null) {
          int lng = patternData3.length;
          int row = patternData3[0].length;
          int col = patternData3[0][0].length;
          matrix3 = new float[lng][row][col];
      }
    }
  }
  
  // パターン・データ・テンソルをスケーリングして配列に置き換える
  private void scaleMatrix() {
    if (patternData2 != null) {
      scaling(patternData2, matrix2); // スケーリング
      matrix3[0] = matrix2; // ３次元化
    }
    else if (patternData3 != null) {
      scaling(patternData3, matrix3); // スケーリング
    }
  }
  
  /**
   * ２次の配列をゼロから１までの単精度値にスケーリングする
   * 計算式：floatMatrix = scaling(original)
   * 呼出し：scaling(original, floatMatrix)
   * 結果は２次元行列floatMatrixに代入される
   * @param original double[][]
   * @param floatMatrix float[][]
   */
  public void scaling(double[][] original, float[][] floatMatrix) {
    int row = original.length;
    int col = original[0].length;
    
    // 最小値、最大値を求める
    float min = Float.MAX_VALUE;
    float max = Float.MIN_VALUE;
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        float val = (float)original[i][j];
        floatMatrix[i][j] = val;
        if (val < min) min = val;
        if (val > max) max = val;
      } // j
    } // i
    // スケーリング
    float diff = max - min;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        floatMatrix[i][j] = (floatMatrix[i][j] - min) / diff;
      } // j
    } // i
  } // scaling()
  
  /**
   * ３次の配列をゼロから１までの単精度値にスケーリングする
   * 計算式：floatMatrix = (float)scaling(original)
   * 呼出し：scaling(original, floatMatrix)
   * 結果は３次元行列floatMatrixに代入される
   * @param original double[][][]
   * @param floatMatrix float[][][]
   */
  public void scaling(double[][][] original, float[][][] floatMatrix) {
//    DoubleTensor original = this;
    int lng = original.length;
    int row = original[0].length;
    int col = original[0][0].length;
    
    // 最小値、最大値を求める
    float min = Float.MAX_VALUE;
    float max = Float.MIN_VALUE;
    
    for (int k = 0; k < lng; k++) {
      for (int i = 0; i < row; i++) {
        for (int j = 0; j < col; j++) {
          float val = (float)original[k][i][j];
          floatMatrix[k][i][j] = val;
          if (val < min) min = val;
          else if (val > max) max = val;
        } // j
      } // i
    } // k
    // スケーリング
    float diff = max - min;
    for (int k = 0; k < lng; k++) {
      for (int i = 0; i < row; i++) {
        for (int j = 0; j < col; j++) {
          floatMatrix[k][i][j] = (floatMatrix[k][i][j] - min) / diff;
        } // j
      } // i
    } // k
  } // scaling()
  
  // 表示を更新する
  @Override
  public void update(Graphics g) {
    Dimension dim = this.getSize();
    Image offImage = createImage(dim.width, dim.height);
    Graphics offGraphics = offImage.getGraphics();

    if (offImage != null) {
      paint(offGraphics);
      g.drawImage(offImage, 0, 0, null);
    }
  }
  
  /**
   * パターン描画用データを受け取り、再描画を呼ぶ
   * @param patternData double[][] // パターン描画用データ
   */
  public void setMatrix(double[][] patternData) {
    this.patternData2 = patternData;
    repaint();
  }
  
  /**
   * パターン描画用データを受け取り、再描画を呼ぶ
   * @param patternData double[][][] // パターン描画用データ
   */
  public void setMatrix(double[][][] patternData) {
    this.patternData3 = patternData;
    
    repaint();
  }
  
} // PatternDrawer
  
  
  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JButton cardButton;
  private javax.swing.JPanel controlPanel;
  private javax.swing.JPanel imageCard;
  private javax.swing.JPanel mainPanel;
  private java.awt.CardLayout cardLayout;
  private javax.swing.JButton valueButton;
  private javax.swing.JPanel verifyCard;
  // End of variables declaration//GEN-END:variables
} // PatternViewer

// EOF
