/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package view;

import java.awt.CardLayout;
import java.awt.Rectangle;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.StringTokenizer;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import task.Task;
import util.LogEditor;
import util.TimeStamp;

/**
 * <p> 表　題: Class: PropertyViewer</p>
 * <p> 説　明: プロパティの編集（表示・設定・保存・読込）を行う</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.9.21</p>
 */
//public class PropertyViewer extends javax.swing.JFrame {
public class PropertyViewer extends Viewer {

  public static final String FILE_NAME = "Property.ini";
  public static final String BOUNDS_NAME = "propertyBounds";
  private static Console console; // コンソールへの参照
  ArrayList<String> stringList; // プロパティ・ファイルの全内容のリスト

  Class clazz; // アプリケーションのクラス
  Task task; // アプリケーションのインスタンス
  
  String appName; // アプリケーション・クラスの名前
  ArrayList<BoundsProperty> boundsList; // 領域用プロパティのリスト
  ArrayList<Property> parameterList; // パラメータ用プロパティのリスト
  ArrayList<String> buf; // プロパティをファイルへ出力するためのバッファ
  
  /**
   * Creates new form PropertyViewer
   * @param aConsole Console
   */
  public PropertyViewer(Console aConsole) {
    initComponents();
    console = aConsole;
    readProperty(); // プロパティの読み込み
  }
  
  // プロパティの読み込み
//$  private void readProperty() {
  public final void readProperty() {
    // プロパティ・ファイルの読み込み
    stringList = LogEditor.loadText(FILE_NAME);
    if (stringList == null) {
      // プロパティ・ファイルが読込まれなかった
      LogEditor.print("***** ERROR ***** PropertyViewer.readProperty():");
      LogEditor.print("Failed to read a property file.");
      return;
    }

    // プロパティ・ファイルの全内容をプロパティ・ビューワ表示
    int propertyNum = stringList.size();
    for ( int i = 0; i < propertyNum; i++) {
      String line = stringList.get(i);
      writeToTextArea(line + "\n");
    }

    // プロパティ・ファイルから各種プロパティを抽出
    analizeProperty();
    
    // パラメータ欄をパラメータ・テーブルに配置する
    placeParameterStrip();
  }
  
  // プロパティ・ファイル（初期化ファイル）から各種プロパティを抽出
  private void analizeProperty() {
    writeToTextArea("\n" + "アプリとパラメータを抽出" + "\n");
    boundsList = new ArrayList<>(); // 領域型プロパティのリスト
    parameterList = new ArrayList<>(); // パラメータ型プロパティのリスト
      
    int propertyNum = stringList.size();
    for ( int i = 0; i < propertyNum; i++) {
      String line = stringList.get(i); // １行分取り出し
      if (line.startsWith("#")) continue; // コメントは無視する
      if (line.startsWith("[")) continue; // グループ・ラベルは無視する
      
      // アプリケーション名を抽出
      if (line.startsWith("APL")) {
        pickoutAppName(line);
        continue;
      } // APL
      
      // 領域（座標とサイズ）プロパティを抽出
      if (line.startsWith("BND")) {
        pickoutBounds(line);
        continue;
      } // BND
      
      // パラメータ用プロパティを抽出
      if (line.startsWith("PRM")) {
        pickoutParameter(line);
//        continue;
      } // PRM
      
    } // i
    
    // パラメータ用プロパティ・リストの内容を表示
    if (parameterList != null) {
      int prmNum = parameterList.size();
      for ( int i = 0; i < prmNum; i++) {
        Property prop = parameterList.get(i);
        writeToTextArea("Param: Name=" + prop.getName()
            + ", Type=" + prop.getType() + ", Value=" + prop.getValue()
             + ", Comment=" + prop.getComment() + "\n");
      }
    
    }
  } // analizeProperty()
  
  // タスク（ニューラルネットワーク）名を抽出
  private void pickoutAppName(String line) {
    StringTokenizer token = new StringTokenizer(line, ",");
    if (token.hasMoreTokens()) {
      String propertyLabel = token.nextToken(); // プロパティのラベルを読飛し
      if (token.hasMoreTokens()) {
        appName = token.nextToken(); // アプリケーション名を抽出し代入
        writeToTextArea("Application: Name=" + appName + "\n");
        tasknameTextField.setText(appName); // プロパティ・テーブルに表示
        return;
      }
    }
    // トークンが無い
    LogEditor.print("***** WARNING ***** PropertyViewer.pickoutAppName(): ");
    LogEditor.print("There are no next-token.");
  }
  
   // アプリケーション・クラスの名前を得る
  public String getAppName() {
    return appName;
  }
  
  // 領域（座標とサイズ）用プロパティを抽出
  private void pickoutBounds(String line) {
    StringTokenizer token = new StringTokenizer(line, ",");
    String propertyLabel = token.nextToken(); // プロパティのラベルを読飛し
    String propertyType =  token.nextToken(); // プロパティ型を抽出
    Property<Rectangle> prop; // Rectangle型プロパティ
    try {
      Class<?> propClass = Class.forName("view." + propertyType); // 領域型プロパティのクラス
// 2024.2.13      prop = (Property)(propClass.newInstance()); // 領域型プロパティのインスタンス
      prop = (Property)(propClass.getDeclaredConstructor().newInstance()); // 領域型プロパティのインスタンス
    }
    catch (ReflectiveOperationException e) {
      LogEditor.print("***** WARNING ***** PropertyViewer.pickoutBounds():");
      LogEditor.print("ReflectiveOperationException occur.");
      return;
    }
    prop.setType(propertyType); // プロパティ型を代入
    prop.setName(token.nextToken()); // プロパティ名を抽出し代入
    Rectangle rect = new Rectangle(); // Rectangle型（座標とサイズ）
    rect.x = Integer.parseInt(token.nextToken()); // 座標を抽出
    rect.y = Integer.parseInt(token.nextToken());
    rect.width = Integer.parseInt(token.nextToken()); // サイズ抽出
    rect.height = Integer.parseInt(token.nextToken());
    prop.setValue(rect); // 領域（座標とサイズ）を抽出し代入
    prop.setComment(token.nextToken()); // 領域（座標とサイズ）を抽出し代入
    writeToTextArea("Bounds: Name=" + prop.getName() 
        + ", Type=" + prop.getType()
        + ", x=" + prop.getValue().x + ", y=" + prop.getValue().y
        + ", w=" + prop.getValue().width + ", h=" + prop.getValue().height
        +"\n");
    boundsList.add((BoundsProperty)prop); // 領域型プロパティのリスト
  }
  
  // 領域（座標とサイズ）用プロパティのリストを得る
  private ArrayList<BoundsProperty> getBoundsProperty() {
    return  boundsList;
  }
  
  // 所与の名前の領域用プロパティの値を得る
  public Rectangle getBoundsValue(String boundsName) {
    int boundsNum = boundsList.size();
    Rectangle rect = null;
    for (int i = 0; i < boundsNum; i++) {
      BoundsProperty boundsProp = boundsList.get(i);
      String propName = boundsProp.getName();
      if (propName.equals(boundsName)) {
        rect = (Rectangle)boundsProp.getValue();
        break;
      }
    }
    return rect;
  }
  
  // ビューワから領域（座標とサイズ）プロパティを取り出し、
  // 領域プロパティ・リストに戻す
//$  private void restoreBoundsList() {
  public void restoreBoundsList() {
    Rectangle viewerRect; // ビューワの領域（座標とサイズ）
    int boundsNum = boundsList.size();
    for (int i = 0; i < boundsNum; i++) {
      BoundsProperty boundsProp = boundsList.get(i);
      String propName = boundsProp.getName();
      switch (propName) {
        case Console.BOUNDS_NAME:
          viewerRect = console.getBounds();
          boundsProp.setValue(viewerRect);
          break;
        case PropertyViewer.BOUNDS_NAME:
          viewerRect = getBounds();
          boundsProp.setValue(viewerRect);
          break;
        case PatternViewer.BOUNDS_NAME:
          viewerRect = getViewerBounds("patternViewer"); // パターン・ビューワ
          if (viewerRect != null) boundsProp.setValue(viewerRect);
          break;
        case GraphViewer.BOUNDS_NAME:
          viewerRect = getViewerBounds("graphViewer"); // グラフ・ビューワ
          if (viewerRect != null) boundsProp.setValue(viewerRect);
          break;
        default:
          break;
      }
    } // boundsNum
  } // restoreBoundsList()
  
  // ビューワから領域（座標とサイズ）を得る
  private Rectangle getViewerBounds(String viewerName) {
    Rectangle bounds = null;    
    try{  
      // フィールドの取得
      Class superClass = task.getClass().getSuperclass();
      Field viewer = superClass.getDeclaredField(viewerName);
      viewer.setAccessible(true);

      // フィールドから境界を取り出す
      JFrame aViewer = (JFrame)viewer.get(task);
      bounds = aViewer.getBounds();
    }
    catch(NullPointerException e) {
      LogEditor.print("***** WARNING ***** PropertyViewer.getViewerBounds():");
      LogEditor.print("There is no " + viewerName + ".");
    }
    catch(ReflectiveOperationException e) {
      LogEditor.print("***** ERROR ***** PropertyViewer.getViewerBounds():");
      LogEditor.print("ReflectiveOperationException occur.");
    }
    
    return bounds;
  }
  
  // パラメータ用プロパティを抽出
  private void pickoutParameter(String line) {
    StringTokenizer token = new StringTokenizer(line, ",");
    int tokenNum = token.countTokens();
    if (tokenNum < 5) {
      LogEditor.print("***** WARNING ***** PropertyViewer.pickoutParameter():");
      LogEditor.print(" token=" + tokenNum + " < 5.");
      return;
    }
    
    String propertyLabel = token.nextToken(); // プロパティのラベルを読飛し
    String propertyType =  token.nextToken(); // パラメータ型を抽出
    
    Property prop = null; // Rectangle型プロパティ
    switch (propertyType) {
      case Property.TYPE_INT:
        {
          prop = new IntProperty<>();
          prop.setName(token.nextToken()); // パラメータ名を抽出して代入
          String propertyValue = token.nextToken().trim(); // パラメータ値を抽出
          prop.setValue(Integer.parseInt(propertyValue)); // パラメータ値を代入
          break;
        }
      case Property.TYPE_DOUBLE:
        {
          prop = new DoubleProperty<>();
          prop.setName(token.nextToken()); // パラメータ名を抽出して代入
          String propertyValue = token.nextToken().trim(); // パラメータ値を抽出
          prop.setValue(Double.parseDouble(propertyValue)); // パラメータ値を代入
          break;
        }
      case Property.TYPE_FLOAT:
        {
          prop = new FloatProperty<>();
          prop.setName(token.nextToken()); // パラメータ名を抽出して代入
          String propertyValue = token.nextToken().trim(); // パラメータ値を抽出
          prop.setValue(Float.parseFloat(propertyValue)); // パラメータ値を代入
          break;
        }
      case Property.TYPE_STRING:
        {
          prop = new StringProperty<>();
          prop.setName(token.nextToken()); // パラメータ名を抽出して代入
          String propertyValue = token.nextToken().trim(); // パラメータ値を抽出
          prop.setValue(propertyValue); // パラメータ値を代入
          break;
        }
      default:
        break;
    }
    if (prop != null) {
      prop.setComment(token.nextToken()); // パラメータ名を抽出
      parameterList.add(prop); // パラメータ用プロパティ・リストに追加
    }

  }
  
  // パラメータ・リストの値を代入したパラメータ欄を、
  // パラメータ・テーブルに配置する
  private void placeParameterStrip() {
    // 既にパラメータ欄が有れば、すべて消去する
    if (tablePanel.getComponentCount() > 0)  tablePanel.removeAll();
    
    // パラメータ・テーブルを構成
    int pramNum = parameterList.size();
    for (int i = 0; i < pramNum; i++) {
      Property param = parameterList.get(i);
      String name = param.getName();
      PropertyStrip propStrip = new PropertyStrip();
      String type = param.getType();
      propStrip.setType(type);
      propStrip.setName(param.getName());
      String value;
      
      switch (type) {
        case Property.TYPE_INT:
          value = Integer.toString((Integer)param.getValue());
          break;
        case Property.TYPE_DOUBLE:
          value = Double.toString((Double)param.getValue());
          break;
        case Property.TYPE_FLOAT:
          value = Float.toString((Float)param.getValue());
          break;
        case Property.TYPE_STRING:
        default:
          value = (String)param.getValue();
          break;
      }
      propStrip.setValue(value);
      propStrip.setComment(param.getComment());
      tablePanel.add(propStrip);
//      tablePanel.repaint();
      tablePanel.validate(); // パネルの再描画を促す
    }
  } // placeParameterStrip
  
  // パラメータ・テーブルのパラメータ欄からパラメータ・プロパティを取り出し
  // パラメータ・リストに戻す
//$  private void restoreParamerList() {
  public void restoreParamerList() {
    ArrayList<Property> paramList = new ArrayList<>(); // パラメータ・リスト
    int stripNum = tablePanel.getComponentCount();
    for (int i = 0; i < stripNum; i++) {
      PropertyStrip propStrip = (PropertyStrip)tablePanel.getComponent(i);
      JTextField field = (JTextField)propStrip.getComponent(0);
      String type = field.getText(); // 型
      Property prop;
      switch (type) {
        case Property.TYPE_INT:
          prop = new IntProperty();
          field = (JTextField)propStrip.getComponent(2);
          prop.setValue(Integer.parseInt(field.getText())); // 値
          break;
        case Property.TYPE_DOUBLE:
          prop = new DoubleProperty();
          field = (JTextField)propStrip.getComponent(2);
          prop.setValue(Double.parseDouble(field.getText())); // 値
          break;
        case Property.TYPE_FLOAT:
          prop = new DoubleProperty();
          field = (JTextField)propStrip.getComponent(2);
          prop.setValue(Float.parseFloat(field.getText())); // 値
          break;
        case Property.TYPE_STRING:
        default:
          prop = new StringProperty();
          field = (JTextField)propStrip.getComponent(2);
          prop.setValue(field.getText()); // 値
          break;
      }
      
      field = (JTextField)propStrip.getComponent(1);
      prop.setName(field.getText()); // 名前
      field = (JTextField)propStrip.getComponent(3);
      prop.setComment(field.getText()); // コメント
      
      paramList.add(prop);
    } // stripNum
    
    parameterList = paramList; // パラメータ用プロパティのリストを更新
  }
  
  // パラメータの変更をチェックする
  public boolean checkParameterChange() {
    boolean parameterChangedFlag = false;
    JTextField field;
    Property param;
    int stripNum = tablePanel.getComponentCount();
    for (int i = 0; i < stripNum; i++) {
      PropertyStrip propStrip = (PropertyStrip)tablePanel.getComponent(i);
      field = (JTextField)propStrip.getComponent(0);
      String type = field.getText(); // 型
      field = (JTextField)propStrip.getComponent(2);
      param = parameterList.get(i);
      
      // パラメータの値をチェック
      switch (type) {
        case Property.TYPE_INT:
          int iValue = Integer.parseInt(field.getText()); // 値
          int iParam = (Integer)param.getValue();
          if (iValue != iParam) parameterChangedFlag = true;
          break;
        case Property.TYPE_DOUBLE:
          double dValue = Double.parseDouble(field.getText()); // 値
          double dParam = (Double)param.getValue();
          if (dValue != dParam) parameterChangedFlag = true;
          break;
        case Property.TYPE_FLOAT:
          field = (JTextField)propStrip.getComponent(2);
          float fValue = Float.parseFloat(field.getText()); // 値
          float fParam = (Float)param.getValue();
          if (fValue != fParam) parameterChangedFlag = true;
          break;
        case Property.TYPE_STRING:
        default:
          field = (JTextField)propStrip.getComponent(2);
          String sValue = field.getText(); // 値
          String sParam = (String)param.getValue();
          if (!sValue.equals(sParam)) parameterChangedFlag = true;
          break;
      }
      
      // パラメータ名をチェック
      field = (JTextField)propStrip.getComponent(1); // 名前
      String sValue = field.getText(); // 値
      String sParam = (String) param.getName();
      if (!sValue.equals(sParam)) {
        parameterChangedFlag = true;
        LogEditor.print("***** ERROR ***** PropertyViewer.checkParameterChange():");
        LogEditor.print("Parameter name is different."
                + " [name:" + sValue + ", field:" + sParam + "]" );
      }
      
      if (parameterChangedFlag) { // パラメータが変更された
        LogEditor.print("PropertyViewer.checkParameterChange():"
                + "　Parameters[" + i +  "] is changed.");
        break;
      }
    }
    
    if (parameterChangedFlag) { // パラメータが変更された
      return disposal();
    }
    
    return true; // パラメータは変更されていない
  }
  
  // ダイアログを表示し、パラメータ変更を保存または破棄する
  public boolean disposal() {
    LogEditor.print("PropertyViewer.disposal():　disposal() is called.");
    
    // DisposalDialogを表示
    int ans = JOptionPane.showConfirmDialog(
        console, "パラメータの変更は有効になっていません。\n　はい、変更を有効にします。"
        + "\n　いいえ、変更を破棄します。\n　タスクの実行を取消ます。",
        "パラメータが変更されています",
//        JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
        JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE);
    
    switch (ans) {
      case JOptionPane.YES_OPTION:
        // パラメータを保存する
        restoreParamerList(); // パラメータ・テーブルの値をパラメータ・リストに保存する         
        LogEditor.print("PropertyViewer.disposal(): All parameters are saved.");
        break;
      case JOptionPane.NO_OPTION:
        // パラメータの変更を破棄する
        placeParameterStrip(); // パラメータ・リストの値をパラメータ・テーブルに戻す
        LogEditor.print("PropertyViewer.disposal(): Changed parameter are discarded.");
        break;
      default:
        // タスクの実行をキャンセルする
        LogEditor.print("PropertyViewer.disposal(): Task run operation is canceled.");
        return false;
    }
    
    return true;
  }
  
  
  /**
   * This method is called from within the constructor to initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is always
   * regenerated by the Form Editor.
   */
  @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    mainPanel = new javax.swing.JPanel();
    tableCard = new javax.swing.JPanel();
    tableScrollPane = new javax.swing.JScrollPane();
    propertyPanel = new javax.swing.JPanel();
    titlePanel = new javax.swing.JPanel();
    taskLabel = new javax.swing.JLabel();
    typeLabel = new javax.swing.JLabel();
    nameLabel = new javax.swing.JLabel();
    valueLabel = new javax.swing.JLabel();
    commentLabel = new javax.swing.JLabel();
    tasknameTextField = new javax.swing.JTextField();
    tablePanel = new javax.swing.JPanel();
    textCard = new javax.swing.JPanel();
    textScrollPane = new javax.swing.JScrollPane();
    propertyTextArea = new javax.swing.JTextArea();
    controlPanel = new javax.swing.JPanel();
    saveButton = new javax.swing.JButton();
    loadButton = new javax.swing.JButton();
    cardButton = new javax.swing.JButton();
    cancelButton = new javax.swing.JButton();
    setButton = new javax.swing.JButton();

    setTitle("DaiJa : Property viewer");

    mainPanel.setPreferredSize(new java.awt.Dimension(600, 350));
    mainPanel.setLayout(new java.awt.CardLayout());

    tableCard.setPreferredSize(new java.awt.Dimension(600, 350));
    tableCard.setLayout(new java.awt.BorderLayout());

    tableScrollPane.setHorizontalScrollBarPolicy(javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);

    propertyPanel.setPreferredSize(new java.awt.Dimension(600, 348));
    propertyPanel.setLayout(new java.awt.BorderLayout());

    titlePanel.setPreferredSize(new java.awt.Dimension(600, 55));

    taskLabel.setHorizontalAlignment(javax.swing.SwingConstants.TRAILING);
    taskLabel.setText("  Task");

    typeLabel.setText("Type");

    nameLabel.setText("Name");

    valueLabel.setText("Value");

    commentLabel.setText("Comment");

    tasknameTextField.setEditable(false);
    tasknameTextField.setText("Task name");

    javax.swing.GroupLayout titlePanelLayout = new javax.swing.GroupLayout(titlePanel);
    titlePanel.setLayout(titlePanelLayout);
    titlePanelLayout.setHorizontalGroup(
      titlePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(titlePanelLayout.createSequentialGroup()
        .addContainerGap()
        .addGroup(titlePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
          .addComponent(typeLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 34, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(taskLabel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.PREFERRED_SIZE, 34, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
        .addGroup(titlePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
          .addGroup(titlePanelLayout.createSequentialGroup()
            .addComponent(nameLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 80, javax.swing.GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
            .addComponent(valueLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 72, javax.swing.GroupLayout.PREFERRED_SIZE)
            .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
            .addComponent(commentLabel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
          .addComponent(tasknameTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 276, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addContainerGap(266, Short.MAX_VALUE))
    );
    titlePanelLayout.setVerticalGroup(
      titlePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, titlePanelLayout.createSequentialGroup()
        .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        .addGroup(titlePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(taskLabel)
          .addComponent(tasknameTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addGroup(titlePanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(typeLabel)
          .addComponent(nameLabel)
          .addComponent(valueLabel)
          .addComponent(commentLabel))
        .addGap(17, 17, 17))
    );

    propertyPanel.add(titlePanel, java.awt.BorderLayout.NORTH);

    tablePanel.setMinimumSize(new java.awt.Dimension(400, 180));
    tablePanel.setPreferredSize(new java.awt.Dimension(400, 180));
    tablePanel.setLayout(new java.awt.GridLayout(17, 0));
    propertyPanel.add(tablePanel, java.awt.BorderLayout.CENTER);

    tableScrollPane.setViewportView(propertyPanel);

    tableCard.add(tableScrollPane, java.awt.BorderLayout.CENTER);

    mainPanel.add(tableCard, "tableCard");

    textCard.setPreferredSize(new java.awt.Dimension(600, 350));
    textCard.setLayout(new java.awt.BorderLayout());

    propertyTextArea.setColumns(20);
    propertyTextArea.setRows(5);
    textScrollPane.setViewportView(propertyTextArea);

    textCard.add(textScrollPane, java.awt.BorderLayout.CENTER);

    mainPanel.add(textCard, "textCard");

    //CardLayout cardLayout = new CardLayout();
    mainPanel.setLayout(cardLayout);
    getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);

    controlPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
    controlPanel.setPreferredSize(new java.awt.Dimension(600, 24));

    saveButton.setText("Save");
    saveButton.setMargin(new java.awt.Insets(2, 10, 2, 10));
    saveButton.setPreferredSize(new java.awt.Dimension(46, 18));
    saveButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        saveButtonActionPerformed(evt);
      }
    });

    loadButton.setText("Re-read");
    loadButton.setMargin(new java.awt.Insets(2, 10, 2, 10));
    loadButton.setPreferredSize(new java.awt.Dimension(63, 18));
    loadButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        loadButtonActionPerformed(evt);
      }
    });

    cardButton.setText("Next Page");
    cardButton.setMargin(new java.awt.Insets(2, 10, 2, 10));
    cardButton.setPreferredSize(new java.awt.Dimension(76, 18));
    cardButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        cardButtonActionPerformed(evt);
      }
    });

    cancelButton.setText("Cancel");
    cancelButton.setActionCommand("");
    cancelButton.setMargin(new java.awt.Insets(2, 10, 2, 10));
    cancelButton.setPreferredSize(new java.awt.Dimension(57, 18));
    cancelButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        cancelButtonActionPerformed(evt);
      }
    });

    setButton.setText("Set");
    setButton.setMargin(new java.awt.Insets(2, 10, 2, 10));
    setButton.setPreferredSize(new java.awt.Dimension(38, 18));
    setButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        setButtonActionPerformed(evt);
      }
    });

    javax.swing.GroupLayout controlPanelLayout = new javax.swing.GroupLayout(controlPanel);
    controlPanel.setLayout(controlPanelLayout);
    controlPanelLayout.setHorizontalGroup(
      controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(controlPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addComponent(saveButton, javax.swing.GroupLayout.PREFERRED_SIZE, 59, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(loadButton, javax.swing.GroupLayout.PREFERRED_SIZE, 75, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(cardButton, javax.swing.GroupLayout.PREFERRED_SIZE, 85, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(cancelButton, javax.swing.GroupLayout.PREFERRED_SIZE, 65, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(setButton, javax.swing.GroupLayout.PREFERRED_SIZE, 48, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addContainerGap())
    );
    controlPanelLayout.setVerticalGroup(
      controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(controlPanelLayout.createSequentialGroup()
        .addGroup(controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(saveButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(loadButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(cardButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(cancelButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(setButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addGap(0, 2, Short.MAX_VALUE))
    );

    getContentPane().add(controlPanel, java.awt.BorderLayout.SOUTH);

    pack();
  }// </editor-fold>//GEN-END:initComponents

  private void cardButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cardButtonActionPerformed
    cardLayout.next(mainPanel);
  }//GEN-LAST:event_cardButtonActionPerformed

  private void saveButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_saveButtonActionPerformed
    restoreBoundsList(); // ビューワから領域を取り出し、領域プロパティ・リストに戻す
    restoreParamerList(); // パラメータ・テーブルからパラメータ・プロパティを取り出す
    restoreProperty(); // プロパティを再生成し、ファイルに保存する
  }//GEN-LAST:event_saveButtonActionPerformed

  private void loadButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_loadButtonActionPerformed
    readProperty(); // プロパティ・ファイルを再読み込み
  }//GEN-LAST:event_loadButtonActionPerformed

  private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
    placeParameterStrip(); // パラメータ・リストの値をパラメータ・テーブルに
  }//GEN-LAST:event_cancelButtonActionPerformed

  private void setButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_setButtonActionPerformed
    restoreParamerList(); // パラメータ・テーブルの値をパラメータ・リストに戻す
    // 既にアプリケーションが実行されていたら、強制的に
    // アプリケーションのパラメータをパラメータ・リストで置き換える
///    if ((clazz != null) && (task != null)) putParameter(clazz, task);
  }//GEN-LAST:event_setButtonActionPerformed

   /**
   * プロパティをテキストエリアに追加する。
   * @param text String // メッセージ
   */
  public final void writeToTextArea(String text) {
    propertyTextArea.append(text);
  }
  
// 使用しない
//  /**
//   * @param args the command line arguments
//   */
//  public static void main(String args[]) {
//    /* Set the Nimbus look and feel */
//    //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
//    /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
//         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
//     */
//    try {
//      for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
//        if ("Nimbus".equals(info.getName())) {
//          javax.swing.UIManager.setLookAndFeel(info.getClassName());
//          break;
//        }
//      }
//    } catch (ClassNotFoundException | InstantiationException | IllegalAccessException | javax.swing.UnsupportedLookAndFeelException ex) {
//      java.util.logging.Logger.getLogger(PropertyViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
//    }
//    //</editor-fold>
//    
//    //</editor-fold>
//
//    /* Create and display the form */
//    java.awt.EventQueue.invokeLater(new Runnable() {
//      @Override
//      public void run() {
//        new PropertyViewer().setVisible(true);
//      }
//    });
//  }
  
  // 領域をプロパティで置き換え
  public void putBounds() {
    Rectangle rect = getBoundsValue("propertyBounds");
    if (rect != null) {
    setBounds(rect);
    }
    else { // ディフォルト値を設定
    setBounds(600, 0, 400, 400);
    }
  }

  // アプリケーションのパラメータをパラメータ・リストで置き換え
  public void putParameter(Class clazz, Task task) {
    this.clazz = clazz; // アプリケーションのクラス
    this.task = task; // アプリケーションのインスタンス
    
    int prmNum = parameterList.size();
    for (int i = 0; i < prmNum; i++) {
      Property prmProp = parameterList.get(i); // プロパティの取り出し
      String fieldName = prmProp.getName(); // パラメータ名の取り出し
      String fieldType = prmProp.getType(); // パラメータ型の取り出し
      
      Object prmValue = null;
      switch (fieldType) {
        case Property.TYPE_INT:
          // 整数型パラメータか？
          prmValue = (int)(prmProp.getValue()); // パラメータ値の取り出し
          break;
        case Property.TYPE_DOUBLE:
          // 倍精度型パラメータか？
          prmValue = (double)(prmProp.getValue()); // パラメータ値の取り出し
          break;
        case Property.TYPE_STRING:
          // 文字列型パラメータか？
          prmValue = (String)(prmProp.getValue()); // パラメータ値の取り出し
          break;
        case Property.TYPE_FLOAT:
          // 単精度型パラメータか？
          prmValue = (float)(prmProp.getValue()); // パラメータ値の取り出し
          break;
        default:
          LogEditor.print("***** ERROR ***** PropertyViewer.putParameter():");
          LogEditor.print("No such a parameter type:" + fieldType);
          return;
      }
      
      try{  
        // フィールド(str)の取得
        Field prmField = clazz.getDeclaredField(fieldName);
        prmField.setAccessible(true);
        writeToTextArea(fieldName + " Before: " 
            + prmField.get(task).toString() + ", "); 

        // フィールド(str)の変更
        prmField.set(task, prmValue);
        writeToTextArea(" After:" + prmField.get(task).toString() + "\n");
      }
      catch(ReflectiveOperationException e) {
        LogEditor.print("***** WARNING ***** PropertyViewer.putParameter():");
        LogEditor.print("ReflectiveOperationException occur:" + fieldName);
//        return;
      }
    }
  }
  
  // プロパティを再生成し、ファイルに保存する
  public void restoreProperty() {
    buf = new ArrayList<>(100);
    addRL("#Title:DaiJa Property file");
    String taskName = appName.replaceFirst("task.", "");
    String fileName = "DaiJaProperty_" + taskName + "_"
        + TimeStamp.getDateAndTime() +".ini";
    addRL("#File name:" + fileName);
    
    addRL("#");
    addRL("[TIME] #Date and Time");
    addRL("VAL,String,date," + TimeStamp.getDateStamp() + ",年／月／日");
    addRL("VAL,String,time," + TimeStamp.getTimeStamp() + ",時：分：秒");
    
    addRL("#");
    addRL("[BND] #Bounds properties");
    int boundsNum = boundsList.size();
    for (int i = 0; i < boundsNum; i++) {
      BoundsProperty boundsProp = boundsList.get(i);
      String propName = boundsProp.getName();
      Rectangle rect = (Rectangle)boundsProp.getValue();
      addRL("BND,BoundsProperty," + propName + "," 
        + rect.x + "," + rect.y + ","+ rect.width + ","+ rect.height + ","
        + boundsProp.getComment());
    }
    
    addRL("#");
    addRL("[APL] #Application");
    addRL("APL," + appName);
    
    addRL("#");
    addRL("[PRM] #Parameters");
    int parameterNum = parameterList.size();
    for (int i = 0; i < parameterNum; i++) {
      Property prmProp = parameterList.get(i); // プロパティの取り出し
      String prmName = prmProp.getName(); // パラメータ名の取り出し
      String prmType = prmProp.getType(); // パラメータ型の取り出し
      
      String value = String.valueOf(prmProp.getValue());
      String comment = prmProp.getComment(); // コメントの取り出し
      
      addRL("PRM," + prmType + "," + prmName + "," + value + "," + comment);
    }
    
    addRL("#");
    addRL("[EOF] #End of file");
    
    LogEditor.saveText(buf, fileName); // ファイルに保存
  }
  
  // バッファに文字列と改行・復帰を追加する
  private void addRL(String line) {
    buf.add(line + "\r\n");
  }
  
  // カード・レイアウトを追加設定
  CardLayout cardLayout = new CardLayout();
//    mainPanel.setLayout(cardLayout);

  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JButton cancelButton;
  private javax.swing.JButton cardButton;
  private javax.swing.JLabel commentLabel;
  private javax.swing.JPanel controlPanel;
  private javax.swing.JButton loadButton;
  private javax.swing.JPanel mainPanel;
  private javax.swing.JLabel nameLabel;
  private javax.swing.JPanel propertyPanel;
  private javax.swing.JTextArea propertyTextArea;
  private javax.swing.JButton saveButton;
  private javax.swing.JButton setButton;
  private javax.swing.JPanel tableCard;
  private javax.swing.JPanel tablePanel;
  private javax.swing.JScrollPane tableScrollPane;
  private javax.swing.JLabel taskLabel;
  private javax.swing.JTextField tasknameTextField;
  private javax.swing.JPanel textCard;
  private javax.swing.JScrollPane textScrollPane;
  private javax.swing.JPanel titlePanel;
  private javax.swing.JLabel typeLabel;
  private javax.swing.JLabel valueLabel;
  // End of variables declaration//GEN-END:variables
}

// =============================================================================

/**
 * <p> 表　題: Class: Property</p>
 * <p> 説　明: プロパティの保持</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.9.24</p>
 */
abstract class Property<T> {
  // プロパティ型
  static final String TYPE_STRING = "String"; // 文字列型 
  static final String TYPE_INT = "int"; // 整数型 
  static final String TYPE_FLOAT = "float"; // 単精度浮動小数型 
  static final String TYPE_DOUBLE = "double"; // 倍精度浮動小数型 
  static final String TYPE_BOUNDS = "Bounds"; // 座標とサイズ型 
  
  String type; // プロパティ型
  String name; // プロパティ名
  T value; // 値
  String comment; // コメント
  
  String getType() { return type; }
  void setType(String type) { this.type = type; }
  
  String getName() { return name; }
  void setName(String name) { this.name = name; }
  
  T getValue() { return value; }
  void setValue(T value) { this.value = value; }
  
  String getComment() { return comment; }
  void setComment(String comment) { this.comment = comment; }
} // Property

// 文字列型プロパティ
class StringProperty<String> extends Property {
  StringProperty() { type = Property.TYPE_STRING; }
}

// 整数型プロパティ
class IntProperty<Integer> extends Property {
  IntProperty() { type = Property.TYPE_INT; }
}

// 単精度浮動小数型プロパティ
class FloatProperty<Float> extends Property {
  FloatProperty() { type = Property.TYPE_FLOAT; }
}

// 倍精度浮動小数型プロパティ
class DoubleProperty<Double> extends Property {
  DoubleProperty() { type = Property.TYPE_DOUBLE; }
}

// 領域（座標とサイズ）型プロパティ
//class BoundsProperty<Rect> extends Property {
class BoundsProperty<Rectangle> extends Property {
  BoundsProperty() { type = Property.TYPE_BOUNDS; }
}

// EOF
