/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package view;

import java.awt.Dimension;

/**
 * <p> 表　題: Class: Ruler</p>
 * <p> 説　明: 多次元ルーラ</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.8.27</p>
 */
public class Ruler {
}

/**
 * 縦と横の長さ（ピクセル数）を保持するクラス
 Box(int height, int width)
 h: 縦方向のピクセル数
 w: 横方向のピクセル数
 */
class Box {
  int h, w;
  // 要素ゼロの生成子
  Box() {
    h = 0;
    w = 0;
  }
  // 所与の要素での生成子
  Box(int height, int width) {
    h = height;
    w = width;
  }
  
  // 要素ゼロの代理生成子
  static Box box() {
    return new Box();
  }
  
  // 所与の要素での代理生成子
  static Box box(int height, int width) {
    return new Box(height, width);
  }
  
  // 要素をゼロ・クリア
  void clear() {
    h = 0;
    w = 0;
  }
  
  // Dimensionを得る
  Dimension getDimension() {
    return new Dimension(w, h);
  }
  // Numを得る
  Num getNum() {
    return new Num(h, w);
  }
  // Dotを得る
  Dot getDot() {
    return new Dot(w, h);
  }
  
} // Box

/**
 * 縦方向と横方向の要素数を保持するクラス
 *  Num(int row, int col)
 * r: 縦方向の要素数
 * c: 横方向の要素数
 */
class Num {
  int r, c;
  // 要素ゼロの生成子
  Num() {
    r = 0;
    c = 0;
  }
  // 所与の要素での生成子
  Num(int row, int col) {
    r = row;
    c = col;
  }
  
  // 要素ゼロの代理生成子
  static Num num() {
    return new Num();
  }
  // 所与の要素での代理生成子
  static Num num(int row, int col) {
    return new Num(row, col);
  }
  
  // 要素をゼロ・クリア
  void clear() {
    r = 0;
    c = 0;
  }

  // Dimensionを得る
  Dimension getDimension() {
    return new Dimension(c, r);
  }
  // Dimを得る
  Box getDim() {
    return new Box(r, c);
  }
  // Dotを得る
  Dot getDot() {
    return new Dot(c, r);
  }
  
} // Num

/**
 * ２次元変数を保持するクラス Flow Mass
 * x: X軸方向の数量（単精度）
 * y: Y軸方向の数量（単精度）
 */
class Dim {
  float x, y;
  
  // 要素ゼロの生成子
  Dim() { x = 0; y = 0; }
  
  // 所与の要素での生成子
  Dim(float x, float y) { this.x = x; this.y = y; }
  
  // 要素ゼロの代理生成子
  static Dim dim() { return new Dim(); }
  
  // 所与の要素での代理生成子
  static Dim dim(float x, float y) { return new Dim(x, y); }
  
  // 要素をゼロ・クリア
  void clear() { x = 0.0F; y = 0.0F; }
  
} // Dim

/**
 * ２要素を保持するクラス
 * Twin(int m, int n)
 * m: 第１要素
 * n: 第２要素
 */
class Twin {
  int m, n;
  
  // 要素ゼロの生成子
  Twin() { m = 0; n = 0; }
  
  // 所与の要素での生成子
  Twin(int m, int n) { this.m = m; this.n = n; }

  // 要素ゼロの代理生成子
  static Twin twin() { return new Twin(); }
  
  // 所与の要素での代理生成子
  static Twin twin(int m, int n) { return new Twin(m, n); }
  
  // 要素をゼロ・クリア
  void clear() { m = 0; n = 0; }
  
} // Twin

// =============================================================================
/**
 * ２次元グラフ（time, value）を保持するクラス
 * ※Dot（画面用）と対応して用いる
 * Fig(int t, float v)
 * t: 横軸（時間軸、右方向が正）の数量（整数）
 * v: 縦軸（従属変数、上方向が正）の数量（単精度）
 */
class Fig {
  int t; float v;
  float shiftT; float shiftY;
  float magT; float magY;
  
  Edge minMax; // ２次元グラフの範囲
  Quad param; // 座標変換パラメータ

  void setRegion(int tMin, int tMax, float vMin, float vMax) {
    minMax = new Edge(tMin, tMax, vMin, vMax);
  }
  
  void setParam(float shiftT, float magT, float shiftY, float magY) {
    param = new Quad(shiftT, magT, shiftY, magY);
  }
  
  // 要素ゼロの生成子
  Fig() { t = 0; v = 0.0F; }
  
  // 所与の要素での生成子
  Fig(int t, float v) { this.t = t; this.v = v; }
  
  // 要素ゼロの代理生成子
  static Fig fig() { return new Fig(); }
  
  // 所与の要素での代理生成子
  static Fig fig(int t, float v) { return new Fig(t, v); }
  
  // 要素をゼロ・クリア
  void clear() { t = 0; v = 0.0F; }
  
  // FigからDotを得る
  Dot getDot() {
    return new Dot((int)(t * magT + shiftT), -(int)(v * magY + shiftY));
  }
  
  // FigにDotを設定する
  void setDot(Dot dot) {
    t = (int)(((float)dot.e - shiftT) / magT);
    v = -((float)dot.f - shiftY) / magY;
  }

} // Fig

/**
 * ４次元変数を保持するクラス
 * Edge(int l, int r, float b, float t) 
 * ※第３と第４（b,t）の引数の順番に注意
 * l: 左側の数量、または左側の最小値xMin
 * r: 右側の数量、または右側の最大値xMax
 * b: 下側の数量、または下側の最小値yMin
 * t: 上側の数量、または上側の最大値yMax
 */
class Edge {
  int l, r; float b, t;

  // 要素ゼロの生成子
  Edge() { l = 0; r = 0; b = 0F; t = 0F; }
  
  // 所与の要素での生成子
  Edge(int l, int r, float b, float t) {
    this.l = l; this.r = r; this.b = b; this.t = t; }

  // 要素ゼロの代理生成子
  static Edge edge() {
    return new Edge();
  }
  // 所与の要素での代理生成子
  static Edge edge(int l, int r, float b, float t) {
    return new Edge(l, r, b, t);
  }
  
  // 要素をゼロ・クリア
  void clear() { this.l = 0; this.r = 0; this.b = 0F; this.t = 0F; }
  
  final void setEdge(int l, int r, float b, float t) {
    this.l = l; this.r = r; this.b = b; this.t = t;
  }
  
} // Edge

/**
 * ４次元変数を保持するクラス
 * Tetra(float l, float r, float b, float t) {
 * ※第３と第４（b,t）の引数の順番に注意
 * l: 左側の数量、または左側の最小値xMin
 * r: 右側の数量、または右側の最大値xMax
 * b: 下側の数量、または下側の最小値yMin
 * t: 上側の数量、または上側の最大値yMax
 */
class Tetra {
  float l, r, b, t;

  // 要素ゼロの生成子
  Tetra() { l = 0F; r = 0F; b = 0F; t = 0F; }
  
  // 所与の要素での生成子
  Tetra(int l, int r, float b, float t) {
    this.l = l; this.r = r; this.b = b; this.t = t; }

  // 要素ゼロの代理生成子
  static Tetra tetra() {
    return new Tetra();
  }
  // 所与の要素での代理生成子
  static Tetra tetra(int l, int r, float b, float t) {
    return new Tetra(l, r, b, t);
  }
  
  // 要素をゼロ・クリア
  void clear() { l = 0F; r = 0F; b = 0F; t = 0F; }
  
  final void setTetra(float l, float r, float b, float t) {
    this.l = l; this.r = r; this.b = b; this.t = t;
  }
  
} // Tetra

/**
 * ４要素（２要素×２セット）を保持するクラス
 * Quad(float m, float n, float s, float t)
 * m: 第１セット要素１ xMag
 * n: 第１セット要素２ xShift
 * s: 第２セット要素１ yMag
 * t: 第２セット要素２ yShift
 */
class Quad {
  float m, n, s, t;
  
  // 要素ゼロの生成子
  Quad() { m = 0; n = 0; s = 0; t = 0; }
  
  // 所与の要素での生成子
  Quad(float m, float n, float s, float t) {
    this.m = m; this.n = n; this.s = s; this.t = t; }

  // 要素ゼロの代理生成子
  static Quad quad() {
    return new Quad();
  }
  
  // 所与の要素での代理生成子
  static Quad quad(float m, float n, float s, float t) {
    return new Quad(m, n, s, t);
  }
  
  // 要素をゼロ・クリア
  void clear() { m = 0; n = 0; s = 0; t = 0; }
  
} // Quad

// =============================================================================
/**
 * ２次元グラフの画面上での座標（e, f）を示すクラス
 * ※Fig（グラフ用）と対応して用いる
 *  Dot(int e, int f)
 * e: 横軸（左から右への方向が正）の画素数（ピクセル）
 * f: 縦軸（上から下への方向が正）の画素数（ピクセル）
 */
class Dot {
  int e, f;
  
  Edge minMax; // 範囲
  Quad param; // 座標変換パラメータ

  // 要素ゼロの生成子
  Dot() { e = 0; f = 0; }
  
  // 所与の要素での生成子
  Dot(int e, int f) { this.e = e; this.f = f; }
  
  // 要素ゼロの代理生成子
  static Dot dot() { return new Dot(); }
  
  // 所与の要素での代理生成子
  static Dot dot(int x, int y) { return new Dot(x, y); }
  
  // 要素をゼロ・クリア
  void clear() { e = 0; f = 0; }
  
  // DotにFigを設定する
  void setFig(Fig fig) {
    e = (int)(((float)fig.t - fig.shiftT) / fig.magT);
    f = -(int)((fig.v - fig.shiftY) / fig.magY);
  }

} // Dot

/**
 * ４次元変数を保持するクラス
 * Pict(int l, int r, int t, int b)
 * l: 左側の数量、または左端の位置（最小値xMin）
 * r: 右側の数量、または右端の位置（最大値xMax）
 * t: 上側の数量、または上端の位置（最小値yMin）
 * b: 下側の数量、または下端の位置（最大値yMax）
 */
class Pict {
  int l, r, t, b;

  // 要素ゼロの生成子
  Pict() { l = 0; r = 0; t = 0; b = 0; }
  
  // 所与の要素での生成子
  Pict(int l, int r, int t, int b) {
    this.l = l; this.r = r; this.t = t; this.b = b; }

  // 要素ゼロの代理生成子
  static Pict pict() { return new Pict(); }
  
  // 所与の要素での代理生成子
  static Pict pict(int l, int r, int t, int b) {
    return new Pict(l, r, t, b); }
  
  // 要素をゼロ・クリア
  void clear() { this.l = 0; this.r = 0; this.t = 0; this.b = 0; }
  
} // Pict

/**
 * ４次元変数を保持するクラス
 * Rect(int x, int y, int w, int h)
 * x: 画面左から長方形左辺への距離（ピクセル）
 * y: 画面上から長方形上辺への距離（ピクセル）
 * ※(x,y)は長方形の左上の頂点の位置（画面左上を原点とする）
 * w: 長方形の横幅（ピクセル）
 * h: 長方形の高さ（ピクセル）
 */
class Rect {
  int x, y, w, h;

  // 要素ゼロの生成子
  Rect() { x = 0; y = 0; w = 0; h = 0; }
  
  // 所与の要素での生成子
  Rect(int x, int y, int w, int h) {
    this.x = x; this.y = y; this.w = w; this.h = h; }

  // 要素ゼロの代理生成子
  static Rect rect() { return new Rect(); }
  
  // 所与の要素での代理生成子
  static Rect rect(int x, int y, int w, int h) {
    return new Rect(x, y, w, h); }
  
  // 要素をゼロ・クリア
  void clear() { this.x = 0; this.y = 0; this.w = 0; this.h = 0; }
  
} // Rect

// EOF
