/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2021.4.4
 *  Copyright: 2020, 2021
 */
package view;

import java.awt.*;

/**
 * <p> 表　題: SomDrawer</p>
 * <p> 説　明: ＳＯＭ描画</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2007-2021</p>
 * <p> 作成日: 2007.4.12</p>
 * <p> 更新日: 2021.08.28</p>
*/
public class SomDrawer extends Canvas {

  public static final int MAX_INDEX = 1000;//32000;// Data buffer length

  private Image offImage; // Image on the offscreen
  private Graphics offGraphics; // Graphics object
  private Color[] colorPalet; // Color Palet
 
  private static final int X_SPACE = 20; // Ｘ軸余白
  private static final int Y_SPACE = 20;  // Ｙ軸余白

  private float zMax; // Ｕマップ境界等高線
  @SuppressWarnings("FieldMayBeFinal")
  private float zMin;
  
  // SOM
  // 等高線作画
  private static final int MAP_SIZE_X = 400; // マップ高さ(ディフォルト値）
  private static final int MAP_SIZE_Y = 400; // マップ幅(ディフォルト値）
  private int mapSizeX; // マップ高さ
  private int mapSizeY; // マップ幅
  private static final int GRID_X = 40; // マップ縦軸格子数(ディフォルト値）
  private static final int GRID_Y = 40; // マップ横軸格子数(ディフォルト値）
  private int numOfGridX, deltax; // マップ縦軸格子数、縦軸格子幅
  private int numOfGridY, deltay; // マップ横軸格子数、横軸格子幅
  private static final int NUM_OF_CONTOUR = 12; // 等高線の数(ディフォルト値）
  private int maxColor = NUM_OF_CONTOUR + 1; // 色の数(ディフォルト値）
  private int numOfContour; // 等高線の数

  private double [][] f; // SOMビューワに表示されるマップ
  private double [][] out; // 正規化値
  private double fmax, fmin; // 関数最大値、最少値
  private String[][] theLabelMap; // ラベル・マップ
  private boolean evaluateFlag; // 評価マップ表示フラグ　（true：評価マップを表示）
  private boolean borderFlag; // 境界線表示フラグ　（true：境界線を表示）

  // 構築子（変数の初期化）
  @SuppressWarnings("OverridableMethodCallInConstructor")
  public SomDrawer() {
    super(); // GraphDrawerの初期化
    
    // 等高線作画
    mapSizeX = MAP_SIZE_X; // マップ高さ
    mapSizeY = MAP_SIZE_Y; // マップ幅
    setSize(mapSizeY,mapSizeX); // マップサイズ設定
    numOfGridX = GRID_X; // マップ縦軸格子数
    numOfGridY = GRID_Y; // マップ横軸格子数
    numOfContour = NUM_OF_CONTOUR; // 等高線の数
    maxColor = numOfContour + 1;
    
    // Set color palet
    setColorPalet();
  }

  // 関数値（SOMビューワに表示されるマップ）の正規化
  public void dataset( ) {
    fmax=0.0; 
    fmin=1.e+30;
    
    for(int i = 0; i <= numOfGridX; i++) {
      for(int j = 0; j <= numOfGridY; j++) {
        if(f[i][j] >= fmax) fmax = f[i][j];
        if(f[i][j] <= fmin) fmin = f[i][j];
      }
    }
    // 正規化
    for(int i=0; i <= numOfGridX; i++) {
      for(int j=0; j <= numOfGridY; j++) {
        out[i][j] = (f[i][j] - fmin) / (fmax - fmin);
      }
    }
  }

  // -------------------------------------------------------
  // ＡＢ間の交点
  double getDab(int k, double level, double[] d) {
    return (level - d[k+1]) / (d[k] - d[k+1]);
  }
  int getXab(int k, double level, double d, double[] x) {
    return (int)((1.0 - d) * x[k+1] + d * x[k] + 0.5);
  }
  int getYab(int k, double level, double d, double[] y) {
    return (int)((1.0 - d) * y[k+1] + d * y[k] + 0.5);
  }
  
  // ＡＥ間の交点
  double getDae(int k, double level, double[] d) {
    return (level - d[k]) / (d[6] - d[k]);
  }
  int getXae(int k, double level, double d, double[] x) {
    return (int)((1.0 - d) * x[k] + d * x[6] + 0.5);
  }
  int getYae(int k, double level, double d, double[] y) {
    return (int)((1.0 - d) * y[k] + d * y[6] + 0.5);
  }
  
  // ＢＥ間の交点
  double getDbe(int k, double level, double[] d) {
    return (level - d[6]) / (d[k+1] - d[6]);
  }
  int getXbe(int k, double level, double d, double[] x) {
    return (int)((1.0 - d) * x[6] + d * x[k+1] + 0.5);
  }
  int getYbe(int k, double level, double d, double[] y) {
    return (int)((1.0 - d) * y[6] + d * y[k+1] + 0.5);
  }
  
  // -------------------------------------------------------
  // 三角形の塗りつぶし
  void fillDelta(Graphics g, int colorIndex, 
                 double x1, double x2, double x3,
                 double y1, double y2, double y3) {
    Color gc = g.getColor();
    g.setColor(getPaletColor(colorIndex));
    int[] px = {(int)x1, (int)x2, (int)x3, (int)x1};
    int[] py = {(int)y1, (int)y2, (int)y3, (int)y1};
    g.fillPolygon(py, px, 4); // x,y軸を入れ替え
    g.setColor(gc);
  }
  
  // 四角形の塗りつぶし
  void fillQuad(Graphics g, int colorIndex, 
                double x1, double x2, double x3, double x4,
                double y1, double y2, double y3, double y4) {
    Color gc = g.getColor();
    g.setColor(getPaletColor(colorIndex));
    int[] px = {(int)x1, (int)x2, (int)x3, (int)x4, (int)x1};
    int[] py = {(int)y1, (int)y2, (int)y3, (int)y4, (int)y1};
    g.fillPolygon(py, px, 5); // x,y軸を入れ替え
    g.setColor(gc);
  }
  
  // -------------------------------------------------------
  // グラフの描画
  @Override
  public void paint(Graphics g) {
    // 余白の設定
    g.translate(Y_SPACE / 2, X_SPACE / 2);
    // Draw Background
    g.setColor(Color.darkGray);
    g.fillRect(0 , 0, mapSizeY + Y_SPACE * 2, mapSizeX + X_SPACE * 2);
    g.setColor(Color.lightGray);   
    g.translate(Y_SPACE, X_SPACE);
   
    if (f == null) {
      return;
    }
    numOfGridX = f.length - 1; // マップの縦軸格子数
    numOfGridY = f[0].length - 1; // マップの横軸格子数
    
    out = new double [numOfGridX + 1][numOfGridY + 1]; // 正規化値
    deltax = mapSizeX / numOfGridX; // 縦軸格子幅
    if (deltax < 4) deltax = 4;
    deltay = mapSizeY / numOfGridY; // 横軸格子幅
    if (deltay < 4) deltay = 4;
    dataset(); // 関数値の正規化

    // セルの五点の座標値（[0]は使わない）
    double[] x = new double[7];
    double[] y = new double[7];
    // セルの五点の標高値（[0]は使わない）
    double[] d = new double[7];
    
    int i, j; // 格子のインデックス
    int line; //等高線のインデックス
    int k; // 四辺の三角形のインデックス
    int levelIndex; // 三角形の頂点の標高のインデックス（０～７）
    double dab, xab, yab; // 等高線と格子の交点
    double dae, xae, yae;
    double dbe, xbe, ybe;
    double xcent, ycent; // セルの四辺の中央の座標
    double level; //等高線の高さ
    double x1, y1, x2, y2; //セルの四隅の座標
    
    // 等高線の描画
    for(i = 0; i < numOfGridX; i++) {
      for(j = 0; j < numOfGridY; j++) {

        //セルの四隅の座標値
        x1 = i * deltax; x2 = (i + 1) * deltax;
        y1 = j * deltay; y2 = (j + 1) * deltay;
        // セルの四辺の中央の座標値
        xcent = (x1 + x2) / 2.0;
        ycent = (y1 + y2) / 2.0;
        // セルの五点の座標値
        x[1]=x1; x[2]=x2; x[3]=x2; x[4]=x1; x[5]=x[1]; x[6]=xcent;
        y[1]=y1; y[2]=y1; y[3]=y2; y[4]=y2; y[5]=y[1]; y[6]=ycent;
        // セルの五点の標高値
        d[1] = out[i][j]; d[2] = out[i+1][j]; d[3] = out[i+1][j+1];
        d[4] = out[i][j+1]; d[5] = d[1];
        d[6] = (d[1] + d[2] + d[3] + d[4])/4.0;

        // すべての点が同じ値のときはスキップ
          for(k = 1; k <= 4; k++) {
            boolean continuFlag = false;
            
            for(line = 0; line <= numOfContour; line++) { // 等高線の数だけ繰返し
              level = 1. / (double)numOfContour * (double)line; //等高線の高さ

             if (borderFlag) { // 境界線表示フラグ　（true：境界線を表示）
                level = zMax; // Ｕマップ境界等高線
             }
              
              levelIndex = 0; // 全ての頂点は等高線よりも低い
              // 三角形の頂点の高さをチェックして、インデックスを設定する。
              if (d[k] > level) { // 点Ａ＞標高
                levelIndex = levelIndex + 4; // 二進数で０１００
              }
              if (d[k+1] > level) { // 点Ｂ＞標高
                levelIndex = levelIndex + 2; // 二進数で００１０
              }
              if (d[6] > level) { // 点Ｅ＞標高
                levelIndex = levelIndex + 1; // 二進数で０００１
              }
              
              // インデックスに応じて塗りつぶしと等高線を描画
              switch (levelIndex) {
                case 0:
                  continuFlag = true; // lineループを抜ける。
                  break;
                case 1:
                  dae = getDae(k, level, d);
                  xae = getXae(k, level, dae, x);
                  yae = getYae(k, level, dae, y);                 
                  dbe = getDbe(k, level, d);
                  xbe = getXbe(k, level, dbe, x);
                  ybe = getYbe(k, level, dbe, y);                 
                  fillDelta(g, line, x[6], xbe, xae, y[6], ybe, yae);
                  g.drawLine((int)ybe, (int)xbe, (int)yae, (int)xae); // 上下を反転させない
                  break;
                case 2:
                  dab = getDab(k, level, d);
                  xab = getXab(k, level, dab, x);
                  yab = getYab(k, level, dab, y);
                  dbe = getDbe(k, level, d);
                  xbe = getXbe(k, level, dbe, x);
                  ybe = getYbe(k, level, dbe, y);                 
                  fillDelta(g, line, x[k+1], xab, xbe, y[k+1], yab, ybe);
                  g.drawLine((int)yab, (int)xab, (int)ybe, (int)xbe); // 上下を反転させない
                  break;
                case 3:
                  dae = getDae(k, level, d);
                  xae = getXae(k, level, dae, x);
                  yae = getYae(k, level, dae, y);                 
                  dab = getDab(k, level, d);
                  xab = getXab(k, level, dab, x);
                  yab = getYab(k, level, dab, y);
                  fillQuad(g, line, x[k+1], x[6], xae, xab, y[k+1], y[6], yae, yab);
                  g.drawLine((int)yae, (int)xae, (int)yab, (int)xab); // 上下を反転させない
                  break;
                case 4:
                  dab = getDab(k, level, d);
                  xab = getXab(k, level, dab, x);
                  yab = getYab(k, level, dab, y);
                  dae = getDae(k, level, d);
                  xae = getXae(k, level, dae, x);
                  yae = getYae(k, level, dae, y);                 
                  fillDelta(g, line, x[k], xab, xae, y[k], yab, yae);
                  g.drawLine((int)yab, (int)xab, (int)yae, (int)xae); // 上下を反転さない
                  break;
                case 5:
                  dab = getDab(k, level, d);
                  xab = getXab(k, level, dab, x);
                  yab = getYab(k, level, dab, y);
                  dbe = getDbe(k, level, d);
                  xbe = getXbe(k, level, dbe, x);
                  ybe = getYbe(k, level, dbe, y);                 
                  fillQuad(g, line, x[6], x[k], xab, xbe, y[6], y[k], yab, ybe);
                  g.drawLine((int)yab, (int)xab, (int)ybe, (int)xbe); // 上下を反転さない
                  break;
                case 6:
                  dae = getDae(k, level, d);
                  xae = getXae(k, level, dae, x);
                  yae = getYae(k, level, dae, y);                 
                  dbe = getDbe(k, level, d);
                  xbe = getXbe(k, level, dbe, x);
                  ybe = getYbe(k, level, dbe, y);                 
                  fillQuad(g, line, x[k], x[k+1], xbe, xae, y[k], y[k+1], ybe, yae);
                  g.drawLine((int)ybe, (int)xbe, (int)yae, (int)xae); // 上下を反転さない
                  break;
                case 7:
                  fillDelta(g, line, x[k], x[k+1], x[6], y[k], y[k+1], y[6]);
                  break;
                default:
                  break;
              }
              
              if (continuFlag) {
                break; // lineループを抜ける。
              }
            
            } // lineループ（等高線の数の分だけ）
          } //kループ

      }  // jループ（Y方向）
    } // iループ（X方向）
    
    // ラベルを表示する。
    Color sc = g.getColor();
    g.setColor(Color.red);
    for(i = 0; i < numOfGridX; i=i+2) {
      for(j = 0; j < numOfGridY; j=j+2) {
        String ls = theLabelMap[i/2][j/2];
        if (ls != null) {
          int fontHight = g.getFontMetrics().getHeight();
          g.drawString(ls, j * deltay, i * deltax + fontHight/2); // グラフを上下反転させない
        }
      }  // jループ（Y方向）
    } // iループ（X方向）
    g.setColor(sc);
  }

  // 表示を更新する
  @Override
	public void update(Graphics g) {
		Dimension canvasSize = this.getSize();
		offImage = createImage(canvasSize.width, canvasSize.height);
		offGraphics = offImage.getGraphics();

		if (offImage != null) {
			paint(offGraphics);
			g.drawImage(offImage, 0, 0, null);
		}
	}

  // ---------------------------------------------------------------------------
  
  void showLabel(Graphics g, String[][] aLabelMap) {
    Color sc = g.getColor();
    g.setColor(Color.red);
    for(int i = 0; i < numOfGridX; i = i + 2) {
      for(int j = 0; j < numOfGridY; j = j + 2) {
        String ls = aLabelMap[i / 2][j / 2];
        if (ls != null) {
          g.drawString(ls, (numOfGridY - j) * deltay, i * deltax); // 上下を反転
        }
      }  // jループ（Y方向）
    } // iループ（X方向）
    g.setColor(sc);
  }
  
  private void setColorPalet() {
    maxColor = numOfContour + 1;
    colorPalet = new Color[maxColor];
    float aHue, aMag, aDim;
    for (int i = maxColor-1; i >= 0; i--) {
      aHue = 2.0F - (2.0F * (float)i / (float)maxColor);
      if (aHue < 1.0F) {
        if (aHue > 0.8F) {
          aMag = 0.7F;
          aDim = 0.5F;
        }
        else {
          aMag = 1.0F;
          aDim = 1.0F;
        }
      }
      else {
        aHue = aHue - 1.0F;
        if (aHue < 0.4F) {
          aMag = 0.7F;
          aDim = 0.5F;
        }
        else if (aHue < 0.7F) {
          aMag = 0.6F;
          aDim = 0.4F;
        }
        else if (aHue < 0.95F) {
          aMag = 0.55F;
          aDim = 0.35F;
        }
        else {
          aMag = 0.4F;
          aDim = 0.25F;
        }
      }
      colorPalet[i] = Color.getHSBColor(aHue, aMag, aDim);
    }
  }

  private Color getPaletColor(int id) {
    return colorPalet[id];
  }
  
  public Color[] getColorPalet() {
    return colorPalet;
  }
  
  // ---------------------------------------------------------------------------
  // Ｙ軸のビット位置を得る
  int getY(double yValue) {
    // Y軸座標を反転させる
//    return (int)yValue; // 上下を反転させない　2019.1.4
    return mapSizeX - (int)yValue; // 上下を反転させる
  }
  
  // ---------------------------------------------------------------------------
  // X Axis
  // ＵマップＸ軸幅
  public void setUMapXSize(float uMapXSize) {
//    mapWidth = (int)uMapXSize;
    mapSizeX = (int)uMapXSize;
  }
  public float getUMapXSize() {
//    return (float)mapWidth;
    return (float)mapSizeX ;
  }

  // Y Axis
  // ＵマップＹ軸幅
  public void setUMapYSize(float uMapYSize) {
//    mapHight = (int)uMapYSize;
    mapSizeY = (int)uMapYSize;
///    repaint();
  }
  public float getUMapYSize() {
//    return (float)mapHight;
    return (float)mapSizeY;
  }

  // Ｕマップ境界等高線
  public void setZMax(float newMax) {
    zMax = newMax;
///    repaint();
  }
  public float getZMax() {
    return zMax;
  }

  public void setZMin(float newMin) {
    zMin = newMin;
///    repaint();
  }
  public float getZMin() {
    return zMin;
  }

  // 等高線数
  public void setNumOfContour(float numOfContour) {
    this.numOfContour = (int)numOfContour;
    setColorPalet();
///    repaint();
  }
  public float getNumOfContour() {
    return (float)numOfContour;
  }

  // ---------------------------------------------------------------------------
  // 評価マップ表示フラグ　（true：評価マップを表示）
  public boolean getEvaluateFlag() {
    return evaluateFlag;
  }
  public void setEvaluateFlag(boolean evaluateFlag) {
    this.evaluateFlag = evaluateFlag;
  }
  
  // ---------------------------------------------------------------------------
  // 境界線表示フラグ　（true：境界線を表示）
  public boolean getBorderFlag() {
    return borderFlag;
  }
  public void setBorderFlag(boolean borderFlag) {
    this.borderFlag = borderFlag;
  }
  
  // ---------------------------------------------------------------------------
  // 参照ベクタを関数（マップ）に設定する
  public void setCodeMap(double[][] newMapVector) {
    f = newMapVector;
  }
  
  // Ｕマップを関数（マップ）に設定する
  public void setUMap(double[][] newUMatrix) {
    f = newUMatrix;
  }
  
  // ラベル・マップをマップに設定する
  public void setLabelMap(String[][] newLabelMatrix) {
    theLabelMap = newLabelMatrix;
  }
  
} // SomDrawer

// EOF
