/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2021.4.4
 *  Copyright: 2020, 2021
 */
package view;

import java.awt.*;
import java.awt.event.*;
import java.util.ArrayList;
import java.util.StringTokenizer;
import javax.swing.JPanel;
//import javax.swing.JScrollPane;

import util.DJ;
import util.LogEditor;

/**
 * <p> 表　題: SomViewer</p>
 * <p> 説　明: ＳＯＭ表示用ビューワー</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2007-2021</p>
 * <p> 作成日: 2007.4.12</p>
 * <p> 更新日: 2021.08.28</p>
 */

/** ＳＯＭ表示用ビューワー */
//public class SomViewer extends javax.swing.JFrame {
public class SomViewer extends Viewer {

  public static final String BOUNDS_NAME = "somBounds"; // 表示位置と大きさ
  // ＳＯＭタイプ
  public static final int SOM_GENERAL = 0; // 汎用
  public static final int SOM_ANIMAL = 1; // 動物
  public static final int SOM_PLANT = 2; // 植物
  public static final int SOM_LETTER = 3; // 文字
  public static final int SOM_VOICE = 4; // 音声

  public static final String VIEWER_TITLE = "SOM viewer";
  StringBuffer comment; // コメント表示用バッファ
  boolean packFrame = false;

  SomDrawer graph = new SomDrawer();
  float[] dataValue;
  String[] nodeNames;
  static final Point DEFAULT_LOCATION = new Point(595,10); // 暫定位置
  static final Dimension DEFAULT_DIMENSION = new Dimension(470,640); // 暫定領域
  
  static final int MAX_DATA = 16;//8; // 最大グラフ項目数 //#050
  static final int MAX_CONTOUR = 100; // 最大等高線数 20190202
  // 評価方法の選択肢
  static final int TYPE_NOTHING = 0; // 何もしない 20190208
  static final int TYPE_RESIDUE = 1; // 剰余 20190208
  static final int TYPE_QUOTIENT = 2; // 商 20190208
  static final int TYPE_LABEL = 3; // ラベルで判定
//  int maxData; // Number of data
  
  // SOM
  String[] factorNames; // 因子名配列
  int numOfFactor; // 因子ベクタの総数
  double[][] factorVectors; // 因子ベクタ配列[因子ベクタの総数][因子数]
  String[] factorLabels; // 因子ラベル配列[因子ベクタの総数]
  double[][] codeVectorArray; // 参照ベクタ配列[参照ベクタの個数][因子数]
  int dimOfFactor; // 因子ベクタ（参照ベクタ）の次数
  int dimOfCodeX; // Ｘ軸格子数
  int dimOfCodeY; // Ｙ軸格子数
  int numOfCodeVector; // 参照ベクタの個数[Ｘ軸格子数＊Ｙ軸格子数]
  double[][] elemetMap; // 要素マップ
  double learningCount; // 学習回数カウント値
  double[][] uMatrix; // Ｕマップ（Ｕ行列） 
  int dimOfUX; // ＵマップのＸ軸格子数
  int dimOfUY; // ＵマップのＸ軸格子数
  boolean umapFlag; // Ｕマップ選択フラグ
  boolean elementFlag; // 因子マップ選択フラグ
  boolean evaluateFlag; // 評価マップ表示フラグ　（true：評価マップを表示）
  boolean borderFlag; // 境界線表示フラグ　（true：境界線を表示）
  String[][] labelMap; // ラベル・マップ（ラベル行列）
  String elementName; // 要素変数名
  int elementNo; // 要素変数番号
  Color buttonColor; // ボタンのディフォルト背景色
  
  // プロパティ（デフォルト値）
  int numOfTrial = 0; // 試行因子ベクタ数
  int evaluationType = 0; // 評価方法
  int numOfContour = 10; // Ｕマップ等高線数
  int baseContour = 10; // Ｕマップ基準等高線
  double borderContour = 0.2; // Ｕマップ境界等高線
  int uMapXSize = 400; // ＵマップＸ軸表示幅[ドット]
  int uMapYSize = 400; // ＵマップＹ軸表示幅[ドット]
  
  SomGraphViewer_button_actionAdapter buttonAdapter
    = new SomGraphViewer_button_actionAdapter(this);
  SomGraphViewer_setButton_actionAdapter setButtonAdapter
    = new SomGraphViewer_setButton_actionAdapter(this);
  SomGraphViewer_settingTextField_actionAdapter settingTextFieldAdapter
    = new SomGraphViewer_settingTextField_actionAdapter(this);
  SomGraphViewer_rangeTextField_actionAdapter rangeTextFieldAdapter
  = new SomGraphViewer_rangeTextField_actionAdapter(this);

  BorderLayout graphViewerBorderLayout = new BorderLayout();
  Panel mainPanel = new Panel();
  BorderLayout mainPanelBorderLayout = new BorderLayout();
  
//#  JScrollPane graphScrollPane = new javax.swing.JScrollPane();

  Panel graphPanel = new Panel();
  BorderLayout graphBorderLayout = new BorderLayout();
  Panel graphTitlePanel = new Panel();
  Label graphTitileLabel = new Label();
  BorderLayout graphTitlePanelBorderLayout = new BorderLayout();

  JPanel graphsubPanel = new JPanel();
  BorderLayout graphsubBorderLayout = new BorderLayout();
  
  // Operation Panel
  Panel operatePanel = new Panel();
  BorderLayout operatePanelBorderLayout = new BorderLayout();
  // Mode Panel
  Panel modePanel = new Panel(); // モード選択パネル
  GridLayout modePanelGridLayout = new GridLayout();
  // Control Panel
  Panel contorlPanel = new Panel();
  BorderLayout controlBorderLayout = new BorderLayout();
  // Color Palet
  ColorPaletPanel colorPaletPanel = new ColorPaletPanel(graph); // カラーパレットパネル
  // Setting Panel
  Panel settingPanel = new Panel();
  GridLayout settingPanelGridLayout = new GridLayout();
  // Title Panel
  Label modeLabel = new Label(); // モード選択
  Button umapButton = new Button(); // Ｕマップ選択ボタン
  Button elementButton = new Button(); // 要素変数マップ選択ボタン
  Panel valuePanel = new Panel();
  GridLayout valueGridLayout = new GridLayout();
  Label valueLabel = new Label();
  TextField elementNameTextField = new TextField(); // 要素変数名
  TextField onValueTextField = new TextField();
  TextField offValueTextField = new TextField();  
  // Time Panel
  Panel timePanel = new Panel();
  GridLayout timeGridLayout = new GridLayout();
  Button firstButton = new Button(); // 先頭要素選択ボタン
  Button backButton = new Button(); // 前の要素選択ボタン
  Button nextButton = new Button(); // 次の要素選択ボタン
  Button lastButton = new Button(); // 末尾要素選択ボタン
  Panel trialPanel = new Panel(); // 学習回数パネル
  GridLayout trialGridLayout = new GridLayout();
  Label trialLabel = new Label(); // 学習回数ラベル
  TextField trialTextField = new TextField(); // 学習回数
  // マップ設定パネル
  Panel mapSetPanel = new Panel();
  GridLayout mapGridLayout = new GridLayout();
  // 等高線設定パネル
  Label contourLabel = new Label(); // 等高線ラベル
  Panel contourSetPanel = new Panel(); // 等高線設定パネル
  GridLayout gridLayout0 = new GridLayout();
  // RangePanel
  Panel rangePanel = new Panel();
  GridLayout rengPanelGridLayout = new GridLayout();
  // SetRangPanel
  Panel setRangPanel = new Panel();
  GridLayout rangGridLayout = new GridLayout();
  Button evaluateButton = new Button(); // 評価表示ボタン
  Button borderButton = new Button(); // 境界線表示ボタン
  Button setButton = new Button();
  Button saveButton = new Button(); // 参照ベクタ保存ボタン
  // XRangePanel
  Panel xRangePanel = new Panel(); // 因子表示パネル
  GridLayout gridLayout1 = new GridLayout();
  Label xLabel = new Label();
  TextField uMapXSizeTextField = new TextField();
  Label typeLabel = new Label();
  TextField evaluationTypeTextField = new TextField();
  // YRangePanel
  Panel yRangePanel = new Panel(); // 因子選択パネル
  GridLayout gridLayout2 = new GridLayout();
  Label yLabel = new Label();
  TextField uMapYSizeTextField = new TextField();
  Label baseLabel = new Label(); // 評価基準ラベル
  TextField baseTextField = new TextField(); // 評価基準
  // ZRangePanel
  Panel borderSetPanel = new Panel();
  GridLayout gridLayout3 = new GridLayout();
  Label borderLabel = new Label(); // 境界線ラベル
  Panel borderPanel = new Panel(); // 境界線高さ増減パネル
  GridLayout borderGridLayout = new GridLayout();
  
  Button upBorderButton = new Button(); // 境界線高さ増加ボタン
  Button downBorderButton = new Button(); // 境界線高さ低減ボタン
  TextField zMinTextField = new TextField();
  TextField borderContourTextField = new TextField(); // Ｕマップ境界等高線
  Panel contourPanel = new Panel(); // 等高線数増減パネル
  GridLayout contourGridLayout = new GridLayout();
  Button upContourButton = new Button(); // 等高線数追加ボタン
  Button downContourButton = new Button(); // 等高線数削減ボタン
  TextField numOfContourTextField = new TextField(); // 等高線数

  Panel baseSetPanel = new Panel(); // 評価基準設定パネル
  GridLayout baseGridLayout = new GridLayout();

  Panel typeSetPanel = new Panel(); // 評価方法設定パネル
  GridLayout typeGridLayout = new GridLayout();
  
  /** SomViewerを生成する */
  public SomViewer() {
    comment = new StringBuffer(); // コメント表示用バッファ
    comment.append("Data : "); // + anInterfaceCell.getName());
    initialize();
    setBounds();
  }
  
  /**
   * ビューワにコメントを追加する
   * @param comment 
   */
  public void appendComment(String comment) {
    this.comment.append(comment); // コメントを追加
    graphTitileLabel.setText(this.comment.toString()); // コメントを表示
  }
  
  /**
   * ビューワにコメントを設定する
   * @param comment String // コメント
   */
  public void setComment(String comment) {
    graphTitileLabel.setText(comment); // コメントを表示
  }
  
  /** 表示用フレームの領域を設定 */
  public final void setBounds() {
    PropertyViewer propertyViewer = DJ.getPropertyViewer();
    Rectangle rect = propertyViewer.getBoundsValue(BOUNDS_NAME);
    if (rect != null) {
      setBounds(rect);
    }
    else { // ディフォルト値を設定
      setLocation(DEFAULT_LOCATION); // 暫定位置
      setSize(DEFAULT_DIMENSION); // 暫定領域
    }
  }
  
  /** 初期化 */
  private void initialize() {
//$    enableEvents(AWTEvent.WINDOW_EVENT_MASK);
    try {
      initComponents();
    }
    catch(Exception e) {
    }
    
    elementName = "Item Name"; // 要素変数名
    elementNo = 0; // 要素番号
    umapFlag = true; // Ｕマップ選択フラグ
    elementFlag = false; // 因子マップ選択フラグ
    evaluateFlag = false; // 評価マップ表示フラグ（true：評価マップを表示）
    borderFlag = false; // 境界線表示フラグ　（true：境界線を表示）
    buttonColor = borderButton.getBackground(); // ボタンのディフォルト背景色
  }

  /** コンポーネントの初期化 */
  private void initComponents() throws Exception  {
    this.setIconImage(Console.icon16);
    
    this.setTitle("DaiJa : " + VIEWER_TITLE);

    // Set Frame Size and Location
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension frameSize = this.getSize();
    if (frameSize.height > screenSize.height) frameSize.height = screenSize.height;
    if (frameSize.width > screenSize.width) frameSize.width = screenSize.width;

    // Set Layouts
    this.setLayout(graphViewerBorderLayout);
    mainPanel.setLayout(mainPanelBorderLayout);
    mainPanelBorderLayout.setVgap(5);

//    ScrollPaneLayout graphScrollPaneLayout = new ScrollPaneLayout(){
//      public void layoutContainer() {
//         super.layoutContainer(graphPanel);
//         // do some extra work here ...
//      }
//    };
//    graphScrollPane.setLayout(graphScrollPaneLayout);
    
    // Add Compornents
    graphPanel.setLayout(graphBorderLayout);
    graphTitileLabel.setFont(new Font("Dialog", 1, 14));
    graphTitileLabel.setText(comment.toString());
    graphTitlePanel.setLayout(graphTitlePanelBorderLayout);
    operatePanel.setLayout(operatePanelBorderLayout);
    operatePanelBorderLayout.setHgap(5);

    evaluateButton.setLabel("Evaluate"); // 評価表示ボタン
    evaluateButton.addActionListener(setButtonAdapter);
    borderButton.setLabel("Border"); // 境界線表示ボタン
    borderButton.addActionListener(setButtonAdapter);
    
    saveButton.setLabel("Save SOM"); // 参照ベクタ保存ボタン
    saveButton.addActionListener(setButtonAdapter);

    setButton.setLabel("SET");
    setButton.addActionListener(setButtonAdapter);
    
    rangePanel.setLayout(rengPanelGridLayout); //#043
    evaluationTypeTextField.setText("1"); //#043
    evaluationTypeTextField.addActionListener(rangeTextFieldAdapter);
    uMapXSizeTextField.setText("400"); //#043
    uMapXSizeTextField.addActionListener(rangeTextFieldAdapter);
    
    baseTextField.setText("16"); // 評価基準
    baseTextField.addActionListener(rangeTextFieldAdapter);
    uMapYSizeTextField.setText("400"); //#043
    uMapYSizeTextField.addActionListener(rangeTextFieldAdapter);
    
    zMinTextField.setText("0.0"); //#043
    zMinTextField.addActionListener(rangeTextFieldAdapter);
    borderContourTextField.setText("0.5"); // Ｕマップ境界等高線
    borderContourTextField.addActionListener(rangeTextFieldAdapter);
    
    contourLabel.setAlignment(Label.CENTER);
    contourLabel.setText("Contour"); // 等高線ラベル
    upContourButton.setLabel("▲"); // 等高線数追加ボタン
    upContourButton.addActionListener(setButtonAdapter);
    downContourButton.setLabel("▼"); // 等高線数削減ボタン
    downContourButton.addActionListener(setButtonAdapter);
    
    numOfContourTextField.setText("15"); // 等高線数
    numOfContourTextField.addActionListener(rangeTextFieldAdapter);
    
    xLabel.setAlignment(Label.CENTER);
    xLabel.setText("X"); //#043
    yLabel.setAlignment(Label.CENTER);
    yLabel.setText("Y"); //#043
    typeLabel.setAlignment(Label.CENTER);
    typeLabel.setText("Type"); //#043
    baseLabel.setAlignment(Label.CENTER);
    baseLabel.setText("Base"); //#043
    
    borderLabel.setAlignment(Label.CENTER);
    borderLabel.setText("Border"); // 境界線ラベル
    upBorderButton.setLabel("△"); // 界線高さ増加ボタン
    upBorderButton.addActionListener(setButtonAdapter);
    downBorderButton.setLabel("▽"); // 境界線高さ低減ボタン
    downBorderButton.addActionListener(setButtonAdapter);
    
    rengPanelGridLayout.setColumns(1);
    rengPanelGridLayout.setRows(5);
    rengPanelGridLayout.setVgap(3);
    contourSetPanel.setLayout(gridLayout0); // 等高線設定パネル
    xRangePanel.setLayout(gridLayout1); // 因子表示パネル
    yRangePanel.setLayout(gridLayout2); // 因子選択パネル
    borderSetPanel.setLayout(gridLayout3);
    setRangPanel.setLayout(rangGridLayout);
    borderPanel.setLayout(borderGridLayout); // 境界線増減パネル
    contourPanel.setLayout(contourGridLayout); // 等高線数増減パネル
    baseSetPanel.setLayout(baseGridLayout); // 評価基準設定パネル
    typeSetPanel.setLayout(typeGridLayout); // 評価方法設定パネル

    modePanel.setLayout(modePanelGridLayout); // モード選択パネル
    modePanelGridLayout.setColumns(1);
    modePanelGridLayout.setRows(4);
    contorlPanel.setLayout(controlBorderLayout);
    modeLabel.setText("Mode"); // モード選択
    umapButton.setLabel("U-Map"); // Ｕマップ選択ボタン
    umapButton.addActionListener(buttonAdapter);
    elementButton.setLabel("Element"); // 要素変数マップ選択ボタン
    elementButton.addActionListener(buttonAdapter);
    
    settingPanelGridLayout.setColumns(1);
    settingPanelGridLayout.setRows(4);
    timePanel.setLayout(timeGridLayout);
    timeGridLayout.setColumns(4);
    firstButton.setLabel("|<<"); // 先頭要素選択ボタン
    firstButton.addActionListener(buttonAdapter);
    backButton.setLabel("<"); // 前の要素選択ボタン
    backButton.addActionListener(buttonAdapter);
    nextButton.setLabel(">"); // 後の要素選択ボタン
    nextButton.addActionListener(buttonAdapter);
    lastButton.setLabel(">>|"); // 末尾要素選択ボタン
    lastButton.addActionListener(buttonAdapter);
    // Value Panel
    valuePanel.setLayout(valueGridLayout);
    valueGridLayout.setColumns(4);
    valueLabel.setAlignment(Label.CENTER);
    valueLabel.setText("Element");
    elementNameTextField.setText("Name"); // 要素変数名
    offValueTextField.setEditable(false);
    offValueTextField.setText("1");
    onValueTextField.setText("0");
    onValueTextField.addActionListener(settingTextFieldAdapter);
    trialPanel.setLayout(trialGridLayout); // 学習回数パネル
    trialGridLayout.setColumns(2);
    trialLabel.setAlignment(Label.CENTER);
    trialLabel.setText("Trial"); // 学習回数ラベル
    trialTextField.setText("50"); // 学習回数
    trialTextField.addActionListener(settingTextFieldAdapter);
    
    // マップ設定パネル
    mapSetPanel.setLayout(mapGridLayout); // マップ設定パネル
    mapGridLayout.setColumns(4);
    mapSetPanel.add(xLabel, null);
    mapSetPanel.add(uMapXSizeTextField, null);
    mapSetPanel.add(yLabel, null);
    mapSetPanel.add(uMapYSizeTextField, null);
    
   this.add(mainPanel,java.awt.BorderLayout.CENTER);
    
    
   // ########### スクロールバーの追加 #####################################
    
    mainPanel.add(graphTitlePanel, java.awt.BorderLayout.NORTH);
    mainPanel.add(graphPanel, java.awt.BorderLayout.CENTER);
    mainPanel.add(operatePanel, java.awt.BorderLayout.SOUTH);
    
    graphTitlePanel.add(graphTitileLabel, java.awt.BorderLayout.WEST);

//    graphPanel.add(graphScrollPane, java.awt.BorderLayout.CENTER);
//    graphScrollPane.setHorizontalScrollBarPolicy(javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
//    graphScrollPane.setVerticalScrollBarPolicy(javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
//    graphScrollPane.setAutoscrolls(true);
//    graphScrollPane.setPreferredSize(new java.awt.Dimension(400, 400));
////    graphScrollPane.getViewport().add(graph);
//    graphScrollPane.getViewport().add(graphsubPanel);
//    graphScrollPane.setLayout(new javax.swing.ScrollPaneLayout()); // スクロールに追従
            
//    graphsubPanel.add(graph, java.awt.BorderLayout.CENTER);

    graphPanel.add(graph, java.awt.BorderLayout.CENTER);
    
    operatePanel.add(modePanel,  BorderLayout.WEST); // モード選択パネル
    operatePanel.add(contorlPanel,  BorderLayout.CENTER);

    xRangePanel.add(valueLabel, null); // 因子表示ラベル
    xRangePanel.add(elementNameTextField, null); // 因子名
    xRangePanel.add(onValueTextField, null); // 因子インデックス
    xRangePanel.add(offValueTextField, null); // 因子数

    yRangePanel.add(firstButton, null); // 先頭要素選択ボタン
    yRangePanel.add(backButton, null); // 前の要素選択ボタン
    yRangePanel.add(nextButton, null); // 後の要素選択ボタン
    yRangePanel.add(lastButton, null); // 末尾要素選択ボタン

    contourSetPanel.add(contourLabel, null); // 等高線ラベル
    contourSetPanel.add(numOfContourTextField, null); // 等高線数
    contourSetPanel.add(contourPanel, null); // 等高線数増減パネル
    contourPanel.add(upContourButton, null); // 等高線数追加ボタン
    contourPanel.add(downContourButton, null); // 等高線数削減ボタン
    
    contourSetPanel.add(baseSetPanel, null); // 等高線数増減パネル
    baseSetPanel.add(baseLabel, null); // 評価基準ラベル
    baseSetPanel.add(baseTextField, null); // 評価基準
    
    borderSetPanel.add(borderLabel, null); // 境界線ラベル
    borderSetPanel.add(borderContourTextField, null); // Ｕマップ境界等高線
    borderSetPanel.add(borderPanel, null); // 境界線高さ増減パネル
    borderPanel.add(upBorderButton, null); // 境界線高さ増加ボタン
    borderPanel.add(downBorderButton, null); // 境界線高さ低減ボタン
    
    borderSetPanel.add(typeSetPanel, null); // 等高線数増減パネル
    typeSetPanel.add(typeLabel, null); // 評価方法ラベル
    typeSetPanel.add(evaluationTypeTextField, null); // 評価方法
    
    rangePanel.add(setRangPanel, null); // 試行回数、マップ設定パネル
    rangePanel.add(contourSetPanel, null); // 等高線設定パネル
    rangePanel.add(xRangePanel, null); // 因子表示パネル
    rangePanel.add(yRangePanel, null); // 因子選択パネル
    rangePanel.add(borderSetPanel, null); // 境界線設定ボタン

    operatePanel.add(rangePanel, java.awt.BorderLayout.EAST);
    modePanel.add(umapButton, null); // Ｕマップ選択ボタン
    modePanel.add(elementButton, null); // 要素変数選択ボタン
    modePanel.add(evaluateButton, null); // 評価表示ボタン
    modePanel.add(borderButton, null); // 境界線表示ボタン

    contorlPanel.add(colorPaletPanel, java.awt.BorderLayout.CENTER); // カラーパレット
    contorlPanel.add(saveButton, java.awt.BorderLayout.SOUTH); // 参照ベクタ保存ボタン
    
    setRangPanel.add(trialPanel, null);
    trialPanel.add(trialLabel, null); // 学習回数ラベル
    trialPanel.add(trialTextField, null); // 学習回数
    setRangPanel.add(mapSetPanel, null); // 
    
    // 設定パネルの大きさを与える
//    Dimension rangeDim = rangePanel.getSize();
    Dimension rangeDim = new Dimension(260, 120);
    rangePanel.setPreferredSize(rangeDim);
    
    if (packFrame) this.pack();
    else this.validate();
}

  /**
   * プロパティを設定
   * @param numOfTrial int // 試行因子ベクタ数
   * @param evaluationType int // 評価方法
   * @param numOfContour int // Ｕマップ等高線数
   * @param baseContour int // Ｕマップ基準等高線（評価基準）
   * @param borderContour double // Ｕマップ境界等高線
   * @param uMapXSize int // ＵマップＸ軸幅
   * @param uMapYSize int // ＵマップＹ軸幅
   */
  public void setProperty(int numOfTrial, int evaluationType,
          int numOfContour,int baseContour, double borderContour, 
          int uMapXSize, int uMapYSize) {
    
    this.numOfTrial = numOfTrial; // 試行因子ベクタ数
    this.evaluationType = evaluationType; // 評価方法
    this.numOfContour = numOfContour; // Ｕマップ等高線数
    this.baseContour = baseContour; // Ｕマップ基準等高線
    this.borderContour = borderContour; // Ｕマップ境界等高線
    this.uMapXSize = uMapXSize; // ＵマップＸ軸表示幅[ドット]
    this.uMapYSize = uMapYSize; // ＵマップＹ軸表示幅[ドット]
    
    setEvaluationType(String.valueOf(evaluationType)); // 評価方法
    setNumOfContour(String.valueOf(numOfContour)); // Ｕマップ等高線数
    setBaseContour(String.valueOf(baseContour)); // Ｕマップ基準等高線
    setBorderContour(String.valueOf(borderContour)); // Ｕマップ境界等高線
    setUMapXSize(String.valueOf(uMapXSize)); // ＵマップＸ軸幅
    setUMapYSize(String.valueOf(uMapYSize)); // ＵマップＹ軸幅
  }
  
  /**
   * パラメータを設定
   * @param dimOfFactor int // 参照ベクタの次数（因子ベクタの次数）
   * @param dimOfCodeX int // 参照ベクタ配列のＸ軸格子数
   * @param dimOfCodeY  int // 参照ベクタ配列のＹ軸格子数
   * @param factorNames String[] // 因子名配列
   * @param numOfFactor int // 因子ベクタの総数
   * @param factorVectors double[][] // 因子ベクタ配列[因子ベクタの総数][因子数]
   * @param factorLabels String[] // 因子ラベル配列
   */
  public void setParameter(int dimOfFactor, int dimOfCodeX, int dimOfCodeY,
          String[] factorNames,int numOfFactor, double[][] factorVectors, 
          String[] factorLabels) {
    this.factorNames = factorNames; // 因子名配列
    this.dimOfFactor = dimOfFactor; // 参照ベクタの次数
    this.dimOfCodeX = dimOfCodeX; // Ｘ軸格子数
    this.dimOfCodeY = dimOfCodeY; // Ｙ軸格子数
    this.numOfFactor = numOfFactor; //  因子ベクタの総数
    this.factorVectors = factorVectors; // 因子ベクタ配列
    this.factorLabels = factorLabels; // 因子ラベル配列
    
    numOfCodeVector = dimOfCodeX * dimOfCodeY; // 参照ベクタの個数[Ｘ軸格子数＊Ｙ軸格子数]
//    dimOfUX = dimOfCodeX * 2 + 1; // ＵマップのＸ軸格子数
//    dimOfUY = dimOfCodeY * 2 + 1; // ＵマップのＸ軸格子数
    dimOfUX = dimOfCodeX * 2; // ＵマップのＸ軸格子数
    dimOfUY = dimOfCodeY * 2; // ＵマップのＸ軸格子数
    
    elemetMap = new double[dimOfUX][dimOfUY]; // 要素マップ
  }
  
  /**
   * マップへの参照を設定する
   * @param codeVectorArray double[][] // 参照ベクタ配列
   * @param uMatrix double[][] // Ｕマップ（Ｕ行列） 
   * @param labelMap  String[][] // ラベル・マップ（ラベル行列）
   */
  public void setMap(double[][] codeVectorArray,
               double[][] uMatrix, String[][] labelMap) {
    this.codeVectorArray = codeVectorArray;
    this.uMatrix = uMatrix;
    this.labelMap = labelMap;
  }

  // ボタン操作が行われたらマップを更新
  void updateMap() {
    // Ｕマップを設定
    if (umapFlag) { // Ｕマップ選択フラグ
      graph.setUMap(uMatrix); // Ｕマップ（Ｕ行列）        
    } // Ｕマップ
    if (elementFlag) { // 因子マップ選択フラグ
      compositeElementMap(); // 因子マップを生成

      graph.setCodeMap(elemetMap); // 因子マップを設定
      // 参照ベクタの次数を表示      
      offValueTextField.setText(String.valueOf(dimOfFactor));
      // 因子のインデックスと因子名を表示
      onValueTextField.setText(String.valueOf(elementNo));
      elementName = factorNames[elementNo]; // 因子名配列
      elementNameTextField.setText(elementName); // 因子名
    } // 因子マップ
    else if (evaluateFlag) { // 評価マップ表示フラグ
      double[][] aRegionMap = regionMap(); // 評価マップを得る
      if (aRegionMap != null) {
        graph.setCodeMap(aRegionMap); // SOM描画に評価マップを設定
      }
    } // 評価マップ

    graph.setLabelMap(labelMap); // ラベル・マップを設定
    trialTextField.setText(String.valueOf(learningCount)); // 学習回数カウント値
     
    graph.repaint();
  }
  
  /**
   * SOMタスクからマップを更新する
   * @param iCount int // 現在までに実行した回数 epoch * numOfFactor + j 
   */
  public void updateMap(int iCount) {
    this.learningCount = iCount; // 学習回数カウント値
    updateMap();
  }
  
  /**
   * エレメント表示グラフを作成する
   */
  void compositeElementMap() {
    for (int i = 0; i < dimOfCodeX; i++) {
      int iu = i * 2, iu1 = iu + 1;
      for (int j = 0; j < dimOfCodeY; j++) {
        int ju = j * 2, ju1 = ju + 1;
        if ((i < dimOfCodeX - 1) && (j < dimOfCodeY - 1)) {
          elemetMap[iu][ju] = codeVectorArray[i + j * dimOfCodeX][elementNo];
          elemetMap[iu1][ju] = (elemetMap[iu][ju] +
                  codeVectorArray[(i+1) + j * dimOfCodeX][elementNo]) / 2.0;
          elemetMap[iu][ju1] = (elemetMap[iu][ju] +
                  codeVectorArray[i + (j+1) * dimOfCodeX][elementNo]) / 2.0;
          elemetMap[iu1][ju1] = (elemetMap[iu][ju] +
                  codeVectorArray[(i+1) + (j+1) * dimOfCodeX][elementNo]) / 2.0;
        }
        else if ((i == dimOfCodeX - 1) && (j < dimOfCodeY - 1)) {
          elemetMap[iu][ju] = codeVectorArray[i + j * dimOfCodeX][elementNo];
          elemetMap[iu1][ju] = elemetMap[iu][ju] + (-elemetMap[iu][ju] +
                  codeVectorArray[(i-1) + j * dimOfCodeX][elementNo]) / 2.0;
          elemetMap[iu][ju1] = (elemetMap[iu][ju] +
                  codeVectorArray[i + (j+1) * dimOfCodeX][elementNo]) / 2.0;
          elemetMap[iu1][ju1] = elemetMap[iu][ju1];
        }
        else if ((i < dimOfCodeX - 1) && (j == dimOfCodeY - 1)) {
          elemetMap[iu][ju] = codeVectorArray[i + j * dimOfCodeX][elementNo];
          elemetMap[iu1][ju] = (elemetMap[iu][ju] +
                  codeVectorArray[(i+1) + j * dimOfCodeX][elementNo]) / 2.0;
          if ((i + (j-1) * dimOfCodeX) < 0) {
//            elemetMap[iu][ju1] = elemetMap[iu][ju] + (-elemetMap[iu][ju]) / 2.0;
            elemetMap[iu][ju1] = elemetMap[iu][ju];
          }
          else {
            elemetMap[iu][ju1] = elemetMap[iu][ju] + (-elemetMap[iu][ju] +
                  codeVectorArray[(i) + (j-1) * dimOfCodeX][elementNo]) / 2.0;
          }
          elemetMap[iu1][ju1] = elemetMap[iu1][ju];
        }
        else {
          elemetMap[iu][ju] = codeVectorArray[i + j * dimOfCodeX][elementNo];
          if ((i - 1 < 0) ||(j - 1 < 0) ) {
//            elemetMap[iu1][ju] = elemetMap[iu][ju] + (-elemetMap[iu][ju]) / 2.0;
//            elemetMap[iu][ju1] = elemetMap[iu][ju] + (-elemetMap[iu][ju]) / 2.0;
//            elemetMap[iu1][ju1] = elemetMap[iu][ju] + (-elemetMap[iu][ju]) / 2.0;
            elemetMap[iu1][ju] = elemetMap[iu][ju];
            elemetMap[iu][ju1] = elemetMap[iu][ju];
            elemetMap[iu1][ju1] = elemetMap[iu][ju];
          }
          else {
            elemetMap[iu1][ju] = elemetMap[iu][ju] + (-elemetMap[iu][ju] +
                  codeVectorArray[(i-1) + j * dimOfCodeX][elementNo]) / 2.0;
            elemetMap[iu][ju1] = elemetMap[iu][ju] + (-elemetMap[iu][ju] +
                  codeVectorArray[(i) + (j-1) * dimOfCodeX][elementNo]) / 2.0;
            elemetMap[iu1][ju1] = elemetMap[iu][ju] + (-elemetMap[iu][ju] +
                  codeVectorArray[(i-1) + (j-1) * dimOfCodeX][elementNo]) / 2.0;
          }
        }
      }
    }
  } // compositeElementMap()
  
  // 評価マップを作成する
  double[][] regionMap() {
    double[][] aRegionMap; // 評価マップ
    
    double aNorm; // ノルム（差分の平方和）
    double leastNorm; // ノルムの最少値
    double delta; // 差分
    int[] bestMatches = new int[numOfCodeVector]; // BMUのラベル値
    
    int type = Integer.parseInt(evaluationTypeTextField.
            getText().trim()); // ラベル値の計算方法（０～３）
    int base = Integer.parseInt(baseTextField.
            getText().trim()); // ラベルの種類数

    // 全ての参照ベクタに対して入力ベクタとのノルムを比較する。
    for (int i = 0; i < numOfCodeVector; i++) { // 参照ベクタの個数
      leastNorm = Double.MAX_VALUE; // ノルムの最少値
      bestMatches[i] = 0; // ノルムが最少となる因子ベクタのラベル値
      for (int j = 0; j < numOfFactor; j++) { // 因子ベクタの個数
        aNorm = 0.0; // ノルム（差分の平方和）
        for (int k = 0; k < dimOfFactor; k++) { // 参照ベクタの次数
          // 参照ベクタと入力ベクタのノルムを計算する。
          delta = codeVectorArray[i][k] - factorVectors[j][k]; // 要素の差分
          aNorm = aNorm + delta * delta; // 差分の平方和
          if (aNorm > leastNorm) break; // 平方和が大きければループから抜ける。
        } // for (j < dimOfCode) // 参照ベクタの次数
        if (aNorm < leastNorm) {
          leastNorm = aNorm; // 二乗和の最小値を更新
          // ラベル値を代入
          switch (type) {
            case TYPE_RESIDUE: // 剰余
              bestMatches[i] = j % base; // BMUのラベル値
              break;
            case TYPE_QUOTIENT : // 商
              bestMatches[i] = j / base; // BMUのラベル値
              break;
            case TYPE_LABEL : // ラベルで判定
              String st = factorLabels[j].trim();
              if (!(st.isEmpty())) bestMatches[i] = Integer.parseInt(st); // BMUのラベル値
              break;
            case TYPE_NOTHING : // Do nothing
            default: // Do nothing
              break;
          } // switch
        } // if
      } // for (j < numOfVector) 学習ベクタの個数
    } // for (i < numOfCode) // 参照ベクタの個数
    
    aRegionMap = new double[dimOfCodeX * 2][dimOfCodeY * 2]; // 評価マップ
    for (int i = 0; i < dimOfCodeX; i++) {
      int iu = i * 2;
      for (int j = 0; j < dimOfCodeY; j++) {
        int ju = j * 2;
        aRegionMap[iu][ju] = aRegionMap[iu][ju + 1] = 
        aRegionMap[iu + 1][ju] = aRegionMap[iu + 1][ju + 1] =
                (double)bestMatches[i + j * dimOfCodeX]; // BMUのラベル値を代入
      }
    }
    
    return aRegionMap; // 評価マップ
  }
  
  /** SOMビューワ・タイトルを得る
   * @return VIEWER_TITLE String //ビューワ・タイトル
   */
  public String getViewerTitile() {
    return VIEWER_TITLE;
  }

//-----------------------------------------------------------------------------
  
  // ＵマップＸ軸幅
  public String getUMapXSize() {
    return uMapXSizeTextField.getText().trim();
  }
  public void setUMapXSize(String uMapXSize) {
    uMapXSizeTextField.setText(uMapXSize.trim());
    graph.setUMapXSize( Float.parseFloat(uMapXSize.trim()) );
  }

  // ＵマップＹ軸幅
  public String getUMapYSize() {
    return uMapYSizeTextField.getText().trim();
  }
  public void setUMapYSize(String uMapYSize) {
    uMapYSizeTextField.setText(uMapYSize.trim());
    graph.setUMapYSize( Float.parseFloat(uMapYSize.trim()) );
  }

  // 評価方法
  public String getEvaluationType() {
    return evaluationTypeTextField.getText().trim();
  }
  public void setEvaluationType(String evaluationType) {
    evaluationTypeTextField.setText(evaluationType.trim());
//#    graph.setXRange( Float.parseFloat(evaluationType.trim()) ); // バグ
  }

  // Ｕマップ基準等高線
  public String getBaseContour() {
    return baseTextField.getText().trim();
  }
  public void setBaseContour(String baseContour) {
    baseTextField.setText(baseContour.trim());
//#    graph.setYRange( Float.parseFloat(aRange.trim()) ); // バグ
  }

  // Ｕマップ境界等高線
  public String getBorderContour() {
    return borderContourTextField.getText().trim();
  }
  public void setBorderContour(String borderContour) {
    borderContourTextField.setText(borderContour.trim());
    graph.setZMax( Float.parseFloat(borderContour.trim()) ); // 名称変更は保留
  }

  // Ｕマップ等高線数
  public String getNumOfContour() {
    return numOfContourTextField.getText().trim();
  }
  public void setNumOfContour(String numOfContour) {
    numOfContourTextField.setText(numOfContour.trim());
    graph.setNumOfContour( Float.parseFloat(numOfContour.trim()) );
  }

  /**
   * グラフを再描画する
   */
  public void repaintGraph() {
    graph.repaint();
  }

  // -----------------------------------------------------------------------------
  /**
   * Ｕマップを得る。
   * @return uMatrix double[][] // Ｕマップ
   */
  public double[][] getUMatrix() {
    return uMatrix;
  }

  /** 
   * ラベル・マップを得る
   * @return labelMap String[][] // ラベル・マップ
   */
  public String[][] getLabelMatrix() {
    return labelMap;
  }
  
  /**
   * 学習回数カウント値を得る
   * @return learningCount double // 学習回数カウント値
   */
  public double getLearningCount() {
    return learningCount;
  }
  
  /**
   * 参照ベクタとラベル・マップをファイルに保存する
   */
  public void saveData() {
    
    // DJ.print("SomGraphViewer.setButtonAction()" + " Save SOMボタンがクリックされました。");
        
    // 文字列をバッファに追加
    ArrayList<String> buf = new ArrayList<>(); // ファイル出力用文字列バッファ
    buf.add(comment.toString() + "\r\n"); // コメント
    
    // コメントを取出し
    StringTokenizer commentToken = new StringTokenizer(comment.toString(), ",");
    ArrayList<String> commentList = new ArrayList<>(); // コメント・リスト
    while (commentToken.hasMoreTokens()) {
      commentList.add(commentToken.nextToken()); // コメントを追加
    }
    // ファイル名に因子種別を追加
    StringBuilder fileName = new StringBuilder("SomCode_"); // ファイル名のヘッダ
//    if (commentList.size() > 0) fileName.append(commentList.get(1)).append("-"); // 因子種別
    if (!commentList.isEmpty()) fileName.append(commentList.get(1)).append("-"); // 因子種別
    fileName.append(util.TimeStamp.getDateAndTime()).append(".dat"); // 日時
    
    // プロパティ
    buf.add("Property:" + "\r\n");
    buf.add("Number Of Trial = " + "\r\n");
    buf.add(numOfTrial + "\r\n"); // 試行因子ベクタ数
    buf.add("Evaluation Type = " + "\r\n");
    buf.add(evaluationType + "\r\n"); // 評価方法
    buf.add("Numbe Of Contour = " + "\r\n");
    buf.add(numOfContour + "\r\n"); // Ｕマップ等高線数
    buf.add("Base Contour = " + "\r\n");
    buf.add(baseContour + "\r\n"); // Ｕマップ基準等高線
    buf.add("Border Contour = " + "\r\n");
    buf.add(borderContour + "\r\n"); // Ｕマップ境界等高線
    buf.add("U-Map X-Axis Size = " + "\r\n");
    buf.add(uMapXSize + "\r\n"); // ＵマップＸ軸表示幅[ドット]
    buf.add("U-Map Y-Axis Size = " + "\r\n");
    buf.add(uMapXSize + "\r\n"); // ＵマップＹ軸表示幅[ドット]
    
    // 因子ベクタ
    buf.add("Factor Vector:" + "\r\n");
    buf.add("Dimension of factor = " + "\r\n");
    buf.add(dimOfFactor + "\r\n"); // 因子ベクタの次数
    // 因子名を追加
    buf.add("factorNames:" + "\r\n");
    buf.add("Factor Name = " + "\r\n");
    if (dimOfFactor > 0) {
      buf.add(factorNames[0]);
      for (int k = 1; k < dimOfFactor; k++) {
        buf.add("," + factorNames[k]); // 因子名
      }
    }
    buf.add("\r\n"); // 復帰改行
    
    // 因子ベクタを追加
    buf.add("factorVectors = " + "\r\n");
    buf.add(numOfFactor + "\r\n"); // 因子ベクタ数
    if (numOfFactor > 0) {
      // 最初の行
      addFactor(factorVectors, 0, buf);
      // 残りの行
      for (int j = 1; j < numOfFactor; j++) {
        addFactor(factorVectors, j, buf);
      }
    }
    else {
      buf.add("\r\n"); // 復帰改行
    }
    
    // 因子ラベルを追加
    buf.add("Factor Label:" + "\r\n"); // 因子ラベル
    buf.add("factorLabels = " + "\r\n"); // 因子ラベル配列
    buf.add(numOfFactor + "\r\n"); // 因子ベクタ数
    if (numOfFactor > 0) {
      buf.add(" " + factorLabels[0]); // 最初の因子ラベル
      for (int i = 1; i < numOfFactor; i++) { // 残りの因子ラベル
        buf.add(", " + factorLabels[i]);
      }
    }
    buf.add("\r\n"); // 復帰改行
    
    // 参照ベクタ
    buf.add("Code Vectors:" + "\r\n");
    buf.add("Number of codeVectors =" + "\r\n");
    buf.add(numOfCodeVector + "\r\n"); // 参照ベクタ数
    // 参照ベクタ数を追加
    buf.add("codeVectors = " + "\r\n");
    if (numOfCodeVector > 0) {
      addFactor(codeVectorArray, 0, buf); // 最初の行
      for (int j = 1; j < numOfCodeVector; j++) {
        addFactor(codeVectorArray, j, buf); // 残りの行
      }
    }
    else {
      buf.add("\r\n"); // 復帰改行
    }
            
    // ラベルマップ
    buf.add("Label Map:" + "\r\n");
    int row = labelMap.length; // 行数
    int col = labelMap[0].length; // 列数
    buf.add("Row number of Label Map = " + "\r\n");
    buf.add(row + "\r\n"); // 行数
    buf.add("Collumn number of Label Map = " + "\r\n");
    buf.add(col + "\r\n"); // 列数
    // ラベルマップを追加
    buf.add("Label Map =" + "\r\n");
    if (row > 0) {
      // 最初の行
      addLabel(labelMap, 0, buf);
      // 残りの行
      for (int j = 1; j < row; j++) {
        addLabel(labelMap, j, buf);
      }
    }
    else {
      buf.add("\r\n"); // 復帰改行
    }
    
    LogEditor.saveText(buf, fileName.toString()); // ファイルに保存
  }
  
  // 因子ベクタをバッファに書き込む
  void addFactor(double[][] array, int j, ArrayList<String> buf) {
    int col = array[j].length; // 列数
    if (col > 0) {
      buf.add(" " + array[j][0]);
      for (int i = 1; i < col; i++) {
        buf.add(", " + array[j][i]); // ラベル
      }
      buf.add("\r\n"); // 復帰改行
    }
  }
  
  // ラベルマップをバッファに書き込む
  void addLabel(String[][] array, int j, ArrayList<String> buf) {
//    int row = array.length; // 行数
    int col = array[j].length; // 列数
    if (col > 0) {
      buf.add(" " + array[j][0]);
      for (int i = 1; i < col; i++) {
        buf.add(", " + array[j][i]); // ラベル
      }
      buf.add("\r\n"); // 復帰改行
    }
  }
  
  // ---------------------------------------------------------------------------
  void button_actionPerformed(ActionEvent e) {
    String cmd = e.getActionCommand();

    switch (cmd) {
      case "Graph":
        break;
//$      case "Close":
//        // ビューワーを閉じる。
//        setVisible(false);
//        break;
//$      case "Delete":
//        // ビューワーを閉じて、登録を抹消する。
//        deleteViewer();
//        break;
//$      case "Property":
//        // プロパティ・ダイアログを表示する。
//        openDialog();
//        break;
//$      case "CLEAR":
//        // グラフ描画を消去する。
//        graph.clearGraph();
//        break;
      case "U-Map":
        // Ｕマップを選択する。
//      if (!umapFlag) {
        elementFlag = false; // 因子マップ選択フラグ
        elementButton.setBackground(buttonColor);
        evaluateFlag = false; // 評価マップ表示フラグ
        evaluateButton.setBackground(buttonColor);
        
        borderFlag = false;
        borderButton.setBackground(buttonColor);
        graph.setEvaluateFlag(evaluateFlag); // 評価マップ
        graph.setBorderFlag(borderFlag);
        
        umapFlag = true; // Ｕマップ選択フラグ
        umapButton.setBackground(Color.ORANGE);
        updateMap(); // マップを更新
        break;
      case "Element":
        // 要素変数マップを選択する。
        if (!elementFlag) { // 因子マップ選択フラグ
          umapFlag = false; // Ｕマップ選択フラグ
          umapButton.setBackground(buttonColor);
          evaluateFlag = false; // 評価マップ表示フラグ
          evaluateButton.setBackground(buttonColor);
          
          borderFlag = false; //# 20190115
          borderButton.setBackground(buttonColor);
          graph.setEvaluateFlag(evaluateFlag); //# 評価マップ表示
          graph.setBorderFlag(borderFlag);
          
          elementFlag = true; // 因子マップ選択フラグ
          elementButton.setBackground(Color.ORANGE);
          updateMap(); // マップを更新
        } break;
      case "|<<":
        // 先頭要素選択ボタン
        if (!elementFlag) { // 因子マップ選択フラグ
          umapFlag = false; // Ｕマップ選択フラグ
          umapButton.setBackground(buttonColor);
          elementFlag = true; // 因子マップ選択フラグ
          elementButton.setBackground(Color.ORANGE);
        } elementNo = 0; // 先頭要素番号
        updateMap(); // マップを更新
        break;
      case "<":
        // 前の要素選択ボタン
        if (elementFlag) { // 因子マップ選択フラグ
          int no = elementNo - 1;
          if (no >= 0) {
            elementNo = no; // 前の要素番号
            updateMap(); // マップを更新
          }
        } break;
      case ">":
        // 後の要素選択ボタン
        if (elementFlag) { // 因子マップ選択フラグ
          int no = elementNo + 1;
          if (no < dimOfFactor) {
            elementNo = no; // 後の要素番号
            updateMap(); // マップを更新
          }
        } break;
      case ">>|":
        // 末尾要素選択ボタン
        if (!elementFlag) { // 因子マップ選択フラグ
          umapFlag = false; // Ｕマップ選択フラグ
          umapButton.setBackground(buttonColor);
          elementFlag = true; // 因子マップ選択フラグ
          elementButton.setBackground(Color.ORANGE);
        } elementNo = dimOfFactor - 1; // 末尾要素番号
        updateMap(); // マップを更新
        break;
      default:
        break;
    }
  }

  void setButton_actionPerformed(ActionEvent e) {
    String cmd = e.getActionCommand();

    try {
      switch (cmd) {
        case "SET":
          // Do nothing
          break;
        case "Evaluate": // 評価マップ表示ボタン
          // 評価マップ表示ボタン
          if (evaluateFlag) { // 評価マップ表示フラグ　（true：評価マップを表示）
            evaluateFlag = false;
            evaluateButton.setBackground(buttonColor);
          }
          else {
            // 評価マップを表示する
            umapFlag = false; // Ｕマップ選択フラグ
            umapButton.setBackground(buttonColor);
            elementFlag = false; // 因子マップ選択フラグ
            elementButton.setBackground(buttonColor);

            borderFlag = false; //# 20190115
            borderButton.setBackground(buttonColor);
            evaluateFlag = true; // 評価マップ表示フラグ
            evaluateButton.setBackground(Color.ORANGE);
          }
          graph.setEvaluateFlag(evaluateFlag); // 評価マップ表示グラフ 
          graph.setBorderFlag(borderFlag);
          break;
        case "Border":
          // 境界線表示ボタン
          if (borderFlag) { // 境界線表示フラグ　（true：境界線を表示）
            borderFlag = false;
            borderButton.setBackground(buttonColor);
          }
          else {
            evaluateFlag = false; // 評価マップ表示フラグ
            evaluateButton.setBackground(buttonColor);
            borderFlag = true;
            borderButton.setBackground(Color.ORANGE);
          }
          graph.setEvaluateFlag(evaluateFlag); // 評価マップ表示フラグ
          graph.setBorderFlag(borderFlag);
          break;
          
        case "Save SOM":
          // 参照ベクタ保存ボタン
          
          // 参照ベクタとラベル・マップをファイルに保存
          saveData();

          break;
          
        case "▲":
          {
            numOfContour = Integer.parseInt(numOfContourTextField.getText().trim());
            numOfContourTextField.setText(String.valueOf(numOfContour + 1));
            break;
          }
        case "▼":
          {
            numOfContour = Integer.parseInt(numOfContourTextField.getText().trim());
            numOfContour = numOfContour - 1;
            if (numOfContour > 1) {
              numOfContourTextField.setText(String.valueOf(numOfContour));
            }   break;
          }
        case "△":
          {
            // 境界線高さ増加ボタン
            float borderLevel = Float.parseFloat(borderContourTextField.getText().trim());
            if (borderLevel < 0.0001F) borderLevel = 0.1F;
            borderLevel = borderLevel * 1.05F + 0.00005F;
            int i = (int)(borderLevel * 10000.0F);
            borderLevel = (float)i / 10000.0F;
            if (borderLevel > 1.0F) borderLevel = 1.0F;
            borderContourTextField.setText(String.valueOf(borderLevel));
            break;
          }
        case "▽":
          {
            // 境界線高さ低減ボタン
            float borderLevel = Float.parseFloat(borderContourTextField.getText().trim());
            if (borderLevel < 0.0001F) borderLevel = 0.9F;
            borderLevel = borderLevel * 0.95F + 0.00005F;
            int i = (int)(borderLevel * 10000.0F);
            borderLevel = (float)i / 10000.0F;
            if (borderLevel < 0.0F) borderLevel = 0.0F;
            borderContourTextField.setText(String.valueOf(borderLevel));
            break;
          }
        default:
          break;
      }

      // X Axis
      graph.setUMapXSize(Float.parseFloat(uMapXSizeTextField.getText().trim()) );
      // Y Axis
      graph.setUMapYSize(Float.parseFloat(uMapYSizeTextField.getText().trim()) );
      // Z Axis
      graph.setZMin( Float.parseFloat(zMinTextField.getText().trim()) );
      graph.setZMax(Float.parseFloat(borderContourTextField.getText().trim()) );
      // 等高線数をチェック
      float zRange = Float.parseFloat(numOfContourTextField.getText().trim());
      if (zRange <= MAX_CONTOUR) { // 最大等高線数 20190202
        graph.setNumOfContour(zRange);
      }
      else {
        DJ.print("***** WARNING ***** SomGraphViewer.setButtonAction()"
          + " 等高線数が" + Integer.toString(MAX_CONTOUR) + "を超えています。");
      }
    }
    catch (NumberFormatException nfe) {
      DJ.print("***** ERROR ***** SomGraphViewer.setButton_actionPerformed()"
               + " Number Format Error.");
    }
    
    updateMap(); // マップを更新
    colorPaletPanel.repaint();
  }

  void settingTextField_actionPerformed(ActionEvent e) {
    int anElementNo = (int)Float.parseFloat(onValueTextField.getText().trim()) ; // 要素番号
    if (anElementNo >= dimOfFactor) { // 参照ベクタの次数
      DJ.print("***** ERROR ***** SomGraphViewer.settingTextField_actionPerformed()"
               + " ElementNo > dimOfCode.");
      
      onValueTextField.setText(String.valueOf(elementNo));
    }
    else {
      elementNo = anElementNo; // 要素番号
      updateMap(); // マップを更新
    }
  }
  
//  // スクロール・バーによるグラフ位置の修正
//  private void graphsubPanelAncestorMoved(javax.swing.event.AncestorEvent evt) {                                         
//
////    this.validate();
////    this.doLayout();
//     Dimension d = graphPanel.getSize();
//     this.pack();
//     graphPanel.setSize(d);
////    this.revalidate();
////    this.repaint();
//    
////    Dimension d = this.getSize();
////    this.setSize(d.width, d.height);
//
////    Point p = graphPanel.getLocation();
////    Point p = graphScrollPane.getLocation();
////    Point p = graphsubPanel.getLocation();
//
////     Dimension d = new Dimension(400, 400);
////     Dimension d = graphScrollPane.getSize();
////     Rectangle r = graphsubPanel.getBounds();
//
////    graphPanel.setLocation(p);
////    graphScrollPane.setLocation(p);
////    graphsubPanel.setLocation(p);
////    graph.setLocation(p);
//
////    graphsubPanel.setSize(d);
////    graph.setSize(d);
//
////    graphPanel.validate();
////    graphPanel.repaint();
////    graphScrollPane.validate();
////    graphScrollPane.repaint();
////    graphsubPanel.validate();
////    graphsubPanel.repaint();
////    graph.validate();
////    graph.repaint();
//    
////    Point p = graphsubPanel.getLocation();
////    graph.validate();
////    graph.repaint();
////    graphScrollPane.validate();
////    graphScrollPane.repaint();
////    graph.setLocation(p);
////    graphPanel.validate();
//  }
  
} // SomViewe


class SomGraphViewer_button_actionAdapter implements java.awt.event.ActionListener {
  SomViewer adaptee;
  SomGraphViewer_button_actionAdapter(SomViewer adaptee) {
    this.adaptee = adaptee;
  }
  @Override
  public void actionPerformed(ActionEvent e) {
    adaptee.button_actionPerformed(e);
  }
}

class SomGraphViewer_setButton_actionAdapter implements java.awt.event.ActionListener {
  SomViewer adaptee;
  SomGraphViewer_setButton_actionAdapter(SomViewer adaptee) {
    this.adaptee = adaptee;
  }
  @Override
  public void actionPerformed(ActionEvent e) {
    adaptee.setButton_actionPerformed(e);
  }
}

class SomGraphViewer_settingTextField_actionAdapter implements java.awt.event.ActionListener {
  SomViewer adaptee;
  SomGraphViewer_settingTextField_actionAdapter(SomViewer adaptee) {
    this.adaptee = adaptee;
  }
  @Override
  public void actionPerformed(ActionEvent e) {
    adaptee.settingTextField_actionPerformed(e);
  }
}

class SomGraphViewer_rangeTextField_actionAdapter implements java.awt.event.ActionListener {
  SomViewer adaptee;
  SomGraphViewer_rangeTextField_actionAdapter(SomViewer adaptee) {
    this.adaptee = adaptee;
  }
  @Override
  public void actionPerformed(ActionEvent e) {
    adaptee.setButton_actionPerformed(e);
  }
}

// EOF
