/*
 *  Title: DaiJa_V4 ( Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 3.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package data;

import java.util.ArrayList;
import util.DJ;

/**
 * <p> 表　題: Class: Control</p>
 * <p> 説　明: 制御系</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.07.03</p>
 */
public class Control {
  
  // 制御用定数、パラメータと変数
  private static final long L_ZERO = 0L;
  private static final double D_ZERO = 0.0;
  private static final double SYSTEM_TIME = 0.001; // System Time Interva
  private static final double INTERPOLATE_TIME = 0.01; // Interpolate Interva
  private static final double PSN_FACTOR = 0.000001; // Position Unit Factor
  private double system_clock; // System Clock
  private double system_time; // system Clock Interval
  private double interpolate_time; // Interpolation Interval
  private double interpolate_count; // Interpolation Counter
  private double display_count; // Display Counter
  private double psn_factor; // Position Unit Factor
  
  // シミュレーション
  // プラント Plant
  public static final float DAC_FACTOR = 10.0F / 2048.0F; // 2048pulse to 10volt
  private static final double INVERTER = 1.0; // A/V
  private static final double FRC_CONST = 1.0; // Nm/A
  private static final double INERTIA = 1.0; //0.5; //1.0; // kgm2/sec2
  private static final double FRICTION = 2.0; //2.0; //0.0; // kg/s
  private long dac_buffer; // D/A Converter Buffer
  private long fbc_buffer; // Feedback Counter Buffer
  private double inverter; // Inverter Gain
  private double frc_const; // Force Constant
  private double inertia; // Load Inertia
  private double friction; // Load Friction

  // 解析 Analyze
  private static final double SAMPLING_TIME = 0.001; // Sampling Interval
  private double sampling_time; // Sampling Time
  private double sampling_count; // Sampling Counter	
  private double spd_step; // Step Response
  private double spd_ramp; // Ramp Response
  
  // シミュレーション Simulation
  private static final double SIMULATION_TIME = 0.001; // Simulation Interval
  private double simulation_time; // Simulation Interval
  private double simulation_count; // Simulation Counter
  private double frc_sim; // Force Simulation
  private double spd_sim; // Speed Digital Simulation
  private double psn_sim; // Position Digital Simulation
  private double frc_sim_1; // Force(k-1)
  private double frc_sim_2; // Force(k-2)
  private double psn_sim_1; // Position(k-1)
  private double psn_sim_2; // Position(k-2)
  private double frc_rng; // Force Runge-Kutta Method
  private double spd_rng; // Speed Runge-Kutta Method
  private double psn_rng; // Position Runge-Kutta Method
  private double spd_var; // Speed Variation
  
  // 制御
  // フィードバック処理 Input Feedback
  private static final long FB_SCALE = 1; // Feedback Scaling Factor

  // 補間 Interpolate
  private static final long TIM_REF = 1000; // msec
  private static final long ACC_REF = 2; // um/msec/msec
  private static final long SPD_REF = 1000; //100;	 // um/msec
  private static final long PSN_REF = 100000; //60000; // um
  private long tim_ref; // Time Reference
  private long acc_ref; // Acceleration Reference
  private long spd_ref; // Speed Reference
  private long psn_ref; // Position Reference
  private long acc_cmd; // Acceration Command
  private long spd_cmd; // Speed Command
  private long psn_cmd; // Position Command
  // 位置制御 Position Control
  private static final long PSN_PRM = 0; //25; // Position Loop Gain
  private long psn_prm; // Position Loop Gain
  private long psn_err; // Position Error
  private long spd_mnp; // Speed Manipulate
  // 速度制御 Speed Control
  private static final long FIL_PRM = 0; // 1/1000
  private static final long INT_PRM = 1048; //2048; //1024; //256; //64; // 1/1000
  private static final long DIF_PRM = 0; // 1/1000
  private static final long SPD_PRM = 1024; //512; // 2048; // 1/1000
  private long fil_prm; // Filter Parameter
  private long int_prm; // Integral Parameter
  private long dif_prm; // Differential Parameter
  private long spd_prm; // Speed Loop Gain
  private long spd_err; // Speed Error
  private long spd_err_old; // Old Speed Error
  private long spd_cmp; // Speed Compensation
  private long int_cmp; // Integral Compensation
  private long dif_cmp; // Differential Compensation
  private long frc_mnp; // Force Manipulate
  private long frc_old; // Force Manipulate
  // フィードバック処理 Input Feedback
  private long FBC_PORT;
  private long psn_fed; // Position Feedback
  private long spd_fed; // Speed Feedback
  private long acc_fed; // Acceleration Feedback
  private long frc_fed; // Force Feedback
  private long psn_old; // Old Position
  private long spd_old; // Old Speed
  // Ｄ／Ａ変換器出力 Output D/A Converter
  public long DAC_PORT;
  public static final long DAC_MIN = 0xfffff800; // D/A Minimum Output
  public static final long DAC_MAX = 0x000007ff; // D/A Maximum Output
  
  // 外乱
  public static final double DELTA_FORCE = 0.00005; // 外乱の変化量
  private double massBase = INERTIA; // 基本質量
  
  
  public Control() {
    initControl();
  }
  
  /** 
   * 制御の初期化
   */
  public final void initControl() {
    // 補間
    tim_ref = TIM_REF; // Time Reference
    acc_ref = ACC_REF; // Acceration Reference
    spd_ref = SPD_REF; // Speed Reference
    psn_ref = PSN_REF; // Position Reference
    
    // 制御
    fil_prm = FIL_PRM; // Filter Parameter
    int_prm = INT_PRM; // Integral Parameter
    dif_prm = DIF_PRM; // Differential Parameter
    spd_prm = SPD_PRM; // Speed Loop Gain
    psn_prm = PSN_PRM; // Position Loop Gain
    
    // 解析
    system_time = SYSTEM_TIME; // system Clock Interval
    interpolate_time = INTERPOLATE_TIME; // Interpolation Interval
    sampling_time = SAMPLING_TIME; // Sampling Time
    
    // シミュレーション
    psn_factor = PSN_FACTOR; // Position Unit Factor, 1m to 1um
    
    // プラント
    inverter = INVERTER; // Inverter Gain
    frc_const = FRC_CONST; // Force Constant
    inertia = INERTIA; // Load Inertia
    friction = FRICTION; // Load Friction/
    
    // シミュレーション
    simulation_time = SIMULATION_TIME; // Simulation Interval
    
    resetControl(); // 制御のリセット
  }
  
  /**
   * 制御のリセット
   */
  public void resetControl() {
    // プラント, 解析
    system_clock = D_ZERO; // Sampling Counter
    sampling_count = D_ZERO; // Sampling Counter
    spd_step = D_ZERO; // Step Response
    spd_ramp = D_ZERO; // Ramp Response
    // シミュレーション
    simulation_count = D_ZERO; // Simulation Counter
    dac_buffer = L_ZERO; // D/A Converter Buffer
    fbc_buffer = L_ZERO; // Feedback Counter Buffer
    frc_sim = D_ZERO; // Force Simulation
    spd_sim = D_ZERO; // Speed Digital Simulation
    psn_sim = D_ZERO; // Position Digital Simulation
    frc_sim_1 = D_ZERO; // Force(k-1)
    frc_sim_2 = D_ZERO; // Force(k-2)	
    psn_sim_1 = D_ZERO; // Position(k-1)
    psn_sim_2 = D_ZERO; // Position(k-2)
    frc_rng = D_ZERO; // Force Runge-Kutta Method
    spd_rng = D_ZERO; // Speed Runge-Kutta Method
    psn_rng = D_ZERO; // Position Runge-Kutta Method
    spd_var = D_ZERO; // Speed Variation
    
    // 補間
	acc_cmd = L_ZERO; // Acceration Command
	spd_cmd = L_ZERO; // Speed Command
	psn_cmd = L_ZERO; // Position Command

	// 位置制御
	psn_err = L_ZERO; // Position Error
	spd_mnp = L_ZERO; // Speed Manipulate

	// 速度制御
	spd_err = L_ZERO; // Speed Error
	spd_err_old =L_ZERO; // Old Speed Error
	spd_cmp = L_ZERO; // Speed Compensation
	int_cmp = L_ZERO; // Integral Compensation
	dif_cmp = L_ZERO; // Differential Compensation
	frc_mnp = L_ZERO; // Force Manipulate
	frc_old = L_ZERO; // Force Manipulate

//    // 電流制御 Current Control
//    angle = L_ZERO; // Motor Angle
//    uph_cur = L_ZERO; // U Phase Current
//    vph_cur = L_ZERO; // V Phase Current
//    wph_cur = L_ZERO; // W Phase Current

	// フィードバック処理
	psn_fed = L_ZERO; // Position
	spd_fed = L_ZERO; // Speed
	acc_fed = L_ZERO; // Acceleration
	frc_fed = L_ZERO; // Force
	psn_old = L_ZERO; // Old Position
	spd_old = L_ZERO; // Old Speed
  }
  
  // ---------------------------------------------------------------------------
  private boolean speedControlFlag = true; // true：速度制御を有効にする
  private boolean feedbackFlag = true; // true：フィードバックを有効にする
  
  public void stopSpeedControl() {
    speedControlFlag = false;
  }
  
  public void startSpeedControl() {
    speedControlFlag = true;
  } 
  
  public void stopFeedback() {
    feedbackFlag = false;
  }
  
  public void startFeedback() {
    feedbackFlag = true;
  } 
  
  public double getForce() {
    return frc_mnp;
//    return frc_sim;
  }
  
  /**
   * 加速度を得る
   * @return spd_var double
   */
  public double getAcceleration() {
    return  spd_var;
  }
  
  // 速度を得る
  public double getSpeed() {
    return  spd_rng;
  }
  
  public double getPosition() {
    return psn_rng;
  }
  
  /**
   * 速度のフィードバックを得る
   * @return spd_fed
   */
  public double getSpeedFeedback() {
    return spd_fed;
  }
  
  /**
   * シミュレータへの入力として、力の操作量を設定する
   * @param force double
   */
  public void setForce(double force) {
    frc_mnp = (long)(force * 1.0); // 力の操作量
  }
  
  /**
   * シミュレータへの入力として、速度を設定する
   * @param speed double
   */
  public void setSpeed(double speed) {
    spd_ref = (long)(speed * 1000.0); // 速度
  }
  
  /**
   * 制御パラメータを設定する
   * @param spd_prm int // 比例ゲイン
   * @param int_prm int // 積分ゲイン
   * @param inertia double // 慣性質量
   * @param friction double // 粘性摩擦
   */
  public void setParameter(
          int spd_prm, int int_prm, double inertia, double friction) {
    this.spd_prm = (long)spd_prm;
    this.int_prm = (long)int_prm;
    this.inertia = inertia;
    this.friction = friction;
  }
  
  /**
   * 基本質量を設定する
   * @param mussBase double
   */
  public void setMassBase(double mussBase) {
    this.massBase = mussBase;
  }
  
  /**
   * 外乱を与える
   * @param TryalCount int // 実行回数
   */
  public void disturber(int TryalCount) {
    // 実行回数が10000回を超えたら、質量に外乱を与える
    if (TryalCount < 10000) {
      // Do nothing
    }
    else if (TryalCount < 20000) {
      inertia = inertia - DELTA_FORCE;
    }
    else if (TryalCount < 40000) {
      inertia = inertia + DELTA_FORCE;
    }
    else if (TryalCount < 50000) {
      inertia = inertia - DELTA_FORCE;
    }
    else {
      // Do nothing
    }
  }
  
  /**
   * 質量を変動させる
   */
  public void massChanger() {
//    double var = (+1.0 - Math.cos(2.0 * Math.PI * time)) * 0.5; // 変動分
    double var = (+1.0 - Math.cos(6.0 * Math.PI * psn_rng)) * 0.1; // 変動分
//    inertia = massBase * (1.0 + var);
    inertia = massBase * (1.1 - var);
  }
  
/**
 * 質量を変動させる
 * @param changeRate double // 変動率
 */  
  public void massChanger(double changeRate) {
    double var = (+1.0 - Math.cos(6.0 * Math.PI * psn_rng)) * 0.1; // 変動分
//    inertia = massBase * (1.0 + var * changeRate);
    inertia = massBase * (1.0 + (0.1 - var) * changeRate);
  }
  
  /**
   * 質量を得る
   * @return inertia double
   */
  public double getMass() {
    return inertia;
  }
  
  /**
   * 速度ステップ応答（比例制御のみ）の理論値
   * 正解値とする速度のステップ応答（解析解）
   * @param time double
   * @return spd_step double
   */
  public double speed_step(double time) {
    double gain;
    double sref;
    sref = (double) (spd_ref) * psn_factor / sampling_time;
    gain = (double) (spd_prm) * DAC_FACTOR * inverter
            * frc_const / psn_factor * sampling_time / (double) 1000.0;
    spd_step = gain * sref / (+friction + gain) * (+(double) 1.0
            - Math.exp(-(+friction + gain) / inertia * time));
    return spd_step;
  }
  
  /**
   * 速度Ｓ字カーブ応答（3次多項式）1/10にスケーリング
   *  y=ax^3+bx^2+cx+d
   *  （１）原点Ps=(0,0)と点Pe=(1,1)を通る。 
   *  （２）PsとPeで傾き（加速度）がゼロ。dy/dx|(0.0)=dy/dx|(1,1)=0
   * @param time double
   * @return spd_s double
   */
  public double speed_sCurve(double time) {
    return  (-0.2 * time + 0.3 )* time * time; // S字カーブ   
  }
  
  /**
   * 速度Ｓ字カーブ応答（3次多項式）
   * @param clock double // クロック
   * @return sCurve double // S字カーブ
   */
  public double getS_Curve(double clock) {
    double sCurve = (-2 * clock + 3 )* clock * clock;
    return sCurve; // S字カーブ   
  }
  
  /**
   * 速度ランプ応答（比例制御のみ）の理論値
   * 正解値とする速度のランプ応答（解析解）
   * @param time double
   * @return spd_step double
   */
  public double spd_ramp(double time) {
    double aref = (double)acc_ref * psn_factor
            / sampling_time / sampling_time;
    double gain = (double)spd_prm * DAC_FACTOR * inverter
            * frc_const / psn_factor * sampling_time / (double)1000.0;
    spd_ramp = gain * aref / (friction + gain) * (system_clock
            - inertia / (friction + gain) * ((double) 1.0
            - Math.exp(-(friction + gain) / inertia * system_clock)));
    return spd_ramp; // * 0.05950193; //0.06; // スケーリング
  }
  
  /**
   * 学習データ「操作量（力）と応答（速度）」を抽出する
   * @param pointNum int // データ抽出数
   * @return sampleData double[][] // 抽出した学習データ
   */
  public double[][] getSampleData(int pointNum) {
    double[][] sampleData = new double[2][pointNum];
    
    int adjuster = 1000 / pointNum; // 制御の総実行時間を1000[msec]にするための係数
    for (int i = 0; i < pointNum * adjuster; i++) {
      execute();
      if ((i % adjuster) == 0) {
        int j = i / adjuster;
        sampleData[0][j] = getSpeed(); // 応答データ（速度）
        sampleData[1][j] = getForce(); // 入力データ（力の操作量）
      }
    }
    DJ.print(" Speed", sampleData[0]);
    DJ.print(" Force", sampleData[1]);
    
    return sampleData;
  }
  
  /**
   * 学習データを得る
   * @param pGainNum int // 比例ゲイン数
   * @param iGainNum int // 積分ゲイン数
   * @param pointNum int // 選点数
   * @return dataSet ArrayList<double[][]> // 学習データ
   */
  public ArrayList getDataSet(int pGainNum, int iGainNum, int pointNum) {
    ArrayList<double[][]> dataSet = new ArrayList<>();
    long gpd = SPD_PRM / 10L; // 変化量　 1024/10=102
    long gp = SPD_PRM - (gpd * 6L); // 1024 - 102 * 6 = 512
    long gid = INT_PRM / 1L; // 1024 / 8 = 128;
    long gi = 0; // 256
    
    DJ.print("・応答データ（速度）と入力データ（力の操作量）");
    for (int m = 0; m < pGainNum; m++) {
      gp = gp + gpd * (long)m; // 比例ゲインを変更
      for (int n = 0; n < iGainNum; n++) {
        gi = gid * (long)n; // 積分ゲインを変更
        resetControl();
        spd_prm = gp;
        int_prm = gi;
        DJ._print("[" + m + "," + n + "]:" +
                 " spd_prm=" + spd_prm + ", int_prm=" + int_prm);
        double[][] sampleData = getSampleData(pointNum);
        dataSet.add(sampleData); // 学習データを得る
      }
    }
    // DJ._print("dataNum=", dataSet.size());
    return dataSet;
  }
  
  
  // ---------------------------------------------------------------------------
  /**
   * 制御を実行
   */
  public void execute() {
    // システムクロック更新
    system_clock = system_clock + system_time;	// System Time Interval
    if (system_clock >= (double) 10000.0) {
      system_clock = D_ZERO;
    }

    // シミュレーション・カウント更新
    simulation_count = simulation_count + system_time;
    if (simulation_count >= simulation_time) {
      simulation_count = D_ZERO; // Simulation Counter
      simulate(); // 制御対象をシミュレーション
    }
    
    // 制御・解析	Control and Analyze
    sampling_count = sampling_count + system_time;
    // system Clock Interval
    if (sampling_count >= sampling_time) {
      sampling_count = D_ZERO; // Sampling Counter

      // フィードバック処理ルーチン
      FBC_PORT = fbc_buffer;
      if (feedbackFlag) {
        feedback();
      }

      // 速度制御ルーチン
      if (speedControlFlag) {
        speed_control();
      }

      // 操作量出力ルーチン
      output_dac();
      dac_buffer = DAC_PORT;
    }

    // 補間ルーチンと位置制御ルーチン
    interpolate_count = interpolate_count + system_time; // system Clock Interval
    if (interpolate_count >= interpolate_time) {
      interpolate_count = D_ZERO; // Interpolation Counter

//      // 補間ルーチン
//      if (list7_1) {
//        interpolate();
//      }

//      // 位置制御ルーチン
//      position_control();
    }

  }
  
  /**
   * 速度微分方程式	Speed Differencial Equation
   * @param spd double
   * @return spd_dif double
   */
  public double speed_equation(double spd) {
    double spd_dif; // Speed Differencial
    double dac_out; // DAC Output Voltage
    double current; // Current

    // Ｄ／Ａ変換器出力電圧	 e = (10/2048) * frc_mnp
    dac_out = (double) dac_buffer * DAC_FACTOR;
    // 電流 i = Gi * e
    current = inverter * dac_out; // Current
    // 発生力 tau = Tc * i
    frc_sim = frc_const * current; // Force
    // 速度微分値算出 Calculate Speed Differencial
    spd_dif = (frc_sim - (friction * spd)) / inertia; // Speed Differencial	

    return (spd_dif);
  }

  /**
   *  制御対象シミュレーション	Simulate Plant
   */
  public void simulate() {
    double k1, k2, k3, k4;

    // 速度シミュレーション	Speed Simulation
    k1 = simulation_time * speed_equation(spd_rng);
    k2 = simulation_time * speed_equation(spd_rng + (k1 / 2.0));
    k3 = simulation_time * speed_equation(spd_rng + (k2 / 2.0));
    k4 = simulation_time * speed_equation(spd_rng + k3);
    spd_var = ((k1 + (k2 * 2.0) + (k3 * 2.0) + k4) / 6.0); // Speed Variation
    spd_rng = spd_rng + spd_var;

    // 位置シミュレーション	Position Simulation
    k1 = simulation_time * (spd_rng);
    k2 = simulation_time * (spd_rng + (k1 / 2.0));
    k3 = simulation_time * (spd_rng + (k2 / 2.0));
    k4 = simulation_time * (spd_rng + k3);
    psn_rng = psn_rng + ((k1 + (k2 * 2.0) + (k3 * 2.0) + k4) / 6.0);

    // フィードバックカウンタ	Set Feedbacck Counter
    fbc_buffer = (long) (psn_rng / psn_factor / (double) FB_SCALE);
  }
  
  /**
   * フィードバック処理 Input Feedback
   */
  public void feedback() {
    psn_fed = FBC_PORT * FB_SCALE; // Position Feedback
    // 速度フィードバック算出 Calculate Speed Feedback
    spd_fed = psn_fed - psn_old;
    psn_old = psn_fed;
    // 加速度フィードバック算出	Calculate Acceleration Feedback
    acc_fed = spd_fed - spd_old;
    spd_old = spd_fed;
  }

  /**
   * 速度制御ルーチン Speed Control Routine
   */
  public void speed_control() {

//	if(mf.list2_2){
//		frc_mnp = spd_prm * (long)(sim.inertia) * spd_ref / tim_ref;
//	} // LIST2_2

//	if(mf.list6_1){
		// 速度誤差算出 Calculate Speed Error
		spd_err_old = spd_err;
		spd_err = spd_ref - spd_fed;
		// 高周波遮断フィルタ Filtering Operation
		spd_cmp = ( fil_prm	* ( spd_cmp - spd_err ) / 1000 ) + spd_err;
		// 積分補償 Integral Compensation
//		int_cmp = int_cmp + ( int_prm * spd_cmp );
		int_cmp = int_cmp + ( int_prm * spd_cmp );
		// 微分補償 Differential Compensation
		dif_cmp = dif_prm * ( spd_err - spd_err_old ) / 1000;
		// 速度ループゲイン（比例補償） Proportional Compensation

		frc_mnp = ( spd_cmp	+ ( int_cmp / 1000000 ) + dif_cmp ) * spd_prm / 1000;
        
        
//	} // LIST6_1
  
//    if (mf.list8_1) {
//      // 速度誤差算出 Calculate Speed Error
//      spd_err_old = spd_err;
//      spd_err = spd_mnp - spd_fed;
//      // 高周波遮断フィルタ Filtering Operation
//      spd_cmp = (fil_prm * (spd_cmp - spd_err) / 1000) + spd_err;
//      // 積分補償 Integral Compensation
//      int_cmp = int_cmp + (int_prm * spd_cmp);
//      // 微分補償 Differential Compensation
//      dif_cmp = dif_prm * (spd_err - spd_err_old) / 1000;
//      // 速度ループゲイン（比例補償） Proportional Compensation
//      frc_mnp = (spd_cmp + (int_cmp / 1000000) + dif_cmp)
//              * spd_prm / 1000;
//    } // LIST8_1
  }
  
  /**
   * Ｄ／Ａ変換器出力 Output D/A Converter
   */
  public void output_dac() {
	long	data;
	data = frc_mnp; // Force Manipulate
	if( data > (long)DAC_MAX ) data = (long)DAC_MAX; // D/A Maximum Output
	else if( data < (long)DAC_MIN ) data = (long)DAC_MIN;// D/A Minimum Output
	DAC_PORT = data; // D/A Converter Port
  }

  /**
   * 制御指令値補間 Interpolate Control Command
   */
  public void interpolate() {
	long	psn_dst;
	long	psn_stp;

    psn_dst = psn_ref - psn_cmd;
    if (psn_dst > 0) { // 前進 Forward
      psn_stp = spd_cmd * spd_cmd / acc_ref / 2;
      if (psn_dst > psn_stp) {
        if (spd_cmd < spd_ref) { // Accelaration
          acc_cmd = acc_ref;
          spd_cmd = spd_cmd + acc_cmd;
          if (spd_cmd > spd_ref) {
            spd_cmd = spd_ref;
            acc_cmd = L_ZERO;
          }
        }
      } else if (psn_dst <= psn_stp) {
        if (spd_cmd > acc_ref) { // Decelaration/
          acc_cmd = -acc_ref;
          spd_cmd = spd_cmd + acc_cmd;
          if (spd_cmd < acc_ref) {
            spd_cmd = acc_ref;
            acc_cmd = L_ZERO;
          }
        }
      }
      psn_cmd = psn_cmd + spd_cmd;
      if (psn_cmd > psn_ref) {
        psn_cmd = psn_ref;
        spd_cmd = L_ZERO;
        acc_cmd = L_ZERO;
      }
    } else if (psn_dst < 0) { // 後退 Backward
      psn_stp = spd_cmd * spd_cmd / acc_ref / 2;
      if (psn_dst < (-psn_stp)) {
        if (spd_cmd > (-spd_ref)) { // 加速 Accelaration
          acc_cmd = -acc_ref;
          spd_cmd = spd_cmd + acc_cmd;
          if (spd_cmd < (-spd_ref)) {
            spd_cmd = -spd_ref;
            acc_cmd = L_ZERO;
          }
        }
      } else if (psn_dst >= (-psn_stp)) {
        if (spd_cmd < (-acc_ref)) { // 減速 Decelaration	*/
          acc_cmd = acc_ref;
          spd_cmd = spd_cmd + acc_cmd;
          if (spd_cmd > (-acc_ref)) {
            spd_cmd = -acc_ref;
            acc_cmd = L_ZERO;
          }
        }
      }
      psn_cmd = psn_cmd + spd_cmd;
      if (psn_cmd < psn_ref) {
        psn_cmd = psn_ref;
        spd_cmd = L_ZERO;
        acc_cmd = L_ZERO;
      }
    } else {
      // 停止 Stop
      psn_cmd = psn_ref;
      spd_cmd = L_ZERO;
      acc_cmd = L_ZERO;
    }
  } // interpolate

  /**
   * 位置制御ルーチン Position Control Routine
   */
  public void position_control() {
//	if(mf.list2_3){
//    frc_mnp = psn_prm * (long)(sim.inertia) * 2 * psn_ref / tim_ref / tim_ref;
//	} // LIST2_3

//    if (mf.list8_1) {
//      psn_err = psn_cmd - psn_fed; // 位置差算出 Calculate Position Error
//      spd_mnp = psn_err * psn_prm / 1000;
//    } // LIST8_1
  }
  
} // Control

// EOF
