/*
 *  Title: DaiJa_V5 (Digital-Learning Aide Instrument by Java)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020, 2021, 2022
 */
package meta;

import java.util.ArrayList;
import util.DJ;
//import view.Console;

/**
 * <p> 表　題: Class: Domain</p>
 * <p> 説　明: 領域クラス</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2022</p>
 * <p> 作成日: 2022.11.18</p>
 */
public class Domain extends Entity {
  
  static int numOfDomain = 0; // 領域の個数
  protected int domainNo; // 領域番号
  
  protected ArrayList<Domain> childDomainList; // 子領域リスト
  protected ArrayList<Domain> parentDomainList; // 親領域リスト
 
//  private Console console; // コンソール
  
  private ArrayList<Domain> domainTree; // 領域系統図
  private Assets assets; // 資産
  
  /**
   * Domainクラス生成子
   */
  public Domain() {
    initDomain();
  }
  
  // Domainを初期化
  private void initDomain() {
    domainNo = numOfDomain++; // 領域の個数を更新
    entityName = "Domain" + domainNo; // 領域番号を設定
    
    childDomainList = new ArrayList<>(); // 子領域リスト
    parentDomainList = new ArrayList<>(); // 親領域リスト
    domainTree = new ArrayList<>(); // 領域系統図を生成
    domainTree.add(this); // 自分を先頭に置く
    
    assets = new Assets(); // 資産クラスを生成
    
    DJ.print_("Domain is created! --- ");
  }
  
  /**
   * 領域番号を設定する
   * ※領域番号を強制的に設定するので、使用は慎重に
   * @param domainNo int // 領域番号
   */
  public void setDomainNo(int domainNo) {
    this.domainNo = domainNo;
  }
  
  /**
   * 領域番号を得る
   * @return domainNo int // 領域番号
   */
  public int getDomainNo() {
    return domainNo;
  }

  /**
   * この領域を親領域とし、子領域リストに登録する
   * @param domain Domain // 被登録領域（子領域）
   */
  public void registerDomain(Domain domain) {
    childDomainList.add(domain); // この領域の子領域リストに登録する
    domain.parentDomainList.add(this); // この領域を親領域リストに登録する
    DJ.print("Domain is added!");
  }
  
  /**
   * クラス名と領域名を得る
   * @return 
   */
  public String whoAreYou() {
//    String who = getClass().getCanonicalName(); // meta.Domain
//    String who = getClass().getName(); // meta.Chaos, meta.Domain
//    String who = getClass().getTypeName(); // meta.Chaos, meta.Domain
//    String who = getClass().toGenericString(); // public class meta.Domain
//    String who = getClass().toString(); // meta.Chaos meta.Domain
    String who = getClass().getSimpleName(); // meta.Chaos, Domain
    String parent = "none";
    if (!parentDomainList.isEmpty()) 
        parent = parentDomainList.get(0).getEntityName();
    String child = "none";
    if (!childDomainList.isEmpty()) 
        child = childDomainList.get(0).getEntityName();
    return("Name:" + entityName + ", Class:" + who
            + ", Parent:" + parent + ", Child:" + child);
  }
  
  /**
   * DomainListを得る
   * @return world ArrayList＜Domain＞ // Domainのリスト
   */
  public ArrayList<Domain> getChildDomainList() {
    return childDomainList;
  }
  
  /**
   * この領域の先祖系統のリストを得る
   * 先祖系統のエッジは変化する可能性があるので都度作成する
   * @param  // 対象領域
   * @return ancestorList ArrayList< Domain> // 先祖系統のリスト
   */
  public ArrayList<Domain> getAncestorList() {
    ArrayList<Domain> ancestorList = new ArrayList<>();
    ancestorList = getAncestorDomain(this, ancestorList);
    return ancestorList;
  }
  
  /**
   * 先祖系統のリストを得る（再帰呼出）
   * @param aDomain Domain // 対象領域
   * @param ancestorList ArrayList<Domain> // 先祖系統のリスト
   * @return ancestorList ArrayList<Domain> // 先祖系統のリスト
   */
  private ArrayList<Domain> getAncestorDomain(
       Domain aDomain, ArrayList<Domain> ancestorList){
    
    if (aDomain.parentDomainList == null) { // 親領域リストはnull
      DJ._print("There are no parent-domain list.");
      DJ.print("Domain name:" + aDomain.getEntityName());
      return ancestorList;
    }
    
    DJ.print("Domain name:" + aDomain.getEntityName());
    if (parentDomainList.size() < 1) { // 親領域の個数はゼロ以下
      if (aDomain instanceof Chaos) { // 親領域リストはChaos
        DJ.print("This domain is Chaos.");
      }
      return ancestorList;
    }
    
    Domain parentDomain = aDomain.parentDomainList.get(0);
    if (parentDomain == null) { // 親領域はnull
      DJ._print("Parent domain is null.");
    }
    else { // 親領域はnullではない
      DJ.print("Parent-domain name:" + parentDomain.getEntityName());
      ancestorList.add(parentDomain); // 親領域を先祖系統のリストに追加
      if (parentDomain instanceof Chaos) { // 親領域リストはChaos
        DJ._print("Root domain is Chaos.");
      }
      else { // 親領域はChaosではない
        ancestorList = getAncestorDomain(parentDomain, ancestorList); // 再帰呼出
      }
    }
    
    return ancestorList;
  }
  
  /**
   * 全領域のリストを得る
   * @return allDomainList ArrayList< Domain> // 全領域リスト
   */
  public ArrayList<Domain> getAllDomain() {
    ArrayList<Domain> allDomainList = new ArrayList<>();
    
    // Chaosを得る
    ArrayList<Domain> AncestorList = getAncestorList();
    int size = AncestorList.size();
    Domain topDomain = AncestorList.get(size - 1);
    // 子領域を追加する（再帰呼出）
    allDomainList = appendChildDomain(topDomain, allDomainList);
    
    return allDomainList;
  }
  
  /**
   *  子領域を追加する（再帰呼出）
   * @param aDomain Domain // 対象領域
   * @param domainList ArrayList<Domain> // 全領域リスト
   * @return domainList ArrayList<Domain> // 全領域リスト
   */
  private ArrayList<Domain> appendChildDomain(
          Domain aDomain, ArrayList<Domain> domainList) {

    domainList.add(aDomain);
    
    if (aDomain.childDomainList == null) { // 子領域リストが無い
      DJ._print("Domain.appendChildDomain(): "
              + "Domain[" + aDomain.getEntityName() + "] is null.");
      return domainList;
    }
    
    int numOfChildren = aDomain.childDomainList.size(); // 子領域の個数
    if (numOfChildren < 1) { // 子領域の個数がゼロ以下
      DJ._print("Domain.appendChildDomain(): Domain["
              + aDomain.getEntityName() + "] has no child.");
      return domainList;
    }
    
    for (int i = 0; i < numOfChildren; i++) {
      Domain aChildDomain = aDomain.childDomainList.get(i);
//      domainList.add(aChildDomain);
      appendChildDomain(aChildDomain, domainList); // 再帰呼出 
    }
    
    return domainList;
  }
  
  
  // 未実装
  public ArrayList<Domain> getDomailTree() {
    
    return domainTree; // 領域系統図
  }
  
  
  
  /**
   * 領域リストの文字列化
   * @param aDomainList ArrayList< Domain> // 文字列化される領域
   * @return sb.toString() String // 領域リストの文字列
   */
  public static  String getDomainNames(ArrayList<Domain> aDomainList) {
    StringBuilder sb = new StringBuilder("Domain names:");
    sb.append(aDomainList.get(0).entityName);
    for (int i = 1; i < aDomainList.size(); i++) {
      sb.append(", ").append(aDomainList.get(i).entityName);
    }
    sb.append(".");
    return sb.toString();
  }
  
  /**
   * Domainの先祖系統をログに表示する
   */
  public void printAncestorList() {
    ArrayList<Domain> anAncestorList = getAncestorList();
    
    DJ.print_("先祖系統：");
    if (anAncestorList.size() < 1) {
      DJ.print("There are no ancestor.");
      return;
    }
    
//    printDomainList(anAncestorList);
    DJ.print(getDomainNames(anAncestorList));
  } 
  
  /**
   * 子領域系統のリストをログに表示する
   */
  public void printChildList() {    
    DJ.print_("子領域系統：");
    
    if (childDomainList.size() < 1) {
      DJ.print("There are no child.");
      return;
    }
    
//    printDomainList(childDomainList);
    DJ.print(getDomainNames(childDomainList));
  }
  
  /**
   * 全ての領域をログに表示する
   */
  public void printAllDomain() {
    ArrayList<Domain> allDomainList = getAllDomain();
    if (allDomainList == null) {
      DJ._print("Domain.printAllDomain(): All-Domain-List is null.");
      return;
    }
    if (allDomainList.size() < 1){
      DJ._print("Domain.printAllDomain(): There are no Domain.");
      return;
    }
    DJ.print_("全領域系統：");
//    printDomainList(allDomainList);
    DJ.print(getDomainNames(allDomainList));
  }
  
  
  
  /**
   * 資産を設定する
   * ※資産を強制的に設定するので、使用は慎重に
   * @param assets Assets // 資産
   */
  public void setAssets(Assets assets) {
    this.assets = assets;
  }
  
  /**
   * 資産を得る
   * @return assets Assets // 資産
   */
  public Assets getAssets() {
    return assets;
  }

  
  
  
  
  
  
} // Domain class

// EOF
