/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2025
 */
package quan;

import util.DJ;
import util.comp.CompMat;
import util.comp.CompVec;
import util.comp.Complex;

/**
 * <p> 表　題: Class: QuGate</p>
 * <p> 説　明: 量子ゲート</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024</p>
 * <p> 作成日: 2024.02.13</p>
 */
public class QuGate {
  
  public static final String I_GATE = "I_GATE"; // [[1 0],[0 1]] identity matrix
  public static final String X_GATE = "X_GATE"; // [[0 1],[1 0]] Not Gate
  public static final String Y_GATE = "Y_GATE"; // [[0 1],[-1 0]]
  public static final String Z_GATE = "Z_GATE"; // [[1 0],[0 -1]]
  public static final String H_GATE = "H_GATE"; // 1/√2[[1 1],[1 -1]] Hadamard Gate
  
  public static final String RX_GATE = "RX_GATE"; // [[cosθ/2 -isinθ/2],[-isinθ/2 cosθ/2]]
  public static final String RY_GATE = "RY_GATE"; // [[cosθ/2 -isinθ/2],[ isinθ/2 cosθ/2]]
  public static final String RZ_GATE = "RZ_GATE"; // [[exp(-iθ/2) 0],[0 exp(iθ/2)]]
  public static final String RK_GATE = "RK_GATE"; // [[1 0],[0 exp(i2π/2^(k+1)]]
  
  public static final String CX_GATE = "CX_GATE"; // Control-Not Gate
  public static final String C2X_GATE = "C2X_GATE"; // Control-Control Not Gate
  public static final String W_GATE = "W_GATE"; // QFT Gate: Quantum Fourier Transform
  
  public static final String SWAP_GATE = "SWAP_GATE"; // [[1 0 0 0],[0 0 1 0],[0 1 0 0],[0 0 0 1]] 

  public static final String N_TYPE = "NONCHALANT"; // Nonchalant Type
  public static final String C_TYPE = "CONTROL"; // Control Type
  public static final String X_TYPE = "X_GATE"; // Not Gate Type
  
  public static final Complex BASE_N = new Complex(); // 0.0+0.0i
  public static final Complex BASE_P0 = new Complex(1.0, 0.0); // |0>=1.0+0.0i
  public static final Complex BASE_M0 = new Complex(-1.0, 0.0); // -1.0+0.0i
  public static final Complex BASE_P1 = new Complex(0.0, 1.0); // |1>=0.0+1.0i
  public static final Complex BASE_M1 = new Complex(0.0, -1.0); // 0.0-1.0i

  public static final int NOP = 0; // 処理なし
  public static final int CONTROL = 1; // 制御ビット（複数）
  public static final int TARGET = 2; // 処理対象ビット（１ビットのみ）

  
  private CompMat gateMat; // ゲート行列（複素数行列）
  
  /**
   * 単位量子ゲートを生成する（１量子ビット、単位複素数行列）
   */
  public QuGate() {
    gateMat = new CompMat(2); // ２次元の単位複素数行列
  }
  
  /**
   * 単位量子ゲートを生成する（多量子ビット、単位複素数行列）
   * @param NumOfQuBit int // 量子ビットの個数
   */
  public QuGate(int NumOfQuBit) {
    int numOfProbAmp = QuBit.calcNumOfProbAmp(NumOfQuBit); // 確率振幅数
    gateMat = new CompMat(numOfProbAmp); // 単位複素数行列
  }
  
  /**
   * 量子ゲートを生成する（複素数行列）
   * ※　非正方行列のゲートの生成が可能
   * ※　行と列の個数は２のべき乗（2,4,8...）に制限される
   * @param row int // 行数
   * @param col  int // 列数
   */
  public QuGate(int row, int col) {
    if (!QuBit.checkPowerOfTwo(row) ){ // ２のべき乗ではない
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(int row, int col)" + "\n"
          + " row(=" + row + ") is not power of 2.");
      return;      
    }
    if (!QuBit.checkPowerOfTwo(col) ){ // ２のべき乗ではない
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(int row, int col)" + "\n"
          + " col(=" + col + ") is not power of 2.");
      return;      
    }
    gateMat = new CompMat(row, col); // 複素数行列
  }
  
  /**
   * 量子ゲートを生成する（行列の要素を複素数で指定）
   * @param NumOfQuBit int // 量子ビットの個数
   * @param element Complex // 要素を指定する複素数
   */
  public QuGate(int NumOfQuBit, Complex element) {
    int numOfProbAmp = QuBit.calcNumOfProbAmp(NumOfQuBit);
    gateMat = new CompMat(numOfProbAmp, element); // 複素数行列
  }
 
  /**
   * 量子ゲートを生成する（行列の要素を複素数ベクトルで指定）
   * ※　ベクトルの要素の個数は整数の平方数とする
   * @param element CompVec // 複素数ベクトル
   */
  public QuGate(CompVec element) {
    int length = element.length();
    double sq = Math.sqrt(length); // 複素数行列の次数
    long _dim = Math.round(sq);
    int numOfProbAmp = (int)_dim;
    if (_dim > Integer.MAX_VALUE) {
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(CompVec element)" + "\n"
          + " Length of element (=" + element 
          + ") is too big.");
      return;      
    }
    if ((!QuBit.checkPowerOfTwo(numOfProbAmp))) {
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(CompVec element)" + "\n"
          + " Length of element (=" + element 
          + ") is not power of two.");
      return;      
    }
    gateMat = new CompMat(numOfProbAmp);
    int index;
    for (int i = 0; i < numOfProbAmp; i++) {
      index = i * numOfProbAmp;
      for (int j = 0; j < numOfProbAmp; j++) {
        gateMat.setComp(i, j, element.copyComp(index + j));
      }
    }
  }
  
  /**
   * 量子ゲート生成子（ゲートラベル）
   *   ２量子ビット
   * @param gateLabel String // ゲートラベル
   */
  public QuGate(String gateLabel) {
    if (null == gateLabel) {
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(String gateLabel)" + "\n"
          + " Argument String (gateLabel) is null.");
      return;      
    }
    Complex zero = new Complex(); // 0.0+0.0i
    Complex one = new Complex(1.0, 0.0); // 1.0+0.0i

    switch (gateLabel) {
      case I_GATE:
        Complex[] compArrayI = {one, zero, zero, one};
        gateMat = new CompMat(compArrayI);
        break;
      case X_GATE:
        Complex[] compArrayX = {zero, one, one, zero};
        gateMat = new CompMat(compArrayX);
        break;
      case Y_GATE:
        Complex iOne = new Complex(0.0, 1.0); // 0.0+1.0i
        Complex _iOne = new Complex(0.0, -1.0); // 0.0-1.0i
        Complex[] compArrayY = {zero, _iOne, iOne, zero};
        gateMat = new CompMat(compArrayY);
        break;
      case Z_GATE:
        Complex _one = new Complex(-1.0, 0.0); // -1.0+0.0i
        Complex[] compArrayZ = {one, zero, zero, _one};
        gateMat = new CompMat(compArrayZ);
        break;
      case H_GATE:
        Complex oneRt2 = new Complex(1.0 / Math.sqrt(2.0), 0.0); // 1/√2
        Complex _oneRt2 = new Complex(-oneRt2.getReal(), 0.0); // -1/√2
        Complex[] compArrayH = {oneRt2, oneRt2, oneRt2, _oneRt2};
        gateMat = new CompMat(compArrayH);
        break;
      default:
        Complex[] compArrayNull = {zero, zero, zero, zero};
        gateMat = new CompMat(compArrayNull);
        break;
    }
  }
  
  /**
   * 量子ゲート生成子（ゲートラベル、実数パラメータ）
   *   ２量子ビット、
   * @param gateLabel String // ゲートラベル
   * @param parameter double // パラメータ
   */
  public QuGate(String gateLabel, double parameter) {
    if (null == gateLabel) {
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(String gateLabel, double parameter)" + "\n"
          + " Argument String (gateLabel) is null.");
      return;      
    }
    
    double halfTheta = parameter / 2.0;
    double sinTheat = Math.sin(halfTheta);
    double cosTheat = Math.cos(halfTheta);
    
    Complex zero = new Complex(); // 0.0+0.0i
    Complex one = new Complex(1.0, 0.0); // 1.0+0.0i
    Complex cos = new Complex(cosTheat, 0.0);
    Complex isin = new Complex(0.0, sinTheat);
    Complex _isin = new Complex(0.0, - sinTheat);
    
    switch (gateLabel) {
      case RX_GATE:
        Complex[] compArrayRX = {cos, _isin, _isin, cos};
        gateMat = new CompMat(compArrayRX);
        break;
      case RY_GATE:
        Complex sin = new Complex(sinTheat, 0.0);
        Complex _sin = new Complex(- sinTheat, 0.0);
        Complex[] compArrayRY = {cos, _sin, sin, cos};
        gateMat = new CompMat(compArrayRY);
        break;
      case RZ_GATE:
        // Complex cossin = new Complex(cosTheat, sinTheat);
        // Complex cos_sin = new Complex(cosTheat, - sinTheat);
        isin.add(cos);
        _isin.add(cos);
        Complex[] compArrayRZ = {_isin, zero, zero, isin};
        gateMat = new CompMat(compArrayRZ);
        break;
      case RK_GATE:
        double angle = Math.PI / Math.pow(2, parameter); // k:parameter
        cosTheat = Math.cos(angle);
        sinTheat = Math.sin(angle);
        Complex pk = new Complex(cosTheat, sinTheat);
        Complex[] compArrayRK = {one, zero, zero, pk};
        gateMat = new CompMat(compArrayRK);
        break;
      default:
        Complex[] compArrayNull = {zero, zero, zero, zero};
        gateMat = new CompMat(compArrayNull);
        break;
    }
    
  }

    
  /**
   * 量子ゲート生成子(ラベル、整数パラメータ)
   *   W_GATE：量子フーリエ変換用量子ゲート（多量子ビット）
   *   RK_GATE：位相ゲート（２量子ビット）
   * @param gateLabel String // ゲートラベル
   * @param parameter int // 量子ビット数、k:位相ゲートインデックス
   */
  public QuGate(String gateLabel, int parameter) {
    
    switch (gateLabel) {
      case W_GATE: // 量子フーリエ変換用量子ゲート
        gateMat = createWGate(parameter); // numOfQuBit:parameter
        break;
    
      case RK_GATE: // 位相ゲート（２量子ビット）
        Complex zero = new Complex(); // 0.0+0.0i
        Complex one = new Complex(1.0, 0.0); // 1.0+0.0i
        double angle = Math.PI / Math.pow(2, parameter);
        double cosTheat = Math.cos(angle);
        double sinTheat = Math.sin(angle);
        Complex pk = new Complex(cosTheat, sinTheat);
        Complex[] compArrayRK = {one, zero, zero, pk};
        gateMat = new CompMat(compArrayRK);
        break;
        
      default:
        DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(String gateLabel, int numOfQuBit)" + "\n"
          + " There are no such a gate(" + gateLabel + ").");
        break;
    }
  }
  
  /**
   * Ｗゲート（多次元、量子ビットタイプ配列）の要素行列を生成する
   * @param numOfQuBit int // 量子ビット数
   * @return wCompMat CompMat // Ｗゲートの要素行列
   */
  public final CompMat createWGate(int numOfQuBit) {
    int numOfProbAmp = QuBit.calcNumOfProbAmp(numOfQuBit);
    CompMat wCompMat = new CompMat(numOfProbAmp, numOfProbAmp);
    for (int k = 0; k < numOfProbAmp; k++) {
      for (int j = 0; j < numOfProbAmp; j++) {
        wCompMat.setComp(k, j, getWkj(numOfProbAmp, k, j));
      }
    }
    return wCompMat;
  }
  
  /**
   * Ｗゲートの行列の要素Wkjを生成する
   * @param numOfProbAmp int // 量子ビット数
   * @param k int // 行インデックス
   * @param j int // 列インデックス
   * @return 
   */
  public Complex getWkj(int numOfProbAmp, int k, int j) {
    double kj = 2.0 * Math.PI * k * j / numOfProbAmp; // 2πkj/N
    double real = Math.cos(kj);
    double imag = Math.sin(kj);
    Complex wkj = new Complex(real, imag);
    wkj.div(Math.sqrt(numOfProbAmp)); // wkj = wkj / √N
    
    return wkj;
  }
  
  
  // -----------------------------------------------------
  
  static void funcA() {
    QuGate.funcB();       
  }
  
  static void funcB()  {
    QuGate.funcA();   
  }
  
  /**
   * 多量子ビットの量子ゲートを生成する（ゲートラベルで指定、CX、　C_SWAP）
   * @param gateLabel String // ゲートラベル
   * @param numOfQuBit int // 量子ビット数
   * @param targets int[] // ターゲット量子ビットのインデックス配列
   * @param controls int[] // 制御ビットのインデックス配列
   */
  public QuGate(String gateLabel, int numOfQuBit, int[] targets, int[] controls) {
    if (gateLabel == null) { // 量子ゲートラベル
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(String gateLabel, int numOfBit,"
          + " int[] targets, int[] controls)" + "\n"
          + " Gate Label String (gateLabel) is null.");
      return;      
    }
    if (numOfQuBit <= 0) {
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(String gateLabel, int numOfBit,"
          + " int[] targets, int[] controls)" + "\n"
          + " Number of quBit (qbTypeArray =" + numOfQuBit 
          + ") is Less than eaqual to zero.");
      return;      
    }
       
    switch (gateLabel) {
      case CX_GATE:
        gateMat = createCXGate(numOfQuBit, targets, controls);
        break;
      case SWAP_GATE:
        gateMat = createSwapGate(numOfQuBit, targets, controls);
        break;
      default:
        Complex[] compArrayNull = {BASE_P0, BASE_M0, BASE_N, BASE_P1, BASE_M1};
        gateMat = new CompMat(compArrayNull);
        // DJ.print("# gateMat", gateMat);
        break;
    }
    
  }
  
  /**
   * Xゲートを生成する
   * @param numOfQuBit int // 量子ビット数
   * @param targets int[] // ターゲット量子ビットのインデックス
   * @param controls int[] // コントロール量子ビットのインデックス
   * @return cxCompMat CompMat // ゲート行列
   */
  public final CompMat createCXGate(
    int numOfQuBit, int[] targets, int[] controls) {
    DJ.print_("$ numOfQuBit = ", numOfQuBit);
    int numOfProbAmp = QuBit.calcNumOfProbAmp(numOfQuBit);
    DJ.print(",  $ numOfProbAmp = ", numOfProbAmp);
    CompMat cxCompMat = new CompMat(numOfProbAmp); // 標準ゲート

    int[][] binaly = makeBinalyArray(numOfQuBit); // ２進数マーカー
    // DJ._print("$ binaly = ", binaly);
    
    int numOfTarget = targets.length;
    DJ.print_("$ numOfTarget = ", numOfTarget);
    DJ.print(",  $ targetIndes = ", targets); // ターゲット量子ビットのインデックス
    
    int targetId0 = targets[0];
    // int targetId1 = targets[1];
    
    int numOfControl = controls.length;
    DJ.print_("$ numOfControl = ", numOfControl);
    if (numOfControl < 1) DJ.print("$ controlIndes = null");
    else DJ.print(",  $ controlIndes = ", controls); // コントロール量子ビットのインデックス
    
    int maxTag = (int)Math.pow(2, numOfQuBit - numOfTarget);
    int[] tag0 = new int[maxTag];
    int[] tag1 = new int[maxTag];
    int count0 = 0;
    int count1 = 0;
    
    if (numOfControl <= 0) { // コントロール無し
      for (int i = 0; i < numOfProbAmp; i++) {
        if (binaly[i][targetId0] == 0) {
         tag0[count0++] = i; 
        }
        else if (binaly[i][targetId0] == 1) {
         tag1[count1++] = i; 
        }
      }
    } // コントロール無し、ここまで
    else{ // コントロール有り
      int controlIndex = controls[0];
      DJ.print("$ controlIndex = ", controlIndex);
      for (int i = 0; i < numOfProbAmp; i++) {
        if (checkControl(i, controls, binaly)) { // コントロールが１
          if (binaly[i][targetId0] == 0) {
           tag0[count0++] = i; 
          }
          else if (binaly[i][targetId0] == 1) {
           tag1[count1++] = i; 
          }
        } // コントロールが１、ここまで
      }
    } // コントロール有り、ここまで

    // cxCompMatを設定する
    for (int k = 0; k < count0; k++) {
      cxCompMat.setComp(tag0[k], tag0[k], 0.0, 0.0);
      cxCompMat.setComp(tag0[k], tag1[k], 1.0, 0.0);
      cxCompMat.setComp(tag1[k], tag1[k], 0.0, 0.0);
      cxCompMat.setComp(tag1[k], tag0[k], 1.0, 0.0);
    }
     
    return cxCompMat;
  }
  
  /**
   * Swapゲートを生成する
   * @param numOfQuBit int // 量子ビット数
   * @param targets int[] // ターゲット量子ビットのインデックス
   * @param controls int[] // コントロール量子ビットのインデックス
   * @return cxCompMat CompMat // ゲート行列
   */
  public final CompMat createSwapGate(
    int numOfQuBit, int[] targets, int[] controls) {
    DJ.print_("$ numOfQuBit = ", numOfQuBit);
    int numOfProbAmp = QuBit.calcNumOfProbAmp(numOfQuBit);
    DJ.print(",  $ numOfProbAmp = ", numOfProbAmp);
    CompMat swapCompMat = new CompMat(numOfProbAmp); // 標準ゲート

    int[][] binaly = makeBinalyArray(numOfQuBit); // ２進数マーカー
    // DJ._print("$ binaly = ", binaly);
    
    int numOfTarget = targets.length;
    DJ.print_("$ numOfTarget = ", numOfTarget);
    DJ.print(",  $ targetIndes = ", targets); // ターゲット量子ビットのインデックス

    int targetId0 = targets[0];
    int targetId1 = targets[1];
    
    int numOfControl = controls.length;
    DJ.print_("$ numOfControl = ", numOfControl);
    if (numOfControl < 1) DJ.print("$ controlIndes = null");
    else DJ.print(",  $ controlIndes = ", controls); // コントロール量子ビットのインデックス
    
    int maxTag = (int)Math.pow(2, numOfQuBit - numOfTarget);
    int[] tag0 = new int[maxTag];
    int[] tag1 = new int[maxTag];
    int count0 = 0;
    int count1 = 0;
    

    if (numOfControl <= 0) { // コントロール無し
      for (int i = 0; i < numOfProbAmp; i++) {
        if ((binaly[i][targetId0] == 0) && (binaly[i][targetId1] == 1)) {
          tag0[count0++] = i; 
        }
        else if ((binaly[i][targetId0] == 1) && (binaly[i][targetId1] == 0)) {
          tag1[count1++] = i; 
        }
      }
    } // コントロール無し、ここまで
    else{ // コントロール有り
      int controlIndex = controls[0];
      DJ.print("$ controlIndex = ", controlIndex);
      for (int i = 0; i < numOfProbAmp; i++) {
        if (checkControl(i, controls, binaly)) { // コントロールが１
        if ((binaly[i][targetId0] == 0) && (binaly[i][targetId1] == 1)) {
           tag0[count0++] = i; 
          }
        else if ((binaly[i][targetId0] == 1) && (binaly[i][targetId1] == 0)) {
           tag1[count1++] = i; 
          }
        } // コントロールが１、ここまで
      }
    } // コントロール有り、ここまで
    
    // cxCompMatを設定する
    for (int k = 0; k < count0; k++) {
      swapCompMat.setComp(tag0[k], tag0[k], 0.0, 0.0);
      swapCompMat.setComp(tag0[k], tag1[k], 1.0, 0.0);
      swapCompMat.setComp(tag1[k], tag1[k], 0.0, 0.0);
      swapCompMat.setComp(tag1[k], tag0[k], 1.0, 0.0);
    }
    
    return swapCompMat;
  }
  
  /**
   * ２進数マーカーを生成する
   * @param numOfQuBit
   * @return 
   */
  private int[][] makeBinalyArray(int numOfQuBit) {
    int numOfProbAmp = QuBit.calcNumOfProbAmp(numOfQuBit);
    int[][] binalyMarker = new int[numOfProbAmp][numOfQuBit];
    
    int k;
    for (int i = 0; i < numOfProbAmp; i++) {
      k = i;
      for (int j = 0; j < numOfQuBit; j++) {
        binalyMarker[i][numOfQuBit - 1 - j] = k % 2;
        k = k / 2;
      }
    }
    
    return binalyMarker;
  }
  
  /**
   * 全てのコントロールが１、もしくはコントロールが無いことをチェックする
   * false：コントロールが0
   * @param controls int[] // コントロールの配列
   * @return false/true boolean // チェック結果
   */
  private boolean checkControl(int index, int[] controls, int[][] binaly) {
    boolean flag = true;
    int numOfControl = controls.length;
    if (numOfControl > 0) {
      for (int i = 0; i < numOfControl; i++) {
        if (binaly[index][controls[i]] == 0) {
          flag = false;
          break;
        }
      }
    }
    // DJ.print("$ control:" + Boolean.toString(flag));
    
    return flag;
  }
  
  
  // -----------------------------------------------------
  
  /**
   * 多量子ビットの量子ゲート（ゲートラベルで指定）を生成する
   * @param gateLabel String // ゲートラベル
   * @param numOfBit int // 量子ビット数
   * @param target int // ターゲット量子ビットインデクス
   * @param controls int[] // 制御ビット配列
   */
  public QuGate(String gateLabel, int numOfBit, int target, int[] controls) {
    if (gateLabel == null) { // 量子ゲートラベル
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(String gateLabel, int numOfBit, int target, int[] controls)" + "\n"
          + " Gate Label String (gateLabel) is null.");
      return;      
    }
    if (numOfBit <= 0) {
      DJ.print("***** ERROR ***** "
          + "QuGate.QuGate(String gateLabel, int numOfBit, int target, int[] controls)" + "\n"
          + " Number of quBit (qbTypeArray =" + numOfBit 
          + ") is Less than eaqual to zero.");
      return;      
    }
    
    switch (gateLabel) {
      case CX_GATE:
        int[] qbTypeArray = getQuBitTypeArray(numOfBit, target, controls);
        // DJ.print("# qbTypeArray", qbTypeArray);
        int[] gateMarker = getGateMarker(qbTypeArray);
        // DJ.print("# gateMarker", gateMarker);
        gateMat = createCXGate(gateMarker, controls);
        break;
      default:
        Complex[] compArrayNull = {BASE_P0, BASE_M0, BASE_N, BASE_P1, BASE_M1};
        gateMat = new CompMat(compArrayNull);
        // DJ.print("# gateMat", gateMat);
        break;
    }
    
  }
    
  /**
   * 多量子ビットのタイプ配列（制御ビット、標的ビット）を得る
   * 標的ビットに[1,2]'を設定、制御ビットに|1>=[0 1}'を設定し、
   * その他のビットに要素に奇数の値を設定する。例[3 5]'
   * @param numOfQuBit int // 量子ビット数
   * @param target int // ターゲットとなる量子ビットのインデックス
   * @param controls int[] // コントローラとなる量子ビットのインデックス
   * @return qbTypeArray int[] // 量子ビット設定配列
   */
  public static int[] getQuBitTypeArray(
          int numOfQuBit, int target, int[] controls) {
    int qbTypeLength = numOfQuBit * 2; // 量子ビットタイプ配列の長さ
    int[] qbTypeArray = new int[qbTypeLength]; // 量子ビットタイプ配列
    int cntNumber = controls.length; // 制御ビットの個数
    // すべてのビットに要素に奇数の値を設定
    for (int i = 0; i < qbTypeLength; i += 2) {
      qbTypeArray[i] = i * 2 + 1; // 3,7,11, ...
      qbTypeArray[i + 1] =  qbTypeArray[i] + 2; // 5,9,13, ...
    }
    // 制御ビットに|1>=[0 1]'を設定
    for (int i = 0; i < cntNumber; i++) {
      int cnt2 = controls[i] * 2;
      qbTypeArray[cnt2] = 0;
      qbTypeArray[cnt2 + 1] = 1;
    }
    // ターゲット・ビットに[1 2]'を設定
    int target2 = target * 2;
    qbTypeArray[target2] = 1;
    qbTypeArray[target2 + 1] = 2;
    
    return qbTypeArray;
  }
    
  /**
   * 多量子ビットゲートのマーカ配列を得る
   * @param quBitTypeArray int // 
   * @return gateMarker int [] // ゲートのマーカ配列
   */
  public static int[] getGateMarker(int[] quBitTypeArray) {
    int bitTypeLength = quBitTypeArray.length; // 量子ビットタイプ配列の長さ
    
    if (bitTypeLength % 2 != 0) {
      DJ.print("***** ERROR ***** "
          + "QuGate.getGateMarker(int[] quBitTypeArray)" + "\n"
          + " QuBit-Type-Array length (" + bitTypeLength
          + ") is not even.");
      return null;
    }
    
    int numOfQuBit = bitTypeLength / 2; // 量子ビット数
    int gateOrder = QuBit.calcNumOfProbAmp(numOfQuBit); // 量子ゲートの次数
    int[] gateMarker = new int[gateOrder]; // マーカー配列
    gateMarker[QuBit.PA0] = quBitTypeArray[bitTypeLength - 2]; // 最後から2番目の要素
    gateMarker[QuBit.PA1] = quBitTypeArray[bitTypeLength - 1]; // 最後の要素

    int mkLength = 1; // 暫定マーカ配列の長さ
    for (int i = numOfQuBit - 2; i >= 0; i--) { // 量子ビット数-1の繰り返し
      mkLength *= 2; // 確率振幅の長さはべき乗
      for (int k = 0; k < mkLength; k++) { // 確率振幅の長さの繰り返し
        // 後半の積
        int _pa1 = quBitTypeArray[i*2+1];
        _pa1 = _pa1 * gateMarker[k];
        gateMarker[mkLength + k] = _pa1;
        // 前半の積
        int _pa0 = quBitTypeArray[i*2];
        _pa0 = _pa0 * gateMarker[k];
        gateMarker[k] = _pa0;
      }
    }

    return gateMarker;
  }
  
  /**
   * ＣＸゲート（多次元、量子ビットタイプ配列）の要素行列を生成する
   * @param gateMaker int[] // ペア配列
   * @param controls int[] // 制御量子ビット配列
   * @return cXGateMat CompMat // ゲートの要素行列
   */
  public final CompMat createCXGate(int[] gateMaker, int[] controls) {
    int order = gateMaker.length; // ゲートの次数
    // DJ.print("$ order = ", order);
    CompMat cXGateMat = new CompMat(order); // 単位行列
    // DJ.printF("$ cXGateMat", cXGateMat);
    int numOfControl = controls.length; // 制御ビットの個数
    int numOfQuBit = QuBit.calcNumOfQuBit(order); // 量子ビット数
    
    double power = numOfQuBit -1 - numOfControl;
    int seek = (int) Math.pow(2.0, power); // 探索数
    
    if (seek <= 0) seek = 1; // 探索数
    // DJ.printF("$ seek", seek);
    
    int pairs[][] = new int[seek][2];
    int index = 0;
    for (int k = 0; k < order; k++) {
      if (gateMaker[k] != 0) {
        pairs[index][0] = k;

        for (int i = k + 1; i < order; i++) {
          if (gateMaker[i] == gateMaker[k] * 2) {
            pairs[index][1] = i;
            index++;
            break;
          }
        } // for(i)

      }
      if (index >= seek) {
        break;
      }
    } // for(k)

    // 量子ゲートにペアを設定する
    for (int k = 0; k < seek; k++) {
      cXGateMat.setComp(pairs[k][0], pairs[k][0], 0.0, 0.0);
      cXGateMat.setComp(pairs[k][0], pairs[k][1], 1.0, 0.0);
      cXGateMat.setComp(pairs[k][1], pairs[k][0], 1.0, 0.0);
      cXGateMat.setComp(pairs[k][1], pairs[k][1], 0.0, 0.0);
    }
      
    return cXGateMat;
  }

    
  /**
   * 多量子ビットのコントロール量子ゲートを生成する
   * 制御対象ゲートを１量子ビットの量子ゲートで指定する
   * @param targetGate QuGate // 対象量子ゲート（１量子ビット）
   * @param numOfBit int // 量子ビット数
   * @param target int // ターゲット量子ビットインデクス
   * @param controls int[] // 制御ビット配列
   */
  public QuGate(QuGate targetGate, int numOfBit, int target, int[] controls) {
  
    if (numOfBit < 1) {
      DJ.print("***** ERROR ***** QuGate.QuGate(QuGate targetGate,"
        + " int numOfBit, int target, int[] controls)" + "\n"
        + " Number of quBit (qbTypeArray =" + numOfBit + ") is Less than one.");
      return;      
    }
    
    int[] qbTypeArray = getQuBitTypeArray(numOfBit, target, controls);
    // DJ.print("# qbTypeArray", qbTypeArray);
    int[] gateMarker = getGateMarker(qbTypeArray);
    // DJ.print("# gateMarker", gateMarker);
    gateMat = createMultiGate(targetGate, gateMarker, controls);
    
  }
  
  /**
   * 多量子ビットのコントロール量子ゲートの要素行列を生成する
   * @param targetGate QuGate // 対象量子ゲート（１量子ビット）
   * @param gateMaker int[] // ペア配列
   * @param controls int[] // 制御量子ビット配列
   * @return cXGateMat CompMat // ゲートの要素行列
   */
  public final CompMat createMultiGate(
      QuGate targetGate, int[] gateMaker, int[] controls) {
    int order = gateMaker.length; // ゲートの次数
    // DJ.print("$ order = ", order);
    CompMat cXGateMat = new CompMat(order); // 単位行列
    // DJ.printF("$ cXGateMat", cXGateMat);
    int numOfControl = controls.length; // 制御ビットの個数
    int numOfQuBit = QuBit.calcNumOfQuBit(order); // 量子ビット数
    double power = numOfQuBit -1 - numOfControl;
    int seek = (int) Math.pow(2.0, power); // 探索数
    if (seek <= 0) seek = 1; // 探索数
    // DJ.printF("$ seek", seek);
    
    int pairs[][] = new int[seek][2];
    int index = 0;
    for (int k = 0; k < order; k++) {
      if (gateMaker[k] != 0) {
        pairs[index][0] = k;

        for (int i = k + 1; i < order; i++) {
          if (gateMaker[i] == gateMaker[k] * 2) {
            pairs[index][1] = i;
            index++;
            break;
          }
        } // for(i)

      }
      if (index >= seek) {
        break;
      }
    } // for(k)

    // 量子ゲートにペアを設定する
    for (int k = 0; k < seek; k++) {
      cXGateMat.setComp(pairs[k][0], pairs[k][0], targetGate.copyElement(0, 0));
      cXGateMat.setComp(pairs[k][0], pairs[k][1], targetGate.copyElement(0, 1));
      cXGateMat.setComp(pairs[k][1], pairs[k][0], targetGate.copyElement(1, 0));
      cXGateMat.setComp(pairs[k][1], pairs[k][1], targetGate.copyElement(1, 1));
    }
      
    return cXGateMat;
  }
  
  
  // -----------------------------------------------------
  // 特殊な量子ゲートを生成する静的メソッド makeXxxxxQuGate(argments)
  
   /** ※C-UKゲートはC-RKゲートと同じ
   * コントロール・ユニタリ量子ゲート（C-UKゲート）を生成する
   * @param numOfBit int // 量子ビット数
   * @param target int // ターゲット量子ビット
   * @param controls int[] // 制御量子ビット配列
   * @param parameter int // パラメータk
   * @return cRkt QuGate // コントロール回転量子ゲート
   */
  public static QuGate makeCUkQuGate(
      int numOfBit, int target, int[] controls, int parameter) {
 
      if (numOfBit <= 1) {
      DJ.print("***** ERROR ***** "
          + "QuGate.makeCUQuGate(int numOfBit, int target, int[] controls, int parameter)" + "\n"
          + " Number of quBit (" + numOfBit + ") is less than eaqual to one.");
      return null;      
    }
    
    QuGate cUkGate = new QuGate(numOfBit);

    int[] qbTypeArray = getQuBitTypeArray(numOfBit, target, controls);
    DJ.print("# qbTypeArray", qbTypeArray);
    int[] gateMarker = getGateMarker(qbTypeArray);
    DJ.print("# gateMarker", gateMarker);
    cUkGate.gateMat = cUkGate.createCUkQuGate(gateMarker, controls, parameter);
    DJ.print("# cUkGate.gateMat", cUkGate.gateMat);    
  
    return cUkGate;    
  }
  
  
  /**
   * コントロール・ユニタリ量子ゲート（C-UKゲート）の要素行列を生成する 
   * @param gateMaker int[] // ペア配列
   * @param controls int[] // 制御量子ビット配列
   * @param parameter int // パラメータk
   * @return  cXGateMat CompMat // ゲートの要素行列
   */
  public final CompMat createCUkQuGate(
      int[] gateMaker, int[] controls, int parameter) {
    
    int order = gateMaker.length; // ゲートの次数
    // DJ.print("$ order = ", order);
    CompMat cUkGateMat = new CompMat(order); // 単位行列
    // DJ.printF("$ cUkGateMat", cUkGateMat);
    int numOfControl = controls.length; // 制御ビットの個数
    int numOfQuBit = QuBit.calcNumOfQuBit(order); // 量子ビット数
       
    double power = numOfQuBit -1 - numOfControl;
    int seek = (int) Math.pow(2.0, power); // 探索数  
    
    if (seek <= 0) seek = 1; // 探索数
    // DJ.printF("$ seek", seek);
    
    int pairs[][] = new int[seek][2];
    int index = 0;
    for (int k = 0; k < order; k++) {
      if (gateMaker[k] != 0) {
        pairs[index][0] = k;

        for (int i = k + 1; i < order; i++) {
          if (gateMaker[i] == gateMaker[k] * 2) {
            pairs[index][1] = i;
            index++;
            break;
          }
        } // for(i)

      }
      if (index >= seek) {
        break;
      }
    } // for(k)
    
    // 量子ゲートの要素を求める
    double halfTheta = Math.PI / Math.pow(2, parameter);
    double sinTheat = Math.sin(halfTheta);
    double cosTheat = Math.cos(halfTheta);
    
    // 量子ゲートにペアを設定する
    for (int k = 0; k < seek; k++) {
      cUkGateMat.setComp(pairs[k][0], pairs[k][0], 1.0, 0.0);
      cUkGateMat.setComp(pairs[k][0], pairs[k][1], 0.0, 0.0);
      cUkGateMat.setComp(pairs[k][1], pairs[k][0], 0.0, 0.0);
      cUkGateMat.setComp(pairs[k][1], pairs[k][1], cosTheat, sinTheat);
    }
      
    return cUkGateMat;
  }

  /**
   * コントロール回転量子ゲート（C-RKゲート）を生成する
   * @param numOfBit int // 量子ビット数
   * @param target int // ターゲット量子ビット
   * @param controls int[] // 制御量子ビット配列
   * @param parameter int // パラメータk
   * @return cRkt QuGate // コントロール回転量子ゲート
   */
  public static QuGate makeCRkQuGate(
      int numOfBit, int target, int[] controls, int parameter) {
    if (numOfBit <= 1) {
      DJ.print("***** ERROR ***** "
          + "QuGate.makeCRkQuGate(int numOfBit, int target, int[] controls, int parameter)" + "\n"
          + " Number of quBit (" + numOfBit + ") is less than eaqual to one.");
      return null;      
    }
    
    QuGate cRk = new QuGate(numOfBit);
    int[] qbTypeArray = getQuBitTypeArray(numOfBit, target, controls);
    DJ.print("# qbTypeArray", qbTypeArray);
    int[] gateMarker = getGateMarker(qbTypeArray);
    DJ.print("# gateMarker", gateMarker);
    
    cRk.gateMat = cRk.createCRkQuGate(gateMarker, controls, parameter);
    // DJ.print("# cRk.gateMat", cRk.gateMat);
    
    return cRk;    
  }
  
  /**
   * コントロール回転量子ゲート（C-RKゲート）の要素行列を生成する 
   * @param gateMaker int[] // ペア配列
   * @param controls int[] // 制御量子ビット配列
   * @param parameter int // パラメータk
   * @return  cXGateMat CompMat // ゲートの要素行列
   */
  public final CompMat createCRkQuGate(
      int[] gateMaker, int[] controls, int parameter) {
    
    int order = gateMaker.length; // ゲートの次数
    DJ.print("$ order = ", order);
    CompMat cRkGateMat = new CompMat(order); // 単位行列
    // DJ.printF("$ cXGateMat", cXGateMat);
    int numOfControl = controls.length; // 制御ビットの個数
    int numOfQuBit = QuBit.calcNumOfQuBit(order); // 量子ビット数
       
    double power = numOfQuBit -1 - numOfControl;
    int seek = (int) Math.pow(2.0, power); // 探索数  
    
    if (seek <= 0) seek = 1; // 探索数
    // DJ.printF("$ seek", seek);
    
    int pairs[][] = new int[seek][2];
    int index = 0;
    for (int k = 0; k < order; k++) {
      if (gateMaker[k] != 0) {
        pairs[index][0] = k;

        for (int i = k + 1; i < order; i++) {
          if (gateMaker[i] == gateMaker[k] * 2) {
            pairs[index][1] = i;
            index++;
            break;
          }
        } // for(i)

      }
      if (index >= seek) {
        break;
      }
    } // for(k)
    
    // 量子ゲートの要素を求める
    double halfTheta = Math.PI / Math.pow(2, parameter);
    double sinTheat = Math.sin(halfTheta);
    double cosTheat = Math.cos(halfTheta);
    
    // 量子ゲートにペアを設定する
    for (int k = 0; k < seek; k++) {
      cRkGateMat.setComp(pairs[k][0], pairs[k][0], 1.0, 0.0);
      cRkGateMat.setComp(pairs[k][0], pairs[k][1], 0.0, 0.0);
      cRkGateMat.setComp(pairs[k][1], pairs[k][0], 0.0, 0.0);
      cRkGateMat.setComp(pairs[k][1], pairs[k][1], cosTheat, sinTheat);
    }
      
    return cRkGateMat;
  }
  
  /**
   * スワップ量子ゲート（Swapゲート）を生成する
   * @param numOfQuBit int // 量子ビット数
   * @param targets int[] // ターゲット量子ビット配列
   * @return swapGate QuGate // コントロール回転量子ゲート
   */
  public static QuGate makeSwapQuGate(int numOfQuBit, int[] targets) {
    if (numOfQuBit < 2) {
      DJ.print("***** ERROR ***** "
          + "QuGate.makeSwapQuGate(int numOfQuBit, int[] targets)" + "\n"
          + " Number of quBit (" + numOfQuBit + ") is less than two.");
      return null;      
    }
    
    int numOfTarget = targets.length;
    if (numOfTarget < 2) {
      DJ.print("***** ERROR ***** "
          + "QuGate.makeSwapQuGate(int numOfQuBit, int[] targets)" + "\n"
          + " Number of target (" + numOfTarget + ") is less than two.");
      return null;      
    }
    
    // CXゲート
    int target = targets[1]; // ターゲット量子ビットに第１ターゲット
    DJ.print_("#: numOfQuBits = " + numOfQuBit + ", control = " + target);
    int[] controlArray = {targets[0]}; // コントロール量子ビットに第０ターゲット
    DJ.print(", target", controlArray);
    QuGate cxGate = new QuGate(
        QuGate.CX_GATE, numOfQuBit, target, controlArray); // CXゲート
    DJ.printF("#; cxGate", cxGate);
    
    // XCゲート
    target = targets[0]; // ターゲット量子ビットに第０ターゲット
    DJ.print_("#: numOfQuBits = " + numOfQuBit + ", target0 = " + target);
    controlArray[0] = targets[1]; // コントロール量子ビットに第１ターゲット
    DJ.print(", target1", controlArray);
    QuGate xcGate = new QuGate(
        QuGate.CX_GATE, numOfQuBit, target, controlArray); // XCゲート
    DJ.printF("#; xcGate", xcGate);
    
    // 仮Swapゲート    
    QuGate swapGate = cxGate.product(xcGate);
    DJ.printF("#; Transient swapGate", swapGate);
    // Swapゲート    
    swapGate = swapGate.product(cxGate);
    DJ.printF("#; swapGate", swapGate);
    
    return swapGate;    
  }
  
  /**
   * コントロール・スワップ量子ゲート（CSwapゲート）を生成する
   * @param numOfQuBit int // 量子ビット数
   * @param targets int[] // ターゲット量子ビット配列
   * @param controls int[] // コントロール量子ビット配列
   * @return swapGate QuGate // コントロールSwap量子ゲート
   */
  public static QuGate makeCSwapQuGate(
      int numOfQuBit, int[] targets, int[] controls) {
    QuGate cSwapGate = new QuGate(numOfQuBit);
    
      int[] qbTypeArray = getSwapBitArray(numOfQuBit, targets, controls);
      DJ.print("# qbTypeArray", qbTypeArray);
      int[] gateMarker = getSwapGateMarker(qbTypeArray);
      DJ.print("# gateMarker", gateMarker);
      cSwapGate.gateMat = createSwapGate(gateMarker, controls);
  
    return cSwapGate;    
  }
  
  
  /**
   * スワップ・ゲートの量子ビット配列（制御ビットx1、標的ビットx2）を得る
   * 制御ビットに|1>=[0 1}'を設定、標的ビットに[3,5]'、[7,11]'を設定し、
   * ///// その他のビットに要素に奇数の値を設定する。例[3 5]'
   * その他のビットに[1 0]'を設定する。
   * @param numOfQuBit int // 量子ビット数
   * @param target int // ターゲットとなる量子ビットのインデックス
   * @param controls int[] // コントローラとなる量子ビットのインデックス
   * @return qbTypeArray int[] // 量子ビット設定配列
   */
  public static int[] getSwapBitArray(
          int numOfQuBit, int[] target, int[] controls) {
    int qbTypeLength = numOfQuBit * 2; // 量子ビットタイプ配列の長さ
    int[] qbTypeArray = new int[qbTypeLength]; // 量子ビットタイプ配列
    // すべてのビットに[1 0]'を設定
    for (int i = 0; i < qbTypeLength; i += 2) {
      qbTypeArray[i] =1;
      qbTypeArray[i + 1] = 0;
    }
    
    int cntNumber = controls.length; // 制御ビットの個数
    if (cntNumber > 0) {
      // 制御ビットに|1>=[0 1]'を設定
      for (int i = 0; i < cntNumber; i++) {
        int cnt2 = controls[i] * 2; // 制御ビットのインデックス
        qbTypeArray[cnt2] = 0;
        qbTypeArray[cnt2 + 1] = 1;
      }
    }
    // 標的ビットに[3,5]'、[7,11]'を設定
    int target0 = target[0] * 2;
    qbTypeArray[target0] = 3;
    qbTypeArray[target0 + 1] =5;
    int target1 = target[1] * 2;
    qbTypeArray[target1] = 7;
    qbTypeArray[target1 + 1] =11;
    
    return qbTypeArray;
  }
  
  
  /**
   * スワップ・ゲートのマーカ配列を得る
   * @param quBitTypeArray int // 
   * @return gateMarker int [] // ゲートのマーカ配列
   */
  public static int[] getSwapGateMarker(int[] quBitTypeArray) {
    int bitTypeLength = quBitTypeArray.length; // 量子ビットタイプ配列の長さ
    
    if (bitTypeLength % 2 != 0) {
      DJ.print("***** ERROR ***** "
          + "QuGate.getGateMarker(int[] quBitTypeArray)" + "\n"
          + " QuBit-Type-Array length (" + bitTypeLength
          + ") is not even.");
      return null;
    }
    
    int numOfQuBit = bitTypeLength / 2; // 量子ビット数
    int gateOrder = QuBit.calcNumOfProbAmp(numOfQuBit); // 量子ゲートの次数
    int[] gateMarker = new int[gateOrder]; // マーカー配列
    gateMarker[QuBit.PA0] = quBitTypeArray[bitTypeLength - 2]; // 最後から2番目の要素
    gateMarker[QuBit.PA1] = quBitTypeArray[bitTypeLength - 1]; // 最後の要素

    int mkLength = 1; // 暫定マーカ配列の長さ
    for (int i = numOfQuBit - 2; i >= 0; i--) { // 量子ビット数-1の繰り返し
      mkLength *= 2; // 確率振幅の長さはべき乗
      for (int k = 0; k < mkLength; k++) { // 確率振幅の長さの繰り返し
        // 後半の積
        int _pa1 = quBitTypeArray[i*2+1];
        _pa1 = _pa1 * gateMarker[k];
        gateMarker[mkLength + k] = _pa1;
        // 前半の積
        int _pa0 = quBitTypeArray[i*2];
        _pa0 = _pa0 * gateMarker[k];
        gateMarker[k] = _pa0;
      }
    }

    return gateMarker;
  }
  
  
  /**
   * スワップ・ゲート（多次元、量子ビットタイプ配列）の要素行列を生成する
   * @param gateMaker int[] // ペア配列
   * @param controls int[] // 制御量子ビット配列
   * @return cXGateMat CompMat // ゲートの要素行列
   */
  public static CompMat createSwapGate(int[] gateMaker, int[] controls) {
    int order = gateMaker.length; // ゲートの次数
    DJ.print("$ order = ", order);
    CompMat cSwapGateMat = new CompMat(order); // 単位行列
    // DJ.printF("$ cXGateMat", cXGateMat);
    
    int seek = 1; // 探索数
    int pairs[][] = new int[1][2];
    int index = 0;
    for (int k = 0; k < order; k++) {
      if (gateMaker[k] == 33) {
        pairs[index][0] = k;
        for (int i = k + 1; i < order; i++) {
          if (gateMaker[i] == 35) {
            pairs[index][1] = i;
            index++;
            break;
          }
        } // for(i)
      }
      else if (gateMaker[k] == 35) {
        pairs[index][0] = k;
        for (int i = k + 1; i < order; i++) {
          if (gateMaker[i] == 33) {
            pairs[index][1] = i;
            index++;
            break;
          }
        } // for(i)
      }
      if (index >= seek) {
        break;
      }
    } // for(k)
    
    // 量子ゲートにペアを設定する
    for (int k = 0; k < seek; k++) {
      cSwapGateMat.setComp(pairs[k][0], pairs[k][0], 0.0, 0.0);
      cSwapGateMat.setComp(pairs[k][0], pairs[k][1], 1.0, 0.0);
      cSwapGateMat.setComp(pairs[k][1], pairs[k][0], 1.0, 0.0);
      cSwapGateMat.setComp(pairs[k][1], pairs[k][1], 0.0, 0.0);
    }
      
    return cSwapGateMat;
  }
    
  // -----------------------------------------------------
  // getter, setter
  
  /**
   * 量子ゲート（複素数行列）の次数を得る
   * @return length int // 量子ゲートの次数
   */
  public int getOrder() {
    return gateMat.rowNumber();
  }
  
  /**
   * 量子ゲート（複素数行列）の行数を得る
   * @return  row int //  量子ゲートの行数
   */
  public int rowNumber() {
    return gateMat.rowNumber();
  }
  
  /**
   * 量子ゲート（複素数行列）の列数を得る
   * @return col int // 量子ゲートの列数
   */
  public int colNumber() {
    return gateMat.colNumber();    
  }
  
  /**
   * ゲート行列を得る
   * @return gateMat CompMat // ゲート行列
   */
  public CompMat getGateMat() {
    return gateMat; // 複素数行列
  }
  
  /**
   * ゲート行列を設定する
   * @param gateMat CompMat // ゲート行列
   */
  public void setGateMat(CompMat gateMat) {
    this.gateMat = gateMat;
  }
  
  /**
   * ゲートの要素を得る
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @return element Complex // 要素
   */
  public Complex getElement(int row, int col) {
    return gateMat.getComp(row, col);
  }
  
  /**
   * ゲートの要素を設定する
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @param element Complex // 要素
   */
  public void setElement(int row, int col, Complex element) {
    gateMat.setComp(row, col, element);
  }
  
  /**
   * ゲート行列を新たな複素数行列で得る
   * @return _gateMat CompMat // ゲート行列のコピー
   */
  public CompMat copyGateMat() {
    CompMat _gateMat = gateMat.copyCompMat();
    return _gateMat; // 複素数行列
  }
  
  /**
   * ゲートの要素を新規複素数で得る
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @return element Complex // 要素のコピー
   */
  public Complex copyElement(int row, int col) {
    return gateMat.copyComp(row, col);
  }
  
  /**
   * ゲートの要素を新規複素数で置き換える
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @param element Complex // コピーする要素
   */  
  public void copyElement(int row, int col, Complex element) {
    gateMat.setComp(row, col, element);
  }
  
  /**
   * ゲートの要素を複素数行列で得る
   * @return element Complex[][] // ゲートの要素
   */
  public Complex[][] getElement() {
    return gateMat.getMatrix(); // 複素数行列
  }
  
  /**
   * ゲートの要素を新たな複素数行列で得る
   * @return element Complex[][] // ゲートの要素
   */
  public Complex[][] copyElement() {
    return gateMat.copyMatrix(); // 複素数行列
  }
  
  // -----------------------------------------------------
  //   private CompMat gateMat; // ゲート行列（複素数行列）
  
  /**
   * 量子ゲートの共役転置を得る
   * @return conjuQuGate QuGate // 共役転置な量子ゲート
   */
  public QuGate conjugate() {
    int order = gateMat.rowNumber();
    int numOfQuBit = QuBit.calcNumOfQuBit(order);
    QuGate conjuQuGate = new QuGate(numOfQuBit); // ゲート行列
    CompMat conjuMat = conjuQuGate.gateMat;
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        conjuMat.setComp(j, i, gateMat.getComp(i, j).conjugate());
      }
    }
    return conjuQuGate;
  }
  
  
/**
   * 量子ゲートに量子ビットを適用し時間発展させる
   * @param operand QuBit // 
   * @return 
   */
  public QuBit apply(QuBit operand) {
    Complex[][] _gateMat = gateMat.getMatrix();
    Complex[] _qbVec = operand.getProbAmpArray();
    
    if (_gateMat[0].length != _qbVec.length) {
      DJ.print("***** ERROR ***** "
          + "QuGate.apply(QuBit operand)" + "\n"
          + " gate size (" + _gateMat[0].length
          + ") is not equal to QuBit size ("
          + _qbVec.length + ").");
      return null;
    }

    Complex[] result = Complex.product(_gateMat, _qbVec);
    QuBit resultQB = new QuBit(result);
    return resultQB;
  }
  
  /**
   * QuGateとQuGateの積を求める
   * ※被演算数行列の共役転置行列への置き換えは行われない
   * @param operand QuGate // 演算数複素数ベクトル
   * @return productedGate QuGate // 積:量子ゲート
   */
  public QuGate product(QuGate operand) {
    int row = gateMat.rowNumber();
    int col = gateMat.colNumber();
    int operandRow = operand.rowNumber();
    int operandCol = operand.colNumber();
    
    if (row != operandRow) {
      DJ.print("***** ERROR ***** "
          + "QuGate.innerProduct(QuGate operand)" + "\n"
          + " Element numbers are not same."
          + " target row number is " + row + ", "
          + " operand row number is " + operandRow + ".");
      return null;
    }
    
    QuGate productedGate = new QuGate(col, operandCol);
    CompMat productedMat = productedGate.gateMat;
    for (int i = 0; i < col; i++) {
      for (int j = 0; j < operandCol; j++) {
        for (int k = 0; k < row; k++) {
          Complex element = gateMat.copyComp(i, k); // 共役天使
          element.mul(operand.getElement(k, j));
          productedMat.getComp(i, j).add(element);
        }
      }
    }
    
    return productedGate;
  }
  
  /**
   * QuGateとQuGateの内積を求める
   * ※被演算数行列は共役転置行列に置き換えられる
   * @param operand QuGate // 演算数複素数ベクトル
   * @return productedGate QuGate // 内積:量子ゲート
   */
  public QuGate innerProduct(QuGate operand) {
    int row = gateMat.rowNumber();
    int col = gateMat.colNumber();
    int operandRow = operand.rowNumber();
    int operandCol = operand.colNumber();
    
    if (row != operandRow) {
      DJ.print("***** ERROR ***** "
          + "QuGate.innerProduct(QuGate operand)" + "\n"
          + " Element numbers are not same."
          + " target row number is " + row + ", "
          + " operand row number is " + operandRow + ".");
      return null;
    }
    
    // 被演算数複素数行列を共役転置行列に置き換える
    CompMat _conjugate = gateMat.conjugate();
    QuGate productedGate = new QuGate(col, operandCol);
    CompMat productedMat = productedGate.gateMat;
    for (int i = 0; i < col; i++) {
      for (int j = 0; j < operandCol; j++) {
        for (int k = 0; k < row; k++) {
          Complex conj = _conjugate.copyComp(i, k);
          conj.mul(operand.getElement(k, j));
          productedMat.getComp(i, j).add(conj);
        }
      }
    }
    
    return productedGate;
  }

  
  /**
   * 量子ゲート同士の相互作用により量子ゲートを生成する
   * @param operandGate QuGate // 相互作用させる量子ゲート
   * @return interactedGate QuGate // 相互作用による量子ゲート
   */
  public QuGate interactQuGate(QuGate operandGate) {
    int row0 = gateMat.rowNumber();
    int col0 = gateMat.colNumber();
    CompMat operandMat = operandGate.gateMat;
    int row1 = operandMat.rowNumber();
    int col1 = operandMat.colNumber();
    
    QuGate interactedGate = new QuGate(row0 * row1, col0 * col1);
    for (int i0 = 0; i0 < row0; i0++) {
      for (int j0 = 0; j0 < col0; j0++) {
        
        for (int i1 = 0; i1 < row1; i1++) {
          for (int j1 = 0; j1 < col1; j1++) {
            int i = i0 * row1 + i1;            
            int j = j0 * col1 + j1;
            Complex element = gateMat.copyComp(i0, j0);
            element.mul(operandMat.getComp(i1, j1));
            interactedGate.setElement(i, j, element);
          }
        }
        
      }
    }
    
    return interactedGate;
  }
  
  // -----------------------------------------------------
  // ドイッチュのアルゴリズム
  
  /**
   * ドイッチュのアルゴリズムに対応する仮想ゲート
   * @param functionType int // 関数タイプ
   * @param inputQuBit QuBit // 入力量子ビット（２量子ビット）
   * @return outputQuBit QuBit // 出力量子ビット（２量子ビット）
   */
  public static QuBit deutschGate(int functionType, QuBit inputQuBit) {
    CompVec inputProbAmpVec = inputQuBit.copyProbAmpVec(); // 入力量子ビットの確率振幅
    QuBit quBit0, quBit1; // 第０、１量子ビット（１量子ビット）
    int inCB0; // int inCB1; // 関数入力（古典ビット）
    QuBit outputQuBit; // 出力量子ビット

    //  第０項
    quBit0 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    quBit0.product(inputProbAmpVec.getComp(0)); // 第０項
    inCB0 = 0; // 関数入力（古典ビット）
    int qbVal = deutschFunction(functionType, inCB0); // 関数出力
    // DJ.print_("  $0 関数出力qbVal("+ functionType + ") = ", qbVal);
    if (qbVal == 0)
      quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else
      quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit = quBit0.interactQuBit(quBit1);
    
    // 第１項
    quBit0 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    quBit0.product(inputProbAmpVec.getComp(1)); // 第１項
    if (qbVal == 1)
      quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else
      quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit.sum(quBit0.interactQuBit(quBit1));
    
    // 第２項
    quBit0 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    quBit0.product(inputProbAmpVec.getComp(2)); // 第２項
    if (functionType > 1) qbVal = 1 - qbVal; // 関数出力を反転
    if (qbVal == 0)
      quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else
      quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit.sum(quBit0.interactQuBit(quBit1));
    
    // 第３項
    quBit0 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    quBit0.product(inputProbAmpVec.getComp(3)); // 第３項
    if (qbVal == 1)
      quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else
      quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit.sum(quBit0.interactQuBit(quBit1));
    
    return outputQuBit;
  }
  
  /**
   * ドイッチュのアルゴリズムに対応する仮想ゲート（初代）
   * @param functionType int // 関数タイプ
   * @param inputQuBit QuBit // 入力量子ビット（２量子ビット）
   * @return outputQuBit QuBit // 出力量子ビット（２量子ビット）
   */
  public static QuBit deutschGate_0(int functionType, QuBit inputQuBit) {
    CompVec inputProbAmpVec = inputQuBit.copyProbAmpVec(); // 入力量子ビットの確率振幅
    QuBit quBit0, quBit1;
    int inQB0; // int inQB1;
    QuBit outputQuBit; // 出力量子ビット

    //  第０項
    quBit0 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    quBit0.product(inputProbAmpVec.getComp(0)); // 第０項
    inQB0 = 0;
    int qbVal = deutschFunction(functionType, inQB0); // 関数出力
    DJ.print_("  $0 関数出力qbVal("+ functionType + ") = ", qbVal);
    if (qbVal == 0) quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else            quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit = quBit0.interactQuBit(quBit1);
    
    // 第１項
    quBit0 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    quBit0.product(inputProbAmpVec.getComp(1)); // 第１項
    DJ.print_(", ", qbVal); //  関数出力を表示
    if (qbVal == 1) quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else            quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit.sum(quBit0.interactQuBit(quBit1));
    
    // 第２項
    quBit0 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    quBit0.product(inputProbAmpVec.getComp(2)); // 第２項
    if (functionType > 1) qbVal = 1 - qbVal; // 関数出力を反転
    DJ.print_(", ", qbVal); //  関数出力を表示
    if (qbVal == 0) quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else            quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit.sum(quBit0.interactQuBit(quBit1));
    
    // 第３項
    quBit0 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    quBit0.product(inputProbAmpVec.getComp(3)); // 第３項
    DJ.print(", ", qbVal); //  関数出力を表示
    if (qbVal == 1) quBit1 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    else            quBit1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    outputQuBit.sum(quBit0.interactQuBit(quBit1));
    
    return outputQuBit;
  }
  
  
  /**
   * ドイッチュのアルゴリズムで使用する関数
   * @param functionType int // 関数タイプ
   * @param input int // 入力
   * @return 
   */
  public static int deutschFunction(int functionType, int input) {
    
    if ((input != 0) && (input != 1)) {
      DJ.print("***** ERROR ***** "
          + "QuGate.deutschOracle(int functionType, int input)" + "\n"
          + " Input value is not 0 nor 1."
          + " input value is " + input + ".");
      throw new IllegalArgumentException("引数 input の値が不正です。");
    }
    
    int output;
    
    switch (functionType) {
      case 0: // 定数型関数
        output = 0; // 入力に関わりなく常に０を返す
        break;
      case 1: // 定数型関数
        output = 1; // 入力に関わりなく常に１を返す
        break;
      case 2: // 均衡型関数
         output = input; // 入力と同じ値を返す
        break;
      case 3: // 均衡型関数
      default:
        output = 1 - input; // 入力と異なる値を返す
        break;
    }
    
    return output;
  }
  
  
  // ==================================================================
  // 全体を鍵括弧で括る（行列CompMatと同じ書式）
  // 確率振幅ごとにコンマで区切る
  
  /**  CompMatと同じ
   * 連続する空白文字列を得る
   * @param size int // 空白文字の個数
   * @return space.toString String // 空白文字列
   */
  private String getSpace(int size) {
    StringBuilder space = new StringBuilder(); // 表示文字列
    for (int i = 0; i < size; i++) {
      space.append(" "); // 空白を追加
    }
    return space.toString();
  }
  
  /**
   * 1.量子ゲートの文字列（行列表示）を得る
   * @return String // 複素数行列の文字列
   */
  @Override
  public String toString() {
    int row = gateMat.rowNumber(); // 行数（列の長さ）
    int col = gateMat.colNumber(); // 列数（行の長さ）
    StringBuilder sb = new StringBuilder("\n[");
    int index;
    for (int i = 0; i < row; i++) {
      sb.append("[");
      for (int j = 0; j < col; j++) {
        sb.append(gateMat.getComp(i, j).toString());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

  /** CompMatと同じ
   * 2.複素数行列の文字列（行列表示、枠・書式付き）を得る
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toFormat() {
    int row = gateMat.rowNumber(); // 行の長さ＝行内の要素数（列数）
    int col = gateMat.colNumber(); // 列の長さ＝列内の要素数（行数）
    int maxWide[] = new int[col]; // 数値文字列の長さの最大長 (最小例：-0+0i)
    for (int j = 0; j < col; j++) maxWide[j] = 5;
    
    // 数値文字列の抽出
    String[][] buf = new String[row][col]; // 文字列バッファ
    int[][] wides = new int[row][col]; // 余白数
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        // DJ.print_("i,j=", i);  DJ.print_(", ", j);
        String text = gateMat.getComp(i, j).toFormat();

        if (gateMat.getReal(i, j) >= 0) { // 実部が正
          buf[i][j] = " " + text;
          wides[i][j] = text.length() + 1; // 数値文字列の長さ
        } else {
          buf[i][j] = text;
          wides[i][j] = text.length(); // 数値文字列の長さ
        }
        // DJ.print(",  text=", text);
        // DJ.print_("wides[" + i + "][" + j + "]=", wides[i][j]);

        if (wides[i][j] > maxWide[j]) {
          maxWide[j] = wides[i][j]; // wideを更新
        }
        // DJ.print(",  maxWide[" + j + "]=", maxWide[j]);
      }
    }
    
    StringBuilder sb = new StringBuilder("F\n");
    int index;
    
    String filler = "  ";
    int gap = filler.length();
    int spaces = maxWide[0];
    for (int j = 1; j < col; j++) {
      spaces = spaces + gap + maxWide[j]; 
    }
    sb.append("┌").append(getSpace(spaces)).append("┐\n");
    
    for (int i = 0; i < row; i++) {
      sb.append("│");
      for (int j = 0; j < col; j++) {
        sb.append(buf[i][j]); // 数値
        sb.append(getSpace(maxWide[j] - wides[i][j])).append(filler); // 余白
      }
      index = sb.length();
      sb.replace(index - gap, index, "│\n");
    }
    sb.append("└").append(getSpace(spaces)).append("┘");
    return sb.toString();
  }
  
  /**
   * 3.量子ゲートの文字列（行列表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toString(String label) {
    int row = gateMat.rowNumber(); // 行数（列の長さ）
    int col = gateMat.colNumber(); // 列数（行の長さ）
    StringBuilder sb = new StringBuilder(label + ":\n[");
    int index;
    for (int i = 0; i < row; i++) {
      sb.append("[");
      for (int j = 0; j < col; j++) {
        sb.append(gateMat.getComp(i, j).toString());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }
  
  /**  CompMatと同じ
   * 4.量子ゲートの文字列（行列表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toFormat(String label) {
    String str = this.toFormat();
    
    return label + ":" + str;
  }

} // QuGate Class

// End of file

