/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package task;

import active.Activator;
import java.util.ArrayList;
import java.util.Collections;
import layer.ConvoluteLayer;
import layer.Layer;
import layer.LinearLayer;
import layer.PoolingLayer;
import util.DJ;
import util.LogEditor;
import util.TimeStamp;
import view.GraphViewer;
import view.GraphViewerLauncher;
import view.PatternViewer;
import view.PatternViewerLauncher;

/**
 * <p> 表　題: Class: ConvNet</p>
 * <p> 説　明: CNN（Convolutional Neural Network）による画像認識</p>
 * <p> 　　　: 手書き数字の判別、畳込層＋プーリング層＋中間層＋出力層</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.03.27</p>
 */
public class ConvNet extends Task {
  private boolean trialFlag = false; // 試行フラグ　true:試行時　false:学習時
  double[] inData; // 学習データ
  double[] teachData; // 正解データ
  
  // 学習・試行制御
  int epoch = 500; // 1000; // エポック回数
  int batchNum = 10; //15; //20; // バッチ数
  int interval = 5; //10; // 経過表示間隔
  
  // 学習パラメータ
  double initialCoef = 0.1; //0.01; // 重みとバイアスの初期値係数
  double filterCoef = 0.05; //0.01; // フィルタの初期値係数
  double eta = 0.01; // 学習係数
  double dropOutRate = 0.0; // 0.3; // ドロップ・アウト率 （0.0～0.5）
  // 畳込のパラメータ
  int filterNum = 10; //15; //20; // フィルタの枚数
  int filterHight = 3; // フィルタの高さ
  int filterWidth = filterHight; // フィルタの幅
  int filterStride = 1; // フィルタの移動量
  int padding = 0; // パデング量（畳込層用）
  // プーリングのパラメータ
  int poolingHight = 2; // プーリング枠の高さ
  int poolingWidth = poolingHight; // プーリング枠の幅
  
  // 各層のノード数
  // 入力層
  int inImageHight = 8; // 入力画像の高さ
  int inImageWidth = inImageHight; // 入力画像の幅
  int inNodeNum = inImageHight * inImageWidth; // 入力層のノード数（8x8）
  // 畳込層
  private final int convoNodeHight =
      (inImageHight - filterHight) / filterStride + 1; // 畳込ノードの高さ
  private final int convoNodeWidth = 
      (inImageWidth - filterWidth) / filterStride + 1; // 畳込ノードの幅
  private final int convoNodeNum = filterNum *
      convoNodeHight * convoNodeWidth; // 畳込層の出力ノード数
  // プーリング層
  private final int poolNodeHight =
      (convoNodeHight + padding * 2) / poolingHight; // ノードの高さ
  private final int poolNodeWidth =
      (convoNodeWidth + padding * 2) / poolingWidth; // ノードの幅
  private final int poolNodeNum = filterNum *
      poolNodeHight * poolNodeWidth; // プーリング層の出力ノード数
  // 中間層
  private final int midNodeNum = poolNodeNum; // 中間層のニューロン数
  // 出力層
  private final int outNodeNum = 10; // 出力層のニューロン数
  
  // レイヤー構成
  private ConvoluteLayer convoLayer0; // 畳込層０
  private PoolingLayer poolLayer0; // プーリング層０
  private LinearLayer middleLayer0; // 中間層０
  private LinearLayer middleLayer1; // 中間層１
  private LinearLayer outputLayer; // 出力層
  
  // 正答率の算出
  int trialCount = 0; // 総試行回数
  int correctCount = 0; // 総正答回数
  double correctLevel = 0.5; //0.9; // 正答判定値　0.5～1.0
  double correctRatio; // 正答率
                
  // デバッグ用実行時間計測用変数
//**/  private long startDebug = 0; // 実行開始時刻
//**/  private long endDebug = 0; // 実行終了時刻
//**/  private double lapDebug = 0; // エポック実行時間
//**/  private double totalDebug = 0; // 実行時間
  
  // パターン・ビューワ
  boolean patternViewerFlag; // true:表示、false:非表示
  PatternViewerLauncher patternViewerLauncher; // ランチャ
  PatternViewer patternViewer; // パターン・ビューワ（※名称変更不可）
  double[][][] patternData0, patternData1; // データ
  // グラフ・ビューワ
  int graphShift = 0; // グラフの縦軸を移動
  GraphViewerLauncher graphViewerLauncher; // ランチャ
  GraphViewer graphViewer; // グラフ・ビューワ （※名称変更不可）
  double[] graphData; // データ
  String[] dataName; // データ名
  
  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternData0 = new double[1][10][2]; // 出力と正解（599×10×２）
    patternData1 = new double[filterNum][filterHight][filterWidth]; // フィルタ
    patternViewerLauncher = DJ.pattern(
        PatternViewer.PATTERN_JUDGE, patternData0, "ConvNet0",  // 判定パターンとデータ
        PatternViewer.PATTERN_CHECK, patternData1, "ConvNet1"); // 手書き文字の画像とフィルタ
    
    // グラフ・ビューワー・ランチャーを得る
//    graphViewer.graphShift = 2; // グラフの縦軸を移動
    graphData = new double[4];
    dataName = new String[4]; // データ名
    dataName[0] = "LearnError"; // 学習時の二乗誤差
    dataName[1] = "LearnEntropy"; // 学習時の交差エントロピー誤差
    dataName[2] = "TrialError"; // 試行時の二乗誤差
    dataName[3] = "TrialEntropy"; // 試行時の交差エントロピー誤差
    graphViewerLauncher = DJ.graph(epoch, interval, dataName, graphData);
    
    // ＣＮＮによる画像認識（手書き数字の判別）
    convNet();
  }
  
  // ＣＮＮによる画像認識（手書き数字の判別）
  public void convNet() {
    
    DJ._print("ConvNet.convNet() ============================");
    DJ._print(" コンボリューショナル・ニューラル・ネットワークによる画像認識");
    
    DJ._print("・パラメータ");
    DJ.print_(" 重みとバイアスの初期値係数:initialCoef=", initialCoef);
    DJ.print_(" フィルタの初期値係数:filterCoef=", filterCoef);
    DJ.print_(", 学習係数:eta=", eta);
    DJ.print_(" エポック回数:epoch=", epoch);

    DJ.print_(", バッチ数:batchNum=", batchNum);
    DJ.print(", ドロップ・アウト率:dropOutRate=", dropOutRate);
    DJ.print(", 経過表示間隔:interval=", interval);
    
    DJ._print("・ニューラルネットの構成");
    DJ.print_(" 入力層ノード数:inNodeNum=",inNodeNum);
    DJ.print(", 畳込層ノード数:convoNodeNum=",convoNodeNum);
    DJ.print(" プーリング層ノード数:poolNodeNum=",poolNodeNum);
    DJ.print_(" 中間層ノード数:midNodeNum=",midNodeNum);
    DJ.print(", 出力層ノード数:outNodeNum=",outNodeNum);

    DJ._print("・畳み込みとプーリングの構成");
    DJ.print_(" ファイル数:filterNum=", filterNum);
    DJ.print_(", フィルタの高さ:filterHight=", filterHight);
    DJ.print(", フィルタの幅:filterHight=", filterWidth);
    DJ.print_(" フィルタの移動量:filterStride=", filterStride);
    DJ.print(", パデング量（畳込層用）:padding=", padding);
    DJ.print_(" プーリング枠の高さ:poolingHight=", poolingHight);
    DJ.print(", プーリング枠の幅:poolingWidth=", poolingWidth);
    
    DJ._print("・入力画像をファイルから読込");
    DJ.print(" DaiJa/resorce/DLearn/input_image.txt, コンマ区切り, 1797[枚]×64[画素]");
    double[][] inputImage = LogEditor.loadData("input_image.txt");
//    DJ.print( 入力画像："inputImage", inputImage); // 入力イメージ

    int inputImageNum = inputImage.length; // 入力画像の枚数
    int imageElementNum = inputImage[0].length; // 入力画像の画素数
    DJ.print_(" 入力画像の枚数=" + inputImageNum);
    DJ.print(",  入力画像の要素数=" + imageElementNum);
    double average = DJ.average(inputImage); // 平均
    double stdDev = DJ.stdDev(inputImage); // 標準偏差
    DJ.print(" 入力画像データの平均: average = inputImage.average()=", average);
    DJ.print(" 標準偏差: stdDev = inputImage.stdDev()=", stdDev);
    
    DJ._print("・入力画像データの正規化：平均値 = 0, 標準偏差 = 1");
    double[][] inputImageStd = DJ.normalize(inputImage); // 正規化
//    DJ.print(" 正規化された入力画像データ", inputImageStd);

    double averageNomal = DJ.average(inputImageStd); // 正規化データの平均
    double stdDevNomal = DJ.stdDev(inputImageStd); // 正規化データの準偏差
    DJ.print_(" 正規化された入力画像データの平均:averageNomal=", averageNomal);
    DJ.print(",  標準偏差=", stdDevNomal);
    
    DJ._print("・正解文字列をファイルから読込");
    DJ.print(" プロジェクト・フォルダ/resorce/target_text.txt,"
            + " 整数一桁, コンマ区切り, 1797[文字]");
    int[][] targetText = LogEditor.loadIntData("target_text.txt");
    
    int targetNum = targetText[0].length; // 正解文字の個数
    DJ.print(" 正解文字列の長さ:targetNum=", targetNum);
//    DJ.print("正解文字列：targetText", targetText);
    
    DJ._print("・正解データをone-hot表現で生成, 10[行]×1797[列]");
    double[][] correctData = new double[targetNum][outNodeNum];
    for (int i = 0; i < targetNum; i++) {
      for (int j = 0; j < outNodeNum; j++) {
        correctData[i][j] = 0.0; // one-hot表現
      }
      int k = targetText[0][i];
      correctData[i][k] = 1.0; // one-hot表現
    }
//    DJ.print(" 正解データをone-hot表現:teachData=", teachData);
     
    DJ._print("・ランダムなインデックスを生成, 1797[個]");
    DJ.print(" ランダム・インデックス数:inputImageNum=", inputImageNum);
    ArrayList<Integer> randomIndexList = DJ.permutationRandom(inputImageNum);
    int randomListNum = randomIndexList.size();
    DJ.print(" ランダム・リスト長:ranListNum=", randomListNum);
//    DJ.printList(" インデックス・リスト:indexTensor = ", randomIndexList);
    
    DJ.print(" ランダムなインデックスを2:1で学習用と試行用に割り振り");
    ArrayList<Integer> learnIndexList = new ArrayList<>();
    ArrayList<Integer> trialIndexList = new ArrayList<>();
    for (int i = 0; i < randomListNum; i++) {
      if (i % 3 != 0) learnIndexList.add(randomIndexList.get(i));
      else            trialIndexList.add(randomIndexList.get(i));
    }
    int learnNum = learnIndexList.size(); // 学習用データ数
    int trialNum = trialIndexList.size(); // 試行用データ数
    DJ.print_(" 学習用データ数:learnNum=", learnNum);
    DJ.print(", 試行用データ数:trialNum=", trialNum);
//    DJ.printList(" 学習用インデックス:learnIndexList=", learnIndexList);
//    DJ.printList(" 試行用インデックス:trialIndexList=", trialIndexList);

    // DJ._print("学習データ");
    inData = new double[inNodeNum];
//    DJ.print("inData", inData);
    
    // DJ._print("正解データ");
    teachData = new double[outNodeNum];
    
//    DJ.print("teachData", teachData);
  
    DJ._print("・ニューラルネットの各層の初期化");
    convoLayer0 = new ConvoluteLayer(inImageHight, inImageWidth);
    convoLayer0.initialize(eta, initialCoef,
            filterNum, filterHight, filterWidth, filterStride,
            Layer.ADA_GRAD);
//    ArrayList<double[][]> convo0xList = convoLayer0.get_xList();
    ArrayList<double[][]> convo0yList = convoLayer0.get_yList();
//    ArrayList<double[][]> convo0dEdXList = convoLayer0.get_dEdXList();
//    ArrayList<double[][]> convo0dEdYList = convoLayer0.get_dEdYList(); // 参照を代入（注５）
    
    poolLayer0 = new PoolingLayer(convoNodeHight, convoNodeWidth);
    poolLayer0.initialize(filterNum, padding, poolingHight, poolingWidth);
//#    ArrayList<double[][]> pool0xList = poolLayer0.get_xList(); // 参照を代入（注６）
//    ArrayList<double[][]> pool0yList = poolLayer0.get_yList();
    ArrayList<double[][]> pool0dEdXList = poolLayer0.get_dEdXList();
//    ArrayList<double[][]> pool0dEdYList = poolLayer0.get_dEdYList();
    
    middleLayer0 = new LinearLayer(poolNodeNum, midNodeNum);
    middleLayer0.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
//    double[] mid0X = middleLayer0.getX();
    double[] mid0Y = middleLayer0.getY();
//    double[] mid0C = middleLayer0.getC();
//    double[] mid0E = middleLayer0.getE();
//    double[] mid0dEdX = middleLayer0.getdEdX();
//#    double[] mid0dEdY = middleLayer0.getdEdY(); // 参照を代入（注１）
    
    middleLayer1 = new LinearLayer(midNodeNum, midNodeNum);
    middleLayer1.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
//#    double[] mid1X = middleLayer1.getX(); // 参照を代入（注３）
    double[] mid1Y = middleLayer1.getY();
//    double[] mid1C = middleLayer1.getC();
//    double[] mid1E = middleLayer1.getE();
    double[] mid1dEdX = middleLayer1.getdEdX();
//#    double[] mid1dEdY = middleLayer1.getdEdY(); // 参照を代入（注２）

    outputLayer = new LinearLayer(midNodeNum, outNodeNum);
///    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY); // 恒等関数
    outputLayer.initialize(eta, initialCoef, Activator.SOFTMAX, Layer.ADA_GRAD);
//#    double[] outX = outputLayer.getX(); // 参照を代入（注４）
    double[] outY = outputLayer.getY();
//    double[] outC = outputLayer.getC();
//    double[] outE = outputLayer.getE();
    double[] outdEdX = outputLayer.getdEdX();
//    double[] outdEdY = outputLayer.getdEdY();

    middleLayer0.setdEdY(mid1dEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
    middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
    middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
    outputLayer.setX(mid1Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定

    convoLayer0.set_dEdYList(pool0dEdXList); // （注５）畳込層のdEdYListへ、プーリングのdEdXListへの参照を設定
    poolLayer0.set_xList(convo0yList); // （注６）プーリング層の入力へ、畳込層の出力への参照を設定
    
    // PatternViewerへの参照を得る
    patternViewer = patternViewerLauncher.getPatternViewer();
    // GraphViewerへの参照を得る
    graphViewer = graphViewerLauncher.getGraphViewer();
    if (graphViewer != null) graphViewer.shiftGraphAxis(1); // グラフの縦軸をシフトする
    
    // DJ._print("・学習誤差と試行誤差の計測用データ");
    double learnErrorSum; // 学習時の二乗誤差
    double learnEntropySum; // 学習時の交差エントロピー誤差
    
    // DJ._print("・試行誤差の計測用データ");
    double trialErrorSum; // 試行時の差分二乗誤差
    double trialEntropySum; // 試行時の交差エントロピー誤差
    
    DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | (i == epoch); //経過表示フラグ
      
      if (intervalFlag) {
        trialCount = 0; // 総試行回数（正答率の算出用）
        correctCount = 0; // 総正答回数（正答率の算出用）
      }
    
      // DJ.print("・学習用と試行用のインデックスをそれぞれ別々にシャッフル");
      Collections.shuffle(learnIndexList);
      Collections.shuffle(trialIndexList);
      
      // DJ._print("・学習誤差と試行誤差の初期化");
      learnErrorSum = 0.0; // 学習時の二乗誤差
      learnEntropySum = 0.0; // 学習時の交差エントロピー誤差
      trialErrorSum = 0.0; // 試行時の二乗誤差
      trialEntropySum = 0.0; // 試行時の交差エントロピー誤差

      // DJ.print("・パターン・ビューワ用データの設定");
      patternData0 = new double[trialNum][10][2]; // ID、0-10、出力と正解

      // 全データを学習
      for (int j = 0; j < learnNum; j++) {
      // DJ._print("・エポックの実行ループ開始 -----------------------------");
      // DJ.print(" エポック:i=" + i + ", 学習：j=" + j);
        trialFlag = false; // 試行フラグ　true:試行時　false:学習時
        
        // DJ._print(" ##### 前向き伝播の呼び出し #####");
        // DJ._print("・正規化された入力画像から入力データを取り出し");
        int learnIndex = learnIndexList.get(j); // ランダム・インデックス
        // DJ._print(" 学習用データID：learnIndex=", learnIndex);
        inData = inputImageStd[learnIndex]; // 入力データ
        
        // DJ._print(" 入力データ：inData", inData);
        forward(inData); // 中間層と出力層の順伝播処理を呼び出し

        // DJ._print(" ##### 後向き伝播の呼び出し #####");
        // DJ._print("・one-hot表現された正解データを取り出し");
        teachData = correctData[learnIndex]; // 正解データ
        // DJ.print("教師データ:teachData", teachData);
          
        // DJ._print(" 教師データ：teachData", teachData);
        backward(teachData); // 中間層と出力層の逆伝播処理を呼び出し
        
        // DJ._print("・バッチ処理");
        if ((j % batchNum) == 0 ) {
          // DJ._print(" ##### 更新の呼び出し #####");
          update(); // 中間層と出力層のバイアスと重みの更新の呼び出し
        }
        
        // DJ._print(" ##### 学習時の誤差の保存 #####");
        // 差分二乗誤差の総和
        learnErrorSum += + DJ.getSquareError(outY, teachData);
        // 交差エントロピー誤差の総和
        learnEntropySum += DJ.getEntropyError(outY, teachData);

        // DJ._print(" 試行データによる実行（前向き処理のみ） #####");
        if (j < trialNum) {
          trialFlag = true; // 試行フラグ　true:試行時　false:学習時
          
          // DJ._print(" 試行データでの順伝播の呼び出し");
          // DJ._print("・正規化された試行画像から入力データを取り出し");
          int trialIndex = trialIndexList.get(j); // ランダム・インデックス
          inData = inputImageStd[trialIndex]; // 入力データ
          forward(inData); // 中間層と出力層の順伝播処理を呼び出し
          
          // DJ._print("・one-hot表現された正解データを取り出し");
          teachData = correctData[trialIndex]; // 正解データ

          // DJ._print("・試行データでの誤差の算出");
          trialErrorSum += DJ.getSquareError(outY, teachData);
          trialEntropySum += DJ.getEntropyError(outY, teachData);
          
          if (intervalFlag) { // 実行時間に影響を与える
            // PatternViewerに渡すデータを代入
            for (int k = 0; k < 10; k++) {
              patternData0[j][k][0] = outY[k]; // 出力
              patternData0[j][k][1] = teachData[k]; // 正解
            }
        
            // 正答回数の計測
            trialCount++; // 総試行回数
            int correctNumber = targetText[0][trialIndex]; // 正解数字
            double output = outY[correctNumber];
            if (output >= correctLevel){ // 正答判定値　0.5～1.0
              correctCount++; // 総正答回数
            }
          } // intervalFlag

          trialFlag = false; // 試行フラグ　true:試行時　false:学習時
        } // trial
        
        // DJ._print(" End of one data ---------------");
      } // 全データを学習
      
      // DJ._print(" End of one epoch ---------------");

      // DJ._print("　エポック毎の誤差の平均値を保存する");
      double learnErrorVal = Math.sqrt(learnErrorSum / learnNum);
      double learnEntropyVal = learnEntropySum / learnNum;
      double trialErrorVal = Math.sqrt(trialErrorSum / trialNum);
      double trialEntropyVal = trialEntropySum / trialNum;
      
      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加
      
       // 経過表示インターバル
      if (intervalFlag) {
         DJ.print_(" i=" + i);
        
        // 正答率の算出と表示
        correctRatio = (double)correctCount / (double)trialCount * 100.0;
//#        DJ.print_("総試行回数=", trialCount); // 総試行回数
//#        DJ.print_(", 総正答回数=", correctCount); // 総正答回数
        DJ.print_(", 正答比率=", correctRatio + "[%]"); // 正答率
        
        // patternViewerへフィルタ・データを渡す
        ArrayList<double[][]> filterWeightList =
                convoLayer0.getFilterWeightList(); // フィルタの重み
        for (int k = 0; k < filterNum; k++) {
          for (int m = 0; m < filterHight; m++) {
            for (int n = 0; n < filterWidth; n++) {
              double[][] filterWeight = filterWeightList.get(k);
              patternData1[k][m][n] = filterWeight[m][n];
            }
          }
        } // k
        
        // パターンの更新
        if (patternViewer == null) {
          patternViewer = patternViewerLauncher.getPatternViewer();
        }
//        if (patternViewer != null) patternViewer.updatePattern(patternData0);
        if (patternViewer != null) {
          patternViewer.setVisible(true); // パターン・ビューワを表示する
          patternViewer.updatePattern(patternData0, patternData1);
        }

        // グラフの更新
        if (graphViewer == null) {
          graphViewer = graphViewerLauncher.getGraphViewer();
          if (graphViewer != null)
            graphViewer.shiftGraphAxis(1); // グラフの縦軸をシフトする
        }
        if (graphViewer != null) {
          // GraphViewerへ誤差データを渡す
          graphData[0] = learnErrorVal;
          graphData[1] = learnEntropyVal;
          graphData[2] = trialErrorVal;
          graphData[3] = trialEntropyVal;
          graphViewer.updateGraph(i, graphData);
        }

        DJ.print(", lapTime=" + lapTime + "[msec]");
        
        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            wait(SLEEP_TIME); // タイムアウト付きで待機状態にする。
            // 休止状態
            if (pauseFlag) wait();
            // DJ.print("Resume from wait(sleepTime)");
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** ConvolutionalNN." + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
      
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch --------------------------------------");
    }
    DJ._print(" End of all epoch --------------------------------------------");
    
    DJ._print(" 抽出サンプルによる学習効果の検証 ----------------------------");
    // DJ.print(" 試行データの中からサンプルを抽出して試行出力を表示する");
    trialFlag = true; // 試行フラグ　true:試行時　false:学習時
    for (int j = 0; j< 10; j++) { // 試行用データ数：trialNum=1797/3=599個
      DJ._print("抽出サンプル:" + j);
      int trialIndex = trialIndexList.get(j); // ランダム・インデックス
      int correctDigit = targetText[0][trialIndex]; // 正解文字
      DJ.print("正解文字:correctDigit=", correctDigit);
      
      inData = inputImageStd[trialIndex]; // 試行データ
      // DJ.print("試行データ:inData=", inData);
      forward(inData); // 中間層と出力層の順伝播処理を呼び出し
      
      // 試行出力
      DJ.print("試行出力:outLayer.y=", DJ.reshape(outY, outNodeNum, 1));
    }
    trialFlag = false; // 試行フラグ　true:試行時　false:学習時
   
    DJ._print("・最終誤差: ");
    for (int k = 0; k < graphData.length; k++) {
      DJ.print("  " + dataName[k] + "=" + graphData[k]);
    }
    
    DJ._print_("・試行回数=", trialCount); // 総試行回数
    DJ.print_(", 正答回数=", correctCount); // 総正答回数
    DJ.print("・最終正答比率=", correctRatio + "[%]"); // 正答率
        
    // 実行時間の表示
    DJ.print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    // 処理時間の表示
    DJ.print_("・タスク終了日時：", TimeStamp.getTimeFormated());
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print(", タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
  } // convNet()
  
  /** 順播処理の呼び出し */
  private void forward(double[] x) {
    // 畳込層への入力
    ArrayList<double[][]> xList = convoLayer0.get_xList();
    double[][] x_ = xList.get(0); // フィルタの入力画像は全て同一
    for (int i = 0; i < inImageHight; i++ ) {
      for (int j = 0; j < inImageWidth; j++) {
        x_[i][j] = x[inImageWidth * i + j];
      }
    }
    
//**/    if (trialFlag == false) { // 試行フラグ　false:学習時
//**/      startDebug = System.nanoTime(); // 実行開始時刻
//**/    }
    convoLayer0.forward(); // 畳込層順伝播の呼び出し
//**/    if (trialFlag == false) { // 試行フラグ　false:学習時
//**/      endDebug = System.nanoTime(); // 休止時刻
//**/      lapDebug = (endDebug - startDebug) / 1000000.0;
//**/      totalDebug = totalDebug + lapDebug; // 実行時間
//**/    }

    // DJ._print("----- poolLayer0.forward(convoLayer0.y) -----");
    poolLayer0.forward(); // プーリング層順伝播の呼び出し
    
   // 受け渡しテンソルの形状を変更
    ArrayList<double[][]> yList = poolLayer0.get_yList();
    double[] mid0X = middleLayer0.getX();
    for (int k = 0; k < filterNum; k++) {
      for (int i = 0; i < poolNodeHight; i++ ) {
//        for (int j = 0; j < poolNodeWidth; j++) {
//          mid0X[poolNodeWidth * (poolNodeHight * k + i) + j]
//                  = yList.get(k)[i][j];
//        }
        System.arraycopy(yList.get(k)[i], 0, mid0X,
                poolNodeWidth * (poolNodeHight * k + i), poolNodeWidth);
      }
    }
    
    middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
    middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
    outputLayer.forward(); // 出力層の順伝播処理を呼び出し
  }
  
  /** 逆播処理の呼び出し */
//  private void backward(DoubleTensor teach) {
  private void backward(double[] teach) {
    // DJ._print("----- backward(Tensor teach) -----");
    double[] outC = outputLayer.getC();
//    for (int k = 0; k < outNodeNum; k++) { outC[k] = teach[k]; }
    System.arraycopy(teach, 0, outC, 0, outNodeNum);
    outputLayer.backward(outC); // 出力層の逆伝播処理を呼び出し
    middleLayer1.backward(); // 中間層１の逆伝播処理を呼び出し
    middleLayer0.backward(); // 中間層０の逆伝播処理を呼び出し

    // 受け渡しテンソルの形状を変更
    double[] mid0dEdX = middleLayer0.getdEdX();
    ArrayList<double[][]> dEdYList = poolLayer0.get_dEdYList();
    for (int k = 0; k < filterNum; k++) {
      double[][] dEdY = dEdYList.get(k);
      for (int i = 0; i < poolNodeHight; i++ ) {
        for (int j = 0; j < poolNodeWidth; j++) {
          dEdY[i][j] = mid0dEdX[poolNodeWidth * (poolNodeHight * k + i) + j];
        } // j
      } // i
    } // k
    poolLayer0.backward(); // プーリング層逆伝播の呼び出し
    convoLayer0.backward(); // 畳込層逆伝播の呼び出し
  }
  
  /** 更新処理の呼び出し */
  private void update() {
    // DJ._print("----- update() -----");
    // DJ._print("----- convoLayer0.update(eta) -----");
    convoLayer0.update(); // 畳込層更新の呼び出し
    // DJ._print("----- poolLayer0.update(eta) -----");
    poolLayer0.update(); // プーリング層更新の呼び出し
    // 中間層と出力層のバイアスと重みの更新の呼び出し
    middleLayer0.update(); // 中間層０の更新処理を呼び出し
    middleLayer1.update(); // 中間層１の更新処理を呼び出し
    outputLayer.update(); // 出力層の更新処理を呼び出し
  }

} // ConvNet Class

// End of file

