/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package task.control;

import active.Activator;
import data.Control;
import layer.ControlLayer;
import layer.Layer;
import task.Task;
import util.DJ;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: MassFlutter</p>
 * <p> 説　明: 質量が位置により変化する制御対象の力操作量を推定する。
 *              加速度、速度、位置指令を入力し、制御対象の逆関数を学習する。
 *               中間層２段＋出力層１段（層数切替可能２～３）</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.10.27</p>
 */
public class MassFlutter extends Task {

  static final int THREE_LAYER = 3; // ３層（中間層×２）
  int layerNum = THREE_LAYER; // 層数
  
  // 学習・試行制御
  int epoch = 10000; // エポックの回数
  int batchNum = 1; // 10; // バッチ数
  int interval = 100; // 経過表示間隔
  
  // 学習パラメータ
  double initialCoef = 0.5; // 重みとバイアスの初期値係数
  double eta = 0.4; // 学習係数
  int dataNum = 1000; //1000; // エポック毎の学習データ数　1[s]/1[ms] = 1000 [個]

  // 制御系と制御対象のパラメータ
  private Control control; // 制御系と制御対象
  int speedGain = 1280; //2048 // 比例ゲイン（速度ループゲイン）
  int integralGain = 0; // 積分ゲイン
  double inertia = 1.0; // 慣性質量［kgm^2/sec^2］
  double friction = 2.0; // 粘性摩擦[kg/sec]

  // 各層のノード数
  int inNodeNum = 3; //2; // 入力層のノード数
  int midNodeNum = 15; //8; // 中間層のノード数
  int outNodeNum = 1; // 出力層のノード数
  
  // レイヤー構成
  private ControlLayer middleLayer0; // 中間層０
  private ControlLayer middleLayer1; // 中間層１
  private ControlLayer outputLayer; // 出力層
  
  // 誤差変数
  double squareError; // 二乗誤差
  double meanError; // 二乗誤差データの平均
  double aveError; // エポック毎の誤差の平均値
  
  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; // パターン・ビューワを表示する
    patternData0 = new double[10][dataNum]; // ID,出力(力),速度,正解(速度所望値)
    patternData1 = new double[10][dataNum]; // ID,力操作量,加速度,所望速度
    patternViewerLauncher = DJ.pattern( // 判定パターンとデータ
        PatternViewer.PATTERN_TUNER, patternData0, "MassFlutter:Learning", // 学習中
        PatternViewer.PATTERN_TUNER, patternData1, "MassFlutter:Trial"); // 試行結果
    
    // グラフ・ビューワ・ランチャを得る
    graphShift = 8; // グラフの縦軸を移動
    graphData = new double[2];
    dataName = new String[2]; // データ名
    dataName[0] = "squareError"; // 学習時の二乗誤差
    dataName[1] = "aveError"; // 学習時の平均二乗誤差
    graphViewerLauncher = DJ.graph(epoch, interval, dataName, graphData);
    
    massFlutter(); // タスク本体の呼び出し
  }
  
  /** 
   * 質量が位置により変化する制御対象の力操作量を推定する。
   * 力操作量と質量を推定する。
   */
  public void massFlutter() {
    DJ._print("DaiJa_V3, MassFlutter.massFlutter() ==========================");
    
    DJ._print("・パラメータ");
    DJ.print_(" 重みとバイアスの初期値係数:initialCoef=", initialCoef);
    DJ.print_(", 学習係数:eta=",eta);
    DJ.print(", エポックの回数:epoch=",epoch);
    
    DJ.print_(" バッチ数:batchNum=",batchNum);
    DJ.print_(", 経過表示間隔:interval=",interval);
    DJ.print(", 学習データ数：dataNum=", dataNum);
    
    DJ.print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・各層のノード数");
    DJ.print_(" 入力層ノード数:inNodeNum=",inNodeNum);
    DJ.print_(", 中間層ノード数:midNodeNum=",midNodeNum);
    DJ.print(", 出力層ノード数:outNodeNum=",outNodeNum);
    
    DJ._print("・制御系と制御対象（負荷）を設定");
    control = new Control();
    //#    control.setParameter(speedGain, integralGain, 1.0, 0.0); // パラメータ
    control.setParameter(0, 0, inertia, friction); // パラメータ
    control.setMassBase(inertia); // 基本質量を設定する
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl(); // 制御系をリセット
    
    DJ._print("・入力データと教師データを作成");
    DJ._print(" 制御系から所望速度を収集する");
    //# double[] inputData = new double[dataNum]; // 入力データ
    double[] correctData = new double[dataNum]; // 正解データ
    
    double[] accelCmd = new double[dataNum]; // 加速度目標値
    double[] speedCmd = new double[dataNum]; // 速度目標値
    double[] positionCmd = new double[dataNum]; // 位置目標値
    double spd = 0.0; // 速度目標値生成用補助変数
    double psn = 0.0; // 位置目標値生成用補助変数
    
    for (int j = 0; j < dataNum; j++) {
      double clock = 0.0 + (1.0 / dataNum) * j;
      //# inputData[j] = clock; // クロック
      //# correctData[j] = control.getS_Curve(clock); // 所望速度（Ｓ字カーブ）

      // 加速度目標値を生成
      if (clock < 0.5) 
        accelCmd[j] = (+1.0 - Math.cos(4.0 * Math.PI * clock)) * 2.0;
      else 
        accelCmd[j] = (-1.0 + Math.cos(4.0 * Math.PI * clock)) * 2.0;
      // 速度目標値（所望速度）を生成
      spd = spd + accelCmd[j] / dataNum; // 1000.0;
      speedCmd[j] = spd;
      // 位置目標値を生成
      psn = psn + speedCmd[j] / dataNum; // 1000.0;
      positionCmd[j] = psn;

      correctData[j] = speedCmd[j]; // 所望速度（Sinカーブ応答）;
              
      patternData0[0][j] = clock; // パターン表示（0頁目）の時間軸
      patternData1[0][j] = clock; // パターン表示（1頁目）の時間軸
    }

    // DJ.print(" 順序をランダムにシャッフルしたインデックスのリスト");
    //# ArrayList<Integer> indexList = DJ.permutationRandom(dataNum);
    //# DJ.print("inputData", inputData);
    
    DJ._print("・ニューラルネットの各層の初期化");
    middleLayer0 = new ControlLayer(inNodeNum, midNodeNum);
    if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
      middleLayer0.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
    else // ２層：中間１層を飛ばして中間０層と出力層でデータを受け渡しする
      middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
//      middleLayer0.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
    double[] mid0X = middleLayer0.getX();
    double[] mid0Y = middleLayer0.getY();
    // double[] mid0C = middleLayer0.getC();
    // double[] mid0E = middleLayer0.getE();
    // double[] mid0dEdX = middleLayer0.getdEdX();
    //# double[] mid0dEdY = middleLayer0.getdEdY(); // 参照を代入（注１）

    double[] mid1Y;
    double[] mid1dEdX;
    middleLayer1 = new ControlLayer(midNodeNum, midNodeNum);
    if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
      middleLayer1.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
    else // ２層：中間１層を飛ばして中間０層と出力層でデータを受け渡しする
      middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
//      middleLayer1.initialize(eta, initialCoef, Activator.RELU, Layer.ADA_GRAD);
    // double[] mid1X = middleLayer1.getX(); // 参照を代入（注３）
    mid1Y = middleLayer1.getY();
    // double[] mid1C = middleLayer1.getC();
    // double[] mid1E = middleLayer1.getE();
    mid1dEdX = middleLayer1.getdEdX();
    //# double[] mid1dEdY = middleLayer1.getdEdY(); // 参照を代入（注２）

    outputLayer = new ControlLayer(midNodeNum, outNodeNum);
//    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.SGD);
    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.ADA_GRAD);
    //# double[] outX = outputLayer.getX(); // 参照を代入（注４）
    double[] outY = outputLayer.getY();
    double[] outC = outputLayer.getC();
    /**/    double[] outE = outputLayer.getE();
    double[] outdEdX = outputLayer.getdEdX();
    // ouble[] outdEdY = outputLayer.getdEdY();

    double[] outS = new double[outC.length]; // 正解（所望速度）

    if (layerNum == THREE_LAYER) { // ３層：中間１層を使用する
      middleLayer0.setdEdY(mid1dEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
      middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
      middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
      outputLayer.setX(mid1Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    }
    else { // ２層：中間１層を飛ばして中間０層と出力層でデータを受け渡しする
      middleLayer0.setdEdY(outdEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
      // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
      // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
      outputLayer.setX(mid0Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    }
    
    // DJ._print("PatternViewerへの参照を得る");
    patternViewer = patternViewerLauncher.getPatternViewer();
    // DJ._print("GraphViewerへの参照を得る");
    graphViewer = graphViewerLauncher.getGraphViewer();
    if (graphViewer != null) graphViewer.shiftGraphAxis(graphShift); // グラフの縦軸をシフトする
    
    DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | 
              (i == epoch); //経過表示フラグ

      // DJ._print("・学習用データのインデックスをシャッフル");
      //#      Collections.shuffle(indexList);
      
      // DJ.print("・学習誤差と試行誤差の初期化");
      squareError = 0.0; // 二乗誤差
      meanError = 0.0; // 二乗誤差データの平均
    
      //#      int index = indexList.get(i); // ランダム・インデックス
      
      // 制御をリセット
      control.setParameter(0, 0, inertia, friction);
      control.resetControl();
      
      // 全データを学習
      for (int j = 0; j < dataNum; j++) {
        // DJ._print("・エポックの実行ループ開始 ----------------------------");
        // DJ.print(" エポック:i=" + i + ", 学習データ:j=" + j);
        
        // DJ._print(" ##### 前向き伝播の呼び出し #####");
        // DJ._print("・入力データを取り出し");
        //#        int index = indexList.get(j); // ランダム・インデックス
        int index = j; // ランダム・インデックス
        // DJ._print(" 学習用データID：learnIndex=", learnIndex);
//        mid0X[0] = inputData[index]; // 入力データをランダムに取出し
        mid0X[0] = accelCmd[index] * 0.05; // 加速度目標値
        mid0X[1] = correctData[index]; // 正解データ（所望速度を応答速度として入力）
        mid0X[2] = positionCmd[index]; // 位置目標値
        middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
        if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
          middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
        outputLayer.forward(); // 出力層の順伝播処理を呼び出し
        
        // DJ._print(" 推定された力操作量を制御対象（負荷）に与える");
        double force = outY[0] / Control.DAC_FACTOR; // 推定された力の操作量
        control.setForce(force); // 力の操作量 frc_mnp
        //# control.massChanger(inputData[j]); // 質量を変動させる
        control.massChanger(); // 質量を変動させる
        control.execute(); // 制御対象の実行
        outS[0] = control.getSpeed(); // 応答速度;
        
        // DJ._print(" ##### 後向き伝播の呼び出し #####");
        outC[0] = correctData[index]; // 正解データを取出し
        //# outputLayer.backward(outC); // 出力層の逆伝播処理を呼び出し
        outputLayer.backward(outC, outS); // 出力層の逆伝播処理を呼び出し（逆関数の学習時）
        if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
          middleLayer1.backward(); // 中間層１の逆伝播処理を呼び出し
        middleLayer0.backward(); // 中間層０の逆伝播処理を呼び出し
        
        // DJ._print("・バッチ処理");
        if ((j % batchNum) == 0 ) {
          // DJ._print(" ##### 更新の呼び出し #####");
          middleLayer0.update(); // 中間層０の更新処理を呼び出し
          if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
            middleLayer1.update(); // 中間層１の更新処理を呼び出し
          outputLayer.update(); // 出力層の更新処理を呼び出し
        }

        // DJ._print(" ##### 学習時の誤差の保存 #####");
        if (intervalFlag) { // 実行時間に影響を与える
          // 二乗誤差の算出
          squareError = outE[0] * outE[0] / 2.0;
          meanError = meanError + squareError;
        
          // パターン表示用データの保存
          patternData0[1][j] = force * 0.001; // 推定された力操作量
          patternData0[2][j] = outS[0]; // 応答速度
          patternData0[3][j] = correctData[j]; // 正解（所望速度）
          
          patternData0[4][j] = control.getAcceleration() * 100.0; // 応答加速度
          patternData0[5][j] = control.getMass(); // 質量
          patternData0[6][j] = control.getPosition(); // 応答位置
          
          patternData0[7][j] = accelCmd[j] * 0.1; // 加速度目標値 
          patternData0[8][j] = speedCmd[j]; // 速度目標値
          patternData0[9][j] = positionCmd[j]; // * 0.001; // 位置目標値
          
        }
        
        // DJ._print(" End of one data ---------------");
      } // 全データを学習
      // DJ._print(" End of one epoch ---------------");// DJ._print(" End of one epoch ---------------");
      
      // DJ._print("　エポック毎の誤差の平均値を保存する");
      aveError = meanError / dataNum ;
      
      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

      // 経過表示インターバル
      if (intervalFlag) { // 実行時間に影響を与える
        // DJ._print(" ##### １エポックの実行結果 #####");
        DJ.print_(" i=" + i);
        DJ.print_(", squareError = ", squareError);
        DJ.print(", aveError = ", aveError);

        // グラフ表示用データを代入
        graphData[0] = squareError;
        graphData[1] = aveError;
        updateViewer(i); // ビューワの表示を更新
        
        DJ.print_(", lapTime = ", lapTime); DJ.print("[msec]");
        
        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            wait(SLEEP_TIME); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
    
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch ---------------------------------------");
    }
    DJ._print(" End of all epoch --------------------------------------------");
    
    DJ._print("・エポック実行回数");
    DJ.print(" Last epoch = ", epoch);
    DJ.print_("・最終誤差: ");
    for (int k = 0; k < graphData.length; k++) {
      DJ.print_("  " + dataName[k] + "=" + graphData[k]);
    }
    DJ.print("");
    
    DJ._print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・収集データ数: ", dataNum);
    DJ.print("力操作量の推定値: f~", patternData0[1]);
    DJ.print("応答速度: v", patternData0[2]);
    
    DJ._print("・検証：学習した逆関数で制御対象を駆動し、応答速度を比較する");
    control.setParameter(0, 0, inertia, friction); // パラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl();
//    spd = 0.0;
//    psn = 0.0;

    for (int j = 0; j < dataNum; j++) {
      //# mid0X[0] = inputData[j]; // 入力データを取出し
      mid0X[0] = accelCmd[j] * 0.05; // * 50.0; // 加速度目標値
      // （１）目標値を入力する場合
//**/      mid0X[1] = correctData[j]; // 所望速度
//**/      mid0X[2] = positionCmd[j]; // 位置目標値
      // （２）前回の応答をフィードバックして入力する場合
/**/      mid0X[1] = control.getSpeed(); // 応答速度をフィードバックして入力
/**/      mid0X[2] = control.getPosition(); // 応答位置をフィードバックして入力
      middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
      if (layerNum == THREE_LAYER) // ３層：中間１層を使用する
        middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
      outputLayer.forward(); // 出力層の順伝播処理を呼び出し
      
      double force = outY[0] / Control.DAC_FACTOR; // ＮＮで算出された力操作量
      control.setForce(force); // 力の操作量 frc_mnp
//      control.massChanger(inputData[j]); // 質量を変動させる
      control.massChanger(); // 質量を変動させる
      control.execute(); // 制御対象の実行

      patternData1[1][j] = force * 0.001; // ＮＮで算出された力操作量
      patternData1[2][j] = -10.0; // 
      patternData1[3][j] = control.getMass(); // 質量
      // 制御対象の応答
      patternData1[4][j] = control.getAcceleration() * 100.0; // 応答加速度
      patternData1[5][j] = control.getSpeed(); // 応答速度
      patternData1[6][j] = control.getPosition(); // 応答位置速度
      // 目標値
      patternData1[7][j] = accelCmd[j] * 0.1; // * 100.0; // 加速度目標値
      patternData1[8][j] = speedCmd[j]; // 速度目標値（所望速度）
      patternData1[9][j] = positionCmd[j]; // * 0.001; // 位置目標値
    }
    
    updatePattern();
    
    // 処理時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print_("・タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
    DJ.print(", タスク終了日時：", TimeStamp.getTimeFormated());
    
    DJ._print("##### MassFlutter 終了 #####");
  } // massFlutter()

} // MassFlutter Class

// End of file
