/*
 *  Title: DaiJa_V3 (Deep-learning Activated Implement by JAva)
 *  @author Yoshinari Sasaki
 *  @version 3.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package task.control;

import data.Control;
import layer.ControlLayer;
import task.Task;
import util.DJ;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: ServoControl</p>
 * <p> 説　明: サーボ制御
 *              ・速度のオープンループ制御
 *              ・速度のクローズループ制御</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.10.04</p>
 */
public class ServoControl extends Task {
  
  // 学習・試行制御
  int epoch = 1; // エポックの回数
  int interval = 1; //10; // 経過表示間隔
  
  // 学習パラメータ
  double initialCoef = 0.1; // 重みとバイアスの初期値係数
  double eta = 0.1; // 学習係数
  int dataNum = 1000; // エポック毎の学習データ数

  // 制御系と制御対象のパラメータ
  int speedGain = 2048; // 比例ゲイン（速度ループゲイン）
  int integralGain = 0; // 積分ゲイン
  double inertia = 1.0; // 慣性質量［kgm^2/sec^2］
  double friction = 2.0; // 粘性摩擦係数[kg/sec]

  // 各層のノード数
  int inNodeNum = 2; // 入力層のノード数
  int midNodeNum = 8; // 中間層のノード数
  int outNodeNum = 1; // 出力層のノード数
  
  // レイヤー構成
  private ControlLayer middleLayer0; // 中間層０
  //#  private ControlLayer middleLayer1; // 中間層１
  private ControlLayer outputLayer; // 出力層
  
  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; //true; // false:非表示
    patternData0 = new double[5][dataNum]; // ID,力,加速度,速度,位置
    patternData1 = new double[5][dataNum]; // ID,力操作量,加速度,所望速度
    patternViewerLauncher = DJ.pattern(  // 判定パターン、データ、タイトル
        PatternViewer.PATTERN_TUNER, patternData0, "Open-loop Control",
        PatternViewer.PATTERN_TUNER, patternData1, "Clos-loop Control");
    
    servoControl(); // タスク本体の呼び出し
  }
  
  /** 
   * サーボ制御
   * ・速度のオープンループ制御
   * ・速度のクローズループ制御
   */
  public void servoControl() {
    DJ._print("ServoControl.servoControl() ==========================");
    
    DJ._print("・パラメータ");
    DJ.print(", 学習データ数：dataNum=", dataNum);
    
    DJ.print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・制御系と制御対象を設定");
    Control control = new Control();

    // DJ._print("PatternViewerへの参照を得る");
    patternViewer = patternViewerLauncher.getPatternViewer();
    
    // DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | 
              (i == epoch); //経過表示フラグ

      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

       // 経過表示インターバル
      if (intervalFlag) { // 実行時間に影響を与える
//        DJ.print_(" i=" + i);

        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
//            wait(SLEEP_TIME); // タイムアウト付きで待機状態
            wait(100); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
    
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch ---------------------------------------");
    } // i
    // DJ._print(" End of all epoch --------------------------------------------");
    
    DJ._print("・制御系と制御対象の実行準備");
    
//    control.setParameter(speedGain, integralGain, inertia, friction); // 制御対象のパラメータ
    double[][] trialData = new double[4][dataNum]; // 力操作量と試行結果
    
    DJ._print("（１）オープンループ制御の応答速度");
    int gain = speedGain / 10; //5;
    control.setParameter(gain, 0, inertia, friction); // パラメータ
//    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.stopFeedback(); // フィードバックを止める（無効化する）
    control.resetControl();
    
    for (int j = 0; j < dataNum; j++) {
      double clock = 0.0 + (1.0 / dataNum) * (double)j;
      control.execute();
//      trialData[0][j] = force; // 力操作量
      trialData[0][j] = control.getForce(); // 力操作量
      trialData[1][j] = control.getAcceleration(); // 応答加速度
      trialData[2][j] = control.getSpeed(); // 応答速度
      trialData[3][j] = control.getPosition(); // 応答位置
    
      // 学習されているデータ
      patternData0[0][j] = clock; // 時刻
      patternData0[1][j] = trialData[0][j]; // 力操作量
      patternData0[2][j] = trialData[1][j] * 100.0; // 応答加速度
      patternData0[3][j] = trialData[2][j] * 1.0; // 応答速度
      patternData0[4][j] = trialData[3][j] * 1.0; // 応答位置
    }
    
    DJ._print("（2）クローズループ（PID）制御の応答速度");
      control.startSpeedControl(); // 速度制御を開始する（有効化する）
      control.startFeedback(); // フィードバック処理を開始する（有効化する）
      control.setParameter(speedGain, integralGain, inertia, friction); // パラメータ
      control.resetControl();

    for (int j = 0; j < dataNum; j++) {
      double clock = 0.0 + (1.0 / dataNum) * (double)j;
      control.execute();
      trialData[0][j] = control.getForce(); // 力操作量
      trialData[1][j] = control.getAcceleration(); // 応答加速度
      trialData[2][j] = control.getSpeed(); // 応答速度
      trialData[3][j] = control.getPosition(); // 応答位置
    
      // 学習されているデータ
      patternData1[0][j] = clock; // 時刻
      patternData1[1][j] = trialData[0][j]; //; // 力操作量
      patternData1[2][j] = trialData[1][j] * 100.0; // 応答加速度
      patternData1[3][j] = trialData[2][j] * 1.0; // 応答速度
      patternData1[4][j] = trialData[3][j] * 1.0; // 応答位置
    }
    
    // パターンの更新
    updatePattern();
    
    
    // 処理時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print_("・タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
    DJ.print(", タスク終了日時：", TimeStamp.getTimeFormated());
  } // servoControl()

    
} // ServoControl Class

// End of file
