/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package task.control;

import active.Activator;
import data.Control;
import java.util.ArrayList;
import java.util.StringTokenizer;
import layer.ControlLayer;
import layer.Layer;
import task.Task;
import util.DJ;
import util.LogEditor;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: TargetEmulator</p>
 * <p> 説　明: 制御対象をＮＮで学習した仮想制御対象を提供する
 *      制御対象を学習したＮＮの係数をファイルから読込む
 *      力操作量fを入力すると、推定加速度a^が出力される
 *      推定加速度a^から推定速度v^と推定位置p^が求められる</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2020, 2021</p>
 * <p> 作成日: 2020.08.19</p>
 */
public class TargetEmulator extends Task {

  
  // 学習・試行制御
  int epoch = 1; // エポックの回数
  int interval = 1; // 経過表示間隔
  
  // 学習パラメータ
  double initialCoef = 0.2; // 重みとバイアスの初期値係数
  double eta = 0.2; // 学習係数
  int dataNum = 1000; // エポック毎の学習データ数

  // 制御系と制御対象のパラメータ
  Control control; // 制御系と制御対象
  int speedGain = 1280; //2048 // 比例ゲイン（速度ループゲイン）
  int integralGain = 0; // 積分ゲイン
  double inertia = 1.0; // 慣性質量［kgm^2/sec^2］
  double friction = 2.0; // 粘性摩擦[kg/sec]

  // 各層のノード数
  int inNodeNum = 2; // 入力層のノード数
  int midNodeNum = 8; // 中間層のノード数
  int outNodeNum = 1; // 出力層のノード数
  
  // レイヤー構成
  private ControlLayer middleLayer0; // 中間層０
  //#  private ControlLayer middleLayer1; // 中間層１
  private ControlLayer outputLayer; // 出力層
  
  // 制御対象を学習した仮想制御対象
  private double[] mid0X, mid0U, mid0Y;
  private double[] outX, outU, outY;
  private double force; // 力操作量
  private double acceleration; // 推定加速度
  private double speed; // 推定速度
  private double position; // 推定位置
  
  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; //true:パターン・ビューワを表示 // false:非表示
    patternData0 = new double[10][dataNum]; // ID,力,加速度,速度,位置
    patternData1 = new double[10][dataNum]; // ID,力操作量,加速度,所望速度
    patternViewerLauncher = DJ.pattern( // 判定パターンとデータ
        PatternViewer.PATTERN_TUNER, patternData0, "TargetEmulator:Learnning  Result",
        PatternViewer.PATTERN_TUNER, patternData1, "TargetEmulator:Trial Result");
    
    targetEmulator(); // タスク本体の呼び出し
  }
  
  /** 
   * 制御対象をＮＮで学習した仮想制御対象を提供する
   * 力操作量を入力すると、推定加速度、推定速度、推定位置が算出される
   */
  public void targetEmulator() {
    DJ._print("DaiJa_V3, TargetEmulator.targetEmulator() ====================");
    
    DJ._print("・パラメータ");
    DJ.print_(" 重みとバイアスの初期値係数:initialCoef=", initialCoef);
    DJ.print_(", 学習係数:eta=",eta);
    DJ.print(", エポックの回数:epoch=",epoch);
    
    //#    DJ.print_(", ドロップ・アウト率:dropOutRate=",dropOutRate);
    DJ.print_(", 経過表示間隔:interval=",interval);
    DJ.print(", 学習データ数：dataNum=", dataNum);
    
    DJ.print("・制御系と制御対象のパラメータ");
    DJ.print_(" 比例ゲイン:speedGain=",speedGain);
    DJ.print_(", 積分ゲイン:integralGain=",integralGain);
    DJ.print_(", 慣性質量:inertia=",inertia);
    DJ.print(", 粘性摩擦:friction=",friction);
    
    DJ._print("・各層のノード数");
    DJ.print_(" 入力層ノード数:inNodeNum=",inNodeNum);
    DJ.print_(", 中間層ノード数:midNodeNum=",midNodeNum);
    DJ.print(", 出力層ノード数:outNodeNum=",outNodeNum);
    
    DJ._print("・制御系と制御対象（負荷）を設定");
    control = new Control();
    
    // DJ._print("クロック・データを作成");
    double[] clockData = new double[dataNum]; // クロック配列
    double inc = 1.0 / (double)dataNum;
    double clock = -inc;
    for (int j = 0; j < dataNum; j++) {
      clock = clock + inc;
      clockData[j] = clock; // 入力データ
      patternData0[0][j] = clock; // パターン表示（0頁目）の時間軸
      patternData1[0][j] = clock; // パターン表示（1頁目）の時間軸
    }
    
    // DJ.print(" 順序をランダムにシャッフルしたインデックスのリスト");
    //    ArrayList<Integer> indexList = DJ.permutationRandom(dataNum);
    
    DJ._print("・ニューラルネットの各層の初期化");
    middleLayer0 = new ControlLayer(inNodeNum, midNodeNum);
    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
    //    middleLayer0.initialize(eta, initialCoef, Activator.SIGMOID, Layer.ADA_GRAD);
//    double[] mid0X = middleLayer0.getX();
//    double[] mid0Y = middleLayer0.getY();
    mid0X = middleLayer0.getX();
    mid0Y = middleLayer0.getY();
    // double[] mid0C = middleLayer0.getC();
    // double[] mid0E = middleLayer0.getE();
    // double[] mid0dEdX = middleLayer0.getdEdX();
    //# double[] mid0dEdY = middleLayer0.getdEdY(); // 参照を代入（注１）

    // middleLayer1 = new ControlLayer(midNodeNum, midNodeNum);
    // middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.SGD);
    ////  middleLayer1.initialize(eta, initialCoef, Activator.SIGMOID, Layer.ADA_GRAD);
    //# double[] mid1X = middleLayer1.getX(); // 参照を代入（注３）
    // double[] mid1Y = middleLayer1.getY();
    // double[] mid1C = middleLayer1.getC();
    // double[] mid1E = middleLayer1.getE();
    // double[] mid1dEdX = middleLayer1.getdEdX();
    //# double[] mid1dEdY = middleLayer1.getdEdY(); // 参照を代入（注２）

    outputLayer = new ControlLayer(midNodeNum, outNodeNum);
    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.SGD);
    //    outputLayer.initialize(eta, initialCoef, Activator.IDENTITY, Layer.ADA_GRAD);
    //# double[] outX = outputLayer.getX(); // 参照を代入（注４）
//    double[] outY = outputLayer.getY();
    outY = outputLayer.getY();
    // double[] outC = outputLayer.getC();
    // double[] outE = outputLayer.getE();
    double[] outdEdX = outputLayer.getdEdX();
    // double[] outdEdY = outputLayer.getdEdY();

     //#    double[] outS = new double[outC.length]; // 正解（所望速度）

    // middleLayer0.setdEdY(mid1dEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
    // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
    // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
    // outputLayer.setX(mid1Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    
    middleLayer0.setdEdY(outdEdX); // （注１）中間０層のdEdYへ、中間１層のdEdXの参照を設定
    // middleLayer1.setdEdY(outdEdX); // （注２）中間１層のdEdYへ、出力層のdEdXの参照を設定
    // middleLayer1.setX(mid0Y); // （注３）中間１層の入力mid1Xへ、中間０層の出力mid0Yの参照を設定
    outputLayer.setX(mid0Y); // （注４）出力層の入力outXへ、中間層の出力mid1Yの参照を設定
    
    // DJ._print("PatternViewerへの参照を得る");
    patternViewer = patternViewerLauncher.getPatternViewer();
    
    // DJ._print(" ##### ニューラルネットの学習開始 #####");
    for (int i = 0; i <= epoch; i++) {
      startTime = System.nanoTime(); // 実行開始時刻
      intervalFlag = (i % interval == interval - 1) | 
              (i == epoch); //経過表示フラグ

      // 実行時間の累積
      endTime = System.nanoTime(); // 休止時刻
      double lapTime_ = (endTime - startTime) / 1000000.0;
      if (lapTime_ > 0.0) lapTime = lapTime_; // オーバーフロー対策
      totalTime = totalTime + lapTime; // 経過時間を追加

       // 経過表示インターバル
      if (intervalFlag) { // 実行時間に影響を与える
//        // DJ._print(" ##### １エポックの実行結果 #####");
//        DJ.print_(" i=" + i);
        
        // スレッドの休止（実行速度の調整および経過表示のため）
        synchronized(this) {
          try {
            // DJ.print("Ｅnter to wait(sleepTime)");
            // wait(SLEEP_TIME); // タイムアウト付きで待機状態
            wait(100); // タイムアウト付きで待機状態
            // DJ.print("Resume from wait(sleepTime)");
            if (pauseFlag) wait(); // 休止状態
          }
          catch (InterruptedException e) {
             DJ.print("***** ERROR ***** " + getClass().getName() + "\n"
                 + " Exception occur in wait(sleepTime):" + e.toString());
          }
        } // synchronized()
      } // interval
    
      // 実行処理を強制的に終了させる
      if (abortFlag) {
        DJ._print("##### Abort action requested");
        epoch = i; // 現在のエポック回数iをepochに代入し、実行を強制的に終了させる
      }
      
      // DJ._print(" End of one epoch ---------------------------------------");
    } // i
    // DJ._print(" End of all epoch --------------------------------------------");
    
    DJ._print("・制御対象を学習したNNの精度を検討する");
    control.setParameter(0, 0, inertia, friction); // 制御対象のパラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl();
    
    DJ._print("１．制御対象に力操作量を与た場合の応答速度");
    double[][] trialData = new double[4][dataNum]; // 力操作量と試行結果
    force = 0.0; // 力操作量
    
    for (int j = 0; j < dataNum; j++) {
      clock = clockData[j]; // （２）で使用する
/*      
      // （１）力操作量を与える（台形状）
      int cycle = dataNum / 3;
      if (j < cycle) force = force + 2.0; // 入力データ（力の操作量）
      else if (j < cycle * 2) force = force + 1.0; 
      else force = force - 1.05;
*/ /*
      // （２）力操作量を与える(Ｓ字状)
      force = (-2 * clock + 3 )* clock * clock; // 力の操作量
      // force = force * 2000.0; // スケーリング
      force = force * 500.0; // スケーリング
*/ /**/
      // （３）力操作量を与える（折線ジグザグ状）
      int cycle = dataNum / 5;
      if (j < cycle) force = force + 2.5; // 入力データ（力の操作量）
      else if (j < cycle * 2) force = force - 3.0; 
      else if (j < cycle * 3) force = force + 1.5; 
      else if (j < cycle * 4) force = force - 2.0; 
      else force = force + 0.75;
/**/
      trialData[0][j] = force; // 力操作量
      control.setForce(trialData[0][j]); // 力操作量
      control.execute();
      trialData[1][j] = control.getAcceleration(); // 応答加速度
      trialData[2][j] = control.getSpeed(); // 応答速度
      trialData[3][j] = control.getPosition(); // 応答位置
    }
    
    DJ._print("ニューラルネットの係数（重みとバイアス）の読み込み");
    String fileName = "DaiJa-V4_Emulator.dat";
    ArrayList<double[][]> coeffList = loadCoef(fileName);
//    int dataSize = coeffList.size();
//    DJ.print("dataSize:", dataSize);
      
    double[][] coef;
    coef = coeffList.get(0);
    middleLayer0.setW(coef);
    DJ.print("middleLayer0.w", coef);

    coef = coeffList.get(1);
    middleLayer0.setB(coef[0]);
    DJ.print("middleLayer0.b", coef[0]);

    coef = coeffList.get(2);
    outputLayer.setW(coef);
    DJ.print("outputLayer.w", coef);

    coef = coeffList.get(3);
    outputLayer.setB(coef[0]);
    DJ.print("outputLayer.b", coef[0]);
    
    DJ._print("２．学習されているNNに力操作量を与た場合の応答速度");
    control.setParameter(0, 0, inertia, friction); // 制御対象のパラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl();

    // DJ._print("・制御対象（仮想制御対象）の実行準備");
    initEmulator(); // 仮想制御対象を初期化
    for (int j = 0; j < dataNum; j++) {

      // DJ._print(" 推定された力操作量を仮想制御対象に与え、応答加速度を得る");
      acceleration = setForce(trialData[0][j]); // 力の操作量のみ
      //# acceleration = getAcceleration(); // 推定加速度
      speed = getSpeed(); // 推定速度
      position = getPosition(); // 推定位置
      
      // パターン表示用データ
      patternData0[1][j] = trialData[0][j] * 0.001; // 力操作量
      patternData0[2][j] = -10.0; // ダミー
      patternData0[3][j] = -10.0; // ダミー 
      // 制御対象に力操作量を与た場合の応答
      patternData0[4][j] = trialData[1][j] * 100.0; //2000.0; // 応答加速度
      patternData0[5][j] = trialData[2][j]; // 応答速度
      patternData0[6][j] = trialData[3][j] * 2.0; //10.0; // 応答位置
      // 学習したNNに力操作量を与た場合の応答
      patternData0[7][j] = acceleration * 1.0; // 推定加速度
      patternData0[8][j] = speed * 10.0; // 推定速度
      patternData0[9][j] = position * 20.0; // 推定位置
    }
    
    DJ._print("３．学習されているNNにＳ字状の力操作量を与た場合の応答");
    // 制御対象の準備
    control.setParameter(0, 0, inertia, friction); // 制御対象のパラメータ
    control.stopSpeedControl(); // 速度制御を止める（無効化する）
    control.resetControl();
    
    // 層内の疑似ニューロンの変数をゼロ・リセット
    middleLayer0.clear();
    outputLayer.clear();
    
    acceleration = 0.0; // 加速度の推定値
    speed = 0.0; // 速度の推定値（推定速度の積算値）
    position = 0.0; // 位置の推定値（推定速度の積算値）
    
    for (int j = 0; j < dataNum; j++) {
      clock = clockData[j];
      force = (-2.0 * clock + 3.0 ) * clock * clock; // 
      force = force * 500.0; // スケーリング

      trialData[0][j] = force; // 力操作量
      control.setForce(trialData[0][j]); // 力操作量
      control.execute();
      trialData[1][j] = control.getAcceleration(); // 応答加速度
      trialData[2][j] = control.getSpeed(); // 応答速度
      trialData[3][j] = control.getPosition(); // 応答位置

      mid0X[0] = force * 0.002; // 力操作量
      mid0X[1] =  trialData[2][j]; // 速度(推定値) // 入力データ（速度）

      middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
      //#        middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
      outputLayer.forward(); // 出力層の順伝播処理を呼び出し
      
      acceleration = outY[0]; // 推定加速度
      speed = speed + acceleration / dataNum; // 推定速度
      position = position + speed / dataNum; // 推定位置

      // 学習されているデータ
      // 制御対象にｓ字の力操作量を与た場合の応答
      patternData1[1][j] = trialData[0][j] * 0.001; // 力操作量
      patternData1[2][j] = -10.0; // ダミー
      patternData1[3][j] = -10.0; // ダミー
      // 制御対象の応答
      patternData1[4][j] = trialData[1][j] * 100.0; //500.0; // 応答加速度
      patternData1[5][j] = trialData[2][j]; // * 2.0 // 応答速度
      patternData1[6][j] = trialData[3][j] * 2.0; //10.0; // 応答位置
      // 学習したNfwdNNにｓ字の力操作量を与た場合の応答速度
      patternData1[7][j] = acceleration * 1.0; // 加速度(推定値)
      patternData1[8][j] = speed * 10.0; // 速度(推定値)
      patternData1[9][j] = position * 20.0; // 位置(推定値)
    }
    
    // パターンの更新
    updatePattern();
    
    initEmulator(); // ＮＮの初期化（仮想制御対象の呼び出しの準備のため必須）
    
    // 処理時間の算出
    DJ._print_("・総実行時間：" + (totalTime / 1000.0) + " [sec]");
    double aveTime = totalTime / epoch;
    DJ.print(", 平均実行時間：" + aveTime + " [msec/epoch]");
    finishTime = System.currentTimeMillis(); // タスク開始時刻
    DJ.print_("・タスク処理時間：" + 
            ((finishTime - beginTime) / 1000.0) + " [sec]");
    DJ.print(", タスク終了日時：", TimeStamp.getTimeFormated());
    
    DJ._print("##### TargetEmulator 準備完了：待機中 #####");
  } // targetEmulator()

  
// -----------------------------------------------------------------------------  
  // 制御対象を学習した仮想制御対象
  
  /**
   * 仮想制御対象の初期化
   */
  public void initEmulator() {
    mid0X = middleLayer0.getX();
   
    // 層内の疑似ニューロンの変数をゼロ・リセット
    middleLayer0.clear();
    outputLayer.clear();
    
    // 仮想制御対象の変数をゼロ・リセット
    force = 0.0; // 力操作量
    acceleration = 0.0; // 推定加速度
    speed = 0.0; // 推定速度
    position = 0.0; // 推定位置
  }
  
  /**
   * 仮想制御対象へ力操作量を受け取り、応答加速度を返す
   * @param force_ double // 力操作量
   * @return speed double // 応答加速度
   */
  public double setForce(double force_) {
    // 仮の推定加速度と推定速度を求める
    double acceleration_ = outY[0]; // 仮の推定加速度（１ステップ前の推定加速度）
    double speed_ = speed + acceleration_ / dataNum; // 仮の推定速度

    mid0X[0] = force_ * 0.002; // 力操作量を入力
    mid0X[1] = speed_ * 10.0; // 仮の推定速度を入力　※10.0は係数

    middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
    //#        middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
    outputLayer.forward(); // 出力層の順伝播処理を呼び出し

    acceleration = outY[0]; // 推定加速度
    speed = speed + acceleration / dataNum; // 推定速度
    position = position + speed / dataNum; // 推定位置
    
    return acceleration;
  }
  
  /**
   * 仮想制御対象へ力操作量を受け取り、応答速度を返す
   * @param force_ double // 力操作量
   * @param speed_ double // 速度
   * @return acceleration double // 加速度の推定値
   */
  public double setForce(double force_, double speed_) {
    mid0X[0] = force_ * 0.002; // 力操作量を入力
    mid0X[1] = speed_; // 速度を入力

    middleLayer0.forward(); // 中間０層の順伝播処理を呼び出し
    //# middleLayer1.forward(); // 中間１層の順伝播処理を呼び出し
    outputLayer.forward(); // 出力層の順伝播処理を呼び出し
    
    acceleration = outY[0]; // 推定加速度
    speed = speed + acceleration / dataNum; // 推定速度
    position = position + speed / dataNum; // 推定位置

    return acceleration;
  }
  
  /**
   * 推定加速度を得る
   * @return acceleration double // 推定加速度
   */
  public double getAcceleration() {
    return acceleration; // 推定加速度
  }
  
  /**
   * 推定速度を得る
   * @return speed double // 推定速度
   */
  public double getSpeed() {
    return speed; // 推定速度
  }
  
  /**
   * 推定位置を得る
   * @return position double // 推定位置
   */
  public double getPosition() {
    return position; // 推定位置
  }
  
  // ---------------------------------------------------------------------------
  // 学習済みのニューラルネットの係数（重みとバイアス）を読み込む
  
  /**
   * ニューラルネットの係数（重みとバイアス）の読み込み
   * @param fileName String // ファイル名
   * @return coeffList ArrayList<double[][]> // 重みとバイアスの配列リスト
   */
  public static ArrayList<double[][]> loadCoef(String fileName) {
    ArrayList<String> stringList = LogEditor.getText(fileName);
    DJ.print("File title:", stringList.get(1)); // ファイル・タイトル
    int dataSize = Integer.parseInt(stringList.get(2)); // データ配列数
    //      DJ.print("Data size:", dataSize);
    int lineNum = 2;
    ArrayList<double[][]> coeffList = new ArrayList<>(dataSize);
    for (int k = 0; k < dataSize; k++) {
      lineNum = lineNum + 1;
      int nodeNum = Integer.parseInt(stringList.get(lineNum)); // ノード数
      double[][] coef = new double[nodeNum][];
      for (int i = 0; i < nodeNum; i++) {
        lineNum = lineNum + 1;
        coef[i] = loadLine(stringList.get(lineNum));
      }
      coeffList.add(coef);
    } // k
    return coeffList;
  }

  // 一行分の要素をリストに書き込む
  private static double[] loadLine(String lineString) {
    ArrayList<Double> doubleList = new ArrayList<>();
    StringTokenizer lineToken = new StringTokenizer(lineString, ",");
    while (lineToken.hasMoreTokens()) {
      String nextStr = lineToken.nextToken().trim();
      if ("".equals(nextStr)) {
//$        LogEditor.print("***** WARNING ***** DJ.loadLine():");
//$        LogEditor.print(" Next token in a lineString is void.");
        DJ._print_("***** WARNING ***** TargetEmulator.loadLine():");
        DJ.print(" Next token in a lineString is void.");
        continue;
      }
      double d = Double.parseDouble(nextStr);
      doubleList.add(d); // データ要素
    }
    int listSize = doubleList.size();
    double[] doubleArray = new double[listSize];
    for (int i = 0; i < listSize; i++) {
      doubleArray[i] = doubleList.get(i);
    }
    return doubleArray;
  }
    
} // TargetEmulator Class

// End of file
