/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2025
 */
package task.qu;

import quan.QuBit;
import quan.QuGate;
import task.Task;
import util.DJ;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: QuantumComm</p>
 * <p> 説　明: ドイチュのアルゴリズム</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2025</p>
 * <p> 作成日: 2025.03.12</p>
 */
public class DeutschAlgorithm extends Task {
  
  // 学習パラメータ
  int dataNum = 1000; // エポック毎の学習データ数

  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; //true; // false:非表示
    patternData0 = new double[5][dataNum]; // ID
    patternData1 = new double[5][dataNum]; // ID
    patternViewerLauncher = DJ.pattern(  // 判定パターン、データ、タイトル
        PatternViewer.PATTERN_TUNER, patternData0, "DeutschAlgorithm",
        PatternViewer.PATTERN_TUNER, patternData1, "DeutschAlgorithm");
    
    this.stepMode = true; // タスクのステップ実行モード
    
    deutschAlgorithm(); // タスク本体の呼び出し
  }
  
  /** 
   * ドイチュのアルゴリズム
   */
  public void deutschAlgorithm() {
    this.stepMode = true; // タスクのステップ実行モード
    // DJ.printS(3, "DeutschAlgorithm(1)", this); // Pause Task ------------------------
    DJ._print("DeutschAlgorithm.deutschAlgorithm() ========================");
    DJ.print("■量子計算の開始");

    DJ._print("(1)ドイチュのアルゴリズムの概要");
    DJ.print("(2)オラクル");
    DJ.print("(3)ドイチュのアルゴリズム");
    DJ.print("(4)関数別オラクル");
                
    DJ.printS(1, "DeutschAlgorithm(1)", this); // Pause Task ------------------------
    DJ.print("Code(1)ドイチュのアルゴリズムの概要");
    DJ._print("・ドイチュのアルゴリズムは関数ｆのタイプが、\n"
            + "　定数型か均衡型かを判定するアルゴリズムです。\n"
            + "　定数型は、入力が変化しても出力は定まった一方のみで\n"
            + "　均衡型は、入力が変化すると、出力も変化します。）");
    
    DJ._print("・関数ｆの入力は０か１の２通り、出力も０か１の２通りで\n"
            + "  その組み合わせは４通りになります。\n"
            + "  関数f0：入力が０でも１でも出力は０→f1(0)=f1(1)=0\n"
            + "  関数f1：入力が０でも１でも出力は１→f2(0)=f2(1)=1\n"
            + "  関数f2：入力が０なら出力は０→f3(0)=0"
                    + "、入力が１なら出力は１→f3(1)=1\n"
            + "  関数f3：入力が０なら出力は１→f4(0)=1"
                    + "、入力が１なら出力は０→f4(1)=0");
    
    DJ._print("・関数f0と関数f1は入力が変化しても出力は"
            + "変わらないので定数型Constantです。\n"
            + "  関数f2と関数f3は入力が変化すると"
            + "出力も変わるので均衡型Balancedです。");


    DJ.printS(1, "DeutschAlgorithm(2)", this); // Pause Task ------------------------
    DJ.print("Code(2)オラクル");
    DJ._print("・一般に関数ｆはブラック・ボックスとして扱われ、\n"
            + "  その実装については問いません。つまり入力を与えると\n"
            + "  出力を返す便利な関数ｆとして導入されるため、\n"
            + "  オラクルoracle（神託、神託機械）と呼ばれます。");
    DJ._print("※ここでは量子ゲートクラスQuGateのメソッドとして\n"
            + "  実装しますが、本来の量子ゲートではありません。");
    
    DJ._print("Code(2-1)関数fiの検証");
    int numOfFuncType = 4;
    for (int funcType = 0; funcType < numOfFuncType; funcType++) {
      DJ.print_("・関数タイプ：f", funcType);
      for (int i = 0; i < 2; i++) {
        // DJ.print("・オラクルに入力を与える");
        DJ.print_(",  in = ", i); // オラクルの入力
        int out = QuGate.deutschFunction(funcType, i);
        DJ.print_(" → out = ", out); // オラクルの出力
      }
      DJ.print("");
    }
    
    
    DJ.printS(2, "DeutschAlgorithm(2-2)①", this); // Pause Task ------------------------
    DJ.print("Code(2-2)①オラクル（疑似的な量子ゲートUf）の検証");
    DJ.print("・ドイチュのアルゴリズムに対応する疑似的な量子ゲートUf\n"  
      + "  public QuBit deutschGate(int functionType, QuBit inputQuBit)\n"
      + "  @param functionType int // 関数タイプ \n"  
      + "  @param inputQuBit QuBit // 入力量子ビット（２量子ビット） \n"  
      + "  @return outputQuBit QuBit // 出力量子ビット（２量子ビット）");
    
    DJ._print("・オラクルの入力量子ビットinQuBitと"
            + "出力量子ビットoutQuBitに対応する\n"
      + "  古典ビットをそれぞれ入力inputCBと出力outputCBとします。\n"
      + "                                                         \n"
      + "  inQuBit[i]           ┌───┐   outQuBit             \n"
      + "  第０入力量子ビット ─┤Oracle├─ 第０出力量子ビット   \n"
      + "    inputCB[i][0]      │      │     outputCB[i][0]     \n"
      + "  第１入力量子ビット ─┤  Uf  ├─ 第１出力量子ビット   \n"
      + "    inputCB[i][1]      └───┘     outputCB[i][1]     \n"
      + "  インデックスi = 0,1は入力の２パターンを表わします。    ");
    
    
    DJ.printS(2, "DeutschAlgorithm(2-2)②", this); // Pause Task ------------------------
    DJ.print("Code(2-2)②入力古典ビットと出力古典ビットの関係");
    DJ.print("・オラクル（疑似的な量子ゲートUf）の               \n"
      + "  入力古典ビットinputCBと出力古典ビットoutputCBの関係    \n"
      + "    outputCB[i][0] = inputCB[i][0]                       \n"
      + "    outputCB[i][1] = inputCB[i][1] ⊕ fi(inputCB[i][0])   \n"
      + "  演算子⊕：排他的論理和EX-OR                             \n"
      + "  入力inputCBは 00 と 10 の２パターン（input0のみ変化）  \n"
      + "  ゲートUfの入出力特性                                   \n"
      + "  関数    :f0        :f1         :f2         :f3         \n"
      + "  inputCB :  00  10  :  00   10  :  00   10  :  00   10  \n"
      + "  outputCB:  00  10  :  01   11  :  00   11  :  01   10  ");
    
    

    DJ.printS(2, "DeutschAlgorithm(2-3)", this); // Pause Task ------------------------
    DJ.print("Code(2-3)オラクルへの入力を生成");
    int numOfInPattern = 2; // 入力パターン数
    
    int[][] inputCB = new int[numOfInPattern][2]; // 入力古典ビット×２、０か１
    QuBit e0 = new QuBit(QuBit.STD0_QUBIT); // 標準量子ビット[1 0]'
    QuBit e1 = new QuBit(QuBit.STD1_QUBIT); // 標準量子ビット[0 1]'
    QuBit[][] inputQB = new QuBit[numOfInPattern][2]; // ２パターン、量子ビット×２
    QuBit[] inQuBit = new QuBit[2]; // ２パターン、２量子ビット
    QuBit[] e = {e0, e1}; // 標準量子ビット{[1 0]' [0 1]'}
    
    // DJ.print("・オラクルへの入力パターンを生成します。");
    for (int i = 0; i < numOfInPattern; i++) {
      inputCB[i][0] = i % 2; // 第０入力古典ビット、０か１
      inputCB[i][1] = i / 2; // 第１入力古典ビット、０のみ
      DJ._print("入力パターン:", i
              + ", 古典ビット（２進数）" + inputCB[i][0] + inputCB[i][1] + "");
      DJ.print("　１量子ビット２個での入力");
      inputQB[i][0]= e[inputCB[i][0]]; // 標準量子ビット[1 0]', [0 1]'
      inputQB[i][1]= e[inputCB[i][1]]; // 標準量子ビット[1 0]'
      DJ.print_("  inputQB0 ", inputQB[i][0], QuBit.H_BIT);
      DJ.print(",  inputQB1 ", inputQB[i][1], QuBit.H_BIT);
      
      DJ.print("　２量子ビット１個での入力");
      inQuBit[i] = inputQB[i][0].interactQuBit(inputQB[i][1]);
      DJ.print("  inQuBit ", inQuBit[i], QuBit.H_BIT);
    }
    

    DJ.printS(2, "DeutschAlgorithm(2-4)", this); // Pause Task ------------------------
    DJ.print("Code(2-4)オラクルに質問：４タイプ × ２パターン");
    int[] outputCB = new int[2]; // 出力古典ビット×２、０か１
    
    for (int funcType = 0; funcType < numOfFuncType; funcType++) {
      DJ._print("・関数タイプ：f" + funcType);
      for (int i = 0; i < numOfInPattern; i++) {
        DJ._print("　入力パターン：inputType = ", i);
        
        DJ.print("・オラクルへの入力量子ビット（２量子ビット）：");
        DJ.print("  inQuBit ", inQuBit[i], QuBit.H_BIT);
        
        // DJ.print("・オラクルに質問");
        QuBit outQuBit = QuGate.deutschGate(funcType, inQuBit[i]);
        
        DJ.print("・オラクルの出力量子ビット（２量子ビット１個）");
        DJ.print("  outQuBit", outQuBit, QuBit.H_BIT);
        
        DJ.print("・オラクルの入力と出力を古典ビットで表示します。");
        DJ.print("　出力量子ビットを古典ビットに変換します。");
        double[][] outTransProb = outQuBit.getTransProb();
        DJ.printF("outTransProb", outTransProb);
        if (outTransProb[0][0] == 1.0) outputCB[0] = 0;
                                  else outputCB[0] = 1;
        if (outTransProb[0][1] == 1.0) outputCB[1] = 0;
                                  else outputCB[1] = 1;
        
        DJ._print("・オラクルの入力古典ビット（２進数）"
               + inputCB[i][0] + inputCB[i][1] + "");
        DJ.print("・オラクルの出力古典ビット（２進数）"
               + outputCB[0] + outputCB[1] + "");
      }
      
      if (funcType < numOfFuncType - 1)
        DJ.printS(1, "DeutschAlgorithm(2-4)", this); // Pause Task ------------------------
    }



    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(3, "DeutschAlgorithm(3)", this); // Pause Task ------------------------
    DJ.print("Code(3)ドイチュのアルゴリズムの量子回路");
    
    DJ.print("・量子回路は量子ゲートとオラクルOracleで構成される\n"
      + "                  ┌-┐┌───┐┌-┐┌-┐  \n"
      + "QB0 = [1 0]'-─-─┤H├┤Oracle├┤H├┤M├  \n"
      + "             ┌-┐│ ││      ││ │└-┘  \n"
      + "QB1 = [1 0]'-┤X├┤H├┤  Uf  ├┤I├──-  \n"
      + "             └-┘└-┘└───┘└-┘       ");
    DJ.print("  関数ｆによる疑似的な量子ゲートUfはオラクルと呼ばれます。");
    
    DJ._print("・この量子回路の出力の２量子ビットの第０量子ビットが、\n"
            + "　[1 0]'（古典ビット０）ならば関数は定数型Constantで、\n"
            + "　[0 1]'（古典ビット１）ならば関数は均衡型Balancedです。");
    
    DJ._print("・量子回路への入力");
    DJ.print("  ２個の量子ビットQB0とQB1の初期値はどちらも[1 0]'ですが、\n"
           + "  QB1はXゲートで反転され、[0 1]'になります。" );
        

    DJ.printS(1, "DeutschAlgorithm(3-1)", this); // Pause Task ------------------------
    DJ.print("Code(3-1)入力量子ビットの生成とXゲートで反転");
    
    DJ._print("・２個の標準量子ビットQB0とQB1を生成します。");
    QuBit qb0 = new QuBit(); // [1 0]'
    QuBit qb1 = new QuBit(); // [1 0]'
    DJ.printF("qb0", qb0);
    // qb0:F qb1:F
    // ┌     ┐
    // │ 1+0i│= [1 0]'
    // │ 0+0i│
    // └     ┘
    
    DJ.print("・Xゲートを生成します。");
    QuGate xGate = new QuGate(QuGate.X_GATE);
    DJ.printF("xGate", xGate); // Ｘゲート
    // xGate:F
    // ┌            ┐  ┌      ┐
    // │ 0+0i   1+0i│= │ 0   1│
    // │ 1+0i   0+0i│  │ 1   0│
    // └            ┘  └      ┘
    
    DJ.print("・量子ビットQB1をXゲートで反転します。");
    qb1.apply(xGate);   
    DJ.printF("qb1", qb1); // QB1
    // qb1:F
    // ┌     ┐
    // │ 0+0i│= [0 1]'
    // │ 1+0i│
    // └     ┘
    
    DJ.print("　量子ビットQB0とQB1相互作用させます。QB0 ⊗ QB1");
    QuBit qb0_1 = qb0.interactQuBit(qb1);
    DJ.printF("qb0_1", qb0_1); // QB0_1ビット
    // qb0_1:F
    // ┌     ┐
    // │ 0+0i│= [0 1 0 0]'
    // │ 1+0i│
    // │ 0+0i│
    // │ 0+0i│
    // └     ┘
    
    DJ.printS(1, "DeutschAlgorithm(3-2)", this); // Pause Task ------------------------
    DJ._print("Code(3-2)Hゲート（Hadamardゲート）を生成");
    QuGate hGate = new QuGate(QuGate.H_GATE);
    DJ.printF("hGate", hGate); // Hゲート
    // hGate:F
    // ┌                        ┐       ┌      ┐
    // │ 0.70711+0i   0.70711+0i│= 1/√2│ 1   1│
    // │ 0.70711+0i  -0.70711+0i│       │ 1  -1│
    // └                        ┘       └      ┘
    
    DJ.print("　Hゲート同士を相互作用させます。H ⊗ H");
    QuGate h_hGate = hGate.interactQuGate(hGate);
    DJ.printF("h_hGate", h_hGate); // H_Hゲート
    // h_hQuGate:F
    // ┌                                  ┐
    // │ 0.5+0i   0.5+0i   0.5+0i   0.5+0i│
    // │ 0.5+0i  -0.5+0i   0.5+0i  -0.5+0i│
    // │ 0.5+0i   0.5+0i  -0.5+0i  -0.5+0i│
    // │ 0.5+0i  -0.5+0i  -0.5+0i   0.5-0i│
    // └                                  ┘
    
    DJ._print(" 量子ビットQB0_1を量子ゲートH_Hに適用します。");
    QuBit inputQuBit = h_hGate.apply(qb0_1); // 
    DJ.printF("inputQuBit", inputQuBit);
    // inputQuBit:F 
    // ┌       ┐     ┌  ┐
    // │ 0.5+0i│= 1/2│ 1│←────────┐
    // │-0.5+0i│     │-1│←┐(1) product   │(0) product
    // │ 0.5+0i│     │ 1│←┘     = -0.25  │     = -0.25
    // │-0.5+0i│     │-1│←────────┘
    // └       ┘     └  ┘

    
    DJ.printS(2, "DeutschAlgorithm(3-3)", this); // Pause Task --------------------
    DJ._print("Code(3-3)H_Iゲートを準備");  
    DJ._print("・Iゲートを生成します。");  
    QuGate iGate = new QuGate(QuGate.I_GATE);
    DJ.printF("iGate", iGate); // Ｉゲート
    // iGate:F
    // ┌            ┐
    // │ 1+0i   0+0i│
    // │ 0+0i   1+0i│
    // └            ┘    
    
    DJ.print("・HゲートとIゲートを相互作用させ、H_Iゲートを生成します。");
    QuGate h_iGate = hGate.interactQuGate(iGate);
    DJ.printF("h_iGate", h_iGate); // H_Iゲート
    // h_iGate:F
    // ┌                                                  ┐
    // │ 0.70711+0i   0+0i         0.70711+0i   0+0i      │
    // │ 0+0i         0.70711+0i   0+0i         0.70711+0i│
    // │ 0.70711+0i   0+0i        -0.70711+0i   -0+0i     │
    // │ 0+0i         0.70711+0i   -0+0i       -0.70711+0i│
    // └                                                  ┘
    
    
    DJ.printS(2, "DeutschAlgorithm(3-4)", this); // Pause Task --------------------
    DJ._print("Code(3-4)仮想ゲートを呼び出し\n"  
      + "  public QuBit deutschGate(int functionType, QuBit inputQuBit)\n"
      + "  @param functionType int // 関数タイプ \n"  
      + "  @param inputQuBit QuBit // 入力量子ビット（２量子ビット） \n"  
      + "  @return outputQuBit QuBit // 出力量子ビット（２量子ビット）");  

    DJ._print("　測定後の第０量子ビットの遷移確率は１か０になります。\n"
           + "  遷移確率が[1 0]'ならば第０基底軸に遷移し、\n"
           + "  古典ビットは０になり、関数は定数型Constantです。\n"
           + "  遷移確率が[0 1]'ならば第１基底軸に遷移し、\n"
           + "  古典ビットは１になり、関数は均衡型Balancedです。");
      
    
    numOfFuncType = 4;
    
    for (int funcType = 0; funcType < numOfFuncType; funcType++) {
      DJ._print("funcType", funcType);
      
      QuBit outputQuBit = QuGate.deutschGate(funcType, inputQuBit);
      DJ._printF("outputQuBit", outputQuBit);
      // outputQuBit:F 
      // ┌       ┐      ┌ ┐ ┌ ┐     ┌ ┐ ┌ ┐
      // │ 0.5+0i│= 1/2(│1│+│0│) ⊗ (│1│-│0│)
      // │-0.5+0i│      │0│ │1│     │0│ │1│
      // │ 0.5+0i│      └ ┘ └ ┘     └ ┘ └ ┘
      // │-0.5+0i│
      // └       ┘    

      DJ.print("・オラクルを通過した２量子ビットoutputQuBitを"
             + "  H_Iゲートに適用します。");
      QuBit resultQB = h_iGate.apply(outputQuBit); // 
      DJ.printF("resultQB", resultQB);
      // resultQB:F 
      // ┌           ┐   　    ┌ ┐     ┌ ┐ ┌ ┐
      // │ 0+0i      │=　1/√2(│0│) ⊗ (│1│+│0│)
      // │ 0+0i      │     　  │1│     │0│ │1│
      // │ 0.70711+0i│       　└ ┘     └ ┘ └ ┘
      // │-0.70711+0i│
      // └           ┘
      DJ.print("・２量子ビットresultQBはもつれておらず、\n"
             + "　第０量子ビットを分離できます。");

      DJ._print("・２量子ビットresultQBの遷移確率を求めます。");
      double[][] resultTransProb = resultQB.getTransProb();
      DJ.printF("resultTransProb", resultTransProb);
      //  resultTransProb=
      //  [[1.0, 0.5] ←[1 0]'への遷移確率
      //  [0.0, 0.5]  ←[0 1]'への遷移確率
      //  ] ↑   ↑
      //    │   └ 第１量子ビットの遷移確率
      //    └ 第０量子ビットの遷移確率
      DJ.print("　測定後の第０量子ビットの遷移確率は１か０になります。");
      
      
      if (funcType < numOfFuncType - 1) 
        DJ.printS(1, "DeutschAlgorithm(3-4)", this); // Pause Task --------------------
      
    } // 関数タイプで繰り返し
    
    
        
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(3, "DeutschAlgorithm(4)", this); // Pause Task ------------------------
    DJ.print("Code(4)関数別オラクル");
    
    DJ._print("Code(4-0)①関数f0のオラクル");
    DJ.print("  f0(0) = f0(1) = 0：入力が０でも１でも出力は０");
    
    DJ._print("・関数f0は２個のIゲートにより次のように構成されます。");
    DJ.print("  関数f0のオラクル \n"
      + "    ┌-┐ \n"
      + "    ┤I├ \n"
      + "    ├-┤ \n"
      + "    ┤I├ \n"
      + "    └-┘ ");
    DJ.print(
        "          ┌   ┐ ┌   ┐ ┌       ┐ \n" 
      + "  I ⊗ I = │1 0│⊗│1 0│=│1 0 0 0│ \n"
      + "          │0 1│ │0 1│ │0 1 0 0│ \n"
      + "          └   ┘ └   ┘ │0 0 1 0│ \n"
      + "                          │0 0 0 1│ \n"
      + "                          └       ┘ ");
    QuGate oracleF0Gate = iGate.interactQuGate(iGate);
    DJ.printF("oracleF0Gate", oracleF0Gate); // I_Iゲート
    // oracleF1Gate:F    
    // ┌                          ┐ ┌       ┐
    // │ 1+0i   0+0i   0+0i   0+0i│=│1 0 0 0│
    // │ 0+0i   1+0i   0+0i   0+0i│ │0 1 0 0│
    // │ 0+0i   0+0i   1+0i   0+0i│ │0 0 1 0│
    // │ 0+0i   0+0i   0+0i   1+0i│ │0 0 0 1│
    // └                          ┘ └       ┘
    
    
    DJ.printS(1, "DeutschAlgorithm(4-0)②", this); // Pause Task --------------------
    DJ.print("Code(4-0)②関数f0のオラクルを検証");
    
//    int funcType = 0; // 関数タイプ
    DJ._print("・関数タイプ：f0");
    DJ.print("  オラクルの入力と出力を古典ビットで表示します。");
    for (int i = 0; i < numOfInPattern; i++) {
      DJ._print("　入力パターン：inputType = ", i);
    
      // DJ.print("・オラクルに質問");
      QuBit outQuBit = oracleF0Gate.apply(inQuBit[i]);
      // DJ.printF("outQuBit", outQuBit);
      
      // DJ._print("・出力量子ビットを古典ビットに変換します。");
      double[][] outTransProb = outQuBit.getTransProb(); // 遷移確率
      // DJ.printF("outTransProb", outTransProb);
      if (outTransProb[0][0] == 1.0) outputCB[0] = 0;
                                else outputCB[0] = 1;
      if (outTransProb[0][1] == 1.0) outputCB[1] = 0;
                                else outputCB[1] = 1;
      
      DJ.print("  オラクルの入力古典ビット（２進数）"
             + inputCB[i][0] + inputCB[i][1] + "");   // 00 10
      DJ.print("  オラクルの出力古典ビット（２進数）" // ↓ ↓
             + outputCB[0] + outputCB[1] + "");       // 00 10
    }  
      
        
    DJ.printS(1, "DeutschAlgorithm(4-0)③", this); // Pause Task --------------------
    DJ.print("Code(4-0)③　関数f0のオラクルに入力量子ビットを適用");
    DJ.print("QuBit outF0QuBit = oracleF0Gate.apply(inQuBit);");
    QuBit outF0QuBit = oracleF0Gate.apply(inputQuBit);
    DJ.printF("outF0QuBit", outF0QuBit);
    // outF0QuBit:F 
    // ┌       ┐
    // │ 0.5+0i│←────────┐
    // │-0.5+0i│←┐(1) product   │(0) product
    // │ 0.5+0i│←┘     = -0.25  │     = -0.25
    // │-0.5+0i│←────────┘
    // └       ┘

    DJ.print("・オラクルを通過した量子ビットをH_Iゲートに適用します。");
    QuBit resultQB = h_iGate.apply(outF0QuBit);
    DJ.printF("resultQB", resultQB);
    // resultQB:F 
    // ┌           ┐       ┌  ┐
    // │ 0.70711+0i│= 1/√2│ 1│
    // │-0.70711+0i│       │-1│
    // │ 0+0i      │       │ 0│
    // │ 0+0i      │       │ 0│
    // └           ┘       └  ┘
    
    DJ.printS(1, "DeutschAlgorithm(4-0)④", this); // Pause Task --------------------
    DJ.print("Code(4-0)④２量子ビットresultQBの遷移確率");

    DJ.print(
        "       ┌ ┐ ┌ ┐   ┌ ┐   ┌ ┐    \n" 
      + " 1/√2(│1│-│0│+ 0│0│+ 0│0│) = \n"
      + "       │0│ │1│   │0│   │0│    \n"
      + "       │0│ │0│   │1│   │0│    \n"
      + "       │0│ │0│   │0│   │1│    \n"
      + "       └ ┘ └ ┘   └ ┘   └ ┘    \n"
      + "                                     \n"
      + "       ┌ ┐ ┌ ┐          ┌ ┐ ┌ ┐   \n" 
      + " 1/√2(│1│⊗│1│) - 1/√2(│1│⊗│0│)  \n"
      + "       │0│ │0│          │0│ │1│   \n"
      + "       └ ┘ └ ┘          └ ┘ └ ┘   \n"
      + "         ↑    ↑            ↑    ↑     \n"
      + "         │    第１量子ビット│    第１量子ビット \n"
      + "         第０量子ビット      第０量子ビット       \n");
        
    double[][] resultTransProb = resultQB.getTransProb();
      DJ.printF("resultTransProb", resultTransProb);
      //  resultTransProb=
      //  [[1.0, 0.5] ←[1 0]'への遷移確率
      //  [0.0, 0.5]  ←[0 1]'への遷移確率
      //  ] ↑   ↑
      //    │   └ 第１量子ビットの遷移確率
      //    └ 第０量子ビットの遷移確率
      DJ.print("　第０量子ビットの遷移確率は[1 0]'になり,\n"
             + "  測定により得られる古典ビットは０なので、\n"
             + "  関数f0は定数型Constantであると判定されます。");
    
    
    DJ.printS(2, "DeutschAlgorithm(4-1)①", this); // Pause Task ------------------------
    DJ._print("Code(4-1)①関数f1のオラクル");
    DJ.print("　f1(0) = f1(1) = 1：入力が０でも１でも出力は１");
    
    DJ._print("・関数f1はIゲートとXゲートにより構成されます。");
    DJ._print("  関数f1のオラクル \n"
      + "  ┌-┐ \n"
      + "  ┤I├ \n"
      + "  ├-┤ \n"
      + "  ┤X├ \n"
      + "  └-┘ ");

    // DJ.printF("xGate", xGate); // Ｘゲート
    // xGate:F
    // ┌            ┐
    // │ 0+0i   1+0i│
    // │ 1+0i   0+0i│
    // └            ┘    
    
    DJ._print("  I ⊗ X = \n"
      + " ┌   ┐ ┌   ┐ ┌       ┐  ┌   ┐  \n" 
      + " │1 0│⊗│0 1│=│0 1 0 0│= │X O│  \n"
      + " │0 1│ │1 0│ │1 0 0 0│  │O X│  \n"
      + " └   ┘ └   ┘ │0 0 0 1│  └   ┘  \n"
      + "                 │0 0 1 0│           \n"
      + "                 └       ┘           ");
    
    DJ.print("I ⊗ X");
    QuGate oracleF1Gate = iGate.interactQuGate(xGate);
    DJ.printF("oracleF1Gate", oracleF1Gate); // I_Xゲート
    // oracleF1Gate:F    
    // ┌                          ┐
    // │ 0+0i   1+0i   0+0i   0+0i│
    // │ 1+0i   0+0i   0+0i   0+0i│
    // │ 0+0i   0+0i   0+0i   1+0i│
    // │ 0+0i   0+0i   1+0i   0+0i│
    // └                          ┘    
    
    DJ.printS(1, "DeutschAlgorithm(4-1)②", this); // Pause Task --------------------
    DJ.print("Code(4-1)②　関数f1のオラクルを検証");

    DJ._print("・関数タイプ：f1");
    DJ.print("  オラクルの入力と出力を古典ビットで表示します。");
    for (int i = 0; i < numOfInPattern; i++) {
      DJ._print("　入力パターン：inputType = ", i);
      // DJ.print("・オラクルに質問");
      QuBit outQuBit = oracleF1Gate.apply(inQuBit[i]);
      // DJ.print("・オラクルの出力量子ビット（２量子ビット１個）");
      // DJ.print("  outQuBit", outQuBit, QuBit.H_BIT);
    
      // DJ.print("　出力量子ビットを古典ビットに変換します。");
      double[][] outTransProb = outQuBit.getTransProb();
      // DJ.printF("outTransProb", outTransProb);
      if (outTransProb[0][0] == 1.0) outputCB[0] = 0;
                                else outputCB[0] = 1;
      if (outTransProb[0][1] == 1.0) outputCB[1] = 0;
                                else outputCB[1] = 1;
      
      DJ.print("・オラクルの入力古典ビット（２進数）"
             + inputCB[i][0] + inputCB[i][1] + "");   // 00 10
      DJ.print("・オラクルの出力古典ビット（２進数）" // ↓ ↓
             + outputCB[0] + outputCB[1] + "");       // 01 11
    }

    
    DJ.printS(1, "DeutschAlgorithm(4-1)③", this); // Pause Task --------------------
    DJ.print("Code(4-1)③　関数f1のオラクルに入力量子ビットを適用");
    DJ.print("QuBit outF1QuBit = oracleF1Gate.apply(inQuBit);");
    QuBit outF1QuBit = oracleF1Gate.apply(inputQuBit); // 
    DJ.printF("outF1QuBit", outF1QuBit);
    // outF1QuBit:F 
    // ┌       ┐
    // │-0.5+0i│←────────┐
    // │ 0.5+0i│←┐(1) product   │(0) product
    // │-0.5+0i│←┘     = -0.25  │     = -0.25
    // │ 0.5+0i│←────────┘
    // └       ┘

    DJ.print("・オラクルを通過した量子ビットをH_Iゲートに適用する。");
    resultQB = h_iGate.apply(outF1QuBit); // 
    DJ.printF("resultQB", resultQB);
    // resultQB:F 
    // ┌           ┐       ┌  ┐
    // │-0.70711+0i│= 1/√2│-1│
    // │ 0.70711+0i│       │ 1│
    // │ 0+0i      │       │ 0│
    // │ 0+0i      │       │ 0│
    // └           ┘       └  ┘
    
    DJ._print("・２量子ビットresultQBの遷移確率を求めます。");
    resultTransProb = resultQB.getTransProb();
    DJ.printF("resultTransProb", resultTransProb);
    //  resultTransProb=
    //  [[1.0, 0.5] ←[1 0]'への遷移確率
    //  [0.0, 0.5]  ←[0 1]'への遷移確率
    //  ] ↑   ↑
    //    │   └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("　第０量子ビットの遷移確率は[1 0]'になり,\n"
           + "  測定により得られる古典ビットは０なので、\n"
           + "  関数f1は定数型Constantであると判定されます。");
    

    
    DJ.printS(2, "DeutschAlgorithm(4-2)①", this); // Pause Task ------------------------
    DJ._print("Code(4-2)①関数f2のオラクル");
    DJ.print("  f2(0) = 0, f2(1) = 1：\n"
           + "  入力が０なら出力は０、入力が１なら出力は１");
    DJ._print("・関数f2はCXゲート(control-notゲート)により構成されます。");
    
    DJ._print("  関数f2のオラクル \n"
      + "  ┌-┐ \n"
      + "  ┤C├ \n"
      + "  │|│ \n"
      + "  ┤X├ \n"
      + "  └-┘ ");

    DJ._print(" ＣＸゲート = \n"
      + " ┌       ┐  ┌   ┐  \n" 
      + " │1 0 0 0│= │I O│  \n"
      + " │0 1 0 0│  │O X│  \n"
      + " │0 0 0 1│  └   ┘  \n"
      + " │0 0 1 0│           \n"
      + " └       ┘           ");
    
    int _numOfBits = 2; // 量子ビット数
    int _target = 1; // ターゲット量子ビット
    // DJ.print_("_numOfBits = " + _numOfBits + ", _target = " + _target);
    int[] _controlArray = {0}; // 制御量子ビット
    // DJ.print(", _controlArray", _controlArray);
    // numOfBits = 2, target = 1, controlArray[0]
    QuGate cxGate = new QuGate(
        QuGate.CX_GATE, _numOfBits, _target, _controlArray); // ＣＸゲート
    QuGate oracleF2Gate = new QuGate(_numOfBits);
    oracleF2Gate.setGateMat(cxGate.copyGateMat()); // ＣＸゲートをコピー
    
    DJ.printF("oracleF2Gate", oracleF2Gate);
    // oracleF2Gate:F
    // ┌                          ┐
    // │ 1+0i   0+0i   0+0i   0+0i│
    // │ 0+0i   1+0i   0+0i   0+0i│
    // │ 0+0i   0+0i   0+0i   1+0i│
    // │ 0+0i   0+0i   1+0i   0+0i│
    // └                          ┘
    
    DJ.printS(1, "DeutschAlgorithm(4-2)②", this); // Pause Task --------------------
    DJ.print("Code(4-2)②関数f2のオラクルを検証");

    DJ._print("・関数タイプ：f2");
    DJ.print("・オラクルの入力と出力を古典ビットで表示します。");
    for (int i = 0; i < numOfInPattern; i++) {
      DJ._print("　入力パターン：inputType = ", i);
      // DJ.print("・オラクルに質問");
      QuBit outQuBit = oracleF2Gate.apply(inQuBit[i]);
      // DJ.print("・オラクルの出力量子ビット（２量子ビット１個）");
      // DJ.print("  outQuBit", outQuBit, QuBit.H_BIT);
    
      // DJ.print("　出力量子ビットを古典ビットに変換します。");
      double[][] outTransProb = outQuBit.getTransProb();
      // DJ.printF("outTransProb", outTransProb);
      if (outTransProb[0][0] == 1.0) outputCB[0] = 0;
                                else outputCB[0] = 1;
      if (outTransProb[0][1] == 1.0) outputCB[1] = 0;
                                else outputCB[1] = 1;
      
      DJ.print("・オラクルの入力古典ビット（２進数）"
             + inputCB[i][0] + inputCB[i][1] + "");   // 00 10
      DJ.print("・オラクルの出力古典ビット（２進数）" // ↓ ↓
             + outputCB[0] + outputCB[1] + "");       // 00 11
    }
    
    DJ.printS(1, "DeutschAlgorithm(4-2)③", this); // Pause Task --------------------
    DJ.print("Code(4-2)③関数f2のオラクルに入力量子ビットを適用");
    DJ.print("QuBit outF2QuBit = oracleF2Gate.apply(inQuBit);");
    QuBit outF2QuBit = oracleF2Gate.apply(inputQuBit); // 
    DJ.printF("outF2QuBit", outF2QuBit);
    // outF2QuBit:F 
    // ┌       ┐
    // │ 0.5+0i│←────────┐
    // │-0.5+0i│←┐(1) product   │(0) product
    // │-0.5+0i│←┘     = 0.25   │     = 0.25
    // │ 0.5+0i│←────────┘
    // └       ┘
    
    DJ.print("・オラクルを通過した量子ビットをH_Iゲートに適用します。");
    resultQB = h_iGate.apply(outF2QuBit); // 
    DJ.printF("resultQB", resultQB);
    // resultQB:F 
    // ┌           ┐       ┌  ┐
    // │ 0+0i      │= 1/√2│ 0│
    // │ 0+0i      │       │ 0│
    // │ 0.70711+0i│       │ 1│
    // │-0.70711+0i│       │-1│
    // └           ┘       └  ┘
    //  DJ.print(
    //      "        ┌ ┐   ┌ ┐ ┌ ┐ ┌ ┐    \n" 
    //    + " 1/√2(0│1│+ 0│0│+│0│-│0│) = \n"
    //    + "        │0│   │1│ │0│ │0│    \n"
    //    + "        │0│   │0│ │1│ │0│    \n"
    //    + "        │0│   │0│ │0│ │1│    \n"
    //    + "        └ ┘   └ ┘ └ ┘ └ ┘    \n"
    //    + "                                       \n"
    //    + "       ┌ ┐ ┌ ┐          ┌ ┐ ┌ ┐   \n" 
    //    + " 1/√2(│0│⊗│1│) - 1/√2(│0│⊗│0│)  \n"
    //    + "       │1│ │0│          │1│ │1│   \n"
    //    + "       └ ┘ └ ┘          └ ┘ └ ┘   \n"
    //    + "         ↑    ↑             ↑    ↑    \n"
    //    + "         │    第１量子ビット │    第１量子ビット \n"
    //    + "         第０量子ビット       第０量子ビット       \n");
    
    DJ._print("・２量子ビットresultQBの遷移確率を求めます。");
    resultTransProb = resultQB.getTransProb();
    DJ.printF("resultTransProb", resultTransProb);
    //  resultTransProb=
    //  [[0.0, 0.5] ←[1 0]'への遷移確率
    //  [1.0, 0.5]  ←[0 1]'への遷移確率
    //  ] ↑   ↑
    //    │   └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("　第０量子ビットの遷移確率は[0 1]'になり,\n"
           + "  測定により得られる古典ビットは１なので、\n"
           + "  関数f2は均衡型Balancedであると判定されます。");
    

    
    DJ.printS(2, "DeutschAlgorithm(4-3)①", this); // Pause Task ------------------------
    DJ._print("Code(4-3)①関数f3のオラクル\n"
            + "  f3(0) = 1, f3(1) = 0：");
    DJ.print("　入力が０なら出力は１、入力が１なら出力は０");
    
    DJ._print("・関数f3は以下に示すXI_CX_XIゲートにより構成されます。");
    
    DJ._print("  関数f3のオラクル：XI_CX_XIゲート \n"
      + "  ┌-┐┌-┐┌-┐ \n"
      + "  ┤X├┤C├┤X├ \n"
      + "  ├ ┤├|┤├ ┤ \n"
      + "  ┤I├┤X├┤I├ \n"
      + "  └-┘└-┘└-┘ ");

    DJ.printS(1, "DeutschAlgorithm(4-3)②", this); // Pause Task --------------------
    DJ.print("Code(4-3)②XゲートとIゲートにより構成されるX_Iゲート");
    DJ._print("XIゲート \n"
      + " ┌-┐ \n" 
      + " │X│ \n"
      + " ├ ┤ \n"
      + " │I│ \n"
      + " └-┘ ");

    DJ._print("  X ⊗ I = \n"
      + " ┌   ┐ ┌   ┐ ┌       ┐  ┌   ┐  \n" 
      + " │0 1│⊗│1 0│=│0 0 1 0│= │O I│  \n"
      + " │1 0│ │0 1│ │0 0 0 1│  │I O│  \n"
      + " └   ┘ └   ┘ │1 0 0 0│  └   ┘  \n"
      + "                 │0 1 0 0│           \n"
      + "                 └       ┘           ");
    
    DJ.print("X ⊗ I");
    QuGate xiGate = xGate.interactQuGate(iGate);
    DJ.printF("xiGate", xiGate); // X_Iゲート
    // xiGate:F    
    // ┌                          ┐
    // │ 0+0i   0+0i   1+0i   0+0i│
    // │ 0+0i   0+0i   0+0i   1+0i│
    // │ 1+0i   0+0i   0+0i   0+0i│
    // │ 0+0i   1+0i   0+0i   0+0i│
    // └                          ┘    
    
    DJ.printS(1, "DeutschAlgorithm(4-3)③", this); // Pause Task --------------------
    DJ.print("Code(4-3)③CX_XIゲート");
    DJ.print("cxGate = \n"
      + " ┌       ┐  ┌   ┐  \n" 
      + " │1 0 0 0│= │I O│  \n"
      + " │0 1 0 0│  │O X│  \n"
      + " │0 0 0 1│  └   ┘  \n"
      + " │0 0 1 0│           \n"
      + " └       ┘           ");
    DJ.printF("cxGate", cxGate); // ＣＸゲート
    
    DJ._print("・CXゲートとX_Iゲートの内積によりCX_XIゲートを生成する。");
    DJ._print("  CX・XI = \n"
      + " ┌       ┐' ┌       ┐ ┌       ┐  ┌   ┐  \n" 
      + " │1 0 0 0│・│0 0 1 0│=│0 0 1 0│= │O X│  \n"
      + " │0 1 0 0│  │0 0 0 1│ │0 0 0 1│  │I O│  \n"
      + " │0 0 0 1│  │1 0 0 0│ │0 1 0 0│  └   ┘  \n"
      + " │0 0 1 0│  │0 1 0 0│ │1 0 0 0│           \n"
      + " └       ┘  └       ┘ └       ┘           ");
  
    QuGate cx_xiGate = cxGate.innerProduct(xiGate);
    DJ.printF("cx_xiGate", cx_xiGate); // CX_XIゲート
    // cx_xiGate:F
    // ┌                          ┐
    // │ 0+0i   0+0i   1+0i   0+0i│
    // │ 0+0i   0+0i   0+0i   1+0i│
    // │ 0+0i   1+0i   0+0i   0+0i│
    // │ 1+0i   0+0i   0+0i   0+0i│
    // └                          ┘
    
    DJ.printS(1, "DeutschAlgorithm(4-3)④", this); // Pause Task --------------------
    DJ._print("Code(4-3)④X_IゲートとCX_XIゲートの内積により"
            + "XI_CX_XIゲートを生成");
    DJ.print("  XI_CX・XI  = \n"
      + " ┌       ┐' ┌       ┐ ┌       ┐  ┌   ┐  \n" 
      + " │0 0 1 0│・│0 0 1 0│=│0 1 0 0│= │X O│  \n"
      + " │0 0 0 1│  │0 0 0 1│ │1 0 0 0│  │O I│  \n"
      + " │1 0 0 0│  │0 1 0 0│ │0 0 1 0│  └   ┘  \n"
      + " │0 1 0 0│  │1 0 0 0│ │0 0 0 1│           \n"
      + " └       ┘  └       ┘ └       ┘           ");
 
    DJ._print("・オラクルoracleF3Gate：XI_CX_XIゲート");
    QuGate oracleF3Gate = xiGate.innerProduct(cx_xiGate);
    DJ.printF("oracleF3Gate", oracleF3Gate); // XI_CX_XIゲート
    // oracleF3Gate:F
    // ┌                          ┐
    // │ 0+0i   1+0i   0+0i   0+0i│
    // │ 1+0i   0+0i   0+0i   0+0i│
    // │ 0+0i   0+0i   1+0i   0+0i│
    // │ 0+0i   0+0i   0+0i   1+0i│
    // └                          ┘

      
    DJ.printS(1, "DeutschAlgorithm(4-3)⑤", this); // Pause Task --------------------
    DJ.print("Code(4-3)⑤関数f3のオラクルを検証");

    DJ._print("・関数タイプ：f3");
    DJ.print("・オラクルの入力と出力を古典ビットで表示します。");
    for (int i = 0; i < numOfInPattern; i++) {
      DJ._print("　入力パターン：inputType = ", i);
      // DJ.print("・オラクルに質問");
      QuBit outQuBit = oracleF3Gate.apply(inQuBit[i]);
      // DJ.print("・オラクルの出力量子ビット（２量子ビット１個）");
      // DJ.print("  outQuBit", outQuBit, QuBit.H_BIT);
    
      // DJ.print("　出力量子ビットを古典ビットに変換します。");
      double[][] outTransProb = outQuBit.getTransProb();
      // DJ.printF("outTransProb", outTransProb);
      if (outTransProb[0][0] == 1.0) outputCB[0] = 0;
                                else outputCB[0] = 1;
      if (outTransProb[0][1] == 1.0) outputCB[1] = 0;
                                else outputCB[1] = 1;
      
      DJ.print("・オラクルの入力古典ビット（２進数）"
             + inputCB[i][0] + inputCB[i][1] + "");   // 00 10
      DJ.print("・オラクルの出力古典ビット（２進数）" // ↓ ↓
             + outputCB[0] + outputCB[1] + "");       // 01 10
    }


    
    DJ.printS(1, "DeutschAlgorithm(4-3)⑥", this); // Pause Task --------------------
    DJ.print("Code(4-3)⑥関数f3のオラクルに入力量子ビットを適用");
    DJ.print("QuBit outF3QuBit = oracleF3Gate.apply(inQuBit);");
    QuBit outF3QuBit = oracleF3Gate.apply(inputQuBit); // 
    DJ._print("  oracleF3Gate'・inQuBit= \n"
      + " ┌       ┐'    ┌  ┐     ┌  ┐  \n" 
      + " │0 1 0 0│・1/2│ 1│= 1/2│-1│  \n"
      + " │1 0 0 0│     │-1│     │ 1│  \n"
      + " │0 0 1 0│     │ 1│     │ 1│  \n"
      + " │0 0 0 1│     │-1│     │-1│  \n"
      + " └       ┘     └  ┘     └  ┘  ");
    
    DJ.printF("outF3QuBit", outF3QuBit);
    // outF3QuBit:F 
    // ┌       ┐
    // │-0.5+0i│←────────┐
    // │ 0.5+0i│←┐(1) product   │(0) product
    // │ 0.5+0i│←┘     = 0.25   │     = 0.25
    // │-0.5+0i│←────────┘
    // └       ┘
    
    DJ.print("・オラクルを通過した量子ビットをH_Iゲートに適用します。");
    resultQB = h_iGate.apply(outF3QuBit); // 
    DJ.printF("resultQB", resultQB);
    // resultQB:F
    //  ┌           ┐       ┌  ┐
    //  │ 0+0i      │= 1/√2│ 0│
    //  │ 0+0i      │       │ 0│
    //  │-0.70711+0i│       │-1│
    //  │ 0.70711+0i│       │ 1│
    //  └           ┘       └  ┘
      
      DJ._print("・２量子ビットresultQBの遷移確率を求めます。");
      resultTransProb = resultQB.getTransProb();
      DJ.printF("resultTransProb", resultTransProb);
      //  resultTransProb=
      //  [[0.0, 0.5] ←[1 0]'への遷移確率
      //  [1.0, 0.5]  ←[0 1]'への遷移確率
      //  ] ↑   ↑
      //    │   └ 第１量子ビットの遷移確率
      //    └ 第０量子ビットの遷移確率
      DJ.print("　第０量子ビットの遷移確率は[0 1]'になり,\n"
             + "  測定により得られる古典ビットは１ですので、\n"
             + "  関数f3は均衡型Balancedであると判定されます。");
      
    
    // DJ.printS(2, "DeutschAlgorithm(Ending)", this); // Pause Task ---------------------
    DJ._print("量子計算の完了");
    DJ.print("DeutschAlgorithm.deutschAlgorithm() ===========================");
  } // deutschAlgorithm()
    
} // DeutschAlgorithm Class


// End of file
