/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva))
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2025
 */
package task.qu;

import quan.QuBit;
import quan.QuGate;
import task.Task;
import util.DJ;
import util.TimeStamp;

/**
 * <p> 表　題: Class: PhaseKickback</p>
 * <p> 説　明: 位相キックバック</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024, 2025</p>
 * <p> 作成日: 2025.05.13</p>
 */
public class PhaseKickback extends Task {
  
  // 学習・試行制御
  // int epoch = 2; // エポックの回数
  // int interval = 1; //10; // 経過表示間隔
  
  // 量子計算パラメータ
  // int delta = 30; // 5; //10; //15; // 基底角度の増分（度）

  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    //  patternViewerFlag = true; //true; // false:非表示
    //  patternData0 = new double[5][dataNum]; // 附番,
    //  patternData1 = new double[5][dataNum]; // ID,
    //  patternViewerLauncher = DJ.pattern(  // 判定パターン、データ、タイトル
    //      PatternViewer.PATTERN_TUNER, patternData0, "PhaseKickback",
    //      PatternViewer.PATTERN_TUNER, patternData1, "PhaseKickback");
    
    this.stepMode = true; // タスクのステップ実行モード
    
    phaseKickback(); // タスク本体の呼び出し
  }
  
  /** 
   * 位相キックバック
   */
  public void phaseKickback() {
    DJ._print("PhaseKickback.phaseKickback() ========================");
    DJ.print("(1)位相キックバックの量子回路");
    DJ.print("(2)位相回転ゲートの固有値と固有ベクトル");
    DJ.print("(3)ユニタリ行列にZゲートを使用");
    DJ.print("(4)ユニタリ行列にXゲートを使用");
    DJ.print("(5)ユニタリ行列にSゲートを使用");
    DJ.print("(6)非固有ベクトルの場合");
        
    // this.stepMode = true; // タスクのステップ実行モードON
    DJ.printS(2, "PhaseKickback(1)", this); // Pause Task ------------------
    DJ._print("Code(1)位相キックバックの量子回路");
    DJ.print("・位相キックバックはアダマール・ゲートHと、\n"
           + "　ユニタリ行列Uで構成されるアルゴリズムです。");
    DJ.print("・補助量子ビットと入力量子ビットを与えると、\n"
           + "　U行列の固有値が補助量子ビットに反映されます。");
    
    DJ._print("・位相キックバックの量子回路\n"
      + "               ┌-┐  ┌-┐  ┌-┐                  \n"
      + "qb0 = [1 0]' ─┤H├─┤C├─┤M├ ← 補助量子ビット\n"
      + "               ├-┤  ├|┤  └-┘                  \n"
      + "qb1 = f  ───┤I├─┤U├─      ← 入力量子ビット\n"
      + "               └-┘  └-┘                         \n"
      + "全量子状態  qbf0   qbf1   qbf2     ");
    
    DJ._print("・位相キックバックの量子回路の実行に必要な"
           + "　量子ビットと量子回路を準備します。");

    DJ.printS(1, "PhaseKickback(1-1)", this); // Pause Task ------------
    DJ._print("Code(1-1)標準量子ビットe0QBとe1QBを生成");
    QuBit e0QB = new QuBit(QuBit.STD0_QUBIT);
    QuBit e1QB = new QuBit(QuBit.STD1_QUBIT);
    DJ.printF("e0QB", e0QB);
    DJ.printF("e1QB", e1QB);
    // e0QB:F              e1QB:F
    // ┌     ┐           ┌     ┐
    // │ 1+0i│= [1 0]',  │ 0+0i│= [0 1]'
    // │ 0+0i│           │ 1+0i│
    // └     ┘           └     ┘
    
    DJ.printS(1, "PhaseKickback(1-2)", this); // Pause Task --------------
    DJ.print("Code(1-2)補助量子ビットを生成");
    DJ._print("・標準量子ビットe0QBを、"
           + "補助量子ビットqb0にコピーします。");
    QuBit qb0 = new QuBit(e0QB);
    DJ.printF("qb0", qb0); // 補助量子ビット
    // qb0:F
    // ┌     ┐
    // │ 1+0i│= [1 0]'
    // │ 0+0i│
    // └     ┘
    
    DJ.print("・Iゲート（恒等ゲート）を生成します。");
    QuGate iGate = new QuGate(QuGate.I_GATE);
    DJ.printF("iGate", iGate); // Ｉゲート
    // iGate:F
    // ┌            ┐  ┌    ┐
    // │ 1+0i   0+0i│= │1  0│
    // │ 0+0i   1+0i│  │0  1│
    // └            ┘  └    ┘
    
    DJ.printS(1, "PhaseKickback(1-3)", this); // Pause Task --------------
    DJ.print("Code(1-3)Hゲート（Hadamardゲート）を生成");
    QuGate hQuGate = new QuGate(QuGate.H_GATE);
    DJ.printF("hQuGate", hQuGate); // Hゲート
    // hGate:F
    // ┌                        ┐         ┌    ┐ 
    // │ 0.70711+0i   0.70711+0i│= (1/√2)│1  1│ 
    // │ 0.70711+0i  -0.70711+0i│         │1 -1│
    // └                        ┘         └    ┘
    
    DJ._print("Code(1-4)HIゲートを生成");
    DJ.print("・HゲートとIゲートを相互作用させ、"
            + "２量子ゲート(H ⊗ I)を生成します。");
    QuGate hIQuGate = hQuGate.interactQuGate(iGate);
    DJ.printF("hIQuGate", hIQuGate); // HIゲート
    // hiQuGate:F
    // ┌                                                  ┐         ┌    ┐
    // │ 0.70711+0i   0+0i         0.70711+0i   0+0i      │= (1/√2)│I  I│
    // │ 0+0i         0.70711+0i   0+0i         0.70711+0i│         │I -I│
    // │ 0.70711+0i   0+0i        -0.70711+0i   -0+0i     │         └    ┘
    // │ 0+0i         0.70711+0i   -0+0i       -0.70711+0i│
    // └                                                  ┘
    
    
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(2, "PhaseKickback(2)", this); // Pause Task --------------
    DJ.print("Code(2)位相回転ゲートの固有値と固有ベクトル");

    DJ._print("Code(2-1)①Zゲートを生成");
    DJ.print("・Zゲートを通常の生成子で生成します。");
    QuGate zQuGate = new QuGate(QuGate.Z_GATE);
    DJ.printF("zQuGate", zQuGate);
    //  zQuGate:F
    //  ┌            ┐ ┌    ┐
    //  │ 1+0i   0+0i│=│1  0│
    //  │ 0+0i  -1+0i│ │0 -1│
    //  └            ┘ └    ┘
    DJ.print("・Zゲートの固有値はv0=+1とv1=-1で、\n"
           + "  固有ベクトルはf0=[1 0]'とf1=[0 1]'です。");
    DJ._print("・固有値と固有ベクトルは次式を満たします。\n"
            + "  Z f = v f");
    
    
   DJ.printS(1, "PhaseKickback(2-1)②", this); // Pause Task --------------
    DJ._print("Code(2-1)②Zゲートの固有値と固有ベクトルを確認");
    DJ._print("・固有ベクトルf0をZゲートに適用します。\n"
            + "  Z f0 = Z e0 = +1 [1 0]' = v0 f0");
    QuBit z_e0QB = zQuGate.apply(e0QB);
    DJ.printF("z_e0QB", z_e0QB);
    //  z_e0QB:F 
    //  ┌     ┐     ┌ ┐
    //  │ 1+0i│= +1(│1│)
    //  │ 0+0i│     │0│
    //  └     ┘     └ ┘
    DJ.print("・固有ベクトルと固有値の関係を満たしています。");
    
    DJ._print("・固有ベクトルf1をZゲートに適用します。\n"
            + "  Z f1 = Z e1 = -1 [0 1]' = v1 f1");
    QuBit z_e1QB = zQuGate.apply(e1QB);
    DJ.printF("z_e1QB", z_e1QB);
    //  z_e1QB:F 
    //  ┌     ┐     ┌ ┐
    //  │ 0+0i│= -1(│0│)
    //  │-1+0i│     │1│
    //  └     ┘     └ ┘
    DJ.print("・固有ベクトルと固有値の関係を満たしています。");
    
    
   DJ.printS(1, "PhaseKickback(2-1)③", this); // Pause Task --------------
    DJ._print("Code(2-1)③Zゲートを位相回転ゲートで生成");
    DJ.print("・R0ゲート(Rkゲートのk=0の場合)を生成します。");
    int k = 0;
    DJ.print_("k = ", k);
    QuGate r0Gate = new QuGate(QuGate.RK_GATE, k);
    DJ.printF(",  r0Gate = zQuGate", r0Gate);
    // k = 0,  r0Gate:F
    // ┌            ┐ ┌                   ┐ ┌    ┐
    // │ 1+0i   0+0i│=│1  0               │=│1  0│
    // │ 0+0i  -1+0i│ │0 exp(i2π/2^(k+1))│ │0 -1│
    // └            ┘ └                   ┘ └    ┘
    // オイラーの公式 exp(iθ) = cos(θ) + isin(θ) より、
    // exp(i2π/2^(k+1)) = exp(i2π/2^1) = cos(π) + isin(π) = -1+0i
    
    DJ.print("・Zゲートはk=0のRkゲートR0と同じです。\n"
           + "  Zゲートの固有値はv0=+1とv1=-1で、\n"
           + "  固有ベクトルはf0=[1 0]'とf1=[0 1]'です。");
    
    
    DJ.printS(1, "PhaseKickback(2-2)", this); // Pause Task --------------
    DJ.print("Code(2-2)コントロール・ユニタリ・ゲートC-Uを生成");
    DJ.print("　ユニタリ行列UとしてR0=Z行列を使います。");
    QuGate targetGate = new QuGate(); // ターゲット量子ゲート
    targetGate.setGateMat(r0Gate.copyGateMat());
    int numOfBits = 2; // 量子ビット数
    int target = 1; // ターゲット量子ビットのインデックス
    int[] controlArray = new int[1]; // コントロール量子ビット配列
    controlArray[0] = 0; // コントロール量子ビットのインデックス
    DJ.print_("targetGate = R0-Gate" + "numOfBits = " + numOfBits
          + ", target = " + target + ", controlArray", controlArray);
    DJ.print("QuGate cR0Gate = new QuGate("
          + "targetGate, numOfBits, target, controlArray);");
    QuGate cR0Gate = new QuGate(targetGate, numOfBits, target, controlArray);
    DJ.printF("cR0Gate", cR0Gate); // C-Rk0ゲート
    // cR0Gate:F
    // # qbTypeArray[0, 1, 1, 2]
    // # gateMarker[0, 0, 1, 2]
    // ┌                          ┐ ┌                     ┐ 
    // │ 1+0i   0+0i   0+0i   0+0i│=│ 1   0   0   0       │  
    // │ 0+0i   1+0i   0+0i   0+0i│ │ 0   1   0   0       │ 
    // │ 0+0i   0+0i   1+0i   0+0i│ │ 0   0   1   0       │ 
    // │ 0+0i   0+0i   0+0i  -1+0i│ │ 0   0   0   exp(iπ)│ 
    // └                          ┘ └                     ┘ 
    
    
    
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(2, "PhaseKickback(3)", this); // Pause Task ------------------
    DJ.print("Code(3)ユニタリ行列にZゲートを使用");
    
    DJ._print("Code(3-1)①入力量子ビットに固有ベクトルを使用");
    DJ.print("  入力量子ビットfに固有ベクトルf0=[1 0]'を使用し、\n"
           + "  補助量子ビットqb0に入力量子ビットfを相互作用させ、\n"
           + "　２量子ビットの量子ビット状態qbf0を生成します。");
    QuBit f0QB = new QuBit(e0QB); // [1 0]'
    QuBit qbf0 = qb0.interactQuBit(f0QB);
    DJ.printF("qbf0", qbf0);
    // qbf0:F 
    // ┌     ┐
    // │ 1+0i│= qb0 ⊗ f = [1 0] ⊗ [1 0]'
    // │ 0+0i│
    // │ 0+0i│
    // │ 0+0i│
    // └     ┘

    DJ._print("・量子ビット状態qbf0を量子ゲートhiQuGateに適用します。");
    DJ.print("QuBit qbf1 = hiQuGate.apply(qbf0);");
    QuBit qbf1 = hIQuGate.apply(qbf0);
    DJ.printF("qbf1", qbf1);
    // qbf1:F 
    // ┌           ┐       ┌     ┐ ┌     ┐    ┌     ┐
    // │ 0.70711+0i│=1/√2(│ 1+0i│+│ 0+0i│) ⊗ │ 1+0i│
    // │ 0+0i      │       │ 0+0i│ │ 1+0i│    │ 0+0i│
    // │ 0.70711+0i│       └     ┘ └     ┘    └     ┘
    // │ 0+0i      │
    // └           ┘
    
    
    DJ.printS(1, "PhaseKickback(3-1)②", this); // Pause Task ------------------
    DJ._print("Code(3-1)②量子ビット状態をユニタリ量子ゲートに適用");
    DJ._print("  量子ビット状態qbf1を量子ゲートcR0Gateに適用します。");
    DJ.print("QuBit qbf2 = cR0Gate.apply(qbf1);");
    QuBit qbf2 = cR0Gate.apply(qbf1);
    DJ.printF("qbf2", qbf2);
    // cRk0hi_e10f1QB:F 
    // ┌           ┐            ┌ ┐  ┌ ┐    ┌ ┐
    // │ 0.70711+0i│= (1/√(2))(│1│+1│0│) ⊗ │1│
    // │ 0+0i      │            │0│  │1│    │0│
    // │ 0.70711+0i│            └ ┘  └ ┘    └ ┘
    // │ 0+0i      │
    // └           ┘
    DJ.print("・入力量子ビットが固有ベクトルf0=[1 0]'ですので、\n"
           + "  補助量子ビットには固有値v0=+1が"
           + "位相キックバックされます。");
    
    DJ._print("・量子ビットごとの遷移確率を収集します。");
    double[][] result = qbf2.getTransProb();
    DJ.printF("result", result);
    //  result=
    //  [[0.5, 1]←[1 0]'への遷移確率
    //  [0.5, 0] ←[0 1]'への遷移確率
    //  ] ↑  ↑         
    //    │  └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("・[1 0]'と[0 1]'への遷移確率は共に0.5になります。");
    

    DJ.printS(1, "PhaseKickback(3-2)①", this); // Pause Task ------------------
    DJ._print("Code(3-2)①入力量子ビットに固有ベクトルを使用");
    DJ.print("・入力量子ビットfに固有ベクトルはf1=[0 1]'を使用し、\n"
           + "  補助量子ビットqb0に入力量子ビットfを相互作用させ、\n"
           + "  ２量子ビットの量子ビット状態qbf0を生成します。");
    QuBit f1QB = new QuBit(e1QB); // 
    qbf0 = qb0.interactQuBit(f1QB);
    DJ.printF("qbf0", qbf0);
    // qbf0:F 
    // ┌     ┐
    // │ 0+0i│= qb0 ⊗ f = [1 0] ⊗ [0 1]'
    // │ 1+0i│
    // │ 0+0i│
    // │ 0+0i│
    // └     ┘
    
    DJ._print("・量子ビット状態qbf0を量子ゲートhiQuGateに適用します。");
    DJ.print("QuBit qbf1 = hiQuGate.apply(qbf0);");
    qbf1 = hIQuGate.apply(qbf0);
    DJ.printF("qbf1", qbf1);
    // qbf1:F 
    // ┌           ┐        ┌ ┐ ┌ ┐          ┌ ┐ ┌ ┐ ┌ ┐ ┌ ┐
    // │ 0+0i      │= 1/√2(│0│+│0│) = 1/√2(│1│⊗│0│+│0│⊗│0│)
    // │ 0.70711+0i│        │1│ │0│　        │0│ │1│ │1│ │1│　
    // │ 0+0i      │        │0│ │0│          └ ┘ └ ┘ └ ┘ └ ┘
    // │ 0.70711+0i│        │0│ │1│
    // └           ┘        └ ┘ └ ┘
    
    
    DJ.printS(1, "PhaseKickback(3-2)②", this); // Pause Task ------------------
    DJ._print("Code(3-2)②量子ビット状態をユニタリ量子ゲートに適用");
    DJ.print("  量子ビット状態qbf1を量子ゲートcRk0Gateに適用します。");
    qbf2 = cR0Gate.apply(qbf1);
    DJ.printF("qbf2", qbf2);
    // cRk0hi_e10f1QB:F 
    // ┌           ┐       ┌ ┐          ┌ ┐  ┌ ┐
    // │ 0+0i      │=1/√2(│1│+ exp(iπ)│0│)⊗│0│
    // │ 0.70711+0i│       │0│          │1│  │1│
    // │ 0+0i      │       └ ┘          └ ┘  └ ┘
    // │-0.70711+0i│               ┌ ┐  ┌ ┐  ┌ ┐
    // └           ┘        =1/√2(│1│-1│0│)⊗│0│
    //                               │0│  │1│  │1│
    //                               └ ┘  └ ┘  └ ┘
    DJ.print("・入力量子ビットが固有ベクトルf1=[0 1]'の場合には"
           + "  補助量子ビットには固有値v1=-1が"
           + "  位相キックバックされます。");
    
    DJ._print("・量子ビットごとの遷移確率を収集します。");
    result = qbf2.getTransProb();
    DJ.printF("result", result);
    //  result=
    //  [[0.5, 0]←[1 0]'への遷移確率
    //  [0.5, 1] ←[0 1]'への遷移確率
    //  ] ↑  ↑         
    //    │  └ 第１量子ビットの遷移確率
    //    └ 第０量子ビットの遷移確率
    DJ.print("・遷移確率は正の値なので確率は同じになります。");
    

    
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(2, "PhaseKickback(4)", this); // Pause Task ------------------
    DJ.print("Code(4)ユニタリ行列にXゲートを使用");
    
    DJ._print("・位相キックバックの量子回路\n"
      + "               ┌-┐  ┌-┐  ┌-┐                  \n"
      + "qb0 = [1 0]' ─┤H├─┤C├─┤M├ ← 補助量子ビット\n"
      + "               ├-┤  ├|┤  └-┘                  \n"
      + "qb1 = f  ───┤I├─┤X├─      ← 入力量子ビット\n"
      + "               └-┘  └-┘                         \n"
      + "全量子状態  qbf0   qbf1   qbf2     ");
    
    DJ._print("・Xゲートを生成");
    QuGate xQuGate = new QuGate(QuGate.X_GATE);
    DJ.printF("xQuGate", xQuGate);
    //  xQuGate:F
    //  ┌            ┐ ┌     ┐
    //  │ 0+0i   1+0i│=│ 0  1│
    //  │ 1+0i   0+0i│ │ 1  0│
    //  └            ┘ └     ┘
    
    
    DJ.printS(1, "PhaseKickback(4-1)", this); // Pause Task --------------
    DJ._print("Code(4-1)Xゲートの固有値と固有ベクトル");
    DJ.print("・Xゲートの固有値はv0=+1とv1=-1で、固有ベクトルは\n"
           + "  f0=(1/√(2))[1 +1]'とf1=(1/√(2))[1 -1]'です。");
    
    DJ._print("・固有ベクトルf0とf1を生成します。");
    f0QB = hQuGate.apply(e0QB); // +(1/√(2))[1 +1]'
    DJ.printF("f0QB", f0QB); // 固有ベクトルf0
    //  f0QB:F 
    //  ┌           ┐          ┌ ┐ ┌ ┐
    //  │ 0.70711+0i│= 1/√(2)(│1│+│0│)
    //  │ 0.70711+0i│          │0│ │1│
    //  └           ┘          └ ┘ └ ┘
    
    f1QB = hQuGate.apply(e1QB); // +(1/√(2))[1 -1]'
    DJ.printF("f1QB", f1QB); // 固有ベクトルf1
    //  f1QB:F 
    //  ┌           ┐          ┌ ┐ ┌ ┐
    //  │ 0.70711+0i│= 1/√(2)(│1│-│0│)
    //  │-0.70711+0i│          │0│ │1│
    //  └           ┘          └ ┘ └ ┘
    
    
    DJ.printS(1, "PhaseKickback(4-2)", this); // Pause Task --------------
    DJ._print("Code(4-2)固有値と固有ベクトルを確認");
    DJ.print("・Xゲートに固有ベクトルf0を適用します。\n"
           + "  X f0 = X (1/√(2))[1 +1]'=  (1/√(2))[1 +1]'= v0 f0");
    QuBit x_f0QB = xQuGate.apply(f0QB);
    DJ.printF("x_f0QB", x_f0QB);
    //  x_f0QB:F 
    //  ┌           ┐             ┌ ┐ ┌ ┐
    //  │ 0.70711+0i│= +1(1/√(2)(│1│+│0│))
    //  │ 0.70711+0i│             │0│ │1│
    //  └           ┘             └ ┘ └ ┘
    DJ.print("・固有値v0と固有ベクトルf0の関係を満たしています。");

    DJ._print("・Xゲートに固有ベクトルf1を適用します。\n"
            + "  X f1 = X (1/√(2))[1 -1]'= -(1/√(2))[1 -1]'= v1 f1");
    QuBit x_f1QB = xQuGate.apply(f1QB);
    DJ.printF("x_f1QB", x_f1QB);
    //  x_f1QB:F 
    //  ┌           ┐             ┌ ┐ ┌ ┐
    //  │-0.70711+0i│= -1(1/√(2)(│1│-│0│))
    //  │ 0.70711+0i│             │0│ │1│
    //  └           ┘             └ ┘ └ ┘
    DJ.print("・固有値v1と固有ベクトルf1の関係を満たしています。");


    DJ.printS(1, "PhaseKickback(4-3)", this); // Pause Task --------------
    DJ.print("Code(4-3)コントロールNotゲートを生成");
    targetGate = new QuGate(); // ターゲット量子ゲート
    targetGate.setGateMat(xQuGate.copyGateMat());
    numOfBits = 2; // 量子ビット数
    target = 1; // ターゲット量子ビットのインデックス
    controlArray = new int[1]; // コントロール量子ビット配列
    controlArray[0] = 0; // コントロール量子ビットのインデックス
    DJ.print("targetGate = X-Gate" + ", numOfBits = " + numOfBits
          + ", target = " + target + ", controlArray", controlArray);
    DJ.print("QuGate cxQuGate = new QuGate("
          + "targetGate, numOfBits, target, controlArray);");
    QuGate cxQuGate = new QuGate(targetGate, numOfBits, target, controlArray);
    DJ.printF("cxQuGate", cxQuGate); // C-Xゲート
    // cxGate:F
    // ┌                          ┐ ┌           ┐ ┌      ┐
    // │ 1+0i   0+0i   0+0i   0+0i│=│ 1  0  0  0│=│ I  0 │
    // │ 0+0i   1+0i   0+0i   0+0i│ │ 0  1  0  0│ │ 0  X │
    // │ 0+0i   0+0i   0+0i   1+0i│ │ 0  0  0  1│ └      ┘
    // │ 0+0i   0+0i   1+0i   0+0i│ │ 0  0  1  0│ 
    // └                          ┘ └           ┘ 
    
    
    DJ.printS(1, "PhaseKickback(4-4)", this); // Pause Task --------------
    DJ.print("Code(4-4)HIゲートとC-Xゲートを合成");
     
    QuGate cXhIGate = cxQuGate.innerProduct(hIQuGate);
    DJ.printF("cXhIGate", cXhIGate);
    //  cXhIGate:F
    //  ┌                                                  ┐
    //  │ 0.70711+0i   0+0i         0.70711+0i   0+0i      │
    //  │ 0+0i         0.70711+0i   0+0i         0.70711+0i│
    //  │ 0+0i         0.70711+0i   0+0i        -0.70711+0i│
    //  │ 0.70711+0i   0+0i        -0.70711+0i   0+0i      │
    //  └                                                  ┘
    //           ┌           ┐ ┌      ┐
    // = (1/√(2)│ 1  0  1  0│=│ I  I │
    //           │ 0  1  0  1│ │ X -X │
    //           │ 0  1  0 -1│ └      ┘
    //           │ 1  0 -1  0│ 
    //           └           ┘
    
    
    
    DJ.printS(1, "PhaseKickback(4-5)①", this); // Pause Task --------------
    DJ._print("Code(4-5)①入力量子ビットに固有ベクトルf0を使用");
    DJ.print("・補助量子ビットqb0に入力量子ビットf0を相互作用させ、\n"
           + "  ２量子ビットの量子ビット状態qbf0を生成します。");
    qbf0 = qb0.interactQuBit(f0QB); // [1 0]' ⊗ (1/√(2))[1 +1]'
    DJ.printF("qbf0", qbf0);
    // qbf0:F 
    // ┌           ┐
    // │ 0.70711+0i│= qb0 ⊗ f = [1 0]' ⊗ (1/√(2))[1 +1]'
    // │ 0.70711+0i│
    // │ 0+0i      │
    // │ 0+0i      │
    // └           ┘
    
    DJ.print("・量子ビット状態qbf0を合成ゲートcXhIGateに適用します。");
    qbf2 = cXhIGate.apply(qbf0);
    DJ.printF("qbf2", qbf2);
    // qbf2:F 
    //  ┌       ┐      ┌ ┐ ┌ ┐   ┌ ┐ ┌ ┐
    //  │ 0.5+0i│= 1/2(│1│+│0│)⊗(│1│+│0│)
    //  │ 0.5+0i│      │0│ │1│   │0│ │1│
    //  │ 0.5+0i│      └ ┘ └ ┘   └ ┘ └ ┘
    //  │ 0.5+0i│
    //  └       ┘
    DJ.print("・入力量子ビットが固有ベクトルf0=[1 0]'の場合は\n"
           + "  補助量子ビットに固有値v0=+1が"
           + "位相キックバックされます。");


    DJ.printS(1, "PhaseKickback(4-5)②", this); // Pause Task --------------
    DJ._print("Code(4-5)②入力量子ビットに固有ベクトルf1を使用");
    DJ.print("  補助量子ビットqb0に入力量子ビットf1を相互作用させ、\n"
           + "　２量子ビットの量子ビット状態qbf0を生成します。");
    qbf0 = qb0.interactQuBit(f1QB); // [1 0]' ⊗ (1/√(2))[1 -1]'
    DJ.printF("qbf0", qbf0);
    // qbf0:F 
    // ┌           ┐
    // │ 0.70711+0i│= qb0 ⊗ f1 = [1 0]' ⊗ (1/√(2))[1 -1]'
    // │-0.70711+0i│
    // │ 0+0i      │
    // │-0+0i      │
    // └           ┘
    
    DJ.print("・量子ビット状態qbf0を合成ゲートcXhIGateに適用します。");
    qbf2 = cXhIGate.apply(qbf0);
    DJ.printF("qbf2", qbf2);
    // qbf2:F 
    //  ┌       ┐      ┌ ┐ ┌ ┐   ┌ ┐ ┌ ┐
    //  │ 0.5+0i│= 1/2(│1│-│0│)⊗(│1│-│0│)
    //  │-0.5+0i│      │0│ │1│   │0│ │1│
    //  │-0.5+0i│      └ ┘ └ ┘   └ ┘ └ ┘
    //  │ 0.5+0i│
    //  └       ┘
    DJ.print("・入力量子ビットが固有ベクトルf1=[0 1]'の場合は\n"
           + "  補助量子ビットに固有値v1=-1が"
           + "位相キックバックされます。");


    
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(2, "PhaseKickback(5)", this); // Pause Task --------------
    DJ.print("Code(5)ユニタリ行列にSゲートを使用");
    
    DJ._print("・位相キックバックの量子回路\n"
      + "               ┌-┐  ┌-┐  ┌-┐                  \n"
      + "qb0 = [1 0]' ─┤H├─┤C├─┤M├ ← 補助量子ビット\n"
      + "               ├-┤  ├|┤  └-┘                  \n"
      + "qb1 = f  ───┤I├─┤S├─      ← 入力量子ビット\n"
      + "               └-┘  └-┘                         \n"
      + "全量子状態  qbf0   qbf1   qbf2     ");
    
    DJ._print("Code(5-1)Sゲートを生成");
    DJ.print("・SゲートはR1ゲート(Rkゲートのk=1の場合)で"
           + "生成します。");
    k = 1;
    DJ.print_("k = ", k);
    QuGate sQuGate = new QuGate(QuGate.RK_GATE, k);
    DJ.printF(", sQuGate=r1Gate", sQuGate);
    // k = 1, sQuGate=r1Gate:F
    // ┌            ┐ ┌                   ┐ ┌    ┐
    // │ 1+0i   0+0i│=│1  0               │=│1  0│
    // │ 0+0i   0+1i│ │0 exp(i2π/2^(k+1))│ │0  i│
    // └            ┘ └                   ┘ └    ┘
    // オイラーの公式 exp(iθ) = cos(θ) + isin(θ) より、
    // exp(i2π/2^(k+1)) = exp(i2π/2^2) = cos(π/2) + isin(π/2) = i
    
    DJ.print("・Sゲート（R1ゲート）の固有値は v0=+1 と v1=+i で、\n"
           + "　固有ベクトルは f0=[1 0]' と f1=[0 1]'です。");
    

    DJ.printS(1, "PhaseKickback(5-2)", this); // Pause Task --------------
    DJ._print("Code(5-2)固有値と固有ベクトルを確認");
    DJ.print("・Sゲートに固有ベクトルf0を適用します。\n"
           + "  S f0 = S [1 0]'= +1 [1 0]'= v0 f0");
    f0QB = new QuBit(e0QB); // [1 0]'
    QuBit s_f0QB = sQuGate.apply(f0QB);
    DJ.printF("s_f0QB", s_f0QB);
    //  s_f0QB:F 
    //  ┌     ┐    ┌ ┐
    //  │ 1+0i│= +1│0│= v0 f0
    //  │ 0+0i│    │1│
    //  └     ┘    └ ┘
    DJ.print("・固有値v0と固有ベクトルf0の関係を満たしています。");

    DJ._print("・Sゲートに固有ベクトルf1を適用します。\n"
            + "  S f1 = S [0 1]'= +i [0 1]'= v1 f1");
    f1QB = new QuBit(e1QB); // [0 1]'
    QuBit s_f1QB = sQuGate.apply(f1QB);
    DJ.printF("s_f1QB", s_f1QB);
    //  s_f1QB:F 
    //  ┌     ┐    ┌ ┐
    //  │ 0+0i│= +i│0│= vi f1
    //  │ 0+1i│    │1│
    //  └     ┘    └ ┘
    DJ.print("・固有値v1と固有ベクトルf1の関係を満たしています。");

    
    DJ.printS(1, "PhaseKickback(5-3)", this); // Pause Task --------------
    DJ.print("Code(5-3)コントロールSゲートC-Sを生成");
    DJ._print("・ユニタリ行列UとしてSゲートを使います。");
    targetGate = new QuGate(); // ターゲット量子ゲート
    targetGate.setGateMat(sQuGate.copyGateMat());
    numOfBits = 2; // 量子ビット数
    target = 1; // ターゲット量子ビットのインデックス
    controlArray = new int[1]; // コントロール量子ビット配列
    controlArray[0] = 0; // コントロール量子ビットのインデックス
    DJ.print("targetGate = S_Gate" + ", numOfBits = " + numOfBits
          + ", target = " + target + ", controlArray", controlArray);
    DJ.print("QuGate csGate = new QuGate("
          + "targetGate, numOfBits, target, controlArray);");
    QuGate csQuGate = new QuGate(targetGate, numOfBits, target, controlArray);
    DJ.printF("csQuGate", csQuGate); // C-Sゲート
    // csQuGate:F
    // ┌                          ┐ ┌               ┐ ┌    ┐
    // │ 1+0i   0+0i   0+0i   0+0i│=│ 1   0   0   0 │=│I  0│  
    // │ 0+0i   1+0i   0+0i   0+0i│ │ 0   1   0   0 │ │0  S│
    // │ 0+0i   0+0i   1+0i   0+0i│ │ 0   0   1   0 │ └    ┘
    // │ 0+0i   0+0i   0+0i   0+1i│ │ 0   0   0   i │ 
    // └                          ┘ └               ┘ 
    
    DJ.printS(1, "PhaseKickback(5-4)", this); // Pause Task --------------
    DJ.print("Code(5-4)HIゲートとC-Sゲートを合成");
     
    QuGate cShIGate = csQuGate.innerProduct(hIQuGate);
    DJ.printF("cShIGate", cShIGate);
    //  cShIGate:F
    //  ┌                                                  ┐
    //  │ 0.70711+0i   0+0i         0.70711+0i   0+0i      │
    //  │ 0+0i         0.70711+0i   0+0i         0.70711+0i│
    //  │ 0.70711+0i   0+0i        -0.70711+0i   0+0i      │
    //  │ 0+0i         0-0.70711i   0+0i        -0+0.70711i│
    //  └                                                  ┘
    //           ┌           ┐ ┌        ┐
    // = (1/√(2)│ 1  0  1  0│=│ I   I  │
    //           │ 0  1  0  1│ │ SZ -SZ │
    //           │ 1  0 -1  0│ └        ┘
    //           │ 0 -i  0 +i│ 
    //           └           ┘
    
    DJ.printS(1, "PhaseKickback(5-5)①", this); // Pause Task --------------
    DJ._print("Code(5-5)①入力量子ビットに固有ベクトルf0を使用");
    DJ.print("・補助量子ビットqb0に入力量子ビットf0を相互作用させ、\n"
           + "  ２量子ビットの量子ビット状態qbf0を生成します。");
    qbf0 = qb0.interactQuBit(f0QB); // [1 0]' ⊗ [1 0]'
    DJ.printF("qbf0", qbf0);
    // qbf0:F 
    // ┌     ┐
    // │ 1+0i│= qb0 ⊗ f = [1 0]' ⊗ [1 0]'
    // │ 0+0i│
    // │ 0+0i│
    // │ 0+0i│
    // └     ┘
    
    DJ.print("・量子ビット状態qbf0を合成ゲートcShIGateに適用します。");
    qbf2 = cShIGate.apply(qbf0);
    DJ.printF("qbf2", qbf2);
    // qbf2:F 
    //  ┌           ┐           ┌ ┐  ┌ ┐    ┌ ┐
    //  │ 0.70711+0i│=(1/√(2))(│1│+1│0│) ⊗ │1│
    //  │ 0+0i      │           │0│  │1│    │0│
    //  │ 0.70711+0i│           └ ┘  └ ┘    └ ┘
    //  │ 0+0i      │
    //  └           ┘
    DJ.print("・入力量子ビットが固有ベクトルf0=[1 0]'の場合は\n"
           + "  補助量子ビットに固有値v0=+1が"
           + "位相キックバックされます。");

    
    DJ.printS(1, "PhaseKickback(5-5)②", this); // Pause Task --------------
    DJ._print("Code(5-5)②入力量子ビットに固有ベクトルf1を使用");
    DJ.print("・補助量子ビットqb0に入力量子ビットf1を相互作用させ、\n"
           + "  ２量子ビットの量子ビット状態qbf0を生成します。");
    qbf0 = qb0.interactQuBit(f1QB); // [1 0]' ⊗ [0 1]'
    DJ.printF("qbf0", qbf0);
    // qbf0:F 
    // ┌     ┐
    // │ 0+0i│= qb0 ⊗ f = [1 0]' ⊗ [0 1]'
    // │ 1+0i│
    // │ 0+0i│
    // │ 0+0i│
    // └     ┘
    
    DJ.print("・量子ビット状態qbf0を合成ゲートcShIGateに適用します。");
    qbf2 = cShIGate.apply(qbf0);
    DJ.printF("qbf2", qbf2);
    // qbf2:F 
    //  ┌           ┐           ┌ ┐  ┌ ┐    ┌ ┐
    //  │ 0+0i      │=(1/√(2))(│1│+i│0│) ⊗ │0│
    //  │ 0.70711+0i│           │0│  │1│    │1│
    //  │ 0+0i      │           └ ┘  └ ┘    └ ┘
    //  │ 0-0.70711i│
    //  └           ┘
    DJ.print("・入力量子ビットが固有ベクトルf0=[0 1]'の場合は\n"
           + "  補助量子ビットに固有値v1=+iが"
           + "位相キックバックされます。");

    

    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(2, "PhaseKickback(6)", this); // Pause Task ------------------
    DJ.print("Code(6)非固有ベクトルの場合");
    DJ.print("・入力量子ビットfに非固有ベクトルを使い、\n"
           + "  位相キックバックを調べます。");
    
    DJ._print("Code(6-1)入力量子ビットf0を角度θ=π/6で設定");
    double theta = Math.PI / 6.0; // 30度
    f0QB = QuBit.rotateRadianQuBit(theta); // f0 = [√(3/2)  1/2]'
    DJ.printF("f0QB", f0QB);
    // f0QB:F 
    // ┌           ┐        ┌  ┐      ┌  ┐
    // │ 0.86603+0i│ = √3/2│ 1│＋ 1/2│ 0│
    // │ 0.5+0i    │        │ 0│      │ 1│
    // └           ┘        └  ┘      └  ┘
    
    DJ.print("・補助量子ビットqb0に入力量子ビットf0を\n"
           + "  相互作用させ、２量子ビットqbf0を生成します。");
    qbf0 = qb0.interactQuBit(f0QB); // [1 0]'⊗[√(3/2)  1/2]'
    DJ.printF("qbf0", qbf0);
    // qbf0:F
    // ┌           ┐          ┌  ┐        ┌  ┐
    // │ 0.86603+0i│ = (√3/2)│ 1│＋ (1/2)│ 0│
    // │ 0.5+0i    │          │ 0│        │ 1│
    // │ 0+0i      │          │ 0│        │ 0│
    // │ 0+0i      │          │ 0│        │ 0│
    // └           ┘          └  ┘        └  ┘

    
    DJ.printS(1, "PhaseKickback(6-2)", this); // Pause Task ------------------
    DJ.print("Code(6-2)量子ビット状態qbf0を合成ゲートcXhIGateに適用");
    qbf2 = cXhIGate.apply(qbf0);
    DJ.printF("qbf2", qbf2);
    // qbf2:F 
    // ┌           ┐             ┌ ┐ ┌ ┐      ┌ ┐ ┌ ┐
    // │ 0.61237+0i│= 1/√2(√3/2│1│⊗│1│ + 1/2│1│⊗│0│
    // │ 0.35355+0i│             │0│ │0│      │0│ │1│
    // │ 0.35355+0i│             └ ┘ └ ┘      └ ┘ └ ┘
    // │ 0.61237+0i│             ┌ ┐ ┌ ┐      ┌ ┐ ┌ ┐
    // └           ┘         +1/2│0│⊗│1│+√3/2│0│⊗│0│)
    //                             │1│ │0│      │1│ │1│
    //                             └ ┘ └ ┘      └ ┘ └ ┘
    
    // ┌           ┐         ┌ ┐        ┌ ┐      ┌ ┐
    // │ 0.61237+0i│= (1/√2)│1│⊗ (√3/2│1│ + 1/2│0│)
    // │ 0.35355+0i│         │0│        │0│      │1│
    // │ 0.35355+0i│         └ ┘        └ ┘      └ ┘
    // │ 0.61237+0i│         ┌ ┐      ┌ ┐        ┌ ┐
    // └           ┘ +(1/√2)│0│⊗ (1/2│1│ + √3/2│0│)
    //                         │1│      │0│        │1│
    //                         └ ┘      └ ┘        └ ┘
    
    DJ.print("・入力量子ビットが非固有ベクトルf0=[√(3/2)  1/2]'の\n"
           + "  場合は、補助量子ビットに(1/√2)が"
           + "位相キックバックされます。");
    
   
    // this.stepMode = true; // タスクのステップ実行モード
    DJ._print("##### PhaseKickbackの処理を終了します　#####");
  } // phaseKickback()

} // PhaseKickback Class

// End of file
