/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2025
 */
package task.qu;

import quan.QuBit;
import java.util.Random;
import quan.QuBasis;
import task.Task;
import util.comp.Complex;
import util.DJ;
import util.TimeStamp;
import view.PatternViewer;

/**
 * <p> 表　題: Class: QuantumComm</p>
 * <p> 説　明: 量子通信</p>
 * <p>         ・BB84プロトコル（量子鍵配布）
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2025</p>
 * <p> 作成日: 2025.01.13</p>
 */
public class QuantumComm extends Task {
  
  // 学習パラメータ
  int dataNum = 1000; // エポック毎の学習データ数

  /**
   * Taskクラスで新しく作られたスレッドから呼び出される
   */
  @Override
  public void runTask() {
    DJ._print("・タスク開始日時：", TimeStamp.getTimeFormated());
    beginTime = System.currentTimeMillis(); // タスク開始時刻
    
    // パターン・ビューワ・ランチャを得る
    patternViewerFlag = true; //true; // false:非表示
    patternData0 = new double[5][dataNum]; // ID
    patternData1 = new double[5][dataNum]; // ID
    patternViewerLauncher = DJ.pattern(  // 判定パターン、データ、タイトル
        PatternViewer.PATTERN_TUNER, patternData0, "QuantumComm",
        PatternViewer.PATTERN_TUNER, patternData1, "QuantumComm");
    
    this.stepMode = true; // タスクのステップ実行モード
    
    quantumComm(); // タスク本体の呼び出し
  }
  
  /** 
   * 量子通信
   * BB84プロトコル（量子鍵配布）
   */
  public void quantumComm() {
    DJ._print("QuantumComm.quantumComm() ========================");
    DJ.print("■量子計算の開始");

    DJ._print("(1)暗号通信：ＢＢ８４プロトコル");
    DJ.print("(2)織姫と彦星の量子通信:盗聴がない場合");
    DJ.print("(3)カササギ（喜鵲）が盗聴した場合");
    
    
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(3, "QuantumComm(1)", this); // Pause Task ------------------------
    DJ.print("(1)暗号通信：ＢＢ８４プロトコル");
    DJ.print("・こと座の織姫Vegaは天の川の対岸のわし座の彦星Altairに、\n"
           + "　2進数の文字列を量子ビットで送信します。しかし、\n"
           + "　天帝Jade Emperorははくちょう座のカササギDenebに命じて\n"
           + "　二人の通信を盗聴させます。（※著者のフィクションです。）");
    DJ._print("・織姫と彦星は２組の基底を用意します。\n"
           + "  織姫はランダムに基底を選択し、量子ビット送信します。\n"
           + "　彦星もランダムに基底を選択し、量子ビット受信します。\n"
           + "　両者は基底が一致している場合の量子ビットを抽出して、\n"
           + "　カササギ（喜鵲）による盗聴の有無を調べます。");
    
    DJ._print("Code(1-1)①暗号鍵の長さと疑似乱数の初期値");
    int maxRand = 2; // 乱数の上限
    int keyLength = 50 * 4; // 5000 * 4; // 必要な乱数の個数 = 暗号鍵の長さ * 4
    DJ.print("keyLength = ", keyLength); // keyLength = 200
    long vBinaryRandomSeed = 5; // 織姫の疑似乱数の初期値：２進数用
    long vBasisRandomSeed = 8;  // 織姫の疑似乱数の初期値：基底選択用
    long aBasisRandomSeed = 11; // 彦星の疑似乱数の初期値：基底選択用
    long dBasisRandomSeed = 17; // 喜鵲の疑似乱数の初期値：基底選択用
    
    
    DJ.printS(1, "QuantumComm(1-1)②", this); // Pause Task --------------------
    DJ.print("Code(1-1)②織姫Vegaが送信する２進数（乱数）");
    Random randForBinary = new Random(vBinaryRandomSeed);
    int[] vBinaryRandArray = new int[keyLength]; // 織姫が送信する２進数配列
    int sum = 0; // ２進数の合計
    for (int i = 0; i < keyLength; i++) {
      vBinaryRandArray[i] = randForBinary.nextInt(maxRand);
      sum = sum + vBinaryRandArray[i];
    }
    DJ.print("vBinaryRandArray", vBinaryRandArray);
    // binaryRandArray[1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1...]
    DJ.print("sum = ", sum); // sum = 96
    
    DJ.printS(1, "QuantumComm(1-1)③", this); // Pause Task --------------------
    DJ.print("Code(1-1)③織姫Vegaの基底選択用乱数（２進数）");
    Random vRandForBasis = new Random(vBasisRandomSeed);
    int[] vBasisRandArray = new int[keyLength]; // 基底選択用乱数配列（２進数）
    sum = 0; // ２進数の合計
    for (int i = 0; i < keyLength; i++) {
      vBasisRandArray[i] = vRandForBasis.nextInt(maxRand);
      sum = sum + vBasisRandArray[i];
    }
    DJ.print("vBasisRandArray", vBasisRandArray);
    // vBasisRandArray[1, 1, 1, 0, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 1, 0, 0, 1, 1...]
    DJ.print("sum = ", sum); // sum = 98
    
    DJ.print("Code(1-1)④彦星（Altair）の基底選択用乱数（２進数）");
    Random aRandForBasis = new Random(aBasisRandomSeed);
    int[] aBasisRandArray = new int[keyLength]; // 基底選択用乱数配列（２進数）
    sum = 0; // ２進数の合計
    for (int i = 0; i < keyLength; i++) {
      aBasisRandArray[i] = aRandForBasis.nextInt(maxRand);
      sum = sum + aBasisRandArray[i];
    }
    DJ.print("aBasisRandArray", aBasisRandArray);
    // aBasisRandArray[1, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 0...]
    DJ.print("sum = ", sum); // sum = 87
    
    DJ.print("Code(1-1)⑤カササギ（喜鵲）Denebの基底選択用乱数（２進数）");
    Random dRandForBasis = new Random(dBasisRandomSeed);
    int[] dBasisRandArray = new int[keyLength]; // 基底選択用乱数配列（２進数）
    sum = 0; // ２進数の合計
    for (int i = 0; i < keyLength; i++) {
      dBasisRandArray[i] = dRandForBasis.nextInt(maxRand);
      sum = sum + dBasisRandArray[i];
    }
    /**/ DJ.print("dBasisRandArray", dBasisRandArray);
    // dBasisRandArray[1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 1, 0, 1...]
    DJ.print("sum = ", sum); // sum = 115
    
    
    DJ.printS(2, "QuantumComm(1-2)", this); // Pause Task ----------------------
    DJ._print("Code(1-2)織姫と彦星が使う基底");
    DJ.print("①標準基底（１量子ビット）");
    int numOfQuBit = 1; // 量子ビット数
    QuBasis standardBasis = new QuBasis(numOfQuBit);
    DJ.print("standardBasis", standardBasis);
    // standardBasis
    // ┌                    ┐
    // │ 1.0+0.0i│ 0.0+0.0i│
    // │ 0.0+0.0i│ 1.0+0.0i│
    // └                    ┘
    
    DJ.print("②４５度（π/4）回転した基底");
    double pi = Math.PI;
    double theta = pi / 4.0; // 45 degree
    QuBasis rotate45Basis = new QuBasis(theta);
    DJ.printF("rotate45Basis", rotate45Basis);
    // rotate45Basis:F
    // ┌                        ┐
    // │ 0.70711+0i│-0.70711+0i│
    // │ 0.70711+0i│ 0.70711+0i│
    // └                        ┘


    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(3, "QuantumComm(2)", this); // Pause Task -----------------
    DJ.print("Code(2)盗聴がない場合の織姫と彦星の通信");
    DJ._print("・織姫がランダムに基底を選択して量子ビットを測定し、送信します。\n"
            + "　彦星も量子ビットを受信し、ランダムに基底を選択し測定します。");
    
    DJ.print_("Code(2-1)送信する二進数の個数：");
    DJ.print("size = ", keyLength);
    
    int vBinaryId; // 織姫が送信する古典ビットのインデックス
    int vBasisId; // 織姫の基底選択用乱数のインデックス
    QuBit vQuBit; // 織姫が送信する量子ビット
    
    int aBasisId; // 彦星の基底選択用乱数のインデックス
    QuBit aQuBit; // 彦星が受信した量子ビット
    int[] aBinaryArray = new int[keyLength]; // 彦星が受信した古典ビット配列（２進数）

    
    DJ._print("Code(2-2)－－－－－  通信開始  －－－－－");
    for (int i = 0; i < keyLength; i++) { // 全試行回数

      // 織姫の送信
      vBinaryId = vBinaryRandArray[i]; // 織姫が送信する古典ビット配列（２進数）
      // DJ.print("vBinaryId[" + i + "] = ", vBinaryId);
      vBasisId = vBasisRandArray[i]; // 織姫の基底選択用乱数
      // DJ.print("vBasisId[" + i + "] = ", vBasisId);
      if (vBasisId == QuBasis.BASIS_0) { // 標準基底を選択
        // 古典ビットに対応する標準基底の量子ビット
        vQuBit = standardBasis.getBasisQuBit(vBinaryId);
      }
      else { // ４５度回転した基底を選択
        // 古典ビットに対応する４５度回転した基底の量子ビット
        vQuBit = rotate45Basis.getBasisQuBit(vBinaryId);
      }
      // 選択した基底で織姫が送信した量子ビット
      // DJ.printF("vQuBit[" + i + "] = ", vQuBit);
      
      // 彦星の受信
      aBasisId = aBasisRandArray[i]; // 彦星の基底選択用乱数
      // DJ.print("aBasisId[" + i + "] = ", aBasisId);
      if (aBasisId == QuBasis.BASIS_0) { // 標準基底を選択
        // 受信した量子ビットを標準基底で測定し、遷移した量子ビット
        aQuBit = standardBasis.measurement(vQuBit);
      }
      else { // ４５度回転した基底を選択
        // 受信した量子ビットを４５度回転した基底で測定し、遷移した量子ビット
        aQuBit = rotate45Basis.measurement(vQuBit);
      }
      // 彦星が受信した量子ビット
      // DJ.printF("aQuBit[" + i + "]", aQuBit);
           
      // 受信した量子ビットより古典ビットを得る
      Complex probAmp = aQuBit.getProbAmp(QuBit.PA0); // 第０確率振幅
      if (aBasisId == QuBasis.BASIS_0) { // 標準基底を選択
        if (probAmp.getReal() > 0.5) aBinaryArray[i] = 0;
        else                         aBinaryArray[i] = 1;
      }
      else { // ４５度回転した基底を選択
        if (probAmp.getReal() > 0.0) aBinaryArray[i] = 0;
        else                         aBinaryArray[i] = 1;
      }
      // 彦星が受信した古典ビット列（２進数）
      // DJ.print("aBinaryArray[" + i + "] = ", aBinaryArray[i]);
      
      // DJ.printS(1, "QuantumComm(2-2)", this); // Pause Task -----------------
    }
    
    
    DJ.printS(2, "QuantumComm(2-3)", this); // Pause Task -----------------
    DJ._print("Code(2-3)－－－－－  通信完了  －－－－－");
    
    DJ._print("・通信結果");
    DJ.print("　織姫が送信する古典ビット配列、２進数");
    /**/ DJ.print("vBinaryRandArray", vBinaryRandArray);
    DJ.print("　彦星が受信した古典ビット配列、２進数");
    /**/ DJ.print("aBinaryArray", aBinaryArray);
    
    DJ.print("・単純に織姫と彦星の送受信値が一致した回数");
    int accord = 0; // 織姫と彦星の古典ビット値が一致した回数
    for (int i = 0; i < keyLength; i++) {
      if (vBinaryRandArray[i] ==  aBinaryArray[i]) accord++;
    }
    DJ.print_("accord = ", accord); 
    DJ.print("回、　全試行回数 = ", keyLength);
    // accord = 161回、　全試行回数 = 200
    // accord = 15018回、　全試行回数 = 20000

    // 15018/20000 = 0.7509≒3/4=1/2+1/4, 
    // 1/2は偶然基底が一致、更に残りの半分の半分は確率的に半分は一致
    
    DJ.printS(1, "QuantumComm(2-4)", this); // Pause Task ----------------------
    DJ._print("Code(2-4)基底と送受信値が一致した試行を抽出する");
    int sameBasis = 0; // 基底が一致した回数
    int correctCase = 0; // 送受信値が一致した回数
    for (int i = 0; i < keyLength; i++) {
      // 織姫の基底選択用乱数と彦星の基底選択用乱数
      if (vBasisRandArray[i] == aBasisRandArray[i]) { // 基底が一致
        sameBasis++;
         // 織姫が送信する古典ビットと彦星が受信した古典ビット
        if (vBinaryRandArray[i] == aBinaryArray[i]) { // 送受信値が一致
          correctCase++;
        }
      }
    }
    
    //vBasisRandArray[i]; // 織姫の基底選択用乱数
    //aBasisRandArray[i]; // 彦星の基底選択用乱数
    DJ.print("・試行" + keyLength + "回の内、基底が一致した回数");
    DJ.print("sameBasis = ", sameBasis); // 基底が一致した回数
    // sameBasis = 107
    // sameBasis = 10027
    
    // vBinaryRandArray = new int[keyLength]; // 織姫が送信した２進数値
    // int[] aBinaryArray = new int[keyLength]; // 彦星が受信した２進数値
    DJ.print("・更にそのうち、送受信値が一致した回数 \n"
           + "　（基底が同じならば送受信値も一致するので、\n"
           + "　基底が一致した回数と等しい）。");
    DJ.print("correctCase = ", correctCase); // 送受信値が一致した回数
    // correctCase = 107
    // correctCase = 10027
    
    
    
    this.stepMode = true; // タスクのステップ実行モード
    DJ.printS(3, "QuantumComm(3)", this); // Pause Task -----------------
    DJ.print("(3)カササギが盗聴する場合");
    
    DJ._print("Code(3-1)織姫がランダムな基底で量子ビットを送信し、\n"
            + "　カササギ（喜鵲）がランダムな基底でそれを盗聴します。\n"
            + "　カササギは盗聴で遷移した量子ビットを彦星に転送します。\n"
            + "　さらに彦星もランダムな基底で量子ビットを受信します。");
    
    int dBasisId; // カササギ（喜鵲）の基底選択用乱数のインデックス
    QuBit dQuBit; // 喜鵲が受信した量子ビット（喜鵲が転送する量子ビット）
    int[] dBinaryArray = new int[keyLength]; // 喜鵲が受信した古典ビット配列（２進数）
    
    DJ._print("Code(3-2)－－－－－  通信開始  －－－－－");
    for (int i = 0; i < keyLength; i++) { // 全試行回数
      
      // Code(3-2)①織姫の送信
      vBinaryId = vBinaryRandArray[i]; // 織姫が送信する古典ビット配列（２進数）
      // DJ._print("vBinaryId[" + i + "] = ", vBinaryId);
      vBasisId = vBasisRandArray[i]; // 織姫の基底選択用乱数
      // DJ.print("vBasisId[" + i + "] = ", vBasisId);
      if (vBasisId == QuBasis.BASIS_0) { // 標準基底を選択
        // 古典ビットに対応する標準基底の量子ビット
        vQuBit = standardBasis.getBasisQuBit(vBinaryId);
      }
      else { // ４５度回転した基底を選択
        // 古典ビットに対応する４５度回転した基底の量子ビット
        vQuBit = rotate45Basis.getBasisQuBit(vBinaryId);
      }
      // 選択した基底で織姫が送信した量子ビット
      // DJ.printF_("vQuBit[" + i + "] = ", vQuBit);
    
      // Code(3-2)②カササギ（喜鵲）の受信と転信
      dBasisId = dBasisRandArray[i]; // カササギ（喜鵲）の基底選択用乱数
      // DJ.print("dBasisId[" + i + "] = ", dBasisId);
      if (dBasisId == QuBasis.BASIS_0) // 標準基底を選択
        // 受信した量子ビットを標準基底で測定し、遷移した量子ビット
        dQuBit = standardBasis.measurement(vQuBit);
      else // ４５度回転した基底を選択
        // 受信した量子ビットを４５度回転した基底で測定し、遷移した量子ビット
        dQuBit = rotate45Basis.measurement(vQuBit);
      // カササギ（喜鵲）が受信した量子ビット
      // DJ.printF_("dQuBit[" + i + "]", dQuBit);
      
      // 受信した量子ビットより古典ビットを得る
      Complex probAmp = dQuBit.getProbAmp(QuBit.PA0); // 第０確率振幅
      if (dBasisId == QuBasis.BASIS_0) { // 標準基底を選択
        if (probAmp.getReal() > 0.5) dBinaryArray[i] = 0;
        else                         dBinaryArray[i] = 1;
      }
      else { // ４５度回転した基底を選択
        if (probAmp.getReal() > 0.0) dBinaryArray[i] = 0;
        else                         dBinaryArray[i] = 1;
      }
      // カササギ（喜鵲）が受信した古典ビット列（２進数）
      // DJ.print("dBinaryArray[" + i + "] = ", dBinaryArray[i]);
      
      // Code(3-2)③彦星の受信
      aBasisId = aBasisRandArray[i]; // 彦星の基底選択用乱数
      // DJ.print("aBasisId[" + i + "] = ", aBasisId);
      if (aBasisId == QuBasis.BASIS_0) // 標準基底を選択
        // 受信した量子ビットを標準基底で測定し、遷移した量子ビット
        aQuBit = standardBasis.measurement(dQuBit);
      else // ４５度回転した基底を選択
        // 受信した量子ビットを４５度回転した基底で測定し、遷移した量子ビット
        aQuBit = rotate45Basis.measurement(dQuBit);
      // 彦星が受信した量子ビット
      // DJ.printF_("aQuBit[" + i + "]", aQuBit);
      
      // 受信した量子ビットより古典ビットを得る
      probAmp = aQuBit.getProbAmp(QuBit.PA0); // 第０確率振幅
      if (aBasisId == QuBasis.BASIS_0) { // 標準基底を選択
        if (probAmp.getReal() > 0.5) aBinaryArray[i] = 0;
        else                         aBinaryArray[i] = 1;
      }
      else { // ４５度回転した基底を選択
        if (probAmp.getReal() > 0.0) aBinaryArray[i] = 0;
        else                         aBinaryArray[i] = 1;
      }
      // 彦星が受信した古典ビット列（２進数）
      // DJ.print("aBinaryArray[" + i + "] = ", aBinaryArray[i]);
      
      // DJ.printS(1, "QuantumComm(3-2)", this); // Pause Task -----------------
    }
      
      
    DJ.printS(2, "QuantumComm(3-3)", this); // Pause Task -----------------
    DJ._print("Code(3-3)－－－－－  通信完了  －－－－－");
    
    DJ._print("・通信結果");
    DJ.print("　織姫が送信した２進数列vBinaryRandArray");
    DJ.print("vBinaryRandArray", vBinaryRandArray); // 古典ビット配列
    DJ.print("　カササギ（喜鵲）が受信した２進数列dBinaryArray");
    DJ.print("dBinaryArray", dBinaryArray); // 古典ビット配列
    DJ.print("　彦星が受信した２進数列aBinaryArray");
    DJ.print("aBinaryArray", aBinaryArray); // 古典ビット配列
      
    DJ._print("・単純に織姫と彦星の送受信値が一致した回数");
    accord = 0; // 織姫と彦星の古典ビット値が一致した回数
    for (int i = 0; i < keyLength; i++) {
      if (vBinaryRandArray[i] ==  aBinaryArray[i]) accord++;
    }
    DJ.print_("accord = ", accord); 
    DJ.print("回、  全試行回数 = ", keyLength);
    // accord = 122回、　全試行回数 = 200
    // accord = 12495回、　全試行回数 = 20000
     
    // vBasisRandArray[i]; // 織姫の基底選択用乱数
    // aBasisRandArray[i]; // 彦星の基底選択用乱数
    
    
    DJ.printS(1, "QuantumComm(3-4)", this); // Pause Task ---------------------
    DJ._print("Code(3-4)織姫と彦星の基底が一致した回数と、\n"
      + "  基底および送受信値が両方とも一致した回数を数えます。");
    sameBasis = 0; // 基底が一致した回数
    correctCase = 0; // 送受信値が一致した回数
    for (int i = 0; i < keyLength; i++) {
      // 織姫と彦星の基底選択用乱数
      if (vBasisRandArray[i] == aBasisRandArray[i]) { // 基底が一致
        sameBasis++;
         // 織姫が送信する古典ビットと彦星が受信した古典ビット
        if (vBinaryRandArray[i] == aBinaryArray[i]) { // 送受信値が一致
          correctCase++;
        }
      }
    }
    

    DJ._print("・試行" + keyLength + "回の内、基底が一致した回数");
    DJ.print("sameBasis = ", sameBasis); // 基底が一致した回数
    // sameBasis = 107, 10027
    
    // vBinaryRandArray = new int[keyLength]; // 織姫が送信した２進数値
    // int[] aBinaryArray = new int[keyLength]; // 彦星が受信した２進数値
    DJ.print("　更にそのうち、送受信値が一致した回数");
    DJ.print("correctCase = ", correctCase); // 送受信値が一致した回数
    // correctCase = 77, 7488
    
    DJ._print( "・基底が同じならば送受信値も一致するので、\n"
             + "　基底が一致した回数と等しくなるはずですが、\n"
             + "　盗聴により 1/2*1/2*1/2 =1/8 は誤るので、\n"
             + "　受信値が一致する割合は 1/2-1/8=3/8=0.375 になります。");
    double colectRate = (double)correctCase / (double)keyLength;
    DJ.print("一致した割合 = " + correctCase + " / " + keyLength + " = ",
        colectRate);
    // 一致した割合 = 77 / 200 = 0.385
    // 一致した割合 = 7488 / 20000 = 0.3744
    
    DJ._print("・送受信値が一致した回数が約1/2→約3/8に減少したことから、\n"
            + "　盗聴が行われたと推定されます。");


    
    DJ.printS(2, "QuantumComm(Ending)", this); // Pause Task ---------------------
    DJ._print("量子計算の完了");
    DJ.print("QuantumComm.quantumComm() ===========================");
  } // quantumComm()
    
} // QuantumComm Class

// End of file
