/*
 *  Title: DaiJa_V5 ( Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package util;

import java.text.DecimalFormat;
import view.Console;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Random;
import quan.QuBasis;
import quan.QuBit;
import quan.QuGate;
import task.Task;
import util.comp.CompMat;
import util.comp.CompVec;
import util.comp.Complex;
import view.GraphViewerLauncher;
import view.PatternViewerLauncher;
import view.PropertyViewer;
import view.SomViewerLauncher;

/**
 * <p> 表　題: Class: DJ</p>
 * <p> 説　明: DaiJa汎用関数</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020, 2021</p>
 * <p> 作成日: 2019.1.22</p>
 */ 
public class DJ {
  
//  private static Console console;
  private static Console console;
  private static Random random;
  
  // 逆正規化のために保存
  private static double average = 0.0; // 全要素の平均
  private static double stdDev = 1.0; // 全要素の標準偏差
  
  private static final String DEFAULT_FILE_NAME = "DaiJa_text"; // ディフォルト・ファイル名
  

  /**
   * DaiJa汎用関数クラスDJ生成子
   * @param aConsole Console // コンソール
   */
  public DJ(Console aConsole) {
    console = aConsole;
  }
  
  public static boolean checkDJ() {
    return console != null;
  }
  
  // 関数 ---------------------------------------------------------------------
  public static Random getRandom() {
    if (random == null) {
      random = new Random((long)314159265); // 乱数クラス
    }
    return random;
  }
  
  /**
   * 要素の順序をシャッフルしたランダムな順番のリストを得る
   * @param num int // リストの要素数
   * @return list ArrayList＜Integer＞ // 順序を変えたリスト
   */
  public static ArrayList<Integer> permutationRandom(int num) {
      ArrayList<Integer> list = new ArrayList<>();
      for(int i = 0 ; i < num ; i++) {
          list.add(i); // listに値を入れる。この段階では昇順
      }
      Collections.shuffle(list); // シャッフルして、順序を変える
      return list; // シャッフルした結果
  }
  
  // シグモイド関数
  public static double sigmoid(double x) {
    double y = 1.0 / (1.0 + Math.exp(-x));
    return y;
  }
  
  // Hyperbolic Tangent
  public static double tanh(double x) {
    double y = Math.tanh(x);
    return y;
  }
  
  public static double reLU(double x) {
    double y; 
    if (x <= 0) y = 0.0;
    else y = x;
    return y;
  }
  
  public static double leakyReLU(double x) {
    double y; 
    if (x <= 0) y = 0.01 * x;
    else y = x;
    return y;
  }
  
  /**
   * 論理変数値bを整数値iに変換する
   * if b=false then i=0
   * if b=true  then i=1
   * @param b boolean // 論理変数値
   * @return i int // 整数値
   */
  public static int boolToInt(boolean b) {
    int i;
    if (b) i = 1; else i = 0;
    return i;
  }
 
  // ---------------------------------------------------------------------------
  /**
   * ２次の配列の全要素の平均を得る
   * @param original double[][] // ２次の配列
   * @return average double // 全要素の平均値
   */
  public static double average(double[][] original) {
    double sum = 0.0;  
    int row = original.length;
    int col = original[0].length;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        sum = sum + original[i][j];
      }
    }
      
    int elementNumber = row * col;
    // ゼロ割を回避
    if (sum != 0.0) {
      if (elementNumber <= 0) {
        LogEditor.print("***** ERROR ***** DJ.avarage(): ");
        LogEditor.print("Element number of array <= 0.");
        return Double.NaN;
      }
    }
      
    double average_ = sum / (double)elementNumber;
    return average_;
  } // average()
   
  /**
   * ２次の配列の全要素の標準偏差を得る
   * @param original double[][] // ２次の配列
   * @return stdDev double // 全要素の標準偏差
   */
  public static double stdDev(double[][] original) {
    double average_ = average(original); // 全要素の平均

    double sum = 0.0;  
    int row = original.length;
    int col = original[0].length;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        double diff = original[i][j] - average_;
        sum = sum + diff * diff;
      }
    }
      
    int elementNumber = row * col;
    // ゼロ割を回避
    if (sum != 0.0) {
      if (elementNumber <= 0) {
        LogEditor.print("***** ERROR ***** DJr.stdDev(): ");
        LogEditor.print("Element number of array <= 0.");
        return Double.NaN;
      }
    }
      
    double stdDev_ = Math.sqrt(sum / elementNumber);
    return stdDev_;
  } // stdDev()
  
  /**
   * ２次の配列の全要素を正規化
   * 計算式：normalizedTensor = normalize(original)
   * 呼出し：normalizedTensor = DJ.normalize(original)
   * @param original double[][] // 正規化された２次の配列
   * @return normalizedTensor DoubleTensor // 全要素で正規化されたダブル・テンソル
   */
  public static double[][] normalize(double[][] original) {

    average = average(original); // 全要素の平均
    stdDev = stdDev(original); // 全要素の標準偏差
    if (stdDev <= Double.MIN_VALUE) { // ゼロ割を回避
      LogEditor.print("***** ERROR ***** DJ.normalize(): ");
      LogEditor.print("Standard Deviation is neary Zero.");
      return null;
    }
    
    int row = original.length;
    int col = original[0].length;
    double[][] normalizedArray = new double[row][col];
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        normalizedArray[i][j] = (original[i][j] - average) / stdDev;
      } // col
    } // row
    return normalizedArray;
  } // normalize()
  
  
  /**
   * １次の配列を２次の配列に変換
   * 変形例 配列(64, 1) -> (8, 8)に変形
   * @param original double[] // １次の配列
   * @param resultRow int // ２次の配列の行数
   * @param resultCol int // ２次の配列の列数
   * @return result double[][] // ２次の配列
   */
  public static double[][] reshape(
          double[] original, int resultRow, int resultCol) {
    double[][] result = new double[resultRow][resultCol];
    int k = 0;
    for (int i = 0; i < resultRow; i++) {
      for (int j = 0; j < resultCol; j++) {
//        int k = i * resultCol + j;
        result[i][j] =original[k];
        k = k + 1;
      }
    }
    return result;
  }

  /**
   * 差分二乗誤差の総和
   * @param y double[] // 出力層の出力
   * @param teach double[] // 正解
   * @return sum double // 差の二乗誤差の総和
   */
  public static double getSquareError(double[] y, double[] teach) {
    double sum = 0.0;
    int nodeNum = y.length;
    for (int k = 0; k < nodeNum; k++) {
      double deff = (y[k] - teach[k]);
      sum = sum + deff * deff;
    }
    return sum;
  }
  
  /**
   * 交差エントロピー誤差の総和
   * @param y DoubleTensor // 出力層の出力
   * @param teach DoubleTensor // 正解
   * @return sum double // 交差エントロピー誤差の総和
   */
  public static double getEntropyError(double[] y, double[] teach) {
    double sum = 0.0;
    int nodeNum = y.length;
    for (int k = 0; k < nodeNum; k++) {
      sum = sum - teach[k] * Math.log(y[k] + 1.0E-7);
    }
    return sum;
  }
  
  // 改行 ----------------------------------------------------------------------
  
  /**
   * 改行：空白行を１行挿入
   */
  public static void printL() {
    LogEditor.print(""); // プリント後の改行有り
  }
    
  /**
   * 改行：空白行を挿入
   * @param num int // 挿入する行数
   */
  public static void printL(int num) {
    StringBuilder sb = new StringBuilder();
    
    for (int i = 0; i < num - 1; i++) {
      sb.append("\n");
    }
        
    LogEditor.print(sb.toString()); // プリント後の改行有り
  }
  
  
  // タスクのステップ実行 ------------------------------------------------------
  
  /**
   * タスクのステップ実行と空白行の挿入
   * @param num int // 挿入する行数
   * @param comment String // 
   * @param task Task // 一時停止するタスク
   */
  public static void printS(int num, String comment, Task task) {
    if (task.stepMode) { // true:ステップ送り実行モード、false:連続実行モード
      LogEditor.print("\n### 一時停止中 ### " + comment + "（Run/Pauseボタンで実行再開）");
      task.stepMode();
    }

    printL(num); // 改行：空白行を挿入
  }
  
  
  
  
  // 数値プリント ==============================================================
  
  /**
   * 整数値をプリント
   * @param val int // 整数値
   */
  public static void print_(int val) {
    LogEditor.print_(Integer.toString(val)); // プリント後の改行無し
  }
  public static void print(int val) {
    LogEditor.print(Integer.toString(val)); // プリント後の改行有り
  }
  public static void _print_(int val) {
    LogEditor._print_(Integer.toString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(int val) {
    LogEditor._print(Integer.toString(val));// プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * 単精度実数値をプリント
   * @param val float // 単精度実数値
   */
  public static void print_(float val) {
    LogEditor.print_(Float.toString(val)); // プリント後の改行無し
  }
  public static void print(float val) {
    LogEditor.print(Float.toString(val)); // プリント後の改行有り
  }
  public static void _print_(float val) {
    LogEditor._print_(Float.toString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(float val) {
    LogEditor._print(Float.toString(val));// プリント前の改行有り、プリント後の改行有り
  }

  /**
   * 倍精度実数値をプリント
   * @param val float // 倍精度実数値
   */
  public static void print_(double val) {
    LogEditor.print_(Double.toString(val)); // プリント後の改行無し
  }
  public static void print(double val) {
    LogEditor.print(Double.toString(val)); // プリント後の改行有り
  }
  public static void _print_(double val) {
    LogEditor._print_(Double.toString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(double val) {
    LogEditor._print(Double.toString(val));// プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * 倍精度実数値をプリント（書式付き）
   * @param val Complex // 複素数値
   */
  public static void printF_(double val) {
    LogEditor.print_(toFormat(val)); // プリント後の改行無し
  }
  public static void printF(double val) {
    LogEditor.print(toFormat( val)); // プリント後の改行有り
  }
  public static void _printF_(double val) {
    LogEditor._print_(toFormat(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _printF(double val) {
    LogEditor._print(toFormat(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * 複素数値をプリント
   * @param val Complex // 複素数値
   */
  public static void print_(Complex val) {
    LogEditor.print_(val.toString()); // プリント後の改行無し
  }
  public static void print(Complex val) {
    LogEditor.print(val.toString()); // プリント後の改行有り
  }
  public static void _print_(Complex val) {
    LogEditor._print_(val.toString()); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(Complex val) {
    LogEditor._print(val.toString());// プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージをプリント
   * @param message String // メッセージ
   */
  public static void print_(String message) {
    LogEditor.print_(message); // プリント後の改行無し
  }
  public static void print(String message) {
    LogEditor.print(message); // プリント後の改行有り
  }
  public static void _print_(String message) {
    LogEditor._print_(message); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message) {
    LogEditor._print(message); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージと文字列をプリント
   * @param message String // メッセージ
   * @param val String // 文字列
   */
   public static void print_(String message, String val) {
    LogEditor.print_(message + val); // プリント後の改行無し
  }
  public static void print(String message, String val) {
    LogEditor.print(message + val); // プリント後の改行有り
  }
   public static void _print_(String message, String val) {
    LogEditor._print_(message + val); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, String val) {
    LogEditor._print(message + val); // プリント前の改行有り、プリント後の改行有り
  }
   
  /**
   * メッセージと整数値をプリント
   * @param message String // メッセージ
   * @param val int // 整数値
   */
  public static void print_(String message, int val) {
    LogEditor.print_(message + Integer.toString(val)); // プリント後の改行無し
  }
 public static void print(String message, int val) {
    LogEditor.print(message + Integer.toString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, int val) {
    LogEditor._print_(message + Integer.toString(val)); // プリント前の改行有り、プリント後の改行無し
  }
 public static void _print(String message, int val) {
    LogEditor._print(message + Integer.toString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージと単精度実数値をプリント
   * @param message String // メッセージ
   * @param val float // 単精度実数値
   */
  public static void print_(String message, float val) {
    LogEditor.print_(message + Float.toString(val)); // プリント後の改行無し
  }
  public static void print(String message, float val) {
    LogEditor.print(message + Float.toString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, float val) {
    LogEditor._print_(message + Float.toString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, float val) {
    LogEditor._print(message + Float.toString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージと倍精度実数値をプリント
   * @param message String // メッセージ
   * @param val double // 倍精度実数値
   */
  public static void print_(String message, double val) {
    LogEditor.print_(message + Double.toString(val)); // プリント後の改行無し
  }
  public static void print(String message, double val) {
    LogEditor.print(message + Double.toString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, double val) {
    LogEditor._print_(message + Double.toString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, double val) {
    LogEditor._print(message + Double.toString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージと倍精度実数値をプリント（書式付き）
   * @param message String // メッセージ
   * @param val Complex // 複素数値
   */
  public static void printF_(String message, double val) {
    LogEditor.print_(toFormat(message, val)); // プリント後の改行無し
  }
  public static void printF(String message, double val) {
    LogEditor.print(toFormat(message, val)); // プリント後の改行有り
  }
  public static void _printF_(String message, double val) {
    LogEditor._print_(toFormat(message, val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _printF(String message, double val) {
    LogEditor._print(toFormat(message, val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * 倍精度実数値（ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @param val double // 倍精度実数値
   * @return sb String // 複素数の文字列
   */
  public static String toFormat(String label, double val) {
    StringBuilder sb = new StringBuilder(label + ":F ");
    sb.append(toFormat(val));
    return sb.toString();
  }
  
  /**
   * 倍精度実数値の文字列を得る（書式付き）
   * @param val double // 倍精度実数値
   * @return String // 複素数の文字列
   */
  public static String toFormat(double val) {
    DecimalFormat fmt = new DecimalFormat("##.#####");
    // fmt.setRoundingMode(RoundingMode.FLOOR); // 丸め
    // fmt.setRoundingMode(RoundingMode.DOWN); // 切り捨て
    
    String string = fmt.format(val);
    if (string.contains("-")) {
      return string;
    } else {
      return " " + string;
    }
  }
  
  /**
   * メッセージと複素数をプリント
   * @param message String // メッセージ
   * @param val Complex // 複素数値
   */
  public static void print_(String message, Complex val) {
    LogEditor.print_(val.toString(message)); // プリント後の改行無し
  }
  public static void print(String message, Complex val) {
    LogEditor.print(val.toString(message)); // プリント後の改行有り
  }
  public static void _print_(String message, Complex val) {
    LogEditor._print_(val.toString(message)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, Complex val) {
    LogEditor._print(val.toString(message)); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージと複素数をプリント（書式付き）
   * @param message String // メッセージ
   * @param val Complex // 複素数値
   */
  public static void printF_(String message, Complex val) {
    LogEditor.print_(val.toFormat(message)); // プリント後の改行無し
  }
  public static void printF(String message, Complex val) {
    LogEditor.print(val.toFormat(message)); // プリント後の改行有り
  }
  public static void _printF_(String message, Complex val) {
    LogEditor._print_(val.toFormat(message)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _printF(String message, Complex val) {
    LogEditor._print(val.toFormat(message)); // プリント前の改行有り、プリント後の改行有り
  }
  
  //  メッセージと配列のプリント --------------------ーー-----------------------
  /**
   * メッセージと整数配列をプリント
   * @param message String // メッセージ
   * @param val int[] // 整数配列
   */
  public static void print_(String message, int[] val) {
    LogEditor.print_(message + arrayToString(val)); // プリント後の改行無し
  }
 public static void print(String message, int[] val) {
    LogEditor.print(message + arrayToString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, int[] val) {
    LogEditor._print_(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行無し
  }
 public static void _print(String message, int[] val) {
    LogEditor._print(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  public static String arrayToString(int[] array) {
    StringBuilder sb = new StringBuilder();
    // sb.append("=\n");
    sb.append("[").append(array[0]);
    for (int i = 1; i < array.length; i++) {
      sb.append(", ").append(array[i]);
    }
    sb.append("]");
    return sb.toString();
  }
  
  
  // 配列プリント --------------------------------------------------------------
  
  /**
   * メッセージと２次の整数配列をプリント
   * @param message String // メッセージ
   * @param val int[][] // ２次の整数配列
   */
  public static void print_(String message, int[][] val) {
    LogEditor.print_(message + arrayToString(val)); // プリント後の改行無し
  }
 public static void print(String message, int[][] val) {
    LogEditor.print(message + arrayToString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, int[][] val) {
    LogEditor._print_(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行無し
  }
 public static void _print(String message, int[][] val) {
    LogEditor._print(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  public static String arrayToString(int[][] array) {
    StringBuilder sb = new StringBuilder();
    sb.append("=\n");
    sb.append("[[").append(array[0][0]);
    for (int j = 1; j < array[0].length; j++) {
      sb.append(", ").append(array[0][j]);
    }
    sb.append("]\n");
    for (int i = 1; i < array.length; i++) {
      sb.append("[").append(array[i][0]);
      for (int j = 1; j < array[i].length; j++) {
        sb.append(", ").append(array[i][j]);
      }
      sb.append("]\n");
    }
    sb.append("]");
    return sb.toString();
  }
  
  /**
   * メッセージと単精度配列をプリント
   * @param message String // メッセージ
   * @param val int[] // 単精度配列
   */
  public static void print_(String message, float[] val) {
    LogEditor.print_(message + arrayToString(val)); // プリント後の改行無し
  }
 public static void print(String message, float[] val) {
    LogEditor.print(message + arrayToString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, float[] val) {
    LogEditor._print_(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行無し
  }
 public static void _print(String message, float[] val) {
    LogEditor._print(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  public static String arrayToString(float[] array) {
    StringBuilder sb = new StringBuilder();
    sb.append("=\n");
    sb.append("[").append(array[0]);
    for (int i = 1; i < array.length; i++) {
      sb.append(", ").append(array[i]);
    }
    sb.append("]");
    return sb.toString();
  }
  
  /**
   * メッセージと２次の単精度配列をプリント
   * @param message String // メッセージ
   * @param val float[][] // ２次の単精度実数配列
   */
  public static void print_(String message, float[][] val) {
    LogEditor.print_(message + arrayToString(val)); // プリント後の改行無し
  }
  public static void print(String message, float[][] val) {
    LogEditor.print(message + arrayToString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, float[][] val) {
    LogEditor._print_(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, float[][] val) {
    LogEditor._print(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  public static String arrayToString(float[][] array) {
    StringBuilder sb = new StringBuilder();
    sb.append("=\n");
    sb.append("[[").append(array[0][0]);
    for (int j = 1; j < array[0].length; j++) {
      sb.append(", ").append(array[0][j]);
    }
    sb.append("]\n");
    for (int i = 1; i < array.length; i++) {
      sb.append("[").append(array[i][0]);
      for (int j = 1; j < array[i].length; j++) {
        sb.append(", ").append(array[i][j]);
      }
      sb.append("]\n");
    }
    sb.append("]");
    return sb.toString();
  }
  
  /**
   * メッセージと倍精度実数配列をプリント
   * @param message String // メッセージ
   * @param val double[] // 倍精度実数配列
   */
  public static void print_(String message, double[] val) {
    LogEditor.print_(message + arrayToString(val)); // プリント後の改行無し
  }
  public static void print__(String message, double[] val) {
    LogEditor.print_(message + arrayToString_(val)); // プリント後の改行無し
  }
  public static void print(String message, double[] val) {
    LogEditor.print(message + arrayToString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, double[] val) {
    LogEditor._print_(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, double[] val) {
    LogEditor._print(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  // "="の後の改行無し
  public static String arrayToString_(double[] array) {
    StringBuilder sb = new StringBuilder();
    sb.append("=");
    sb.append("[").append(array[0]);
    for (int i = 1; i < array.length; i++) {
      sb.append(", ").append(array[i]);
    }
//    sb.append("]\n");
    return sb.toString();
  }
  
  // "="の後で改行
  public static String arrayToString(double[] array) {
    StringBuilder sb = new StringBuilder();
    sb.append("=\n");
    sb.append("[").append(array[0]);
    for (int i = 1; i < array.length; i++) {
      sb.append(", ").append(array[i]);
    }
    sb.append("]");
    return sb.toString();
  }
  
  /**
   * メッセージと倍精度実数配列をプリント（書式付き）
   * @param message String // メッセージ
   * @param val double[] // 倍精度実数配列
   */
  public static void printF_(String message, double[] val) {
    LogEditor.print_(message + arrayToFormat(val)); // プリント後の改行無し
  }
  public static void printF__(String message, double[] val) {
    LogEditor.print_(message + arrayToFormat_(val)); // プリント後の改行無し
  }
  public static void printF(String message, double[] val) {
    LogEditor.print(message + arrayToFormat(val)); // プリント後の改行有り
  }
  public static void _printF_(String message, double[] val) {
    LogEditor._print_(message + arrayToFormat(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _printF(String message, double[] val) {
    LogEditor._print(message + arrayToFormat(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  // "="の後の改行無し
  public static String arrayToFormat_(double[] array) {
    DecimalFormat fmt = new DecimalFormat("##.#####"); ///
    StringBuilder sb = new StringBuilder();
    sb.append("=");
    sb.append("[").append(fmt.format(array[0])); ///
    for (int i = 1; i < array.length; i++) {
      sb.append(", ").append(fmt.format(array[i])); ///
    }
//    sb.append("]\n");
    return sb.toString();
  }
  
  // "="の後で改行
  public static String arrayToFormat(double[] array) {
    DecimalFormat fmt = new DecimalFormat("##.#####"); ///
    StringBuilder sb = new StringBuilder();
    sb.append("=\n");
    sb.append("[").append(fmt.format(array[0])); ///
    for (int i = 1; i < array.length; i++) {
      sb.append(", ").append(fmt.format(array[i])); ///
    }
    sb.append("]");
    return sb.toString();
  }
  
  /**
   * メッセージと２次の倍精度実数配列をプリント
   * @param message String // メッセージ
   * @param val double[][ // ２次の倍精度実数配列
   */
  public static void print_(String message, double[][] val) {
    LogEditor.print_(message + arrayToString(val)); // プリント後の改行無し
  }
  public static void print(String message, double[][] val) {
    LogEditor.print(message + arrayToString(val)); // プリント後の改行有り
  }
  public static void _print_(String message, double[][] val) {
    LogEditor._print_(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, double[][] val) {
    LogEditor._print(message + arrayToString(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  public static String arrayToString(double[][] array) {
    StringBuilder sb = new StringBuilder();
    sb.append("=\n");
    sb.append("[[").append(array[0][0]); ///
    for (int j = 1; j < array[0].length; j++) {
      sb.append(", ").append(array[0][j]); ///
    }
    sb.append("]\n");
    for (int i = 1; i < array.length; i++) {
      sb.append("[").append(array[i][0]); ///
      for (int j = 1; j < array[i].length; j++) {
        sb.append(", ").append(array[i][j]); ///
      }
      sb.append("]\n");
    }
    sb.append("]");
    return sb.toString();
  }
  
  /**
   * メッセージと２次の倍精度実数配列をプリント（書式付き）
   * @param message String // メッセージ
   * @param val double[][ // ２次の倍精度実数配列
   */
  public static void printF_(String message, double[][] val) {
    LogEditor.print_(message + arrayToFormat(val)); // プリント後の改行無し
  }
  public static void printF(String message, double[][] val) {
    LogEditor.print(message + arrayToFormat(val)); // プリント後の改行有り
  }
  public static void _printF_(String message, double[][] val) {
    LogEditor._print_(message + arrayToFormat(val)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _printF(String message, double[][] val) {
    LogEditor._print(message + arrayToFormat(val)); // プリント前の改行有り、プリント後の改行有り
  }
  
  public static String arrayToFormat(double[][] array) {
    DecimalFormat fmt = new DecimalFormat("##.#####"); ///
    StringBuilder sb = new StringBuilder();
    sb.append("=\n");
    sb.append("[[").append(fmt.format(array[0][0])); ///
    for (int j = 1; j < array[0].length; j++) {
      sb.append(", ").append(fmt.format(array[0][j])); ///
    }
    sb.append("]\n");
    for (int i = 1; i < array.length; i++) {
      sb.append("[").append(fmt.format(array[i][0])); ///
      for (int j = 1; j < array[i].length; j++) {
        sb.append(", ").append(fmt.format(array[i][j])); ///
      }
      sb.append("]\n");
    }
    sb.append("]");
    return sb.toString();
  }
  
  
  
  // 複素数配列(Complex[])の文字列をプリント -----------------------------------
  
  /**
   * 複素数配列をプリント
   * @param val Complex[] // 複素数配列
   */
  public static void print_(Complex[] val) {
    LogEditor.print_(Complex.toString(val)); } // プリント後の改行無し
  public static void print(Complex[] val) {
    LogEditor.print(Complex.toString(val)); } // プリント後の改行有り
  public static void _print_(Complex[] val) {
    LogEditor._print_(Complex.toString(val)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(Complex[] val) {
    LogEditor._print(Complex.toString(val)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 複素数配列をプリント行・列選択表示）
   * @param val Complex[] // 複素数配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   */
  public static void print_(Complex[] val, boolean orient) {
    LogEditor.print_(Complex.toString(val, orient)); } // プリント後の改行無し
  public static void print(Complex[] val, boolean orient) {
    LogEditor.print(Complex.toString(val, orient)); } // プリント後の改行有り
  public static void _print_(Complex[] val, boolean orient) {
    LogEditor._print_(Complex.toString(val, orient)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(Complex[] val, boolean orient) {
    LogEditor._print(Complex.toString(val, orient)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 複素数配列をプリント（書式付き）
   * @param val Complex[] // 複素数配列
   */
  public static void printF_(Complex[] val) {
    LogEditor.print_(Complex.toFormat(val)); } // プリント後の改行無し（書式付き）
  public static void printF(Complex[] val) {
    LogEditor.print(Complex.toFormat(val)); } // プリント後の改行有り（書式付き）
  public static void _printF_(Complex[] val) {
    LogEditor._print_(Complex.toFormat(val)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(Complex[] val) {
    LogEditor._print(Complex.toFormat(val)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * 複素数配列をプリント（行・列選択表示、書式付き）
   * @param val Complex[] // 複素数配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   */
  public static void printF_(Complex[] val, boolean orient) {
    LogEditor.print_(Complex.toFormat(val, orient)); } // プリント後の改行無し（書式付き）
  public static void printF(Complex[] val, boolean orient) {
    LogEditor.print(Complex.toFormat(val, orient)); } // プリント後の改行有り（書式付き）
  public static void _printF_(Complex[] val, boolean orient) {
    LogEditor._print_(Complex.toFormat(val, orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(Complex[] val, boolean orient) {
    LogEditor._print(Complex.toFormat(val, orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * メッセージと複素数配列をプリント
   * @param message String // メッセージ
   * @param val Complex[] // 複素数配列
   */
  public static void print_(String message, Complex[] val) {
    LogEditor.print_(Complex.toString(message, val)); } // プリント後の改行無し
  public static void print(String message, Complex[] val) {
    LogEditor.print(Complex.toString(message, val)); } // プリント後の改行有り
  public static void _print_(String message, Complex[] val) {
    LogEditor._print_(Complex.toString(message, val)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, Complex[] val) {
    LogEditor._print(Complex.toString(message, val)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * メッセージと複素数配列をプリント（行・列選択表示、メッセージ付き）
   * @param message String // メッセージ
   * @param val Complex[] // 複素数配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   */
  public static void print_(String message, Complex[] val, boolean orient) {
    LogEditor.print_(Complex.toString(message, val, orient)); } // プリント後の改行無し
  public static void print(String message, Complex[] val, boolean orient) {
    LogEditor.print(Complex.toString(message, val, orient)); } // プリント後の改行有り
  public static void _print_(String message, Complex[] val, boolean orient) {
    LogEditor._print_(Complex.toString(message, val, orient)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, Complex[] val, boolean orient) {
    LogEditor._print(Complex.toString(message, val, orient)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * メッセージと複素数配列をプリント（書式付き）
   * @param message String // メッセージ
   * @param val Complex[] // 複素数配列
   */
  public static void printF_(String message, Complex[] val) {
    LogEditor.print_(Complex.toFormat(message, val)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, Complex[] val) {
    LogEditor.print(Complex.toFormat(message, val)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, Complex[] val) {
    LogEditor._print_(Complex.toFormat(message, val)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, Complex[] val) {
    LogEditor._print(Complex.toFormat(message, val)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * メッセージと複素数配列をプリント（行・列選択表示、メッセージ付き、書式付き）
   * @param message String // メッセージ
   * @param val Complex[] // 複素数配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   */
  public static void printF_(String message, Complex[] val, boolean orient) {
    LogEditor.print_(Complex.toFormat(message, val, orient)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, Complex[] val, boolean orient) {
    LogEditor.print(Complex.toFormat(message, val, orient)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, Complex[] val, boolean orient) {
    LogEditor._print_(Complex.toFormat(message, val, orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, Complex[] val, boolean orient) {
    LogEditor._print(Complex.toFormat(message, val, orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  
  // 複素数行列の文字列をプリント ----------------------------------------------
  
  /**
   * 複素数行列をプリント
   * @param val Complex[][] // 複素数行列
   */
  public static void print_(Complex[][] val) {
    LogEditor.print_(Complex.toString(val)); } // プリント後の改行無し
  public static void print(Complex[][] val) {
    LogEditor.print(Complex.toString(val)); } // プリント後の改行有り
  public static void _print_(Complex[][] val) {
    LogEditor._print_(Complex.toString(val)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(Complex[][] val) {
    LogEditor._print(Complex.toString(val)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 複素数行列をプリント（書式付き）
   * @param val Complex[][] // 複素数行列
   */
  public static void printF_(Complex[][] val) {
    LogEditor.print_(Complex.toFormat(val)); } // プリント後の改行無し
  public static void printF(Complex[][] val) {
    LogEditor.print(Complex.toFormat(val)); } // プリント後の改行有り
  public static void _printF_(Complex[][] val) {
    LogEditor._print_(Complex.toFormat(val)); } // プリント前の改行有り、プリント後の改行無し
  public static void _printF(Complex[][] val) {
    LogEditor._print(Complex.toFormat(val)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * メッセージと複素数行列をプリント
   * @param message String // メッセージ
   * @param val Complex[][] // 複素数行列
   */
  public static void print_(String message, Complex[][] val) {
    LogEditor.print_(Complex.toString(message, val)); } // プリント後の改行無し
  public static void print(String message, Complex[][] val) {
    LogEditor.print(Complex.toString(message, val)); } // プリント後の改行有り
  public static void _print_(String message, Complex[][] val) {
    LogEditor._print_(Complex.toString(message, val)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, Complex[][] val) {
    LogEditor._print(Complex.toString(message, val)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * メッセージと複素数行列をプリント（書式付き）
   * @param message String // メッセージ
   * @param val Complex[][] // 複素数行列
   */
  public static void printF_(String message, Complex[][] val) {
    LogEditor.print_(Complex.toFormat(message, val)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, Complex[][] val) {
    LogEditor.print(Complex.toFormat(message, val)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, Complex[][] val) {
    LogEditor._print_(Complex.toFormat(message, val)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, Complex[][] val) {
    LogEditor._print(Complex.toFormat(message, val)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  
  // 複素数配列の文字列をプリント ----------------------------------------------
  
  /**
   * 複素数配列をプリント
   * @param val CompVec // 複素数配列
   */
  public static void print_(CompVec val) {
    LogEditor.print_(val.toString()); } // プリント後の改行無し
  public static void print(CompVec val) {
    LogEditor.print(val.toString()); } // プリント後の改行有り
  public static void _print_(CompVec val) {
    LogEditor._print_(val.toString()); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(CompVec val) {
    LogEditor._print(val.toString()); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 複素数配列をプリント（行・列選択表示）
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @param val CompVec // 複素数配列
   */
  public static void print_(boolean orient, CompVec val) {
    LogEditor.print_(val.toString(orient)); } // プリント後の改行無し
  public static void print(boolean orient, CompVec val) {
    LogEditor.print(val.toString(orient)); } // プリント後の改行有り
  public static void _print_(boolean orient, CompVec val) {
    LogEditor._print_(val.toString(orient)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(boolean orient, CompVec val) {
    LogEditor._print(val.toString(orient)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 複素数配列をプリント（書式付き）
   * @param val CompVec // 複素数配列
   */
  public static void printF_(CompVec val) {
    LogEditor.print_(val.toFormat()); } // プリント後の改行無し（書式付き）
  public static void printF(CompVec val) {
    LogEditor.print(val.toFormat()); } // プリント後の改行有り（書式付き）
  public static void _printF_(CompVec val) {
    LogEditor._print_(val.toFormat()); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(CompVec val) {
    LogEditor._print(val.toFormat()); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * 複素数配列をプリント（行・列選択表示、書式付き）
   * @param val CompVec // 複素数配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   */
  public static void printF_(CompVec val, boolean orient) {
    LogEditor.print_(val.toFormat(orient)); } // プリント後の改行無し（書式付き）
  public static void printF(CompVec val, boolean orient) {
    LogEditor.print(val.toFormat(orient)); } // プリント後の改行有り（書式付き）
  public static void _printF_(CompVec val, boolean orient) {
    LogEditor._print_(val.toFormat(orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(CompVec val, boolean orient) {
    LogEditor._print(val.toFormat(orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * メッセージと複素数配列をプリント（メッセージ付き）
   * @param message String // メッセージ
   * @param val CompVec // 複素数配列
   */
  public static void print_(String message, CompVec val) {
//    LogEditor.print_(message + val.toString()); // プリント後の改行無し
    LogEditor.print_(val.toString(message)); } // プリント後の改行無し
  public static void print(String message, CompVec val) {
//    LogEditor.print(message + val.toString()); // プリント後の改行有り
    LogEditor.print(val.toString(message)); } // プリント後の改行有り
  public static void _print_(String message, CompVec val) {
//    LogEditor._print_(message + val.toString()); // プリント前の改行有り、プリント後の改行無し
    LogEditor._print_(val.toString(message)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, CompVec val) {
//    LogEditor._print(message + val.toString()); // プリント前の改行有り、プリント後の改行有り
    LogEditor._print(val.toString(message)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * メッセージと複素数配列をプリント（行・列選択表示、メッセージ付き）
   * @param message String // メッセージ
   * @param val CompVec // 複素数配列
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   */
  public static void print_(String message, CompVec val, boolean orient) {
//    LogEditor.print_(message + val.toString(orient)); // プリント後の改行無し
    LogEditor.print_(val.toString(message, orient)); } // プリント後の改行無し
  public static void print(String message, CompVec val, boolean orient) {
//    LogEditor.print(message + val.toString(orient)); // プリント後の改行有り
    LogEditor.print(val.toString(message, orient)); } // プリント後の改行有り
  public static void _print_(String message, CompVec val, boolean orient) {
//    LogEditor._print_(message + val.toString(orient)); // プリント前の改行有り、プリント後の改行無し
    LogEditor._print_(val.toString(message, orient)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, CompVec val, boolean orient) {
//    LogEditor._print(message + val.toString(orient)); // プリント前の改行有り、プリント後の改行有り
    LogEditor._print(val.toString(message, orient)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * メッセージと複素数配列をプリント（メッセージ付き、書式付き）
   * @param message String // メッセージ
   * @param val CompVec // 複素数配列
   */
  public static void printF_(String message, CompVec val) {
//    LogEditor.print_(message + val.toFormat()); // プリント後の改行無し（書式付き）
    LogEditor.print_(val.toFormat(message)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, CompVec val) {
//    LogEditor.print(message + val.toFormat()); // プリント後の改行有り（書式付き）
    LogEditor.print(val.toFormat(message)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, CompVec val) {
//    LogEditor._print_(message + val.toFormat()); // プリント前の改行有り、プリント後の改行無し（書式付き）
    LogEditor._print_(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, CompVec val) {
//    LogEditor._print(message + val.toFormat()); // プリント前の改行有り、プリント後の改行有り（書式付き）
    LogEditor._print(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * メッセージと複素数配列をプリント（行・列選択表示、メッセージ付き、書式付き）
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）
   * @param message String // メッセージ
   * @param val CompVec // 複素数配列
   */
  public static void printF_(String message, CompVec val, boolean orient) {
//    LogEditor.print_(message + val.toFormat(orient)); // プリント後の改行無し（書式付き）
    LogEditor.print_(val.toFormat(message, orient)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, CompVec val, boolean orient) {
//    LogEditor.print(message + val.toFormat(orient)); // プリント後の改行有り（書式付き）
    LogEditor.print(val.toFormat(message, orient)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, CompVec val, boolean orient) {
//    LogEditor._print_(message + val.toFormat(orient)); // プリント前の改行有り、プリント後の改行無し（書式付き）
    LogEditor._print_(val.toFormat(message, orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, CompVec val, boolean orient) {
//    LogEditor._print(message + val.toFormat(orient)); // プリント前の改行有り、プリント後の改行有り（書式付き）
    LogEditor._print(val.toFormat(message, orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  
  // 複素数行列の文字列をプリント ----------------------------------------------
  
  /**
   * 複素数行列をプリント
   * @param val CompMat // 複素数行列
   */
  public static void print_(CompMat val) {
    LogEditor.print_(val.toString()); // プリント後の改行無し
  }
  public static void print(CompMat val) {
    LogEditor.print(val.toString()); // プリント後の改行有り
  }
  public static void _print_(CompMat val) {
    LogEditor._print_(val.toString()); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(CompMat val) {
    LogEditor._print(val.toString()); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * 複素数行列をプリント（書式付き）
   * @param val CompMat // 複素数行列
   */
  public static void printF_(CompMat val) {
    LogEditor.print_(val.toFormat()); // プリント後の改行無し
  }
  public static void printF(CompMat val) {
    LogEditor.print(val.toFormat()); // プリント後の改行有り
  }
  public static void _printF_(CompMat val) {
    LogEditor._print_(val.toFormat()); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _printF(CompMat val) {
    LogEditor._print(val.toFormat()); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージと複素数行列をプリント
   * @param message String // メッセージ
   * @param val CompMat // 複素数行列
   */
  public static void print_(String message, CompMat val) {
    LogEditor.print_(val.toString(message)); // プリント後の改行無し
  }
  public static void print(String message, CompMat val) {
    LogEditor.print(val.toString(message)); // プリント後の改行有り
  }
  public static void _print_(String message, CompMat val) {
    LogEditor._print_(val.toString(message)); // プリント前の改行有り、プリント後の改行無し
  }
  public static void _print(String message, CompMat val) {
    LogEditor._print(val.toString(message)); // プリント前の改行有り、プリント後の改行有り
  }
  
  /**
   * メッセージと複素数行列をプリント（書式付き）
   * @param message String // メッセージ
   * @param val CompMat // 複素数行列
   */
  public static void printF_(String message, CompMat val) {
    LogEditor.print_(val.toFormat(message)); // プリント後の改行無し（書式付き）
  }
  public static void printF(String message, CompMat val) {
    LogEditor.print(val.toFormat(message)); // プリント後の改行有り（書式付き）
  }
  public static void _printF_(String message, CompMat val) {
    LogEditor._print_(val.toFormat(message)); // プリント前の改行有り、プリント後の改行無し（書式付き）
  }
  public static void _printF(String message, CompMat val) {
    LogEditor._print(val.toFormat(message)); // プリント前の改行有り、プリント後の改行有り（書式付き）
  }
  
  // 量子ゲートの文字列をプリント ----------------------------------------------
  
  /**
   * 1.量子ゲートをプリント
   * @param val QuGate // 量子ゲート
   */
  public static void print_(QuGate val) {
    LogEditor.print_(val.toString()); } // プリント後の改行無し
  public static void print(QuGate val) {
    LogEditor.print(val.toString()); } // プリント後の改行有り
  public static void _print_(QuGate val) {
    LogEditor._print_(val.toString()); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(QuGate val) {
    LogEditor._print(val.toString()); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 2.量子ゲートをプリント（書式付き）
   * @param val CompMat // 複素数行列
   */
  public static void printF_(QuGate val) {
    LogEditor.print_(val.toFormat()); } // プリント後の改行無し
  public static void printF(QuGate val) {
    LogEditor.print(val.toFormat()); } // プリント後の改行有り
  public static void _printF_(QuGate val) {
    LogEditor._print_(val.toFormat()); } // プリント前の改行有り、プリント後の改行無し
  public static void _printF(QuGate val) {
    LogEditor._print(val.toFormat()); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 3.メッセージと量子ゲートをプリント
   * @param message String // メッセージ
   * @param val CompMat // 複素数行列
   */
  public static void print_(String message, QuGate val) {
    LogEditor.print_(val.toString(message)); } // プリント後の改行無し
  public static void print(String message, QuGate val) {
    LogEditor.print(val.toString(message)); } // プリント後の改行有り
  public static void _print_(String message, QuGate val) {
    LogEditor._print_(val.toString(message)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, QuGate val) {
    LogEditor._print(val.toString(message)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 4.メッセージと量子ゲートをプリント（書式付き）
   * @param message String // メッセージ
   * @param val CompMat // 複素数行列
   */
  public static void printF_(String message, QuGate val) {
    LogEditor.print_(val.toFormat(message)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, QuGate val) {
    LogEditor.print(val.toFormat(message)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, QuGate val) {
    LogEditor._print_(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, QuGate val) {
    LogEditor._print(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  // 基底の文字列をプリント ----------------------------------------------------
  
  /**
   * 1.基底をプリント
   * @param val QuBasis // 基底
   */
  public static void print_(QuBasis val) {
    LogEditor.print_(val.toString()); } // プリント後の改行無し
  public static void print(QuBasis val) {
    LogEditor.print(val.toString()); } // プリント後の改行有り
  public static void _print_(QuBasis val) {
    LogEditor._print_(val.toString()); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(QuBasis val) {
    LogEditor._print(val.toString()); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 2.基底をプリント（書式付き）
   * @param val QuBasis // 基底
   */
  public static void printF_(QuBasis val) {
    LogEditor.print_(val.toFormat()); } // プリント後の改行無し
  public static void printF(QuBasis val) {
    LogEditor.print(val.toFormat()); } // プリント後の改行有り
  public static void _printF_(QuBasis val) {
    LogEditor._print_(val.toFormat()); } // プリント前の改行有り、プリント後の改行無し
  public static void _printF(QuBasis val) {
    LogEditor._print(val.toFormat()); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 3.メッセージと基底をプリント
   * @param message String // メッセージ
   * @param val QuBasis // 基底
   */
  public static void print_(String message, QuBasis val) {
    LogEditor.print_(val.toString(message)); } // プリント後の改行無し
  public static void print(String message, QuBasis val) {
    LogEditor.print(val.toString(message)); } // プリント後の改行有り
  public static void _print_(String message, QuBasis val) {
    LogEditor._print_(val.toString(message)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, QuBasis val) {
    LogEditor._print(val.toString(message)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 4.メッセージと基底をプリント（書式付き）
   * @param message String // メッセージ
   * @param val QuBasis // 基底
   */
  public static void printF_(String message, QuBasis val) {
    LogEditor.print_(val.toFormat(message)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, QuBasis val) {
    LogEditor.print(val.toFormat(message)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, QuBasis val) {
    LogEditor._print_(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, QuBasis val) {
    LogEditor._print(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  // 量子ビットの文字列をプリント ----------------------------------------------
  
  /**
   * 1.量子ビットをプリント（書式付き）
   * @param val QuBit // 量子ビット
   */
  public static void print_(QuBit val) {
    LogEditor.print_(val.toString()); } // プリント後の改行無し（書式付き）
  public static void print(QuBit val) {
    LogEditor.print(val.toString()); } // プリント後の改行有り（書式付き）
  public static void _print_(QuBit val) {
    LogEditor._print_(val.toString()); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _print(QuBit val) {
    LogEditor._print(val.toString()); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * 1-1.量子ビットをプリント（行・列選択表示、書式付き）
   * @param val QuBit // 量子ビット
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）  
   */
  public static void print_(QuBit val, boolean orient) {
    LogEditor.print_(val.toString(orient)); } // プリント後の改行無し（書式付き）
  public static void print(QuBit val, boolean orient) {
    LogEditor.print(val.toString(orient)); } // プリント後の改行有り（書式付き）
  public static void _print_(QuBit val, boolean orient) {
    LogEditor._print_(val.toString(orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _print(QuBit val, boolean orient) {
    LogEditor._print(val.toString(orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * 2.量子ビットをプリント（書式付き）
   * @param val QuBit // 量子ビット
   */
  public static void printF_(QuBit val) {
    LogEditor.print_(val.toFormat()); } // プリント後の改行無し（書式付き）
  public static void printF(QuBit val) {
    LogEditor.print(val.toFormat()); } // プリント後の改行有り（書式付き）
  public static void _printF_(QuBit val) {
    LogEditor._print_(val.toFormat()); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(QuBit val) {
    LogEditor._print(val.toFormat()); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * 2-1.量子ビットをプリント（行・列選択表示、書式付き）
   * @param val QuBit // 量子ビット
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）  
   */
  public static void printF_(QuBit val, boolean orient) {
    LogEditor.print_(val.toFormat(orient)); } // プリント後の改行無し（書式付き）
  public static void printF(QuBit val, boolean orient) {
    LogEditor.print(val.toFormat(orient)); } // プリント後の改行有り（書式付き）
  public static void _printF_(QuBit val, boolean orient) {
    LogEditor._print_(val.toFormat(orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(QuBit val, boolean orient) {
    LogEditor._print(val.toFormat(orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * 3.量子ビットをプリント（行表示、ラベル付き）
   * @param message String // メッセージ
   * @param val QuBit // 量子ビット
   */
  public static void print_(String message, QuBit val) {
    LogEditor.print_(val.toString(message)); } // プリント後の改行無し
//    LogEditor.print_(message + val.toString()); } // プリント後の改行無し
  public static void print(String message, QuBit val) {
    LogEditor.print(val.toString(message)); } // プリント後の改行有り
//    LogEditor.print(message + val.toString()); } // プリント後の改行有り
  public static void _print_(String message, QuBit val) {
    LogEditor._print_(val.toString(message)); } // プリント前の改行有り、プリント後の改行無し
//    LogEditor._print_(message + val.toString()); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, QuBit val) {
    LogEditor._print(val.toString(message)); } // プリント前の改行有り、プリント後の改行有り
//    LogEditor._print(message + val.toString()); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 3-1.量子ビットをプリント（行・列選択表示、ラベル付き）
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）  
   * @param message String // メッセージ
   * @param val QuBit // 量子ビット
   */
  public static void print_(String message, QuBit val, boolean orient) {
    LogEditor.print_(val.toString(message, orient)); } // プリント後の改行無し
//    LogEditor.print_(message + val.toString(orient)); } // プリント後の改行無し
  public static void print(String message, QuBit val, boolean orient) {
    LogEditor.print(val.toString(message, orient)); } // プリント後の改行有り
//    LogEditor.print(message + val.toString(orient)); } // プリント後の改行有り
  public static void _print_(String message, QuBit val, boolean orient) {
    LogEditor._print_(val.toString(message, orient)); } // プリント前の改行有り、プリント後の改行無し
//    LogEditor._print_(message + val.toString(orient)); } // プリント前の改行有り、プリント後の改行無し
  public static void _print(String message, QuBit val, boolean orient) {
    LogEditor._print(val.toString(message, orient)); } // プリント前の改行有り、プリント後の改行有り
//    LogEditor._print(message + val.toString(orient)); } // プリント前の改行有り、プリント後の改行有り
  
  /**
   * 4.メッセージと量子ビットをプリント（行表示、ラベル付き、書式付き）
   * @param message String // メッセージ
   * @param val QuBit // 量子ビット
   */
  public static void printF_(String message, QuBit val) {
    LogEditor.print_(val.toFormat(message)); } // プリント後の改行無し（書式付き）
//    LogEditor.print_(message + val.toFormat()); } // プリント後の改行無し（書式付き）
  public static void printF(String message, QuBit val) {
    LogEditor.print(val.toFormat(message)); } // プリント後の改行有り（書式付き）
//    LogEditor.print(message + val.toFormat()); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, QuBit val) {
    LogEditor._print_(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
//    LogEditor._print_(message + val.toFormat()); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, QuBit val) {
    LogEditor._print(val.toFormat(message)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
//    LogEditor._print(message + val.toFormat()); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  /**
   * 4-1.メッセージと量子ビットをプリント（行・列選択表示、ラベル付き、書式付き）
   * @param orient boolean // 配列の表示方向 true：列（縦）false：行（横）  
   * @param message String // メッセージ
   * @param val QuBit // 量子ビット
   */
  public static void printF_(String message, QuBit val, boolean orient) {
//    LogEditor.print_(val.toFormat(orient, message)); } // プリント後の改行無し（書式付き）
    LogEditor.print_(val.toFormat(message, orient)); } // プリント後の改行無し（書式付き）
//    LogEditor.print_(message + val.toFormat(orient)); } // プリント後の改行無し（書式付き）
  public static void printF(String message, QuBit val, boolean orient) {
//    LogEditor.print(val.toFormat(orient, message)); } // プリント後の改行有り（書式付き）
    LogEditor.print(val.toFormat(message, orient)); } // プリント後の改行有り（書式付き）
//    LogEditor.print(message + val.toFormat(orient)); } // プリント後の改行有り（書式付き）
  public static void _printF_(String message, QuBit val, boolean orient) {
//    LogEditor._print_(val.toFormat(orient, message)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
    LogEditor._print_(val.toFormat(message, orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
//    LogEditor._print_(message + val.toFormat(orient)); } // プリント前の改行有り、プリント後の改行無し（書式付き）
  public static void _printF(String message, QuBit val, boolean orient) {
//    LogEditor._print(val.toFormat(orient, message)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
    LogEditor._print(val.toFormat(message, orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
//    LogEditor._print(message + val.toFormat(orient)); } // プリント前の改行有り、プリント後の改行有り（書式付き）
  
  
  
  
  // リストをプリント-----------------------------------------------------------
  
  /**
   * メッセージとリストの内容をコンマで区切ってプリントする
   * @param message
   * @param list 
   */
  public static void printList_(String message, ArrayList list) {
    StringBuilder sb = listToStringBuffer(list);
    LogEditor.print_(message + sb.toString());
  }
  public static void printList(String message, ArrayList list) {
    StringBuilder sb = listToStringBuffer(list);
    LogEditor.print(message + sb.toString());
  }
  public static void _printList_(String message, ArrayList list) {
    StringBuilder sb = listToStringBuffer(list);
    LogEditor._print_(message + sb.toString());
  }
  public static void _printList(String message, ArrayList list) {
    StringBuilder sb = listToStringBuffer(list);
    LogEditor._print(message + sb.toString());
  }
  
  public static StringBuilder listToStringBuffer(ArrayList list) {
    StringBuilder sb = new StringBuilder();
    sb.append(list.get(0)); // 先頭にはコンマを付けない
    for (int i = 1; i < list.size(); i++) {
      sb.append(", ").append(list.get(i));
    }
    return sb;
  }
  
  public static String stringArrayToString(String[] stringArray) {
    StringBuilder sb = new StringBuilder();
    sb.append(stringArray[0]); // 先頭にはコンマを付けない
    for (int i = 1; i < stringArray.length; i++) {
      sb.append(", ").append(stringArray[i]);
    }
    return sb.toString();
  }
  
  
  // ビューワとランチャの生成 ==================================================
  /**
   * PatternViewerLauncherを生成する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternArray0 double[][] // パターン描画用データの配列
   * @param patternComment0 String // パターン・コメント
   * @return patternVL PatternViewerLauncher // PatternViewerLauncherへの参照
   */
  public static PatternViewerLauncher pattern(
          int patternType0, double[][] patternArray0, String patternComment0) {
    PatternViewerLauncher patternVL =
        new PatternViewerLauncher(patternType0, patternArray0, patternComment0);
    return patternVL;
  }
  
  /**
   * PatternViewerLauncherを生成する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternTensor0 double[][] // パターン描画用データの配列
   * @param patternComment0 String // パターン・コメント
   * @param patternType1 int // 描画パターン・タイプ
   * @param patternTensor1 double[][] // パターン描画用データの配列
   * @param patternComment1 String // パターン・コメント
   * @return patternVL PatternViewerLauncher // PatternViewerLauncherへの参照
   */
  public static PatternViewerLauncher pattern(
      int patternType0, double[][] patternTensor0, String patternComment0,
      int patternType1, double[][] patternTensor1, String patternComment1) {
    
    PatternViewerLauncher patternVL = new PatternViewerLauncher(
        patternType0, patternTensor0, patternComment0,
        patternType1, patternTensor1, patternComment1);
    return patternVL;
  }
  
  /**
   * PatternViewerLauncherを生成する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternTensor0 double[][][] // パターン描画用データの配列
   * @param patternComment0 String // パターン・コメント
   * @param patternType1 int // 描画パターン・タイプ
   * @param patternTensor1 double[][] // パターン描画用データの配列
   * @param patternComment1 String // パターン・コメント
   * @return patternVL PatternViewerLauncher // PatternViewerLauncherへの参照
   */
  public static PatternViewerLauncher pattern(
      int patternType0, double[][][] patternTensor0, String patternComment0,
      int patternType1, double[][] patternTensor1, String patternComment1) {
    
    PatternViewerLauncher patternVL = new PatternViewerLauncher(
        patternType0, patternTensor0, patternComment0,
        patternType1, patternTensor1, patternComment1);
    return patternVL;
  }
  
  /**
   * PatternViewerLauncherを生成する
   * @param patternType0 int // 描画パターン・タイプ
   * @param patternTensor0 double[][][] // パターン描画用データの配列
   * @param patternComment0 String // パターン・コメント
   * @param patternType1 int // 描画パターン・タイプ
   * @param patternTensor1 double[][][] // パターン描画用データの配列
   * @param patternComment1 String // パターン・コメント
   * @return patternVL PatternViewerLauncher // PatternViewerLauncherへの参照
   */
  public static PatternViewerLauncher pattern(
      int patternType0, double[][][] patternTensor0, String patternComment0,
      int patternType1, double[][][] patternTensor1, String patternComment1) {
    
    PatternViewerLauncher patternVL = new PatternViewerLauncher(
        patternType0, patternTensor0, patternComment0,
        patternType1, patternTensor1, patternComment1);
    return patternVL;
  }
  
   /**
   * GraphViewerLauncherを生成する
   * @param epoch int // エポック回数
   * @param interval int // 表示間隔
   * @param dataName String[] // データ名
   * @param graphData double[] // グラフ描画用データの配列
   * @return graphVL GraphViewerLauncher // GraphViewerLauncherへの参照
   */
  public static GraphViewerLauncher graph(
      int epoch, int interval, String[] dataName, double[] graphData) {
    GraphViewerLauncher graphVL =
        new GraphViewerLauncher(epoch, interval, dataName, graphData);
    return graphVL;
  }
  
  /**
   * SomViewerLauncherを生成する
   * @param somType int // 描画パターン・タイプ
   * @param somComment String // パターン・コメント
   * @return somViewerLauncher SomViewerLauncher // SomViewerLauncherへの参照
   */
  public static SomViewerLauncher launchSomViewer(int somType, String somComment) {
    SomViewerLauncher somViewerLauncher =
        new SomViewerLauncher(somType, somComment);
    return somViewerLauncher;
  }
  
  /**
   * プロパティ・ビューワを得る
   * @return propertyViewer　PropertyViewer // プロパティ・ビューワ
   */
  public static PropertyViewer getPropertyViewer() {
    return console.getPropertyViewer();
  }
  
} // DJ

// EOF
