/*
 *  Title: DaiJa_V4 ( Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package util;

import java.awt.FileDialog;
import java.awt.Frame;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import static util.LogEditor.addLog;

/**
 * <p> 表　題: Class: FileAccessor</p>
 * <p> 説　明: ファイルの入出力処理</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020, 2021</p>
 * <p> 作成日: 2019.4.24</p>
 * <p> 更新日: 2019.4.30</p>
 */
public class FileAccessor {

  private static FileAccessor fileAccessor; // 自分自身への参照。
  private static Frame parent; // ダイアログ表示用親フレーム

  // ファイルアクセス用ディフォルト変数
  static final String FILE_NAME = "Daija-";
//  static final String EXTENSION = "CSV";
  static final String EXTENSION = "ini";

  // ファイルアクセス用変数
  BufferedWriter fileWriter;
  BufferedReader fileReader;
  String fileName;
  String fileFullName;
  String fileDirectory;
  File fileId;

  // 初期分別フィルタ用変数
  boolean continueFlag;

  /** FileAccessorのインスタンスを返す。
   *  FileAccessorが生成されていない場合は、本来の構築子で生成する。
   *  親フレームは設定されない。
   * @return fileAccessor FileAccessor // fileAccessorへの参照
   */
  public static FileAccessor createFileAccessor() {
    if (fileAccessor == null) {
      fileAccessor = new FileAccessor();
    }
    return fileAccessor;
  }

  /** FileAccessorのインスタンスを返す。
   *  FileAccessorが生成されていない場合は、本来の構築子で生成する。
   *  @param parentFrame Frame // 親フレーム
   *  @return fileAccessor FileAccessor // fileAccessorへの参照
   */
  public static FileAccessor createFileAccessor(Frame parentFrame) {
    if (fileAccessor == null) {
      fileAccessor = new FileAccessor(parentFrame);
    }
    else {
      parent = parentFrame;
    }
    return fileAccessor;
  }

  /** インスタンスの構築子。
   *  Singltonにするために、構築子はプライベートになっている。
   */
  private FileAccessor() {
    // Do noting
  }
  private FileAccessor(Frame parentFrame) {
    parent = parentFrame;
  }

  /** ファイル・リーダーをオープンする。
   *  @ param String targetFileFullName ファイル名（ディレクトリ、拡張子を含む）
   *  @ return boolean true:ファイル・オープン成功  false:ファイル・オープン失敗
   */
  private boolean openFileReader(String targetFileFullName) {
    if (targetFileFullName == null) {
      addLog("FileAccessor.openFileReader(): TargetFileFullName is null");
      return false;
    }
    try {
      fileId = new File(targetFileFullName);
      fileReader = new BufferedReader(new FileReader(fileId));
      this.fileFullName = targetFileFullName;
    }
    catch (IOException ex) {
      addLog("FileAccessor.openFileReader():"
          + " Failed to open [" + targetFileFullName + "]."
          + "\n" +"IOException = " + ex.toString());
      return false;
    }
    return true;
  }

  /** ファイル・ライターをオープンする。
   *  @ param String targetFileFullName ファイル名ファイル名（ディレクトリ、拡張子を含む）
   *  @ return boolean true:オープン成功  false:オープン失敗
   */
  private boolean openFileWriter(String targetFileFullName) {
    try {
      fileId = new File(targetFileFullName);
      fileWriter = new BufferedWriter(new FileWriter(fileId));
      this.fileFullName = targetFileFullName;
    }
    catch (IOException ex) {
      addLog("FileAccessor.openFileWriter():"
          + " Failed to open [" + targetFileFullName + "]."
          + "\n" +"IOException = " + ex.toString());
      return false;
    }
    return true;
  }

  /** ファイル・リーダーをクローズする。
   *  @ return boolean true:ファイル・オープン成功  false:ファイル・オープン失敗
   */
  private boolean closeFileReader() {
    try {
      if (fileReader != null) fileReader.close();
    }
    catch (IOException ex) {
      addLog("FileAccessor.closeFileReader():"
          + "Failed to close [" + fileFullName + "]."
          + "\n" +"IOException = " + ex.toString());
      return false;
    }
    return true;
  }

  /** ファイル・ライターをクローズする。
   *  @ return boolean true:ファイル・オープン成功  false:ファイル・オープン失敗
   */
  private boolean closeFileWriter() {
    try {
      if (fileWriter != null) {
        fileWriter.close();
      }
    }
    catch (IOException ex) {
      addLog("FileAccessor.closeFileWriter():"
          + "Failed to close [" + fileFullName + "]."
          + "\n" +"IOException = " + ex.toString());
      return false;
    }
    return true;
  }

  /**
   * テキスト・ファイルを読み込む。
   * @param textBuffer ArrayList＜String> // 読み込んだテキストが入るリスト
   * @param newFileName String // ファイル名
   * @return boolean true:ファイル読込成功  false:ファイル読込失敗
   */
  public boolean loadTextFile(ArrayList<String> textBuffer, String newFileName) {
    fileName = newFileName;
    
    if (parent == null) {
      fileFullName = fileName;
    }
    else {
      FileDialog fd = new FileDialog(parent, "Read Text File", FileDialog.LOAD);
      fd.setFile(fileName);
      fd.setVisible(true);
      fileName = fd.getFile();
      if (fileName == null) {
        addLog("FileAccessor.readTextFile(): Failed to get file name."
         + "\n" + "Filename = " + fileName);
        return false;
      }
      else {
        fileDirectory = fd.getDirectory();
        fileFullName = fileDirectory + fileName;
      }
    }

    return readTextFile(textBuffer, fileFullName);
  }

  /**
   * テキスト・ファイルを読み込む。
   * @param textBuffer ArrayList＜String> // 読み込んだテキストが入るリスト
   * @param newFileName String // ファイル名
   * @return boolean true:ファイル読込成功  false:ファイル読込失敗
   */
  public boolean getTextFile(ArrayList<String> textBuffer, String newFileName) {
    fileName = newFileName;
    fileFullName = fileDirectory + fileName;
    return readTextFile(textBuffer, fileFullName);
  }
  
  public boolean readTextFile(ArrayList<String> textBuffer, String newFileName) {
    fileName = newFileName;
    fileFullName = newFileName;
    boolean fileReadFlag = false;
    String buf;
    StringBuilder stringBuffer = new StringBuilder();

    if (!openFileReader(fileFullName)) {
      addLog("FileAccessor.readTextFile(): Failed to open ["
         + fileFullName + "].");
      return false;
    }

    try {
      while (true) {
        buf = fileReader.readLine();
        if (buf == null) {
          break;
        }
        else {
          textBuffer.add(buf);
        }
      }
      if (!closeFileReader()) {
        addLog("FileAccessor.readTextFile(): Failed to close ["
               + fileFullName + "].");
        return false;
      }
    }
    catch (IOException ex) {
      addLog("FileAccessor.readTextFile(): Failed to read ["
          + fileFullName + "].\n" +"IOException = " + ex.toString());
      return false;
    }
    return true;
  }

  /**
   * テキスト・ファイルを保存する。
   * ※ファイル・ダイアログにファイルの仮名を表示する
   * @param textBuffer ArrayList＜String> // ファイルへ保存するデータ
   * @param newFileName String // ファイル名称
   * @return　resultFlag boolean // true:保存成功  false:保存失敗
   */
  public boolean saveTextFile(ArrayList<String> textBuffer, String newFileName) {
    fileName = newFileName + "_" + TimeStamp.getDateAndTime() + "." + EXTENSION;
    
    if (parent != null) {
      FileDialog fd = new FileDialog(parent, "Save Text File", FileDialog.SAVE);
      fd.setFile(fileName);
      fd.setVisible(true);
      fileName = fd.getFile();
      if (fileName != null) {
        fileDirectory = fd.getDirectory();
        fileFullName = fileDirectory + fileName;
      }
      else {
        addLog("FileAccessor.writeTextFile(): File name is null.");
        return false;
      }
    }
    
    return writeTextFile(textBuffer, fileFullName);
  }
  
  /**
   * テキスト・ファイルを書出す。
   * @param textBuffer ArrayList＜String> // ファイルへ出力する文字列
   * @return boolean // true:ファイル書出し成功  false:ファイル書出し失敗
   */
  public boolean writeTextFile(ArrayList<String> textBuffer) {
    if (parent == null) {
      fileFullName =  FILE_NAME + "." + EXTENSION;
    }
    else {
      FileDialog fd = new FileDialog(parent, "Write Text File", FileDialog.SAVE);
      fileName = FILE_NAME + TimeStamp.getDateAndTime() + "." + EXTENSION;
      fd.setFile(fileName);
      fd.setVisible(true);
      fileName = fd.getFile();
      if (fileName == null) {
        addLog("FileAccessor.writeTextFile(): Failed to get file name.");
        return false;
      }
      else {
        fileDirectory = fd.getDirectory();
        fileFullName = fileDirectory + fileName;
      }
    }

    return writeTextFile(textBuffer, fileFullName);
  }

  public boolean writeTextFile(ArrayList<String> textBuffer, String newFileName) {
    if (!openFileWriter(newFileName)) return false;

    int i,j;
    try {
//      fileWriter.write("#" + newFileName +"\n"); // ファイルへヘッダーを書き込み

      Iterator anEnum = textBuffer.listIterator();
      while (anEnum.hasNext()) {
        fileWriter.write((String)anEnum.next());
      }
      if (!closeFileWriter()) return false;
    }
    catch (IOException ex) {
      addLog("FileAccessor.writeTextFile(): Failed to access ["
          + newFileName + "].\n" +"IOException = " + ex.toString());
      return false;
    }
    addLog("FileAccessor.writeTextFile(): [" + fileFullName + "] is saved.");
    return true;
  }

} // FileAccessor

// EOF
