/*
 *  Title: DaiJa_V4 ( Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package util;

import view.Console;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.StringTokenizer;
import task.Task;
import view.PropertyViewer;

/**
 * <p> 表　題: Class: Console</p>
 * <p> 説　明: メッセージを編集する</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020, 2021</p>
 * <p> 作成日: 2019.1.20</p>
 * 
 * ログやエラーを表示したりファイルへ保存したりする。
 * ファイルにアクセスする場合は、FileAccessorクラスを生成する必要がある。
 */
public class LogEditor {

  Task task; // アプリケーション実行用タスク
  
  private static Console console; // コンソールへの参照
  private static PropertyViewer propertyViewer; // プロパティ・ビューワ
  private static FileAccessor fileAccessor;
  private static final String DEFAULT_FILE_NAME = "DaiJa_text"; // ディフォルト・ファイル名
  private static final String FILE_NAME = "Daija_Log";
  private static final String EXTENSION = "dat";
//$  private static final String LOG_FILE_NAME = "DaiJa_V4_Log.dat"; // ログ・ファイル名
  private static final String LOG_FILE_NAME = FILE_NAME + "." + EXTENSION; // ログ・ファイル名

  private static final ArrayList<String> READ_TEXT_BUFFER = new ArrayList<>(100);
  private static final ArrayList<String> WRITE_TEXT_BUFFER = new ArrayList<>(100);
  
  public static final boolean LOG_ACTIVE = true; // ログ実行中
  public static final boolean LOG_STOP = false; // ログ停止中
  private static boolean logActiveFlag = LOG_STOP; // ログ実行中・停止中フラグ

  public static final int ITEM_ERROR = 0; // エラー
  public static final int ITEM_MESSAGE = 1; // メッセージ
  private static final boolean[] LOG_STATE = new boolean[2];

  public static boolean ERR; // エラー・ステータス・フラグ
  public static boolean MSG; // メッセージ・ステータス・フラグ
  
  private static final boolean NG = false; // フォールス・フラグ
  private static final boolean OK = true; // トゥルー・フラグ
  
  /**
   * インスタンスの構築子。
   * @param aConsole Console // 
   */
  public LogEditor(Console aConsole) {
    console = aConsole;
    fileAccessor = FileAccessor.createFileAccessor(console);
    initializelogStatus();
    updateLogStatusFlag();
  }
  
  /** ログ・ステータスを初期化する。 */
  private void initializelogStatus() {
    LOG_STATE[ITEM_ERROR] = true;
    LOG_STATE[ITEM_MESSAGE] = true;
  }

  /** ログ・ステータスの設定が変わったときに、フラグを更新する。 */
  private void updateLogStatusFlag() {
    ERR = LOG_STATE[ITEM_ERROR];
    MSG = LOG_STATE[ITEM_MESSAGE];
  }
  
  /** ログ・ステータスを、強制的にリセットする。 */
  private void resetLogStatusFlag() {
    ERR = false;
    MSG = false;
    addLog("LogEditor.resetLogStatusFlag(): LogStatusFlags are reset.");
  }
  
  /**
   * 引数で与えられたメッセージを表示エリアとファイルバッファへ追加する。
   * @param message String // メッセージ
   */
  public static void addLog(String message) {
    if (ERR || MSG) {
//      writeToTextBuffer(message);
      WRITE_TEXT_BUFFER.add(message + "\n");
//      writeToTextArea(message); 
      if (console != null)  console.writeToTextArea(message + "\n");
    }
  }
  
  /**
   * メッセージをプリントして改行しない
   * @param message String
   */
  public static void print_(String message) {
    if (ERR || MSG) {
      WRITE_TEXT_BUFFER.add(message);
      if (console != null)  console.writeToTextArea(message);
    }
  }
  
  /**
   * メッセージをプリントして改行する
   * @param message String
   */
  public static void print(String message) {
    addLog(message);
  }
  
  /**
   * 改行してからメッセージをプリントして、再度の改行はしない
   * @param message String
   */
  public static void _print_(String message) {
    if (ERR || MSG) {
      WRITE_TEXT_BUFFER.add("\n" + message);
      if (console != null)  console.writeToTextArea("\n" + message);
    }
  }
  
  /**
   * 改行してからメッセージをプリントして、再度改行する
   * @param message String
   */
  public static void _print(String message) {
    addLog("\n" + message);
  }
  
//$
//  /**
//   * メッセージをファイルバッファに追加する。
//   * @param message String // メッセージ
//   */
//  private static void writeToTextBuffer(String message) {
////    this.WRITE_TEXT_BUFFER.add(message + "\n");
//    WRITE_TEXT_BUFFER.add(message + "\n");
//  }

//$
//  /**
//   * メッセージをテキストエリアに追加する。
//   * @param message String // メッセージ
//   */
//  private static void writeToTextArea(String message) {
//    if (console != null)  console.writeToTextArea(message + "\n");
//  }
  
  /** ログ開始・停止フラグを設定する。
   */
  public void setLogActive() {
    if (logActiveFlag == LOG_ACTIVE) {
      updateLogStatusFlag(); // ログ・ステータスを、更新（元に戻す）する。
      addLog("LogEditor.setLogActive(): Logging is set.");
      logActiveFlag = LOG_STOP;
    }
    else {
      addLog("LogEditor.setLogActive(): Logging is going to reset.");
      resetLogStatusFlag(); // ログ・ステータスを、強制的にリセットする。
      logActiveFlag = LOG_ACTIVE;
    }
  }

  /** ログをファイルにダンプする。
   *  ダンプしたバッファはクリアされる。
   *  @return boolean // OK: true, NG: false
   */
  public boolean dumpLog() {
    if (fileAccessor == null) {
      addLog("LogEditor.dumpLog(): fileAccessor is null.");
      addLog("LogEditor.dumpLog(): Log is not dumped.");
      return NG;
    }
    else {
      String logFileName = FILE_NAME + TimeStamp.getDateAndTime() + "." + EXTENSION;
      fileAccessor.writeTextFile(WRITE_TEXT_BUFFER, logFileName);
      WRITE_TEXT_BUFFER.clear();
      addLog("LogEditor.dumpLog(): Log is dumped.");
      return OK;
    }
  }

  /** ログをファイルからロードする。
   *  ロードする前にバッファがクリアされる。
   *  ロードしたファイルはLogFrameに表示される。
   *  @return boolean // OK: true, NG: false
   */
  public boolean loadLog() {
    if (console != null) {
      addLog("LogEditor.loadLog(): Console is null.");
      addLog("LogEditor.loadLog(): Log is not loaded.");
      return NG;
    }
    if (fileAccessor == null) {
      addLog("LogEditor.loadLog(): fileAccessor is null.");
      addLog("LogEditor.loadLog(): Log is not loaded.");
      return NG;
    }
    else {
      READ_TEXT_BUFFER.clear();
//$      if (fileAccessor.readTextFile(READ_TEXT_BUFFER)) {
      if (fileAccessor.loadTextFile(READ_TEXT_BUFFER, LOG_FILE_NAME)) {
        Iterator anIterator = READ_TEXT_BUFFER.iterator();
        while (anIterator.hasNext()) {
//          writeToTextArea((String)anIterator.next());
          console.writeToTextArea((String)anIterator.next() + "\n");
        }
        addLog("LogEditor.loadLog(): Log is loaded.");
        return OK;
      }
      else {
        addLog("LogEditor.loadLog(): Loading is failed.");
        addLog("LogEditor.loadLog(): Log is not loaded.");
        return NG;
      }
    }
  }

  public void clearLog() {
    WRITE_TEXT_BUFFER.clear();
    READ_TEXT_BUFFER.clear();
    addLog("LogEditor.clearLog(): Log buffer is cleared.");
  }
  
  /**
   * テキストファイルの読み込み
   * ※デフォルトの仮称ファイル名("Daija.txt")
   * @return READ_TEXT_BUFFER ArrayList // 読み込んだ文字列
   */
  public static ArrayList<String> loadText() {
    return loadText(DEFAULT_FILE_NAME);
  }
  
  /**
   * テキストファイルの読み込み
   * @param fileName String // ファイル名
   * @return READ_TEXT_BUFFER ArrayList // 読み込んだ文字列
   */
  public static ArrayList<String> loadText(String fileName) {
    if (fileAccessor == null) {
      addLog("LogEditor.loadText(): fileAccessor is null.");
      addLog("LogEditor.loadText(): Text file is not loaded.");
      console.setVisible(true);
      return null;
    }
    
    READ_TEXT_BUFFER.clear();
    if (fileAccessor.loadTextFile(READ_TEXT_BUFFER, fileName)) {
      addLog("LogEditor.loadLog(): Text file is loaded.");
      return READ_TEXT_BUFFER;
    }
    else {
      addLog("LogEditor.loadLog(): Loading is failed.");
      addLog("LogEditor.loadLog(): Text file is not loaded.");
      return null;
    }
  }
  
  /**
   * テキストファイルの読み込み
   * @param fileName String // ファイル名
   * @return READ_TEXT_BUFFER ArrayList // 読み込んだ文字列
   */
  public static ArrayList<String> getText(String fileName) {
    READ_TEXT_BUFFER.clear();
    if (fileAccessor.getTextFile(READ_TEXT_BUFFER, fileName)) {
      addLog("LogEditor.loadLog(): Text file is loaded.");
      return READ_TEXT_BUFFER;
    }
    else {
      addLog("LogEditor.loadLog(): Loading is failed.");
      addLog("LogEditor.loadLog(): Text file is not loaded.");
      return null;
    }
  }
  
  /**
   * テキストファイルの保存
   * @param elementList ArrayList // 保存する文字列
   * @return resultFlag boolean // 保存の成否
   */
  public static boolean saveText(ArrayList<String> elementList) {
    return saveText(elementList, DEFAULT_FILE_NAME);
  }
  
  /**
   * テキストファイルの保存
   * @param elementList ArrayList // 保存する文字列
   * @param fileName String // ファイル名
   * @return resultFlag boolean // 保存の成否
   */
  public static boolean saveText(
      ArrayList<String> elementList, String fileName) {
    if (fileAccessor == null) {
      addLog("LogEditor.saveText(): fileAccessor is null.");
      addLog("LogEditor.saveText(): Text file is not saved.");
      return NG;
    }
    else {
//      fileAccessor.saveTextFile(elementList, fileName);
      fileAccessor.writeTextFile(elementList, fileName);
      addLog("LogEditor.saveText(): Text file is saved.");
      return OK;
    }
  }
  
  // ---------------------------------------------------------------------------
  /**
   * ファイルから２次の整数配列を読込
   * @param file String // ファイル
   * @return intArray int[][] // ２次の整数配列
   */
  public static int[][] loadIntData(String file) {
    ArrayList<String> stringList = LogEditor.getText(file);
    int rowNum = stringList.size();
    if (rowNum < 1) {
      LogEditor.print("***** ERROR ***** LogEditor.loadIntData():");
      LogEditor.print(" Number of row: " + rowNum + " < 1");
      return null;
    }
    ArrayList<Integer> arrayList = new ArrayList<>();
    String lineString = stringList.get(0);
    loadIntLine(lineString, arrayList);
    
    int columnNum = arrayList.size();
    int[][] intArray = new int[rowNum][columnNum];
    
    for (int j = 1; j < columnNum; j++) {
      intArray[0][j] = arrayList.get(j);
    }
    
    for (int i = 1; i < rowNum; i++) {
      lineString = stringList.get(i);
      arrayList.clear();
      loadIntLine(lineString, arrayList);
      for (int j = 1; j < columnNum; j++) {
        intArray[i][j] = arrayList.get(j);
      }
    }
    
    return intArray;
  } // loadtxt()
  
  // 一行分の要素をリストに書き込む
  private static void loadIntLine(String lineString, ArrayList<Integer> arrayList) {
    
    StringTokenizer lineToken = new StringTokenizer(lineString, ",");
    while (lineToken.hasMoreTokens()) {
      String nextStr = lineToken.nextToken().trim();
      if ("".equals(nextStr)) {
        LogEditor.print("***** WARNING ***** LogEditor.loadIntLine():");
        LogEditor.print(" Next token in a lineString is void.");
        continue;
      }
      int d = Integer.parseInt(nextStr);
      arrayList.add(d); // データ要素
    }
  } // loadLine()
  
  // ---------------------------------------------------------------------------
  /**
   * ファイルから２次の配列を読込
   * @param file String // ファイル名
   * @return doubleArray double[][] // ダブル・テンソル
   */
  public static double[][] loadData(String file) {
    ArrayList<String> stringList = LogEditor.getText(file);
    int rowNum = stringList.size();
    if (rowNum < 1) {
      LogEditor.print("***** ERROR ***** LogEditor.loadData():");
      LogEditor.print(" Number of row: " + rowNum + " < 1");
      return null;
    }
    ArrayList<Double> arrayList = new ArrayList<>();
    String lineString = stringList.get(0);
    loadLine(lineString, arrayList);
    int columnNum = arrayList.size();
    double[][] doubletArray = new double[rowNum][columnNum];
    for (int j = 1; j < columnNum; j++) {
      doubletArray[0][j] = arrayList.get(j);
    }
    for (int i = 1; i < rowNum; i++) {
      lineString = stringList.get(i);
      arrayList.clear();
      loadLine(lineString, arrayList);
      for (int j = 1; j < columnNum; j++) {
        doubletArray[i][j] = arrayList.get(j);
      }
    }
    return doubletArray;
  }
  
  // 一行分の要素をリストに書き込む
  private static void loadLine(String lineString, ArrayList<Double> arrayList) {
    StringTokenizer lineToken = new StringTokenizer(lineString, ",");
    while (lineToken.hasMoreTokens()) {
      String nextStr = lineToken.nextToken().trim();
      if ("".equals(nextStr)) {
        LogEditor.print("***** WARNING ***** LogEditor.loadtxt():");
        LogEditor.print(" Next token in a lineString is void.");
        continue;
      }
      double d = Double.parseDouble(nextStr);
      arrayList.add(d); // データ要素
    }
  }
  
  /**
   * テキストデータの保存
   * @param text String[]  // 保存する文字列の配列
   * @param data double[][] // 保存するデータの２次元配列
   * @param fileName String // 保存するファイルの名称
   */
  public static void saveData(String[] text, double[][] data, String fileName) {
    ArrayList<String> buf = new ArrayList<>(text.length + data.length);
    // 文字列をバッファに追加
    for (int i = 0; i < text.length; i++) {
      buf.add(text[i] + "\r\n");
    }
    // データをバッファに追加
    for (int i = 0; i < data.length; i++) {
      StringBuilder str = new StringBuilder();
      str.append(data[i][0]);
      for (int j = 1; j < data[0].length; j++) {
        str.append(",").append(data[i][j]);
      }
      buf.add(str.toString() + "\r\n");
    }
    LogEditor.saveText(buf, fileName); // ファイルに保存
  }
  
} // LogEditor

// EOF
