/*
 *  Title: DaiJa_V5 (Digital-learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 5.0
 *  @since 2022.8.18
 *  Copyright: 2020 - 2024
 */
package util.comp;

import java.util.Random;
import quan.QuBit;
import util.DJ;

/**
 * <p> 表　題: Class: CompMat</p>
 * <p> 説　明: 複素数行列</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2024</p>
 * <p> 作成日: 2024.03.04</p>
 */
public class CompMat {
  
  public static long RANDOM_SEED = 314; // 疑似乱数初期値
    
  private Complex[][] complexMatrix; // 複素数配列（２次元）
  
  /**
   * デフォルト（要素0個）の複素数行列を生成する
   */
  public CompMat() {
//    complexMatrix = new Complex[1][1];
//    complexMatrix[0][0] = new Complex();    
  }
  
  /**
   * 引数で与えられたサイズの複素数行列を生成する
   * @param row int // 行数（列の長さ）
   * @param col int // 列数（行の長さ）
   */
  public CompMat(int row, int col) {
    complexMatrix = new Complex[row][col];
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        complexMatrix[i][j] = new Complex();
      }
    }
  }
  
// 単位複素数行列とシグネチャが同じになるので割愛する。
// 代わりにCompMat(int order, int order)を用いる。
//  /**
//   * 引数で与えられたサイズの複素数正方行列を生成する
//   * @param order int // 正方行列の次数
//   */
//  public CompMat(int order) {
//    complexMatrix = new Complex[order][order];
//    
//    for (int i = 0; i < order; i++) {
//      for (int j = 0; j < order; j++) {
//        complexMatrix[i][j] = new Complex();
//      }
//    }
//  }
  
  /**
   * 単位複素数行列（正方行列）を生成する
   * @param order int // 行列の次数
   */
  public CompMat(int order) {
    complexMatrix = new Complex[order][order];
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        complexMatrix[i][j] = new Complex(0.0, 0.0);
      }
      complexMatrix[i][i] = new Complex(1.0, 0.0);
    }
  }

  /**
   * 複素数行列（正方行列）を生成し、複素数を設定する
   * @param order int // 正方行列の次数
   * @param element Complex // 設定する複素数
   */
  public CompMat(int order, Complex element) {
    complexMatrix = new Complex[order][order];
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        complexMatrix[i][j] = element.copyComplex();
      }
    }
  }
  
  /**
   * 複素数行列（正方行列）を生成し、複素数を設定する
   * @param dataArray Complex[] // 設定する複素数配列
   */
  public CompMat(Complex[] dataArray) {
    int length = dataArray.length;
    int order = (int)Math.sqrt(length);
    complexMatrix = new Complex[order][order];
    for (int i = 0; i < order; i++) {
      for (int j = 0; j < order; j++) {
        complexMatrix[i][j] = dataArray[i * order + j];
      }
    }
  }
  
  /**
   * 複素数行列（正方行列）を設定する
   * @param dataMat Complex[][] // 設定する複素数行列
   */
  public CompMat(Complex[][] dataMat) {
//    int length = dataMat.length;
//    int order = (int)Math.sqrt(length);
//    complexMatrix = new Complex[order][order];
//    for (int i = 0; i < order; i++) {
//      for (int j = 0; j < order; j++) {
//        complexMatrix[i][j] = dataMat[i][j];
//      }
//    }
    complexMatrix = dataMat;
  }
  
//  private Complex[][] complexMatrix; // 複素数行列
  
  /**
   * 量子ビットで量子ビット行列を生成する
   * @param quBitArray QuBit[] // 量子ビットの配列
   */
  public CompMat(QuBit[] quBitArray) {
    int length = quBitArray.length;
    complexMatrix = new Complex[length][length];
    for (int i = 0; i < length; i++) {
      QuBit aQuBit = quBitArray[i];
      for (int j = 0; j < length; j++) {
        complexMatrix[j][i] = aQuBit.getProbAmp(j);
      }
    }
  }
  
  /**
   * 複素数行列を乱数で生成する
   * @param row int // 行数（列・縦列配列の要素数）
   * @param col int // 列数（行・横配列の要素数）
   * @param randomSeed long // 疑似乱数初期値
   */
  public CompMat(int row, int col, long randomSeed) {
    Random random = new Random(randomSeed); // 疑似乱数
    random = new Random(randomSeed);
    complexMatrix = new Complex[row][col];
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        // 量子ビットの確率振幅を乱数で設定
        complexMatrix[i][j] = new Complex(random.nextDouble(), 0.0);
      }
    }
  }
  
  
  
  

  // ----------------------------------------------------
  /**
   * 複素数行列の要素の実部を得る
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @return real double // 実部
   */
  public double getReal(int row, int col) {
    return complexMatrix[row][col].getReal();
  }

  /**
   * 複素数行列の要素の実部を設定する
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @param real double // 実部
   */
  public void setReal(int row, int col, double real) {
    complexMatrix[row][col].setReal(real);
  }

  /**
   * 複素数行列の要素の虚部を得る
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @return imag Double // 虚部
   */
  public double getImag(int row, int col) {
    return complexMatrix[row][col].getImag();
  }

  /**
   * 複素数行列の要素の虚部を設定する
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @param imag Double // 虚部
   */
  public void setImag(int row, int col, double imag) {
    complexMatrix[row][col].setImag(imag);
  }

  /**
   * 複素数行列の要素を得る
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @return element Complex // 要素
   */
  public Complex getComp(int row, int col) {
    return complexMatrix[row][col];
  }

  /**
   * 複素数行列の要素を新規複素数で得る
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @return element Complex //  要素
   */
  public Complex copyComp(int row, int col) {
    return complexMatrix[row][col].copyComplex();
  }

  /**
   * 複素数行列の要素を複素数のプリミティブで設定する
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @param real double // 実部
   * @param imag double // 虚部
   */
  public void setComp(int row, int col, double real, double imag) {
    complexMatrix[row][col].setComplex(real, imag);
  }

  /**
   * 複素数行列の要素を複素数のプリミティブで設定する
   * @param row int // 要素の行インデックス
   * @param col int // 要素の列インデックス
   * @param element Complex // 要素
   */
  public void setComp(int row, int col, Complex element) {
    complexMatrix[row][col].setComplex(element);
  }

  /**
   * 複素数行列の行を得る
   * @param rowNumber int // 行番号
   * @return 
   */
  public CompVec getRow(int rowNumber) {
    CompVec row = new CompVec(complexMatrix[rowNumber]);
    return row;
  }
  
  /**
   * 複素数行列の列をコピーする
   * @param colNumber int // コピーする列のインデックス
   * @return resultVec CompVec // コピーした列ベクトル
   */
  public CompVec copyColumn(int colNumber) {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    
    CompVec resultVec = new CompVec(row);
    for (int i = 0; i < row; i ++) {
      Complex aComp = complexMatrix[i][colNumber];
      resultVec.setComp(i, aComp.getReal(), aComp.getImag());
    }
    return resultVec;
  }

// このメソッドは不要、thisCompMat[row][col] = complex;  
// /**
//   * 複素数行列の要素を複素数で置き換える
//   * @param row int // 要素の行インデックス
//   * @param col int // 要素の列インデックス
//   * @param complex Complex // 要素（規複素数数)
//   */
//  public void replaceComp(int row, int col, Complex complex) {
//    complexMatrix[row][col] = complex;
//  }

  /**
   * 複素数行列を得る
   * @return complexArray Complex[][] // 複素数配列
   */
  public Complex[][] getMatrix() {
    return complexMatrix;
  }
  
  /**
   * 複素数行列を新規複素数行列で得る
   * @return 
   */
  public CompMat copyCompMat() {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    CompMat _compMat = new CompMat(row, col);
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        _compMat.setComp(i, j, complexMatrix[i][j]);
      }
    }
    return _compMat;
  }
  
  /**
   * 複素数行列を新規複素数行列で得る
   * @return complexArray Complex[][] // 複素数配列
   */
  public Complex[][] copyMatrix() {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    CompMat _compMat = new CompMat(row, col);
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        _compMat.setComp(i, j, complexMatrix[i][j]);
      }
    }
    return _compMat.complexMatrix;
  }
  
  /**
   * 複素数行列を複素数行列のプリミティブで設定する（）
   * @param complexMatrix complex[][] // 複素数行列
   */
  public void setArray(Complex[][] complexMatrix) {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        this.complexMatrix[i][j].setComplex(complexMatrix[i][j]);
      }
    }
  }
  
  
  // ------------------------------------------------------
   /**
   * 複素数行列の行数を得る
   * @return row int //  複素行列の行数
   */
  public int rowNumber() {
    return complexMatrix.length;
  }
 
   /**
   * 複素数行列の列数を得る
   * @return col int // 複素数行列の列数
   */
  public int colNumber() {
    return complexMatrix[0].length;
  }
  
  // ------------------------------------------------------
  /**
   * 複素数行列の実数倍
   * ※被演算行列は積（演算結果）で上書きされる
   * @param real  double // 乗算実数
   */
  public void product(double real) {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        complexMatrix[i][j].mul(real);
      }
    }
  }
  
  /**
   * 複素数行列の複素数倍
   * ※被演算行列は積（演算結果）で上書きされる
   * @param comp Complex // 乗算複素数
   */
  public void product(Complex comp) {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        complexMatrix[i][j].mul(comp);
      }
    }
  }
    
  /**
   * 複素数行列と複素数行列の積
   * ※被演算行列は積（演算結果）で上書きされる
   * 対応する要素毎の積を求める
   * @param operand CompMat // 演算数複素数行列
   */
  public void product(CompMat operand) {
    int row = complexMatrix.length;
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "CompMat.product(CompMat)" + "\n"
          + " Number of row (=" + row
          + ") is less than one.");
      return;
    }
    int col = complexMatrix[0].length;
    if (col < 1) {
      DJ.print("***** ERROR ***** "
          + "CompMat.product(CompMat)" + "\n"
          + " Number of col (=" + col
          + ") is less than one.");
      return;
    }
    int operandRow = operand.rowNumber();
    if (operandRow != row) {
      DJ.print("***** ERROR ***** "
          + "CompMat.product(CompMat)" + "\n"
          + " Number of row is not same.\n"
          + " target matrix row is " + row + ", \n"
          + " operand matrix length is " + operandRow + ".");
      return;
    }
    int operandCol = operand.colNumber();
    if (operandCol != col) {
      DJ.print("***** ERROR ***** "
          + "CompMat.product(CompMat)" + "\n"
          + " Number of col is not same.\n"
          + " target matrix col is " + col + ", \n"
          + " operand matrix length is " + operandCol + ".");
      return;
    }

    Complex[][] operandMatrix = operand.getMatrix();
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        complexMatrix[i][j].mul(operandMatrix[i][j]);
      }
    }
  }
  
  
  // ------------------------------------------------------
  /** QuBasisクラスに移植
   * 複素数行列の共役を得る
   * @return _conjugate CompMat // 共役な複素数行列
   */
  public CompMat conjugate() {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    CompMat _conjugate = new CompMat(col, row); // 転置
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        _conjugate.setComp(j, i, complexMatrix[i][j].conjugate());
      }
    }
    return _conjugate;
  }
  
  /** デバッグ済み
   * CompMatとCompVecの内積を求める
   * ※被演算数行列は共役転置行列に置き換えられる
   * @param operand CompVec // 演算数複素数ベクトル
   * @return _innerProduct CompVec // 複素数ベクトル
   */
  public CompVec innerProduct(CompVec operand) {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    int operandlength = operand.length();
    if (row != operandlength) {
      DJ.print("***** ERROR ***** "
          + "Complex.innerProduct(CompMat)" + "\n"
          + " Element numbers are not same."
          + " target row number is " + row + ", "
          + " operand length is " + operandlength + ".");
      return null;
    }

    // 被演算数複素数行列を共役転置行列に置き換える
    CompMat _conjugate = this.conjugate();
    CompVec _innerProduct = new CompVec(col);
    for (int i = 0; i < col; i++) {
      for (int k = 0; k < row; k++) {
        Complex conj = _conjugate.copyComp(i, k);
        conj.mul(operand.getComp(k));
        _innerProduct.getComp(i).add(conj);
      }
    }
    return _innerProduct;
  }
  
  /** デバッグ済み
   * CompMatとCompMatの内積を求める
   * ※被演算数行列は共役転置行列に置き換えられる
   * @param operand Complex[] // 演算数複素数ベクトル
   * @return _innerProduct Complex // 複素数
   */
  public CompMat innerProduct(CompMat operand) {
    int row = complexMatrix.length;
    int col = complexMatrix[0].length;
    int operandRow = operand.rowNumber();
    int operandCol = operand.colNumber();
    if (row != operandRow) {
      DJ.print("***** ERROR ***** "
          + "Complex.innerProduct(CompMat)" + "\n"
          + " Element numbers are not same."
          + " target row number is " + row + ", "
          + " operand row number is " + operandRow + ".");
      return null;
    }

    // 被演算数複素数行列を共役転置行列に置き換える
    CompMat _conjugate = this.conjugate();
    CompMat _innerProduct = new CompMat(col, operandCol);
    for (int i = 0; i < col; i++) {
      for (int j = 0; j < operandCol; j++) {
        for (int k = 0; k < row; k++) {
          Complex conj = _conjugate.copyComp(i, k);
          conj.mul(operand.getComp(k, j));
          _innerProduct.getComp(i, j).add(conj);
        }
      }
    }
    return _innerProduct;
  }
  
  /** デバッグ済み、QuBasisクラスに移植
   * CompMatとCompVecの内積を求める
   * //　param target CompMat // 被演算数複素数行列
   * @param operand QuBit // 演算数量子ビット
   * @return _innerProducts QuBit // 複素数ベクトル
   */
  public QuBit innerProduct(QuBit operand) {
    int row = complexMatrix.length;
    if (row < 1) {
      DJ.print("***** ERROR ***** "
          + "CompMat.innerProduct(QuBit operand)" + "\n"
          + " target row number is less than one.");
      return null;
    }
    int col = complexMatrix[0].length;
    if (col%2 != 0) {
      DJ.print("***** ERROR ***** "
          + "CompMat.innerProduct(QuBit operand)" + "\n"
          + " target column number is (" + col
          + ") is not even number.");
    }
//    int numOfQuBits = operand.getNumOfQuBit();
//    int operandLength = numOfQuBits * 2;
    int operandLength = operand.getProbAmpLength();
    if (row != operandLength) {
      DJ.print("***** ERROR ***** "
          + "Complex.innerProduct(QuBit operand)" + "\n"
          + " Element numbers are not same.\n"
          + " target row number is " + row + ", \n"
          + " operand length is " + operandLength + ".");
      return null;
    }

    // 被演算数複素数行列は共役行列に置き換える
    CompMat _conjugate = this.conjugate();
//    QuBit resultQB = new QuBit(col / 2);
    QuBit resultQB = new QuBit(col / 2);
    CompVec _innerProduct = resultQB.getProbAmp();
    for (int i = 0; i < col; i++) {
      for (int k = 0; k < operandLength; k++) {
        Complex conj = _conjugate.copyComp(i, k);
        conj.mul(operand.getProbAmp(k));
        _innerProduct.getComp(i).add(conj);
      }
    }
    return resultQB;
  }
  
  
  
  // private Complex[][] complexMatrix; // 複素数行列


  // ------------------------------------------------------
  // 複素ベクトルの比較
  public boolean equals(CompMat cm){
    
    if (this == cm) return true; // アドレスが同じ

    int thisRow = this.rowNumber();
    if (thisRow != cm.rowNumber()) return false; // 行数が違う
    int thisCol = this.colNumber();
    if (thisCol != cm.colNumber()) return false; // 列数が違う
    
    for (int i = 0; i < thisRow; i++ ) {
      for (int j = 0; j < thisCol; j++ ) {
        Complex thisComp = complexMatrix[i][j];
        if (!thisComp.equals(cm.complexMatrix[i][j])) {
          return false; // 値が違う
        }
      }
    }

    return true; // 値が同じ
  }
  
  
  
  
  // ==================================================================
  // CompMatクラスでの複素数行列の表示は数学的表示に従う
  // 要素の区切りにカンマを付けない
  // 行（横）表示では行全体を[]で括り、要素毎には括らない
  
  /**
   * 連続する空白文字列を得る
   * @param size int // 空白文字の個数
   * @return space.toString String // 空白文字列
   */
  private String getSpace(int size) {
    StringBuilder space = new StringBuilder(); // 表示文字列
    for (int i = 0; i < size; i++) {
      space.append(" "); // 空白を追加
    }
    return space.toString();
  }
  
  /**
   * 1.複素数行列の文字列（行列表示）を得る
   * @return sb String // 複素数行列の文字列
   */
  @Override
  public String toString() {
    int row = complexMatrix.length; // 行数（列の長さ）
    int col = complexMatrix[0].length; // 列数（行の長さ）
    StringBuilder sb = new StringBuilder("[");
    int index;
    for (int i = 0; i < row; i++) {
      sb.append("[");
      for (int j = 0; j < col; j++) {
        sb.append(complexMatrix[i][j].toString());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

//  /**
//   * 2.複素数行列の文字列（行列表示、書式付き）を得る
//   * @return sb.toString() // 複素数行列の文字列
//   */
//  public String toFormatOld() {
//    int row = complexMatrix.length; // 行数（列の長さ）
//    int col = complexMatrix[0].length; // 列数（行の長さ）
//    StringBuilder sb = new StringBuilder("F\n[");
//    int index;
//    for (int i = 0; i < row; i++) {
//      sb.append("[");
//      for (int j = 0; j < col; j++) {
//        sb.append(complexMatrix[i][j].toFormat());
//        sb.append("  "); // 書式表示では要素間のコンマは付けない
//      }
//      index = sb.length();
//      sb.replace(index - 2, index, "]\n");
//    }
//    index = sb.length();
//    sb.replace(index - 1, index, "]");
//    return sb.toString();
//  }
  
  /**
   * 2.複素数行列の文字列（行列表示、枠・書式付き）を得る
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toFormat() {
    int row = complexMatrix.length; // 行の長さ＝行内の要素数（列数）
    int col = complexMatrix[0].length; // 列の長さ＝列内の要素数（行数）
    int maxWide[] = new int[col]; // 数値文字列の長さの最大長 (最小例：-0+0i)
    for (int j = 0; j < col; j++) maxWide[j] = 5;
    
    // 数値文字列の抽出
    String[][] buf = new String[row][col]; // 文字列バッファ
    int[][] wides = new int[row][col]; // 余白数
    
    for (int i = 0; i < row; i++) {
      for (int j = 0; j < col; j++) {
        // DJ.print_("i,j=", i);  DJ.print_(", ", j);
        String text = complexMatrix[i][j].toFormat();

        if (complexMatrix[i][j].getReal() >= 0) { // 実部が正
          buf[i][j] = " " + text;
          wides[i][j] = text.length() + 1; // 数値文字列の長さ
        } else {
          buf[i][j] = text;
          wides[i][j] = text.length(); // 数値文字列の長さ
        }
        // DJ.print(",  text=", text);
        // DJ.print_("wides[" + i + "][" + j + "]=", wides[i][j]);

        if (wides[i][j] > maxWide[j]) {
          maxWide[j] = wides[i][j]; // wideを更新
        }
        // DJ.print(",  maxWide[" + j + "]=", maxWide[j]);
      }
    }
    
//    DJ._print("buf=");
//    for (int i = 0; i < row; i++) {
//      DJ.print_(" ", i);
//      for (int j = 0; j < col; j++) {
//         DJ.print_(" ", buf[i][j]);
//      }
//         DJ.print("");
//    }
    
    StringBuilder sb = new StringBuilder("F\n");
    int index;
    
    String filler = "  ";
    int gap = filler.length();
    int spaces = maxWide[0];
    for (int j = 1; j < col; j++) {
      spaces = spaces + gap + maxWide[j]; 
    }
    sb.append("┌").append(getSpace(spaces)).append("┐\n");
    
    for (int i = 0; i < row; i++) {
      sb.append("│");
      for (int j = 0; j < col; j++) {
        sb.append(buf[i][j]); // 数値
        sb.append(getSpace(maxWide[j] - wides[i][j])).append(filler); // 余白
      }
      index = sb.length();
      sb.replace(index - gap, index, "│\n");
    }
    sb.append("└").append(getSpace(spaces)).append("┘\n");
    return sb.toString();
  }
  
  /**
   * 3.複素数行列の文字列（行列表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toString(String label) {
    int row = complexMatrix.length; // 行数（列の長さ）
    int col = complexMatrix[0].length; // 列数（行の長さ）
    StringBuilder sb = new StringBuilder(label + ":\n[");
    int index;
    for (int i = 0; i < row; i++) {
      sb.append("[");
      for (int j = 0; j < col; j++) {
        sb.append(complexMatrix[i][j].toString());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

//  /**
//   * 4.複素数行列の文字列（行列表示、ラベル付き、書式付き）を得る
//   * @param label String // 表示用ラベル
//   * @return sb.toString() // 複素数行列の文字列
//   */
//  public String toFormatOld(String label) {
//    int row = complexMatrix.length; // 行数（列の長さ）
//    int col = complexMatrix[0].length; // 列の長さ
//    StringBuilder sb = new StringBuilder(label + ":F\n[");
//    int index;
//    for (int i = 0; i < row; i++) {
//      sb.append("[");
//      for (int j = 0; j < col; j++) {
//        sb.append(complexMatrix[i][j].toFormat());
//        sb.append("  ");
//      }
//      index = sb.length();
//      sb.replace(index - 2, index, "]\n");
//    }
//    index = sb.length();
//    sb.replace(index - 1, index, "]");
//    return sb.toString();
//  }
  
  /**
   * 4.複素数行列の文字列（行列表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @return sb.toString() // 複素数行列の文字列
   */
  public String toFormat(String label) {
    String str = this.toFormat();
    
    return label + ":" + str;
  }
  
  
  
  
  
  
  
  
  // ----------------------------------------------------
  // CompMatクラスでの複素数行列の表示は数学的表示に従う
  // 要素の区切りにカンマを付けない
  // 行（横）表示では行全体を[]で括り、要素毎には括らない
  
  /**
   * 1.複素数行列の文字列（行列表示）を得る
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toString(Complex[][] complexArray) {
    int row = complexArray.length; // 行数（列の長さ）
    int col = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder("[");
    int index;
    for (int i = 0; i < row; i++) {
      sb.append("[");
      for (int j = 0; j < col; j++) {
        sb.append(complexArray[i][j].toString());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

  /** 
   * 2.複素数行列の文字列（行列表示、書式付き）を得る
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toFormat(Complex[][] complexArray) {
    int colLength = complexArray.length; // 行数（列の長さ）
    int order = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder("F\n[");
    int index;
    for (int i = 0; i < colLength; i++) {
      sb.append("[");
      for (int j = 0; j < order; j++) {
        sb.append(complexArray[i][j].toFormat());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

  /** 
   * 3.複素数行列の文字列（行列表示、ラベル付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toString(String label, Complex[][] complexArray) {
    int colLength = complexArray.length; // 行数（列の長さ）
    int order = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder(label + ":\n[");
    int index;
    for (int i = 0; i < colLength; i++) {
//        sb.append("(").append(i).append("):");
      sb.append("[");
      for (int j = 0; j < order; j++) {
        sb.append(complexArray[i][j].toString());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }

  /**
   * 4.複素数行列の文字列（行列表示、ラベル付き、書式付き）を得る
   * @param label String // 表示用ラベル
   * @param complexArray Complex[][] // 複素数の行列
   * @return sb.toString() // 複素数行列の文字列
   */
  public static String toFormat(String label, Complex[][] complexArray) {
    int colLength = complexArray.length; // 行数（列の長さ）
    int order = complexArray[0].length; // 列の長さ
    StringBuilder sb = new StringBuilder(label + ":F\n[");
    int index;
    for (int i = 0; i < colLength; i++) {
      sb.append("[");
      for (int j = 0; j < order; j++) {
        sb.append(complexArray[i][j].toFormat());
        sb.append("  ");
      }
      index = sb.length();
      sb.replace(index - 2, index, "]\n");
    }
    index = sb.length();
    sb.replace(index - 1, index, "]");
    return sb.toString();
  }
  

} // CompMat Class

// End of file

