/*
 *  Title: DaiJa_V4 (Digital-Learning Aide Instrument by JAva)
 *  @author Yoshinari Sasaki
 *  @version 4.0
 *  @since 2020.7.1
 *  Copyright: 2020, 2021
 */
package view;

import java.awt.BorderLayout;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Rectangle;
import java.util.ArrayList;
import util.DJ;
import util.TimeStamp;

/**
 * <p> 表　題: Class: GraphViewer</p>
 * <p> 説　明: 時系列グラフ（２次元t-x）を表示</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.9.8</p>
 */
//public class GraphViewer extends javax.swing.JFrame {
public class GraphViewer extends Viewer {
  
  public static final String BOUNDS_NAME = "graphBounds";
  
  int epoch; // エポック回数
  int interval; // 経過表示間隔
  int sampleNum; // サンプリング数（エポック数 / 経過表示間隔）
  int sampleCount; // サンプリング回数
  int epochCount; // エポック回数
  
  // グラフ描画用変数
  int dataNum; // データ個数
  String[] dataName; // データ名称
  double[] graphData; // グラフ描画用データ：２次の配列
  ArrayList<float[]> dataList; // グラフ描画用データのリスト
  private Edge axes; // グラフ枠の設定
  GraphDrawer grapher; // グラフ描画用クラス：Canvasのサブクラス
  
  /**
   * 時系列グラフを表示する
   * @param epoch int // エポック回数
   * @param interval int // 表示間隔
   * @param dataName String[] // データ名
   * @param graphData double[] // グラフ描画用データ
   */  
  public GraphViewer(int epoch, int interval, String[] dataName, double[] graphData) {
    initComponents();
    
    this.epoch = epoch; // エポック回数
    this.interval = interval; // 表示間隔
    this.sampleNum = epoch / interval; // サンプリング数（表示回数）
    sampleCount = 0; // サンプリング回数を初期化
    epochCount = 0; // エポック回数を初期化
    this.dataNum = dataName.length;
    this.dataName = dataName;
    dataList = new ArrayList<>();
    
    this.graphData = graphData;
    float[] dataSet = getVector(graphData);
    dataList.add(dataSet);
    sampleCount = sampleCount + 1; // サンプリング回数
    
    setBounds(); // 領域を設定
    
    grapher = new GraphDrawer();
    mainPanel.add(grapher, BorderLayout.CENTER);
  }
  
  /**
   * ２次の配列から単精度の配列を取り出す
   * @param aVector double[] // 倍精度浮動少数の配列
   * @return dataSet float[length] // 単精度の配列
   */
  private float[] getVector(double[] aVector) {
    int length = aVector.length;
    float[] dataSet = new float[length]; // １サンプル分のデータ
    for (int i = 0; i < length; i++) {
      dataSet[i] = (float)aVector[i];
    }
    return dataSet;
  }
  
  // 領域を設定
  public final void setBounds() {
    PropertyViewer propertyViewer = DJ.getPropertyViewer();
    Rectangle rect = propertyViewer.getBoundsValue("graphBounds");
    if (rect != null) {
      setBounds(rect);
    }
    else { // ディフォルト値を設定
      setBounds(0, 400, 600, 400);
    }
  }
  
  /**
   * グラフを描画するテンソルを受け取り、再描画する
   * @param epochCount int 
   * @param graphData Tensor
   */
  public void updateGraph(int epochCount, double[] graphData) {
    this.epochCount = epochCount;
    float[] dataSet = getVector(graphData);
    dataList.add(dataSet);
    sampleCount = sampleCount + 1; // サンプリング回数
    grapher.repaint();
  }
  
  /**
   * グラフの縦軸をシフトする
   * @param shift int // シフト回数
   * @return result boolean // 
   */
  public boolean shiftGraphAxis(int shift) {
    if (grapher == null) return false;
    grapher.shiftEdge(shift);
    return true;
  }
  
  /**
   * This method is called from within the constructor to initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is always
   * regenerated by the Form Editor.
   */
  @SuppressWarnings("unchecked")
  // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
  private void initComponents() {

    mainPanel = new javax.swing.JPanel();
    graphCanvas = new java.awt.Canvas();
    controlPanel = new javax.swing.JPanel();
    upButton = new javax.swing.JButton();
    downButton = new javax.swing.JButton();
    frontButton = new javax.swing.JButton();
    backButton = new javax.swing.JButton();

    setTitle("DaiJa : Graph viewer");
    setFocusable(false);
    setLocation(new java.awt.Point(0, 400));
    setName("graphFrame"); // NOI18N
    setSize(new java.awt.Dimension(600, 400));

    mainPanel.setAutoscrolls(true);
    mainPanel.setPreferredSize(new java.awt.Dimension(600, 200));
    mainPanel.setLayout(new java.awt.BorderLayout());
    mainPanel.add(graphCanvas, java.awt.BorderLayout.PAGE_START);

    getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);

    controlPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder());
    controlPanel.setPreferredSize(new java.awt.Dimension(600, 24));
    controlPanel.setRequestFocusEnabled(false);

    upButton.setText("Up");
    upButton.setPreferredSize(new java.awt.Dimension(72, 18));
    upButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        upButtonActionPerformed(evt);
      }
    });

    downButton.setText("Down");
    downButton.setMaximumSize(new java.awt.Dimension(72, 18));
    downButton.setMinimumSize(new java.awt.Dimension(72, 18));
    downButton.setPreferredSize(new java.awt.Dimension(72, 18));
    downButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        downButtonActionPerformed(evt);
      }
    });

    frontButton.setText("Front");
    frontButton.setPreferredSize(new java.awt.Dimension(72, 18));
    frontButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        frontButtonActionPerformed(evt);
      }
    });

    backButton.setText("Back");
    backButton.setPreferredSize(new java.awt.Dimension(72, 18));
    backButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(java.awt.event.ActionEvent evt) {
        backButtonActionPerformed(evt);
      }
    });

    javax.swing.GroupLayout controlPanelLayout = new javax.swing.GroupLayout(controlPanel);
    controlPanel.setLayout(controlPanelLayout);
    controlPanelLayout.setHorizontalGroup(
      controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(controlPanelLayout.createSequentialGroup()
        .addContainerGap()
        .addComponent(downButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(upButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 284, Short.MAX_VALUE)
        .addComponent(backButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
        .addComponent(frontButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
        .addContainerGap())
    );
    controlPanelLayout.setVerticalGroup(
      controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
      .addGroup(controlPanelLayout.createSequentialGroup()
        .addGroup(controlPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
          .addComponent(downButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(upButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(backButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
          .addComponent(frontButton, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
        .addGap(0, 2, Short.MAX_VALUE))
    );

    getContentPane().add(controlPanel, java.awt.BorderLayout.SOUTH);

    pack();
  }// </editor-fold>//GEN-END:initComponents

  private void upButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_upButtonActionPerformed
    axes.setEdge(axes.l, axes.r, axes.b, axes.t / 5.0F);
    grapher.repaint();
  }//GEN-LAST:event_upButtonActionPerformed

  private void downButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_downButtonActionPerformed
    axes.setEdge(axes.l, axes.r, axes.b, axes.t * 5.0F);
    grapher.repaint();
  }//GEN-LAST:event_downButtonActionPerformed
  
  int lastBtn = 2;
  
  private void backButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_backButtonActionPerformed
    if (lastBtn <= 2) {
      axes.setEdge(0, axes.r / 2, axes.b, axes.t);
      lastBtn = lastBtn - 1;
    }
    else {
      sampleNum = epoch / interval; // サンプリング数（表示回数）
      axes.setEdge(0, sampleNum, axes.b, axes.t);
      lastBtn = -3;
    }
    grapher.repaint();
  }//GEN-LAST:event_backButtonActionPerformed

  private void frontButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_frontButtonActionPerformed
    sampleNum = epoch / interval; // サンプリング数（表示回数）
    if (lastBtn > 0) {
      axes.setEdge((int)(sampleNum - (sampleNum / lastBtn)), sampleNum,
                         axes.b, axes.t);
      lastBtn = lastBtn + 1;
    }
    else {
      axes.setEdge(0, sampleNum, axes.b, axes.t);
      lastBtn = +2;
    }
    grapher.repaint();
  }//GEN-LAST:event_frontButtonActionPerformed

// 使わない
//  /**
//   * @param args the command line arguments
//   */
//  public static void main(String args[]) {
//    /* Set the Nimbus look and feel */
//    //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
//    /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
//         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
//     */
//    try {
//      for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
//        if ("Nimbus".equals(info.getName())) {
//          javax.swing.UIManager.setLookAndFeel(info.getClassName());
//          break;
//        }
//      }
//    } catch (ClassNotFoundException ex) {
//      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
//    } catch (InstantiationException ex) {
//      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
//    } catch (IllegalAccessException ex) {
//      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
//    } catch (javax.swing.UnsupportedLookAndFeelException ex) {
//      java.util.logging.Logger.getLogger(PatternViewer.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
//    }
//    //</editor-fold>
//
//    /* Create and display the form */
//    java.awt.EventQueue.invokeLater(new Runnable() {
//      @Override
//      public void run() {
//        new GraphViewer().setVisible(true);
//      }
//    });
//  }
  
  /**
 * <p> 表　題: Class: PatternDrawer</p>
 * <p> 説　明: ２次元パターンを描画する</p>
 * <p> 著　者: Yoshinari Sasaki</p>
 * <p> 著作権: Copyright (c) 2019, 2020</p>
 * <p> 作成日: 2019.8.27</p>
 */
public class GraphDrawer extends Canvas {
  
  // グラフ描画用変数
  private static final String TAG_ERROR = "Error"; // 縦軸名称;
  private static final String TAG_EPOCH = "Epoch"; // 横軸名称;
  
  private static final int FRAME_SPACE = 2; // 10;
  private static final int FRAME_DOUBLE = FRAME_SPACE * 2;
  private static final int AXIS_LEFT = 40; //20; // 左側座標軸スペース
  private static final int AXIS_RIGHT = 0; // 右側座標軸スペース
  private static final int AXIS_TOP = 15; //10; // 上部座標軸スペース
  private static final int AXIS_BOTOM = 20; // 下部座標軸スペース
  private static final int AXIS_SUB = 5; // 座標軸補助線長
  
  private static final int MAX_COLOR = 11;
	private final Color[] colorPalet; // カラーパレット
  
  // グラフ描画用変数
  private Box screen; // 画面サイズ
  private Rect aide; // 補助表示枠
  private Rect board; // 表示枠
  
  // グラフ描画用パラメータ
  private Quad param; // 座標変換用パラメータ(xMag,xShift,yMag,yShift)
  
  // 生成と初期化
  GraphDrawer() {
    // カラー・パレットを生成
    colorPalet = new Color[MAX_COLOR];
    setColorPalet();
    
    // グラフ枠の設定
    axes = Edge.edge();
    sampleNum = epoch / interval; // サンプリング数（表示回数）
    axes.setEdge(0, sampleNum, 0.0F, 2.5F);
  }
  
  // 縦軸をシフトする
  public void shiftEdge(int shift) {
    axes.setEdge(axes.l, axes.r, axes.b, axes.t / (float)Math.pow(5, shift));
  }
  
  // グラフを描画
  @Override
  public void paint(Graphics g) {
    // 画面のサイズを取得（毎回実施）
    Dimension size = getSize();
    screen = Box.box(size.height, size.width);
    
    // 背景塗りつぶし
    g.setColor(Color.getHSBColor(0.1F, 0.1F, 0.2F));
    g.fillRect(0 , 0, screen.w, screen.h);
    
    // 補助表示枠
    aide = Rect.rect(FRAME_SPACE, FRAME_SPACE,
        screen.w - FRAME_DOUBLE, screen.h - FRAME_DOUBLE);
    
    // 表示枠
    g.setColor(Color.getHSBColor(0.13F, 1.0F, 1.0F)); // アンバー色
    board = Rect.rect(aide.x + AXIS_LEFT, aide.y + AXIS_TOP,
        aide.w - AXIS_LEFT - AXIS_RIGHT, aide.h - AXIS_TOP - AXIS_BOTOM);
    g.drawRect(board.x, board.y, board.w, board.h);    // 表示枠の描画
    // 座標軸名称
    g.drawString(TAG_ERROR, board.x - AXIS_LEFT, board.y + board.h / 2);
    g.drawString(TAG_EPOCH, board.x + board.w / 2,
                board.y + board.h + AXIS_BOTOM);
    // 縦軸座標値
    g.setColor(Color.getHSBColor(0.13F, 0.5F, 0.5F));
    for (int j = 0; j <= 5; j++) {
      float axis = axes.t * (float)(5 - j) / 5.0F;
      g.drawString(String.format("%.2e", axis), 
          board.x - AXIS_LEFT + AXIS_SUB * 1, board.y + board.h * j / 5);
    }
    
    // 横軸座標値
    int xMax = axes.r * interval;
    int xMin = axes.l * interval;
    int xWidth = xMax - xMin;
    for (int j = 0; j < 5; j++) {
      float axis = xWidth * (float)j / 5.0F + xMin;
      g.drawString(String.format("%.0f", axis),
          board.x + board.w * j / 5,
                         board.y + board.h + AXIS_BOTOM - AXIS_SUB);
    }
    String axis = String.valueOf(xWidth + xMin);
    g.drawString(axis, board.x + board.w - AXIS_SUB * 5,
                       board.y + board.h + AXIS_BOTOM - AXIS_SUB);
    // 座標軸補助線
    for (int j = 1; j < 5; j++) {
      g.drawLine(board.x - AXIS_SUB, board.y + board.h * j / 5,
                 board.x + board.w, board.y + board.h  * j / 5);
    }
    for (int j = 1; j < 5; j++) {
      g.drawLine(board.x + board.w * j / 5, board.y,
                 board.x + board.w * j / 5, board.y + board.h + AXIS_SUB);
    }
    
    // 日時を表示
    g.setColor(Color.WHITE);
    g.drawString(TimeStamp.getTimeFormated(), 50, 15);
    
    // エポックを表示
    g.setColor(Color.YELLOW);
    g.drawString("Epoch", screen.w - 100, 35);  
    String epochStr = String.valueOf(epochCount); // エポック回数
    g.drawString(epochStr, screen.w - 60, 35);  
    
    // グラフを描画
    drawGraph(g);
    
    // データ名と数値をカラー表示
    writeData(g);
  }
  
  // グラフを描画
  private void drawGraph(Graphics g) {
    updateParameter(); // グラフを描画用パラメータ（毎回実施）

    // グラフの描画
    for (int k = 2; k < (sampleCount); k++) {
      float[] sDataSet = dataList.get(k - 1);
      float[] eDataSet = dataList.get(k);
      
      for (int i = 0; i < dataNum; i++) {
        Fig sFig = Fig.fig(k - 2, sDataSet[i]);
        Fig eFig = Fig.fig(k - 1, eDataSet[i]);
        Dot sDot = getDot(sFig);
        Dot eDot = getDot(eFig);
        g.setColor(getPaletColor(i));
        g.drawLine(sDot.e, sDot.f, eDot.e, eDot.f);
      } // i
    } // k
    
  } // drawGraph
  
  // グラフ描画用パラメータの更新
  private void updateParameter() {
    // 座標変換用パラメータ(xMag, xShift, yMag, yShift)
    float xMag =  (float)board.w / (float)(axes.r - axes.l);
    float xShift = - (float)axes.l * xMag + (float)board.x;
    float yMag = - (float)board.h / (axes.t - axes.b);
    float yShift = - axes.b * yMag + (float)(board.y + board.h);
    param = Quad.quad(xMag, xShift, yMag, yShift);     
  }
  
  // 座標変換
  private Dot getDot(Fig fig) {
    int e = (int)((float)fig.t * param.m + param.n);
    int f = (int)(fig.v * param.s + param.t);
    return Dot.dot(e, f);
  }
  
  // データ名と数値をカラー表示
  private void writeData(Graphics g) {
    for (int i = 0; i < dataNum; i++) {
      g.setColor(getPaletColor(i));
      g.drawString(dataName[i], 72, 20 * i + 45);  
      float val = (float)graphData[i];
      g.drawString(String.valueOf(val), 152, 20 * i + 45);  
    }
  }
  
  // 表示を更新する
  @Override
  public void update(Graphics g) {
    Dimension dim = getSize();
    Image offImage = createImage(dim.width, dim.height);
    Graphics offGraphics = offImage.getGraphics();

    if (offImage != null) {
      paint(offGraphics);
      g.drawImage(offImage, 0, 0, null);
    }
  }
  
  // カラーパレットの設定
  private void setColorPalet() {
    float aHue = 0.0F;
    float aMag = 0.9F; //1.0F;
    float aDim = 0.9F; //1.0F;
    for (int i = 0; i < MAX_COLOR; i++) {
      aHue = 2.0F * (float)i / (float)MAX_COLOR;
      if (aHue > 1.0F) {
        aHue = aHue - 1.0F;
        aMag = 0.6F;
        aDim = 0.8F;
      }
      if (i == 5) aMag = 0.6F;
      if (i == 13) aMag = 0.3F;
      colorPalet[i] = Color.getHSBColor(aHue, aMag, aDim);
    }
  }

  // カラーパレットから色を得る
  private Color getPaletColor(int id) {
    return colorPalet[id];
  }
  
} // GraphDrawer


  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JButton backButton;
  private javax.swing.JPanel controlPanel;
  private javax.swing.JButton downButton;
  private javax.swing.JButton frontButton;
  private java.awt.Canvas graphCanvas;
  private javax.swing.JPanel mainPanel;
  private javax.swing.JButton upButton;
  // End of variables declaration//GEN-END:variables
} // GraphViewer

// EOF

