using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Drawing;
using System.Diagnostics;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using System.Text;
using System.Linq;

/* シンプルホットキー記録・再生ツール *****************************************************/
/* Win32 API構造体とデリゲート ******************************************************************/
[StructLayout(LayoutKind.Sequential)]
public struct POINT
{
    public int x;
    public int y;
}

[StructLayout(LayoutKind.Sequential)]
public struct RECT
{
    public int left;
    public int top;
    public int right;
    public int bottom;
}

[StructLayout(LayoutKind.Sequential)]
public struct MONITORINFO
{
    public uint cbSize;
    public RECT rcMonitor;
    public RECT rcWork;
    public uint dwFlags;
}

[StructLayout(LayoutKind.Sequential)]
public struct MSLLHOOKSTRUCT
{
    public POINT pt;
    public uint mouseData;
    public uint flags;
    public uint time;
    public IntPtr dwExtraInfo;
}

[StructLayout(LayoutKind.Sequential)]
public struct KBDLLHOOKSTRUCT
{
    public uint vkCode;
    public uint scanCode;
    public uint flags;
    public uint time;
    public IntPtr dwExtraInfo;
}

public delegate IntPtr LowLevelMouseProc(int nCode, IntPtr wParam, IntPtr lParam);
public delegate IntPtr LowLevelKeyboardProc(int nCode, IntPtr wParam, IntPtr lParam);

/* ホットキー情報クラス *********************************************************************************/
public class HotkeyInfo
{
    public int Id { get; set; }
    public bool UseCtrl { get; set; }
    public bool UseShift { get; set; }
    public bool UseAlt { get; set; }
    public Keys Key { get; set; }
    public string Description { get; set; }
    public bool IsRegistered { get; set; }
    public List<RecordedAction> RecordedActions { get; set; }
    public bool IsRecording { get; set; }
    public bool ContinuousExecution { get; set; } // 連打実行フラグを追加
    public bool IsPaused { get; set; } // 一時停止フラグを追加
    public bool FastExecution { get; set; } // 高速実行フラグを追加
    public int LoopCount { get; set; } // ループ回数を追加
    
    public HotkeyInfo()
    {
        Description = "";
        IsRegistered = false;
        RecordedActions = new List<RecordedAction>();
        IsRecording = false;
        ContinuousExecution = false; // デフォルトはOFF
        IsPaused = false; // デフォルトは実行可能
        FastExecution = false; // デフォルトはOFF
        LoopCount = 1; // デフォルトは1回
    }
    
    public string GetHotkeyString()
    {
        List<string> parts = new List<string>();
        if (UseCtrl) parts.Add("Ctrl");
        if (UseShift) parts.Add("Shift");
        if (UseAlt) parts.Add("Alt");
        parts.Add(Key.ToString());
        return string.Join("+", parts);
    }
    
    public uint GetModifiers()
    {
        uint modifiers = 0;
        if (UseCtrl) modifiers |= 0x0002; // MOD_CONTROL
        if (UseShift) modifiers |= 0x0004; // MOD_SHIFT
        if (UseAlt) modifiers |= 0x0001; // MOD_ALT
        return modifiers;
    }
}

/* 記録用イベントクラス *********************************************************************************/
public class RecordedAction
{
    public string Type { get; set; }
    public DateTime Timestamp { get; set; }
    public int DelayMs { get; set; }
    public int X { get; set; }
    public int Y { get; set; }
    public uint VirtualKeyCode { get; set; }
    public string KeyName { get; set; }
    public string Details { get; set; }
    public int WParam { get; set; }
    public uint MouseData { get; set; }
    public bool IsKeyDown { get; set; }
    public string Modifiers { get; set; }
    
    public RecordedAction()
    {
        Type = "";
        KeyName = "";
        Details = "";
        Modifiers = "";
        DelayMs = 0;
    }
    
    public string ToFileString()
    {
        try
        {
            return string.Format("{0}|{1}|{2}|{3}|{4}|{5}|{6}|{7}|{8}|{9}|{10}|{11}",
                Type ?? "",
                Timestamp.ToBinary(),
                DelayMs,
                X,
                Y,
                VirtualKeyCode,
                KeyName ?? "",
                Details ?? "",
                WParam,
                MouseData,
                IsKeyDown,
                Modifiers ?? "");
        }
        catch
        {
            return "";
        }
    }
    
    public static RecordedAction FromFileString(string line)
    {
        string[] parts = line.Split('|');
        if (parts.Length != 12) return null;
        
        try
        {
            return new RecordedAction
            {
                Type = parts[0],
                Timestamp = DateTime.FromBinary(long.Parse(parts[1])),
                DelayMs = int.Parse(parts[2]),
                X = int.Parse(parts[3]),
                Y = int.Parse(parts[4]),
                VirtualKeyCode = uint.Parse(parts[5]),
                KeyName = parts[6],
                Details = parts[7],
                WParam = int.Parse(parts[8]),
                MouseData = uint.Parse(parts[9]),
                IsKeyDown = bool.Parse(parts[10]),
                Modifiers = parts[11]
            };
        }
        catch
        {
            return null;
        }
    }
}

/* 修飾キー状態保存クラス *********************************************************************************/
public class ModifierKeyState
{
    public bool LeftCtrl { get; set; }
    public bool RightCtrl { get; set; }
    public bool LeftShift { get; set; }
    public bool RightShift { get; set; }
    public bool LeftAlt { get; set; }
    public bool RightAlt { get; set; }
    public bool LeftWin { get; set; }
    public bool RightWin { get; set; }
    
    public ModifierKeyState()
    {
        LeftCtrl = false;
        RightCtrl = false;
        LeftShift = false;
        RightShift = false;
        LeftAlt = false;
        RightAlt = false;
        LeftWin = false;
        RightWin = false;
    }
    
    public bool HasAnyModifier()
    {
        return LeftCtrl || RightCtrl || LeftShift || RightShift || 
               LeftAlt || RightAlt || LeftWin || RightWin;
    }
}

/* シンプルホットキー管理フォーム *****************************************************************/
public class SimpleHotkeyForm : Form
{
    // Win32 API
    [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
    private static extern IntPtr SetWindowsHookEx(int idHook, LowLevelMouseProc lpfn, IntPtr hMod, uint dwThreadId);

    [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
    private static extern IntPtr SetWindowsHookEx(int idHook, LowLevelKeyboardProc lpfn, IntPtr hMod, uint dwThreadId);

    [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
    [return: MarshalAs(UnmanagedType.Bool)]
    private static extern bool UnhookWindowsHookEx(IntPtr hhk);

    [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
    private static extern IntPtr CallNextHookEx(IntPtr hhk, int nCode, IntPtr wParam, IntPtr lParam);

    [DllImport("kernel32.dll", CharSet = CharSet.Auto, SetLastError = true)]
    private static extern IntPtr GetModuleHandle(string lpModuleName);

    [DllImport("user32.dll")]
    private static extern short GetKeyState(int nVirtKey);

    [DllImport("user32.dll")]
    private static extern bool RegisterHotKey(IntPtr hWnd, int id, uint fsModifiers, uint vk);

    [DllImport("user32.dll")]
    private static extern bool UnregisterHotKey(IntPtr hWnd, int id);

    [DllImport("user32.dll")]
    private static extern IntPtr WindowFromPoint(POINT Point);

    [DllImport("user32.dll")]
    private static extern bool IsWindowVisible(IntPtr hWnd);

    [DllImport("user32.dll")]
    private static extern bool IsIconic(IntPtr hWnd);

    [DllImport("user32.dll")]
    private static extern void mouse_event(uint dwFlags, uint dx, uint dy, uint dwData, IntPtr dwExtraInfo);

    [DllImport("user32.dll")]
    private static extern void keybd_event(byte bVk, byte bScan, uint dwFlags, IntPtr dwExtraInfo);

    [DllImport("user32.dll")]
    private static extern bool SetCursorPos(int X, int Y);

    [DllImport("user32.dll")]
    private static extern short GetAsyncKeyState(int vKey);

    [DllImport("kernel32.dll")]
    private static extern uint GetLastError();

    [DllImport("user32.dll")]
    private static extern bool GetCursorPos(out POINT lpPoint);

    private static Mutex appMutex = null;
    private const string MUTEX_NAME = "Global\\SimpleHotkeyRecorderAndPlayer_SingleInstance_Mutex";

    private static bool alreadyShowedDuplicateMessage = false;

    // 仮想キーコード定数
    private const int VK_LCONTROL = 0xA2;
    private const int VK_RCONTROL = 0xA3;
    private const int VK_LSHIFT = 0xA0;
    private const int VK_RSHIFT = 0xA1;
    private const int VK_LMENU = 0xA4;    // Alt
    private const int VK_RMENU = 0xA5;    // Alt
    private const int VK_LWIN = 0x5B;
    private const int VK_RWIN = 0x5C;

    // 定数
    private const int WH_MOUSE_LL = 14;
    private const int WH_KEYBOARD_LL = 13;
    private const int WM_LBUTTONDOWN = 0x0201;
    private const int WM_RBUTTONDOWN = 0x0204;
    private const int WM_MBUTTONDOWN = 0x0207;
    private const int WM_LBUTTONUP = 0x0202;
    private const int WM_RBUTTONUP = 0x0205;
    private const int WM_MBUTTONUP = 0x0208;
    private const int WM_MOUSEMOVE = 0x0200;
    private const int WM_MOUSEWHEEL = 0x020A;
    private const int WM_KEYDOWN = 0x0100;
    private const int WM_KEYUP = 0x0101;
    private const int WM_SYSKEYDOWN = 0x0104;
    private const int WM_SYSKEYUP = 0x0105;
    private const int WM_HOTKEY = 0x0312;

    // マウスイベント定数
    private const uint MOUSEEVENTF_MOVE = 0x0001;
    private const uint MOUSEEVENTF_LEFTDOWN = 0x0002;
    private const uint MOUSEEVENTF_LEFTUP = 0x0004;
    private const uint MOUSEEVENTF_RIGHTDOWN = 0x0008;
    private const uint MOUSEEVENTF_RIGHTUP = 0x0010;
    private const uint MOUSEEVENTF_MIDDLEDOWN = 0x0020;
    private const uint MOUSEEVENTF_MIDDLEUP = 0x0040;
    private const uint MOUSEEVENTF_WHEEL = 0x0800;
    private const uint MOUSEEVENTF_ABSOLUTE = 0x8000;

    // キーボードイベント定数
    private const uint KEYEVENTF_KEYUP = 0x0002;

    // フック用のプロシージャとハンドル
    private LowLevelMouseProc mouseProc;
    private LowLevelKeyboardProc keyboardProc;
    private IntPtr mouseHookID = IntPtr.Zero;
    private IntPtr keyboardHookID = IntPtr.Zero;

    // UI コンポーネント
    private ListView hotkeyListView;
    private CheckBox ctrlCheckBox;
    private CheckBox shiftCheckBox;
    private CheckBox altCheckBox;
    private TextBox keyTextBox;
    private TextBox descriptionTextBox;
    private Button addHotkeyButton;
    private Button changeHotkeyButton; // 変更ボタンを追加
    private Button removeHotkeyButton;
    private Button recordButton;
    private Button clearRecordButton;
    private Button saveButton;
    private Button loadButton;
    private Button exitButton;
    private Label statusLabel;
    private ListView actionListView;
    private CheckBox mouseEventsCheckBox;
    private CheckBox keyboardEventsCheckBox;
    private CheckBox mouseMoveCheckBox;

    // 状態管理
    private bool isRecording = false;
    private bool isPlaying = false;
    private bool forceStopRequested = false; // 強制停止フラグを追加
    private bool isUpdatingSelection = false; // 選択状態更新中フラグを追加
    private DateTime lastEventTime = DateTime.MinValue;
    private List<HotkeyInfo> registeredHotkeys = new List<HotkeyInfo>();
    private Dictionary<uint, bool> keyStates = new Dictionary<uint, bool>();
    private int nextHotkeyId = 1;
    private HotkeyInfo currentRecordingHotkey = null;
    private HotkeyInfo rememberedHotkeyForRecording = null; // 記録開始時に保持するホットキー
    
    // 修飾キー状態の保存用
    private ModifierKeyState savedModifierState = null;
    
    // 強制停止用のホットキーID
    private const int STOP_HOTKEY_ID = 9999;
    
    // ウィンドウ復帰用のホットキーID
    private const int SHOW_WINDOW_HOTKEY_ID = 9998;
    
    // Escキー監視用のフック
    private IntPtr escKeyHookID = IntPtr.Zero;
    private LowLevelKeyboardProc escKeyProc;
    
    // ホットキーリストの列挙を定義（可読性向上）
    private enum HotkeyListColumn
    {
        ID = 0,
        Hotkey = 1,
        Description = 2,
        RecordCount = 3,
        LoopCount = 4, // ループ列を追加
        ContinuousExecution = 5,
        FastExecution = 6,
        IsPaused = 7,
        Status = 8
    }

    public SimpleHotkeyForm()
    {
        // ★修正：二重起動チェック（最初に実行）
        if (!CheckSingleInstance())
        {
            // 二重起動の場合は例外をスローしてコンストラクタを中断
            throw new InvalidOperationException("Application is already running.");
        }

        InitializeComponent();
        mouseProc = MouseHookProc;
        keyboardProc = KeyboardHookProc;
        escKeyProc = EscKeyHookProc;
        
        // Escキー監視用の低レベルキーボードフックを設定
        escKeyHookID = SetHook(escKeyProc, WH_KEYBOARD_LL);
    }

    // 単一インスタンスチェックのメソッド
    private bool CheckSingleInstance()
    {
        bool createdNew = false;
        
        try
        {
            // 初期所有権なしでMutexを作成
            appMutex = new Mutex(false, MUTEX_NAME, out createdNew);
            
            if (createdNew)
            {
                // 新規作成された = 初回起動
                appMutex.WaitOne(); // 所有権を取得
                System.Diagnostics.Debug.WriteLine("Mutex新規作成: 初回起動");
                return true;
            }
            else
            {
                // 既に存在する = 二重起動の可能性
                // 所有権取得を試みる（即座に判定）
                if (appMutex.WaitOne(0, false))
                {
                    // 取得できた = 前回異常終了したMutex
                    System.Diagnostics.Debug.WriteLine("Mutex既存だが取得可能: 前回異常終了");
                    return true;
                }
                else
                {
                    // 取得できない = 本当に二重起動している
                    System.Diagnostics.Debug.WriteLine("Mutex既存で取得不可: 二重起動");
                    
                    // ★修正：まだメッセージを表示していない場合のみ表示
                    if (!alreadyShowedDuplicateMessage)
                    {
                        alreadyShowedDuplicateMessage = true;
                        
                        MessageBox.Show(
                            "ホットキーツールは既に起動しています。\n" +
                            "タスクトレイまたはタスクマネージャーを確認してください。\n\n" +
                            "ウィンドウを表示するには Ctrl+Alt+H を押してください。",
                            "二重起動エラー",
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Warning);
                    }
                    
                    appMutex.Close();
                    appMutex = null;
                    return false;
                }
            }
        }
        catch (AbandonedMutexException)
        {
            // 前のプロセスが異常終了した場合 - 続行可能
            System.Diagnostics.Debug.WriteLine("AbandonedMutexException: 前回異常終了");
            return true;
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("Mutex作成エラー: " + ex.Message);
            // エラーが発生しても続行を許可
            return true;
        }
    }

    private void InitializeComponent()
    {
        this.Text = "Hotkey Recorder & Player";
        this.Size = new Size(1200, 700); // 幅を100px増加
        this.FormBorderStyle = FormBorderStyle.Sizable;
        this.StartPosition = FormStartPosition.CenterScreen;
        this.MinimumSize = new Size(1100, 600); // 最小幅も調整
        this.ShowInTaskbar = false; // タスクバーに表示しない
        this.BackColor = Color.FromArgb(245, 245, 245); // ライトグレー背景
        
        // モダンなフォント設定
        Font defaultFont = new Font("Meiryo UI", 9, FontStyle.Regular);
        Font titleFont = new Font("Meiryo UI", 11, FontStyle.Bold);
        Font smallFont = new Font("Meiryo UI", 8, FontStyle.Regular);
        this.Font = defaultFont;

        // メインパネル（左側：ホットキー設定）
        Panel hotkeyPanel = new Panel();
        hotkeyPanel.Location = new Point(10, 10);
        hotkeyPanel.Size = new Size(550, 280);
        hotkeyPanel.BorderStyle = BorderStyle.None;
        hotkeyPanel.BackColor = Color.White;
        this.Controls.Add(hotkeyPanel);

        Label hotkeyTitle = new Label();
        hotkeyTitle.Text = "ホットキー設定";
        hotkeyTitle.Location = new Point(15, 15);
        hotkeyTitle.Size = new Size(200, 25);
        hotkeyTitle.Font = titleFont;
        hotkeyTitle.ForeColor = Color.FromArgb(50, 50, 50);
        hotkeyPanel.Controls.Add(hotkeyTitle);

        // 修飾キーチェックボックス
        ctrlCheckBox = new CheckBox();
        ctrlCheckBox.Text = "Ctrl";
        ctrlCheckBox.Location = new Point(15, 50);
        ctrlCheckBox.Size = new Size(60, 23);
        ctrlCheckBox.FlatStyle = FlatStyle.Flat;
        hotkeyPanel.Controls.Add(ctrlCheckBox);

        shiftCheckBox = new CheckBox();
        shiftCheckBox.Text = "Shift";
        shiftCheckBox.Location = new Point(85, 50);
        shiftCheckBox.Size = new Size(60, 23);
        shiftCheckBox.FlatStyle = FlatStyle.Flat;
        hotkeyPanel.Controls.Add(shiftCheckBox);

        altCheckBox = new CheckBox();
        altCheckBox.Text = "Alt";
        altCheckBox.Location = new Point(155, 50);
        altCheckBox.Size = new Size(60, 23);
        altCheckBox.FlatStyle = FlatStyle.Flat;
        hotkeyPanel.Controls.Add(altCheckBox);

        // キー入力
        Label keyLabel = new Label();
        keyLabel.Text = "キー:";
        keyLabel.Location = new Point(15, 80);
        keyLabel.Size = new Size(40, 20);
        keyLabel.ForeColor = Color.FromArgb(100, 100, 100);
        hotkeyPanel.Controls.Add(keyLabel);

        keyTextBox = new TextBox();
        keyTextBox.Location = new Point(60, 78);
        keyTextBox.Size = new Size(100, 23);
        keyTextBox.BorderStyle = BorderStyle.FixedSingle;
        keyTextBox.MaxLength = 1;
        keyTextBox.KeyDown += KeyTextBox_KeyDown;
        hotkeyPanel.Controls.Add(keyTextBox);

        // 説明
        Label descLabel = new Label();
        descLabel.Text = "説明:";
        descLabel.Location = new Point(180, 80);
        descLabel.Size = new Size(40, 20);
        descLabel.ForeColor = Color.FromArgb(100, 100, 100);
        hotkeyPanel.Controls.Add(descLabel);

        descriptionTextBox = new TextBox();
        descriptionTextBox.Location = new Point(225, 78);
        descriptionTextBox.Size = new Size(150, 23);
        descriptionTextBox.BorderStyle = BorderStyle.FixedSingle;
        hotkeyPanel.Controls.Add(descriptionTextBox);

        // ボタン（フラットスタイル）
        addHotkeyButton = new Button();
        addHotkeyButton.Text = "追加";
        addHotkeyButton.Location = new Point(395, 77);
        addHotkeyButton.Size = new Size(70, 25);
        addHotkeyButton.FlatStyle = FlatStyle.Flat;
        addHotkeyButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        addHotkeyButton.BackColor = Color.White;
        addHotkeyButton.Click += AddHotkeyButton_Click;
        hotkeyPanel.Controls.Add(addHotkeyButton);

        // 変更ボタンを追加
        changeHotkeyButton = new Button();
        changeHotkeyButton.Text = "変更";
        changeHotkeyButton.Location = new Point(475, 77);
        changeHotkeyButton.Size = new Size(70, 25);
        changeHotkeyButton.FlatStyle = FlatStyle.Flat;
        changeHotkeyButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        changeHotkeyButton.BackColor = Color.White;
        changeHotkeyButton.Click += ChangeHotkeyButton_Click;
        hotkeyPanel.Controls.Add(changeHotkeyButton);

        removeHotkeyButton = new Button();
        removeHotkeyButton.Text = "削除";
        removeHotkeyButton.Location = new Point(15, 110);
        removeHotkeyButton.Size = new Size(60, 25);
        removeHotkeyButton.FlatStyle = FlatStyle.Flat;
        removeHotkeyButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        removeHotkeyButton.BackColor = Color.White;
        removeHotkeyButton.Click += RemoveHotkeyButton_Click;
        hotkeyPanel.Controls.Add(removeHotkeyButton);

        recordButton = new Button();
        recordButton.Text = "記録開始";
        recordButton.Location = new Point(85, 110);
        recordButton.Size = new Size(80, 25);
        recordButton.FlatStyle = FlatStyle.Flat;
        recordButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        recordButton.BackColor = Color.White;
        recordButton.Click += RecordButton_Click;
        hotkeyPanel.Controls.Add(recordButton);

        clearRecordButton = new Button();
        clearRecordButton.Text = "記録クリア";
        clearRecordButton.Location = new Point(175, 110);
        clearRecordButton.Size = new Size(80, 25);
        clearRecordButton.FlatStyle = FlatStyle.Flat;
        clearRecordButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        clearRecordButton.BackColor = Color.White;
        clearRecordButton.Click += ClearRecordButton_Click;
        hotkeyPanel.Controls.Add(clearRecordButton);

        // ホットキーリスト
        hotkeyListView = new ListView();
        hotkeyListView.Location = new Point(15, 145);
        hotkeyListView.Size = new Size(620, 120); // 幅を100px増加
        hotkeyListView.View = View.Details;
        hotkeyListView.FullRowSelect = true;
        hotkeyListView.GridLines = false;
        hotkeyListView.BorderStyle = BorderStyle.None;
        hotkeyListView.BackColor = Color.White;
        hotkeyListView.Columns.Add("ID", 30);
        hotkeyListView.Columns.Add("ホットキー", 100);
        hotkeyListView.Columns.Add("説明", 90);
        hotkeyListView.Columns.Add("記録数", 50);
        hotkeyListView.Columns.Add("ループ", 50); // ループ列を追加
        hotkeyListView.Columns.Add("連打", 45);
        hotkeyListView.Columns.Add("高速", 45);
        hotkeyListView.Columns.Add("停止", 45);
        hotkeyListView.Columns.Add("状態", 60);
        hotkeyListView.SelectedIndexChanged += HotkeyListView_SelectedIndexChanged;
        hotkeyListView.DoubleClick += HotkeyListView_DoubleClick;
        hotkeyPanel.Controls.Add(hotkeyListView);

        // 記録設定パネル（右上）
        Panel recordPanel = new Panel();
        recordPanel.Location = new Point(570, 10);
        recordPanel.Size = new Size(510, 120);
        recordPanel.BorderStyle = BorderStyle.None;
        recordPanel.BackColor = Color.White;
        this.Controls.Add(recordPanel);

        Label recordTitle = new Label();
        recordTitle.Text = "記録設定";
        recordTitle.Location = new Point(15, 15);
        recordTitle.Size = new Size(200, 25);
        recordTitle.Font = titleFont;
        recordTitle.ForeColor = Color.FromArgb(50, 50, 50);
        recordPanel.Controls.Add(recordTitle);

        // 記録オプション
        mouseEventsCheckBox = new CheckBox();
        mouseEventsCheckBox.Text = "マウスイベント";
        mouseEventsCheckBox.Location = new Point(15, 50);
        mouseEventsCheckBox.Size = new Size(120, 23);
        mouseEventsCheckBox.FlatStyle = FlatStyle.Flat;
        mouseEventsCheckBox.Checked = true;
        recordPanel.Controls.Add(mouseEventsCheckBox);

        keyboardEventsCheckBox = new CheckBox();
        keyboardEventsCheckBox.Text = "キーボードイベント";
        keyboardEventsCheckBox.Location = new Point(145, 50);
        keyboardEventsCheckBox.Size = new Size(130, 23);
        keyboardEventsCheckBox.FlatStyle = FlatStyle.Flat;
        keyboardEventsCheckBox.Checked = true;
        recordPanel.Controls.Add(keyboardEventsCheckBox);

        mouseMoveCheckBox = new CheckBox();
        mouseMoveCheckBox.Text = "マウス移動も記録";
        mouseMoveCheckBox.Location = new Point(285, 50);
        mouseMoveCheckBox.Size = new Size(130, 23);
        mouseMoveCheckBox.FlatStyle = FlatStyle.Flat;
        mouseMoveCheckBox.Checked = false;
        recordPanel.Controls.Add(mouseMoveCheckBox);

        // ファイル操作ボタン
        saveButton = new Button();
        saveButton.Text = "個別保存";
        saveButton.Location = new Point(15, 80);
        saveButton.Size = new Size(70, 28);
        saveButton.FlatStyle = FlatStyle.Flat;
        saveButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        saveButton.BackColor = Color.White;
        saveButton.Click += SaveButton_Click;
        recordPanel.Controls.Add(saveButton);

        loadButton = new Button();
        loadButton.Text = "個別読込";
        loadButton.Location = new Point(95, 80);
        loadButton.Size = new Size(70, 28);
        loadButton.FlatStyle = FlatStyle.Flat;
        loadButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        loadButton.BackColor = Color.White;
        loadButton.Click += LoadButton_Click;
        recordPanel.Controls.Add(loadButton);

        Button saveAllButton = new Button();
        saveAllButton.Text = "全保存";
        saveAllButton.Location = new Point(175, 80);
        saveAllButton.Size = new Size(60, 28);
        saveAllButton.FlatStyle = FlatStyle.Flat;
        saveAllButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        saveAllButton.BackColor = Color.White;
        saveAllButton.Click += (s, ev) => {
            if (registeredHotkeys.Count == 0)
            {
                MessageBox.Show("保存するホットキーがありません。", "警告", 
                    MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            SaveFileDialog saveFileDialog = new SaveFileDialog();
            saveFileDialog.Filter = "全ホットキーファイル (*.hkall)|*.hkall|すべてのファイル (*.*)|*.*";
            saveFileDialog.Title = "全ホットキーデータを保存";
            saveFileDialog.FileName = string.Format("all_hotkeys_{0}.hkall", 
                DateTime.Now.ToString("yyyyMMdd_HHmmss"));

            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                try
                {
                    SaveAllHotkeysToFile(saveFileDialog.FileName);
                    MessageBox.Show(string.Format("{0}個のホットキーデータを保存しました。", registeredHotkeys.Count), "完了", 
                        MessageBoxButtons.OK, MessageBoxIcon.Information);
                    statusLabel.Text = "全ホットキー保存完了";
                }
                catch (Exception ex)
                {
                    MessageBox.Show(string.Format("保存エラー: {0}", ex.Message), "エラー", 
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            saveFileDialog.Dispose();
        };
        recordPanel.Controls.Add(saveAllButton);

        Button loadAllButton = new Button();
        loadAllButton.Text = "全読込";
        loadAllButton.Location = new Point(245, 80);
        loadAllButton.Size = new Size(60, 28);
        loadAllButton.FlatStyle = FlatStyle.Flat;
        loadAllButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        loadAllButton.BackColor = Color.White;
        loadAllButton.Click += (s, ev) => {
            OpenFileDialog openFileDialog = new OpenFileDialog();
            openFileDialog.Filter = "全ホットキーファイル (*.hkall)|*.hkall|すべてのファイル (*.*)|*.*";
            openFileDialog.Title = "全ホットキーデータを読み込み";

            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                try
                {
                    var loadedHotkeys = LoadAllHotkeysFromFile(openFileDialog.FileName);
                    
                    if (loadedHotkeys != null && loadedHotkeys.Count > 0)
                    {
                        if (MessageBox.Show(
                            string.Format("現在のホットキー設定を全て削除して、{0}個のホットキーを読み込みますか？", loadedHotkeys.Count),
                            "確認", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
                        {
                            // 現在のホットキーを全て解除（登録されているもののみ）
                            foreach (var hotkey in registeredHotkeys)
                            {
                                if (hotkey.IsRegistered)
                                {
                                    UnregisterHotKey(this.Handle, hotkey.Id);
                                }
                            }
                            
                            registeredHotkeys.Clear();
                            actionListView.Items.Clear();
                            
                            // 新しいホットキーを登録（修正：RegisterHotkeyReliableを使用）
                            foreach (var hotkey in loadedHotkeys)
                            {
                                hotkey.Id = nextHotkeyId++;
                                
                                // 一時停止中でない場合のみホットキーを登録
                                if (!hotkey.IsPaused)
                                {
                                    bool success = RegisterHotkeyReliable(hotkey);
                                }
                                else
                                {
                                    // 一時停止中の場合は登録しない
                                    hotkey.IsRegistered = false;
                                }
                                
                                registeredHotkeys.Add(hotkey);
                            }
                            
                            RefreshHotkeyList();
                            MessageBox.Show(string.Format("{0}個のホットキーを読み込みました。", loadedHotkeys.Count), "完了", 
                                MessageBoxButtons.OK, MessageBoxIcon.Information);
                            statusLabel.Text = string.Format("全ホットキー読み込み完了 - {0}個", loadedHotkeys.Count);
                        }
                    }
                    else
                    {
                        MessageBox.Show("有効なホットキーデータが見つかりませんでした。", "警告", 
                            MessageBoxButtons.OK, MessageBoxIcon.Warning);
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show(string.Format("読み込みエラー: {0}", ex.Message), "エラー", 
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            openFileDialog.Dispose();
        };
        recordPanel.Controls.Add(loadAllButton);

        exitButton = new Button();
        exitButton.Text = "終了";
        exitButton.Location = new Point(415, 80);
        exitButton.Size = new Size(80, 28);
        exitButton.FlatStyle = FlatStyle.Flat;
        exitButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        exitButton.BackColor = Color.White;
        exitButton.Font = new Font("Meiryo UI", 9, FontStyle.Bold);
        exitButton.Click += (s, ev) => {
            this.Close();
        };
        recordPanel.Controls.Add(exitButton);

        // ステータスラベル
        statusLabel = new Label();
        statusLabel.Text = "ホットキーを追加してください";
        statusLabel.Location = new Point(570, 140);
        statusLabel.Size = new Size(510, 23);
        statusLabel.Font = defaultFont;
        statusLabel.ForeColor = Color.FromArgb(80, 80, 80);
        this.Controls.Add(statusLabel);
        
        // Escキーの説明パネル
        Panel escPanel = new Panel();
        escPanel.Location = new Point(570, 170);
        escPanel.Size = new Size(510, 30);
        escPanel.BackColor = Color.FromArgb(220, 50, 50);
        this.Controls.Add(escPanel);
        
        Label escKeyLabel = new Label();
        escKeyLabel.Text = "ESC キー: 記録中・再生中いつでも即座に停止　　　Ctrl+Alt+H: ウィンドウ表示";
        escKeyLabel.Location = new Point(0, 0);
        escKeyLabel.Size = new Size(510, 30);
        escKeyLabel.Font = new Font("Meiryo UI", 9, FontStyle.Bold);
        escKeyLabel.ForeColor = Color.White;
        escKeyLabel.TextAlign = ContentAlignment.MiddleCenter;
        escKeyLabel.BackColor = Color.Transparent;
        escPanel.Controls.Add(escKeyLabel);

        // 使用方法パネル
        Panel instructionPanel = new Panel();
        instructionPanel.Location = new Point(10, 295);
        instructionPanel.Size = new Size(1170, 50); // 幅を100px増加
        instructionPanel.BackColor = Color.White;
        this.Controls.Add(instructionPanel);
        
        Label instructionLabel = new Label();
        instructionLabel.Text = "使用方法";
        instructionLabel.Location = new Point(15, 5);
        instructionLabel.Size = new Size(100, 20);
        instructionLabel.Font = new Font("Meiryo UI", 9, FontStyle.Bold);
        instructionLabel.ForeColor = Color.FromArgb(50, 50, 50);
        instructionPanel.Controls.Add(instructionLabel);
        
        Label instructionDetailLabel = new Label();
        instructionDetailLabel.Text = "1. ホットキー追加 → 2. リストから選択 → 3. 記録開始 → 4. 操作を実行 → 5. ESCキーで停止 → 6. ホットキー押下で再生　※ループ列をダブルクリックで回数変更";
        instructionDetailLabel.Location = new Point(15, 25);
        instructionDetailLabel.Size = new Size(1140, 20); // 幅を100px増加
        instructionDetailLabel.Font = smallFont;
        instructionDetailLabel.ForeColor = Color.FromArgb(100, 100, 100);
        instructionPanel.Controls.Add(instructionDetailLabel);

        // 動作履歴編集パネル
        Panel editPanel = new Panel();
        editPanel.Location = new Point(10, 350);
        editPanel.Size = new Size(1170, 35); // 幅を100px増加
        editPanel.BackColor = Color.White;
        this.Controls.Add(editPanel);

        Button deleteActionButton = new Button();
        deleteActionButton.Text = "選択削除";
        deleteActionButton.Location = new Point(15, 5);
        deleteActionButton.Size = new Size(80, 25);
        deleteActionButton.FlatStyle = FlatStyle.Flat;
        deleteActionButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        deleteActionButton.BackColor = Color.White;
        deleteActionButton.Click += DeleteActionButton_Click;
        editPanel.Controls.Add(deleteActionButton);

        Button editDelayButton = new Button();
        editDelayButton.Text = "待ち時間編集";
        editDelayButton.Location = new Point(105, 5);
        editDelayButton.Size = new Size(90, 25);
        editDelayButton.FlatStyle = FlatStyle.Flat;
        editDelayButton.FlatAppearance.BorderColor = Color.FromArgb(150, 150, 150);
        editDelayButton.BackColor = Color.White;
        editDelayButton.Click += EditDelayButton_Click;
        editPanel.Controls.Add(editDelayButton);

        Label editHelpLabel = new Label();
        editHelpLabel.Text = "動作を選択 → 削除ボタン or Deleteキー / 待ち時間列をダブルクリックで変更 / F1-F12・無変換・英数キーは修飾キーなしOK";
        editHelpLabel.Location = new Point(210, 8);
        editHelpLabel.Size = new Size(800, 20); // 幅を100px増加
        editHelpLabel.Font = smallFont;
        editHelpLabel.ForeColor = Color.FromArgb(100, 100, 100);
        editPanel.Controls.Add(editHelpLabel);

        // 動作履歴ラベル
        Label actionHistoryLabel = new Label();
        actionHistoryLabel.Text = "動作履歴";
        actionHistoryLabel.Location = new Point(10, 390);
        actionHistoryLabel.Size = new Size(300, 20);
        actionHistoryLabel.Font = titleFont;
        actionHistoryLabel.ForeColor = Color.FromArgb(50, 50, 50);
        this.Controls.Add(actionHistoryLabel);

        actionListView = new ListView();
        actionListView.Location = new Point(10, 415);
        actionListView.Size = new Size(1170, 240); // 幅を100px増加
        actionListView.View = View.Details;
        actionListView.FullRowSelect = true;
        actionListView.GridLines = false;
        actionListView.BorderStyle = BorderStyle.FixedSingle;
        actionListView.BackColor = Color.White;
        actionListView.Anchor = AnchorStyles.Top | AnchorStyles.Bottom | AnchorStyles.Left | AnchorStyles.Right;
        actionListView.Columns.Add("No", 50);
        actionListView.Columns.Add("イベント種別", 150);
        actionListView.Columns.Add("座標/キー", 120);
        actionListView.Columns.Add("待ち時間(ms)", 100);
        actionListView.Columns.Add("詳細", 300);
        actionListView.Columns.Add("タイムスタンプ", 150);
        actionListView.DoubleClick += ActionListView_DoubleClick;
        actionListView.KeyDown += (s, ev) => {
            if (ev.KeyCode == Keys.Delete)
            {
                DeleteSelectedActions();
                ev.Handled = true;
            }
        };
        this.Controls.Add(actionListView);

        // フォーム終了時の処理
        this.FormClosing += SimpleHotkeyForm_FormClosing;
        
        // フォーム読み込み時の処理（ハンドル作成後に自動読み込み）
        this.Load += (s, ev) => {
            // フォームのハンドルが確実に作成されてから自動読み込み
            if (!this.IsHandleCreated)
            {
                this.CreateHandle();
            }
            AutoLoadHotkeys();
            
            // ウィンドウ復帰用のホットキー（Ctrl+Alt+H）を登録
            RegisterHotKey(this.Handle, SHOW_WINDOW_HOTKEY_ID, 0x0002 | 0x0001, (uint)Keys.H); // MOD_CONTROL | MOD_ALT
        };
    }

    private void KeyTextBox_KeyDown(object sender, KeyEventArgs e)
    {
        // 特殊キーの処理
        keyTextBox.Text = e.KeyCode.ToString();
        e.Handled = true;
    }

    // 修飾キーなしで登録可能なキーかどうかを判定するヘルパーメソッド
    private bool IsAllowedWithoutModifiers(Keys key)
    {
        // F1～F24キー
        if (key >= Keys.F1 && key <= Keys.F24)
            return true;
        
        // 無変換キー (IMENonconvert)
        if (key == Keys.IMENonconvert)
            return true;
        
        // 英数キー (IMEConvert - 注：日本語キーボードでは変換キー)
        if (key == Keys.IMEConvert)
            return true;
        
        // ProcessKey (229) - 一部のIME関連キー
        if (key == Keys.ProcessKey)
            return true;
        
        return false;
    }

    private void AddHotkeyButton_Click(object sender, EventArgs e)
    {
        try
        {
            // キーの検証
            if (string.IsNullOrWhiteSpace(keyTextBox.Text))
            {
                MessageBox.Show("キーを指定してください。", "入力エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            Keys key;
            if (!Enum.TryParse(keyTextBox.Text.ToUpper(), out key))
            {
                MessageBox.Show("無効なキーです。", "入力エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            // 修飾キーなしで登録可能なキーかチェック
            bool isAllowedWithoutModifiers = IsAllowedWithoutModifiers(key);

            // 修飾キーの確認（特定のキー以外は修飾キー必須）
            if (!isAllowedWithoutModifiers && 
                !ctrlCheckBox.Checked && !shiftCheckBox.Checked && !altCheckBox.Checked)
            {
                MessageBox.Show(
                    "以下のキー以外は、修飾キー（Ctrl/Shift/Alt）が必要です：\n" +
                    "・ファンクションキー（F1～F12）\n" +
                    "・無変換キー（IMENonconvert）\n" +
                    "・英数/変換キー（IMEConvert）\n" +
                    "・ProcessKey", 
                    "入力エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            // 重複チェック
            foreach (var existing in registeredHotkeys)
            {
                if (existing.UseCtrl == ctrlCheckBox.Checked &&
                    existing.UseShift == shiftCheckBox.Checked &&
                    existing.UseAlt == altCheckBox.Checked &&
                    existing.Key == key)
                {
                    MessageBox.Show("同じホットキーが既に登録されています。", "重複エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                    return;
                }
            }

            // ホットキー情報を作成
            HotkeyInfo hotkey = new HotkeyInfo();
            hotkey.Id = nextHotkeyId++;
            hotkey.UseCtrl = ctrlCheckBox.Checked;
            hotkey.UseShift = shiftCheckBox.Checked;
            hotkey.UseAlt = altCheckBox.Checked;
            hotkey.Key = key;
            hotkey.Description = descriptionTextBox.Text;

            // ホットキーを登録（修正：RegisterHotkeyReliableを使用）
            bool success = RegisterHotkeyReliable(hotkey);
            if (success)
            {
                registeredHotkeys.Add(hotkey);
                RefreshHotkeyList();
                
                // 追加したホットキーを自動選択（動作履歴も自動表示される）
                SelectHotkeyInList(hotkey);
                
                statusLabel.Text = string.Format("ホットキー '{0}' を登録しました。記録を開始してください。", hotkey.GetHotkeyString());

                // 入力フィールドをクリア
                ctrlCheckBox.Checked = false;
                shiftCheckBox.Checked = false;
                altCheckBox.Checked = false;
                keyTextBox.Text = "";
                descriptionTextBox.Text = "";
            }
            else
            {
                MessageBox.Show("ホットキーの登録に失敗しました。\n他のアプリケーションが同じキーを使用している可能性があります。", 
                    "登録エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("ホットキー追加エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }

    // ホットキー変更ボタンのクリックイベント
    private void ChangeHotkeyButton_Click(object sender, EventArgs e)
    {
        try
        {
            // 選択されたホットキーを取得
            HotkeyInfo selectedHotkey = GetSelectedHotkey();
            if (selectedHotkey == null)
            {
                MessageBox.Show("変更するホットキーをリストから選択してください。", "選択エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            // キーの検証
            if (string.IsNullOrWhiteSpace(keyTextBox.Text))
            {
                MessageBox.Show("変更先のキーを指定してください。", "入力エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            Keys newKey;
            if (!Enum.TryParse(keyTextBox.Text.ToUpper(), out newKey))
            {
                MessageBox.Show("無効なキーです。", "入力エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            // 修飾キーなしで登録可能なキーかチェック
            bool isAllowedWithoutModifiers = IsAllowedWithoutModifiers(newKey);

            // 修飾キーの確認（特定のキー以外は修飾キー必須）
            if (!isAllowedWithoutModifiers && 
                !ctrlCheckBox.Checked && !shiftCheckBox.Checked && !altCheckBox.Checked)
            {
                MessageBox.Show(
                    "以下のキー以外は、修飾キー（Ctrl/Shift/Alt）が必要です：\n" +
                    "・ファンクションキー（F1～F12）\n" +
                    "・無変換キー（IMENonconvert）\n" +
                    "・英数/変換キー（IMEConvert）\n" +
                    "・ProcessKey", 
                    "入力エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            // 変更前と同じ場合はスキップ
            if (selectedHotkey.UseCtrl == ctrlCheckBox.Checked &&
                selectedHotkey.UseShift == shiftCheckBox.Checked &&
                selectedHotkey.UseAlt == altCheckBox.Checked &&
                selectedHotkey.Key == newKey)
            {
                MessageBox.Show("変更前と同じホットキーです。", "情報", MessageBoxButtons.OK, MessageBoxIcon.Information);
                return;
            }

            // 重複チェック（自分以外）
            foreach (var existing in registeredHotkeys)
            {
                if (existing.Id != selectedHotkey.Id &&
                    existing.UseCtrl == ctrlCheckBox.Checked &&
                    existing.UseShift == shiftCheckBox.Checked &&
                    existing.UseAlt == altCheckBox.Checked &&
                    existing.Key == newKey)
                {
                    MessageBox.Show("同じホットキーが既に登録されています。", "重複エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                    return;
                }
            }

            // 変更前のホットキー文字列を保存
            string oldHotkeyString = selectedHotkey.GetHotkeyString();

            // 現在のホットキー登録を解除
            if (selectedHotkey.IsRegistered)
            {
                UnregisterHotkeyReliable(selectedHotkey);
            }

            // ホットキー情報を更新
            selectedHotkey.UseCtrl = ctrlCheckBox.Checked;
            selectedHotkey.UseShift = shiftCheckBox.Checked;
            selectedHotkey.UseAlt = altCheckBox.Checked;
            selectedHotkey.Key = newKey;
            if (!string.IsNullOrWhiteSpace(descriptionTextBox.Text))
            {
                selectedHotkey.Description = descriptionTextBox.Text;
            }

            // 一時停止中でない場合は新しいホットキーを登録
            if (!selectedHotkey.IsPaused)
            {
                bool success = RegisterHotkeyReliable(selectedHotkey);
                if (!success)
                {
                    MessageBox.Show("新しいホットキーの登録に失敗しました。\n他のアプリケーションが同じキーを使用している可能性があります。", 
                        "登録エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    
                    // 登録失敗時は一時停止状態にする
                    selectedHotkey.IsPaused = true;
                }
            }

            RefreshHotkeyList();
            
            // 変更したホットキーを再選択（動作履歴も自動表示される）
            SelectHotkeyInList(selectedHotkey);
            
            statusLabel.Text = string.Format("ホットキーを '{0}' → '{1}' に変更しました。", oldHotkeyString, selectedHotkey.GetHotkeyString());
            statusLabel.ForeColor = Color.Green;

            // 入力フィールドをクリア
            ctrlCheckBox.Checked = false;
            shiftCheckBox.Checked = false;
            altCheckBox.Checked = false;
            keyTextBox.Text = "";
            descriptionTextBox.Text = "";
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("ホットキー変更エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }

    private void RemoveHotkeyButton_Click(object sender, EventArgs e)
    {
        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey == null)
        {
            MessageBox.Show("削除するホットキーをリストから選択してください。", "選択エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        try
        {
            // 登録されている場合のみ解除（一時停止中は既に解除されている）
            if (selectedHotkey.IsRegistered)
            {
                UnregisterHotKey(this.Handle, selectedHotkey.Id);
            }
            
            registeredHotkeys.Remove(selectedHotkey);
            RefreshHotkeyList();
            actionListView.Items.Clear();
            
            statusLabel.Text = string.Format("ホットキー '{0}' を削除しました", selectedHotkey.GetHotkeyString());
            statusLabel.ForeColor = Color.Orange;
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("ホットキー削除エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }

    // 選択されたホットキーを取得するヘルパーメソッド
    private HotkeyInfo GetSelectedHotkey()
    {
        try
        {
            if (hotkeyListView.SelectedItems.Count == 0)
                return null;

            ListViewItem selectedItem = hotkeyListView.SelectedItems[0];
            if (selectedItem.SubItems.Count == 0)
                return null;

            int hotkeyId;
            if (!int.TryParse(selectedItem.SubItems[0].Text, out hotkeyId))
                return null;

            return registeredHotkeys.Find(h => h.Id == hotkeyId);
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("GetSelectedHotkey エラー: " + ex.Message);
            return null;
        }
    }

    // 指定したホットキーをリストで選択状態にするヘルパーメソッド
    // 修正：選択後に必ず動作履歴を表示する
    private void SelectHotkeyInList(HotkeyInfo hotkey)
    {
        try
        {
            if (hotkey == null || hotkeyListView == null || hotkeyListView.IsDisposed)
                return;
                
            isUpdatingSelection = true; // 更新中フラグをセット
            
            hotkeyListView.SelectedItems.Clear();
            
            foreach (ListViewItem item in hotkeyListView.Items)
            {
                if (item == null || item.SubItems.Count == 0)
                    continue;
                    
                int itemId;
                if (int.TryParse(item.SubItems[0].Text, out itemId) && itemId == hotkey.Id)
                {
                    item.Selected = true;
                    item.EnsureVisible();
                    break;
                }
            }
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("SelectHotkeyInList エラー: " + ex.Message);
        }
        finally
        {
            isUpdatingSelection = false; // 更新中フラグをリセット
        }
        
        // 選択したホットキーの動作履歴を表示
        DisplayActionsForHotkey(hotkey);
        
        // 記録ボタンの状態を更新
        if (recordButton != null && !recordButton.IsDisposed)
        {
            if (hotkey.IsRecording)
            {
                recordButton.Text = "記録停止";
            }
            else
            {
                recordButton.Text = "記録開始";
            }
        }
        
        // 選択されたホットキーの情報を入力フィールドに表示
        if (!this.IsDisposed)
        {
            ctrlCheckBox.Checked = hotkey.UseCtrl;
            shiftCheckBox.Checked = hotkey.UseShift;
            altCheckBox.Checked = hotkey.UseAlt;
            keyTextBox.Text = hotkey.Key.ToString();
            descriptionTextBox.Text = hotkey.Description;
        }
    }

    // 記録中のホットキーの選択状態を確実に維持するヘルパーメソッド
    private void EnsureRecordingHotkeySelected()
    {
        if (isRecording && currentRecordingHotkey != null && hotkeyListView != null && !hotkeyListView.IsDisposed)
        {
            HotkeyInfo currentSelected = GetSelectedHotkey();
            if (currentSelected == null || currentSelected.Id != currentRecordingHotkey.Id)
            {
                SelectHotkeyInList(currentRecordingHotkey);
            }
        }
    }

    // 現在の修飾キー状態を保存し、すべての修飾キーを解放するメソッド（高速化版）
    private async Task SaveAndReleaseAllModifierKeys(bool isFastExecution = false)
    {
        try
        {
            // 現在の修飾キー状態を保存
            savedModifierState = new ModifierKeyState();
            savedModifierState.LeftCtrl = (GetAsyncKeyState(VK_LCONTROL) & 0x8000) != 0;
            savedModifierState.RightCtrl = (GetAsyncKeyState(VK_RCONTROL) & 0x8000) != 0;
            savedModifierState.LeftShift = (GetAsyncKeyState(VK_LSHIFT) & 0x8000) != 0;
            savedModifierState.RightShift = (GetAsyncKeyState(VK_RSHIFT) & 0x8000) != 0;
            savedModifierState.LeftAlt = (GetAsyncKeyState(VK_LMENU) & 0x8000) != 0;
            savedModifierState.RightAlt = (GetAsyncKeyState(VK_RMENU) & 0x8000) != 0;
            savedModifierState.LeftWin = (GetAsyncKeyState(VK_LWIN) & 0x8000) != 0;
            savedModifierState.RightWin = (GetAsyncKeyState(VK_RWIN) & 0x8000) != 0;
            
            // デバッグ出力：保存した状態
            System.Diagnostics.Debug.WriteLine(string.Format("修飾キー保存: LCtrl={0}, RCtrl={1}, LShift={2}, RShift={3}, LAlt={4}, RAlt={5}, LWin={6}, RWin={7}",
                savedModifierState.LeftCtrl, savedModifierState.RightCtrl, savedModifierState.LeftShift, savedModifierState.RightShift,
                savedModifierState.LeftAlt, savedModifierState.RightAlt, savedModifierState.LeftWin, savedModifierState.RightWin));
            
            // 現在押下されている修飾キーをチェックして解放
            List<int> keysToRelease = new List<int>();
            
            // Ctrl キー
            if (savedModifierState.LeftCtrl)
                keysToRelease.Add(VK_LCONTROL);
            if (savedModifierState.RightCtrl)
                keysToRelease.Add(VK_RCONTROL);
            
            // Shift キー
            if (savedModifierState.LeftShift)
                keysToRelease.Add(VK_LSHIFT);
            if (savedModifierState.RightShift)
                keysToRelease.Add(VK_RSHIFT);
            
            // Alt キー
            if (savedModifierState.LeftAlt)
                keysToRelease.Add(VK_LMENU);
            if (savedModifierState.RightAlt)
                keysToRelease.Add(VK_RMENU);
            
            // Windows キー
            if (savedModifierState.LeftWin)
                keysToRelease.Add(VK_LWIN);
            if (savedModifierState.RightWin)
                keysToRelease.Add(VK_RWIN);
            
            System.Diagnostics.Debug.WriteLine(string.Format("解放するキー数: {0}", keysToRelease.Count));
            
            // 押下されているキーを一括解放（待機時間を短縮）
            foreach (int key in keysToRelease)
            {
                keybd_event((byte)key, 0, KEYEVENTF_KEYUP, IntPtr.Zero);
                System.Diagnostics.Debug.WriteLine(string.Format("キー解放: VK={0:X2}", key));
            }
            
            // 高速実行モードでない場合のみ待機時間を設ける
            if (keysToRelease.Count > 0 && !isFastExecution)
            {
                await Task.Delay(10); // 通常モードでは10ms待機
                System.Diagnostics.Debug.WriteLine("修飾キー解放後の待機: 10ms");
            }
            else if (isFastExecution && keysToRelease.Count > 0)
            {
                System.Diagnostics.Debug.WriteLine("高速実行モード: 修飾キー解放後の待機をスキップ");
            }
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("修飾キー保存・解放エラー: " + ex.Message);
        }
    }

    // 再生中の修飾キー監視タスク
    private async Task WatchModifierKeysAsync(CancellationToken cancellationToken)
    {
        try
        {
            while (!cancellationToken.IsCancellationRequested && savedModifierState != null)
            {
                await Task.Delay(50, cancellationToken); // 50msごとにチェック
                
                // 現在の修飾キー状態をチェック
                bool currentLeftCtrl = (GetAsyncKeyState(VK_LCONTROL) & 0x8000) != 0;
                bool currentRightCtrl = (GetAsyncKeyState(VK_RCONTROL) & 0x8000) != 0;
                bool currentLeftShift = (GetAsyncKeyState(VK_LSHIFT) & 0x8000) != 0;
                bool currentRightShift = (GetAsyncKeyState(VK_RSHIFT) & 0x8000) != 0;
                bool currentLeftAlt = (GetAsyncKeyState(VK_LMENU) & 0x8000) != 0;
                bool currentRightAlt = (GetAsyncKeyState(VK_RMENU) & 0x8000) != 0;
                bool currentLeftWin = (GetAsyncKeyState(VK_LWIN) & 0x8000) != 0;
                bool currentRightWin = (GetAsyncKeyState(VK_RWIN) & 0x8000) != 0;
                
                // ユーザーが再生中に修飾キーを離した場合、復旧対象から除外
                if (savedModifierState.LeftCtrl && currentLeftCtrl)
                {
                    savedModifierState.LeftCtrl = false; // 復旧しない
                }
                if (savedModifierState.RightCtrl && currentRightCtrl)
                {
                    savedModifierState.RightCtrl = false;
                }
                if (savedModifierState.LeftShift && currentLeftShift)
                {
                    savedModifierState.LeftShift = false;
                }
                if (savedModifierState.RightShift && currentRightShift)
                {
                    savedModifierState.RightShift = false;
                }
                if (savedModifierState.LeftAlt && currentLeftAlt)
                {
                    savedModifierState.LeftAlt = false;
                }
                if (savedModifierState.RightAlt && currentRightAlt)
                {
                    savedModifierState.RightAlt = false;
                }
                if (savedModifierState.LeftWin && currentLeftWin)
                {
                    savedModifierState.LeftWin = false;
                }
                if (savedModifierState.RightWin && currentRightWin)
                {
                    savedModifierState.RightWin = false;
                }
            }
        }
        catch (OperationCanceledException)
        {
            // キャンセルされた場合は正常終了
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("修飾キー監視エラー: " + ex.Message);
        }
    }

    // ホットキーを確実に解放するメソッド
    private bool UnregisterHotkeyReliable(HotkeyInfo hotkey)
    {
        try
        {
            if (!hotkey.IsRegistered)
            {
                System.Diagnostics.Debug.WriteLine(string.Format("ホットキー '{0}' は既に未登録", hotkey.GetHotkeyString()));
                return true;
            }

            // ハンドルが作成されていることを確認
            if (!this.IsHandleCreated)
            {
                System.Diagnostics.Debug.WriteLine("ハンドルが作成されていないため解放をスキップ");
                hotkey.IsRegistered = false;
                return true;
            }

            System.Diagnostics.Debug.WriteLine(string.Format("ホットキー解放開始: '{0}' ID={1}", hotkey.GetHotkeyString(), hotkey.Id));
            
            // 複数回試行して確実に解除
            bool success = false;
            for (int retry = 0; retry < 3; retry++)
            {
                success = UnregisterHotKey(this.Handle, hotkey.Id);
                
                if (success)
                {
                    System.Diagnostics.Debug.WriteLine(string.Format("ホットキー解放成功: 試行{0}回目", retry + 1));
                    break;
                }
                System.Threading.Thread.Sleep(50);
            }
            
            if (!success)
            {
                // それでも失敗する場合は、全ホットキーを一旦解除して再登録
                System.Diagnostics.Debug.WriteLine("単独解放失敗 - 全ホットキー再登録処理を実行");
                return ReregisterAllHotkeysExcept(hotkey);
            }
            
            hotkey.IsRegistered = false;
            return true;
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine(string.Format("ホットキー解放エラー: {0}", ex.Message));
            return false;
        }
    }
    
    // ホットキーを確実に登録するメソッド
    private bool RegisterHotkeyReliable(HotkeyInfo hotkey)
    {
        try
        {
            if (hotkey.IsRegistered)
            {
                System.Diagnostics.Debug.WriteLine(string.Format("ホットキー '{0}' は既に登録済み", hotkey.GetHotkeyString()));
                return true;
            }

            // ハンドルが作成されていない場合は作成
            if (!this.IsHandleCreated)
            {
                this.CreateHandle();
                System.Diagnostics.Debug.WriteLine("ハンドル作成を実行");
            }

            System.Diagnostics.Debug.WriteLine(string.Format("ホットキー登録開始: '{0}' ID={1}", hotkey.GetHotkeyString(), hotkey.Id));
            
            // 念のため、登録前に一度解除を試みる
            UnregisterHotKey(this.Handle, hotkey.Id);
            System.Threading.Thread.Sleep(50);
            
            // 登録を試行
            bool success = RegisterHotKey(this.Handle, hotkey.Id, hotkey.GetModifiers(), (uint)hotkey.Key);
            
            if (!success)
            {
                uint errorCode = GetLastError();
                System.Diagnostics.Debug.WriteLine(string.Format("RegisterHotKey失敗 - エラーコード: {0} (0x{0:X})", errorCode));
                
                // エラーコード1409（既に登録済み）の場合は、新しいIDで再試行
                if (errorCode == 1409)
                {
                    System.Diagnostics.Debug.WriteLine("既に登録済みエラー - 新しいIDで再試行");
                    hotkey.Id = nextHotkeyId++;
                    success = RegisterHotKey(this.Handle, hotkey.Id, hotkey.GetModifiers(), (uint)hotkey.Key);
                }
            }
            
            if (success)
            {
                hotkey.IsRegistered = true;
                System.Diagnostics.Debug.WriteLine("ホットキー登録成功");
            }
            
            return success;
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine(string.Format("ホットキー登録エラー: {0}", ex.Message));
            return false;
        }
    }
    
    // 特定のホットキー以外を全て再登録するメソッド
    private bool ReregisterAllHotkeysExcept(HotkeyInfo excludeHotkey)
    {
        try
        {
            System.Diagnostics.Debug.WriteLine("全ホットキー再登録処理開始");
            
            // 現在登録されているホットキーを記憶
            List<HotkeyInfo> toReregister = new List<HotkeyInfo>();
            foreach (var hotkey in registeredHotkeys)
            {
                if (hotkey.IsRegistered && hotkey.Id != excludeHotkey.Id && !hotkey.IsPaused)
                {
                    toReregister.Add(hotkey);
                }
            }
            
            // 全て解除
            System.Diagnostics.Debug.WriteLine("全ホットキー解除");
            foreach (var hotkey in registeredHotkeys)
            {
                if (hotkey.IsRegistered)
                {
                    UnregisterHotKey(this.Handle, hotkey.Id);
                    hotkey.IsRegistered = false;
                }
            }
            
            System.Threading.Thread.Sleep(100);
            
            // 必要なものだけ再登録
            System.Diagnostics.Debug.WriteLine("必要なホットキーを再登録");
            foreach (var hotkey in toReregister)
            {
                bool success = RegisterHotKey(this.Handle, hotkey.Id, hotkey.GetModifiers(), (uint)hotkey.Key);
                hotkey.IsRegistered = success;
                System.Diagnostics.Debug.WriteLine(string.Format("再登録: {0} = {1}", hotkey.GetHotkeyString(), success));
            }
            
            System.Diagnostics.Debug.WriteLine("全ホットキー再登録処理完了");
            return true;
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine(string.Format("全ホットキー再登録エラー: {0}", ex.Message));
            return false;
        }
    }
    
    // 回避策：全ホットキー解除・再登録による強制一時停止
    private void RestoreModifierKeysSync()
    {
        try
        {
            if (savedModifierState == null)
            {
                System.Diagnostics.Debug.WriteLine("修飾キー復元: savedModifierState が null");
                return; // 復元する状態がない場合は何もしない
            }
            
            if (!savedModifierState.HasAnyModifier())
            {
                System.Diagnostics.Debug.WriteLine("修飾キー復元: 復元すべき修飾キーがない");
                savedModifierState = null;
                return; // 復元する状態がない場合は何もしない
            }

            // デバッグ出力：復元する状態
            System.Diagnostics.Debug.WriteLine(string.Format("修飾キー復元開始: LCtrl={0}, RCtrl={1}, LShift={2}, RShift={3}, LAlt={4}, RAlt={5}, LWin={6}, RWin={7}",
                savedModifierState.LeftCtrl, savedModifierState.RightCtrl, savedModifierState.LeftShift, savedModifierState.RightShift,
                savedModifierState.LeftAlt, savedModifierState.RightAlt, savedModifierState.LeftWin, savedModifierState.RightWin));

            // 復元開始前の待機時間を短縮（100ms → 20ms）
            Thread.Sleep(20);

            List<int> keysToRestore = new List<int>();

            // 保存された状態に基づいて修飾キーを一括復元（個別待機を削除）
            if (savedModifierState.LeftCtrl)
            {
                keybd_event((byte)VK_LCONTROL, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_LCONTROL);
            }
            if (savedModifierState.RightCtrl)
            {
                keybd_event((byte)VK_RCONTROL, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_RCONTROL);
            }
            if (savedModifierState.LeftShift)
            {
                keybd_event((byte)VK_LSHIFT, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_LSHIFT);
            }
            if (savedModifierState.RightShift)
            {
                keybd_event((byte)VK_RSHIFT, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_RSHIFT);
            }
            if (savedModifierState.LeftAlt)
            {
                keybd_event((byte)VK_LMENU, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_LMENU);
            }
            if (savedModifierState.RightAlt)
            {
                keybd_event((byte)VK_RMENU, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_RMENU);
            }
            if (savedModifierState.LeftWin)
            {
                keybd_event((byte)VK_LWIN, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_LWIN);
            }
            if (savedModifierState.RightWin)
            {
                keybd_event((byte)VK_RWIN, 0, 0, IntPtr.Zero); // 押下
                keysToRestore.Add(VK_RWIN);
            }

            System.Diagnostics.Debug.WriteLine(string.Format("復元したキー数: {0}", keysToRestore.Count));
            foreach (int key in keysToRestore)
            {
                System.Diagnostics.Debug.WriteLine(string.Format("キー復元: VK={0:X2}", key));
            }

            // 復元完了後の待機時間を短縮（50ms → 10ms）
            Thread.Sleep(10);
            
            // 復元完了後、保存状態をクリア
            savedModifierState = null;
            System.Diagnostics.Debug.WriteLine("修飾キー復元完了、状態をクリア");
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("修飾キー復元エラー: " + ex.Message);
        }
    }

    private void RecordButton_Click(object sender, EventArgs e)
    {
        if (isRecording)
        {
            // 記録停止：保持していたホットキーで停止
            if (rememberedHotkeyForRecording != null)
            {
                StopRecordingForHotkey(rememberedHotkeyForRecording);
                // 停止後に保持していたホットキーを再選択
                SelectHotkeyInList(rememberedHotkeyForRecording);
                rememberedHotkeyForRecording = null;
            }
        }
        else
        {
            // 記録開始：選択されたホットキーを保持して開始
            HotkeyInfo selectedHotkey = GetSelectedHotkey();
            if (selectedHotkey == null)
            {
                MessageBox.Show("記録するホットキーをリストから選択してください。", "選択エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }

            // 記録開始時にホットキーを保持
            rememberedHotkeyForRecording = selectedHotkey;
            StartRecordingForHotkey(selectedHotkey);
        }
    }

    private void ClearRecordButton_Click(object sender, EventArgs e)
    {
        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey == null)
        {
            MessageBox.Show("クリアするホットキーをリストから選択してください。", "選択エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        if (MessageBox.Show(string.Format("ホットキー '{0}' の記録をクリアしますか？", 
            selectedHotkey.GetHotkeyString()), "確認", 
            MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
        {
            selectedHotkey.RecordedActions.Clear();
            
            // クリア後、即座にUI更新と状態反映を強制実行
            RefreshHotkeyList();
            DisplayActionsForHotkey(selectedHotkey);
            
            // 選択状態を維持
            SelectHotkeyInList(selectedHotkey);
            
            statusLabel.Text = string.Format("ホットキー '{0}' の記録をクリアしました", selectedHotkey.GetHotkeyString());
            statusLabel.ForeColor = Color.Orange;
            
            // アプリケーションの更新を強制的に処理
            Application.DoEvents();
        }
    }

    private void RefreshHotkeyList()
    {
        if (hotkeyListView == null || hotkeyListView.IsDisposed)
            return;
            
        // 現在選択されているホットキーIDを保存
        int selectedHotkeyId = -1;
        if (hotkeyListView.SelectedItems.Count > 0)
        {
            int.TryParse(hotkeyListView.SelectedItems[0].SubItems[0].Text, out selectedHotkeyId);
        }

        hotkeyListView.Items.Clear();
        
        if (registeredHotkeys == null)
            return;
        
        foreach (var hotkey in registeredHotkeys)
        {
            if (hotkey == null)
                continue;
            ListViewItem item = new ListViewItem(hotkey.Id.ToString());
            item.SubItems.Add(hotkey.GetHotkeyString());
            item.SubItems.Add(hotkey.Description);
            item.SubItems.Add(hotkey.RecordedActions.Count.ToString());
            item.SubItems.Add(hotkey.LoopCount.ToString()); // ループ回数を追加
            item.SubItems.Add(hotkey.ContinuousExecution ? "ON" : "OFF");
            item.SubItems.Add(hotkey.FastExecution ? "ON" : "OFF");
            item.SubItems.Add(hotkey.IsPaused ? "停止" : "実行");
            
            string status = "";
            if (hotkey.IsRecording)
            {
                status = "記録中";
                item.BackColor = Color.FromArgb(255, 245, 230); // 薄いオレンジ
            }
            else if (hotkey.IsPaused)
            {
                status = "一時停止";
                item.BackColor = Color.FromArgb(245, 245, 245); // ライトグレー
            }
            else if (hotkey.RecordedActions.Count > 0)
            {
                status = "記録済み";
                item.BackColor = Color.FromArgb(240, 248, 255); // 薄い青
            }
            else
            {
                status = "未記録";
                item.BackColor = Color.White;
            }
            item.SubItems.Add(status);
            
            hotkeyListView.Items.Add(item);

            // 以前選択されていたホットキーを再選択
            if (hotkey.Id == selectedHotkeyId)
            {
                item.Selected = true;
                item.EnsureVisible();
                // 選択された行に強調色を設定（非アクティブ時も視認性向上）
                item.ForeColor = Color.Black;
                item.Font = new Font(item.Font, FontStyle.Bold);
            }
            else
            {
                // 選択されていない行は通常フォントに
                item.Font = hotkeyListView.Font;
                item.ForeColor = Color.Black;
            }
        }
    }

    // ホットキーリストのダブルクリックイベント（連打実行・高速実行・一時停止フラグ・ループ回数切り替え）
    private void HotkeyListView_DoubleClick(object sender, EventArgs e)
    {
        if (hotkeyListView.SelectedItems.Count == 0)
            return;

        // ダブルクリックされた位置を確認
        Point clickPoint = hotkeyListView.PointToClient(Control.MousePosition);
        ListViewHitTestInfo hitTest = hotkeyListView.HitTest(clickPoint);
        
        if (hitTest.SubItem != null)
        {
            HotkeyInfo selectedHotkey = GetSelectedHotkey();
            if (selectedHotkey != null)
            {
                int columnIndex = hitTest.Item.SubItems.IndexOf(hitTest.SubItem);
                
                // ループ列がクリックされた場合
                if (columnIndex == (int)HotkeyListColumn.LoopCount)
                {
                    // ループ回数を編集
                    string input = PromptForInput("ループ回数を入力してください (1-999):", selectedHotkey.LoopCount.ToString());
                    if (!string.IsNullOrEmpty(input))
                    {
                        int newLoopCount;
                        if (int.TryParse(input, out newLoopCount) && newLoopCount >= 1 && newLoopCount <= 999)
                        {
                            selectedHotkey.LoopCount = newLoopCount;
                            RefreshHotkeyList();
                            SelectHotkeyInList(selectedHotkey);
                            statusLabel.Text = string.Format("ホットキー '{0}' のループ回数を {1}回 に変更しました", 
                                selectedHotkey.GetHotkeyString(), newLoopCount);
                            statusLabel.ForeColor = Color.Green;
                        }
                        else
                        {
                            MessageBox.Show("1から999の間の数値を入力してください。", "入力エラー", 
                                MessageBoxButtons.OK, MessageBoxIcon.Warning);
                        }
                    }
                }
                // 連打実行列がクリックされた場合
                else if (columnIndex == (int)HotkeyListColumn.ContinuousExecution)
                {
                    // 連打実行フラグを切り替え
                    selectedHotkey.ContinuousExecution = !selectedHotkey.ContinuousExecution;
                    RefreshHotkeyList();
                    
                    // 切り替え後も選択状態を維持
                    SelectHotkeyInList(selectedHotkey);
                    
                    statusLabel.Text = string.Format("ホットキー '{0}' の連打実行を {1} にしました", 
                        selectedHotkey.GetHotkeyString(), selectedHotkey.ContinuousExecution ? "ON" : "OFF");
                    statusLabel.ForeColor = Color.Green;
                }
                // 高速実行列がクリックされた場合
                else if (columnIndex == (int)HotkeyListColumn.FastExecution)
                {
                    // 高速実行フラグを切り替え
                    selectedHotkey.FastExecution = !selectedHotkey.FastExecution;
                    RefreshHotkeyList();
                    
                    // 切り替え後も選択状態を維持
                    SelectHotkeyInList(selectedHotkey);
                    
                    statusLabel.Text = string.Format("ホットキー '{0}' の高速実行を {1} にしました", 
                        selectedHotkey.GetHotkeyString(), selectedHotkey.FastExecution ? "ON" : "OFF");
                    statusLabel.ForeColor = Color.Green;
                }
                // 一時停止列がクリックされた場合
                else if (columnIndex == (int)HotkeyListColumn.IsPaused)
                {
                    // 一時停止フラグを切り替え
                    bool newPausedState = !selectedHotkey.IsPaused;
                    
                    System.Diagnostics.Debug.WriteLine(string.Format("一時停止切り替え開始: ホットキー='{0}', 現在の状態=IsPaused:{1}, IsRegistered:{2}, 新しい状態={3}", 
                        selectedHotkey.GetHotkeyString(), selectedHotkey.IsPaused, selectedHotkey.IsRegistered, newPausedState ? "一時停止" : "実行可能"));
                    
                    try
                    {
                        bool success = false;
                        
                        if (newPausedState)
                        {
                            // 一時停止：ホットキーの登録を解除
                            System.Diagnostics.Debug.WriteLine("一時停止処理開始");

                            success = UnregisterHotkeyReliable(selectedHotkey);
                            
                            if (success)
                            {
                                selectedHotkey.IsPaused = true;
                                statusLabel.Text = string.Format("ホットキー '{0}' を一時停止しました", selectedHotkey.GetHotkeyString());
                                statusLabel.ForeColor = Color.Orange;
                                System.Diagnostics.Debug.WriteLine("一時停止処理完了");
                            }
                            else
                            {
                                MessageBox.Show(string.Format("ホットキー '{0}' の一時停止に失敗しました。", selectedHotkey.GetHotkeyString()), 
                                    "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
                            }
                        }
                        else
                        {
                            // 再開：ホットキーを再登録
                            System.Diagnostics.Debug.WriteLine("再開処理開始");
                            success = RegisterHotkeyReliable(selectedHotkey);
                            
                            if (success)
                            {
                                selectedHotkey.IsPaused = false;
                                statusLabel.Text = string.Format("ホットキー '{0}' の一時停止を解除しました", selectedHotkey.GetHotkeyString());
                                statusLabel.ForeColor = Color.Green;
                                System.Diagnostics.Debug.WriteLine("再開処理完了");
                            }
                            else
                            {
                                MessageBox.Show(string.Format("ホットキー '{0}' の再開に失敗しました。\n他のアプリケーションが同じキーを使用している可能性があります。", 
                                    selectedHotkey.GetHotkeyString()), 
                                    "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
                            }
                        }
                        
                        if (success)
                        {
                            RefreshHotkeyList();
                            // 切り替え後も選択状態を維持
                            SelectHotkeyInList(selectedHotkey);
                        }
                        
                        System.Diagnostics.Debug.WriteLine(string.Format("一時停止切り替え完了: ホットキー='{0}', 最終状態=IsPaused:{1}, IsRegistered:{2}", 
                            selectedHotkey.GetHotkeyString(), selectedHotkey.IsPaused, selectedHotkey.IsRegistered));
                    }
                    catch (Exception ex)
                    {
                        System.Diagnostics.Debug.WriteLine(string.Format("一時停止切り替えで例外発生: {0}", ex.Message));
                        MessageBox.Show(string.Format("一時停止切り替えエラー: {0}", ex.Message), "エラー", 
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }
    }

    private void HotkeyListView_SelectedIndexChanged(object sender, EventArgs e)
    {
        // 選択状態更新中の場合は処理をスキップ（無限ループ防止）
        if (isUpdatingSelection)
            return;

        // 記録中は記録中のホットキーの選択を維持
        if (isRecording && currentRecordingHotkey != null)
        {
            HotkeyInfo currentSelected = GetSelectedHotkey();
            if (currentSelected == null || currentSelected.Id != currentRecordingHotkey.Id)
            {
                // 記録中のホットキーを強制的に選択状態に戻す
                SelectHotkeyInList(currentRecordingHotkey);
            }
            return;
        }

        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey != null)
        {
            DisplayActionsForHotkey(selectedHotkey);
            
            // 記録ボタンの状態を更新
            if (recordButton != null && !recordButton.IsDisposed)
            {
                if (selectedHotkey.IsRecording)
                {
                    recordButton.Text = "記録停止";
                }
                else
                {
                    recordButton.Text = "記録開始";
                }
            }
            
            // 選択されたホットキーの情報を入力フィールドに表示（変更時に便利）
            ctrlCheckBox.Checked = selectedHotkey.UseCtrl;
            shiftCheckBox.Checked = selectedHotkey.UseShift;
            altCheckBox.Checked = selectedHotkey.UseAlt;
            keyTextBox.Text = selectedHotkey.Key.ToString();
            descriptionTextBox.Text = selectedHotkey.Description;
            
            // 選択状態を視覚的に強調
            HighlightSelectedHotkey();
        }
        else
        {
            // 選択がない場合
            if (actionListView != null && !actionListView.IsDisposed)
                actionListView.Items.Clear();
            if (recordButton != null && !recordButton.IsDisposed)
                recordButton.Text = "記録開始";
            
            // 選択が外れたときに全ての行を通常フォントに戻す
            if (hotkeyListView != null && !hotkeyListView.IsDisposed)
            {
                foreach (ListViewItem item in hotkeyListView.Items)
                {
                    item.Font = hotkeyListView.Font;
                    item.ForeColor = Color.Black;
                }
            }
        }
    }

    private void DisplayActionsForHotkey(HotkeyInfo hotkey)
    {
        if (hotkey == null || actionListView == null || actionListView.IsDisposed)
            return;
            
        actionListView.Items.Clear();
        
        for (int i = 0; i < hotkey.RecordedActions.Count; i++)
        {
            var action = hotkey.RecordedActions[i];
            if (action == null)
                continue;
            
            ListViewItem item = new ListViewItem((i + 1).ToString());
            item.SubItems.Add(action.Type);
            
            string coordKey = "";
            if (action.Type.Contains("マウス"))
            {
                coordKey = string.Format("({0}, {1})", action.X, action.Y);
            }
            else if (action.Type.Contains("キーボード"))
            {
                coordKey = string.Format("{0}{1}", action.Modifiers, action.KeyName);
            }
            item.SubItems.Add(coordKey);
            
            item.SubItems.Add(action.DelayMs.ToString());
            item.SubItems.Add(action.Details);
            item.SubItems.Add(action.Timestamp.ToString("HH:mm:ss.fff"));
            
            // 記録中の場合、最新のアクションをハイライト
            if (isRecording && currentRecordingHotkey != null && 
                currentRecordingHotkey.Id == hotkey.Id && i == hotkey.RecordedActions.Count - 1)
            {
                item.BackColor = Color.LightYellow;
            }
            else
            {
                item.BackColor = Color.White;
            }
            
            actionListView.Items.Add(item);
        }
        
        if (!isRecording)
        {
            statusLabel.Text = string.Format("ホットキー '{0}' の動作履歴: {1}個のアクション", 
                hotkey.GetHotkeyString(), hotkey.RecordedActions.Count);
            statusLabel.ForeColor = Color.Blue;
        }
    }

    // Escキー専用のキーボードフックプロシージャ
    private IntPtr EscKeyHookProc(int nCode, IntPtr wParam, IntPtr lParam)
    {
        if (nCode >= 0)
        {
            KBDLLHOOKSTRUCT hookStruct = (KBDLLHOOKSTRUCT)Marshal.PtrToStructure(lParam, typeof(KBDLLHOOKSTRUCT));
            
            // Escキーが押された時
            if ((int)wParam == WM_KEYDOWN && hookStruct.vkCode == 0x1B) // VK_ESCAPE = 0x1B
            {
                // 再生中または記録中の場合は強制停止
                if (isPlaying || isRecording)
                {
                    System.Diagnostics.Debug.WriteLine("Escキー検出 - 強制停止実行");
                    ForceStopExecution();
                    // Escキーを他のアプリケーションに渡さない（イベントを食い止める）
                    return (IntPtr)1;
                }
            }
        }
        
        // CallNextHookEx(escKeyHookID, nCode, wParam, lParam);
        return (IntPtr)0;
    }

    protected override void WndProc(ref Message m)
    {
        if (m.Msg == WM_HOTKEY)
        {
            int hotkeyId = m.WParam.ToInt32();
            
            // ウィンドウ復帰用のホットキー（Ctrl+Alt+H）の処理
            if (hotkeyId == SHOW_WINDOW_HOTKEY_ID)
            {
                // ウィンドウを最前面に表示
                this.WindowState = FormWindowState.Normal;
                this.TopMost = true;
                this.Activate();
                this.TopMost = false;
                return;
            }
            
            // 通常のホットキー処理
            HotkeyInfo hotkey = registeredHotkeys.Find(h => h.Id == hotkeyId);
            if (hotkey != null)
            {
                ExecuteHotkeyAction(hotkey);
            }
        }
        base.WndProc(ref m);
    }

    // 強制停止処理（修正：記録停止後にホットキーを再選択）
    private void ForceStopExecution()
    {
        System.Diagnostics.Debug.WriteLine(string.Format("ForceStopExecution: isPlaying={0}, isRecording={1}", isPlaying, isRecording));
        
        if (isPlaying)
        {
            forceStopRequested = true;
            isPlaying = false;
            
            if (!this.IsDisposed)
            {
                this.BeginInvoke(new Action(() => {
                    if (this.IsDisposed)
                        return;
                        
                    statusLabel.Text = "再生を強制停止しました (Esc)";
                    statusLabel.ForeColor = Color.Red;
                    
                    // アクションリストの背景色をリセット
                    if (actionListView != null && !actionListView.IsDisposed)
                    {
                        foreach (ListViewItem item in actionListView.Items)
                        {
                            item.BackColor = Color.White;
                            item.Selected = false;
                        }
                    }
                    
                    // ボタンの状態を元に戻す
                    recordButton.Enabled = true;
                    addHotkeyButton.Enabled = true;
                    changeHotkeyButton.Enabled = true;
                    removeHotkeyButton.Enabled = true;
                    clearRecordButton.Enabled = true;
                    loadButton.Enabled = true;
                    saveButton.Enabled = true;
                    hotkeyListView.Enabled = true;
                    
                    System.Diagnostics.Debug.WriteLine("再生強制停止完了");
                }));
            }
        }
        else if (isRecording && currentRecordingHotkey != null)
        {
            // 記録中の場合は記録を停止
            var hotkeyToStop = currentRecordingHotkey;
            if (!this.IsDisposed)
            {
                this.BeginInvoke(new Action(() => {
                    if (this.IsDisposed)
                        return;
                        
                    StopRecordingForHotkey(hotkeyToStop);
                    
                    // 記録停止後にホットキーを再選択（選択が外れないように）
                    SelectHotkeyInList(hotkeyToStop);
                    
                    statusLabel.Text = "記録を強制停止しました (Esc)";
                    statusLabel.ForeColor = Color.Red;
                    
                    System.Diagnostics.Debug.WriteLine("記録強制停止完了");
                }));
            }
        }
    }

    private void ExecuteHotkeyAction(HotkeyInfo hotkey)
    {
        if (hotkey == null)
            return;
            
        this.BeginInvoke(new Action(async () => {
            // フォームが破棄されている場合は何もしない
            if (this.IsDisposed)
                return;
                
            // 一時停止中の場合は何もしない
            if (hotkey.IsPaused)
            {
                statusLabel.Text = string.Format("ホットキー '{0}' は一時停止中です", hotkey.GetHotkeyString());
                statusLabel.ForeColor = Color.Orange;
                return;
            }
            
            // 再生中の場合は現在の再生を中断して新しい操作を開始（連打対応）
            if (isPlaying)
            {
                forceStopRequested = true; // 現在の再生を停止要求
                isPlaying = false; // 現在の再生を中断
                // 少し待って前の処理を確実に停止
                await Task.Delay(50);
                forceStopRequested = false; // フラグをリセット
            }
            
            bool shouldRestoreKeys = false;
            
            try
            {
                // 修飾キーを使用しているホットキーの場合のみ、修飾キーの状態を保存して解放
                bool hasModifiers = hotkey.UseCtrl || hotkey.UseShift || hotkey.UseAlt;
                if (hasModifiers)
                {
                    await SaveAndReleaseAllModifierKeys(hotkey.FastExecution);
                    shouldRestoreKeys = true;
                    System.Diagnostics.Debug.WriteLine("ホットキー押下時に修飾キーを解放");
                }
                else
                {
                    System.Diagnostics.Debug.WriteLine("修飾キーなしのホットキーのため、修飾キー解放をスキップ");
                }
                
                if (hotkey.IsRecording)
                {
                    // 記録中の場合は記録を停止
                    StopRecordingForHotkey(hotkey);
                    // 停止後にそのホットキーを再選択
                    SelectHotkeyInList(hotkey);
                }
                else if (hotkey.RecordedActions.Count > 0)
                {
                    // 記録されたアクションがある場合は再生（ループ対応）
                    await PlayActionsForHotkey(hotkey);
                }
                else
                {
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(string.Format("ホットキー実行エラー: {0}", ex.Message), "エラー", 
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                // エラー時にフラグをリセット
                isPlaying = false;
            }
            finally
            {
                // 修飾キーの復元処理（連打実行がONの場合のみ）
                if (shouldRestoreKeys && hotkey.ContinuousExecution)
                {
                    try
                    {
                        System.Diagnostics.Debug.WriteLine(string.Format("ホットキー '{0}' の連打実行がON - 修飾キー復元を実行", hotkey.GetHotkeyString()));
                        RestoreModifierKeysSync();
                    }
                    catch (Exception restoreEx)
                    {
                        System.Diagnostics.Debug.WriteLine("修飾キー復元エラー: " + restoreEx.Message);
                    }
                }
                else
                {
                    if (!shouldRestoreKeys)
                    {
                        System.Diagnostics.Debug.WriteLine("shouldRestoreKeys が false - 修飾キー復元をスキップ");
                    }
                    else if (!hotkey.ContinuousExecution)
                    {
                        System.Diagnostics.Debug.WriteLine(string.Format("ホットキー '{0}' の連打実行がOFF - 修飾キー復元をスキップ", hotkey.GetHotkeyString()));
                        // 連打実行がOFFの場合は保存状態をクリア
                        savedModifierState = null;
                    }
                }
            }
        }));
    }

    private void StartRecordingForHotkey(HotkeyInfo hotkey)
    {
        try
        {
            if (hotkey == null)
                return;
                
            // 既に記録がある場合は上書き確認
            if (hotkey.RecordedActions != null && hotkey.RecordedActions.Count > 0)
            {
                DialogResult result = MessageBox.Show(
                    string.Format("ホットキー '{0}' には既に {1}個の動作が記録されています。\n上書きして新しく記録しますか？", 
                        hotkey.GetHotkeyString(), hotkey.RecordedActions.Count),
                    "上書き確認",
                    MessageBoxButtons.YesNo,
                    MessageBoxIcon.Question);
                
                if (result != DialogResult.Yes)
                {
                    return; // 記録開始をキャンセル
                }
            }

            // 他のホットキーの記録を停止
            foreach (var h in registeredHotkeys)
            {
                if (h.IsRecording && h.Id != hotkey.Id)
                {
                    StopRecordingForHotkey(h);
                }
            }

            // このホットキー専用の記録リストをクリア
            hotkey.RecordedActions.Clear();
            
            // クリア後、即座にUI更新と状態反映を強制実行
            RefreshHotkeyList();
            DisplayActionsForHotkey(hotkey);
            
            lastEventTime = DateTime.MinValue;
            keyStates.Clear();
            currentRecordingHotkey = hotkey;

            if (mouseEventsCheckBox.Checked)
            {
                mouseHookID = SetHook(mouseProc, WH_MOUSE_LL);
            }
            if (keyboardEventsCheckBox.Checked)
            {
                keyboardHookID = SetHook(keyboardProc, WH_KEYBOARD_LL);
            }
            
            isRecording = true;
            hotkey.IsRecording = true;
            
            // 記録中のホットキーを選択状態にして維持
            SelectHotkeyInList(hotkey);
            
            // 記録ボタンの表示を即座に変更
            if (recordButton != null && !recordButton.IsDisposed)
                recordButton.Text = "記録停止";
            
            if (statusLabel != null && !statusLabel.IsDisposed)
            {
                statusLabel.Text = string.Format("記録開始: {0} (再度押下またはEscキーで停止)", hotkey.GetHotkeyString());
            }
            
            RefreshHotkeyList();
            DisplayActionsForHotkey(hotkey);
            
            // 記録中はコントロールを無効化（ホットキーリストは有効のまま）
            if (!this.IsDisposed)
            {
                if (mouseEventsCheckBox != null && !mouseEventsCheckBox.IsDisposed)
                    mouseEventsCheckBox.Enabled = false;
                if (keyboardEventsCheckBox != null && !keyboardEventsCheckBox.IsDisposed)
                    keyboardEventsCheckBox.Enabled = false;
                if (mouseMoveCheckBox != null && !mouseMoveCheckBox.IsDisposed)
                    mouseMoveCheckBox.Enabled = false;
                if (addHotkeyButton != null && !addHotkeyButton.IsDisposed)
                    addHotkeyButton.Enabled = false;
                if (changeHotkeyButton != null && !changeHotkeyButton.IsDisposed)
                    changeHotkeyButton.Enabled = false;
                if (removeHotkeyButton != null && !removeHotkeyButton.IsDisposed)
                    removeHotkeyButton.Enabled = false;
                if (loadButton != null && !loadButton.IsDisposed)
                    loadButton.Enabled = false;
                
                // アプリケーションの更新を強制的に処理
                Application.DoEvents();
            }
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("記録開始エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }

    private void StopRecordingForHotkey(HotkeyInfo hotkey)
    {
        try
        {
            if (mouseHookID != IntPtr.Zero)
            {
                UnhookWindowsHookEx(mouseHookID);
                mouseHookID = IntPtr.Zero;
            }
            if (keyboardHookID != IntPtr.Zero)
            {
                UnhookWindowsHookEx(keyboardHookID);
                keyboardHookID = IntPtr.Zero;
            }
            
            isRecording = false;
            if (hotkey != null)
            {
                hotkey.IsRecording = false;
            }
            currentRecordingHotkey = null;
            
            // 記録ボタンの表示を即座に変更
            if (recordButton != null && !recordButton.IsDisposed)
                recordButton.Text = "記録開始";
            
            if (statusLabel != null && !statusLabel.IsDisposed && hotkey != null)
            {
                statusLabel.Text = string.Format("記録完了: {0} ({1}個のアクション) - ホットキー押下で再生", 
                    hotkey.GetHotkeyString(), hotkey.RecordedActions != null ? hotkey.RecordedActions.Count : 0);
            }
            
            RefreshHotkeyList();
            
            // 記録停止後に確実にホットキーを選択状態にする
            if (hotkey != null)
            {
                SelectHotkeyInList(hotkey);
            }
            else if (rememberedHotkeyForRecording != null)
            {
                SelectHotkeyInList(rememberedHotkeyForRecording);
            }
            
            // コントロールを再有効化
            if (!this.IsDisposed)
            {
                if (mouseEventsCheckBox != null && !mouseEventsCheckBox.IsDisposed)
                    mouseEventsCheckBox.Enabled = true;
                if (keyboardEventsCheckBox != null && !keyboardEventsCheckBox.IsDisposed)
                    keyboardEventsCheckBox.Enabled = true;
                if (mouseMoveCheckBox != null && !mouseMoveCheckBox.IsDisposed)
                    mouseMoveCheckBox.Enabled = true;
                if (addHotkeyButton != null && !addHotkeyButton.IsDisposed)
                    addHotkeyButton.Enabled = true;
                if (changeHotkeyButton != null && !changeHotkeyButton.IsDisposed)
                    changeHotkeyButton.Enabled = true;
                if (removeHotkeyButton != null && !removeHotkeyButton.IsDisposed)
                    removeHotkeyButton.Enabled = true;
                if (loadButton != null && !loadButton.IsDisposed)
                    loadButton.Enabled = true;
            }
            
            // アプリケーションの更新を強制的に処理
            Application.DoEvents();
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("記録停止エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }

    // 選択状態を視覚的に強調するヘルパーメソッド
    private void HighlightSelectedHotkey()
    {
        if (hotkeyListView == null || hotkeyListView.IsDisposed)
            return;
        
        foreach (ListViewItem item in hotkeyListView.Items)
        {
            if (item.Selected)
            {
                // 選択された行を太字にして強調
                item.Font = new Font(hotkeyListView.Font, FontStyle.Bold);
                item.ForeColor = Color.Black;
            }
            else
            {
                // 非選択行は通常のフォント
                item.Font = hotkeyListView.Font;
                item.ForeColor = Color.Black;
            }
        }
    }

    private async Task PlayActionsForHotkey(HotkeyInfo hotkey)
    {
        if (hotkey == null || hotkey.RecordedActions == null || hotkey.RecordedActions.Count == 0)
        {
            if (statusLabel != null && !statusLabel.IsDisposed)
            {
                statusLabel.Text = string.Format("ホットキー '{0}' に記録されたアクションがありません", 
                    hotkey != null ? hotkey.GetHotkeyString() : "null");
                statusLabel.ForeColor = Color.Orange;
            }
            return;
        }

        if (isRecording)
        {
            statusLabel.Text = "記録中は再生できません";
            statusLabel.ForeColor = Color.Red;
            return;
        }

        try
        {
            isPlaying = true;
            forceStopRequested = false; // 強制停止フラグをリセット
            
            // UI更新：再生中は他のボタンを無効化
            if (!this.IsDisposed)
            {
                if (recordButton != null && !recordButton.IsDisposed)
                    recordButton.Enabled = false;
                if (addHotkeyButton != null && !addHotkeyButton.IsDisposed)
                    addHotkeyButton.Enabled = false;
                if (changeHotkeyButton != null && !changeHotkeyButton.IsDisposed)
                    changeHotkeyButton.Enabled = false;
                if (removeHotkeyButton != null && !removeHotkeyButton.IsDisposed)
                    removeHotkeyButton.Enabled = false;
                if (clearRecordButton != null && !clearRecordButton.IsDisposed)
                    clearRecordButton.Enabled = false;
                if (loadButton != null && !loadButton.IsDisposed)
                    loadButton.Enabled = false;
                if (saveButton != null && !saveButton.IsDisposed)
                    saveButton.Enabled = false;
                if (hotkeyListView != null && !hotkeyListView.IsDisposed)
                    hotkeyListView.Enabled = false;
                
                // 初回のステータス表示のみ
                if (statusLabel != null && !statusLabel.IsDisposed)
                {
                    if (hotkey.FastExecution)
                    {
                        statusLabel.Text = string.Format("高速再生中: {0} (Escキーで停止可能)", hotkey.GetHotkeyString());
                    }
                    else
                    {
                        statusLabel.Text = string.Format("再生中: {0} (Escキーで停止可能)", hotkey.GetHotkeyString());
                    }
                }
                
                // UIの更新を即座に反映（初回のみ）
                Application.DoEvents();
            }
            System.Diagnostics.Debug.WriteLine("PlayActionsForHotkey: 再生開始");
            
            // キーボードイベントが含まれている場合は、ホットキーの修飾キーが完全に解放されるまで待機
            bool hasKeyboardEvents = hotkey.RecordedActions.Any(a => a.Type.Contains("キーボード"));
            if (hasKeyboardEvents && !hotkey.FastExecution)
            {
                System.Diagnostics.Debug.WriteLine("キーボードイベント再生のため、修飾キー解放待機");
                await Task.Delay(100); // 通常モードではホットキーの修飾キーが完全に解放されるまで待機
            }
            else if (hasKeyboardEvents && hotkey.FastExecution)
            {
                System.Diagnostics.Debug.WriteLine("高速実行モード: キーボードイベント前の待機をスキップ");
                // 高速実行モードでは待機をスキップ
            }
            
            // ループ回数分実行
            for (int loopIndex = 0; loopIndex < hotkey.LoopCount && isPlaying && !forceStopRequested; loopIndex++)
            {
                // **修正：ループ情報表示を削除（速度優先）**
                
                for (int i = 0; i < hotkey.RecordedActions.Count && isPlaying && !forceStopRequested; i++)
                {
                    var action = hotkey.RecordedActions[i];
                    
                    // **修正：待機時間処理を先に実行**
                    if (hotkey.FastExecution)
                    {
                        // 高速実行フラグがONの場合、待ち時間を0にする
                        await Task.Delay(1);
                    }
                    else
                    {
                        // **修正：通常実行では記録された待機時間をそのまま使用**
                        if (action.DelayMs > 0)
                        {
                            int delayMs = action.DelayMs;
                            
                            // **修正：200ms以下なら一括待機、それ以上は100ms間隔で分割（Esc検知用）**
                            if (delayMs <= 200)
                            {
                                await Task.Delay(delayMs);
                            }
                            else
                            {
                                // 長い待機時間の場合のみ、強制停止チェックのため分割待機
                                int elapsed = 0;
                                while (elapsed < delayMs && !forceStopRequested && isPlaying)
                                {
                                    int waitTime = Math.Min(100, delayMs - elapsed);
                                    await Task.Delay(waitTime);
                                    elapsed += waitTime;
                                    
                                    // **修正：100msごとにEscキーチェックのみ実行**
                                    if (elapsed % 100 == 0)
                                    {
                                        Application.DoEvents();
                                    }
                                }
                            }
                            
                            if (forceStopRequested || !isPlaying) break;
                        }
                    }

                    // 強制停止チェック
                    if (forceStopRequested || !isPlaying)
                    {
                        System.Diagnostics.Debug.WriteLine("再生を強制停止");
                        break;
                    }

                    // アクション実行
                    ExecuteRecordedAction(action);
                    
                    // **修正：UI更新を完全に削除（再生中はハイライトなし）**
                }

                // ループ間の待機（高速実行でない場合のみ）
                if (loopIndex < hotkey.LoopCount - 1 && !forceStopRequested && isPlaying && !hotkey.FastExecution)
                {
                    await Task.Delay(50); // ループ間の短い待機
                }
            }

            if (forceStopRequested)
            {
                if (statusLabel != null && !statusLabel.IsDisposed)
                {
                    statusLabel.Text = string.Format("再生を停止しました: {0}", hotkey.GetHotkeyString());
                    statusLabel.ForeColor = Color.Red;
                }
            }
            else
            {
                if (statusLabel != null && !statusLabel.IsDisposed)
                {
                    statusLabel.Text = string.Format("再生完了: {0} ({1}回ループ)", hotkey.GetHotkeyString(), hotkey.LoopCount);
                    statusLabel.ForeColor = Color.Blue;
                }
            }
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("再生エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
            if (statusLabel != null && !statusLabel.IsDisposed)
            {
                statusLabel.Text = "再生エラー";
                statusLabel.ForeColor = Color.Red;
            }
        }
        finally
        {
            isPlaying = false;
            forceStopRequested = false;
            
            // UI更新：再生終了後に全ての背景色を元に戻し、選択状態をクリア
            if (!this.IsDisposed && actionListView != null && !actionListView.IsDisposed)
            {
                foreach (ListViewItem item in actionListView.Items)
                {
                    item.BackColor = Color.White;
                    item.Selected = false;
                }
            }
            
            // UI更新：再生終了後にボタンの状態を元に戻す
            if (!this.IsDisposed)
            {
                if (recordButton != null && !recordButton.IsDisposed)
                    recordButton.Enabled = true;
                if (addHotkeyButton != null && !addHotkeyButton.IsDisposed)
                    addHotkeyButton.Enabled = true;
                if (changeHotkeyButton != null && !changeHotkeyButton.IsDisposed)
                    changeHotkeyButton.Enabled = true;
                if (removeHotkeyButton != null && !removeHotkeyButton.IsDisposed)
                    removeHotkeyButton.Enabled = true;
                if (clearRecordButton != null && !clearRecordButton.IsDisposed)
                    clearRecordButton.Enabled = true;
                if (loadButton != null && !loadButton.IsDisposed)
                    loadButton.Enabled = true;
                if (saveButton != null && !saveButton.IsDisposed)
                    saveButton.Enabled = true;
                if (hotkeyListView != null && !hotkeyListView.IsDisposed)
                    hotkeyListView.Enabled = true;
                
                Application.DoEvents();
            }
            System.Diagnostics.Debug.WriteLine("PlayActionsForHotkey: 再生終了");
        }
    }

    private void ExecuteRecordedAction(RecordedAction action)
    {
        try
        {
            if (action == null)
                return;
                
            if (action.Type.Contains("マウス"))
            {
                switch (action.Type)
                {
                    case "マウス: 左クリック押下":
                        SetCursorPos(action.X, action.Y);
                        Thread.Sleep(5); // 座標移動の確実な反映のため待機
                        mouse_event(MOUSEEVENTF_LEFTDOWN, 0, 0, 0, IntPtr.Zero);
                        break;
                    case "マウス: 左クリック解放":
                        mouse_event(MOUSEEVENTF_LEFTUP, 0, 0, 0, IntPtr.Zero);
                        break;
                    case "マウス: 右クリック押下":
                        SetCursorPos(action.X, action.Y);
                        Thread.Sleep(5); // 座標移動の確実な反映のため待機
                        mouse_event(MOUSEEVENTF_RIGHTDOWN, 0, 0, 0, IntPtr.Zero);
                        break;
                    case "マウス: 右クリック解放":
                        mouse_event(MOUSEEVENTF_RIGHTUP, 0, 0, 0, IntPtr.Zero);
                        break;
                    case "マウス: 中クリック押下":
                        SetCursorPos(action.X, action.Y);
                        Thread.Sleep(5); // 座標移動の確実な反映のため待機
                        mouse_event(MOUSEEVENTF_MIDDLEDOWN, 0, 0, 0, IntPtr.Zero);
                        break;
                    case "マウス: 中クリック解放":
                        mouse_event(MOUSEEVENTF_MIDDLEUP, 0, 0, 0, IntPtr.Zero);
                        break;
                    case "マウス: 移動":
                        SetCursorPos(action.X, action.Y);
                        break;
                    case "マウス: ホイール":
                        SetCursorPos(action.X, action.Y);
                        Thread.Sleep(5); // 座標移動の確実な反映のため待機
                        int wheelDelta = action.Details == "上" ? 120 : -120;
                        mouse_event(MOUSEEVENTF_WHEEL, 0, 0, (uint)wheelDelta, IntPtr.Zero);
                        break;
                }
            }
            else if (action.Type.Contains("キーボード"))
            {
                byte vkCode = (byte)action.VirtualKeyCode;
                
                // 自身が修飾キーかどうかチェック
                bool isSelfModifierKey = IsModifierKey(action.VirtualKeyCode);
                
                if (action.Type.Contains("押下"))
                {
                    keybd_event(vkCode, 0, 0, IntPtr.Zero);
                    // 修飾キーの押下後は状態が安定するまで少し待機
                    if (isSelfModifierKey)
                    {
                        Thread.Sleep(30); // 修飾キーの状態を確実に反映（少し長めに）
                        System.Diagnostics.Debug.WriteLine(string.Format("修飾キー押下: VK={0:X2} ({1})", vkCode, action.KeyName));
                    }
                    else if (!string.IsNullOrEmpty(action.Modifiers))
                    {
                        // 修飾キー付きの通常キー
                        System.Diagnostics.Debug.WriteLine(string.Format("{0}{1} 押下", action.Modifiers, action.KeyName));
                    }
                }
                else if (action.Type.Contains("解放"))
                {
                    keybd_event(vkCode, 0, KEYEVENTF_KEYUP, IntPtr.Zero);
                    if (isSelfModifierKey)
                    {
                        Thread.Sleep(30); // 修飾キーの状態を確実に反映（少し長めに）
                        System.Diagnostics.Debug.WriteLine(string.Format("修飾キー解放: VK={0:X2} ({1})", vkCode, action.KeyName));
                    }
                    else if (!string.IsNullOrEmpty(action.Modifiers))
                    {
                        // 修飾キー付きの通常キー
                        System.Diagnostics.Debug.WriteLine(string.Format("{0}{1} 解放", action.Modifiers, action.KeyName));
                    }
                }
            }
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine(string.Format("アクション実行エラー: {0}", ex.Message));
        }
    }

    private void SaveButton_Click(object sender, EventArgs e)
    {
        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey == null)
        {
            MessageBox.Show("保存するホットキーをリストから選択してください。", "選択エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        if (selectedHotkey.RecordedActions.Count == 0)
        {
            MessageBox.Show("保存するアクションがありません。", "警告", 
                MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        SaveFileDialog saveFileDialog = new SaveFileDialog();
        saveFileDialog.Filter = "個別ホットキーファイル (*.txt)|*.txt|すべてのファイル (*.*)|*.*";
        saveFileDialog.Title = "個別ホットキーデータを保存";
        saveFileDialog.FileName = string.Format("hotkey_{0}_{1}.txt", 
            selectedHotkey.GetHotkeyString().Replace("+", "_"), 
            DateTime.Now.ToString("yyyyMMdd_HHmmss"));

        if (saveFileDialog.ShowDialog() == DialogResult.OK)
        {
            try
            {
                SaveActionsToFile(saveFileDialog.FileName, selectedHotkey);
                MessageBox.Show("個別ホットキーデータを保存しました。", "完了", 
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
                statusLabel.Text = "個別保存完了";
                statusLabel.ForeColor = Color.Green;
            }
            catch (Exception ex)
            {
                MessageBox.Show(string.Format("保存エラー: {0}", ex.Message), "エラー", 
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
        saveFileDialog.Dispose();
    }

    private void LoadButton_Click(object sender, EventArgs e)
    {
        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey == null)
        {
            MessageBox.Show("読み込み先のホットキーをリストから選択してください。", "選択エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        OpenFileDialog openFileDialog = new OpenFileDialog();
        openFileDialog.Filter = "個別ホットキーファイル (*.txt)|*.txt|すべてのファイル (*.*)|*.*";
        openFileDialog.Title = "個別ホットキーデータを読み込み";

        if (openFileDialog.ShowDialog() == DialogResult.OK)
        {
            try
            {
                var loadedActions = LoadActionsFromFile(openFileDialog.FileName);
                
                if (loadedActions != null && loadedActions.Count > 0)
                {
                    // 既に記録がある場合は上書き確認
                    if (selectedHotkey.RecordedActions.Count > 0)
                    {
                        DialogResult result = MessageBox.Show(
                            string.Format("ホットキー '{0}' には既に {1}個の動作が記録されています。\n上書きして読み込みますか？", 
                                selectedHotkey.GetHotkeyString(), selectedHotkey.RecordedActions.Count),
                            "上書き確認",
                            MessageBoxButtons.YesNo,
                            MessageBoxIcon.Question);
                        
                        if (result != DialogResult.Yes)
                        {
                            openFileDialog.Dispose();
                            return; // 読み込みをキャンセル
                        }
                    }
                    
                    selectedHotkey.RecordedActions = loadedActions;
                    RefreshHotkeyList();
                    DisplayActionsForHotkey(selectedHotkey);
                    MessageBox.Show(string.Format("{0}個のアクションを読み込みました。", loadedActions.Count), "完了", 
                        MessageBoxButtons.OK, MessageBoxIcon.Information);
                    statusLabel.Text = string.Format("個別読み込み完了 - {0}個のアクション", loadedActions.Count);
                    statusLabel.ForeColor = Color.Green;
                }
                else
                {
                    MessageBox.Show("有効なアクションデータが見つかりませんでした。", "警告", 
                        MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(string.Format("読み込みエラー: {0}", ex.Message), "エラー", 
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
        openFileDialog.Dispose();
    }

    private void SaveAllHotkeysToFile(string filePath)
    {
        try
        {
            StreamWriter writer = new StreamWriter(filePath, false, Encoding.UTF8);
            
            writer.WriteLine("# Simple Hotkey Tool - All Hotkeys Data");
            writer.WriteLine("# Generated: " + DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss"));
            writer.WriteLine("# Hotkey Count: " + registeredHotkeys.Count);
            writer.WriteLine("# Format: [HOTKEY]|UseCtrl|UseShift|UseAlt|Key|Description|ContinuousExecution|IsPaused|FastExecution|LoopCount");
            writer.WriteLine("# Format: [ACTION]|Type|Timestamp|DelayMs|X|Y|VirtualKeyCode|KeyName|Details|WParam|MouseData|IsKeyDown|Modifiers");
            writer.WriteLine("");
            
            foreach (var hotkey in registeredHotkeys)
            {
                // ホットキー情報を保存（連打実行フラグ、一時停止フラグ、高速実行フラグ、ループ回数を含む）
                writer.WriteLine(string.Format("[HOTKEY]|{0}|{1}|{2}|{3}|{4}|{5}|{6}|{7}|{8}",
                    hotkey.UseCtrl,
                    hotkey.UseShift,
                    hotkey.UseAlt,
                    hotkey.Key.ToString(),
                    hotkey.Description,
                    hotkey.ContinuousExecution,
                    hotkey.IsPaused,
                    hotkey.FastExecution,
                    hotkey.LoopCount));
                
                // そのホットキーのアクションを保存
                foreach (var action in hotkey.RecordedActions)
                {
                    writer.WriteLine("[ACTION]|" + action.ToFileString());
                }
                
                writer.WriteLine(""); // ホットキー間の区切り
            }
            
            writer.Close();
        }
        catch (Exception ex)
        {
            throw new Exception(string.Format("ファイル保存エラー: {0}", ex.Message));
        }
    }

    private List<HotkeyInfo> LoadAllHotkeysFromFile(string filePath)
    {
        try
        {
            List<HotkeyInfo> hotkeys = new List<HotkeyInfo>();
            HotkeyInfo currentHotkey = null;
            
            StreamReader reader = new StreamReader(filePath, Encoding.UTF8);
            string line;
            while ((line = reader.ReadLine()) != null)
            {
                if (string.IsNullOrWhiteSpace(line) || line.StartsWith("#"))
                    continue;
                
                if (line.StartsWith("[HOTKEY]|"))
                {
                    // 新しいホットキーの開始
                    string[] parts = line.Substring(9).Split('|'); // "[HOTKEY]|"を除去
                    if (parts.Length >= 5)
                    {
                        currentHotkey = new HotkeyInfo();
                        currentHotkey.UseCtrl = bool.Parse(parts[0]);
                        currentHotkey.UseShift = bool.Parse(parts[1]);
                        currentHotkey.UseAlt = bool.Parse(parts[2]);
                        
                        Keys key;
                        if (Enum.TryParse(parts[3], out key))
                        {
                            currentHotkey.Key = key;
                        }
                        
                        currentHotkey.Description = parts[4];
                        
                        // 連打実行フラグの読み込み（古いファイルとの互換性のためtry-catchで囲む）
                        if (parts.Length >= 6)
                        {
                            try
                            {
                                currentHotkey.ContinuousExecution = bool.Parse(parts[5]);
                            }
                            catch
                            {
                                currentHotkey.ContinuousExecution = false; // デフォルト値
                            }
                        }
                        
                        // 一時停止フラグの読み込み（古いファイルとの互換性のためtry-catchで囲む）
                        if (parts.Length >= 7)
                        {
                            try
                            {
                                currentHotkey.IsPaused = bool.Parse(parts[6]);
                            }
                            catch
                            {
                                currentHotkey.IsPaused = false; // デフォルト値
                            }
                        }
                        
                        // 高速実行フラグの読み込み（古いファイルとの互換性のためtry-catchで囲む）
                        if (parts.Length >= 8)
                        {
                            try
                            {
                                currentHotkey.FastExecution = bool.Parse(parts[7]);
                            }
                            catch
                            {
                                currentHotkey.FastExecution = false; // デフォルト値
                            }
                        }
                        
                        // ループ回数の読み込み（古いファイルとの互換性のためtry-catchで囲む）
                        if (parts.Length >= 9)
                        {
                            try
                            {
                                currentHotkey.LoopCount = int.Parse(parts[8]);
                                if (currentHotkey.LoopCount < 1) currentHotkey.LoopCount = 1; // 最小値制限
                            }
                            catch
                            {
                                currentHotkey.LoopCount = 1; // デフォルト値
                            }
                        }
                        
                        hotkeys.Add(currentHotkey);
                    }
                }
                else if (line.StartsWith("[ACTION]|") && currentHotkey != null)
                {
                    // アクションデータ
                    string actionData = line.Substring(9); // "[ACTION]|"を除去
                    RecordedAction action = RecordedAction.FromFileString(actionData);
                    if (action != null)
                    {
                        currentHotkey.RecordedActions.Add(action);
                    }
                }
            }
            reader.Close();
            
            return hotkeys;
        }
        catch (Exception ex)
        {
            throw new Exception(string.Format("ファイル読み込みエラー: {0}", ex.Message));
        }
    }

    // フォーム終了時の処理でクリーンアップ
    private void CleanupRecordingState()
    {
        rememberedHotkeyForRecording = null;
        currentRecordingHotkey = null;
        isRecording = false;
        isUpdatingSelection = false;
        savedModifierState = null;
    }

    private void DeleteActionButton_Click(object sender, EventArgs e)
    {
        DeleteSelectedActions();
    }

    // 選択された動作を削除する共通メソッド
    private void DeleteSelectedActions()
    {
        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey == null)
        {
            MessageBox.Show("ホットキーを選択してください。", "選択エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        if (actionListView.SelectedItems.Count == 0)
        {
            MessageBox.Show("削除する動作を選択してください。", "選択エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        if (isRecording)
        {
            MessageBox.Show("記録中は編集できません。", "編集エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        try
        {
            // 選択されたアクションのインデックスを取得（降順でソートして後ろから削除）
            List<int> selectedIndices = new List<int>();
            foreach (ListViewItem item in actionListView.SelectedItems)
            {
                int index = int.Parse(item.SubItems[0].Text) - 1; // 1ベースから0ベースに変換
                selectedIndices.Add(index);
            }
            selectedIndices.Sort((a, b) => b.CompareTo(a)); // 降順ソート

            // 後ろから削除
            foreach (int index in selectedIndices)
            {
                if (index >= 0 && index < selectedHotkey.RecordedActions.Count)
                {
                    selectedHotkey.RecordedActions.RemoveAt(index);
                }
            }

            RefreshHotkeyList();
            DisplayActionsForHotkey(selectedHotkey);
            
            // 削除後もホットキーの選択状態を維持
            SelectHotkeyInList(selectedHotkey);
            
            statusLabel.Text = string.Format("ホットキー '{0}' から {1}個の動作を削除しました", 
                selectedHotkey.GetHotkeyString(), selectedIndices.Count);
            statusLabel.ForeColor = Color.Orange;
            
            // アプリケーションの更新を強制的に処理
            Application.DoEvents();
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("削除エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }

    private void EditDelayButton_Click(object sender, EventArgs e)
    {
        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey == null)
        {
            MessageBox.Show("ホットキーを選択してください。", "選択エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        if (actionListView.SelectedItems.Count == 0)
        {
            MessageBox.Show("編集する動作を選択してください。", "選択エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        if (isRecording)
        {
            MessageBox.Show("記録中は編集できません。", "編集エラー", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            return;
        }

        EditSelectedActionDelay(selectedHotkey);
    }

    private void ActionListView_DoubleClick(object sender, EventArgs e)
    {
        HotkeyInfo selectedHotkey = GetSelectedHotkey();
        if (selectedHotkey == null || actionListView.SelectedItems.Count == 0 || isRecording)
            return;

        // ダブルクリックされた位置を確認
        Point clickPoint = actionListView.PointToClient(Control.MousePosition);
        ListViewHitTestInfo hitTest = actionListView.HitTest(clickPoint);
        
        if (hitTest.SubItem != null)
        {
            // 待ち時間列（3番目の列）がクリックされた場合
            if (hitTest.Item.SubItems.IndexOf(hitTest.SubItem) == 3)
            {
                EditSelectedActionDelay(selectedHotkey);
            }
        }
    }

    private void EditSelectedActionDelay(HotkeyInfo selectedHotkey)
    {
        try
        {
            ListViewItem selectedItem = actionListView.SelectedItems[0];
            int actionIndex = int.Parse(selectedItem.SubItems[0].Text) - 1; // 1ベースから0ベースに変換
            
            if (actionIndex < 0 || actionIndex >= selectedHotkey.RecordedActions.Count)
                return;

            RecordedAction action = selectedHotkey.RecordedActions[actionIndex];
            string currentDelay = action.DelayMs.ToString();
            
            // シンプルな入力ダイアログを使用
            string input = PromptForInput("待ち時間をミリ秒で入力してください (0-60000):", currentDelay);

            if (!string.IsNullOrEmpty(input))
            {
                int newDelay;
                if (int.TryParse(input, out newDelay) && newDelay >= 0 && newDelay <= 60000)
                {
                    action.DelayMs = newDelay;
                    DisplayActionsForHotkey(selectedHotkey);
                    statusLabel.Text = string.Format("動作 {0} の待ち時間を {1}ms に変更しました", 
                        actionIndex + 1, newDelay);
                    statusLabel.ForeColor = Color.Green;
                }
                else
                {
                    MessageBox.Show("0から60000の間の数値を入力してください。", "入力エラー", 
                        MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }
            }
        }
        catch (Exception ex)
        {
            MessageBox.Show(string.Format("編集エラー: {0}", ex.Message), "エラー", 
                MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
    }

    // シンプルな入力ダイアログ
    private string PromptForInput(string message, string defaultValue)
    {
        Form prompt = new Form();
        prompt.Width = 400;
        prompt.Height = 160;
        prompt.Text = "入力";
        prompt.StartPosition = FormStartPosition.CenterParent;
        prompt.FormBorderStyle = FormBorderStyle.FixedDialog;
        prompt.MaximizeBox = false;
        prompt.MinimizeBox = false;

        Label textLabel = new Label();
        textLabel.Left = 10;
        textLabel.Top = 10;
        textLabel.Width = 360;
        textLabel.Height = 40;
        textLabel.Text = message;
        prompt.Controls.Add(textLabel);

        TextBox inputBox = new TextBox();
        inputBox.Left = 10;
        inputBox.Top = 50;
        inputBox.Width = 280;
        inputBox.Text = defaultValue;
        prompt.Controls.Add(inputBox);

        Button buttonOk = new Button();
        buttonOk.Text = "OK";
        buttonOk.Left = 300;
        buttonOk.Top = 57;
        buttonOk.Width = 75;
        buttonOk.DialogResult = DialogResult.OK;
        prompt.Controls.Add(buttonOk);

        Button buttonCancel = new Button();
        buttonCancel.Text = "キャンセル";
        buttonCancel.Left = 300;
        buttonCancel.Top = 85;
        buttonCancel.Width = 75;
        buttonCancel.DialogResult = DialogResult.Cancel;
        prompt.Controls.Add(buttonCancel);

        prompt.AcceptButton = buttonOk;
        prompt.CancelButton = buttonCancel;

        return prompt.ShowDialog() == DialogResult.OK ? inputBox.Text : "";
    }

    private void SaveActionsToFile(string filePath, HotkeyInfo hotkey)
    {
        try
        {
            StreamWriter writer = new StreamWriter(filePath, false, Encoding.UTF8);
            
            writer.WriteLine("# Simple Hotkey Tool Action Data");
            writer.WriteLine("# Generated: " + DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss"));
            writer.WriteLine("# Hotkey: " + hotkey.GetHotkeyString());
            writer.WriteLine("# Description: " + hotkey.Description);
            writer.WriteLine("# Action Count: " + hotkey.RecordedActions.Count);
            writer.WriteLine("# Format: Type|Timestamp|DelayMs|X|Y|VirtualKeyCode|KeyName|Details|WParam|MouseData|IsKeyDown|Modifiers");
            writer.WriteLine("");
            
            foreach (var action in hotkey.RecordedActions)
            {
                writer.WriteLine(action.ToFileString());
            }
            
            writer.Close();
        }
        catch (Exception ex)
        {
            throw new Exception(string.Format("ファイル保存エラー: {0}", ex.Message));
        }
    }

    private List<RecordedAction> LoadActionsFromFile(string filePath)
    {
        try
        {
            List<RecordedAction> actions = new List<RecordedAction>();
            
            StreamReader reader = new StreamReader(filePath, Encoding.UTF8);
            string line;
            while ((line = reader.ReadLine()) != null)
            {
                if (string.IsNullOrWhiteSpace(line) || line.StartsWith("#"))
                    continue;
                
                RecordedAction action = RecordedAction.FromFileString(line);
                if (action != null)
                {
                    actions.Add(action);
                }
            }
            reader.Close();
            
            return actions;
        }
        catch (Exception ex)
        {
            throw new Exception(string.Format("ファイル読み込みエラー: {0}", ex.Message));
        }
    }

    private IntPtr SetHook(LowLevelMouseProc proc, int hookType)
    {
        Process curProcess = Process.GetCurrentProcess();
        ProcessModule curModule = curProcess.MainModule;
        return SetWindowsHookEx(hookType, proc, GetModuleHandle(curModule.ModuleName), 0);
    }

    private IntPtr SetHook(LowLevelKeyboardProc proc, int hookType)
    {
        Process curProcess = Process.GetCurrentProcess();
        ProcessModule curModule = curProcess.MainModule;
        return SetWindowsHookEx(hookType, proc, GetModuleHandle(curModule.ModuleName), 0);
    }

    private bool IsClickOnThisForm(int x, int y)
    {
        try
        {
            POINT pt = new POINT { x = x, y = y };
            IntPtr hwnd = WindowFromPoint(pt);
            
            // このフォーム自体のハンドルチェック
            if (hwnd == this.Handle)
                return true;
            
            // すべてのコントロールを再帰的にチェック
            return IsChildOfForm(hwnd, this);
        }
        catch
        {
            return false;
        }
    }

    private bool IsChildOfForm(IntPtr hwnd, Control parent)
    {
        try
        {
            // 親コントロール自身のハンドルチェック
            if (hwnd == parent.Handle)
                return true;
            
            // 子コントロールを再帰的にチェック
            foreach (Control control in parent.Controls)
            {
                if (hwnd == control.Handle)
                    return true;
                
                // さらに子コントロールがある場合は再帰的にチェック
                if (control.HasChildren && IsChildOfForm(hwnd, control))
                    return true;
            }
            
            return false;
        }
        catch
        {
            return false;
        }
    }

    private IntPtr MouseHookProc(int nCode, IntPtr wParam, IntPtr lParam)
    {
        if (nCode >= 0 && isRecording && currentRecordingHotkey != null)
        {
            try
            {
                MSLLHOOKSTRUCT hookStruct = (MSLLHOOKSTRUCT)Marshal.PtrToStructure(lParam, typeof(MSLLHOOKSTRUCT));
                
                // ツール自身への操作は記録しない
                if (IsClickOnThisForm(hookStruct.pt.x, hookStruct.pt.y))
                {
                    return CallNextHookEx(mouseHookID, nCode, wParam, lParam);
                }
                
                string eventType = "";
                string details = "";
                bool shouldRecord = false;

                switch ((int)wParam)
                {
                    case WM_LBUTTONDOWN:
                        eventType = "マウス: 左クリック押下";
                        shouldRecord = true;
                        break;
                    case WM_LBUTTONUP:
                        eventType = "マウス: 左クリック解放";
                        shouldRecord = true;
                        break;
                    case WM_RBUTTONDOWN:
                        eventType = "マウス: 右クリック押下";
                        shouldRecord = true;
                        break;
                    case WM_RBUTTONUP:
                        eventType = "マウス: 右クリック解放";
                        shouldRecord = true;
                        break;
                    case WM_MBUTTONDOWN:
                        eventType = "マウス: 中クリック押下";
                        shouldRecord = true;
                        break;
                    case WM_MBUTTONUP:
                        eventType = "マウス: 中クリック解放";
                        shouldRecord = true;
                        break;
                    case WM_MOUSEMOVE:
                        if (mouseMoveCheckBox.Checked)
                        {
                            eventType = "マウス: 移動";
                            shouldRecord = true;
                        }
                        break;
                    case WM_MOUSEWHEEL:
                        eventType = "マウス: ホイール";
                        int delta = (short)((hookStruct.mouseData >> 16) & 0xFFFF);
                        details = delta > 0 ? "上" : "下";
                        shouldRecord = true;
                        break;
                }

                if (shouldRecord && !string.IsNullOrEmpty(eventType))
                {
                    DateTime currentTime = DateTime.Now;
                    int delayMs = 0;
                    
                    if (lastEventTime != DateTime.MinValue)
                    {
                        TimeSpan timeDiff = currentTime - lastEventTime;
                        delayMs = (int)timeDiff.TotalMilliseconds;
                    }

                    // GetCursorPosで論理座標を取得して記録（DPIスケーリング対応）
                    POINT logicalPoint;
                    GetCursorPos(out logicalPoint);
                    
                    var recordedAction = new RecordedAction
                    {
                        Type = eventType,
                        Timestamp = currentTime,
                        DelayMs = delayMs,
                        X = logicalPoint.x, // 論理座標で保存
                        Y = logicalPoint.y, // 論理座標で保存
                        Details = details,
                        WParam = (int)wParam,
                        MouseData = hookStruct.mouseData
                    };
                    
                    System.Diagnostics.Debug.WriteLine(string.Format("マウス記録: 物理({0},{1}) → 論理({2},{3}) イベント={4}",
                        hookStruct.pt.x, hookStruct.pt.y, logicalPoint.x, logicalPoint.y, eventType));
                    
                    currentRecordingHotkey.RecordedActions.Add(recordedAction);

                    this.BeginInvoke(new Action(() => {
                        try
                        {
                            // フォームが破棄されている場合は何もしない
                            if (this.IsDisposed || currentRecordingHotkey == null)
                                return;
                            
                            // 記録中のホットキーの選択状態を維持
                            EnsureRecordingHotkeySelected();
                            
                            // ホットキーリストの記録数を更新
                            RefreshHotkeyList();
                            
                            // 記録中のホットキーの動作履歴をリアルタイム更新
                            DisplayActionsForHotkey(currentRecordingHotkey);
                            
                            // 最新のアクションまでスクロール
                            if (actionListView.Items.Count > 0)
                            {
                                actionListView.Items[actionListView.Items.Count - 1].EnsureVisible();
                                actionListView.Items[actionListView.Items.Count - 1].BackColor = Color.LightYellow; // 最新アクションをハイライト
                                
                                // 前のアクションのハイライトを解除
                                if (actionListView.Items.Count > 1)
                                {
                                    actionListView.Items[actionListView.Items.Count - 2].BackColor = Color.White;
                                }
                            }
                            
                            // ステータス更新
                            statusLabel.Text = string.Format("記録中: {0} ({1}個のアクション)", 
                                currentRecordingHotkey.GetHotkeyString(), currentRecordingHotkey.RecordedActions.Count);
                            
                            lastEventTime = currentTime;
                        }
                        catch (Exception ex)
                        {
                            System.Diagnostics.Debug.WriteLine("UI更新エラー: " + ex.Message);
                        }
                    }));
                }
            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine(string.Format("Mouse hook error: {0}", ex.Message));
            }
        }

        return CallNextHookEx(mouseHookID, nCode, wParam, lParam);
    }

    private IntPtr KeyboardHookProc(int nCode, IntPtr wParam, IntPtr lParam)
    {
        if (nCode >= 0 && isRecording && currentRecordingHotkey != null)
        {
            try
            {
                KBDLLHOOKSTRUCT hookStruct = (KBDLLHOOKSTRUCT)Marshal.PtrToStructure(lParam, typeof(KBDLLHOOKSTRUCT));
                
                // Escキーが押された場合は記録を停止（記録中のEscキー処理）
                if ((int)wParam == WM_KEYDOWN && hookStruct.vkCode == 0x1B) // VK_ESCAPE = 0x1B
                {
                    System.Diagnostics.Debug.WriteLine("記録中にEscキー検出 - 記録停止");
                    ForceStopExecution();
                    return (IntPtr)1; // Escキーイベントを食い止める
                }
                
                string eventType = "";
                string keyName = "";
                bool isKeyDown = false;
                bool shouldRecord = false;

                switch ((int)wParam)
                {
                    case WM_KEYDOWN:
                    case WM_SYSKEYDOWN:
                        eventType = "キーボード: キー押下";
                        isKeyDown = true;
                        
                        if (keyStates.ContainsKey(hookStruct.vkCode) && keyStates[hookStruct.vkCode])
                        {
                            shouldRecord = false;
                        }
                        else
                        {
                            keyStates[hookStruct.vkCode] = true;
                            shouldRecord = true;
                        }
                        break;
                        
                    case WM_KEYUP:
                    case WM_SYSKEYUP:
                        eventType = "キーボード: キー解放";
                        isKeyDown = false;
                        
                        if (keyStates.ContainsKey(hookStruct.vkCode) && keyStates[hookStruct.vkCode])
                        {
                            keyStates[hookStruct.vkCode] = false;
                            shouldRecord = true;
                        }
                        else
                        {
                            shouldRecord = false;
                        }
                        break;
                }

                try
                {
                    Keys key = (Keys)hookStruct.vkCode;
                    keyName = key.ToString();
                }
                catch
                {
                    keyName = string.Format("VK_{0:X2}", hookStruct.vkCode);
                }

                // 矢印キーや通常キーの場合、現在の修飾キー状態を記録
                string modifiers = "";
                bool isArrowKey = (hookStruct.vkCode >= 0x25 && hookStruct.vkCode <= 0x28); // VK_LEFT(0x25) ~ VK_DOWN(0x28)
                bool isNormalKey = !IsModifierKey(hookStruct.vkCode); // 修飾キー以外
                
                if (shouldRecord && isNormalKey && isKeyDown)
                {
                    // 現在押されている修飾キーの状態を取得
                    List<string> modList = new List<string>();
                    if ((GetAsyncKeyState(VK_LCONTROL) & 0x8000) != 0 || (GetAsyncKeyState(VK_RCONTROL) & 0x8000) != 0)
                        modList.Add("Ctrl");
                    if ((GetAsyncKeyState(VK_LSHIFT) & 0x8000) != 0 || (GetAsyncKeyState(VK_RSHIFT) & 0x8000) != 0)
                        modList.Add("Shift");
                    if ((GetAsyncKeyState(VK_LMENU) & 0x8000) != 0 || (GetAsyncKeyState(VK_RMENU) & 0x8000) != 0)
                        modList.Add("Alt");
                    
                    if (modList.Count > 0)
                    {
                        modifiers = string.Join("+", modList) + "+";
                        System.Diagnostics.Debug.WriteLine(string.Format("修飾キー付きキー記録: {0}{1}", modifiers, keyName));
                    }
                }

                if (shouldRecord && !string.IsNullOrEmpty(eventType))
                {
                    DateTime currentTime = DateTime.Now;
                    int delayMs = 0;
                    
                    if (lastEventTime != DateTime.MinValue)
                    {
                        TimeSpan timeDiff = currentTime - lastEventTime;
                        delayMs = (int)timeDiff.TotalMilliseconds;
                    }

                    var recordedAction = new RecordedAction
                    {
                        Type = eventType,
                        Timestamp = currentTime,
                        DelayMs = delayMs,
                        VirtualKeyCode = hookStruct.vkCode,
                        KeyName = keyName,
                        IsKeyDown = isKeyDown,
                        Modifiers = modifiers
                    };
                    
                    currentRecordingHotkey.RecordedActions.Add(recordedAction);

                    this.BeginInvoke(new Action(() => {
                        try
                        {
                            // フォームが破棄されている場合は何もしない
                            if (this.IsDisposed || currentRecordingHotkey == null)
                                return;
                            
                            // 記録中のホットキーの選択状態を維持
                            EnsureRecordingHotkeySelected();
                            
                            // ホットキーリストの記録数を更新
                            RefreshHotkeyList();
                            
                            // 記録中のホットキーの動作履歴をリアルタイム更新
                            DisplayActionsForHotkey(currentRecordingHotkey);
                            
                            // 最新のアクションまでスクロール
                            if (actionListView.Items.Count > 0)
                            {
                                actionListView.Items[actionListView.Items.Count - 1].EnsureVisible();
                                actionListView.Items[actionListView.Items.Count - 1].BackColor = Color.LightYellow; // 最新アクションをハイライト
                                
                                // 前のアクションのハイライトを解除
                                if (actionListView.Items.Count > 1)
                                {
                                    actionListView.Items[actionListView.Items.Count - 2].BackColor = Color.White;
                                }
                            }
                            
                            // ステータス更新
                            statusLabel.Text = string.Format("記録中: {0} ({1}個のアクション)", 
                                currentRecordingHotkey.GetHotkeyString(), currentRecordingHotkey.RecordedActions.Count);
                            
                            lastEventTime = currentTime;
                        }
                        catch (Exception ex)
                        {
                            System.Diagnostics.Debug.WriteLine("UI更新エラー: " + ex.Message);
                        }
                    }));
                }
            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine(string.Format("Keyboard hook error: {0}", ex.Message));
            }
        }

        return CallNextHookEx(keyboardHookID, nCode, wParam, lParam);
    }

    // 修飾キーかどうかを判定するヘルパーメソッド
    private bool IsModifierKey(uint vkCode)
    {
        return vkCode == 0xA0 || vkCode == 0xA1 ||  // VK_LSHIFT, VK_RSHIFT
               vkCode == 0xA2 || vkCode == 0xA3 ||  // VK_LCONTROL, VK_RCONTROL
               vkCode == 0xA4 || vkCode == 0xA5 ||  // VK_LMENU, VK_RMENU
               vkCode == 0x5B || vkCode == 0x5C;    // VK_LWIN, VK_RWIN
    }

    private void SimpleHotkeyForm_FormClosing(object sender, FormClosingEventArgs e)
    {
        // ×ボタンで終了時は確認ダイアログを表示
        if (e.CloseReason == CloseReason.UserClosing)
        {
            DialogResult result = MessageBox.Show(
                "ホットキーツールを終了しますか？",
                "終了確認",
                MessageBoxButtons.YesNo,
                MessageBoxIcon.Question);
            
            if (result == DialogResult.No)
            {
                e.Cancel = true;
                return;
            }
        }
        
        // 終了時に自動保存
        AutoSaveHotkeys();
        
        // 記録状態のクリーンアップ
        CleanupRecordingState();
        
        // Escキー監視フックを解除
        if (escKeyHookID != IntPtr.Zero)
        {
            UnhookWindowsHookEx(escKeyHookID);
            escKeyHookID = IntPtr.Zero;
        }
        
        // ウィンドウ復帰用のホットキーを解除
        UnregisterHotKey(this.Handle, SHOW_WINDOW_HOTKEY_ID);
        
        // すべてのホットキーを解除（登録されているもののみ）
        foreach (var hotkey in registeredHotkeys)
        {
            if (hotkey.IsRegistered)
            {
                UnregisterHotKey(this.Handle, hotkey.Id);
            }
        }
        
        // フックを解除
        if (mouseHookID != IntPtr.Zero)
        {
            UnhookWindowsHookEx(mouseHookID);
        }
        if (keyboardHookID != IntPtr.Zero)
        {
            UnhookWindowsHookEx(keyboardHookID);
        }
        
        // ★修正：Mutexを解放
        if (appMutex != null)
        {
            try
            {
                appMutex.ReleaseMutex();
                appMutex.Close();
            }
            catch
            {
                // 既に解放されている場合のエラーを無視
            }
            finally
            {
                appMutex = null;
            }
        }
    }

    // 自動保存メソッド
    private void AutoSaveHotkeys()
    {
        try
        {
            // 実行中のフォルダパスを取得（PowerShell実行時に対応）
            string currentDir = Environment.CurrentDirectory;
            string autoSaveFile = Path.Combine(currentDir, "auto_hotkeys.hkall");
            
            System.Diagnostics.Debug.WriteLine("自動保存パス: " + autoSaveFile);
            System.Diagnostics.Debug.WriteLine("登録ホットキー数: " + registeredHotkeys.Count);
            
            // ホットキーが0個でも空ファイルとして保存
            SaveAllHotkeysToFile(autoSaveFile);
            
            System.Diagnostics.Debug.WriteLine("自動保存完了");
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("自動保存エラー: " + ex.Message);
            // エラーが発生してもアプリケーションを終了
        }
    }

    // 自動読み込みメソッド（修正：RegisterHotkeyReliableを使用）
    private void AutoLoadHotkeys()
    {
        try
        {
            // 実行中のフォルダパスを取得（PowerShell実行時に対応）
            string currentDir = Environment.CurrentDirectory;
            string autoSaveFile = Path.Combine(currentDir, "auto_hotkeys.hkall");
            
            System.Diagnostics.Debug.WriteLine("自動読み込みパス: " + autoSaveFile);
            
            if (!File.Exists(autoSaveFile))
            {
                System.Diagnostics.Debug.WriteLine("自動保存ファイルが存在しません");
                return;
            }

            var loadedHotkeys = LoadAllHotkeysFromFile(autoSaveFile);
            
            if (loadedHotkeys != null && loadedHotkeys.Count > 0)
            {
                // 新しいホットキーを登録（一時停止中でなければ）
                foreach (var hotkey in loadedHotkeys)
                {
                    hotkey.Id = nextHotkeyId++;
                    
                    // 一時停止中でない場合のみホットキーを登録（修正：RegisterHotkeyReliableを使用）
                    if (!hotkey.IsPaused)
                    {
                        System.Diagnostics.Debug.WriteLine(string.Format("【起動時登録】ホットキー '{0}' ID={1}, Modifiers=0x{2:X}, Key={3}", 
                            hotkey.GetHotkeyString(), hotkey.Id, hotkey.GetModifiers(), hotkey.Key));
                        
                        bool success = RegisterHotkeyReliable(hotkey);
                        
                        System.Diagnostics.Debug.WriteLine(string.Format("【起動時登録結果】成功={0}, IsRegistered={1}", success, hotkey.IsRegistered));
                    }
                    else
                    {
                        // 一時停止中の場合は登録しない
                        hotkey.IsRegistered = false;
                        System.Diagnostics.Debug.WriteLine(string.Format("【起動時スキップ】ホットキー '{0}' は一時停止中のため登録せず", hotkey.GetHotkeyString()));
                    }
                    
                    registeredHotkeys.Add(hotkey);
                }
                
                RefreshHotkeyList();
                statusLabel.Text = string.Format("前回のホットキー設定を復元しました（{0}個）", loadedHotkeys.Count);
                statusLabel.ForeColor = Color.Green;
                System.Diagnostics.Debug.WriteLine("自動読み込み完了: " + loadedHotkeys.Count + "個");
            }
            else
            {
                System.Diagnostics.Debug.WriteLine("読み込み可能なホットキーがありませんでした");
            }
        }
        catch (Exception ex)
        {
            System.Diagnostics.Debug.WriteLine("自動読み込みエラー: " + ex.Message);
            // エラーが発生しても起動処理は続行
            statusLabel.Text = "ホットキーを追加してください";
            statusLabel.ForeColor = Color.Blue;
        }
    }
}

/* シンプルホットキー管理コントローラー *******************************************************************************/
public class SimpleHotkeyController
{
    private SimpleHotkeyForm managerForm;

    public SimpleHotkeyController()
    {
        try
        {
            managerForm = new SimpleHotkeyForm();
        }
        catch (InvalidOperationException)
        {
            // 二重起動の場合（メッセージは既にSimpleHotkeyFormで表示済み）
            managerForm = null;
        }
        catch (Exception ex)
        {
            // その他の予期しないエラー
            MessageBox.Show(
                "アプリケーションの初期化に失敗しました。\n" + ex.Message,
                "エラー",
                MessageBoxButtons.OK,
                MessageBoxIcon.Error);
            managerForm = null;
        }
    }

    public void ShowManager()
    {
        if (managerForm != null && !managerForm.IsDisposed)
        {
            managerForm.Show();
        }
    }

    public void ShowManagerDialog()
    {
        if (managerForm != null && !managerForm.IsDisposed)
        {
            managerForm.ShowDialog();
        }
    }

    public void CloseManager()
    {
        if (managerForm != null && !managerForm.IsDisposed)
        {
            managerForm.Close();
        }
    }
}