﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Runtime.InteropServices;
//
// Windows Forms C# アプリケーションによる動作サンプルを提供します。(遅延バインディングで利用)
//-------------------------------------------------------------------------------------
//
// 1.JwCADInfoをインストール。
//
// 2.エクスプローラでイメージ、プレビュー表示を確認。
//
// 3.Microsoft Visual Studio 2019 Community でこのプロジェクトをロード。
//
// 4.実行環境に合わせて、64bitまたは32bitでビルドしてください。
//
namespace JwView
{
	public partial class Form1 : Form
	{
		/// <summary>
		/// 設定データファイル（適当にファイルパスを指定してください）
		/// </summary>
		const	string	c_UserFile = @"e:\temp\UserFile.xxx";

		/// <summary>
		/// プレビューウィンドウハンドル
		/// </summary>
		IntPtr	m_hViewWnd;

		/// <summary>
		/// イメージサイズ
		/// </summary>
		int		m_ImageSize;

		/// <summary>
		/// 対象ファイルのパス名
		/// </summary>
		string	m_strFile;

		/// <summary>
		/// 遅延バインディング
		/// </summary>
		LateBinding m_Binding;

		/// <summary>
		/// バインディングする関数宣言
		/// C++での関数宣言
		/// HBITMAP	GetImage(int size, LPCWSTR file);
		/// HWND	GetImageView(LPCWSTR file, const RECT& rect, HWND hParent);
		/// bool	InfoLoad(LPCWSTR file);
		/// bool	InfoSave(LPCWSTR file);
		/// </summary>
		public	struct		RECT { public int l, t, r, b; }
		public	delegate	IntPtr	GetImage(int size, [MarshalAs(UnmanagedType.LPWStr)]string file);
		public	delegate	IntPtr	GetImageView([MarshalAs(UnmanagedType.LPWStr)]string file, ref RECT rect, IntPtr hParent);
		public	delegate	bool	InfoLoad([MarshalAs(UnmanagedType.LPWStr)]string file);
		public	delegate	bool	InfoSave([MarshalAs(UnmanagedType.LPWStr)]string file);

		/// <summary>
		/// 呼び出し関数ポインタ
		/// </summary>
		GetImage	 getImage;
		GetImageView getImageView;
		InfoLoad	 infoLoad;
		InfoSave	 infoSave;

		/// <summary>
		/// 初期化
		/// </summary>
		public	Form1()
		{
			this.InitializeComponent();

			m_ImageSize = 256;
			radioButton6.Checked = true;

			// 遅延バインディング
			string	path;
			if(GetInstallPath(out path))
			{
				// 失敗した場合はフォルダ選択ボタンを押してください
				button3.Enabled = !LoadLibrary(path);
			}
			// 前回の設定情報を復元する
			if(System.IO.File.Exists(c_UserFile) && !infoLoad(c_UserFile))
			{
				MessageBox.Show("設定ファイルの復元に失敗しました", "Load error!!", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}
		/// <summary>
		/// 閉じるボタン
		/// </summary>
		/// <param name="sender">送信元</param>
		/// <param name="e">パラメータ</param>
		private void Button1_Click(object sender, EventArgs e)
		{
			// 設定情報を保存する
			if(!infoSave(c_UserFile))
			{
				MessageBox.Show("設定ファイルの保存に失敗しました", "Save error!!", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			Close();
		}
		/// <summary>
		/// イメージの更新
		/// </summary>
		/// <param name="bImage">イメージ更新</param>
		/// <param name="bView">プレビュー更新</param>
		private void ImageUpdate(bool bImage, bool bView)
		{
			if(button3.Enabled)
			{
				MessageBox.Show("DLLパスを指定してください", "DLL error!!", MessageBoxButtons.OK, MessageBoxIcon.Error);
				button3.Focus();
				return;
			}
			if(!System.IO.File.Exists(m_strFile))
			{
				MessageBox.Show("ファイルを指定してください", "File error!!", MessageBoxButtons.OK, MessageBoxIcon.Error);
				return;
			}
			try
			{
				if(bImage)
				{
					// 指定サイズのイメージ取得
					IntPtr	hBmp = getImage(m_ImageSize, m_strFile);
					if(IntPtr.Zero != hBmp)
					{
						// ピクチャーボックスにイメージを貼り付ける
						Image img = Image.FromHbitmap(hBmp);
						pictureBox1.Image = null;
						pictureBox1.Image = img;

						// ハンドル解放
						Win32APICall.DeleteObject(hBmp);
					}
					else
					{
						MessageBox.Show("イメージ取得に失敗しました", "GetImage error!!", MessageBoxButtons.OK, MessageBoxIcon.Error);
					}
				}
				if(bView)
				{
					// 前回のプレビューを破棄
					if(IntPtr.Zero != m_hViewWnd)
					{
						Win32APICall.DestroyWindow(m_hViewWnd);
					}
					// ピクチャーボックスのサイズにプレビューを貼り付ける
					RECT	rect = new RECT
					{
						l = pictureBox2.Left,
						t = pictureBox2.Top,
						r = pictureBox2.Right,
						b = pictureBox2.Bottom
					};
					// プレビュー取得
					m_hViewWnd = getImageView(m_strFile, ref rect, pictureBox2.Handle);
					if(IntPtr.Zero == m_hViewWnd)
					{
						MessageBox.Show("プレビュー取得に失敗しました", "GetImageView error!!", MessageBoxButtons.OK, MessageBoxIcon.Error);
					}
				}
			}
			catch(Exception e)
			{
				MessageBox.Show(e.Message, "JwCADInfo error!!",MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}
		/// <summary>
		/// プレビューのサイズ変更
		/// </summary>
		/// <param name="sender">送信元</param>
		/// <param name="e">パラメータ</param>
		private void OnSizeChanged(object sender, EventArgs e)
		{
			const short SWP_NOMOVE		= 0x0002;
			const short SWP_NOZORDER	= 0x0004;
			const short SWP_NOACTIVATE	= 0x0010;
			Win32APICall.SetWindowPos(m_hViewWnd, 0, 0, 0, pictureBox2.Width, pictureBox2.Height,
				SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE);
		}
		/// <summary>
		/// ファイル選択ボタン
		/// </summary>
		/// <param name="sender">送信元</param>
		/// <param name="e">パラメータ</param>
		private void button2_Click(object sender, EventArgs e)
		{
			OpenFileDialog ofd = new OpenFileDialog();
			ofd.Filter = "JwCAD File(*.jww;*.jws;*.jwc;*.jwk)|*.jww;*.jws;*.jwc;*.jwk";
			ofd.Title  = "ファイルを選択してください";
			if(ofd.ShowDialog() == DialogResult.OK)
			{
				m_strFile = ofd.FileName;
				ImageUpdate(true, true);
			}
		}
		/// <summary>
		/// Radioボタンの選択
		/// </summary>
		/// <param name="sender">送信元</param>
		/// <param name="e">パラメータ</param>
		private void OnRadioChange(object sender, EventArgs e)
		{
			RadioButton radio = sender as RadioButton;
			if(null == radio)
				return;

			switch(radio.Text)
			{
			case	"16":	m_ImageSize = 16;	break;
			case	"32":	m_ImageSize = 32;	break;
			case	"48":	m_ImageSize = 48;	break;
			case	"64":	m_ImageSize = 64;	break;
			case	"128":	m_ImageSize = 128;	break;
			case	"256":	m_ImageSize = 256;	break;
			}
			ImageUpdate(true, false);
		}
		/// <summary>
		/// JwCADInfo インストールフォルダの選択ボタン
		/// </summary>
		/// <param name="sender">送信元</param>
		/// <param name="e">パラメータ</param>
		private void button3_Click(object sender, EventArgs e)
		{
			FolderBrowserDialog	fbd = new FolderBrowserDialog();
			fbd.Description = "JwCADInfo のインストールフォルダを指定してください。";
		//	fbd.RootFolder = Environment.SpecialFolder.ProgramFiles;
			// 最初に選択するフォルダを指定する
			fbd.SelectedPath = "C:\\Program files\\JwCADInfo";
			fbd.ShowNewFolderButton = false;
	
			// ダイアログを表示する
			if(fbd.ShowDialog(this) != DialogResult.OK)
				return;

			if(!LoadLibrary(fbd.SelectedPath))
			{
				MessageBox.Show("モジュールの取得に失敗しました", "Folder error!!", MessageBoxButtons.OK, MessageBoxIcon.Error);
				return;
			}
			button3.Enabled = false;
		}
		/// <summary>
		/// ライブラリのロード
		/// </summary>
		/// <param name="path">インストールパス</param>
		/// <returns>結果</returns>
		private bool LoadLibrary(string path)
		{
			// モジュールハンドラ取得
			try
			{
				m_Binding = new LateBinding(path + "\\JwCADTool.dll");
				getImage	 = (GetImage)	 m_Binding.GetDelegate("GetImage",	   typeof(GetImage));
				getImageView = (GetImageView)m_Binding.GetDelegate("GetImageView", typeof(GetImageView));
				infoLoad	 = (InfoLoad)	 m_Binding.GetDelegate("InfoLoad",	   typeof(InfoLoad));
				infoSave	 = (InfoSave)	 m_Binding.GetDelegate("InfoSave",	   typeof(InfoSave));
			}
			catch(Exception)
			{
				// モジュールの取得に失敗しました
				return	false;
			}
			return	true;
		}
		/// <summary>
		/// JwCADInfo インストールフォルダの取得
		/// </summary>
		/// <param name="path">インストールパス</param>
		/// <returns>結果</returns>
		private bool GetInstallPath(out string path)
		{
			path = "";
			string	clsid = "{48C7EAC4-1795-432B-9679-B47373014A97}";
			string	dllPath = "";

			Microsoft.Win32.RegistryKey regRootKey = Microsoft.Win32.Registry.ClassesRoot.OpenSubKey("CLSID", false);
			if(regRootKey != null)
			{
				Microsoft.Win32.RegistryKey regClsidKey = regRootKey.OpenSubKey(clsid, false);
				if(regClsidKey != null)
				{
					Microsoft.Win32.RegistryKey regServerKey = regClsidKey.OpenSubKey("InprocServer32", false);
					if(regServerKey != null)
					{
						dllPath = (string)regServerKey.GetValue("");
						regServerKey.Close();
					}
					regClsidKey.Close();
				}
				regRootKey.Close();
			}
			if(System.IO.File.Exists(dllPath))
			{
				path = System.IO.Path.GetDirectoryName(dllPath);
				return	true;
			}
			return	false;
		}
	}
	/// <summary>
	/// 遅延バインディング
	/// </summary>
	public class LateBinding : IDisposable
	{
		private	IntPtr	_module;	// モジュールハンドル

		/// <summary>
		/// DLL名を指定して遅延バインディングを行うオブジェクトを生成します。
		/// </summary>
		/// <param name="filename">バインドするDLL名</param>
		public LateBinding(string filename)
		{
			_module = Win32APICall.LoadLibrary(filename);
			if(_module != IntPtr.Zero)
			{
				return;
			}
			// 例外スロー
			int result = Marshal.GetHRForLastWin32Error();
			throw Marshal.GetExceptionForHR(result);
		}
		/// <summary>
		/// 指定した名前を持つアンマネージ関数ポインタをデリゲートに変換します。
		/// </summary>
		/// <param name="procName">アンマネージ関数名</param>
		/// <param name="delegateType">変換するデリゲートのType</param>
		/// <returns>変換したデリゲート</returns>
		public Delegate GetDelegate(string procName, Type delegateType)
		{
			IntPtr ptr = Win32APICall.GetProcAddress(_module, procName);
			if(ptr != IntPtr.Zero)
			{
				Delegate	d = Marshal.GetDelegateForFunctionPointer(ptr, delegateType);
				return	d;
			}
			// 例外スロー
			int result = Marshal.GetHRForLastWin32Error();
			throw Marshal.GetExceptionForHR(result);
		}
		/// <summary>
		/// 破棄
		/// </summary>
		public void Dispose()
		{
			if(_module != IntPtr.Zero)
			{
				Win32APICall.FreeLibrary(_module);
			}
		}
	}
	/// <summary>
	/// Win32 APIコール
	/// </summary>
	public class Win32APICall
	{
		[DllImport("gdi32.dll")]
		internal static extern int DeleteObject(System.IntPtr hobject);
		[DllImport("user32.dll")]
		internal static extern bool DestroyWindow(IntPtr hWnd);
		[DllImport("user32.dll")]
		internal static extern IntPtr SetWindowPos(IntPtr hWnd, int hAfter, int x, int Y, int cx, int cy, int wFlags);
		[DllImport("kernel32", CharSet = CharSet.Unicode, SetLastError = true)]
		internal static extern IntPtr LoadLibrary(string lpFileName);
		[DllImport("kernel32", SetLastError = true)]
		internal static extern bool FreeLibrary(IntPtr hModule);
		[DllImport("kernel32", CharSet = CharSet.Ansi, SetLastError = true, ExactSpelling = false)]
		internal static extern IntPtr GetProcAddress(IntPtr hModule, string lpProcName);
	}
}
