const { app, BrowserWindow, ipcMain, shell, dialog } = require('electron');
const path = require('path');
const fs = require('fs');
const sharp = require('sharp');

let mainWindow;

function createWindow() {
    mainWindow = new BrowserWindow({
        width: 800,
        height: 600,
        maxWidth: 800,
        maxHeight: 800,
        webPreferences: {
            preload: path.join(__dirname, 'preload.js'),
            contextIsolation: true,
            nodeIntegration: false
        },
        backgroundColor: '#1e1e1e',
        frame: true, // Standard OS frame for now, could go frameless for custom UI if preferred
        // titleBarStyle: 'hidden' // Optional: for custom titlebar
    });

    mainWindow.setMenu(null); // Remove default menu bar

    mainWindow.loadFile('index.html');
}

app.whenReady().then(() => {
    createWindow();

    app.on('activate', () => {
        if (BrowserWindow.getAllWindows().length === 0) createWindow();
    });
});

app.on('window-all-closed', () => {
    if (process.platform !== 'darwin') app.quit();
});

// --- Image Processing Logic ---

// Helper to calculate perceived brightness
async function getBrightness(imageBuffer) {
    const { data, info } = await sharp(imageBuffer)
        .raw()
        .toBuffer({ resolveWithObject: true });

    let sum = 0;
    // Standard luminance formula: 0.2126*R + 0.7152*G + 0.0722*B
    // Data is R,G,B,R,G,B... (assuming 3 channels, could be 4 with alpha)
    const channels = info.channels;
    for (let i = 0; i < data.length; i += channels) {
        const r = data[i];
        const g = data[i + 1];
        const b = data[i + 2];
        sum += (0.2126 * r + 0.7152 * g + 0.0722 * b);
    }
    return sum / (data.length / channels);
}

ipcMain.handle('process-images', async (event, { filePaths, settings }) => {
    const results = [];
    const { resizeEnabled, targetWidth, targetHeight, outputMode, outputFormat } = settings;

    for (const filePath of filePaths) {
        try {
            const fileDir = path.dirname(filePath);
            const fileName = path.basename(filePath, path.extname(filePath));
            const originalExt = path.extname(filePath).slice(1); // remove dot

            // Determine output format
            const format = outputFormat === 'original' ? originalExt : outputFormat;

            // Determine output path
            let outputDir;
            if (outputMode === 'dedicated') {
                outputDir = path.join(fileDir, 'processed');
                if (!fs.existsSync(outputDir)) {
                    fs.mkdirSync(outputDir, { recursive: true });
                }
            } else {
                outputDir = fileDir;
            }

            const outputFilename = `${fileName}_adjusted.${format}`;
            const outputPath = path.join(outputDir, outputFilename);

            // Start Sharp pipeline
            let pipeline = sharp(filePath);

            // 1. Analyze Brightness and Adjust
            // New Strategy: Screen Blend Mode
            const stats = await pipeline.stats();
            const meanLuminance = (0.2126 * stats.channels[0].mean + 0.7152 * stats.channels[1].mean + 0.0722 * stats.channels[2].mean);

            console.log(`Processing ${fileName}: Luminance = ${meanLuminance}`);

            // Thresholds for "Screen" blending
            // Screen works by inverting, multiplying, and inverting back. It always lightens.
            // Formula: 1 - (1-A)*(1-B)

            const buffer = await pipeline.toBuffer(); // We need the buffer for composition

            if (meanLuminance < 130) {
                const composites = [];

                // First layer: Always apply if < 130
                composites.push({
                    input: buffer,
                    blend: 'screen'
                });

                // Second layer: Apply if Very Dark (< 60)
                if (meanLuminance < 60) {
                    composites.push({
                        input: buffer,
                        blend: 'screen'
                    });
                }

                pipeline = sharp(buffer).composite(composites);
            }
            // Else: Leave valid exposure as is

            // 2. Resize
            if (resizeEnabled && (targetWidth || targetHeight)) {
                // Resize logic remains same
                const w = targetWidth ? parseInt(targetWidth) : null;
                const h = targetHeight ? parseInt(targetHeight) : null;
                pipeline = pipeline.resize({
                    width: w,
                    height: h,
                    fit: 'inside'
                });
            }

            // 3. Format & Save
            await pipeline
                .toFormat(format)
                .toFile(outputPath);

            results.push({ success: true, original: filePath, output: outputPath });
        } catch (error) {
            console.error(`Error processing ${filePath}:`, error);
            results.push({ success: false, original: filePath, error: error.message });
        }
    }
    return results;
});

ipcMain.handle('select-folder', async () => {
    const result = await dialog.showOpenDialog(mainWindow, {
        properties: ['openDirectory']
    });
    return result.filePaths[0];
});
