const dropZone = document.getElementById('dropZone');
const previewArea = document.getElementById('previewArea');
const statusUrl = document.getElementById('statusUrl');

// Settings Inputs
const enableResizeCheckbox = document.getElementById('enableResize');
const targetWidthInput = document.getElementById('targetWidth');
const targetHeightInput = document.getElementById('targetHeight');
const resizeInputsDiv = document.getElementById('resizeInputs');
const outputModeSelect = document.getElementById('outputMode');
const outputFormatSelect = document.getElementById('outputFormat');

// Handle Resize Toggle
// Handle Resize Toggle
enableResizeCheckbox.addEventListener('change', (e) => {
    toggleResizeInputs(e.target.checked);
    saveSettings();
});

function toggleResizeInputs(enabled) {
    if (enabled) {
        resizeInputsDiv.classList.remove('disabled');
        targetWidthInput.disabled = false;
        targetHeightInput.disabled = false;
    } else {
        resizeInputsDiv.classList.add('disabled');
        targetWidthInput.disabled = true;
        targetHeightInput.disabled = true;
    }
}

// Persistence Logic
function saveSettings() {
    const settings = {
        resizeEnabled: enableResizeCheckbox.checked,
        targetWidth: targetWidthInput.value,
        targetHeight: targetHeightInput.value,
        outputMode: outputModeSelect.value,
        outputFormat: outputFormatSelect.value
    };
    localStorage.setItem('blimSettings', JSON.stringify(settings));
}

function loadSettings() {
    const saved = localStorage.getItem('blimSettings');
    if (saved) {
        try {
            const settings = JSON.parse(saved);
            enableResizeCheckbox.checked = settings.resizeEnabled;
            targetWidthInput.value = settings.targetWidth || '';
            targetHeightInput.value = settings.targetHeight || '';
            outputModeSelect.value = settings.outputMode || 'original';
            outputFormatSelect.value = settings.outputFormat || 'original';

            toggleResizeInputs(settings.resizeEnabled);
        } catch (e) {
            console.error('Failed to load settings', e);
        }
    }
}

// Add change listeners to all inputs for auto-saving
[targetWidthInput, targetHeightInput, outputModeSelect, outputFormatSelect].forEach(el => {
    el.addEventListener('change', saveSettings);
    el.addEventListener('input', saveSettings); // for text inputs
});

// Initialize
loadSettings();

// Handle Drag & Drop Visuals
['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
    dropZone.addEventListener(eventName, preventDefaults, false);
    document.body.addEventListener(eventName, preventDefaults, false); // Allow dropping anywhere
});

function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
}

['dragenter', 'dragover'].forEach(eventName => {
    dropZone.addEventListener(eventName, highlight, false);
});

['dragleave', 'drop'].forEach(eventName => {
    dropZone.addEventListener(eventName, unhighlight, false);
});

function highlight(e) {
    dropZone.classList.add('drag-over');
}

function unhighlight(e) {
    dropZone.classList.remove('drag-over');
}

// Handle File Drop
dropZone.addEventListener('drop', handleDrop, false);

function handleDrop(e) {
    const dt = e.dataTransfer;
    const files = dt.files;

    if (files.length > 0) {
        // Debug
        // alert(`Dropped ${files.length} files. First: ${files[0].name}`);
        handleFiles(files);
    } else {
        // alert('No files dropped.');
    }
}

async function handleFiles(files) {
    if (!window.api || !window.api.getPathForFile) {
        alert('エラー: アプリのAPIが読み込まれていません。アプリを再起動してください。');
        return;
    }

    let rawPaths = [];
    try {
        rawPaths = Array.from(files).map(f => window.api.getPathForFile(f));
    } catch (e) {
        alert('ファイルパスの取得中にエラーが発生しました: ' + e.message);
        return;
    }

    const filePaths = rawPaths.filter(p => p && isImage(p));

    if (filePaths.length === 0) {
        alert('有効な画像ファイルが見つかりませんでした。\n検出されたパス: ' + JSON.stringify(rawPaths));
        return;
    }

    statusUrl.innerText = `${filePaths.length} 枚の画像を処理中...`;

    // Switch UI to show grid if not already
    // For now we keep the drop zone visible but maybe overlay results or append
    // Let's clear the drop message if it's the first time
    const dropMessage = dropZone.querySelector('.drop-message');
    if (dropMessage) dropMessage.style.display = 'none';

    // Use the dropZone itself as the container or ensure previewArea is visible inside it
    // Actually, let's just append results to previewArea which we will move INSIDE dropZone or make dropZone the container
    // My HTML structure had separate dropZone and previewArea.
    // Let's hide dropZone default content and show previewArea.
    previewArea.classList.remove('hidden');
    // Move previewArea inside dropZone? No, let's just keep them separate but swap visibility or have them stack.
    // Current CSS: dropZone takes flex 1. previewArea has results.
    // Let's make dropZone smaller or just overlay.
    // Simpler: Just append to previewArea and hide dropZone text.
    dropZone.style.border = 'none'; // removing border to look like gallery
    dropZone.appendChild(previewArea); // Move it in if not already

    const settings = {
        resizeEnabled: enableResizeCheckbox.checked,
        targetWidth: targetWidthInput.value,
        targetHeight: targetHeightInput.value,
        outputMode: outputModeSelect.value,
        outputFormat: outputFormatSelect.value
    };

    const results = await window.api.processImages({ filePaths, settings });

    statusUrl.innerText = `完了しました。`;

    results.forEach(res => {
        if (res.success) {
            addPreview(res.output);
        } else {
            console.error(res.error);
        }
    });
}

function isImage(path) {
    return /\.(jpg|jpeg|png|webp|bmp|tiff)$/i.test(path);
}

function addPreview(filePath) {
    const div = document.createElement('div');
    div.className = 'preview-item';

    const img = document.createElement('img');
    img.src = `file://${filePath}?t=${new Date().getTime()}`; // cache bust

    const badge = document.createElement('div');
    badge.className = 'status-badge';
    badge.innerText = '修正完了';

    div.appendChild(img);
    div.appendChild(badge);
    previewArea.appendChild(div);
}
