/*------------------------------------------------------------------------------
    インクルード
------------------------------------------------------------------------------*/
#include "../common.h"

/*------------------------------------------------------------------------------
    書式：BOOL Calc(WCHAR *, double *)
    引数：WCHAR *  -
          double * -
    戻値：BOOL     -
    機能：文字式を計算する。
------------------------------------------------------------------------------*/
INT32 Polish::Calc(const WCHAR *const str, INT32 *val)
{
	double ans;
	INT32 res;
	INT32 len;
	WCHAR *pStr;
	
	len = wcslen(str)+1;
	len *= 2;
	*val = 0.0;
	
	pStr = new WCHAR[len];
	wmemset(pStr, 0x0000, len);
	
	AddPlus(pStr, (WCHAR *)str);
	res = Convert(pStr);
	if (res != PLSH_NOERROR) {
		PStackTerminate();
		DStackTerminate();
		delete [] pStr;
		return res;
	}
	
	if (cntD == 1) {
		DStackPop(&ans);
		PStackTerminate();
		DStackTerminate();
		delete [] pStr;
		
		if (ans < -2147483647) {ans = -2147483647;}
		if (2147483647 < ans)  {ans =  2147483647;}
		
		*val = (INT32)ans;
		return PLSH_NOERROR;
	}
	else {
		PStackTerminate();
		DStackTerminate();
		delete [] pStr;
		return PLSH_ERR_DECODEREMAIN;
	}
}

/*------------------------------------------------------------------------------
    書式：void AddPlus(WCHAR *, WCHAR *)
    引数：WCHAR *  -
          WCHAR *  -
    機能：[-]の前に[+]をつける
------------------------------------------------------------------------------*/
void Polish::AddPlus(WCHAR *dst, WCHAR *src)
{
	WCHAR *p1;
	WCHAR *p2;
	
	p1 = src;
	p2 = dst;
	
	while (*p1 != L'\0') {
		if (*p1 == L'-') {
			*p2 = L'+';  p2++;
			*p2 = L'-';  p2++;
		}
		else {
			*p2 = *p1;  p2++;
		}
		p1++;
	}
}

/*------------------------------------------------------------------------------
    書式：INT32 Convert(WCHAR *)
    引数：WCHAR *  -
    戻値：BOOL     -
    機能：逆ポーランド記法に変換。
------------------------------------------------------------------------------*/
BOOL Polish::Convert(WCHAR *str)
{
	INT32 res;
	INT32 len;
	INT32 ope;
	INT32 term;
	WCHAR o;
	INT32 a, b;
	WCHAR *p;
	INT32 sta, end;
	INT32 l_sta, l_end;
	INT32 r_sta, r_end;
	
	RemoveSpace(str);
	
	if (!CheckBlock(str)) {return PLSH_ERR_CHECKBLOCK;}
	if (!IsFormula(str))  {return PLSH_ERR_FORMULA;}
	if (!Expression(str)) {return PLSH_ERR_EXPRESSION;}
	
	len = wcslen(str);
	if (len == 0) {return PLSH_ERR_LEN;}
	
	PStackInit();
	DStackInit();
	
	sta = 0;
	end = len-1;
	if (IsRemoveBlock(str+sta, end-sta+1)) {sta++;  end--;}
	
	PStackPush(L'@', sta, end);
	
	while (0 < cntP) {
		PStackPop(&o, &a, &b);
		
		if (o != L'@') {
			res = PushDecode(0.0, o);
			if (res != PLSH_NOERROR) {return res;}
			continue;
		}
		
		p   = str+a;
		len = b-a+1;
		
		ope = FindOpe(p, len, &term);
		
		if (ope == -1) {
			if (len != 0) {
				PushDecode(wcstod(p, NULL), L'@');
			}
		}
		else {
			ope += a;
			o = str[ope];
			
			r_sta = ope+1;
			r_end = b;
			l_sta = a;
			l_end = ope-1;
			
			if (IsRemoveBlock(str+r_sta, r_end-r_sta+1)) {r_sta++;  r_end--;}
			if (IsRemoveBlock(str+l_sta, l_end-l_sta+1)) {l_sta++;  l_end--;}
			
			if (term == 1) {
				if (o == L'+') {o = L'#';}
				if (o == L'-') {o = L'_';}
			}
			
			PStackPush(o, 0, 0);
			
			if (r_sta <= r_end) {
				PStackPush(L'@', r_sta, r_end);
			}
			else {
				return PLSH_ERR_TERMCNTR;
			}
			
			if (l_sta <= l_end) {
				PStackPush(L'@', l_sta, l_end);
			}
			else {
				if (term != 1) {
					return PLSH_ERR_TERMCNTL;
				}
			}
		}
	}
	return PLSH_NOERROR;
}

/*------------------------------------------------------------------------------
    書式：INT32 PushDecode(double, WCHAR)
    引数：double  -
          WCHAR   -
    戻値：INT32   -
    機能：復号に渡す
------------------------------------------------------------------------------*/
INT32 Polish::PushDecode(double val, WCHAR o)
{
	double val1;
	double val2;
	
	if (o == L'@') {
		DStackPush(val);
		return PLSH_NOERROR;
	}
	else if (o == L'#') {
		if (!DStackPop(&val1)) {return PLSH_ERR_PUSHDECODE;}
		val1 *= +1;
		DStackPush(val1);
	}
	else if (o == L'_') {
		if (!DStackPop(&val1)) {return PLSH_ERR_PUSHDECODE;}
		val1 *= -1;
		DStackPush(val1);
	}
	else if (o == L'+') {
		if (!DStackPop(&val2)) {return PLSH_ERR_PUSHDECODE;}
		if (!DStackPop(&val1)) {return PLSH_ERR_PUSHDECODE;}
		val1 = val1 + val2;
		DStackPush(val1);
	}
	else if (o == L'-') {
		if (!DStackPop(&val2)) {return PLSH_ERR_PUSHDECODE;}
		if (!DStackPop(&val1)) {return PLSH_ERR_PUSHDECODE;}
		val1 = val1 - val2;
		DStackPush(val1);
	}
	else if (o == L'*') {
		if (!DStackPop(&val2)) {return PLSH_ERR_PUSHDECODE;}
		if (!DStackPop(&val1)) {return PLSH_ERR_PUSHDECODE;}
		val1 = val1 * val2;
		DStackPush(val1);
	}
	else if (o == L'/') {
		if (!DStackPop(&val2)) {return PLSH_ERR_PUSHDECODE;}
		if (!DStackPop(&val1)) {return PLSH_ERR_PUSHDECODE;}
		if (val2 == 0)         {return PLSH_ERR_DIV0;}
		val1 = val1 / val2;
		DStackPush(val1);
	}
	return PLSH_NOERROR;
}

/*------------------------------------------------------------------------------
    書式：void RemoveSpace(WCHAR *)
    引数：WCHAR *  -
    機能：空白とタブ文字を削除
------------------------------------------------------------------------------*/
void Polish::RemoveSpace(WCHAR *str)
{
	WCHAR *dst = str;
	WCHAR *src = str;
	
	while (*src) {
		if ((*src == L' ') || (*src == L'\t')) {
			src++;
			continue;
		}
		*dst = *src;
		dst++;
		src++;
	}
	*dst = L'\0';
}

/*------------------------------------------------------------------------------
    書式：BOOL CheckBlock(WCHAR *)
    引数：WCHAR *  -
    戻値：BOOL     -
    機能：括弧の確認
------------------------------------------------------------------------------*/
BOOL Polish::CheckBlock(WCHAR *str)
{
	INT32 i;
	INT32 depth;
	
	depth = 0;
	for (i = 0; str[i] != L'\0'; i++) {
		if (str[i] == L'(') {depth--;}
		if (str[i] == L')') {depth++;}
		
		if (depth > 0) {return FALSE;}
	}
	
	if (depth == 0) {return TRUE;}
	else            {return FALSE;}
}

/*------------------------------------------------------------------------------
    書式：BOOL IsFormula(WCHAR *)
    引数：WCHAR *  -
    機能：数式かどうか確認
------------------------------------------------------------------------------*/
BOOL Polish::IsFormula(WCHAR *str)
{
	INT32 i;
	
	for (i = 0; str[i] != L'\0'; i++) {
		switch (str[i]) {
		case L'0':
		case L'1':
		case L'2':
		case L'3':
		case L'4':
		case L'5':
		case L'6':
		case L'7':
		case L'8':
		case L'9':
		case L'.':
		case L'+':
		case L'-':
		case L'*':
		case L'/':
		case L'(':
		case L')':  break;
		default:   return FALSE;
		}
	}
	return TRUE;
}

/*------------------------------------------------------------------------------
    書式：BOOL Expression(WCHAR *)
    引数：WCHAR *  -
    戻値：BOOL     -
    機能：表現が間違っていないか
------------------------------------------------------------------------------*/
BOOL Polish::Expression(WCHAR *str)
{
	INT32 i;
	INT32 j;
	
	for (i = 0; str[i] != L'\0'; i++) {
		if (str[i] == L'(') {
			for (j = i-1; 0 <= j; j--) {
				if (isdigit(str[j]) || str[j] == L'.') {
					return FALSE;
				}
				else if (IsOperand(str[j])) {
					break;
				}
			}
		}
		else if (str[i] == L')') {
			for (j = i+1; str[j] != L'\0'; j++) {
				if (isdigit(str[j]) || str[j] == L'.') {
					return FALSE;
				}
				else if (IsOperand(str[j])) {
					break;
				}
			}
		}
	}
	return TRUE;
}

/*------------------------------------------------------------------------------
    書式：INT32 FindOpe(WCHAR *, INT32, INT32 *)
    引数：WCHAR *  -
          INT32    -
          INT32 *  -
    戻値：INT32    -
    機能：最も優先度の低い演算子を探す
------------------------------------------------------------------------------*/
INT32 Polish::FindOpe(WCHAR *str, INT32 len, INT32 *pTerm)
{
	INT32 i;
	INT32 j;
	INT32 idx;
	INT32 depth;
	INT32 deep;
	INT32 prio;
	INT32 min;
	INT32 term;
	INT32 tmp;
	
	idx = -1;
	
	depth = 0;
	deep  = -100;
	prio  = 0;
	min   = 100;
	term  = 2;
	tmp   = 2;
	
	for (i = 0; i < len; i++) {
		if (str[i] == L'(') {depth--;  continue;}
		if (str[i] == L')') {depth++;  continue;}
		
		switch (str[i]) {
			case L'+':
				if (i == 0) {
					prio = 3;
					term = 1;
				}
				else {
					prio = 1;
					term = 2;
				}
				
				for (j = i-1; 0 <= j; j--) {
					if (IsOperand(str[j])) {
						prio = 3;
						term = 1;
						break;
					}
					
					if (isdigit(str[j]) || (str[j] == L'.')) {
						break;
					}
				}
				break;
			case L'-':
				prio = 3;
				term = 1;
				break;
			case L'*':
			case L'/':
				prio = 2;
				term = 2;
				break;
			default:
				continue;
		}
		
		if (deep < depth) {
			deep = depth;
			min  = prio;
			idx  = i;
			tmp  = term;
		}
		else if (deep == depth)  {
			if (prio < min) {
				deep = depth;
				min  = prio;
				idx  = i;
				tmp  = term;
			}
		}
	}
	
	*pTerm = tmp;
	return idx;
}

/*------------------------------------------------------------------------------
    書式：BOOL IsRemoveBlock(WCHAR *, INT32)
    引数：WCHAR *  -
          INT32    -
    戻値：BOOL     -
    機能：外側の括弧を削除していいかどうか
------------------------------------------------------------------------------*/
BOOL Polish::IsRemoveBlock(WCHAR *str, INT32 len)
{
	INT32 i;
	INT32 depth;
	
	if (len == 0) {return FALSE;}
	
	if ((str[0] != L'(') || (str[len-1] != L')')) {return FALSE;}
	
	depth = -1;
	for (i = 1; i <= (len-2); i++) {
		if (str[i] == L'(') {depth--;}
		if (str[i] == L')') {depth++;}
		
		if (depth == 0) {return FALSE;}
	}
	return TRUE;
}

/*------------------------------------------------------------------------------
    書式：BOOL IsOperand(WCHAR)
    引数：WCHAR   -
    戻値：BOOL    -
    機能：演算子かどうか
------------------------------------------------------------------------------*/
BOOL Polish::IsOperand(WCHAR c)
{
	switch (c) {
	case L'+':
	case L'-':
	case L'*':
	case L'/':  return TRUE;
	default:    return FALSE;
	}
}

/*------------------------------------------------------------------------------
    書式：void PStackInit(void)
    機能：スタックを初期化
------------------------------------------------------------------------------*/
void Polish::PStackInit(void)
{
	rootP.front = NULL;
	rootP.back  = NULL;
	rootP.ope   = L'\0';
	rootP.left  = 0;
	rootP.right = 0;
	pCurP       = &rootP;
	cntP        = 0;
}

/*------------------------------------------------------------------------------
    書式：void PStackPush(WCHAR, INT32, INT32)
    引数：WCHAR   -
          INT32   -
          INT32   -
    機能：スタックに追加
------------------------------------------------------------------------------*/
void Polish::PStackPush(WCHAR ope, INT32 left, INT32 right)
{
	st_PStack *pMall;
	st_PStack *pCur;
	
	if (pCurP->back != NULL) {exit(1);}
	
	pMall = new st_PStack();
	pCur  = pCurP;
	
	pCur->back   = pMall;
	pMall->front = pCur;
	pMall->back  = NULL;
	pMall->ope   = ope;
	pMall->left  = left;
	pMall->right = right;
	
	cntP++;
	pCurP = pMall;
}

/*------------------------------------------------------------------------------
    書式：void PStackPop(WCHAR *, INT32 *, INT32 *)
    引数：WCHAR *   -
          INT32 *   -
          INT32 *   -
    機能：スタックから取得
------------------------------------------------------------------------------*/
void Polish::PStackPop(WCHAR *pOpe, INT32 *pLeft, INT32 *pRight)
{
	st_PStack *pMall;
	st_PStack *pCur;
	
	if (cntP <= 0) {exit(1);}
	
	pMall = pCur = pCurP;
	
	*pOpe   = pCur->ope;
	*pLeft  = pCur->left;
	*pRight = pCur->right;
	
	pCur = pCur->front;
	pCur->back = NULL;
	
	delete pMall;
	cntP--;
	
	pCurP = pCur;
}

/*------------------------------------------------------------------------------
    書式：void PStackTerminate(void)
    機能：スタックを強制終了
------------------------------------------------------------------------------*/
void Polish::PStackTerminate(void)
{
	WCHAR o;
	INT32 a, b;
	
	while (0 < cntP) {PStackPop(&o, &a, &b);}
}

/*------------------------------------------------------------------------------
    書式：void DStackInit(void)
    機能：スタックを初期化
------------------------------------------------------------------------------*/
void Polish::DStackInit(void)
{
	rootD.front = NULL;
	rootD.back  = NULL;
	rootD.val   = 0.0;
	pCurD       = &rootD;
	cntD        = 0;
}

/*------------------------------------------------------------------------------
    書式：void DStackPush(double)
    引数：double  -
    機能：スタックに追加
------------------------------------------------------------------------------*/
void Polish::DStackPush(double val)
{
	st_DStack *pMall;
	st_DStack *pCur;
	
	if (pCurD->back != NULL) {return;}
	
	pMall = new st_DStack();
	pCur  = pCurD;
	
	pCur->back   = pMall;
	pMall->front = pCur;
	pMall->back  = NULL;
	pMall->val   = val;
	
	cntD++;
	pCurD = pMall;
}

/*------------------------------------------------------------------------------
    書式：BOOL DStackPop(double *)
    引数：double *   -
    機能：スタックから取得
------------------------------------------------------------------------------*/
BOOL Polish::DStackPop(double *pVal)
{
	st_DStack *pMall;
	st_DStack *pCur;
	
	if (cntD <= 0) {
		DStackTerminate();
		return FALSE;
	}
	
	pMall = pCur = pCurD;
	
	*pVal = pCur->val;
	
	pCur = pCur->front;
	pCur->back = NULL;
	
	delete pMall;
	cntD--;
	
	pCurD = pCur;
	
	return TRUE;
}

/*------------------------------------------------------------------------------
    書式：void DStackTerminate(void)
    機能：スタックを強制終了
------------------------------------------------------------------------------*/
void Polish::DStackTerminate(void)
{
	double val;
	
	while (0 < cntD) {DStackPop(&val);}
}
