﻿#include	"AjcInternal.h"
///************************************************************************************************************///
//																												//
//	２Ｄ／３Ｄベクトル演算																						//
//																												//
///************************************************************************************************************///

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	double	SubAdjustPM1(double n);

//==============================================================================================================//
//	S I N 関 連	（度単位）																						//
//==============================================================================================================//
AJCEXPORT	double		WINAPI AjcSin  (double degree)	{return sin (AJC_DEG2RAD(degree));			}
AJCEXPORT	double		WINAPI AjcSinh (double degree)	{return sinh(AJC_DEG2RAD(degree));			}
AJCEXPORT	double		WINAPI AjcASin (double x)		{return AJC_RAD2DEG(asin(SubAdjustPM1(x)));	}
//==============================================================================================================//
//	C O S 関 連	（度単位）																						//
//==============================================================================================================//
AJCEXPORT	double		WINAPI AjcCos  (double degree)	{return cos (AJC_DEG2RAD(degree));			}
AJCEXPORT	double		WINAPI AjcCosh (double degree)	{return cosh(AJC_DEG2RAD(degree));			}
AJCEXPORT	double		WINAPI AjcACos (double x)		{return AJC_RAD2DEG(acos(SubAdjustPM1(x)));	}
//==============================================================================================================//
//	T A N 関 連	（度単位）																						//
//==============================================================================================================//
AJCEXPORT	double		WINAPI AjcTan				(double degree)			{return tan (AJC_DEG2RAD(degree));	}
AJCEXPORT	double		WINAPI AjcTanh				(double degree)			{return tanh(AJC_DEG2RAD(degree));	}
AJCEXPORT	double		WINAPI AjcATan				(double x)				{return AJC_RAD2DEG(atan (x));		}
AJCEXPORT	double		WINAPI AjcATan2				(double y, double x)	{return AJC_RAD2DEG(atan2(y, x));	}

//==============================================================================================================//
//	３Ｄベクトル加算																							//
//																												//
//	引　数	：	pV1, pV2  - 加算するベクトル情報のアドレス														//
//				pR		  - 加算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 加算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC WINAPI AjcV3dAdd(PCAJC3DVEC pV1, PCAJC3DVEC pV2, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	rsu;

	if (pV1 != NULL && pV2 != NULL && pR != NULL) {
		rsu.x = pV1->x + pV2->x;
		rsu.y = pV1->y + pV2->y;
		rsu.z = pV1->z + pV2->z;
		rc = (PAJC3DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄベクトル減算																							//
//																												//
//	引　数	：	pV1, pV2  - 減算するベクトル情報のアドレス														//
//				pR		  - 減算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 減算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC WINAPI AjcV3dSub(PCAJC3DVEC pV1, PCAJC3DVEC pV2, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	rsu;

	if (pV1 != NULL && pV2 != NULL && pR != NULL) {
		rsu.x = pV1->x - pV2->x;
		rsu.y = pV1->y - pV2->y;
		rsu.z = pV1->z - pV2->z;
		rc = (PAJC3DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//==============================================================================================================//
//	３Ｄベクトル乗算（ベクトルをｎ倍する）																		//
//																												//
//	引　数	：	pV		  - 乗算するベクトル情報のアドレス														//
//				n		  - 乗算する値																			//
//				pR		  - 乗算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 乗算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC WINAPI AjcV3dMult(PCAJC3DVEC pV, double n, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	rsu;

	if (pV != NULL && pR != NULL) {
		rsu.x = pV->x * n;
		rsu.y = pV->y * n;
		rsu.z = pV->z * n;
		rc = (PAJC3DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//==============================================================================================================//
//	３Ｄベクトルの除算																							//
//																												//
//	引　数	：	pV		  - 除算するベクトル情報のアドレス														//
//				n		  - 除算する値																			//
//				pR		  - 除算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 除算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dDiv(PCAJC3DVEC pV, double n, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	rsu;

	if (pV != NULL && pR != NULL) {
		rsu.x = pV->x / n;
		rsu.y = pV->y / n;
		rsu.z = pV->z / n;
		rc = (PAJC3DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//==============================================================================================================//
//	３Ｄ線分情報設定（始点ポイントと方向ベクトル設定）															//
//																												//
//	引　数	：	p1, p2	  - 線分の両端ポイント情報のアドレス													//
//				pR		  - 線分情報を格納するバッファのアドレス												//
//																												//
//	戻り値	：	≠NULL : 線分情報を格納したバッファのアドレス(=pR)												//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DLVEC	WINAPI AjcV3dSetLineVec(PCAJC3DVEC p1, PCAJC3DVEC p2, PAJC3DLVEC pR)
{
	PAJC3DLVEC	rc = NULL;

	if (p1 != NULL && p2 != NULL && pR != NULL) {
		memcpy(&pR->p, p1, sizeof(AJC3DVEC));
		AjcV3dSub(p2, p1, &pR->v);
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ線分情報設定（両端ポイント設定）																		//
//																												//
//	引　数	：	p1, p2	  - 線分の両端ポイント情報のアドレス													//
//				pR		  - 線分情報を格納するバッファのアドレス												//
//																												//
//	戻り値	：	≠NULL : 線分情報を格納したバッファのアドレス(=pR)												//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DLINE	WINAPI AjcV3dSetLinePoint(PCAJC3DVEC p1, PCAJC3DVEC p2, PAJC3DLINE pR)
{
	PAJC3DLINE	rc = NULL;

	if (p1 != NULL && p2 != NULL && pR != NULL) {
		memcpy(&pR->p1, p1, sizeof(AJC3DVEC));
		memcpy(&pR->p2, p2, sizeof(AJC3DVEC));
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ三角形情報設定																							//
//																												//
//	引　数	：	p1, p2, p3	- 三角形の頂点ポイント情報のアドレス												//
//				pR			- 線分情報を格納するバッファのアドレス												//
//																												//
//	戻り値	：	≠NULL : 線分情報を格納したバッファのアドレス(=pR)												//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DTRI	WINAPI AjcV3dSetTriPoint(PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PAJC3DTRI pR)
{
	PAJC3DTRI	rc = NULL;

	if (p1 != NULL && p2 != NULL && p3 != NULL && pR != NULL) {
		memcpy(&pR->p1, p1, sizeof(AJC3DVEC));
		memcpy(&pR->p2, p2, sizeof(AJC3DVEC));
		memcpy(&pR->p3, p3, sizeof(AJC3DVEC));
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄベクトルの長さ算出																						//
//																												//
//	引　数	：	pV		  - 長さを算出するベクトル情報のアドレス												//
//																												//
//	戻り値	：	≠ -1 : ベクトルの長さ																			//
//				＝ -1 : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV3dLength(PCAJC3DVEC pV)
{
	double rc = -1;

	if (pV != NULL) {
		rc = sqrt(pV->x * pV->x + pV->y * pV->y + pV->z * pV->z);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ線分の中点算出																							//
//																												//
//	引　数	：	pL,		  - 中点を算出するラインベクトルのアドレス												//
//				pR		  - 中点を格納するバッファのアドレス													//
//																												//
//	戻り値	：	≠NULL : 中点を格納したバッファのアドレス(=pR)													//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dLVecCenter(PCAJC3DLVEC pL, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	v;

	if (pL != NULL && pR != NULL) {
		AjcV3dDiv(&pL->v, 2, &v);
		AjcV3dAdd(&pL->p, &v, pR);
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ線分の中点算出																							//
//																												//
//	引　数	：	pL,		  - 中点を算出する線分情報のアドレス													//
//				pR		  - 中点を格納するバッファのアドレス													//
//																												//
//	戻り値	：	≠NULL : 中点を格納したバッファのアドレス(=pR)													//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dLineCenter(PCAJC3DLINE pL, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	v;

	if (pL != NULL && pR != NULL) {
		AjcV3dAdd(&pL->p1, &pL->p2, &v);
		AjcV3dDiv(&v, 2, pR);
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄベクトルの外積算出																						//
//																												//
//	引　数	：	pV1, pV2  - 外積を算出するベクトル情報のアドレス												//
//				pR		  - 算出した外積を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 外積を格納したバッファのアドレス(=pR)													//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC WINAPI AjcV3dOuter(PCAJC3DVEC pV1, PCAJC3DVEC pV2, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	rsu;

	if (pV1 != NULL && pV2 != NULL && pR != NULL) {
		rsu.x = pV1->y * pV2->z - pV1->z * pV2->y;
		rsu.y = pV1->z * pV2->x - pV1->x * pV2->z;
		rsu.z = pV1->x * pV2->y - pV1->y * pV2->x;
		rc = (PAJC3DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	PAJC3DVEC WINAPI AjcV3dOuter3P(PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	v1, v2;

	if (p1 != NULL && p2 != NULL && p3 != NULL && pR != NULL) {
		AjcV3dSub(p2, p1, &v1);
		AjcV3dSub(p3, p2, &v2);
		rc = AjcV3dOuter(&v1, &v2, pR);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄベクトルの内積算出																						//
//																												//
//	引　数	：	pV1, pV2  - 内積を算出するベクトル情報のアドレス												//
//																												//
//	戻り値	：	算出した内積値（パラメタエラー時は０を返す）													//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV3dInner(PCAJC3DVEC pV1, PCAJC3DVEC pV2)
{
	double	rc = 0.0;

	if (pV1 != NULL && pV2 != NULL) {
		rc = pV1->x * pV2->x + pV1->y * pV2->y + pV1->z * pV2->z;
	}
	return rc;
}
//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - //
AJCEXPORT	double WINAPI AjcV3dInner3P(PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PAJC3DVEC pR)
{
	double		rc = 0.0;
	AJC3DVEC	v1, v2;

	if (p1 != NULL && p2 != NULL && p3 != NULL && pR != NULL) {
		AjcV3dSub(p2, p1, &v1);
		AjcV3dSub(p3, p2, &v2);
		rc = AjcV3dInner(&v1, &v2);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ三角形の法線ベクトル算出																				//
//																												//
//	引　数	：	pT		  - 三角形情報のアドレス																//
//				pR		  - 法線ベクトルを格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 法線ベクトルを格納したバッファのアドレス(=pR)											//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dPlaneVec(PCAJC3DTRI pT, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	t1, t2;

	if (pT != NULL && pR != NULL) {
		AjcV3dSub(&pT->p2, &pT->p1, &t1);
		AjcV3dSub(&pT->p3, &pT->p1, &t2);
		AjcV3dOuter(&t1, &t2, pR);
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄベクトルの単位ベクトル算出																				//
//																												//
//	引　数	：	pV		  - 単位ベクトルを算出するベクトル情報のアドレス										//
//				pR		  - 算出した単位ベクトルを格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 算出した単位ベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dNormal(PCAJC3DVEC pV , PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	rsu;
	double		len;

	if (pV != NULL && pR != NULL) {
		len   = AjcV3dLength(pV);
		if (len != 0.0) {
			rsu.x = pV->x / len;
			rsu.y = pV->y / len;
			rsu.z = pV->z / len;
			rc = (PAJC3DVEC)memcpy(pR, &rsu, sizeof *pR);
		}
		else {
			rc = (PAJC3DVEC)memset(pR, 0, sizeof *pR);
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄベクトルの角度算出																						//
//																												//
//	引　数	：	pV1, pV2  - ベクトルの角度を算出する２つのベクトル情報のアドレス								//
//																												//
//	戻り値	：	≠ -1 : ベクトル角度（0～180度，但し、ゼロベクトルが指定された場合は－１を返す）				//
//				＝ -1 : ゼロベクトル／パラメタエラー															//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV3dTheta(PCAJC3DVEC pV1, PCAJC3DVEC pV2)
{
	double	rc = -1.0;

	if (pV1 != NULL && pV2 != NULL) {
		if ((pV1->x != 0.0 || pV1->y != 0.0 || pV1->z != 0.0) && (pV2->x != 0.0 || pV2->y != 0.0 || pV2->z != 0.0)) {
			rc = AjcV3dInner(pV1, pV2) / (AjcV3dLength(pV1) * AjcV3dLength(pV2));
			rc = acos(SubAdjustPM1(rc));
			rc = AJC_RAD2DEG(rc);
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄポイントから直線へ垂直な方向ベクトル算出																//
//																												//
//	引　数	：	pL		  - ライン情報のアドレス（線分の始点と方向ベクトル）									//
//				pP		  - ポイント情報のアドレス																//
//				pR		  - 算出した方向ベクトルベクトルを格納するバッファのアドレス							//
//																												//
//	戻り値	：	≠NULL ; 算出した方向ベクトルベクトルを格納したバッファのアドレス(=pR)							//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dVertVecP2L(PCAJC3DLVEC pL , PCAJC3DVEC pP , PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	t1;

	if (pL != NULL && pP != NULL && pR != NULL) {
		AjcV3dSub(pP, &pL->p, &t1);
		AjcV3dOuter(&pL->v, &t1, &t1);
		AjcV3dOuter(&pL->v, &t1, &t1);
		rc = (PAJC3DVEC)memcpy(pR, &t1, sizeof *pR);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄポイントから直線までの長さを算出																		//
//																												//
//	引　数	：	pL		  - ライン情報のアドレス（線分の始点と方向ベクトル）									//
//				pP		  - ポイント情報のアドレス																//
//				pR		  - ポイントからラインへの方向ベクトルを格納するバッファのアドレス（不要時はNULL)		//
//																												//
//	戻り値	：	≠ -1 : ポイントからライン上のポイントまでの長さ												//
//				＝ -1 : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV3dDistP2L(PCAJC3DLVEC pL , PCAJC3DVEC pP , PAJC3DVEC pR)
{
	double		rc = -1;
	AJC3DVEC	t1, t2;

	if (pL != NULL && pP != NULL) {
		AjcV3dVertVecP2L(pL, pP, &t1);
		AjcV3dNormal(&t1, &t1);
		AjcV3dSub(&pL->p, pP, &t2);
		rc = AjcV3dInner(&t1, &t2);
		if (pR != NULL) {
			pR->x = t1.x * rc;
			pR->y = t1.y * rc;
			pR->z = t1.z * rc;
		}
	}
	return rc;
}
//==============================================================================================================//
//	２つの３Ｄポイント間の距離算出																				//
//																												//
//	引　数	：	pV1, pV2  - 長さを算出するベクトル情報のアドレス												//
//																												//
//	戻り値	：	≠ -1 : 算出したベクトルの長さ																	//
//				＝ -1 : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV3dDistP2P(PCAJC3DVEC pV1, PCAJC3DVEC pV2)
{
	double	rc = -1;
	double	xl, yl, zl;

	if (pV1 != NULL && pV2 != NULL) {
		xl = pV2->x - pV1->x;
		yl = pV2->y - pV1->y;
		zl = pV2->z - pV1->z;
		rc = sqrt(xl * xl + yl * yl + zl * zl);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄラインの交点算出																						//
//																												//
//	引　数	：	pL1, pL2  - 交点を算出する２つのライン情報（線分の始点と方向ベクトル）のアドレス				//
//				pR		  - 算出した交点を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 交点を格納したバッファのアドレス(=pR)													//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dCrossL2L(PCAJC3DLVEC pL1, PCAJC3DLVEC pL2, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	t1;
	double		cost;
	double		l;

	if (pL1 != NULL && pL2 != NULL && pR != NULL) {
		l = AjcV3dDistP2L(pL1, &pL2->p, &t1);
		if (memcmp(pL1, pL2, sizeof *pL1) != 0) {
			if (memcmp(&pL1->p, &pL2->p, sizeof pL1->p) != 0) {
				cost = AjcV3dInner(&pL2->v, &t1) / (l * AjcV3dLength(&pL2->v));
				l *= (1.0 / cost);
				AjcV3dNormal(&pL2->v, &t1);
				t1.x *= l;
				t1.y *= l;
				t1.z *= l;
				AjcV3dAdd(&t1, &pL2->p, pR);
			}
			else {
				memcpy(pR, &pL1->p, sizeof *pR);
			}
		}
		else {
			memcpy(pR, &pL1->p, sizeof *pR);
		}
		rc = pR;
	}

	return rc;
}
//==============================================================================================================//
//	３Ｄラインの交点算出（交差しない場合は、最も接近する点の中点を求める）										//
//																												//
//	引　数	：	pL1, pL2  - 交点を算出する２つのライン情報（線分の始点と方向ベクトル）のアドレス				//
//				pR		  - 算出した交点を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 交点を格納したバッファのアドレス(=pR)													//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dCrossL2LEx(PCAJC3DLVEC pL1, PCAJC3DLVEC pL2, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	double		a, b1, b2, c1, c2, s, t;

	if (pL1 != NULL && pL2 != NULL && pR != NULL) {
		a  = (pL1->v.x * pL2->v.x) + (pL1->v.y * pL2->v.y) + (pL1->v.z * pL2->v.z);
		b1 = (pL1->v.x * pL1->v.x) + (pL1->v.y * pL1->v.y) + (pL1->v.z * pL1->v.z);
		b2 = (pL2->v.x * pL2->v.x) + (pL2->v.y * pL2->v.y) + (pL2->v.z * pL2->v.z);
		c1 = ((pL2->p.x - pL1->p.x) * pL1->v.x) + ((pL2->p.y - pL1->p.y) * pL1->v.y) + ((pL2->p.z -pL1->p.z) * pL1->v.z);
		c2 = ((pL2->p.x - pL1->p.x) * pL2->v.x) + ((pL2->p.y - pL1->p.y) * pL2->v.y) + ((pL2->p.z -pL1->p.z) * pL2->v.z);

		s = (-c1 * a  + c2 * b1) / ((a * a) - (b1 * b2));
		t = (-c1 * b2 + c2 * a ) / ((a * a) - (b1 * b2));

		pR->x = ((pL1->p.x + pL1->v.x * t) + (pL2->p.x + pL2->v.x * s)) / 2.0;
		pR->y = ((pL1->p.y + pL1->v.y * t) + (pL2->p.y + pL2->v.y * s)) / 2.0;
		pR->z = ((pL1->p.z + pL1->v.z * t) + (pL2->p.z + pL2->v.z * s)) / 2.0;

		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・点からの垂線と平面の交点を求める																		//
//																												//
//	引　数	：	pL		  - 平面の点から垂直な線ベクトル情報のアドレス											//
//				pP		  - ポイント情報のアドレス																//
//				pR		  - 交点情報を格納するバッファのアドレス												//
//																												//
//	戻り値	：	≠NULL : 交点情報を格納したバッファのアドレス(=pR)												//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dCrossP2F(PCAJC3DLVEC pL, PCAJC3DVEC pP, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	v1;
	AJC3DLVEC	lv;
	AJC3DVEC	p;

	if (pL != NULL && pP != NULL && pR != NULL) {
		memcpy(&lv.p, pP, sizeof lv.p);
		AjcV3dVertVecP2L(pL , pP , &lv.v);
		AjcV3dCrossL2L(pL, &lv, &p);
		AjcV3dSub(&pL->p, &p, &v1);
		AjcV3dAdd(pP, &v1, pR);
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	同一平面上で、線分に直行する方向ベクトル算出																//
//																												//
//	引　数	：	pL		  - 平面上の線分情報のアドレス（平面上の始点と終点）									//
//				pV		  - 平面からの法線ベクトル情報のアドレス												//
//				pR		  - 同一平面上で直行する方向ベクトル情報を格納するバッファのアドレス					//
//																												//
//	戻り値	：	≠NULL : 同一平面上で直行する方向ベクトル情報を格納したバッファのアドレス(=pR)					//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dOrthoVecOnPlane(PCAJC3DLINE pL, PCAJC3DVEC pV, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	t1;

	if (pL != NULL && pV != NULL && pR != NULL) {
		AjcV3dSub(&pL->p2, &pL->p1, &t1);
		AjcV3dOuter(&t1, pV, pR);
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・ベクトルと行列の掛け算																				//
//																												//
//	引　数	：	pV		  - 被乗数ベクトル（ベクトル）															//
//				pM		  - 乗数（行列）																		//
//				pR		  - 乗算結果のベクトルを格納するバッファのアドレス										//
//																												//
//	戻り値	：	≠NULL : 乗算結果のベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dMultMat(PCAJC3DVEC	 pV, PCAJC3DMAT pM, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	v;

	if (pV != NULL && pM != NULL && pR != NULL) {
		v.x = pM->m[0][0] * pV->x + pM->m[0][1] * pV->y + pM->m[0][2] * pV->z;
		v.y = pM->m[1][0] * pV->x + pM->m[1][1] * pV->y + pM->m[1][2] * pV->z;
		v.z = pM->m[2][0] * pV->x + pM->m[2][1] * pV->y + pM->m[2][2] * pV->z;
		rc = (PAJC3DVEC)memcpy(pR, &v, sizeof *pR);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・ベクトルをＸ軸周りに回転																				//
//																												//
//	引　数	：	pV		  - 回転するベクトル																	//
//				t		  - 回転角度（度）																		//
//				pR		  - 回転結果のベクトルを格納するバッファのアドレス										//
//																												//
//	戻り値	：	≠NULL : 回転結果のベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dRotateX(PCAJC3DVEC pV, double t, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DMAT	mat;
	double		s, c;

	if (pV != NULL && pR != NULL) {
		t = AJC_DEG2RAD(t);
		s = sin(t);
		c = cos(t);

		mat.m[0][0] =  1;	mat.m[0][1] =  0;	mat.m[0][2] =  0;
		mat.m[1][0] =  0;	mat.m[1][1] =  c;	mat.m[1][2] = -s;
		mat.m[2][0] =  0;	mat.m[2][1] =  s;	mat.m[2][2] =  c;

		AjcV3dMultMat(pV, &mat, pR);

		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・ベクトルをＹ軸周りに回転																				//
//																												//
//	引　数	：	pV		  - 回転するベクトル																	//
//				t		  - 回転角度（度）																		//
//				pR		  - 回転結果のベクトルを格納するバッファのアドレス										//
//																												//
//	戻り値	：	≠NULL : 回転結果のベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dRotateY(PCAJC3DVEC pV, double t, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DMAT	mat;
	double		s, c;

	if (pV != NULL && pR != NULL) {
		t = AJC_DEG2RAD(t);
		s = sin(t);
		c = cos(t);

		mat.m[0][0] =  c;	mat.m[0][1] =  0;	mat.m[0][2] =  s;
		mat.m[1][0] =  0;	mat.m[1][1] =  1;	mat.m[1][2] =  0;
		mat.m[2][0] = -s;	mat.m[2][1] =  0;	mat.m[2][2] =  c;

		AjcV3dMultMat(pV, &mat, pR);

		rc = pR;
	}

	return rc;
}
//==============================================================================================================//
//	３Ｄ・ベクトルをＺ軸周りに回転																				//
//																												//
//	引　数	：	pV		  - 回転するベクトル																	//
//				t		  - 回転角度（度）																		//
//				pR		  - 回転結果のベクトルを格納するバッファのアドレス										//
//																												//
//	戻り値	：	≠NULL : 回転結果のベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dRotateZ(PCAJC3DVEC	 pV, double t, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DMAT	mat;
	double		s, c;

	if (pV != NULL && pR != NULL) {
		t = AJC_DEG2RAD(t);
		s = sin(t);
		c = cos(t);

		mat.m[0][0] =  c;	mat.m[0][1] = -s;	mat.m[0][2] =  0;
		mat.m[1][0] =  s;	mat.m[1][1] =  c;	mat.m[1][2] =  0;
		mat.m[2][0] =  0;	mat.m[2][1] =  0;	mat.m[2][2] =  1;

		AjcV3dMultMat(pV, &mat, pR);

		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・ベクトルを任意の軸周りに回転																			//
//																												//
//	引　数	：	pV		  - 回転するベクトル																	//
//				t		  - 回転角度（度）																		//
//				pAny	  - 任意の回転軸ベクトル																//
//				pR		  - 回転結果のベクトルを格納するバッファのアドレス										//
//																												//
//	戻り値	：	≠NULL : 回転結果のベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dRotateAny(PCAJC3DVEC  pV, double t, PCAJC3DVEC pAny, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DMAT	mat;
	AJC3DVEC	v;
	double	k, s, c;

	if (pV != NULL && pAny != NULL && pR != NULL) {
		t = AJC_DEG2RAD(t);
		s = sin(t);
		c = cos(t);
		k = 1.0 - c;

		memcpy(&v, pAny, sizeof v);
		AjcV3dNormal(&v, &v);

		mat.m[0][0] = v.x * v.x * k + c;		mat.m[0][1] = v.x * v.y * k - v.z * s;	mat.m[0][2] = v.z * v.x * k + v.y * s;
		mat.m[1][0] = v.x * v.y * k + v.z * s;	mat.m[1][1] = v.y * v.y * k + c;		mat.m[1][2] = v.y * v.z * k - v.x * s;
		mat.m[2][0] = v.z * v.x * k - v.y * s;	mat.m[2][1] = v.y * v.z * k + v.x * s;	mat.m[2][2] = v.z * v.z * k + c;

		AjcV3dMultMat(pV, &mat, pR);

		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・任意の直角なベクトルを求める																			//
//																												//
//	引　数	：	pV		  - ベクトル																			//
//				pR		  - ベクトルに直角な任意のベクトルを格納するバッファのアドレス							//
//																												//
//	戻り値	：	≠NULL : ベクトルに直角な任意のベクトルを格納したバッファのアドレス(=pR)						//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dAnyOrthoVec(PCAJC3DVEC pV, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	v;
	int			sel;

	if (pV != NULL && pR != NULL) {
		sel = (((pV->x != 0) << 2) | ((pV->y != 0) << 1) | (pV->z != 0));
		v.x = v.y = v.z = 0;
		switch (sel) {
					/* xyz */
			case 0:	/* 000 */													break;
			case 1:	/* 001 */	v.x = pV->z;									break;
			case 2:	/* 010 */									v.z =  pV->y;	break;
			case 3:	/* 011 */					v.y =  pV->z;	v.z = -pV->y;	break;
			case 4:	/* 100 */					v.y =  pV->x;					break;
			case 5:	/* 101 */	v.x = pV->z;					v.z = -pV->x;	break;
			case 6:	/* 110 */	v.x = pV->y;	v.y = -pV->x;					break;
			case 7:	/* 111 */	v.x = pV->x;	v.y =  pV->y;
								v.z = -((pV->x * pV->x + pV->y * pV->y) / pV->z);
								AjcV3dNormal(&v, &v);
								AjcV3dMult(&v, AjcV3dLength(pV), &v);
								break;
		}
		rc = (PAJC3DVEC)memcpy(pR, &v, sizeof *pR);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・平面上の任意の点を平面上で指定角度回転した座標を求める												//
//																												//
//	引　数	：	pP		  - 回転する平面上の点データのアドレス													//
//				pC		  - 平面の中心と法線情報のアドレス														//
//				theta	  - 回転角度（度）																		//
//				pR		  - 回転後の座標を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 回転後の座標を格納したバッファのアドレス(=pR)											//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC3DVEC	WINAPI AjcV3dRotateOnPlane(PCAJC3DVEC pP, PCAJC3DLVEC pC, double theta, PAJC3DVEC pR)
{
	PAJC3DVEC	rc = NULL;
	AJC3DVEC	v;

	if (pP != NULL && pC != NULL && pR != NULL) {
		memcpy(&v, pP, sizeof(AJC3DVEC));
		AjcV3dSub(&v, &pC->p, &v);
		AjcV3dRotateAny(&v, theta, &pC->v, &v);
		AjcV3dAdd(&v, &pC->p, pR);
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・任意の３点を通る円の中心と半径を求める																//
//																												//
//	引　数	：	p1, p2, p3	- 任意の３点																		//
//				pR			- 円の中心を格納するバッファのアドレス（不要時はNULL)								//
//																												//
//	戻り値	：	≠ -1  : 円の半径																				//
//				＝ -1  : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV3dCalcCircle(double x1, double y1, double z1,
											double x2, double y2, double z2,
											double x3, double y3, double z3, PAJC3DVEC pR, PAJC3DVEC pV)
{
	AJC3DVEC	p1, p2, p3;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	p3.x = x3; p3.y = y3; p3.z = z3;
	return AjcV3dCalcCircleVEx(&p1, &p2, &p3, pR, pV, NULL);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcV3dCalcCircleEx(double x1, double y1, double z1,
											  double x2, double y2, double z2,
											  double x3, double y3, double z3, PAJC3DVEC pR, PAJC3DVEC pV, PAJC3DCIRINFO pCir)
{
	AJC3DVEC	p1, p2, p3;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	p3.x = x3; p3.y = y3; p3.z = z3;
	return AjcV3dCalcCircleVEx(&p1, &p2, &p3, pR, pV, pCir);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcV3dCalcCircleV(PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PAJC3DVEC pR, PAJC3DVEC pV)
{
	return AjcV3dCalcCircleVEx(p1, p2, p3, pR, pV, NULL);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcV3dCalcCircleVEx(PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PAJC3DVEC pR, PAJC3DVEC pV, PAJC3DCIRINFO pCir)
{
	double		rc = -1;		//	円の半径
	double		L12, L13, L23;	//	３つの線分の長さ
	double		t;				//	２つのラインベクトルの開き角度
	AJC3DLVEC	lvA, lvB;		//	２つのラインベクトル
	AJC3DVEC	pcA, pcB;		//	２つのラインの中点
	AJC3DVEC	vh;				//	平面を表す法線
	AJC3DVEC	pc;				//	２つの直行ラインの交点(円の中心点）

	if (p1 != NULL && p2 != NULL && p3 != NULL) {
		if (memcmp(p1, p2, sizeof(AJC3DVEC)) != 0 && memcmp(p1, p3, sizeof(AJC3DVEC)) != 0 && memcmp(p2, p3, sizeof(AJC3DVEC)) != 0) {
			//	３つの線分の長さ設定
			L12 = AjcV3dDistP2P(p1, p2);
			L13 = AjcV3dDistP2P(p1, p3);
			L23 = AjcV3dDistP2P(p2, p3);
			//	最小線分を除く２つのラインベクトル設定(lvA, lvB)
			if		(L12 >= L13 && L12 >= L23) {
				AjcV3dSetLineVec(p1, p2, &lvA);
				if (L13 >= L23) AjcV3dSetLineVec(p1, p3, &lvB);
				else			AjcV3dSetLineVec(p2, p3, &lvB);
			}
			else if (L13 >= L12 && L13 >= L23) {
				AjcV3dSetLineVec(p1, p3, &lvA);
				if (L12 >= L23) AjcV3dSetLineVec(p1, p2, &lvB);
				else	 		AjcV3dSetLineVec(p2, p3, &lvB);
			}
			else if (L23 >= L12 && L23 >= L13) {
				AjcV3dSetLineVec(p2, p3, &lvA);
				if (L12 >= L13) AjcV3dSetLineVec(p1, p2, &lvB);
				else			AjcV3dSetLineVec(p1, p3, &lvB);
			}
			//	２つのベクトルの開き角度チェック
			t = AjcV3dTheta(&lvA.v, &lvB.v);
			if (t != 0.0 && t != 180.0) {
				//	２つの線分から法線算出(vh)
				AjcV3dOuter(&lvA.v, &lvB.v, &vh);
				if (pV != NULL) {
					memcpy(pV, &vh, sizeof(AJC3DVEC));
				}
				//	２つのラインの中点算出
				AjcV3dLVecCenter(&lvA, &pcA);
				AjcV3dLVecCenter(&lvB, &pcB);
				//	戻り情報(1)
				if (pCir != NULL) {
					//	円に内接する２つの直線
					memcpy(&pCir->lt1.p1, &lvA.p, sizeof(AJC3DVEC)); AjcV3dAdd(&lvA.p, &lvA.v, &pCir->lt1.p2);
					memcpy(&pCir->lt2.p1, &lvB.p, sizeof(AJC3DVEC)); AjcV3dAdd(&lvB.p, &lvB.v, &pCir->lt2.p2);
				}
				//	同一平面上で２つのラインに直行するベクトル算出
				AjcV3dOuter(&lvA.v, &vh, &lvA.v);
				AjcV3dOuter(&lvB.v, &vh, &lvB.v);
				//	直行ラインの始点を２つの線分の中点とする
				memcpy(&lvA.p, &pcA, sizeof(AJC3DVEC));
				memcpy(&lvB.p, &pcB, sizeof(AJC3DVEC));
				//	直行ラインの交点（円の中心）算出
				AjcV3dCrossL2L(&lvA, &lvB, &pc);
				if (pR != NULL) {
					memcpy(pR, &pc, sizeof(AJC3DVEC));
				}
				//	円の半径算出
				rc = AjcV3dDistP2P(&pc, p1);
				//	戻り情報(2)
				if (pCir != NULL) {
					//	円に内接する２つの直線の中点から円中心への直線
					memcpy(&pCir->lc1.p1, &pcA, sizeof(AJC3DVEC)); memcpy(&pCir->lc1.p2, &pc, sizeof(AJC3DVEC));
					memcpy(&pCir->lc2.p1, &pcB, sizeof(AJC3DVEC)); memcpy(&pCir->lc2.p2, &pc, sizeof(AJC3DVEC));
					//	円の中心，法線と半径
					memcpy(&pCir->lvc.p, &pc, sizeof(AJC3DVEC));
					memcpy(&pCir->lvc.v, &vh, sizeof(AJC3DVEC));
					pCir->cr = rc;
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ・球面上の任意の４点から球の中心と半径を求める															//
//																												//
//	引　数	：	p1, p2, p3, p4	- 任意の４点																	//
//				pR				- 円の中心を格納するバッファのアドレス（不要時はNULL)							//
//																												//
//	戻り値	：	≠ -1  : 球の半径																				//
//				＝ -1  : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV3dCalcSphere(double x1, double y1, double z1,
											double x2, double y2, double z2,
											double x3, double y3, double z3,
											double x4, double y4, double z4, PAJC3DVEC pR)
{
	AJC3DVEC	p1, p2, p3, p4;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	p3.x = x3; p3.y = y3; p3.z = z3;
	p4.x = x4; p4.y = y4; p4.z = z4;
	return AjcV3dCalcSphereVEx(&p1, &p2, &p3, &p4, pR, NULL);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcV3dCalcSphereEx(double x1, double y1, double z1,
											  double x2, double y2, double z2,
											  double x3, double y3, double z3,
											  double x4, double y4, double z4, PAJC3DVEC pR, PAJC3DSPHINFO pSph)
{
	AJC3DVEC	p1, p2, p3, p4;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	p3.x = x3; p3.y = y3; p3.z = z3;
	p4.x = x4; p4.y = y4; p4.z = z4;
	return AjcV3dCalcSphereVEx(&p1, &p2, &p3, &p4, pR, pSph);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcV3dCalcSphereV(PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PCAJC3DVEC p4, PAJC3DVEC pR)
{
	return AjcV3dCalcSphereVEx(p1, p2, p3, p4, pR, NULL);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcV3dCalcSphereVEx(PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PCAJC3DVEC p4, PAJC3DVEC pR, PAJC3DSPHINFO pSph)
{
	double		rc = -1;
	AJC3DLVEC	lv1, lv2;		//	２つの円の中心からの垂線ベクトル
	double		cr1, cr2;		//	２つの円の半径
	AJC3DVEC	pc;				//	球の中心

	if (p1 != NULL && p2 != NULL && p3 != NULL && p4 != NULL) {
		if (memcmp(p1, p2, sizeof(AJC3DVEC)) != 0 && memcmp(p1, p3, sizeof(AJC3DVEC)) != 0 && memcmp(p1, p4, sizeof(AJC3DVEC)) != 0 &&
			memcmp(p2, p3, sizeof(AJC3DVEC)) != 0 && memcmp(p2, p4, sizeof(AJC3DVEC)) != 0 && memcmp(p3, p4, sizeof(AJC3DVEC)) != 0) {
			//	２つの内接円算出
			if (pSph == NULL) {
				cr1 = AjcV3dCalcCircleVEx(p1, p2, p3, &lv1.p, &lv1.v, NULL);
				cr2 = AjcV3dCalcCircleVEx(p2, p3, p4, &lv2.p, &lv1.v, NULL);
			}
			else {
				cr1 = AjcV3dCalcCircleVEx(p1, p2, p3, &lv1.p, &lv1.v, &pSph->cif1);
				cr2 = AjcV3dCalcCircleVEx(p2, p3, p4, &lv2.p, &lv2.v, &pSph->cif2);
			}
			//	２つの円算出結果チェック
			if (cr1 != -1 && cr2 != -1) {
				//	２つの円中心からの垂線の交点算出（球の中心）
				AjcV3dCrossL2L(&lv1, &lv2, &pc);
				if (pR != NULL) {
					memcpy(pR, &pc, sizeof(AJC3DVEC));
				}
				//	球の半径算出
				rc = AjcV3dDistP2P(p1, &pc);
				//	戻り情報
				if (pSph != NULL) {
					//	内接円中心から球中心への直線
					memcpy(&pSph->ls1.p1, &lv1.p, sizeof(AJC3DVEC));	memcpy(&pSph->ls1.p2, &pc, sizeof(AJC3DVEC));
					memcpy(&pSph->ls2.p1, &lv2.p, sizeof(AJC3DVEC));	memcpy(&pSph->ls2.p2, &pc, sizeof(AJC3DVEC));
					//	球の中心と半径
					memcpy(&pSph->sc, &pc, sizeof(AJC3DVEC));
					pSph->sr = rc;
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄベクトル加算																							//
//																												//
//	引　数	：	pV1, pV2  - 加算するベクトル情報のアドレス														//
//				pR		  - 加算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 加算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC WINAPI AjcV2dAdd(PCAJC2DVEC pV1, PCAJC2DVEC pV2, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV1 != NULL && pV2 != NULL && pR != NULL) {
		rsu.x = pV1->x + pV2->x;
		rsu.y = pV1->y + pV2->y;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PAJC2DVEC WINAPI AjcVPtAdd(LPPOINT pV1, LPPOINT pV2, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV1 != NULL && pV2 != NULL && pR != NULL) {
		rsu.x = pV1->x + pV2->x;
		rsu.y = pV1->y + pV2->y;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄベクトル減算																							//
//																												//
//	引　数	：	pV1, pV2  - 減算するベクトル情報のアドレス														//
//				pR		  - 減算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 減算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC WINAPI AjcV2dSub(PCAJC2DVEC pV1, PCAJC2DVEC pV2, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV1 != NULL && pV2 != NULL && pR != NULL) {
		rsu.x = pV1->x - pV2->x;
		rsu.y = pV1->y - pV2->y;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PAJC2DVEC WINAPI AjcVPtSub(LPPOINT pV1, LPPOINT pV2, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV1 != NULL && pV2 != NULL && pR != NULL) {
		rsu.x = (double)pV1->x - (double)pV2->x;
		rsu.y = (double)pV1->y - (double)pV2->y;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//==============================================================================================================//
//	２Ｄベクトル乗算（ベクトルをｎ倍する）																		//
//																												//
//	引　数	：	pV		  - 乗算するベクトル情報のアドレス														//
//				n		  - 乗算する値																			//
//				pR		  - 乗算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 乗算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC WINAPI AjcV2dMult(PCAJC2DVEC pV, double n, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV != NULL && pR != NULL) {
		rsu.x = pV->x * n;
		rsu.y = pV->y * n;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PAJC2DVEC WINAPI AjcVptMult(LPPOINT pV, double n, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV != NULL && pR != NULL) {
		rsu.x = (double)pV->x * n;
		rsu.y = (double)pV->y * n;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//==============================================================================================================//
//	２Ｄベクトルの除算																							//
//																												//
//	引　数	：	pV		  - 除算するベクトル情報のアドレス														//
//				n		  - 除算する値																			//
//				pR		  - 除算結果のベクトル情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 除算結果のベクトル情報を格納したバッファのアドレス(=pR)								//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC	WINAPI AjcV2dDiv(PCAJC2DVEC pV, double n, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV != NULL && pR != NULL) {
		rsu.x = pV->x / n;
		rsu.y = pV->y / n;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PAJC2DVEC	WINAPI AjcVPtDiv(LPPOINT pV, double n, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;

	if (pV != NULL && pR != NULL) {
		rsu.x = (double)pV->x / n;
		rsu.y = (double)pV->y / n;
		rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
	}
	return pR;
}
//==============================================================================================================//
//	２Ｄベクトルの長さ算出																						//
//																												//
//	引　数	：	pV		  - 長さを算出するベクトル情報のアドレス												//
//																												//
//	戻り値	：	≠ -1 : ベクトルの長さ																			//
//				＝ -1 : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV2dLength(PCAJC2DVEC pV)
{
	double rc = -1;

	if (pV != NULL) {
		rc = sqrt(pV->x * pV->x + pV->y * pV->y);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcVPtLength(LPPOINT pV)
{
	double rc = -1;

	if (pV != NULL) {
		rc = sqrt((double)pV->x * (double)pV->x + (double)pV->y * (double)pV->y);
	}
	return rc;
}

//==============================================================================================================//
//	２Ｄベクトルの外積算出																						//
//																												//
//	引　数	：	pV1, pV2  - 外積を算出するベクトル情報のアドレス												//
//				pR		  - 算出した外積を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 外積を格納したバッファのアドレス(=pR)													//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double WINAPI AjcV2dOuter(PCAJC3DVEC pV1, PCAJC2DVEC pV2)
{
	double		rc = 0;

	if (pV1 != NULL && pV2 != NULL) {
		rc = pV1->x * pV2->y - pV1->y * pV2->x;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double WINAPI AjcVPtOuter(LPPOINT pV1, LPPOINT pV2)
{
	double		rc = 0;

	if (pV1 != NULL && pV2 != NULL) {
		rc = (double)pV1->x * (double)pV2->y - (double)pV1->y * (double)pV2->x;
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄベクトルの内積算出																						//
//																												//
//	引　数	：	pV1, pV2  - 内積を算出するベクトル情報のアドレス												//
//																												//
//	戻り値	：	算出した内積値（パラメタエラー時は０を返す）													//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV2dInner(PCAJC2DVEC pV1, PCAJC2DVEC pV2)
{
	double	rc = 0.0;

	if (pV1 != NULL && pV2 != NULL) {
		rc = pV1->x * pV2->x + pV1->y * pV2->y;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcVPtInner(LPPOINT pV1, LPPOINT pV2)
{
	double	rc = 0.0;

	if (pV1 != NULL && pV2 != NULL) {
		rc = (double)pV1->x * (double)pV2->x + (double)pV1->y * (double)pV2->y;
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄベクトルの単位ベクトル算出																				//
//																												//
//	引　数	：	pV		  - 単位ベクトルを算出するベクトル情報のアドレス										//
//				pR		  - 算出した単位ベクトルを格納するバッファのアドレス									//
//																												//
//	戻り値	：	≠NULL : 算出した単位ベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC	WINAPI AjcV2dNormal(PCAJC2DVEC pV , PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;
	double		len;

	if (pV != NULL && pR != NULL) {
		len   = AjcV2dLength(pV);
		if (len != 0.0) {
			rsu.x = pV->x / len;
			rsu.y = pV->y / len;
			rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
		}
		else {
			rc = (PAJC2DVEC)memset(pR, 0, sizeof *pR);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PAJC2DVEC	WINAPI AjcVPtNormal(LPPOINT pV , PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	rsu;
	double		len;

	if (pV != NULL && pR != NULL) {
		len   = AjcVPtLength(pV);
		if (len != 0.0) {
			rsu.x = (double)pV->x / len;
			rsu.y = (double)pV->y / len;
			rc = (PAJC2DVEC)memcpy(pR, &rsu, sizeof *pR);
		}
		else {
			rc = (PAJC2DVEC)memset(pR, 0, sizeof *pR);
		}
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄベクトルの角度算出																						//
//																												//
//	引　数	：	pV1, pV2  - ベクトルの角度を算出する２つのベクトル情報のアドレス								//
//																												//
//	戻り値	：	≠ -1 : ベクトル角度（0～180度，但し、ゼロベクトルが指定された場合は－１を返す）				//
//				＝ -1 : ゼロベクトル／パラメタエラー															//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV2dTheta(PCAJC2DVEC pV1, PCAJC2DVEC pV2)
{
	double	rc = -1.0;

	if (pV1 != NULL && pV2 != NULL) {
		if ((pV1->x != 0.0 || pV1->y != 0.0) && (pV2->x != 0.0 || pV2->y != 0.0)) {
			rc = AjcV2dInner(pV1, pV2) / (AjcV2dLength(pV1) * AjcV2dLength(pV2));
			rc = acos(SubAdjustPM1(rc));
			rc = AJC_RAD2DEG(rc);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcVPtTheta(LPPOINT pV1, LPPOINT pV2)
{
	double	rc = -1.0;

	if (pV1 != NULL && pV2 != NULL) {
		if ((pV1->x != 0 || pV1->y != 0) && (pV2->x != 0 || pV2->y != 0)) {
			rc = AjcVPtInner(pV1, pV2) / (AjcVPtLength(pV1) * AjcVPtLength(pV2));
			rc = acos(SubAdjustPM1(rc));
			rc = AJC_RAD2DEG(rc);
		}
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄポイントから直線へ垂直な方向ベクトル算出																//
//																												//
//	引　数	：	pL		  - ライン情報のアドレス（線分の始点と方向ベクトル）									//
//				pP		  - ポイント情報のアドレス																//
//				pR		  - 算出した方向ベクトルベクトルを格納するバッファのアドレス							//
//																												//
//	戻り値	：	≠NULL ; 算出した方向ベクトルベクトルを格納したバッファのアドレス(=pR)							//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC	WINAPI AjcV2dVertVecP2L(PCAJC2DLVEC pL , PCAJC2DVEC pP , PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC3DLVEC	vl;
	AJC3DVEC	vp;
	AJC3DVEC	vo;

	if (pL != NULL && pP != NULL && pR != NULL) {
		vl.p.x = pL->p.x;  vl.p.y = pL->p.y;  vl.p.z = 0;
		vl.v.x = pL->v.x;  vl.v.y = pL->v.y;  vl.v.z = 0;
		vp.x   = pP->x;    vp.y   = pP->y;	  vp.z	 = 0;

		AjcV3dSub  (&vp  , &vl.p, &vo);
		AjcV3dOuter(&vl.v, &vo	, &vo);
		AjcV3dOuter(&vl.v, &vo	, &vo);

		pR->x = vo.x;
		pR->y = vo.y;
		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄポイントから直線までの長さを算出																		//
//																												//
//	引　数	：	pL		  - ライン情報のアドレス（線分の始点と方向ベクトル）									//
//				pP		  - ポイント情報のアドレス																//
//				pR		  - ポイントからラインへの方向ベクトルを格納するバッファのアドレス（不要時はNULL)		//
//																												//
//	戻り値	：	≠ -1 : ポイントからライン上のポイントまでの長さ												//
//				＝ -1 : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV2dDistP2L(PCAJC2DLVEC pL , PCAJC2DVEC pP , PAJC2DVEC pR)
{
	double		rc = -1;
	AJC3DLVEC	vl;
	AJC3DVEC	vp;
	AJC3DVEC	t1, t2;

	if (pL != NULL && pP != NULL) {
		vl.p.x = pL->p.x;  vl.p.y = pL->p.y;  vl.p.z = 0;
		vl.v.x = pL->v.x;  vl.v.y = pL->v.y;  vl.v.z = 0;
		vp.x   = pP->x;    vp.y   = pP->y;	  vp.z	 = 0;

		AjcV3dVertVecP2L(&vl, &vp, &t1);
		AjcV3dNormal(&t1, &t1);
		AjcV3dSub(&vl.p, &vp, &t2);
		rc = AjcV3dInner(&t1, &t2);
		if (pR != NULL) {
			pR->x = t1.x * rc;
			pR->y = t1.y * rc;
		}
	}
	return rc;
}
//==============================================================================================================//
//	２つの２Ｄポイント間の距離算出																				//
//																												//
//	引　数	：	pV1, pV2  - 長さを算出するベクトル情報のアドレス												//
//																												//
//	戻り値	：	≠ -1 : 算出したベクトルの長さ																	//
//				＝ -1 : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	double	WINAPI AjcV2dDistP2P(PCAJC2DVEC pV1, PCAJC2DVEC pV2)
{
	double	rc = -1;
	double	xl, yl;

	if (pV1 != NULL && pV2 != NULL) {
		xl = pV2->x - pV1->x;
		yl = pV2->y - pV1->y;
		rc = sqrt(xl * xl + yl * yl);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	double	WINAPI AjcVPtDistP2P(LPPOINT pV1, LPPOINT pV2)
{
	double	rc = -1;
	double	xl, yl;

	if (pV1 != NULL && pV2 != NULL) {
		xl = (double)pV2->x - (double)pV1->x;
		yl = (double)pV2->y - (double)pV1->y;
		rc = sqrt(xl * xl + yl * yl);
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄラインの交点算出																						//
//																												//
//	引　数	：	pL1, pL2  - 交点を算出する２つのライン情報（線分の始点と方向ベクトル）のアドレス				//
//				pR		  - 算出した交点を格納するバッファのアドレス											//
//																												//
//	戻り値	：	≠NULL : 交点を格納したバッファのアドレス(=pR)													//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC	WINAPI AjcV2dCrossL2L(PCAJC2DLVEC pL1, PCAJC2DLVEC pL2, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	t1;
	double		cost;
	double		l;

	if (pL1 != NULL && pL2 != NULL && pR != NULL) {
		l = AjcV2dDistP2L(pL1, &pL2->p, &t1);
		if (memcmp(pL1, pL2, sizeof *pL1) != 0) {
			if (memcmp(&pL1->p, &pL2->p, sizeof pL1->p) != 0) {
				cost = AjcV2dInner(&pL2->v, &t1) / (l * AjcV2dLength(&pL2->v));
				l *= (1.0 / cost);
				AjcV2dNormal(&pL2->v, &t1);
				t1.x *= l;
				t1.y *= l;
				AjcV2dAdd(&t1, &pL2->p, pR);
			}
			else {
				memcpy(pR, &pL1->p, sizeof *pR);
			}
		}
		else {
			memcpy(pR, &pL1->p, sizeof *pR);
		}
		rc = pR;
	}

	return rc;
}
//==============================================================================================================//
//	２Ｄ・ベクトル回転																							//
//																												//
//	引　数	：	pV		  - 回転するベクトル																	//
//				t		  - 回転角度（度）																		//
//				pR		  - 回転結果のベクトルを格納するバッファのアドレス										//
//																												//
//	戻り値	：	≠NULL : 回転結果のベクトルを格納したバッファのアドレス(=pR)									//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC	WINAPI AjcV2dRotate(PCAJC2DVEC pV, double t, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	double		s, c;
	AJC2DVEC	v;

	if (pV != NULL && pR != NULL) {
		t = AJC_DEG2RAD(t);
		s = sin(t);
		c = cos(t);

		v.x = pV->x * c - pV->y * s;
		v.y = pV->x * s + pV->y * c;
		memcpy(pR, &v, sizeof(AJC2DVEC));

		rc = pR;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PAJC2DVEC	WINAPI AjcVPtRotate(LPPOINT pV, double t, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	double		s, c;

	if (pV != NULL && pR != NULL) {
		t = AJC_DEG2RAD(t);
		s = sin(t);
		c = cos(t);

		pR->x = (double)pV->x * c - (double)pV->y * s;
		pR->y = (double)pV->x * s + (double)pV->y * c;

		rc = pR;
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄ・任意の直角なベクトルを求める																			//
//																												//
//	引　数	：	pV		  - ベクトル																			//
//				pR		  - ベクトルに直角な任意のベクトルを格納するバッファのアドレス							//
//																												//
//	戻り値	：	≠NULL : ベクトルに直角な任意のベクトルを格納したバッファのアドレス(=pR)						//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	PAJC2DVEC	WINAPI AjcV2dAnyOrthoVec(PCAJC2DVEC pV, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	v;
	int			sel;

	if (pV != NULL && pR != NULL) {
		sel = ( ((pV->x != 0) << 1) | (pV->y != 0) );
		v.x = v.y = 0;
		switch (sel) {
					/* xy */
			case 0:	/* 00 */										break;
			case 1:	/* 01 */	v.x =  pV->y;						break;
			case 2:	/* 10 */					v.y =  pV->x;		break;
			case 3:	/* 11 */	v.x = pV->y;	v.y = -pV->x;		break;
								AjcV2dNormal(&v, &v);
								AjcV2dMult(&v, AjcV2dLength(pV), &v);
								break;
		}
		rc = (PAJC2DVEC)memcpy(pR, &v, sizeof *pR);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PAJC2DVEC	WINAPI AjcVPtAnyOrthoVec(LPPOINT pV, PAJC2DVEC pR)
{
	PAJC2DVEC	rc = NULL;
	AJC2DVEC	v;
	int			sel;

	if (pV != NULL && pR != NULL) {
		sel = ( ((pV->x != 0) << 1) | (pV->y != 0) );
		v.x = v.y = 0;
		switch (sel) {
					/* xy */
			case 0:	/* 00 */										break;
			case 1:	/* 01 */	v.x =  pV->y;						break;
			case 2:	/* 10 */					v.y =  pV->x;		break;
			case 3:	/* 11 */	v.x = pV->y;	v.y = -pV->x;		break;
								AjcV2dNormal(&v, &v);
								AjcV2dMult(&v, AjcVPtLength(pV), &v);
								break;
		}
		rc = (PAJC2DVEC)memcpy(pR, &v, sizeof *pR);
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄ・整数型ベクトルへ変換																					//
//																												//
//	引　数	：	pV		  - ベクトル																			//
//				pR		  - ベクトルに直角な任意のベクトルを格納するバッファのアドレス							//
//																												//
//	戻り値	：	≠NULL : 整数型ベクトルを格納したバッファのアドレス(=pR)										//
//				＝NULL : パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	LPPOINT	WINAPI AjcVPtSetIntVec(PAJC2DVEC pV, LPPOINT pR)
{
	LPPOINT	pPt = NULL;

	if (pV != NULL && pR != NULL) {
		pR->x = (int)pV->x;
		pR->y = (int)pV->y;
		pPt   = pR;
	}
	return pPt;
}

//--------------------------------------------------------------------------------------------------------------//
//	数値を－１．０～＋１．０の範囲に調整する																	//
//																												//
//	引　数	：	m		  - －１．０～＋１．０の範囲に調整する数値												//
//																												//
//	戻り値	：	－１．０～＋１．０の範囲に調整した数値															//
//--------------------------------------------------------------------------------------------------------------//
static	double	SubAdjustPM1(double n)
{
	if		(n > +1.0) n = +1.0;
	else if (n < -1.0) n = -1.0;

	return n;
}
