﻿#include	"AjcInternal.h"

#define	INST_ID			0x902256B4
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)

//**************************************************************************************************************//
//																												//
//	平衡２分木（ＡＶＬ木）																						//
//																												//
//**************************************************************************************************************//

#define		AVL_ENTER	if (pW->fMultiThread) EnterCriticalSection(&pW->cs);
#define		AVL_LEAVE	if (pW->fMultiThread) LeaveCriticalSection(&pW->cs);

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	int SubInsNode (HAJCAVL pW, PAJCAVLNODE *pp);
static	int SubInsDataNode(HAJCAVL pW, PAJCAVLNODE *pp);
static	int	SubDelNode (HAJCAVL pW, PAJCAVLNODE *pa, PAJCAVLNODE *pp);
static	VO	SubDelAllNodes(HAJCAVL pW, PAJCAVLNODE *pp);
static	VO	SubEnumNodesUpSeq  (HAJCAVL pW, PAJCAVLNODE p);
static	VO	SubEnumNodesDownSeq(HAJCAVL pW, PAJCAVLNODE p);
static	VO	LeftRotate (PAJCAVLNODE *pp);
static	VO	RightRotate(PAJCAVLNODE *pp);
static	int	KeyComp(HAJCAVL pW, UX k1, UX k2);


//==============================================================================================================//
//	ＡＶＬインスタンスの生成																					//
//																												//
//	引　数	：	cbp 	- コールバックパラメタ																	//
//				cbComp	- キー比較関数（キー値をそのまま比較する場合は、ＮＵＬＬ）								//
//				cbNtcDelNodeData - 削除ノードデータ通知用コールバック（不要時は、ＮＵＬＬ）						//
//																												//
//	戻り値	：	≠NULL - ＯＫ（インスタンスハンドル）															//
//				＝NULL - エラー																					//
//==============================================================================================================//
AJCEXPORT	HAJCAVL	WINAPI	AjcAvlCreate(UX		cbp,
										 int   (CALLBACK *cbComp)(UX key1, UX key2, UX cbp),
										 VO    (CALLBACK *cbNtcDelNodeData)(UX key, VOP pNodeData, UI len, UI nest, UX cbp))
{
	HAJCAVL	pW;

	if ((pW = (HAJCAVL)AJCMEM(sizeof(AJCAVL))) != NULL) {
		memset(pW, 0, sizeof *pW);
		pW->InstID	= INST_ID;
		pW->pRoot	= NULL;
		pW->cbp 	= cbp;
		pW->cbComp	= cbComp;
		pW->cbNtcDel= cbNtcDelNodeData;
	}
	return pW;
}

//==============================================================================================================//
//	ＡＶＬインスタンスの削除																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcAvlDelete(HAJCAVL pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->w_nest = 0;
		SubDelAllNodes(pW, &pW->pRoot);
		if (pW->fMultiThread) {
			DeleteCriticalSection(&pW->cs);
		}
		free(pW);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	マルチスレッドの許可／禁止																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				fEnable - FALSE : マルチスレッド禁止															//
//						  TRUE	: マルチスレッド許可															//
//																												//
//	戻り値	：	前回の許可／禁止状態																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcAvlEnableMultiThread(HAJCAVL pW, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		rc = pW->fMultiThread;
		if (fEnable) {
			if (!pW->fMultiThread) {
				InitializeCriticalSection(&pW->cs);
				pW->fMultiThread = TRUE;
			}
		}
		else {
			if (pW->fMultiThread) {
				pW->fMultiThread = FALSE;
				DeleteCriticalSection(&pW->cs);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	コールバックパラメタの設定																					//
//																												//
//	引　数	：	pW		- インスタンスワークのアドレス															//
//				cbp 	- コールバックパラメタ																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcAvlSetCbp(HAJCAVL pW, UX cbp)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->cbp		= cbp;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	ノード挿入																									//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				key 	  - キー																				//
//				pNodeData - 挿入するノードデータのアドレス														//
//				len 	  - 挿入するノードデータのバイト数（０～６５５３５）									//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - 同一キーのノードあり／ノードメモリ割り当て失敗／パラメタエラー							//
//==============================================================================================================//
AJCEXPORT BOOL WINAPI AjcAvlInsNode 	(HAJCAVL pW, UX key, C_VOP pNodeData, UI len)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (len <= 65535) {
			pW->w_key	= key;
			pW->w_pData = pNodeData;
			pW->w_len	= len;
			pW->w_rsu	= TRUE;
			pW->w_nest	= 0;

			SubInsNode(pW, &pW->pRoot);

			rc = pW->w_rsu;
		}
		AVL_LEAVE
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static int SubInsNode(HAJCAVL pW, PAJCAVLNODE *pp)
{
	int			deltaH = 0;
	PAJCAVLNODE	p = *pp;

	if (pW->w_rsu == TRUE) {

		if (p == NULL) {
			if ((*pp = p = (PAJCAVLNODE)AJCMEM(sizeof(AJCAVLNODE) + pW->w_len)) != NULL) {
				if (pW->w_pData && pW->w_len) {
					memcpy(p + 1, pW->w_pData, pW->w_len);
				}
				p->key = pW->w_key;
				p->bal = 0;
				p->len = (UW)(pW->w_len);
				p->l = p->r = NULL;
				deltaH = 1;
				pW->NodeCount++;
			}
			else {
				pW->w_rsu = FALSE;
			}
		}
		else {
			switch (KeyComp(pW, pW->w_key, p->key)) {
				case 1:
					if (SubInsNode(pW, &p->r)) {
						p->bal++;
						if (p->bal == 1) 
							deltaH = 1;
						else if (p->bal == 2) {
							if (p->r->bal == -1)
								RightRotate(&p->r);
							LeftRotate(pp);
						}
					}
					break;

				case -1:
					if (SubInsNode(pW, &p->l)) {
						p->bal--;
						if (p->bal == -1)
							deltaH = 1;
						else if (p->bal == -2) {
							if (p->l->bal == 1)
								LeftRotate(&p->l);
							RightRotate(pp);
						}
					}
					break;

				default:
					pW->w_rsu = FALSE;
					break;
			}
		}
	}
	return deltaH;
}
//==============================================================================================================//
//	データノード挿入（データのアドレスをキーとしたノード挿入）													//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pNodeData - 挿入するノードデータのアドレス														//
//				len 	  - 挿入するノードデータのバイト数（０～６５５３５）									//
//																												//
//	戻り値	：	≠NULL	- キー値（ノードデータのアドレス）														//
//				＝NULL	- ノードメモリ割り当て失敗／パラメタエラー												//
//==============================================================================================================//
AJCEXPORT UX WINAPI AjcAvlInsDataNode 	(HAJCAVL pW, C_VOP pNodeData, UI len)
{
	UX		rc = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pNodeData != NULL && len <= 65535) {
			pW->w_len = len;
			if (pW->w_pNode = (PAJCAVLNODE)AJCMEM(sizeof(AJCAVLNODE) + pW->w_len)) {
				pW->w_key	= (UX)(pW->w_pNode + 1);
				memcpy((VOP)pW->w_key, pNodeData, len);
				pW->w_pData = pNodeData;
				pW->w_rsu	= TRUE;
				pW->w_nest	= 0;

				SubInsDataNode(pW, &pW->pRoot);

				if (pW->w_rsu) rc = pW->w_key;
				else		   free((VOP)pW->w_pNode);
			}
		}
		AVL_LEAVE
	}

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static int SubInsDataNode(HAJCAVL pW, PAJCAVLNODE *pp)
{
	int			deltaH = 0;
	PAJCAVLNODE	p = *pp;

	if (pW->w_rsu == TRUE) {

		if (p == NULL) {
			*pp = p = (PAJCAVLNODE)pW->w_pNode;
			p->key = pW->w_key;
			p->bal = 0;
			p->len = (UW)(pW->w_len);
			p->l   = p->r = NULL;
			deltaH = 1;
			pW->NodeCount++;
		}
		else {
			switch (KeyComp(pW, (UX)pW->w_key, p->key)) {
				case 1:
					if (SubInsDataNode(pW, &p->r)) {
						p->bal++;
						if (p->bal == 1) 
							deltaH = 1;
						else if (p->bal == 2) {
							if (p->r->bal == -1)
								RightRotate(&p->r);
							LeftRotate(pp);
						}
					}
					break;

				case -1:
					if (SubInsDataNode(pW, &p->l)) {
						p->bal--;
						if (p->bal == -1)
							deltaH = 1;
						else if (p->bal == -2) {
							if (p->l->bal == 1)
								LeftRotate(&p->l);
							RightRotate(pp);
						}
					}
					break;

				default:
					pW->w_rsu = FALSE;
					break;
			}
		}
	}
	return deltaH;
}

//==============================================================================================================//
//	ノード取得（ノード検索）																					//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				key 	  - 検索するノードのキー																//
//				pBuf	  - 見つかったノードデータを格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//				lBuf	  - 見つかったノードデータを格納するバッファのバイト数									//
//																												//
//	戻り値	：	≠-1 - ＯＫ（当該ノード・データのバイト数）														//
//				＝-1 - 指定されたキーが見つからない																//
//==============================================================================================================//
AJCEXPORT int WINAPI AjcAvlGetNodeByKey 	(HAJCAVL pW, UX key, VOP pBuf, UI lBuf)
{
	int			rc = -1;
	PAJCAVLNODE	p;
	int			comp_rsu;
	UI			nest = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pW->pRoot) {
			p = pW->pRoot;
			while ((comp_rsu = KeyComp(pW, key, p->key)) != 0) {
				if (comp_rsu < 0) p = p->l;
				else			  p = p->r;
				if (p == NULL) break;
				nest++;
			}
			if (p != NULL) {
				if (pBuf   != NULL) memcpy(pBuf, p + 1, __min(p->len, lBuf));
				rc = p->len;
			}
		}
		AVL_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	ノードポインタ取得（ノード検索）																			//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				key 	  - 検索するノードのキー																//
//				pLen	  - 見つかったノードデータのバイト数を格納するバッファのアドレス（不要時はＮＵＬＬ）	//
//																												//
//	戻り値	：	≠NULL - ＯＫ（当該ノード・データへのポインタ）													//
//				＝NULL - 指定されたキーが見つからない／パラメタエラー											//
//==============================================================================================================//
AJCEXPORT VOP WINAPI AjcAvlGetNodePtr	  (HAJCAVL pW, UX key, UIP pLen)
{
	VOP			rc = NULL;
	PAJCAVLNODE	p;
	int			comp_rsu;
	UI			nest = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pW->pRoot) {
			p = pW->pRoot;
			while ((comp_rsu = KeyComp(pW, key, p->key)) != 0) {
				if (comp_rsu < 0) p = p->l;
				else			  p = p->r;
				if (p == NULL) break;
				nest++;
			}
			if (p != NULL) {
				if (pLen   != NULL) *pLen = p->len;
				rc = (p + 1);
			}
		}
		AVL_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	ノードのキー値取得																							//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pNode	  - ノード・データへのポインタ															//
//																												//
//	戻り値	：	ノードのキー値																					//
//==============================================================================================================//
AJCEXPORT UX WINAPI AjcAvlGetNodeKey	  (HAJCAVL pW, C_VOP pNode)
{
	UX		rc = -1;
	PAJCAVLNODE	pHd = (PAJCAVLNODE)pNode;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		pHd = (PAJCAVLNODE)pNode;		//	ノードデータアドレス
		pHd--;							//	ノードヘッダアドレス
		rc = pHd->key;
		AVL_LEAVE
	}
	return rc;
}


//==============================================================================================================//
//	先頭ノード取得																								//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pKey	  - 先頭ノードのキー値を格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//				pBuf	  - 見つかったノードデータを格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//				lBuf	  - 見つかったノードデータを格納するバッファのバイト数									//
//																												//
//	戻り値	：	≠-1 - ＯＫ（当該ノード・データのバイト数）														//
//				＝-1 - ノードなし																				//
//==============================================================================================================//
AJCEXPORT int WINAPI AjcAvlGetTopNode	  (HAJCAVL pW, UXP pKey, VOP pBuf, UI lBuf)
{
	int			rc = -1;
	PAJCAVLNODE	p;
	UI			nest = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pW->pRoot) {
			p = pW->pRoot;
			while (p->l != NULL) {
				p = p->l;
				nest++;
			}
			if (pKey   != NULL) *pKey	= p->key;
			if (pBuf   != NULL) memcpy(pBuf, p + 1, __min(p->len, lBuf));
			rc = p->len;
		}
		AVL_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	最終ノード取得																								//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pKey	  - 先頭ノードのキー値を格納するバッファのアドレス（不要時はＮＵＬＬ）					//
//				pBuf	  - 見つかったノードデータを格納するバッファのアドレス（不要時はＮＵＬＬ）				//
//				lBuf	  - 見つかったノードデータを格納するバッファのバイト数									//
//																												//
//	戻り値	：	≠-1 - ＯＫ（当該ノード・データのバイト数）														//
//				＝-1 - ノードなし																				//
//==============================================================================================================//
AJCEXPORT int WINAPI AjcAvlGetLastNode	   (HAJCAVL pW, UXP pKey, VOP pBuf, UI lBuf)
{
	int			rc = -1;
	PAJCAVLNODE	p;
	UI			nest = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pW->pRoot) {
			p = pW->pRoot;
			while (p->r != NULL) {
				p = p->r;
				nest++;
			}
			if (pKey   != NULL) *pKey	= p->key;
			if (pBuf   != NULL) memcpy(pBuf, p + 1, __min(p->len, lBuf));
			rc = p->len;
		}
		AVL_LEAVE
	}
	return rc;
}

//==============================================================================================================//
//	ノード置換																									//
//																												//
//	引　数	：	pW		  	- インスタンスハンドル																//
//				key 	  	- キー																				//
//				pNodeData	- 置換する新ノードデータのアドレス													//
//				len 	 	- 置換する新ノードデータのバイト数（０～６５５３５）								//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - 指定されたキーが見つからない／ノードメモリ割り当て失敗／パラメタエラー					//
//==============================================================================================================//
AJCEXPORT BOOL WINAPI AjcAvlRepNode 	(HAJCAVL pW, UX key, C_VOP pNodeData, UI len)
{
	BOOL		rc = FALSE;
	PAJCAVLNODE	p, q;
	PAJCAVLNODE	*psv;
	int			comp_rsu;
	UI			nest = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pW->pRoot) {
			psv = &pW->pRoot; p = pW->pRoot;
			while ((comp_rsu = KeyComp(pW, key, p->key)) != 0) {
				if (comp_rsu < 0) {psv = &p->l; p = p->l;}
				else			  {psv = &p->r; p = p->r;}
				if (p == NULL) break;
				nest++;
			}
			if (p != NULL) {
				if (p->len == len) {
					if (pW->cbNtcDel) pW->cbNtcDel(p->key, p + 1, p->len, nest + 1, pW->cbp);
					memcpy(p + 1, pNodeData, len);
					rc = TRUE;
				}
				else {
					if (q = (PAJCAVLNODE)AJCMEM(sizeof(AJCAVLNODE) + len)) {
						if (pW->cbNtcDel) pW->cbNtcDel(p->key, p + 1, p->len, nest + 1, pW->cbp);
						memcpy(q	, p 	   , sizeof(AJCAVLNODE));
						memcpy(q + 1, pNodeData, len			   );
						*psv = q;
						free(p);	
						rc = TRUE;
					}
				}
			}
		}
		AVL_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	ノード挿入／置換																							//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				key		  - 置換するノードのキー値																//
//				pNodeData - 置換するノードデータのアドレス														//
//				len		  - 置換するノードデータのバイト数														//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - ノードメモリ割り当て失敗／パラメタエラー												//
//==============================================================================================================//
AJCEXPORT BOOL WINAPI AjcAvlInsOrRepNode(HAJCAVL pW, UX key, C_VOP pNodeData, UI len)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (pNodeData != NULL) {
			if (AjcAvlGetNodePtr(pW, key, NULL) == NULL) {
				rc = AjcAvlInsNode(pW, key, pNodeData, len);
			}
			else {
				rc = AjcAvlRepNode(pW, key, pNodeData, len);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ノード削除																									//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				key 	  - 削除するノードのキー																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - 指定されたキーのノードが見つからない／パラメタエラー									//
//==============================================================================================================//
AJCEXPORT BOOL WINAPI AjcAvlDelNode 	(HAJCAVL pW, UX key)
{
	BOOL	rc = TRUE;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		pW->w_key  = key;
		pW->w_rsu  = FALSE;
		pW->w_nest = 0;
		SubDelNode(pW, &pW->pRoot, &pW->pRoot);
		rc = pW->w_rsu;
		AVL_LEAVE
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	int	SubDelNode(HAJCAVL pW, PAJCAVLNODE *pa, PAJCAVLNODE *pp)
{
	PAJCAVLNODE		p = *pp;
	int				deltaH = 0;

	pW->w_nest++;

	if (p != NULL) {
		switch (KeyComp(pW, pW->w_key, p->key)) {
			case -1:
				if (SubDelNode(pW, &p, &p->l)) {
					p->bal++;
					if (p->bal == 0)
						deltaH = 1;
					else if (p->bal == 2) {
						if (p->r->bal == -1) 
							RightRotate(&p->r);
						LeftRotate(pp);
						if ((*pp)->bal == 0)
							deltaH = 1;
					}
				}
				break;

			case 1:
				if (SubDelNode(pW, &p, &p->r)) {
					p->bal--;
					if (p->bal == 0)
						deltaH = 1;
					else if (p->bal == -2) {
						if (p->l->bal == 1)
							LeftRotate(&p->l);
						RightRotate(pp);
						if ((*pp)->bal == 0) 
							deltaH = 1;
					}
				}
				break;

			default:
				if (p->r == NULL) {
					if (pW->cbNtcDel) pW->cbNtcDel(p->key, p + 1, p->len, pW->w_nest, pW->cbp);
					*pp = p->l;
					pW->NodeCount--;
					free(p);
					pW->w_rsu = TRUE;
					return 1;
				}
				else if (p->l == NULL) {
					if (pW->cbNtcDel) pW->cbNtcDel(p->key, p + 1, p->len, pW->w_nest, pW->cbp);
					*pp = p->r;
					pW->NodeCount--;
					free(p);
					pW->w_rsu = TRUE;
					return 1;
				}
				else {
					pW->s = p;
					pW->q = p->l;
					while (pW->q->r != NULL) {
						pW->s = pW->q;
						pW->q = pW->q->r;
					}

					if (*pa != p) {
						if ((*pa)->l == p) (*pa)->l = pW->q;
						else			   (*pa)->r = pW->q;
					}
					else {
						*pa = pW->q;
					}
					if		(pW->s->l == pW->q) pW->s->l = p;
					else						pW->s->r = p;

					pW->w = p->l;	p->l   = pW->q->l;	 pW->q->l	= pW->w;
					pW->w = p->r;	p->r   = pW->q->r;	 pW->q->r	= pW->w;
					pW->b = p->bal; p->bal = pW->q->bal; pW->q->bal = pW->b;

					p = pW->q;
					if (SubDelNode(pW, &p, &p->l)) {
						p->bal++;
						if (p->bal == 0)
							deltaH= 1;
						else if (p->bal == 2) {
							if (p->r->bal == -1) 
								RightRotate(&p->r);
							LeftRotate(pp);
							if ((*pp)->bal == 0)
								deltaH = 1;
						}
					}
				}
				break;
		}
	}

	pW->w_nest--;

	return deltaH;
}
//==============================================================================================================//
//	全ノード削除																								//
//																												//
//	引　数	：	pW			 - インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - パラメタエラー																			//
//==============================================================================================================//
AJCEXPORT BOOL WINAPI AjcAvlDelAllNodes (HAJCAVL pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		pW->w_nest = 0;
		SubDelAllNodes(pW, &pW->pRoot);
		pW->NodeCount = 0;
		AVL_LEAVE
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	VO	SubDelAllNodes(HAJCAVL pW, PAJCAVLNODE *pp)  // pa=Parent, pb=child //
{
	pW->w_nest++;
	if (*pp != NULL) {
		SubDelAllNodes(pW, &(*pp)->l);
		SubDelAllNodes(pW, &(*pp)->r);
		if (pW->cbNtcDel) pW->cbNtcDel((*pp)->key, (*pp) + 1, (*pp)->len, pW->w_nest, pW->cbp);
		free(*pp);
		*pp = NULL;
	}
	pW->w_nest--;
}
//==============================================================================================================//
//	文字列ノード挿入																							//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pStr	  - 挿入する文字列のアドレス															//
//																												//
//	戻り値	：	≠０ - ＯＫ（キー値＝ノードデータ（文字列）のアドレス）											//
//				＝０ - 同一キーのノードあり／ノードメモリ割り当て失敗／パラメタエラー							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlInsStrNodeA	  (HAJCAVL pW, C_BCP pStr)
{
	UX		rc = 0;
	UX		len;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pStr != NULL && (len = strlen(pStr) + 1 + sizeof(UX)) <= 65535) {
			pW->w_len = (UI)len;
			if (pW->w_pNode = (PAJCAVLNODE)AJCMEM(sizeof(AJCAVLNODE) + pW->w_len)) {
				memset(pW->w_pNode, 0, sizeof(AJCAVLNODE) + pW->w_len);
				pW->w_key	= (UX)(pW->w_pNode + 1);
				strcpy((BCP)pW->w_key, pStr);
				pW->w_pData = (BCP)pStr;
				pW->w_rsu	= TRUE;
				pW->w_nest	= 0;

				SubInsDataNode(pW, &pW->pRoot);

				if (pW->w_rsu) rc = pW->w_key;
				else		   free((VOP)pW->w_pNode);
			}
		}
		AVL_LEAVE
	}

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlInsStrNodeW	  (HAJCAVL pW, C_WCP pStr)
{
	UX		rc = 0;
	UX		len;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pStr != NULL && (len = wcslen(pStr) * 2 + 2 + sizeof(UX)) <= 65535) {
			pW->w_len = (UI)len;
			if (pW->w_pNode = (PAJCAVLNODE)AJCMEM(sizeof(AJCAVLNODE) + pW->w_len)) {
				memset(pW->w_pNode, 0, sizeof(AJCAVLNODE) + pW->w_len);
				pW->w_key	= (UX)(pW->w_pNode + 1);
				wcscpy((WCP)pW->w_key, pStr);
				pW->w_rsu	= TRUE;
				pW->w_nest	= 0;
				SubInsDataNode(pW, &pW->pRoot);
				if (pW->w_rsu) rc = pW->w_key;
				else		   free((VOP)pW->w_pNode);
			}
		}
		AVL_LEAVE
	}

	return rc;
}
//==============================================================================================================//
//	文字列ノード取得																							//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pStr	  - キーデータ（文字列）のアドレス														//
//																												//
//	戻り値	：	≠０ - ＯＫ（キー値（文字列）のアドレス）														//
//				＝０ - 同一キーのノードなし／ノードメモリ割り当て失敗／パラメタエラー							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlGetStrNodeA(HAJCAVL pW, C_BCP pStr)
{
	UX		rc = 0;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			rc = (UX)AjcAvlGetNodePtr(pW, (UX)pStr, NULL);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlGetStrNodeW(HAJCAVL pW, C_WCP pStr)
{
	UX		rc = 0;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			rc = (UX)AjcAvlGetNodePtr(pW, (UX)pStr, NULL);
		}
	}
	return rc;
}
//==============================================================================================================//
//	文字列ノード挿入／取得																						//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pStr	  - 取得するノードのキーデータ（文字列）のアドレス										//
//																												//
//	戻り値	：	≠０ - ＯＫ（キー値＝ノードデータ（文字列）のアドレス）											//
//				＝０ - ノードメモリ割り当て失敗／パラメタエラー													//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlInsOrGetStrNodeA(HAJCAVL pW, C_BCP pStr)
{
	UX		rc = 0;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if ((rc = (UX)AjcAvlGetStrNodeA(pW, pStr)) == 0) {
				rc = AjcAvlInsStrNodeA(pW, pStr);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlInsOrGetStrNodeW(HAJCAVL pW, C_WCP pStr)
{
	UX		rc = 0;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if ((rc = (UX)AjcAvlGetStrNodeW(pW, pStr)) == 0) {
				rc = AjcAvlInsStrNodeW(pW, pStr);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	文字列ノード削除																							//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pStr	  - 挿入／取得する文字列のアドレス														//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - 当該ノード無し／ノードメモリ割り当て失敗／パラメタエラー								//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlDelStrNodeA(HAJCAVL pW, C_BCP pStr)
{
	UX		rc = 0;
	UX		key;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if ((key = (UX)AjcAvlGetNodePtr(pW, (UX)pStr, NULL)) != 0) {
				rc = AjcAvlDelNode(pW, key);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlDelStrNodeW(HAJCAVL pW, C_WCP pStr)
{
	UX		rc = 0;
	UX		key;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if ((key = (UX)AjcAvlGetNodePtr(pW, (UX)pStr, NULL)) != 0) {
				rc = AjcAvlDelNode(pW, key);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	文字列ノードにデータを関連付ける																			//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pStr	  - 挿入／取得する文字列のアドレス														//
//				UX		  - データ																				//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - 当該ノード無し／ノードメモリ割り当て失敗／パラメタエラー								//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlSetStrNodeDataA(HAJCAVL pW, C_BCP pStr, UX data)
{
	BOOL	rc = FALSE;
	union {BCP pS; UXP pD;} u;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if (u.pS = (BCP)AjcAvlGetStrNodeA(pW, pStr)) {
				u.pS += (strlen(u.pS) + 1);
				*u.pD = data;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT UX WINAPI AjcAvlSetStrNodeDataW(HAJCAVL pW, C_WCP pStr, UX data)
{
	BOOL	rc = FALSE;
	union {WCP pS; UXP pD;} u;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if (u.pS = (WCP)AjcAvlGetStrNodeW(pW, pStr)) {
				u.pS += (wcslen(u.pS) + 1);
				*u.pD = data;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	文字列ノードに関連付けたデータを取得																		//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				pStr	  - 挿入／取得する文字列のアドレス														//
//				pData	  - データ値を格納するバッファ															//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - 当該ノード無し／ノードメモリ割り当て失敗／パラメタエラー								//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcAvlGetStrNodeDataA(HAJCAVL pW, C_BCP pStr, UXP pData)
{
	BOOL	rc = FALSE;
	union {BCP pS; UXP pD;} u;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if (u.pS = (BCP)AjcAvlGetStrNodeA(pW, pStr)) {
				if (pData != NULL) {
					u.pS += (strlen(u.pS) + 1);
					*pData = *u.pD;
				}
				rc = TRUE;
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcAvlGetStrNodeDataW(HAJCAVL pW, C_WCP pStr, UXP pData)
{
	BOOL	rc = FALSE;
	union {WCP pS; UXP pD;} u;

	if (IS_MY_INST(pW)) {
		if (pStr != NULL) {
			if (u.pS = (WCP)AjcAvlGetStrNodeW(pW, pStr)) {
				if (pData != NULL) {
					u.pS += (wcslen(u.pS) + 1);
					*pData = *u.pD;
				}
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ノード数取得																								//
//																												//
//	引　数	：	pW			 - インスタンスハンドル																//
//																												//
//	戻り値	：	ノード数（パラメタエラー時は、０を返す）														//
//==============================================================================================================//
AJCEXPORT UI WINAPI AjcAvlGetCount(HAJCAVL pW)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		rc = pW->NodeCount;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT UI WINAPI AjcAvlNumber(HAJCAVL pW)
{
	return AjcAvlGetCount(pW);
}


//==============================================================================================================//
//	全ノード取得																								//
//																												//
//	引　数	：	pW			 - インスタンスハンドル																//
//				cbNtcNode	 - 取得したノードを通知するコールバック関数（不要時はＮＵＬＬ）						//
//				fDownSeq	 - FALSE:昇順，TRUE:降順															//
//																												//
//	戻り値	：	ノード数（エラー時は、０を返す）																//
//==============================================================================================================//
AJCEXPORT UI WINAPI AjcAvlEnumNodes (HAJCAVL pW, AJCAVL_CBNTC cbNtcNode, BOOL fDownSeq)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (cbNtcNode) {
			pW->w_cb   = cbNtcNode;
			pW->w_nest = 0;
			pW->w_cont = TRUE;
			pW->w_cbp  = pW->cbp;
			if (!fDownSeq) {
				SubEnumNodesUpSeq  (pW, pW->pRoot);
			}
			else {
				SubEnumNodesDownSeq(pW, pW->pRoot);
			}
		}
		rc = pW->NodeCount;
		AVL_LEAVE
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	VO	SubEnumNodesUpSeq(HAJCAVL pW, PAJCAVLNODE p)
{
	if (pW->w_cont) {
		pW->w_nest++;
		if (p != NULL) {
			SubEnumNodesUpSeq(pW, p->l);
			if (pW->w_cont) {
				pW->w_cont = pW->w_cb(p->key, p + 1, p->len, pW->w_nest, pW->w_cbp);
			}
			SubEnumNodesUpSeq(pW, p->r);
		}
		pW->w_nest--;
	}
}
//--------------------------------------------------------------------------------------------------------------//
static	VO	SubEnumNodesDownSeq(HAJCAVL pW, PAJCAVLNODE p)
{
	if (pW->w_cont) {
		pW->w_nest++;
		if (p != NULL) {
			SubEnumNodesDownSeq(pW, p->r);
			if (pW->w_cont) {
				pW->w_cont = pW->w_cb(p->key, p + 1, p->len, pW->w_nest, pW->w_cbp);
			}
			SubEnumNodesDownSeq(pW, p->l);
		}
		pW->w_nest--;
	}
}

//==============================================================================================================//
//	全ノード取得（コールバックパラメタ指定）																	//
//																												//
//	引　数	：	pW			 - インスタンスハンドル																//
//				cbp			 - コールバックパラメタ																//
//				cbNtcNode	 - 取得したノードを通知するコールバック関数（不要時はＮＵＬＬ）						//
//				fDownSeq	 - FALSE:昇順，TRUE:降順															//
//																												//
//	戻り値	：	ノード数																						//
//==============================================================================================================//
AJCEXPORT UI   WINAPI AjcAvlEnumNodesEx (HAJCAVL pW, UX cbp, AJCAVL_CBNTC cbNtcNode, BOOL fDownSeq)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (cbNtcNode) {
			pW->w_cb   = cbNtcNode;
			pW->w_nest = 0;
			pW->w_cont = TRUE;
			pW->w_cbp  = cbp;
			if (!fDownSeq) {
				SubEnumNodesUpSeq  (pW, pW->pRoot);
			}
			else {
				SubEnumNodesDownSeq(pW, pW->pRoot);
			}
		}
		rc = pW->NodeCount;
		AVL_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードへのポインタ配列生成																				//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				pNum		- 配列要素数を格納するバッファのアドレス（不要時はNULL)								//
//				fDownSeq	- FALSE:昇順，TRUE:降順																//
//																												//
//	戻り値	：	≠NULL : 全ノードへのポインタ配列の先頭アドレス													//
//				＝NULL : エラー																					//
//==============================================================================================================//
typedef struct {
	UI			ix;
	PAJCAVLPTR	pArr;
} CBP_CREATE_PTR_ARR, *PCBP_CREATE_PTR_ARR;
//--------------------------------------------------------------------------------------------------------------//
static	BOOL CALLBACK cbAvlCreatePtrArr(UX key, VOP pNodeData, UI len, UI nest, UX cbp)
{
	PCBP_CREATE_PTR_ARR	pCbp = (PCBP_CREATE_PTR_ARR)cbp;
	pCbp->pArr[pCbp->ix].key	= key;
	pCbp->pArr[pCbp->ix].pNode	= (VOP)pNodeData;
	pCbp->pArr[pCbp->ix].len	= len;
	pCbp->ix++;
	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	PCAJCAVLPTR	WINAPI	AjcAvlCreatePtrArr (HAJCAVL pW, UIP pNum, BOOL fDownSeq)
{
	PCAJCAVLPTR			rc = NULL;
	CBP_CREATE_PTR_ARR	Cbp;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		Cbp.ix = 0;
		if (Cbp.pArr  =  (PAJCAVLPTR)AJCMEM(sizeof(AJCAVLPTR) * (pW->NodeCount + 1))) {
			memset((VOP)Cbp.pArr, 0,		sizeof(AJCAVLPTR) * (pW->NodeCount + 1));
			AjcAvlEnumNodesEx(pW, (UX)&Cbp, cbAvlCreatePtrArr, fDownSeq);
			if (pNum != NULL) *pNum = Cbp.ix;
			rc = Cbp.pArr;
		}
		AVL_LEAVE
	}
	return rc;
}
//==============================================================================================================//
//	全ノードへのポインタ配列解放																				//
//																												//
//	引　数	：	pW			- インスタンスハンドル																//
//				ppArr		- 全ノードへのポインタ配列の先頭アドレス											//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcAvlReleasePtrArr(HAJCAVL pW, PCAJCAVLPTR pArr)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		AVL_ENTER
		if (pArr != NULL) free((VOP)pArr);
		AVL_LEAVE
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	ＡＶＬツリー左回転																							//
//--------------------------------------------------------------------------------------------------------------//
static VO LeftRotate(PAJCAVLNODE *pp)
{
	PAJCAVLNODE	p = *pp;
	PAJCAVLNODE	r;

	*pp = r = p->r;
	p->r = r->l;
	r->l = p;
	p->bal--;
	if (r->bal > 0)
		p->bal -= r->bal;
	r->bal--;
	if (p->bal < 0)
		r->bal += p->bal;
}
//--------------------------------------------------------------------------------------------------------------//
//	ＡＶＬツリー右回転																							//
//--------------------------------------------------------------------------------------------------------------//
static VO RightRotate(PAJCAVLNODE *pp)
{
	PAJCAVLNODE	p = *pp;
	PAJCAVLNODE	l;

	*pp = l = p->l;
	p->l = l->r;
	l->r = p;
	p->bal++;
	if (l->bal < 0) 
		p->bal -= l->bal;
	l->bal++;
	if (p->bal > 0) 
		l->bal += p->bal;
}
//--------------------------------------------------------------------------------------------------------------//
//	キー比較																									//
//--------------------------------------------------------------------------------------------------------------//
static int KeyComp(HAJCAVL pW, UX k1, UX k2)
{
	int		rc;

	if (pW->cbComp) {
		rc = pW->cbComp(k1, k2, pW->cbp);
		if		(rc < 0) rc = -1;
		else if (rc > 0) rc =  1;
		else			 rc =  0;
	}
	else {
		if		(k1 < k2) rc = -1;
		else if (k1 > k2) rc =	1;
		else			  rc =	0;
	}
	return rc;
}

