﻿#include	"AjcInternal.h"

#define	INST_ID			0xB567DA28
#define	IS_MY_INST(P)	(P != NULL && P->InstID == INST_ID)

//**************************************************************************************************************//
//																												//
//	Ｃ／Ｃ＋＋言語の字句分解																					//
//																												//
//**************************************************************************************************************//

//--------------------------------------------------------------------------------------------------------------//
//	予約語テーブル																								//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	EAJCTKCODE	tkn;
	C_WCP		pStr;
	UI			len;
	BOOL		fCPP;
} AJCTKTABLE;

static	const AJCTKTABLE CtkRsvTbl[] = {
	{	EAJCTK_RSV_ASM		,	L"asm"				,  3, FALSE},
	{	EAJCTK_RSV_AUTO		,	L"auto"				,  4, FALSE},
	{	EAJCTK_RSV_BREAK	,	L"break"			,  5, FALSE},
	{	EAJCTK_RSV_CASE		,	L"case"				,  4, FALSE},
	{	EAJCTK_RSV_CATCH	,	L"catch"			,  5, TRUE },
	{	EAJCTK_RSV_CHAR		,	L"char"				,  4, FALSE},
	{	EAJCTK_RSV_CLASS	,	L"class"			,  5, TRUE },
	{	EAJCTK_RSV_CONST	,	L"const"			,  5, FALSE},
	{	EAJCTK_RSV_CONCAST	,	L"const_cast"		, 10, TRUE },
	{	EAJCTK_RSV_CONTINUE	,	L"continue"			,  8, FALSE},
	{	EAJCTK_RSV_DEFAULT	,	L"default"			,  7, FALSE},
	{	EAJCTK_RSV_DEFINE	,	L"define"			,  6, FALSE},
	{	EAJCTK_RSV_DEFINED	,	L"defined"			,  7, FALSE},
	{	EAJCTK_RSV_DELETE	,	L"delete"			,  6, TRUE },
	{	EAJCTK_RSV_DO		,	L"do"				,  2, FALSE},
	{	EAJCTK_RSV_DOUBLE	,	L"double"			,  6, FALSE},
	{	EAJCTK_RSV_DYNCAST	,	L"dynamic_cast"		, 12, TRUE },
	{	EAJCTK_RSV_ELIF		,	L"elif"				,  4, FALSE},
	{	EAJCTK_RSV_ELSE		,	L"else"				,  4, FALSE},
	{	EAJCTK_RSV_ENDIF	,	L"endif"			,  5, FALSE},
	{	EAJCTK_RSV_ENUM		,	L"enum"				,  4, FALSE},
	{	EAJCTK_RSV_EXTERN	,	L"extern"			,  6, FALSE},
	{	EAJCTK_RSV_FLOAT	,	L"float"			,  5, FALSE},
	{	EAJCTK_RSV_FOR		,	L"for"				,  3, FALSE},
	{	EAJCTK_RSV_FRIEND	,	L"friend"			,  6, TRUE },
	{	EAJCTK_RSV_GOTO		,	L"goto"				,  4, FALSE},
	{	EAJCTK_RSV_IF		,	L"if"				,  2, FALSE},
	{	EAJCTK_RSV_IFDEF	,	L"ifdef"			,  5, FALSE},
	{	EAJCTK_RSV_IFNDEF	,	L"ifndef"			,  6, FALSE},
	{	EAJCTK_RSV_INCLUDE	,	L"include"			,  7, FALSE},
	{	EAJCTK_RSV_INLINE	,	L"inline"			,  6, FALSE},
	{	EAJCTK_RSV_INT		,	L"int"				,  3, FALSE},
	{	EAJCTK_RSV_LONG		,	L"long"				,  4, FALSE},
	{	EAJCTK_RSV_NEW		,	L"new"				,  3, TRUE },
	{	EAJCTK_RSV_OPERATOR	,	L"operator"			,  8, TRUE },
	{	EAJCTK_RSV_PRAGMA	,	L"pragma"			,  6, FALSE},
	{	EAJCTK_RSV_PRIVATE	,	L"private"			,  7, TRUE },
	{	EAJCTK_RSV_PROTECTED,	L"protected"		,  9, TRUE },
	{	EAJCTK_RSV_PUBLIC	,	L"public"			,  6, TRUE },
	{	EAJCTK_RSV_REGISTER	,	L"register"			,  8, FALSE},
	{	EAJCTK_RSV_REICAST	,	L"reinterpret_cast"	, 16, TRUE },
	{	EAJCTK_RSV_RETURN	,	L"return"			,  6, FALSE},
	{	EAJCTK_RSV_SHORT	,	L"short"			,  5, FALSE},
	{	EAJCTK_RSV_SIGNED	,	L"signed"			,  6, FALSE},
	{	EAJCTK_RSV_SIZEOF	,	L"sizeof"			,  6, FALSE},
	{	EAJCTK_RSV_STATIC	,	L"static"			,  6, FALSE},
	{	EAJCTK_RSV_STACAST	,	L"static_cast"		, 11, TRUE },
	{	EAJCTK_RSV_STRUCT	,	L"struct"			,  6, FALSE},
	{	EAJCTK_RSV_SWITCH	,	L"switch"			,  6, FALSE},
	{	EAJCTK_RSV_TEMPLATE	,	L"template"			,  8, TRUE },
	{	EAJCTK_RSV_THIS		,	L"this"				,  4, TRUE },
	{	EAJCTK_RSV_THROW	,	L"throw"			,  5, TRUE },
	{	EAJCTK_RSV_TRY		,	L"try"				,  3, TRUE },
	{	EAJCTK_RSV_TYPEDEF	,	L"typedef"			,  7, FALSE},
	{	EAJCTK_RSV_UNDEF	,	L"undef"			,  5, FALSE},
	{	EAJCTK_RSV_UNION	,	L"union"			,  5, FALSE},
	{	EAJCTK_RSV_UNSIGNED	,	L"unsigned"			,  8, FALSE},
	{	EAJCTK_RSV_VIRTUAL	,	L"virtual"			,  7, TRUE },
	{	EAJCTK_RSV_VOID		,	L"void"				,  4, FALSE},
	{	EAJCTK_RSV_VOLATILE	,	L"volatile"			,  8, FALSE},
	{	EAJCTK_RSV_WCHAR_T	,	L"wchar_t"			,  7, FALSE},
	{	EAJCTK_RSV_WHILE	,	L"while"			,  5, FALSE}
};

#define	MAX_RSVTBL	((sizeof CtkRsvTbl) / (sizeof CtkRsvTbl[0]))
#define	AJCTKRSV_MAXLEN	16

//--------------------------------------------------------------------------------------------------------------//
//	デリミタテーブル																							//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	EAJCTKCODE	tkn;
	C_WCP		pStr;
	UI			lStr;
	BOOL		fCPP;
} AJCTKDLMTABLE;

static	const AJCTKDLMTABLE CtkDlmTbl[] = {
	{	EAJCTK_DLM_LSPART		,	L"("	, 1 , FALSE},
	{	EAJCTK_DLM_LMPART		,	L"{"	, 1 , FALSE},
	{	EAJCTK_DLM_LLPART		,	L"["	, 1 , FALSE},
	{	EAJCTK_DLM_RSPART		,	L")"	, 1 , FALSE},
	{	EAJCTK_DLM_RMPART		,	L"}"	, 1 , FALSE},
	{	EAJCTK_DLM_RLPART		,	L"]"	, 1 , FALSE},

	{	EAJCTK_DLM_SCOPE		,	L"::"	, 2 , TRUE },
	{	EAJCTK_DLM_IDDOT		,	L".*"	, 2 , TRUE },
	{	EAJCTK_DLM_IDARROW		,	L"->*"	, 3 , TRUE },

	{	EAJCTK_DLM_LAND			,	L"&&"	, 2 , FALSE},
	{	EAJCTK_DLM_LOR			,	L"||"	, 2 , FALSE},

	{	EAJCTK_DLM_ARROW		,	L"->"	, 2 , FALSE},
	{	EAJCTK_DLM_QUEST		,	L"?"	, 1 , FALSE},
	{	EAJCTK_DLM_COLON		,	L":"	, 1 , FALSE},
	{	EAJCTK_DLM_SEMICOL		,	L";"	, 1 , FALSE},
	{	EAJCTK_DLM_COMMA		,	L","	, 1 , FALSE},
	{	EAJCTK_DLM_VARG			,	L"..."	, 3 , FALSE},
	{	EAJCTK_DLM_DOT			,	L"."	, 1 , FALSE},

	{	EAJCTK_DLM_SHREQ		,	L">>="	, 3 , FALSE},
	{	EAJCTK_DLM_SHLEQ		,	L"<<="	, 3 , FALSE},
	{	EAJCTK_DLM_SHR			,	L">>"	, 2 , FALSE},
	{	EAJCTK_DLM_SHL			,	L"<<"	, 2 , FALSE},

	{	EAJCTK_DLM_EQEQ			,	L"=="	, 2 , FALSE},
	{	EAJCTK_DLM_NOTEQ		,	L"!="	, 2 , FALSE},
	{	EAJCTK_DLM_LOEQ			,	L"<="	, 2 , FALSE},
	{	EAJCTK_DLM_HIEQ			,	L">="	, 2 , FALSE},
	{	EAJCTK_DLM_LO			,	L"<"	, 1 , FALSE},
	{	EAJCTK_DLM_HI			,	L">"	, 1 , FALSE},

	{	EAJCTK_DLM_PLUSEQ		,	L"+="	, 2 , FALSE},
	{	EAJCTK_DLM_MINUSEQ		,	L"-="	, 2 , FALSE},
	{	EAJCTK_DLM_MULTEQ		,	L"*="	, 2 , FALSE},
	{	EAJCTK_DLM_DIVEQ		,	L"/="	, 2 , FALSE},
	{	EAJCTK_DLM_MODEQ		,	L"%="	, 2 , FALSE},
	{	EAJCTK_DLM_ANDEQ		,	L"&="	, 2 , FALSE},
	{	EAJCTK_DLM_OREQ			,	L"|="	, 2 , FALSE},
	{	EAJCTK_DLM_XOREQ		,	L"^="	, 2 , FALSE},

	{	EAJCTK_DLM_SHARP2		,	L"##"	, 2 , FALSE},
	{	EAJCTK_DLM_PLUS2		,	L"++"	, 2 , FALSE},
	{	EAJCTK_DLM_MINUS2		,	L"--"	, 2 , FALSE},

	{	EAJCTK_DLM_SHARP		,	L"#"	, 1 , FALSE},
	{	EAJCTK_DLM_PLUS			,	L"+"	, 1 , FALSE},
	{	EAJCTK_DLM_MINUS		,	L"-"	, 1 , FALSE},
	{	EAJCTK_DLM_MULT			,	L"*"	, 1 , FALSE},
	{	EAJCTK_DLM_DIV			,	L"/"	, 1 , FALSE},
	{	EAJCTK_DLM_MOD			,	L"%"	, 1 , FALSE},
	{	EAJCTK_DLM_AND			,	L"&"	, 1 , FALSE},
	{	EAJCTK_DLM_OR			,	L"|"	, 1 , FALSE},
	{	EAJCTK_DLM_XOR			,	L"^"	, 1 , FALSE},
	{	EAJCTK_DLM_EQ			,	L"="	, 1 , FALSE},
	{	EAJCTK_DLM_NOT			,	L"~"	, 1 , FALSE},
	{	EAJCTK_DLM_LNOT			,	L"!"	, 1 , FALSE}
};
#define	MAX_DLMTBL	((sizeof CtkDlmTbl) / (sizeof CtkDlmTbl[0]))

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubGetToken(HAJCTK pW, WCP pStrBuf, UI lStrBuf);
static	UI		gtkSymbol(HAJCTK pW);
static	UI		gtkHexa(HAJCTK pW);
static	UI		gtkDecimal(HAJCTK pW);
static	UI		gtkOctal(HAJCTK pW);
static	UI		gtkString(HAJCTK pW);
static	UI		gtkDelimiter(HAJCTK pW);
static	UI		gtkComment(HAJCTK pW);
static	UI		gtkIncPath(HAJCTK pW);
static	UI		SubChkCommentEnd(HAJCTK pW, WCP pe);
static	UI		SubChkLineEnd(WCP pe);
static	UI		gtkSpace(HAJCTK pW);
static	UI		gtkEndOfLine(HAJCTK pW);
static	UI		gtkLineCont(HAJCTK pW);
static	UI		gtkSuffix(HAJCTK pW, WCP pe);
static	BOOL	SetTknStr(HAJCTK pW, WCP pTkn, UI lTkn, WCP pStrBuf, UI lStrBuf);
static	VO		SetPreproTokenHist(HAJCTK pW);
static	BOOL	ReadLineIfEol(HAJCTK pW);

//==============================================================================================================//
//	インスタンス生成																							//
//																												//
//	引　数	：	flag	  - 機能フラグ(AJCTKFLG_XXXXX)															//
//				cbGetLine - １行リード用コールバック関数のアドレス												//
//				xp		  - コールバックパラメタ																//
//																												//
//	戻り値	：	≠NULL : インスタンスハンドル																	//
//				＝NULL : エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCTK	WINAPI	AjcCtkCreateA  (UI flag, PCBAJCTKA cbGetLine, UX xp)
{
	HAJCTK	pW = NULL;

//	if (cbGetLine != NULL) {	// コールバックを後で設定する場合もあるので、NULLはチェックしない
		if (pW = (HAJCTK)AJCMEM(sizeof(AJCTK))) {
			memset(pW, 0, sizeof(AJCTK));
			pW->lLineBuf = 1024;
			if ((flag & AJCTKFLG_LBF_MASK) != 0) {
				pW->lLineBuf *= ((((flag & AJCTKFLG_LBF_MASK) >> 24) & 0xFF));
			}
			if (pW->pLineBuf = (WCP)AJCMEM(pW->lLineBuf * 2)) {
				memset(pW->pLineBuf, 0, pW->lLineBuf * 2);
				pW->InstID	 = INST_ID;
				pW->flag	 = flag;
				pW->pLine	 = pW->pLineBuf;
				pW->xp		 = xp;
				pW->u.cbCtkA = cbGetLine;
			}
			else {
				free(pW);
				pW = NULL;
			}
		}
//	}

	return pW;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	HAJCTK	WINAPI	AjcCtkCreateW  (UI flag, PCBAJCTKW cbGetLine, UX xp)
{
	HAJCTK	pW = NULL;

//	if (cbGetLine != NULL) {	// コールバックを後で設定する場合もあるので、NULLはチェックしない
		if (pW = (HAJCTK)AJCMEM(sizeof(AJCTK))) {
			memset(pW, 0, sizeof(AJCTK));
			pW->lLineBuf = 1024;
			if ((flag & AJCTKFLG_LBF_MASK) != 0) {
				pW->lLineBuf *= ((((flag & AJCTKFLG_LBF_MASK) >> 24) & 0xFF));
			}
			if (pW->pLineBuf = (WCP)AJCMEM(pW->lLineBuf * 2)) {
				memset(pW->pLineBuf, 0, pW->lLineBuf * 2);
				pW->InstID	 = INST_ID;
				pW->flag	 = flag;
				pW->pLine	 = pW->pLineBuf;
				pW->xp		 = xp;
				pW->u.cbCtkW = cbGetLine;
				pW->fUnicode = TRUE;
			}
			else {
				free(pW);
				pW = NULL;
			}
		}
//	}

	return pW;
}
//==============================================================================================================//
//	リセット（インスタンス生成時の状態に戻す）																	//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcCtkReset	 (HAJCTK pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->tkn			 = (EAJCTKCODE)0;
		pW->suffix		 = (EAJCTKSUF)0;
		pW->ctkf		 = 0;
		pW->err			 = 0;
		pW->lno			 = 0;
		pW->loc			 = 0;
		pW->len			 = 0;
		pW->pLineBuf[0]  = 0;
		pW->pLine		 = pW->pLineBuf;
		pW->PpHistIx	 = 0;
		memset(pW->PpHist, 0, sizeof pW->PpHist);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	コールバック設定																							//
//																												//
//	引　数	：	pW		  - インスタンスハンドル																//
//				cbGetLine - １行リード用コールバック関数のアドレス												//
//				xp		  - コールバックパラメタ																//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCtkSetCallBackA	 (HAJCTK pW, PCBAJCTKA cbGetLine, UX xp)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW) && cbGetLine != NULL) {
		pW->xp		 = xp;
		pW->u.cbCtkA = cbGetLine;
		rc = TRUE;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCtkSetCallBackW	 (HAJCTK pW, PCBAJCTKW cbGetLine, UX xp)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW) && cbGetLine != NULL) {
		pW->xp		 = xp;
		pW->u.cbCtkW = cbGetLine;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	コールバックパラメタ設定																					//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				xp		- コールバックパラメタ																	//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcCtkSetCallBackParam	 (HAJCTK pW, UX xp)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->xp	= xp;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	インスタンス消去																							//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcCtkDelete   (HAJCTK pW)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		if (pW->pLineBuf != NULL) free(pW->pLineBuf);
		free(pW);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	機能フラグ設定																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				flag	- 機能フラグ(AJCTKFLG_XXXXX)															//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcCtkSetFlag	(HAJCTK pW, UI flag)
{
	BOOL	rc = FALSE;

	if (IS_MY_INST(pW)) {
		pW->flag = flag;
	}
	return rc;
}
//==============================================================================================================//
//	機能フラグ取得																								//
//																												//
//	引　数	：	pW		- インスタンスハンドル																	//
//				flag	- 機能フラグ(AJCTKFLG_XXXXX)															//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcCtkGetFlag	(HAJCTK pW)
{
	UI	rc = 0;

	if (IS_MY_INST(pW)) {
		rc = pW->flag;
	}
	return rc;
}
//==============================================================================================================//
//	トークン取得（ＧＥＴ＝取り出し）																			//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pStrBuf  - トークン文字列を格納するバッファのアドレス											//
//				lStrBuf  - トークン文字列を格納するバッファのバイト数／文字数									//
//																												//
//	戻り値	：	TRUE  - 有効なトークンあり																		//
//				FALSE - ＥＯＦ／ワークメモリ割り当て失敗														//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCtkGetTokenA(HAJCTK pW, BCP pStrBuf, UI lStrBuf)
{
	BOOL	rc = FALSE;
	WCP		pTmp;

	if (pStrBuf != NULL) {
		if (pTmp = (WCP)AJCMEM(lStrBuf * 2)) {
			*pTmp = 0;
			if (rc = AjcCtkGetTokenW(pW, pTmp, lStrBuf)) {
				WideCharToMultiByte(CP_ACP, 0, pTmp, -1, pStrBuf, lStrBuf, NULL, NULL);
				pStrBuf[lStrBuf - 1] = 0;
			}
			free(pTmp);
		}
		else if (IS_MY_INST(pW)) {
			pW->err |= AJCTKERR_MEMALLOC;
		}
	}
	else {
		rc = AjcCtkGetTokenW(pW, NULL, 0);
	}

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCtkGetTokenW(HAJCTK pW, WCP pStrBuf, UI lStrBuf)
{
	BOOL	rc = TRUE;

	if (IS_MY_INST(pW)) {
		pW->err = AJCTKERR_OK;
		//	トークン取得
		if (!(pW->ctkf & AJCTKF_ISLASTPEEK)) {						//	直前は「CtkGetToken」？
			rc = SubGetToken(pW, pStrBuf, lStrBuf);					//		次のトークン取得
		}
		else {														//	直前は「CtkPeekToken」？
			if (!(pW->ctkf & AJCTKF_EOF)) {							//		ＥＯＦ以外？
				SetTknStr(pW, &pW->pLineBuf[pW->loc],				//			トークン文字列格納
							 pW->len, pStrBuf, lStrBuf);
			}
			else {													//		ＥＯＦ？
				if (pStrBuf != NULL && lStrBuf >= 1) *pStrBuf = 0;	//			トークン文字列 = ""
				rc = FALSE;
			}
		}
		pW->ctkf &= ~AJCTKF_ISLASTPEEK;								//	直前は「CtkGetToken」の旨、設定	
	}
	else {
		rc = FALSE;
	}

	return rc;
}
//==============================================================================================================//
//	トークン取得（ＰＥＥＫ＝取り出さない）																		//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pStrBuf  - トークン文字列を格納するバッファのアドレス											//
//				lStrBuf  - トークン文字列を格納するバッファのバイト数／文字数									//
//																												//
//	戻り値	：	TRUE  - 有効なトークンあり																		//
//				FALSE - ＥＯＦ／ワークメモリ割り当て失敗														//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCtkPeekTokenA(HAJCTK pW, BCP pStrBuf, UI lStrBuf)
{
	BOOL	rc	 = FALSE;
	WCP		pTmp = NULL;

	if (pStrBuf != NULL && lStrBuf != 0) {
		if (pTmp = (WCP)AJCMEM(lStrBuf * 2)) {
			*pTmp = 0;
			if (rc = AjcCtkPeekTokenW(pW, pTmp, lStrBuf)) {
				WideCharToMultiByte(CP_ACP, 0, pTmp, -1, pStrBuf, lStrBuf, NULL, NULL);
				pStrBuf[lStrBuf - 1] = 0;
			}
			free(pTmp);
		}
		else if (IS_MY_INST(pW)) {
			pW->err |= AJCTKERR_MEMALLOC;
		}
	}
	else {
		rc = AjcCtkPeekTokenW(pW, NULL, 0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcCtkPeekTokenW(HAJCTK pW, WCP pStrBuf, UI lStrBuf)
{
	BOOL	rc = TRUE;

	if (IS_MY_INST(pW)) {
		if (!(pW->ctkf & AJCTKF_ISLASTPEEK)) {						//	直前は「CtkGetToken」？	
			rc = SubGetToken(pW, pStrBuf, lStrBuf);					//		次のトークン取得
		}
		else {														//	直前は「CtkPeekToken」？
			if (!(pW->ctkf & AJCTKF_EOF)) {							//		ＥＯＦ以外？
				SetTknStr(pW, &pW->pLineBuf[pW->loc],				//			トークン文字列格納
							 pW->len, pStrBuf, lStrBuf);
			}
			else {													//		ＥＯＦ？
				if (pStrBuf != NULL && lStrBuf >= 1) *pStrBuf = 0;	//			トークン文字列 = ""
				rc = FALSE;
			}
		}
		pW->ctkf |= AJCTKF_ISLASTPEEK;								//	直前は「CtkPeekToken」の旨、設定
	}
	else rc = FALSE;

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	トークン取得																								//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pStrBuf  - トークン文字列を格納するバッファのアドレス											//
//				lStrBuf  - トークン文字列を格納するバッファのバイト数／文字数									//
//																												//
//	戻り値	：	TRUE  - 有効なトークンあり																		//
//				FALSE - ＥＯＦ																					//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubGetToken(HAJCTK pW, WCP pStrBuf, UI lStrBuf)
{
	BOOL	rc = TRUE;
	UI		len;

	//----- 字句毎の情報初期化 -----//
	pW->ctkf &= ~(AJCTKF_PPTOP | AJCTKF_MACNAME | AJCTKF_MACWITHARG | AJCTKF_MACARG | AJCTKF_MACBODY | AJCTKF_WIDECHAR);
	pW->suffix = EAJCTKSUF_NONE;
	pW->err = AJCTKERR_OK;

	do {
		if (!ReadLineIfEol(pW)) {rc = FALSE; break;}

		if ((pW->flag & AJCTKFLG_INCSPACE ) != 0) {
			do {
				if ((len = gtkIncPath  (pW)) != 0) break;
				if ((len = gtkComment  (pW)) != 0) break;
				if ((len = gtkSpace    (pW)) != 0) break;
				if ((len = gtkEndOfLine(pW)) != 0) break;
				if ((len = gtkLineCont (pW)) != 0) break;
				if ((len = gtkString   (pW)) != 0) break;		//	ワイド文字（L"" / L''）識別のため、シンボルよりも先に解析
				if ((len = gtkSymbol   (pW)) != 0) break;
				if ((len = gtkHexa	   (pW)) != 0) break;
				if ((len = gtkDecimal  (pW)) != 0) break;
				if ((len = gtkOctal    (pW)) != 0) break;
				len = gtkDelimiter(pW);
			} while (0);
			pW->loc = ((UI)((UX)pW->pLine - (UX)pW->pLineBuf)) / 2;
			SetTknStr(pW, pW->pLine, len, pStrBuf, lStrBuf);
			pW->pLine += len;
			pW->len = len;
		}
		else {
			if ((len = gtkIncPath(pW)) == 0) {
				do {
					do {
						if ((len = gtkComment  (pW)) != 0) break;
						if ((len = gtkSpace    (pW)) != 0) break;
						if ((len = gtkEndOfLine(pW)) != 0) break;
						if ((len = gtkLineCont (pW)) != 0) break;
					} while (0);
					pW->pLine += len;
					if (!ReadLineIfEol(pW)) {rc = FALSE; goto sgt_exit;}
				} while (len != 0);
				do {
					if ((len = gtkString   (pW)) != 0) break;		//	ワイド文字（L"" / L''）識別のため、シンボルよりも先に解析
					if ((len = gtkSymbol   (pW)) != 0) break;
					if ((len = gtkHexa	   (pW)) != 0) break;
					if ((len = gtkDecimal  (pW)) != 0) break;
					if ((len = gtkOctal    (pW)) != 0) break;
					len = gtkDelimiter(pW);
				} while (0);
			}
			pW->loc = ((UI)((UX)pW->pLine - (UX)pW->pLineBuf)) / 2;
			SetTknStr(pW, pW->pLine, len, pStrBuf, lStrBuf);
			pW->pLine += len;
			pW->len = len;
		}
		//----- トークンの次に空白ありを示すフラグ設定 ---//
		if (*pW->pLine == L' ' || *pW->pLine == L'\t' ) pW->ctkf |=  AJCTKF_NXTSPC;
		else											pW->ctkf &= ~AJCTKF_NXTSPC;
		//----- マクロ引数／ボディフラグ -----------------//
		if ((pW->ctkf & AJCTKF_PREPRO) != 0) {					//	プリプロセス内？
			if (pW->ctkf & AJCTKF_SRHMACARGEND) {				//		マクロ引数 終端検索中？
				pW->ctkf |= AJCTKF_MACARG;						//			マクロ引数ラグセット
			}
			else if (pW->ctkf & AJCTKF_FINDMACARGEND) {			//		マクロ引数 終検出状態？
				pW->ctkf &= ~AJCTKF_FINDMACARGEND;				//			マクロ引数終検出状態解除
				pW->ctkf |=  AJCTKF_SRHMACBODY;					//			→マクロボディ検索中
			}
			//	マクロボディフラグセット
			else if (pW->ctkf & AJCTKF_SRHMACBODY) {			//		マクロボディ検索中？
				pW->ctkf |= AJCTKF_MACBODY;						//			マクロボディフラグセット
			}
		}

	} while(0);
sgt_exit:;
	return rc;
}
//**************************************************************************************************************//
//	トークン解析関数群																							//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//																												//
//	戻り値	：	≠0 - トークンの長さ（バイト数）																//
//				＝0 - 該当しないトークン																		//
//**************************************************************************************************************//
//--------------------------------------------------------------------------------------------------------------//
//	●シンボル（英字／アンダバー）																				//
//--------------------------------------------------------------------------------------------------------------//
#define	GTK_SYMCHAR(P, F)	if		((pW->flag & AJCTKFLG_INCSYM_MBSTR) && AjcIsBigCharW(P)) {					\
								fSymChar = TRUE;  fRsv = FALSE;													\
								if (MAjcIsLeadW(*P) && MAjcIsTrailW(*(P + 1))) cl = 2;							\
								else										   cl = 1;							\
							}																					\
							else if ((pW->flag & AJCTKFLG_INCSYM_DOLLAR) && *P == L'$') {						\
								fSymChar = TRUE;  cl = 1;  fRsv = FALSE;										\
							}																					\
							else if ((pW->flag & AJCTKFLG_INCSYM_ATMARK) && *P == L'@') {						\
								fSymChar = TRUE;  cl = 1;  fRsv = FALSE;										\
							}																					\
							else if (*P == L'_' || F(*P)) {														\
								fSymChar = TRUE;  cl = 1;														\
							}																					\
							else fSymChar = FALSE

//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkSymbol(HAJCTK pW)
{
	UI		len = 0;					//	シンボルの長さ
	BOOL	fSymChar;					//	シンボル構成文字フラグ
	UI		cl;							//	シンボル構成文字の長さ（1 or 2)
	BOOL	fRsv = TRUE;				//	予約語検索フラグ
	UI		i, r, l;					//	予約語検索ワーク
	SI		cmp;						//	・
	WCP		pe;							//	シンボルの直後の文字アドレス
	WC		tkn[AJCTKRSV_MAXLEN + 1];	//	予約語バッファ

	GTK_SYMCHAR(pW->pLine, MAjcAscIsAlphaW);
	if (fSymChar) {
		//----- トークン抽出 -----------//
		pW->tkn = EAJCTK_USR_NAME;
		pe	= pW->pLine + cl;
		len =			  cl;

		GTK_SYMCHAR(pe, MAjcAscIsAlNumW);
		while (fSymChar) {
			pe	+= cl;
			len += cl;
			GTK_SYMCHAR(pe, MAjcAscIsAlNumW);
		}
		//----- ポート名(P3.12等)判断 --//
		if (pW->flag & AJCTKFLG_DOTSYMBOL) {
			UI		dl = len;
			WCP		pd = pe;
			if (*pd == L'.') {
				pd++; dl++;
				if (MAjcAscIsDigitW(*pd)) {
					do {
						pd++; dl++;
					} while(MAjcAscIsDigitW(*pd));
					pe	= pd;
					len = dl;
				}
			}
		}
		//----- トークンコード設定 -----//
		if (len <= AJCTKRSV_MAXLEN && fRsv) {
			wcsncpy(tkn, pW->pLine, len);
			tkn[len] = 0;
			l=0; r=MAX_RSVTBL-1;
			while (l < r) {
				i = (l + r) / 2;
				cmp = wcscmp(CtkRsvTbl[i].pStr, tkn);
				if		(cmp == 0) {
					if ((pW->flag & AJCTKFLG_CPLUSPLUS) != 0 || !CtkRsvTbl[i].fCPP) {
						pW->tkn = CtkRsvTbl[i].tkn;
					}
					goto stk_exit;
				}
				else if (cmp < 0) l = i + 1;
				else			  r = i;
			}
			if (wcscmp(CtkRsvTbl[l].pStr, tkn) == 0) {
				if ((pW->flag & AJCTKFLG_CPLUSPLUS) != 0 || !CtkRsvTbl[l].fCPP) {
					pW->tkn = CtkRsvTbl[l].tkn;
				}
			}
stk_exit:;
		}
		//----- マクロチェック ---------//
		if ((pW->ctkf & AJCTKF_PREPRO) != 0) {					//	プリプロセス内？
			//	マクロ名チェック
			if (pW->PpHistIx == 2						&&		//		トークン数＝２		＆
				pW->PpHist[0] == EAJCTK_DLM_SHARP		&&		//		先頭＝"#"			＆
				pW->PpHist[1] == EAJCTK_RSV_DEFINE) {			//		２番＝"define" ？
				pW->ctkf |= AJCTKF_MACNAME;						//			「マクロ名」
				if (*pe == L'(') {								//			マクロ名の直後は "(" （引数付きマクロ）？
					pW->ctkf |= AJCTKF_MACWITHARG;				//				「引数付マクロ」
					pW->ctkf |= AJCTKF_SRHMACARGTOP;			//				→マクロ引数先頭 '(' 検索状態
				}
				else {											//			マクロの直後は '(' 以外？（引数無しマクロ）？
					pW->ctkf |= AJCTKF_FINDMACARGEND;			//				仮に引数終端検知をセットし、次からマクロボディとする
				}
			}
		}
		//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --//
		SetPreproTokenHist(pW);
		//----- 行頭フラグクリアー -----//
		pW->ctkf &= ~AJCTKF_TOL;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●１６進数（'０ｘ' / '・・・H'）																			//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkHexa(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe;

	//	先頭が「０ｘ」
	if (wcsnicmp(pW->pLine, L"0x", 2) == 0) {
		pW->tkn = EAJCTK_VAL_HEXA;
		pe	= pW->pLine + 2;
		len = 2;
		if (MAjcAscIsXDigitW(*pe)) {
			while (MAjcAscIsXDigitW(*pe)) {
				pe++; len++;
			}
			len += gtkSuffix(pW, pe);
		}
		else {
			pW->err |= AJCTKERR_INVALID_HEXADECIMAL;
		}
		//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --//
		SetPreproTokenHist(pW);
		//----- 行頭フラグクリアー -----//
		pW->ctkf &= ~AJCTKF_TOL;
	}
	//	末尾が「H」
	else if (pW->flag & AJCTKFLG_ASMHEX) {
		pe	= pW->pLine;
		len = 0;
		while (MAjcIsXDigitW(*pe)) {
			pe++; len++;
		}
		//	末尾が「H」ならば、トークン情報設定
		if (*pe == L'H' || *pe == L'h') {
			pe++; len++;
			pW->tkn = EAJCTK_VAL_HEXA_H;
			//----- マクロボディフラグ -----//
			if ((pW->ctkf & AJCTKF_PREPRO) != 0) {					//	プリプロセス内？
				//	マクロボディフラグセット
				if (pW->ctkf & AJCTKF_SRHMACBODY) {					//		マクロボディ検索中？
					pW->ctkf |= AJCTKF_MACBODY;						//			マクロボディフラグ接っと
				}
			}
			//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --//
			SetPreproTokenHist(pW);
			//----- 行頭フラグクリアー -----//
			pW->ctkf &= ~AJCTKF_TOL;
		}
		//	末尾が「H」以外ならば、トークン不成立
		else len = 0;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●１０進整数／実数（数字 or 数字＋ドット）																	//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkDecimal(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe;

	if (MAjcAscIsDigitW(*pW->pLine) || (*pW->pLine == '.' && MAjcAscIsDigitW(*(pW->pLine + 1)))) {
		pW->tkn = EAJCTK_VAL_DECIMAL;
		pe	= pW->pLine;
		len = 0;
		while (MAjcAscIsDigitW(*pe)) {
			pe++; len++;
		}
		if (*pe == L'.') {
			pW->tkn = EAJCTK_VAL_REAL;
			pe++; len++;
			while (MAjcAscIsDigitW(*pe)) {
				pe++; len++;
			}
		}
		if (*pe == L'E' || *pe == L'e') {
			pW->tkn = EAJCTK_VAL_REAL;
			pe++; len++;
			if (*pe == L'+' || *pe == L'-') {
				pe++; len++;
			}
			if (MAjcAscIsDigitW(*pe)) {
				while (MAjcAscIsDigitW(*pe)) {
					pe++; len++;
				}
			}
			else {
				pW->err |= AJCTKERR_INVALID_REALNUMBER;
			}
		}
		if (pW->err == 0) {
			len += gtkSuffix(pW, pe);
		}
		//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --//
		SetPreproTokenHist(pW);
		//----- 行頭フラグクリアー -----//
		pW->ctkf &= ~AJCTKF_TOL;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●８進数（'０'）																							//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkOctal(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe;

	if (*pW->pLine == L'0') {
		pW->tkn = EAJCTK_VAL_OCTAL;
		pe	= pW->pLine + 1;
		len = 1;
		while (MAjcAscIsDigitW(*pe)) {
			if (*pe >= L'8') {
				pW->err |= AJCTKERR_INVALID_OCTALNUMBER;
			}
			pe++; len++;
		}
		len += gtkSuffix(pW, pe);
		//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --//
		SetPreproTokenHist(pW);
		//----- 行頭フラグクリアー -----//
		pW->ctkf &= ~AJCTKF_TOL;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●文字列（ " , ' , L" or L' ）																				//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkString(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe;
	WC		c;

	if ((*pW->pLine == L'"' || *pW->pLine == L'\'') ||
			 (*pW->pLine == L'L' && (*(pW->pLine + 1) == L'"' || *(pW->pLine + 1) == L'\''))) {
		if (*pW->pLine == L'L') {
			pW->ctkf |= AJCTKF_WIDECHAR;
			pW->pLine++;
		}
		if (*pW->pLine == L'"') pW->tkn = EAJCTK_STR_QUOTE;
		else					pW->tkn = EAJCTK_STR_APOST;
		c	= *pW->pLine;
		pe	= pW->pLine + 1;
		len = 1;
		while (*pe != 0  &&  *pe != c) {
			//	アポストロフィ(')で囲まれた文字列は、エスケース文字(\)を認識しない場合
			if (pW->tkn == EAJCTK_STR_APOST && (pW->flag & AJCTKFLG_APOST_NOESC)) {
				pe++; len++;
			}
			//	エスケース文字(\)を認識する場合
			else {
				if (*pe == L'\\') {
					pe++; len++;
					if (*pe != 0) {
						pe++; len++;
					}
				}
				else {
					pe++; len++;
				}
			}
		}
		if (*pe == c) {pe++; len++;}
		else		  {pW->err |= AJCTKERR_LFINSTR;}
		//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --//
		SetPreproTokenHist(pW);
		//----- 行頭フラグクリアー -----//
		pW->ctkf &= ~AJCTKF_TOL;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●デリミタ																									//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkDelimiter(HAJCTK pW)
{
	UI		len = 0;
	UI		i;
	WCP		pe = pW->pLine;

	for (i=0; i<MAX_DLMTBL; i++) {
		if (wcsncmp(CtkDlmTbl[i].pStr, pW->pLine, CtkDlmTbl[i].lStr) == 0 &&
			 ((pW->flag & AJCTKFLG_CPLUSPLUS) != 0 || !CtkDlmTbl[i].fCPP)) {
			break;
		}
	}
	if (i < MAX_DLMTBL) {
		len = CtkDlmTbl[i].lStr;
		pe	= pW->pLine + len;
		pW->tkn = CtkDlmTbl[i].tkn;
	}
	else {
		len = 1;
		pe	= pW->pLine + 1;
		pW->tkn  = EAJCTK_DLM_UNKNOWN;
		pW->err |= AJCTKERR_INVALID_DELIMITER;
	}
	//----- プリプロセス開始チェック -----//
	if ((pW->ctkf & AJCTKF_PREPRO) == 0	&&				//	プリプロセス文外	＆
		(pW->ctkf & AJCTKF_TOL	 ) != 0	&&				//	行頭				＆
		pW->tkn == EAJCTK_DLM_SHARP) {					//	'#' ？
		pW->ctkf = (AJCTKF_PREPRO | AJCTKF_PPTOP);		//		「プリプロセス先頭」設定

		pW->PpHistIx = 0;								//		プリプロセス・トークンカウンタ クリアー	
	}
	//----- マクロ引数開始／終了チェック -//
	if (pW->ctkf & AJCTKF_PREPRO) {
		//	マクロ名チェック
		if (pW->PpHistIx >= 2						&&		//		トークン数＝２		＆
			pW->PpHist[0] == EAJCTK_DLM_SHARP		&&		//		先頭＝"#"			＆
			pW->PpHist[1] == EAJCTK_RSV_DEFINE) {			//		２番＝"define" ？
			if (pW->ctkf & AJCTKF_SRHMACARGTOP) {			//			マクロ引数先頭 '(' 検索状態？
				if (pW->tkn == EAJCTK_DLM_LSPART) {			//				トークン＝ "(" ？
					pW->ctkf |=  AJCTKF_MACARG;				//					フラグ（マクロ引数）セット
					pW->ctkf &= ~AJCTKF_SRHMACARGTOP;		//					マクロ引数先頭 '(' 検索状態解除
					pW->ctkf |=  AJCTKF_SRHMACARGEND;		//					マクロ引数終端 ')' 検索状態セット
				}
			}
			else if (pW->ctkf & AJCTKF_SRHMACARGEND) {		//			マクロ引数終端 ')' 検索状態？
				if (pW->tkn == EAJCTK_DLM_RSPART) {			//				トークン＝ ")" ？
					pW->ctkf |=  AJCTKF_MACARG;				//					マクロ引数ラグセット
					pW->ctkf &= ~AJCTKF_SRHMACARGEND;		//					マクロ引数終端 ')' 検索状態解除
					pW->ctkf |=  AJCTKF_FINDMACARGEND;		//					マクロボディ検索状態セット
				}
			}
		}
	}
	//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --------//
	SetPreproTokenHist(pW);
	//----- 行頭フラグクリアー -----------//
	pW->ctkf &= ~AJCTKF_TOL;

	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●#include <path> の 「path」部分																			//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkIncPath(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe = pW->pLine;

	if ((pW->ctkf & AJCTKF_INCOMMENT) == 0	&&		//	コメント外			＆
		(pW->ctkf & AJCTKF_PREPRO) != 0		&&		//	プリプロセス内		＆
		pW->PpHistIx == 3					&&		//	トークン数＝３		＆
		pW->PpHist[0] == EAJCTK_DLM_SHARP	&&		//	先頭＝"#"			＆
		pW->PpHist[1] == EAJCTK_RSV_INCLUDE	&&		//	２番＝"include" 	＆
		pW->PpHist[2] == EAJCTK_DLM_LO) {			//	３番＝"<" ？

		pW->tkn  = EAJCTK_PATHNAME;

		while (*pe != L'>' && *pe != L'\r' && *pe != L'\n' && *pe != 0) {
			pe++;
			len++;
		}

		//----- ﾌﾟﾘﾌﾟﾛｾｽ ﾄｰｸﾝ履歴設定 --------//
		SetPreproTokenHist(pW);
		//----- 行頭フラグクリアー -----------//
		pW->ctkf &= ~AJCTKF_TOL;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●コメント																									//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkComment(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe = pW->pLine;

	if ((pW->ctkf & AJCTKF_INCOMMENT) != 0) {
		pW->tkn = EAJCTK_COMMENT;
		len = SubChkCommentEnd(pW, pe);
	}
	else if (wcsncmp(pe, L"/*", 2) == 0) {
		pW->tkn   = EAJCTK_COMMENT;
		pW->ctkf |= AJCTKF_INCOMMENT;
		pe	+= 2;
		len += 2;
		len += SubChkCommentEnd(pW, pe);
	}
	else if (wcsncmp(pe, L"//", 2) == 0) {
		pW->tkn   = EAJCTK_LINECOMMENT;
		pe	+= 2;
		len += 2;
		len += SubChkLineEnd(pe);
	}

	return len;
}
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubChkCommentEnd(HAJCTK pW, WCP pe)
{
	UI		len = 0;

	while (*pe != L'\r' && *pe != L'\n' && *pe != 0 && wcsncmp(pe, L"*/", 2) != 0) {
		pe++;
		len++;
	}
	if (wcsncmp(pe, L"*/", 2) == 0) {
		pe	+= 2;
		len += 2;
		pW->ctkf &= ~AJCTKF_INCOMMENT;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
static	UI	SubChkLineEnd(WCP pe)
{
	UI		len = 0;

	while (*pe != L'\r' && *pe != L'\n' && *pe != 0) {
		pe++;
		len++;
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●空白																										//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkSpace(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe = pW->pLine;

	if (*pe == L' ' || *pe == L'\t') {
		pW->tkn = EAJCTK_SPACE;
		while (*pe == L' ' || *pe == L'\t') {
			pe++;
			len++;
		}
	}

	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●改行																										//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkEndOfLine(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe = pW->pLine;

	if (*pe == L'\r' || *pe == L'\n') {
		pW->tkn = EAJCTK_EOL;
		while (*pe == L'\r' || *pe == L'\n') {
			pe++;
			len++;
		}
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	●行継続記号（￥）																							//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkLineCont(HAJCTK pW)
{
	UI		len = 0;
	WCP		pe = pW->pLine;
	WCP		pn = pe + 1;

	if (*pe == L'\\' && (*pn == L'\r' || *pn == L'\n' || *pn == 0)) {
		pW->ctkf |= AJCTKF_LCONT;
		pW->tkn = EAJCTK_LINECONT;
		len = 1;
	}

	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	数値定数のサフィクス・チェック																				//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				pe		 - 数値トークンの直後のアドレス															//
//																												//
//	戻り値	：	≠0 - サフィクスの長さ（バイト数）																//
//				＝0 - サフィクスなし																			//
//--------------------------------------------------------------------------------------------------------------//
static	UI	gtkSuffix(HAJCTK pW, WCP pe)
{
	UI		len;
	WCP		ps;

	//----- 整数のサフィクス・チェック -----//
	if (pW->tkn == EAJCTK_VAL_DECIMAL || pW->tkn == EAJCTK_VAL_HEXA || pW->tkn == EAJCTK_VAL_OCTAL) {
		ps = pe;
		for (len=0; MAjcAscIsAlphaW(*pe); len++, pe++);
		if (len != 0) {
			if		(len == 3 && wcsnicmp(ps, L"ULL", 3) == 0) {		//	ULL	
				pW->suffix = EAJCTKSUF_ULL;
			}
			else if (len == 2 && wcsnicmp(ps, L"LL", 2) == 0) {			//	LL
				pW->suffix = EAJCTKSUF_LL;
			}
			else if (len == 2 && (wcsnicmp(ps, L"UL", 2) == 0  ||		//	UL / LU
								  wcsnicmp(ps, L"LU", 2) == 0)) {
				pW->suffix = EAJCTKSUF_UL;
			}
			else if (len == 1 && (*ps == L'L' || *ps == L'l')) {		//	L
				pW->suffix = EAJCTKSUF_L;
			}
			else {														//	その他
				pW->err |= AJCTKERR_INVALID_SUFFIX;
			}
		}
	}
	//----- 実数のサフィクス・チェック -----//
	else if (pW->tkn == EAJCTK_VAL_REAL) {
		ps = pe;
		for (len=0; MAjcAscIsAlphaW(*pe); len++, pe++);
		if (len != 0) {
			if		(len == 1 && (*ps == L'F' || *ps == L'f')) {		//	F
				pW->suffix = EAJCTKSUF_FLOAT;
			}
			else if (len == 1 && (*ps == L'D' || *ps == L'd')) {		//	D
				pW->suffix = EAJCTKSUF_DOUBLE;
			}
			else if (len == 1 && (*ps == L'L' || *ps == L'l')) {		//	L
				pW->suffix = EAJCTKSUF_LDOUBLE;
			}
			else {														//	その他
				pW->err |= AJCTKERR_INVALID_SUFFIX;
			}
		}
	}
	return len;
}
//--------------------------------------------------------------------------------------------------------------//
//	トークン文字列をバッファに格納																				//
//																												//
//	引　数	：	pTkn	 - トークンの先頭アドレス																//
//				lTkn	 - トークンのバイト数																	//
//				pStrBuf  - トークン文字列を格納するバッファのアドレス											//
//				lStrBuf  - トークン文字列を格納するバッファのバイト数											//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - バッファオーバー																		//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SetTknStr(HAJCTK pW, WCP pTkn, UI lTkn, WCP pStrBuf, UI lStrBuf)
{
	BOOL	rc = TRUE;

	do {
		if (pStrBuf == NULL   ) {pW->err |= AJCTKERR_NULLPTR;	break;}
		if (lStrBuf == 0	  ) {pW->err |= AJCTKERR_ZEROSIZE;	break;}

		if (lTkn < lStrBuf) {
			wcsncpy(pStrBuf, pTkn, lTkn);
			*(pStrBuf + lTkn) = 0;
			rc = TRUE;	
		}
		else {
			wcsncpy(pStrBuf, pTkn, lStrBuf - 1);
			*(pStrBuf + (lStrBuf - 1)) = 0;
			pW->ctkf |= AJCTKERR_BUFOVER;
			rc = FALSE;
		}
	} while(0);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//	プリプロセスのトークン履歴設定																				//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - ＥＯＦ																					//
//--------------------------------------------------------------------------------------------------------------//
static	VO	SetPreproTokenHist(HAJCTK pW)
{
	if ((pW->ctkf & AJCTKF_PREPRO) != 0) {
		if (pW->PpHistIx < AJCTKMAX_PPHIST) {
			pW->PpHist[pW->PpHistIx] = pW->tkn;
		}
		pW->PpHistIx++;
	}
}

//--------------------------------------------------------------------------------------------------------------//
//	行末ならば１行リード																						//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//																												//
//	戻り値	：	TRUE  - ＯＫ																					//
//				FALSE - ＥＯＦ／ワークメモリ割り当て失敗														//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	ReadLineIfEol(HAJCTK pW)
{
	BOOL	rc = TRUE;
	BCP		pTmpBuf = NULL;

	if (!(pW->ctkf & AJCTKF_EOF)) {	//	ＥＯＦ未検出？
		if (*pW->pLine == 0) {		//	行末？
			do {
				//----- ワイド文字 -----------------------------------------------------------------------------//
				if (pW->fUnicode) {
					if (pW->u.cbCtkW != NULL && pW->u.cbCtkW(pW->pLineBuf, pW->lLineBuf, pW->xp)) {	//	１行リード，ＥＯＦ以外？
						pW->pLineBuf[pW->lLineBuf - 1] = 0;											//		末尾に強制的に終端を設定
						if (!(pW->ctkf & AJCTKF_LCONT)) {											//		行継続記号（￥）なし
							pW->ctkf &= ~AJCTKF_PREPRO;												//			「プリプロセス文外」
						}
						pW->ctkf &= ~AJCTKF_LCONT;													//		「行継続記号出現」ﾌﾗｸﾞｸﾘｱｰ
						pW->ctkf |= AJCTKF_TOL;														//		「行頭」
						pW->pLine = pW->pLineBuf;													//		行ポインタ初期化
						pW->lno++;																	//		行番号更新
					}
					else {																			//	ＥＯＦ？
						pW->ctkf &= ~AJCTKF_USER;													//		ユーザ通知フラグクリアー
						pW->ctkf |= AJCTKF_EOF;														//		「ＥＯＦ」フラグ設定
						pW->tkn   = EAJCTK_EOF;														//		トークンコード設定
						if (pW->ctkf & AJCTKF_INCOMMENT) {											//		コメントは閉じていない？
							pW->err |= AJCTKERR_EOFINCOMMENT;										//			エラーフラグ設定
						}
						rc = FALSE;																	//		戻り値＝ＥＯＦ
						break;
					}
				}
				else {
				//----- バイト文字 -----------------------------------------------------------------------------//
					if (pTmpBuf = (BCP)AJCMEM(pW->lLineBuf * 2)) {									//ワイド文字、一時バイトバッファ確保成功？
						memset(pTmpBuf, 0, pW->lLineBuf * 2);
						if (pW->u.cbCtkA != NULL && pW->u.cbCtkA(pTmpBuf, pW->lLineBuf, pW->xp)) {	//	１行リード，ＥＯＦ以外？
							pTmpBuf[pW->lLineBuf * 2 - 1] = 0;										//		末尾に強制的に終端を設定
							MultiByteToWideChar(CP_ACP, 0, pTmpBuf, -1,								//		ワイド文字→バイト文字変換
									 pW->pLineBuf, pW->lLineBuf);									//		・
							if (!(pW->ctkf & AJCTKF_LCONT)) {										//		行継続記号（￥）なし
								pW->ctkf &= ~AJCTKF_PREPRO;											//			「プリプロセス文外」
							}
							pW->ctkf &= ~AJCTKF_LCONT;												//		「行継続記号出現」ﾌﾗｸﾞｸﾘｱｰ
							pW->ctkf |=  AJCTKF_TOL;												//		「行頭」
							pW->pLine = pW->pLineBuf;												//		行ポインタ初期化
							pW->lno++;																//		行番号更新
						}
						else{																		//	ＥＯＦ？
							pW->ctkf &= ~AJCTKF_USER;												//		ユーザ通知フラグクリアー
							pW->ctkf |= AJCTKF_EOF;													//		「ＥＯＦ」フラグ設定
							pW->tkn   = EAJCTK_EOF;													//		トークンコード設定
							if (pW->ctkf & AJCTKF_INCOMMENT) {										//		コメントは閉じていない？
								pW->err |= AJCTKERR_EOFINCOMMENT;									//			エラーフラグ設定
							}
							rc = FALSE;																//		戻り値＝ＥＯＦ
							break;
						}
					}
					else {																			//ワイド文字バッファ確保失敗？
						pW->err |= AJCTKERR_MEMALLOC;
						rc = FALSE;
						break;
					}
				}
			} while (pW->pLineBuf[0] == 0);

			if (pTmpBuf != NULL) free(pTmpBuf);
		}
	}
	else {							//	ＥＯＦ検出済？
		rc = FALSE;					//		戻り値＝ＥＯＦ
	}
	return rc;
}
//==============================================================================================================//
//	トークン文字列取得																							//
//																												//
//	引　数	：	tkn 	 - トークンコード																		//
//																												//
//	戻り値	：	≠NULL - トークン文字列のアドレス																//
//				＝NULL - 不正なトークンコード																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	C_BCP	WINAPI	AjcCtkGetTokenStringA(EAJCTKCODE tkn)
{
	static	BC	buf[32];
	C_BCP	rc = buf;

	if		(tkn >= EAJCTK_RSV_MINCODE && tkn <= EAJCTK_RSV_MAXCODE) {
		WideCharToMultiByte(CP_ACP, 0, CtkRsvTbl[tkn - EAJCTK_RSV_MINCODE].pStr, -1, buf, AJCTSIZE(buf), NULL, NULL);
	}
	else if (tkn >= EAJCTK_DLM_MINCODE && tkn <= EAJCTK_DLM_MAXCODE) {
		WideCharToMultiByte(CP_ACP, 0, CtkDlmTbl[tkn - EAJCTK_DLM_MINCODE].pStr, -1, buf, AJCTSIZE(buf), NULL, NULL);
	}
	else {
		rc = NULL;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	C_WCP	WINAPI	AjcCtkGetTokenStringW(EAJCTKCODE tkn)
{
	C_WCP	rc;

	if		(tkn >= EAJCTK_RSV_MINCODE && tkn <= EAJCTK_RSV_MAXCODE) {
		rc = CtkRsvTbl[tkn - EAJCTK_RSV_MINCODE].pStr;
	}
	else if (tkn >= EAJCTK_DLM_MINCODE && tkn <= EAJCTK_DLM_MAXCODE) {
		rc = CtkDlmTbl[tkn - EAJCTK_DLM_MINCODE].pStr;
	}
	else {
		rc = NULL;
	}
	return rc;
}
//==============================================================================================================//
//	トークンの桁位置取得（ＣＴＫ＿ＬＯＣマクロ処理）															//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//																												//
//	戻り値	：	トークンの桁位置																				//
//																												//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcCtkGetTokenLocationA(HAJCTK pW)
{
	UI		rc = 0;
	UI		ix = 0;

	if (IS_MY_INST(pW)) {
		rc = WideCharToMultiByte(CP_ACP, 0, pW->pLineBuf, pW->loc, NULL, 0, NULL, NULL);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcCtkGetTokenLocationW(HAJCTK pW)
{
	UI		rc = 0;

	if (IS_MY_INST(pW)) {
		rc = pW->loc;
	}
	return rc;
}

//==============================================================================================================//
//	ＴＡＢを考慮したトークンの桁位置取得（ＣＴＫ＿ＰＯＳマクロ処理）											//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//				TabStep  - ＴＡＢステップ幅																		//
//																												//
//	戻り値	：	トークンの桁位置																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcCtkGetTokenPositionA(HAJCTK pW, UI TabStep)
{
	UI		rc = 0;
	UI		ix = 0;
	BCP		pTmp = NULL;

	if (IS_MY_INST(pW)) {
		if (pTmp = (BCP)AJCMEM(pW->lLineBuf * 2)) {
			memset(pTmp, 0, pW->lLineBuf * 2);
			WideCharToMultiByte(CP_ACP, 0, pW->pLineBuf, pW->loc, pTmp, pW->lLineBuf * 2, NULL, NULL);
			while (ix < pW->lLineBuf * 2 && pTmp[ix] != 0) {
				if (pTmp[ix] == '\t') {
					if (TabStep != 0) {
						rc += TabStep;
						rc -= (rc % TabStep);
					}
					ix++;
				}
				else {
					if (_ismbblead(pTmp[ix])) {ix += 2; rc += 2;}
					else					  {ix++;	rc++;	}
				}
			}
			free(pTmp);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcCtkGetTokenPositionW(HAJCTK pW, UI TabStep)
{
	UI		rc = 0;
	UI		ix = 0;

	if (IS_MY_INST(pW)) {
		while (ix < pW->loc) {
			if (pW->pLineBuf[ix] == L'\t') {
				if (TabStep != 0) {
					rc += TabStep;
					rc -= (rc % TabStep);
				}
				ix++;
			}
			else {
				if (MAjcIsLeadW(pW->pLineBuf[ix])) ix += 2;
				else							   ix++;
				rc++;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	インスタンスのコピーを取得																					//
//																												//
//	引　数	：	pW		 - インスタンスハンドル																	//
//																												//
//	戻り値	：	≠NULL : 成功（コピーされたインスタンスハンドル）												//
//				＝NULL : エラー																					//
//																												//
//	備　考	：	複製ハンドル(hDup)を使用した場合、元のハンドル(hCtk)操作の続きとなります。						//
//				例えば、AjcCtkGetToken(hDup・・)を実行した場合、元のハンドルで最後に取得したトークンの次の		//
//			　トークンを取得します。																			//
//				元のハンドル（インスタンス）は更新されませんので、AjcCtkGetToken(hCtk・・)は、（通常通り）		//
//			　最後に実行した AjcCtkGetToken(hCtk) の次のトークンを取得します。									//
//				コピーされたインスタンスハンドルは、AjcCtkDelete(hDup)により消去しなければなりません。			//
//==============================================================================================================//
AJCEXPORT	HAJCTK	WINAPI	AjcCtkGetReplicatedHandle(HAJCTK pW)
{
	HAJCTK	rc = NULL;
	UX		loc;

	if (IS_MY_INST(pW)) {
		do {
			//	複製インスタンス割り当て
			if ((rc 		  = (HAJCTK)AJCMEM(sizeof(AJCTK))) == NULL) {break;}
			//	複製インスタンスへインスタンスデータコピー
			memcpy(rc, pW, sizeof(AJCTK));
			//	複製インスタンス内の行バッファ割り当て
			if ((rc->pLineBuf = (WCP   )AJCMEM(rc->lLineBuf * 2)) == NULL) {free(rc); rc = NULL; break;}
			//	複製インスタンス内の行バッファポインタ設定
			loc = ((UX)pW->pLine - (UX)pW->pLineBuf) / 2;
			rc->pLine = rc->pLineBuf + loc;
			//	複製インスタンス内の行バッファへデータコピー
			memcpy(rc->pLineBuf, pW->pLineBuf, pW->lLineBuf * 2);
		} while(0);
	}
	return rc;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//																												//
//	ＭＳＩＬ用サポートコード																					//
//																												//
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//##############################################################################################################//
//--------------------------------------------------------------------------------------------------------------//
//	現在のトークン情報取得																						//
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	EAJCTKCODE	WINAPI	AjcCtkGetCurToken		(HAJCTK pW)					{return AJCTK_TOKEN	(pW);			}
AJCEXPORT	EAJCTKSUF	WINAPI	AjcCtkGetCurSuffix		(HAJCTK pW)					{return AJCTK_SUFFIX(pW);			}
AJCEXPORT	UI			WINAPI	AjcCtkGetCurFlag		(HAJCTK pW)					{return AJCTK_FLAG	(pW);			}
AJCEXPORT	UI			WINAPI	AjcCtkGetCurLineNumber	(HAJCTK pW)					{return AJCTK_LINE	(pW);			}
AJCEXPORT	UI			WINAPI	AjcCtkGetCurPosition	(HAJCTK pW, UI TabWidth)	{return AJCTK_POS	(pW, TabWidth);	}
AJCEXPORT	UI			WINAPI	AjcCtkGetCurError		(HAJCTK pW)					{return AJCTK_ERROR	(pW);			}

AJCEXPORT	BOOL		WINAPI	AjcCtkIsUsrSym			(EAJCTKCODE tkn)			{return AJCTKIS_USRSYM	(tkn);		}
AJCEXPORT	BOOL		WINAPI	AjcCtkIsRsvSym			(EAJCTKCODE tkn)			{return AJCTKIS_RSVSYM	(tkn);		}
AJCEXPORT	BOOL		WINAPI	AjcCtkIsValue			(EAJCTKCODE tkn)			{return AJCTKIS_VALUE 	(tkn);		}
AJCEXPORT	BOOL		WINAPI	AjcCtkIsString			(EAJCTKCODE tkn)			{return AJCTKIS_STRING	(tkn);		}
AJCEXPORT	BOOL		WINAPI	AjcCtkIsPathName		(EAJCTKCODE tkn)			{return AJCTKIS_PATHNAME(tkn);		}
AJCEXPORT	BOOL		WINAPI	AjcCtkIsDelimit			(EAJCTKCODE tkn)			{return AJCTKIS_DELIMIT	(tkn);		}

AJCEXPORT	BOOL		WINAPI	AjcCtkIsSymbol			(EAJCTKCODE tkn)			{return AJCTKIS_SYMBOL	(tkn);		}
AJCEXPORT	BOOL		WINAPI	AjcCtkIsValSym			(EAJCTKCODE tkn)			{return AJCTKIS_VALSYM	(tkn);		}
AJCEXPORT	BOOL		WINAPI	AjcCtkIsSpace			(EAJCTKCODE tkn)			{return AJCTKIS_SPACE	(tkn);		}

