﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	チェックサム																								//
//																												//
//**************************************************************************************************************//

//**************************************************************************************************************//
//																												//
//	バイトサム																									//
//																												//
//**************************************************************************************************************//
/*==============================================================================================================*/
/*	Function	:	バイトストリームのチェックサム算出（単純加算）												*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	バイトストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UB	WINAPI	AjcCalcByteSum(C_VOP sp, UL len)
{
	UB		r = 0;

	if (sp != NULL) {
		while (len--) {
			r += *((C_UBP)sp)++;
		}
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	バイトストリームのチェックサム算出（１の補数）												*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	バイトストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UB	WINAPI	AjcCalcByteSumN(C_VOP sp, UL len)
{
	UB		r = 0;

	if (sp != NULL) {
		while (len--) {
			r += *((C_UBP)sp)++;
		}
		r = ~r;
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	バイトストリームのチェックサム算出（２の補数）												*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	バイトストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UB	WINAPI	AjcCalcByteSumS(C_VOP sp, UL len)
{
	UB		r = 0;

	if (sp != NULL) {
		while (len--) {
			r += *((C_UBP)sp)++;
		}
		r = ( ~r + 1 );
	}
	return( r );
}
/*==============================================================================================================*/
/*	Function	:	バイトストリームのチェックサム算出（排他論理和）											*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	バイトストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UB	WINAPI	AjcCalcByteSumX(C_VOP sp, UL len)
{
	UB		r = 0;

	if (sp != NULL) {
		while (len--) {
			r ^= *((C_UBP)sp)++;
		}
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	バイトＳＵＭ設定関数群（ストリームの末尾１バイトにバイトサムを設定）						*/
/*																												*/
/*	Argument	:	rec - ストリームの先頭アドレス																*/
/*					rl  - ストリームバイト数																	*/
/*																												*/
/*	Return		:	TRUE  - ＯＫ																				*/
/*					FALSE - パラメタエラー																		*/
/*==============================================================================================================*/
AJCEXPORT BOOL WINAPI AjcSetByteSum (VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		*((UBP)((UBP)rec + (rl - 1))) = AjcCalcByteSum (rec, rl - 1);
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetByteSumN(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		*((UBP)((UBP)rec + (rl - 1))) = AjcCalcByteSumN(rec, rl - 1);
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetByteSumS(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		*((UBP)((UBP)rec + (rl - 1))) = AjcCalcByteSumS(rec, rl - 1);
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetByteSumX(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		*((UBP)((UBP)rec + (rl - 1))) = AjcCalcByteSumX(rec, rl - 1);
		rc = TRUE;
	}
	return rc;
}

/*==============================================================================================================*/
/*	Function	:	バイトＳＵＭチェック関数群（ストリームの末尾１バイトに設定されているバイトサムチェック）	*/
/*																												*/
/*	Argument	:	rec - ストリームの先頭アドレス																*/
/*					rl  - ストリームバイト数																	*/
/*																												*/
/*	Return		:	TRUE ：チェックＯＫ																			*/
/*					FALSE：チェックＮＧ／パラメタエラー															*/
/*==============================================================================================================*/
AJCEXPORT BOOL WINAPI AjcChkByteSum (C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		rc = ( (*((C_UBP)((C_UBP)rec + (rl - 1))) == AjcCalcByteSum (rec, rl - 1)) * TRUE );
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkByteSumN(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		rc = ( (*((C_UBP)((C_UBP)rec + (rl - 1))) == AjcCalcByteSumN(rec, rl - 1)) * TRUE );
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkByteSumS(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		rc = ( (*((C_UBP)((C_UBP)rec + (rl - 1))) == AjcCalcByteSumS(rec, rl - 1)) * TRUE );
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkByteSumX(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 1) {
		rc = ( (*((C_UBP)((C_UBP)rec + (rl - 1))) == AjcCalcByteSumX(rec, rl - 1)) * TRUE );
	}
	return rc;
}
//**************************************************************************************************************//
//																												//
//	ワードサム																									//
//																												//
//**************************************************************************************************************//
/*==============================================================================================================*/
/*	Function	:	ワードストリームのチェックサム算出（単純加算）												*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSum(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r += *((const UW *)sp)++;
			len -= 2;
		}
		if (len) {
			r += *((C_UBP)sp);
		}
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	ワードストリームのチェックサム算出（１の補数）												*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSumN(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r += *((const UW *)sp)++;
			len -= 2;
		}
		if (len) {
			r += *((C_UBP)sp);
		}
		r = ~r;
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	ワードストリームのチェックサム算出（２の補数）												*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSumS(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r += *((const UW *)sp)++;
			len -= 2;
		}
		if (len) {
			r += *((C_UBP)sp);
		}
		r = ( ~r + 1 );
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	ワードストリームのチェックサム算出（排他論理和）											*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（パラメタエラー時は、０を返す）							*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSumX(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r ^= *((const UW *)sp)++;
			len -= 2;
		}
		if (len) {
			r ^= *((C_UBP)sp);
		}
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	ワードＳＵＭ設定関数群（ストリームの末尾２バイトへワードサムを設定）						*/
/*																												*/
/*	Argument	:	rec - ストリームの先頭アドレス																*/
/*					rl  - ストリームバイト数																	*/
/*																												*/
/*	Return		:	TRUE  - ＯＫ																				*/
/*					FALSE - パラメタエラー																		*/
/*==============================================================================================================*/
AJCEXPORT BOOL WINAPI AjcSetWordSum (VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL) {
		if (rl >= 2) {
			*((UW *)((UBP)rec + (rl - 2))) = AjcCalcWordSum (rec, rl - 2);
			rc = TRUE;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetWordSumN(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL) {
		if (rl >= 2) {
			*((UW *)((UBP)rec + (rl - 2))) = AjcCalcWordSumN(rec, rl - 2);
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetWordSumS(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL) {
		if (rl >= 2) {
			*((UW *)((UBP)rec + (rl - 2))) = AjcCalcWordSumS(rec, rl - 2);
			rc = TRUE;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetWordSumX(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL) {
		if (rl >= 2) {
			*((UW *)((UBP)rec + (rl - 2))) = AjcCalcWordSumX(rec, rl - 2);
			rc = TRUE;
		}
	}
	return rc;
}

/*==============================================================================================================*/
/*	Function	:	ワードＳＵＭチェック関数群（ストリームの末尾２バイトに設定されているワードサムチェック）	*/
/*																												*/
/*	Argument	:	rec - ストリームの先頭アドレス																*/
/*					rl  - ストリームバイト数																	*/
/*																												*/
/*	Return		:	＝１：チェックＯＫ																			*/
/*					＝０：チェックＮＧ／パラメタエラー															*/
/*==============================================================================================================*/
AJCEXPORT BOOL WINAPI AjcChkWordSum (C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( *((const UW *)((C_UBP)rec + (rl - 2))) == AjcCalcWordSum (rec, rl - 2));
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkWordSumN(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( *((const UW *)((C_UBP)rec + (rl - 2))) == AjcCalcWordSumN(rec, rl - 2));
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkWordSumS(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( *((const UW *)((C_UBP)rec + (rl - 2))) == AjcCalcWordSumS(rec, rl - 2));
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkWordSumX(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( *((const UW *)((C_UBP)rec + (rl - 2))) == AjcCalcWordSumX(rec, rl - 2));
	}
	return rc;
}

//**************************************************************************************************************//
//																												//
//	ワードサム（逆エンディアン）																				//
//																												//
//**************************************************************************************************************//
/*==============================================================================================================*/
/*	Function	:	逆のエンディアンでワードストリームのチェックサム算出（単純加算）							*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（戻り値は、当該処理装置のエンディアン形式）				*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSumR(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r += AjcExcWord(*((const UW *)sp)++);
			len -= 2;
		}
		if (len) {
			r += *((C_UBP)sp);
		}
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	逆のエンディアンでワードストリームのチェックサム算出（１の補数）							*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（戻り値は、当該処理装置のエンディアン形式）				*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSumNR(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r += AjcExcWord(*((const UW *)sp)++);
			len -= 2;
		}
		if (len) {
			r += *((C_UBP)sp);
		}
		r = ~r;
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	逆のエンディアンでワードストリームのチェックサム算出（２の補数）							*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（戻り値は、当該処理装置のエンディアン形式）				*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSumSR(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r += AjcExcWord(*((const UW *)sp)++);
			len -= 2;
		}
		if (len) {
			r += *((C_UBP)sp);
		}
		r = ( ~r + 1 );
	}
	return( r );
}
/*==============================================================================================================*/
/*	Function	:	逆のエンディアンでワードストリームのチェックサム算出（排他論理和）							*/
/*																												*/
/*	Argument	:	sp  - バイトストリームの先頭アドレス														*/
/*					len - バイト数																				*/
/*																												*/
/*	Return		:	ワードストリームのチェックサム値（戻り値は、当該処理装置のエンディアン形式）				*/
/*==============================================================================================================*/
AJCEXPORT	UW	WINAPI	AjcCalcWordSumXR(C_VOP sp, UL len)
{
	UW		r = 0;

	if (sp != NULL) {
		while (len >= 2) {
			r ^= AjcExcWord(*((const UW *)sp)++);
			len -= 2;
		}
		if (len) {
			r ^= *((C_UBP)sp);
		}
	}
	return( r );
}

/*==============================================================================================================*/
/*	Function	:	逆のエンディアンでワードＳＵＭ設定関数群（ストリームの末尾２バイトへワードサムを設定）		*/
/*																												*/
/*	Argument	:	rec - ストリームの先頭アドレス																*/
/*					rl  - ストリームバイト数																	*/
/*																												*/
/*	Return		:	TRUE  - ＯＫ																				*/
/*					FALSE - パラメタエラー																		*/
/*==============================================================================================================*/
AJCEXPORT BOOL WINAPI AjcSetWordSumR (VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		*((UW *)((UBP)rec + (rl - 2))) = AjcExcWord(AjcCalcWordSumR (rec, rl - 2));
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetWordSumNR(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		*((UW *)((UBP)rec + (rl - 2))) = AjcExcWord(AjcCalcWordSumNR(rec, rl - 2));
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetWordSumSR(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		*((UW *)((UBP)rec + (rl - 2))) = AjcExcWord(AjcCalcWordSumSR(rec, rl - 2));
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcSetWordSumXR(VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		*((UW *)((UBP)rec + (rl - 2))) = AjcExcWord(AjcCalcWordSumXR(rec, rl - 2));
		rc = TRUE;
	}
	return rc;
}

/*==============================================================================================================*/
/*	Function	:	逆のエンディアンでワードＳＵＭチェック関数群（ｽﾄﾘｰﾑの末尾2ﾊﾞｲﾄに設定されているﾜｰﾄﾞｻﾑﾁｪｯｸ）	*/
/*																												*/
/*	Argument	:	rec - ストリームの先頭アドレス																*/
/*					rl  - ストリームバイト数																	*/
/*																												*/
/*	Return		:	＝１：チェックＯＫ																			*/
/*					＝０：チェックＮＧ／パラメタエラー															*/
/*==============================================================================================================*/
AJCEXPORT BOOL WINAPI AjcChkWordSumR (C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( (*(const UW *)((C_UBP)rec + (rl - 2))) == AjcExcWord(AjcCalcWordSumR (rec, rl - 2)));
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkWordSumNR(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( (*(const UW *)((C_UBP)rec + (rl - 2))) == AjcExcWord(AjcCalcWordSumNR(rec, rl - 2)));
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkWordSumSR(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( (*(const UW *)((C_UBP)rec + (rl - 2))) == AjcExcWord(AjcCalcWordSumSR(rec, rl - 2)));
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL WINAPI AjcChkWordSumXR(C_VOP rec, UL rl)
{
	BOOL	rc = FALSE;

	if (rec != NULL && rl >= 2) {
		rc = ( (*(const UW *)((C_UBP)rec + (rl - 2))) == AjcExcWord(AjcCalcWordSumXR(rec, rl - 2)));
	}
	return rc;
}


