﻿#include	"AjcInternal.h"
#include	"AjcCipCommon.h"

//**************************************************************************************************************//
//																												//
//	Ｃ言語風インタプリタ（ＡＰＩ）																				//
//																												//
//**************************************************************************************************************//
//==============================================================================================================//
//	実行中止																									//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT BOOL			WINAPI AjcCipStop			(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		pW->fRunStop = TRUE;
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	オプションの設定																							//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				fEnableOption	- 「option」文の解析許可（TRUE:解析する，FALSE:解析しない）						//
//				NumType			- 数値のタイプ（AJCIPT_INTEGER or AJCIPT_REAL）									//
//				fSigned			- 整数「AJCIPT_INTEGER」時の符号の有無											//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipSetOption	(HWND hwnd, BOOL fEnableOption, EAJCCIPTYP NumType, BOOL fSigned)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		pW->fEnableOption	= (fEnableOption != FALSE);
		pW->fSigned			= (fSigned		 != FALSE);
		if (NumType == AJCIPT_INTEGER || NumType == AJCIPT_REAL) {
			pW->NumType = NumType;
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ダブルクリック時の動作設定																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				fNtcDblClk	- TRUE	- 親ウインドへ通知															//
//							  FALSE - _OnDblClk()実行															//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL			WINAPI AjcCipSetDblClkAct	(HWND hwnd, BOOL fNtcDblClk)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		pW->fNtcDblClk = fNtcDblClk;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	コード（マクロテキストファイル）の読み出し																	//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pMacFile	- コードファイルパス																//
//																												//
//	戻り値	：	TRUE  - 成功（実行可）																			//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipLoad	(HWND hwnd, C_BCP pMacFile)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		do {
			pW->fRunReady = FALSE;

			if (pMacFile == NULL) break;							//	ファイルパスチェック
			AjcSnPrintFA(pW->MacFile, MAX_PATH, "%s", pMacFile);
			_splitpath(pW->MacFile, pW->MacDrv, pW->MacDir, pW->MacFName, pW->MacFExt);
			_makepath (pW->MacPath, pW->MacDrv, pW->MacDir, NULL		, NULL		 );

			if (!(rc = CipReadMacroText(pW))) break;				//	コードファイル読み出し
			if (!(rc = CipAnalize	   (pW))) break;				//	コード解析（プリコンパイル）

			pW->fRunReady = TRUE;									//	「実行準備完了」

			rc = TRUE;
		} while(0);
	}

	return rc;
}
//==============================================================================================================//
//	コード（マクロテキスト）の実行																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pCall		- 関数呼び出し情報																	//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
//----- 標準イベントキュー消去時のコールバック -----------------------------------------------------------------//
static VO CALLBACK cbRemoveStdEvt(VOP pDat, UX cbp)
{
	HAJCCIP			pW	  = (HAJCCIP)cbp;
	PAJCCIPCALL		pCall = *((PAJCCIPCALL *)pDat);

	if (pCall != NULL) {
		AjcCipCallInfoDelete(pW->hWndBack, pCall);					//		呼び出し情報破棄
	}
}
//----- ＳＣＰイベントキュー消去時のコールバック ---------------------------------------------------------------//
static VO CALLBACK cbRemoveScpEvt(VOP pDat, UX cbp)
{
	HAJCCIP			pW	 = (HAJCCIP)cbp;
	PAJCCIPSCPEVT	p	 = (PAJCCIPSCPEVT)pDat;

	if (p->pDat != NULL) {
		free(p->pDat);
		p->pDat = NULL;
	}
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI AjcCipExec	(HWND hwnd, PAJCCIPCALL pCall)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;
	UI			i;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		do {
			if (!pW->fRunReady) {
				pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_NOTREADY, 0,
											  LNGSEL("マクロ実行の準備ができていません。",
													 "Not ready for macro execution."));
				break;
			}
			if (pW->fExec) {
				pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_EXEC, 0,
											  LNGSEL("マクロは既に実行中です。",
													 "Macro is already executing."));
				break;
			}

			//----- 実行前処理 ---------------------------------------------------------------------------------//
			//	エラーコードリセット
			pW->ErrCod = AJCIP_ERR_OK;
			//	フラグ情報初期化
			pW->fRunStop	   = FALSE;							//	実行中止フラグ
			pW->fExeStop	   = FALSE;							//	実行中断フラグ( _Stop()  )
			pW->fExePause	   = FALSE;							//	実行中断フラグ( _Pause() )
			pW->fExeFileSearch = FALSE;							//	ファイル検索中フラグ
			pW->fBusyOnChkBox  = FALSE;							//	_OnChkBox()実行中フラグ
			pW->fExeEventBusy  = FALSE;							//	イベント実行中フラグ
			pW->fExeStdEvtBusy = FALSE;							//	標準  イベント実行中フラグ
			pW->fExeComEvtBusy = FALSE;							//	ＣＯＭイベント実行中フラグ
			pW->nExeDisStdEvt  = 0;								//	標準  イベント許可状態
			pW->nExeDisComEvt  = 0;								//	ＣＯＭイベント許可状態
			pW->mskExeTimEvt   = 0;								//	タイマイベント動作中フラグ
			pW->fExeRClick	   = 0;								//	右クリック発生フラグ
			pW->fExeDblClk	   = 0;								//	ダブルクリック発生フラグ
			memset(pW->fExeTimer, 0, sizeof pW->fExeTimer);		//	タイマイベント発生フラグ
			pW->fExeEnaStdPopupMenu = FALSE;					//	右クリックによる標準ポップアップの許可状態(FALSE:禁止)
			//	Exit()テキスト初期化
			memset(pW->ExitText, 0, sizeof pW->ExitText);
			//	標準イベントキュー生成
			pW->hFQueStdEvt = AjcFQueCreate(sizeof(PAJCCIPCALL ), (UX)pW, cbRemoveStdEvt);
			//	ＳＣＰイベントキュー生成
			pW->hFQueScpEvt = AjcFQueCreate(sizeof(AJCCIPSCPEVT), (UX)pW, cbRemoveScpEvt);
			//	ウインドテキスト消去
			AjcVthSetTitleText(pW->hWndVth, NULL, -1, -1, NULL);
			//	ステップトレース情報初期化
			pW->fStepPause		= FALSE;						//	一時停止中フラグ
			pW->nStepCount		= 0;							//	ステップ実行カウンタ
			pW->fStepMsgBoxBusy	= FALSE;						//	メッセージボックス表示中フラグ
			pW->pStepTkn		= NULL;							//	ステップトレース表示用トークンアドレス
			pW->SvStepLno		= -1;							//	ステップトレース表示用行番号退避
			pW->SvBrkLno		= -1;							//	ブレーク処理用行番号退避
			pW->SvStepTkn		= EAJCTK_INVALID;				//	ステップトレース表示用トークンコード退避
			pW->ExeStepNo		= 0;							//	ダイナミックステップ番号
			//	ＴＭＣウインドクローズ
			if (pW->hWndTmcBack != NULL) {DestroyWindow(pW->hWndTmcBack); pW->hWndTmcBack = NULL;}
			//	デフォルトのプロファイルセクション設定
			AjcSnPrintFA(pW->ExeSect, sizeof pW->ExeSect, "%s\\DefSect", pW->MacFName);

			//----- 関数実行開始 -------------------------------------------------------------------------------//
			//	実行開始通知
			pW->fExec		 = TRUE;
			pW->ixIfNest	 = 0;
			pW->fStepSvPause = ~pW->fStepPause;
			pW->fStepSvExec  = ~pW->fExec;
			CipStepNtcExec(pW, FALSE);
			//	先頭の関数実行
			rc = CipExeFuncCall(pW, pCall, NULL);
			//	実行終了通知
			pW->fExec = FALSE;
			CipStepNtcExec(pW, FALSE);

			//----- 実行後処理 ---------------------------------------------------------------------------------//
			//	全タイマ停止
			CipGsrStopAllTimer(pW);
			//	右クリックを標準に戻す
			AjcVthSetNtcRClk(pW->hWndVth, FALSE, 0, 0);
			//	ダブルクリック行テキスト解放
			if (pW->pExeDblClkLine != NULL) {
				free(pW->pExeDblClkLine);
				pW->pExeDblClkLine = NULL;
			}
			//	標準イベントキュー解放
			if (pW->hFQueStdEvt != NULL) {
				PAJCCIPCALL pCall;
				while (AjcFQueDeque(pW->hFQueStdEvt, &pCall)) {
					AjcCipCallInfoDelete(pW->hWndBack, pCall);
				}
				AjcFQueDelete(pW->hFQueStdEvt);
				pW->hFQueStdEvt = NULL;
			}
			//	ＳＣＰイベントキュー解放
			if (pW->hFQueScpEvt != NULL) {
				AjcFQueDelete(pW->hFQueScpEvt);
				pW->hFQueScpEvt = NULL;
			}
			//	ＳＣＰ消去,ＳＣＰ通知用ウインド破棄
			for (i = 0; i < AJCCIPMAX_SCP; i++) {
				if (pW->ExeScp[i].hScp != NULL) {
					AjcScpDelete(pW->ExeScp[i].hScp);
					pW->ExeScp[i].hScp = NULL;
				}
				if (pW->ExeScp[i].hwnd != NULL) {
					DestroyWindow(pW->ExeScp[i].hwnd);
					pW->ExeScp[i].hwnd = NULL;
				}
			}
			memset(pW->ExeScp, 0, sizeof pW->ExeScp);
			//	ファイルクローズ
			for (i = 1; i < AJCCIPMAX_FILES; i++) {
				if (pW->ExeFile[i].hFile != NULL) {
					fclose(pW->ExeFile[i].hFile);
					pW->ExeFile[i].hFile = NULL;
				}
			}
			memset(pW->ExeFile, 0, sizeof pW->ExeFile);
			//	メモリブロック解放
			for (i = 1; i < AJCCIPMAX_MEMBLK; i++) {
				if (pW->ExeMemBlk[i].pMemAddr != NULL) {
					free(pW->ExeMemBlk[i].pMemAddr);
					pW->ExeMemBlk[i].pMemAddr = NULL;
				}
			}
			memset(pW->ExeMemBlk, 0, sizeof pW->ExeMemBlk);
			//	ボタンクローズ
			for (i = 0; i < MAX_BUTTON; i++) {
				if (pW->hButton[i] != NULL) {
					DestroyWindow(pW->hButton[i]);
					pW->hButton[i] = NULL;
				}
			}
			pW->mskBtnValid = 0;
			//	チェックボックスクローズ
			for (i = 0; i < MAX_CHKBOX; i++) {
				if (pW->hChkBox[i] != NULL) {
					DestroyWindow(pW->hChkBox[i]);
					pW->hChkBox[i] = NULL;
				}
			}
			pW->mskChkValid = 0;
			//	外部関数呼び出し時の引数情報
			if (pW->ppNtcAgv != NULL) {
				free(pW->ppNtcAgv);
				pW->ppNtcAgv = NULL;
			}
			//	ボタン，チェックボックスの高さクリアー
			pW->BtnChkHeight  = 0;

			//	VT100ウインド配置
			CipGsrMoveVT100Window(pW);
		} while(0);
	}
	return rc;
}
//==============================================================================================================//
//	ユーザイベントの実行																						//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pCall		- 関数呼び出し情報																	//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipEvent	(HWND hwnd, PAJCCIPCALL pCall)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		do {
			if (!pW->fExec) {
				pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_EXEC, 0,
											  LNGSEL("マクロは実行されていません。",
													 "Macro is not executed."));
				break;
			}
			pCall->pReturn = pW->pCur;						//	戻りアドレス（ダミー，実際にはエンキュー時に再設定される）
			AjcFQueEnque(pW->hFQueStdEvt, (VOP)&pCall);		//	イベント・エンキュー
			rc = TRUE;
		} while(0);
	}
	return rc;
}

//==============================================================================================================//
//	文字列プリント																								//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pFmt		- 出力文字列																		//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT BOOL			WINAPI AjcCipPutS			(HWND hwnd, C_BCP pTxt)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		AjcVthPutTextA(pW->hWndVth, pTxt, -1);
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	書式文字列プリント																							//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pFmt		- 書式文字列																		//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT BOOL			WINAPI AjcCipPrintF			(HWND hwnd, C_BCP pFmt, ...)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;
	va_list 	vls;
	BC			txt[MAX_PATH + 1024];

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		va_start(vls, pFmt);
		_vsnprintf(txt, sizeof txt, pFmt, vls);
		va_end	(vls);
		txt[(sizeof txt) - 1] = 0;

		if (pW->hWndVth != NULL) {
			AjcVthPutTextA(pW->hWndVth, txt, -1);
		}
		else {
			printf("%s", txt);
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	ユーザ関数からのエラー通知																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pCall		- 関数呼び出し情報																	//
//				pFmt		- 書式文字列																		//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcCipNtcErrByUser	(HWND hwnd, PCAJCCIPCALL pCall, C_BCP pFmt, ...)
{
	HAJCCIP			pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL			rc = FALSE;
	AJCCIPEVPERR	cbp;
	va_list 		vls;
	UI				stl;					//	"ファイルパス名(行番号)" の長さ

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		//	ファイルパス名，行番号設定
		AjcSnPrintFA(pW->NtcTxt, sizeof pW->NtcTxt, "%s(%4d): ", pW->MacFile, pCall->lno);
		stl = (UI)strlen(pW->NtcTxt);
		//	メッセージ設定
		va_start(vls, pFmt);
		_vsnprintf(&pW->NtcTxt[stl], (sizeof pW->NtcTxt) - stl, pFmt, vls);
		va_end	(vls);
		pW->NtcTxt[(sizeof pW->NtcTxt) - 1] = 0;
		//	エラー通知
		cbp.err  = AJCIP_ERR_USER;
		cbp.lno  = pCall->lno;
		cbp.pMsg = pW->NtcTxt;
		SendMessage(pW->hParent, WM_COMMAND, MAKELONG(pW->CtrlId, AJCCIPN_ERR),
											 fCmdWithHdl ? (LPARAM)pW->hWndBack : (LPARAM)&cbp);
		//	エラーコード設定
		pW->ErrCod = AJCIP_ERR_USER;
		//	戻り値＝正常
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	_Exit()マクロ関数により指定されたテキストの取得																//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	≠NULL	? 正常（_Exit()で指定されたテキストへのポインタ）										//
//				＝NULL	? エラー																				//
//==============================================================================================================//
AJCEXPORT C_BCP			WINAPI AjcCipGetExitText	(HWND hwnd)
{
	C_BCP	rc = NULL;

	HAJCCIP			pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		rc = pW->ExitText;
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたファイル名取得																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				buf	  - ファイル名を格納するバッファのアドレス													//
//																												//
//	戻り値	：	TRUE:ファイル名あり，FALSE:ファイル名なし														//
//==============================================================================================================//
AJCEXPORT BOOL			WINAPI AjcCipGetDroppedFile (HWND hwnd, BC buf[MAX_PATH])
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pW->hWndVth != NULL) {
			AjcVthGetDroppedFileA  (pW->hWndVth, buf);
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	ドロップされたディレクトリ名取得																			//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				buf	  - ディレクトリ名を格納するバッファのアドレス												//
//																												//
//	戻り値	：	TRUE:ディレクトリ名あり，FALSE:ディレクトリ名なし												//
//==============================================================================================================//
AJCEXPORT BOOL 			WINAPI AjcCipGetDroppedDir	(HWND hwnd, BC buf[MAX_PATH])
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pW->hWndVth != NULL) {
			AjcVthGetDroppedDirA  (pW->hWndVth, buf);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたディレクトリ名取得																			//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				buf	  - ディレクトリ名を格納するバッファのアドレス												//
//																												//
//	戻り値	：	TRUE:ディレクトリ名あり，FALSE:ディレクトリ名なし												//
//==============================================================================================================//
AJCEXPORT BOOL 			WINAPI AjcCipGetDroppedDirEx(HWND hwnd, BC buf[MAX_PATH], BOOL fTailIsDelimiter)
{
	BOOL		rc = FALSE;

	if (rc = AjcCipGetDroppedDir(hwnd, buf)) {
		if (fTailIsDelimiter) AjcPathCatA(buf, "", MAX_PATH);
		else				  PathRemoveBackslashA(buf);
	}
	return rc;
}

//==============================================================================================================//
//	関数呼び出し情報の初期化																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pFuncName	- 関数名																			//
//																												//
//	戻り値	：	≠NULL - 成功（生成した関数呼び出し情報のアドレス）												//
//				＝NULL - 失敗																					//
//==============================================================================================================//
AJCEXPORT PAJCCIPCALL	WINAPI AjcCipCallInfoInit	(HWND hwnd, C_BCP pFuncName)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	PAJCCIPCALL	pCallInfo  = NULL;
	PAJCCIPFUNC	pFunc	= NULL;
	UI			stl;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pFuncName != NULL && *pFuncName != 0) {															//	関数名正常？
			stl = (UI)strlen(pFuncName);																	//		関数名の長さ設定
			if (pCallInfo = (PAJCCIPCALL)AJCMEM(sizeof(AJCCIPCALL))) {										//		呼び出し情報割り当て，成功？
				memset(pCallInfo, 0, sizeof(AJCCIPCALL));													//			呼び出し情報クリアー
				if ((pFunc = (PAJCCIPFUNC)AjcAvlGetNodePtr(pW->hAvlFun, (UX)pFuncName, NULL)) != NULL) {	//			関数名検索，成功？
					pCallInfo->lno = pFunc->pEntry->lno;													//				関数の行番号設定
					if (pCallInfo->pFunName = (BCP)AJCMEM(stl + 1)) {										//				関数名領域確保，成功？
						strcpy(pCallInfo->pFunName, pFuncName);												//					関数名コピー
						if ((pCallInfo->hQueAgv = AjcFQueCreate(sizeof(AJCCIPAGV), 0, NULL)) != NULL) {		//					引数キュー生成，成功？
							pCallInfo->rchVmg = pW->sFni[0].hVmg;											//						関数の仮戻り変数設定
							if (*pFuncName == '$') {														//						・
								pCallInfo->rcpSym = "$@rc";													//						・
							}
							else {																			//						・
								pCallInfo->rcpSym = "@rc";													//						・
							}
							pCallInfo->rcIx = 0;															//						・ 
						}
						else {																				//					引数キュー生成失敗？
							AjcCipCallInfoDelete(pW->hWndBack, pCallInfo);									//						呼び出し情報破棄
							pCallInfo = NULL;																//						戻り値＝ＮＵＬＬ
						}
					}
					else {																					//				関数名領域確保失敗
						AjcCipCallInfoDelete(pW->hWndBack, pCallInfo);										//					呼び出し情報破棄
						pCallInfo = NULL;																	//					戻り値＝ＮＵＬＬ
					}
				}
				else {																						//			件数名検索失敗？
					AjcCipCallInfoDelete(pW->hWndBack, pCallInfo);											//				呼び出し情報破棄
					pCallInfo = NULL;																		//				戻り値＝ＮＵＬＬ
				}
			}
		}
	}
	return pCallInfo;
}
//==============================================================================================================//
//	関数呼び出し情報の破棄																						//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pCallInfo	- 初期化する関数呼び出し情報のバッファ												//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipCallInfoDelete	(HWND hwnd, PAJCCIPCALL pCallInfo)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;
	PAJCCIPAGV	pNode;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		//	関数名解放
		if (pCallInfo->pFunName != NULL) {
			free(pCallInfo->pFunName);
			pCallInfo->pFunName = NULL;
		}
		//	引数キュー解放
		if (pCallInfo->hQueAgv != NULL) {
			if (pNode = (PAJCCIPAGV)AjcFQueTopNode(pCallInfo->hQueAgv)) {
				do {
					if (pNode->typ == AJCIPT_STRING && pNode->val.pStr != NULL) {
						free((VOP)pNode->val.pStr);
						pNode->val.pStr = NULL;
					}
				} while(pNode = (PAJCCIPAGV)AjcFQueNextNode(pCallInfo->hQueAgv, pNode));
			}
			AjcFQueDelete(pCallInfo->hQueAgv);
			pCallInfo->hQueAgv = NULL;
		}
		//	呼び出し情報解放
		free(pCallInfo);
		//	戻り値＝正常
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	関数呼び出し情報へ引数（整数）追加																			//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pCallInfo	- 関数呼び出し情報																	//
//				value		- 引数の値																			//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcCipCallInfoInsInteger	(HWND hwnd, PAJCCIPCALL pCallInfo, SLL value)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;
	AJCCIPAGV	agv;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pCallInfo->hQueAgv != NULL) {
			agv.seq		= pCallInfo->nArg;
			agv.typ		= AJCIPT_INTEGER;
			agv.val.sll = value;
			agv.fArr	= FALSE;
			if (rc = AjcFQueEnque(pCallInfo->hQueAgv, &agv)) {
				pCallInfo->nArg++;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	関数呼び出し情報へ引数（実数）追加																			//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pCallInfo	- 関数呼び出し情報																	//
//				value		- 引数の値																			//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcCipCallInfoInsReal	(HWND hwnd, PAJCCIPCALL pCallInfo, double value)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;
	AJCCIPAGV	agv;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pCallInfo->hQueAgv != NULL) {
			agv.typ		= AJCIPT_REAL;
			agv.val.dbl = value;
			agv.fArr	= FALSE;
			if (rc = AjcFQueEnque(pCallInfo->hQueAgv, &agv)) {
				pCallInfo->nArg++;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	関数呼び出し情報へ引数（文字列）追加																		//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pCallInfo		- 関数呼び出し情報																//
//				value			- 引数の値																		//
//																												//
//	戻り値	：	TRUE  - 成功																					//
//				FALSE - 失敗																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcCipCallInfoInsString	(HWND hwnd, PAJCCIPCALL pCallInfo, C_BCP pStr)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;
	AJCCIPAGV	agv;
	UI			stl = (UI)strlen(pStr);

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pCallInfo->hQueAgv != NULL) {
			agv.typ		= AJCIPT_STRING;
			agv.fArr	= FALSE;
			if (agv.val.pStr = (C_BCP)AJCMEM(stl + 1)) {
				strcpy((BCP)agv.val.pStr, pStr);
				if (rc = AjcFQueEnque(pCallInfo->hQueAgv, &agv)) {
					pCallInfo->nArg++;
					rc = TRUE;
				}
				else {
					free((VOP)agv.val.pStr);
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	関数の戻り値取得（整数）																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pNum		- 戻り値の配列要素数を格納するバッファ（不要時はNULL)								//
//																												//
//	戻り値	：	≠NULL - 戻り値（64Bit整数値）配列の先頭アドレス												//
//				＝NULL - 失敗																					//
//==============================================================================================================//
AJCEXPORT C_ULLP	WINAPI AjcCipGetRetInt(HWND hwnd, UIP pNum)
{
	HAJCCIP			pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	C_ULLP			rc = NULL;
	PCAJCVMGNODEA	pNode;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pNode = AjcVmgGetNodeA(pW->sFni[0].hVmg, "@rc")) {
			if (pNum != NULL) *pNum = pNode->num;
			rc = (C_ULLP)pNode->arr.pSLL;
		}
	}
	return rc;
}
//==============================================================================================================//
//	関数の戻り値取得（実数）																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pNum		- 戻り値の配列要素数を格納するバッファ（不要時はNULL)								//
//																												//
//	戻り値	：	≠NULL - 戻り値（実数値）配列の先頭アドレス														//
//				＝NULL - 失敗																					//
//==============================================================================================================//
AJCEXPORT const double * WINAPI AjcCipGetRetReal(HWND hwnd, UIP pNum)
{
	HAJCCIP			pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	const double *	rc = NULL;
	PCAJCVMGNODEA	pNode;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pNode = AjcVmgGetNodeA(pW->sFni[0].hVmg, "@rc")) {
			if (pNum != NULL) *pNum = pNode->num;
			rc = pNode->arr.pDBL;
		}
	}
	return rc;
}
//==============================================================================================================//
//	関数の戻り値取得（文字列）																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pNum		- 戻り値の配列要素数を格納するバッファ（不要時はNULL)								//
//																												//
//	戻り値	：	≠NULL - 戻り値(文字列へのポインタ）配列の先頭アドレス											//
//				＝NULL - 失敗																					//
//==============================================================================================================//
AJCEXPORT const C_BCP *  WINAPI AjcCipGetRetStr(HWND hwnd, UIP pNum)
{
	HAJCCIP			pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	const C_BCP		*rc = NULL;
	PCAJCVMGNODEA	pNode;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pNode = AjcVmgGetNodeA(pW->sFni[0].hVmg, "@rc")) {
			if (pNum != NULL) *pNum = pNode->num;
			rc = pNode->arr.pStr;
		}
	}
	return rc;
}
//==============================================================================================================//
//	トレース表示 ウインド生成																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipCreateTraceWindow(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		rc = CipTrcCreateWindow(pW);
	}
	return rc;
}
//==============================================================================================================//
//	トレース表示 ウインド破棄																					//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipDestroyTraceWindow(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (pW->hTrcBack != NULL) {
			DestroyWindow(pW->hTrcBack);
		}
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	ステップ・トレース ウインド生成																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipCreateStepWindow(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		rc = CipStepCreateWindow(pW);
	}
	return rc;
}
//==============================================================================================================//
//	ステップ・トレース ウインド破棄																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL	WINAPI AjcCipDestroyStepWindow(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		if (!pW->fExec) {
			if (pW->hStepBack != NULL) {
				DestroyWindow(pW->hStepBack);
			}
			rc = TRUE;
		}
		else rc = (pW->hStepBack == NULL);
	}
	return rc;
}
//==============================================================================================================//
//	VT100エミュレーションウインドのハンドル取得																	//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	≠NULL : OK(VT100エミュレーションウインドのハンドル)											//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT HWND			WINAPI AjcCipGetVT100Handle		(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	HWND		rc = NULL;

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		rc = pW->hWndVth;
	}
	return rc;
}
//==============================================================================================================//
//	ステータス情報取得																							//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL			WINAPI AjcCipGetStatus			(HWND hwnd, PAJCCIPSTATE pState)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;

	if (pW != NULL && pW->id == AJCCIP_INSTID && pState != NULL) {
		//	数値タイプ設定
		switch (pW->NumType) {
			case AJCIPT_INTEGER:
				if (pW->fSigned) pState->eType = AJCCIPSTS_SINT;
				else			 pState->eType = AJCCIPSTS_UINT;
				break;
			case AJCIPT_REAL:
				pState->eType = AJCCIPSTS_REAL;
				break;
			default:
				pState->eType = AJCCIPSTS_UNKOWN;
				break;
		}
		//	実行状態設定
		if (pW->fExec) {
			pState->eExec = AJCCIPSTS_EXEC;
		}
		else {
			if (pW->fRunReady) pState->eExec = AJCCIPSTS_READY;
			else			   pState->eExec = AJCCIPSTS_NOTREADY;
		}
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	デバッグ用マクロテキスト解析結果ダンプ出力																	//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
static BOOL CALLBACK cbNtcVar(PCAJCVMGNODEA pVarInfo, UX cbp)
{
	FILE		*hOut = (FILE*)cbp;
	UI			i;
	UBP			p;

	for (i = 0; i < pVarInfo->num; i++) {
		UI	stl = (UI)strlen(pVarInfo->pName);
		if (pVarInfo->num == 1) {
			fprintf(hOut, "%s = ", pVarInfo->pName);
		}
		else {
			if (i == 0) fprintf(hOut, "%s[%2d] = ", pVarInfo->pName, i);
			else		fprintf(hOut, "%*c[%2d] = ", stl, ' ', i);
		}
		switch (pVarInfo->type) {
			case AJCVMGT_INT:	fprintf(hOut, "%d"	, pVarInfo->arr.pSI [i]);	break;
			case AJCVMGT_LONG:	fprintf(hOut, "%lld", pVarInfo->arr.pSLL[i]);	break;
			case AJCVMGT_REAL:	fprintf(hOut, "%f"	, pVarInfo->arr.pDBL[i]);	break;
			case AJCVMGT_STR:
				fprintf(hOut, "\"");
				for (p = (UBP)pVarInfo->arr.pStr[i]; *p != 0; p++) {
					if (*p < 0x20) fprintf(hOut, "\\x%02x", *p);
					else		   fprintf(hOut, "%c"	  , *p);
				}
				fprintf(hOut, "\"");
				break;
		}
		fprintf(hOut, "\n");
	}
	return TRUE;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	WINAPI AjcCipDump(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BOOL		rc = FALSE;
	FILE		*hOut;
	PAJCCIPTKN	pTkn;
	UW			lno, nest;
	UI			i;
	BC			path[MAX_PATH];

	if (pW != NULL && pW->id == AJCCIP_INSTID) {
		GetTempPathA(MAX_PATH, path);
		AjcPathCatA(path, "AjrCip.txt", MAX_PATH);
		if (hOut = fopen(path, "wt")) {
			//	トークン情報ダンプ
			fprintf(hOut, "----- Token information ----------\n");
			if (pTkn = pW->pTknTop) {
				nest = 0;
				lno  = pTkn->lno;
				fprintf(hOut, "[%4d]\n", lno);
				while (pTkn->tkn != EAJCTK_EOF) {
					if (lno != pTkn->lno) {
						lno = pTkn->lno;
						fprintf(hOut, "[%4d]\n", lno);
					}
					if (pTkn->tkn == EAJCTK_DLM_RMPART) {	//	'}'
						nest--;
					}
					fprintf(hOut, "%p:%2d: %*c'", pTkn, nest, (nest + 1) * 2, ' ');
					switch (pTkn->tkn) {
						case EAJCTK_RSV_IF:			//	・if
						case EAJCTK_RSV_ELSE:		//	・else
						case EAJCTK_RSV_WHILE:		//	・while
						case EAJCTK_RSV_FOR:		//	・for
						case EAJCTK_RSV_BREAK:		//	・break
						case EAJCTK_RSV_GOTO:		//	・goto
							fprintf(hOut, "%s JMP=%p", CipGsrTknStr(pTkn), pTkn->u.pJmp);
							switch (pTkn->flgCtrl) {
								case AJCIP_WHILE_FRONT:		fprintf(hOut, " (前置き 'while')");	break;
								case AJCIP_WHILE_TAIL:		fprintf(hOut, " (後置き 'while')");	break;
							}
							break;
						case EAJCTK_DLM_SEMICOL:	//	・;
						case EAJCTK_DLM_RMPART:		//	・}
						case EAJCTK_DLM_RSPART:		//	・)
							if (pTkn->flgCtrl != 0) fprintf(hOut, "%s JMP=%p", CipGsrTknStr(pTkn), pTkn->u.pJmp);
							else					fprintf(hOut, "%s", CipGsrTknStr(pTkn));
							switch (pTkn->flgCtrl) {
								case AJCIP_WHILE_END:	fprintf(hOut, " (前置き 'while' ブロックの終端)"	);	break;
								case AJCIP_FOR_COND:	fprintf(hOut, " ('for' 「条件式」の末尾( ';' )"		);	break;
								case AJCIP_FOR_UPD:		fprintf(hOut, " ('for' 「更新文」の末尾( ')' )"		);	break;
								case AJCIP_FOR_END:		fprintf(hOut, " ('for' ブロックの終端( ';' or '}' )");	break;
								case AJCIP_FUNC_END:	fprintf(hOut, " (関数の末尾)"						);	break;
							}
							break;
						default:					//	・その他
							fprintf(hOut, "%s", CipGsrTknStr(pTkn));
							break;
					}
					fprintf(hOut, "' tkn = %4d\n", pTkn->tkn);
					if		(pTkn->tkn == EAJCTK_DLM_LMPART) {	//	'{'
						nest++;
					}
					pTkn++;
				}
			}
			//	全変数値表示
			for (i = 0; i < AJCCIPMAX_IXVMG; i++) {
				if (pW->sFni[i].hVmg != NULL) {
					fprintf(hOut, "----- Level %d Variables --------------\n", i);
					AjcVmgEnumVarA(pW->sFni[i].hVmg, (UX)hOut, cbNtcVar);
				}
			}
			//	ファイルクローズ
			fclose(hOut);
			//	結果表示
			ShellExecuteA(NULL, "open", path, NULL, NULL, SW_SHOWNORMAL);
			//	戻り値＝正常
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//																												//
//	通知情報取得																								//
//																												//
//==============================================================================================================//
//	ドロップディレクトリ数
AJCEXPORT UI			WINAPI AjcCipGetNtcDirs			(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	UI			rc = 0;

	if (pW != NULL) {
		rc = pW->NtcDirs;
	}
	return rc;
}
//	ドロップファイル数
AJCEXPORT UI			WINAPI AjcCipGetNtcFiles		(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	UI			rc = 0;

	if (pW != NULL) {
		rc = pW->NtcFiles;
	}
	return rc;
}
//	外部関数呼び出し情報
AJCEXPORT PAJCCIPEVPFUN	WINAPI AjcCipGetNtcEvpFun		(HWND hwnd)
{
	HAJCCIP			pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	PAJCCIPEVPFUN	rc = NULL;

	if (pW != NULL) {
		rc = &pW->NtcEvpFun;
	}
	return rc;
}
//	通知テキスト（エラーメッセージ，ダブルクリック行）
AJCEXPORT BCP			WINAPI AjcCipGetNtcText			(HWND hwnd)
{
	HAJCCIP		pW = (HAJCCIP)MAjcGetWindowLong(hwnd, 0);
	BCP			rc = NULL;

	if (pW != NULL) {
		rc = pW->NtcTxt;
	}
	return rc;
}

