﻿#include	"AjcInternal.h"
#include	"AjcCipCommon.h"

//**************************************************************************************************************//
//																												//
//	Ｃ言語風インタプリタ（整数式の評価）																		//
//																												//
//**************************************************************************************************************//

#define	MAX_NEST	32				//	最大ネスト数

#define	ISLV01(TKN)	((TKN) == EAJCTK_DLM_MULT || (TKN) == EAJCTK_DLM_DIV || (TKN) == EAJCTK_DLM_MOD)								/*	*	/	%		*/
#define	ISLV02(TKN)	((TKN) == EAJCTK_DLM_PLUS || (TKN) == EAJCTK_DLM_MINUS)															/*	+	-			*/
#define	ISLV03(TKN)	((TKN) == EAJCTK_DLM_SHL  || (TKN) == EAJCTK_DLM_SHR  )															/*	<<	>>			*/
#define	ISLV04(TKN)	((TKN) == EAJCTK_DLM_LO   || (TKN) == EAJCTK_DLM_HI || (TKN) == EAJCTK_DLM_LOEQ || (TKN) == EAJCTK_DLM_HIEQ)	/*	<	>	<=	>=	*/
#define	ISLV05(TKN)	((TKN) == EAJCTK_DLM_EQEQ || (TKN) == EAJCTK_DLM_NOTEQ)															/*	==	!=			*/
#define	ISLV06(TKN)	((TKN) == EAJCTK_DLM_AND )																						/*	&				*/
#define	ISLV07(TKN)	((TKN) == EAJCTK_DLM_XOR )																						/*	^				*/
#define	ISLV08(TKN)	((TKN) == EAJCTK_DLM_OR  )																						/*	|				*/
#define	ISLV09(TKN)	((TKN) == EAJCTK_DLM_LAND)																						/*	&&				*/
#define	ISLV10(TKN)	((TKN) == EAJCTK_DLM_LOR )																						/*	||				*/
#define	ISLV11(TKN)	((TKN) == EAJCTK_DLM_QUEST)																						/*	? (? exp : exp)	*/
#define	ISLV12(TKN) ((TKN) == EAJCTK_DLM_EQ 	|| (TKN) == EAJCTK_DLM_PLUSEQ || (TKN) == EAJCTK_DLM_MINUSEQ || \
					 (TKN) == EAJCTK_DLM_MULTEQ || (TKN) == EAJCTK_DLM_DIVEQ  || (TKN) == EAJCTK_DLM_MODEQ	 || \
					 (TKN) == EAJCTK_DLM_ANDEQ	|| (TKN) == EAJCTK_DLM_OREQ   || (TKN) == EAJCTK_DLM_XOREQ	 || \
					 (TKN) == EAJCTK_DLM_SHREQ	|| (TKN) == EAJCTK_DLM_SHLEQ)

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	演算タイプ																									//
//																												//
//--------------------------------------------------------------------------------------------------------------//
typedef		SLL		TCALC;

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	式の評価用ローカルワーク																					//
//																												//
//--------------------------------------------------------------------------------------------------------------//
typedef struct {
	UI			FmlNestCnt;					//	ネストカウンタ
	BOOL		fFmlCalc;					//	計算フラグ（CALCULATION(=TRUE ) : 計算する，
											//			   (SYNTAX_ONLY(=FALSE) : 式のシンタックスチェックのみ）
	HAJCFQUE	hFmlLsPart;					//	直前のPART情報（AJCCIPLSPARTのキュー）
} LWORK, *PLWORK;

//--------------------------------------------------------------------------------------------------------------//
//																												//
//	内部サブ関数																								//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC		express		(HAJCCIP pW, PLWORK pL);
static	TCALC		expresses	(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl12		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl12s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl11		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl11s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl10		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl10s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl09		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl09s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl08		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl08s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl07		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl07s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl06		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl06s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl05		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl05s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl04		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl04s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl03		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl03s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl02		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl02s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		Lvl01		(HAJCCIP pW, PLWORK pL);
static	TCALC		Lvl01s		(HAJCCIP pW, PLWORK pL, TCALC n);
static	TCALC		part		(HAJCCIP pW, PLWORK pL);

static	BOOL	SetVarVal(HAJCCIP pW, PLWORK pL, HAJCVMGA hVmg, BCP pName, UI ix, TCALC n);

//==============================================================================================================//
//	式の評価（整数）																							//
//																												//
//	引　数	：	pW			 - インスタンスハンドル																//
//				pVal		 - 式の評価値を格納するバッファのアドレス											//
//				fCalculation - 計算フラグ - CALCULATION(=TRUE)	: 計算する										//
//											SYNTAX_ONLY(=FALSE) : 式のシンタックスチェックのみ					//
//																												//
//	戻り値	：	TRUE ：成功																						//
//				FALSE：失敗																						//
//																												//
//	備　考	：	pW->pCur - 入力時：式の先頭トークン																//
//						   終了時：式の直後のトークン（エラーの場合は、式の途中のトークン）						//
//==============================================================================================================//
BOOL	CipFmlINT(HAJCCIP pW, TCALC *pVal, BOOL fCalculation)
{
	TCALC		n;
	LWORK		lw;

	//	直前のPART情報キュー生成
	if (lw.hFmlLsPart = AjcFQueCreate(sizeof(AJCCIPLSPART), 0, NULL)) {

		//----- 変数初期化 ---------------------------------//
		lw.FmlNestCnt	= 0;					//	ネストカウンタ
		lw.fFmlCalc		= fCalculation;			//	計算フラグ
		//----- 式の最後の語句 -----------------------------//
		pW->pFml		= pW->pCur;

		//----- 演算実行 -----------------------------------//
		n = express(pW, &lw);

		//----- 演算結果格納 -------------------------------//
		if (pVal != NULL) *pVal = n;

		//----- キュー解放 ---------------------------------//
		AjcFQueDelete(lw.hFmlLsPart);
	}
	else {
		pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
							LNGSEL("演算式評価用のワークを確保できません。",
								   "Work of formula allocation failure."));
	}
	return (pW->ErrCod == AJCIP_ERR_OK);
}
//--------------------------------------------------------------------------------------------------------------//
//	E X P R E S S																								//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	express(HAJCCIP pW, PLWORK pL)
{
	TCALC		n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		if (pL->FmlNestCnt < MAX_NEST) {
			pL->FmlNestCnt++;
			n = expresses(pW, pL, Lvl12(pW, pL));
			pL->FmlNestCnt--;
		}
		else {
			pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
								LNGSEL("演算式のネストが深すぎます。",
									   "A nest of an arithmetic expression is too deep."));
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	E X P R E S S E S																							//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	expresses(HAJCCIP pW, PLWORK pL, TCALC n)
{
	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV12(pW->pCur->tkn)) {
		n = express(pW, pL);
	}
	return (pL->fFmlCalc ? n : 1);
}

//--------------------------------------------------------------------------------------------------------------//
//	Lvl12	(	=  +=  -=  *=  /=  %=  &=  |=  ^=  >>=	<<=	)													//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl12(HAJCCIP pW, PLWORK pL)
{
	TCALC		n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl12s(pW, pL, Lvl11(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl12s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl12s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;
	TCALC		m;
	AJCCIPLSPART LsPart;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV12(s = pW->pCur->tkn)) {
		if (AjcFQueDeque(pL->hFmlLsPart, (VOP)&LsPart) && LsPart.fLsVar) {	//	直前は変数
			//	次の語句ポインタ設定＆ＥＯＦチェック
			if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
			//	転送演算
			m = Lvl11(pW, pL);
			if (!((s == EAJCTK_DLM_DIVEQ || s == EAJCTK_DLM_MODEQ) && m == 0)) {
				if (pW->fSigned) {
					switch (s) {
						case EAJCTK_DLM_EQ:		 n	 = m; break;		//	「	 = 」
						case EAJCTK_DLM_PLUSEQ:	 n	+= m; break;		//	「	+= 」
						case EAJCTK_DLM_MINUSEQ: n	-= m; break;		//	「	-= 」
						case EAJCTK_DLM_MULTEQ:	 n	*= m; break;		//	「	*= 」
						case EAJCTK_DLM_DIVEQ:	 n	/= m; break;		//	「	/= 」
						case EAJCTK_DLM_MODEQ:	 n	%= m; break;		//	「	%= 」
						case EAJCTK_DLM_ANDEQ:	 n	&= m; break;		//	「	&= 」
						case EAJCTK_DLM_OREQ:	 n	|= m; break;		//	「	|= 」
						case EAJCTK_DLM_XOREQ:	 n	^= m; break;		//	「	^= 」
						case EAJCTK_DLM_SHREQ:	 n >>= m; break;		//	「 >>= 」
						case EAJCTK_DLM_SHLEQ:	 n <<= m; break;		//	「 <<= 」
					}
				}
				else {
					ULL		un = (ULL)n;
					ULL		um = (ULL)m;
					switch (s) {
						case EAJCTK_DLM_EQ:		 un   = um; break;		//	「	 = 」
						case EAJCTK_DLM_PLUSEQ:	 un  += um; break;		//	「	+= 」
						case EAJCTK_DLM_MINUSEQ: un  -= um; break;		//	「	-= 」
						case EAJCTK_DLM_MULTEQ:	 un  *= um; break;		//	「	*= 」
						case EAJCTK_DLM_DIVEQ:	 un  /= um; break;		//	「	/= 」
						case EAJCTK_DLM_MODEQ:	 un  %= um; break;		//	「	%= 」
						case EAJCTK_DLM_ANDEQ:	 un  &= um; break;		//	「	&= 」
						case EAJCTK_DLM_OREQ:	 un  |= um; break;		//	「	|= 」
						case EAJCTK_DLM_XOREQ:	 un  ^= um; break;		//	「	^= 」
						case EAJCTK_DLM_SHREQ:	 un >>= um; break;		//	「 >>= 」
						case EAJCTK_DLM_SHLEQ:	 un <<= um; break;		//	「 <<= 」
					}
					n = (SLL)un;
				}
				//	変数値設定
				SetVarVal(pW, pL, LsPart.vLs.hVmg, LsPart.vLs.pSym, LsPart.vLs.ix, n);	//	CALCULATION時のみ変数設定
			}
			else {
				pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
									LNGSEL("ゼロ除算が発生しました。",
										   "Division by zero."));
			}
		}
		else {							//	直前は変数以外？
			pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
								LNGSEL("左辺(%s)が変数以外です。",
									   "A left side(%s) is not variable."), CipGsrTknStr(CipGsrGetBeforeToken(pW, pW->pCur)));
		}
	}
	return (pL->fFmlCalc ? n : 1);
}

//--------------------------------------------------------------------------------------------------------------//
//	Lvl11	(	?  :	)																						//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl11(HAJCCIP pW, PLWORK pL)
{
	TCALC		n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl11s(pW, pL, Lvl10(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl11s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl11s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	while (pW->ErrCod == AJCIP_ERR_OK  &&  pW->pCur->tkn == EAJCTK_DLM_QUEST) {		// 「 ? 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	条件＝真
		if (n) {
			//	「真」条件時の上位レベル演算
			n = express(pW, pL);
			if (pW->ErrCod) break;
			//	「:」ならば上位レベル演算
			if (pW->pCur->tkn == EAJCTK_DLM_COLON) {
				//	次の語句ポインタ設定＆ＥＯＦチェック
				if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
				//	「偽」条件時のダミー演算
				express(pW, pL);
			}
			//	「:」以外ならばエラー
			else {
				pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
									LNGSEL("<%s> の前にコロン(:)が必要です。",
										   "Need ':' before <%s>"), CipGsrTknStr(pW->pCur));
			}
		}
		//	条件＝偽
		else {
			//	「真」条件時のダミー演算
			express(pW, pL);
			if (pW->ErrCod) break;
			//	「:」ならば上位レベル演算
			if (pW->pCur->tkn == EAJCTK_DLM_COLON) {
				//	次の語句ポインタ設定＆ＥＯＦチェック
				if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
				//	「偽」条件時の演算
				n = express(pW, pL);
			}
			//	「:」以外ならばエラー
			else {
				pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
									LNGSEL("<%s> の前にコロン(:)が必要です。",
										   "Need ':' before <%s>"), CipGsrTknStr(pW->pCur));
			}
		}
	}

	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl10	(	||	)																							//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl10(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl10s(pW, pL, Lvl09(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl10s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl10s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;
	TCALC		x;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV10(s = pW->pCur->tkn)) {		//	「 || 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	ロジカル論理和「||」
		if (s == EAJCTK_DLM_LOR) {
			x = Lvl09(pW, pL);
			n = (n || x);
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl09	(	&&	)																							//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl09(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl09s(pW, pL, Lvl08(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl09s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl09s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;
	TCALC		x;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV09(s = pW->pCur->tkn)) {		//	「 && 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	ロジカル論理積「&&」
		if (s == EAJCTK_DLM_LAND) {
			x = Lvl08(pW, pL);
			n = (n && x);
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl08	(	|	)																							//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl08(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl08s(pW, pL, Lvl07(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl08s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl08s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV08(s = pW->pCur->tkn)) {		//	「 | 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	論理和「 | 」
		if (s == EAJCTK_DLM_OR) {
			n = (n | Lvl07(pW, pL));
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl07	(	^	)																							//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl07(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl07s(pW, pL, Lvl06(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl07s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl07s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV07(s = pW->pCur->tkn)) {		//	「 ^ 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	排他論理和「 ^ 」
		if (s == EAJCTK_DLM_XOR) {
			n = (n ^ Lvl06(pW, pL));
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl06	(	&	)																							//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl06(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl06s(pW, pL, Lvl05(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl06s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl06s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV06(s = pW->pCur->tkn)) {		//	「 & 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	論理積「&」
		if (s == EAJCTK_DLM_AND) {
			n = (n & Lvl05(pW, pL));
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl05	(	==	!=	)																						//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl05(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl05s(pW, pL, Lvl04(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl05s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl05s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV05(s = pW->pCur->tkn)) {		//	「 ==  != 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	比較演算「==, !=」
		switch (s) {
			case EAJCTK_DLM_EQEQ:	n = (n == Lvl04(pW, pL)); break;		//	「 == 」
			case EAJCTK_DLM_NOTEQ:	n = (n != Lvl04(pW, pL)); break;		//	「 != 」
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl04	(	<	>	<=	>=	)																				//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl04(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl04s(pW, pL, Lvl03(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl04s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl04s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV04(s = pW->pCur->tkn)) {		//	「 <   >   <=  >= 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	大小比較演算「<   >   <=  >=」
		if (pW->fSigned) {
			switch (s) {
				case EAJCTK_DLM_LO:		n = (TCALC)((SLL)n <  (SLL)Lvl03(pW, pL)); break;	//	「 <  」
				case EAJCTK_DLM_HI:		n = (TCALC)((SLL)n >  (SLL)Lvl03(pW, pL)); break;	//	「 >  」
				case EAJCTK_DLM_LOEQ:	n = (TCALC)((SLL)n <= (SLL)Lvl03(pW, pL)); break;	//	「 <= 」
				case EAJCTK_DLM_HIEQ:	n = (TCALC)((SLL)n >= (SLL)Lvl03(pW, pL)); break;	//	「 >= 」
			}
		}
		else {
			switch (s) {
				case EAJCTK_DLM_LO:		n = (TCALC)((ULL)n <  (ULL)Lvl03(pW, pL)); break;	//	「 <  」
				case EAJCTK_DLM_HI:		n = (TCALC)((ULL)n >  (ULL)Lvl03(pW, pL)); break;	//	「 >  」
				case EAJCTK_DLM_LOEQ:	n = (TCALC)((ULL)n <= (ULL)Lvl03(pW, pL)); break;	//	「 <= 」
				case EAJCTK_DLM_HIEQ:	n = (TCALC)((ULL)n >= (ULL)Lvl03(pW, pL)); break;	//	「 >= 」
			}
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl03	(	<<	>>	)																						//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl03(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl03s(pW, pL, Lvl02(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl03s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl03s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV03(s = pW->pCur->tkn)) {		//	「 <<  >> 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	シフト演算「<<	>>」
		if (pW->fSigned) {
			switch (s) {
				case EAJCTK_DLM_SHL:	n = (TCALC)((SLL)n << (SLL)Lvl02(pW, pL)); break;	//	「 << 」
				case EAJCTK_DLM_SHR:	n = (TCALC)((SLL)n >> (SLL)Lvl02(pW, pL)); break;	//	「 >> 」
			}
		}
		else {
			switch (s) {
				case EAJCTK_DLM_SHL:	n = (TCALC)((ULL)n << (ULL)Lvl02(pW, pL)); break;	//	「 << 」
				case EAJCTK_DLM_SHR:	n = (TCALC)((ULL)n >> (ULL)Lvl02(pW, pL)); break;	//	「 >> 」
			}
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl02	(	+	-	)																						//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl02(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl02s(pW, pL, Lvl01(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl02s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl02s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV02(s = pW->pCur->tkn)) {		//	「 +   - 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	加減算「+  -」
		switch (s) {
			case EAJCTK_DLM_PLUS:	n = (n + Lvl01(pW, pL)); break;	//	「 + 」
			case EAJCTK_DLM_MINUS:	n = (n - Lvl01(pW, pL)); break;	//	「 - 」
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl01	(	*	/	% 	)																					//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl01(HAJCCIP pW, PLWORK pL)
{
	TCALC	n = 0;

	if (pW->ErrCod == AJCIP_ERR_OK) {
		n = Lvl01s(pW, pL, part(pW, pL));
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	Lvl01s																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	Lvl01s(HAJCCIP pW, PLWORK pL, TCALC n)
{
	EAJCTKCODE	s;
	TCALC		p;

	while (pW->ErrCod == AJCIP_ERR_OK  &&  ISLV01(s = pW->pCur->tkn)) {		//	「 *   /   % 」
		//	次の語句ポインタ設定＆ＥＯＦチェック
		if (CipFmlSkipToken(pW) == EAJCTK_EOF) break;
		//	演算パーツ（定数や変数値）取得
		p = part(pW, pL);
		//	乗算レベル演算「 * / % 」
		switch (s) {
			case EAJCTK_DLM_MULT:	if (pW->fSigned) n = (TCALC)((SLL)n * (SLL)p);
									else			 n = (TCALC)((ULL)n * (ULL)p);
									break;

			case EAJCTK_DLM_DIV:	if (p) {
										if (pW->fSigned) n = (TCALC)((SLL)n / (SLL)p);
										else			 n = (TCALC)((ULL)n / (ULL)p);
									}
									else {
										pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
															LNGSEL("ゼロ除算が発生しました。",
																   "Division by zero."));
									}
									break;

			case EAJCTK_DLM_MOD:	if (p) {
										if (pW->fSigned) n = (TCALC)((SLL)n % (SLL)p);
										else			 n = (TCALC)((ULL)n % (ULL)p);
									}
									else {
										pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno, 
															LNGSEL("ゼロ除算が発生しました。",
																   "Division by zero."));
									}
									break;
		}
	}
	return (pL->fFmlCalc ? n : 1);
}
//--------------------------------------------------------------------------------------------------------------//
//	P A R T																										//
//--------------------------------------------------------------------------------------------------------------//
static	TCALC	part(HAJCCIP pW, PLWORK pL)
{
	TCALC		n	 = 0;				//	戻り値（数値）
	PAJCCIPTKN	pNxt = pW->pCur + 1;	//	１つ先のトークン
	AJCCIPLSPART LsPart;				//	最終PART情報

	//	直前の語句群が変数であることを示すフラグ クリアー
	memset(&LsPart, 0, sizeof LsPart);

	if (pW->ErrCod == AJCIP_ERR_OK) {	//	エラー未発生？

		//	●左カッコ
		if		(pW->pCur->tkn == EAJCTK_DLM_LSPART) {
			//	次の語句ポインタ設定＆ＥＯＦチェック
			if (CipFmlSkipToken(pW) != EAJCTK_EOF) {			// EOF以外？
				//	演算実行
				n = express(pW, pL);
				if (pW->ErrCod == AJCIP_ERR_OK) {				// 数式エラーなし？
					//	演算後の語句が右括弧ならばスキップ
					if (pW->pCur->tkn == EAJCTK_DLM_RSPART) {
						//	次の語句ポインタ設定（ ')' をスキップ）
						pW->pFml = pW->pCur++;
					}
					//	演算後の語句が右括弧以外ならばエラー
					else {
						pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno,
											LNGSEL("式の表現に誤りがあります。<%s>",
												   "Invalid formula expression <%s>."), CipGsrTknStr(pW->pCur));
					}
				}
			}
		}

		//	●単項演算子（+-~!）
		else if (pW->pCur->tkn == EAJCTK_DLM_PLUS	||		//	+
				 pW->pCur->tkn == EAJCTK_DLM_MINUS	||		//	-
				 pW->pCur->tkn == EAJCTK_DLM_NOT	||		//	~
				 pW->pCur->tkn == EAJCTK_DLM_LNOT) {		//	!
			//	トークン退避
			EAJCTKCODE	s = pW->pCur->tkn;
			//	次の語句ポインタ設定＆ＥＯＦチェック
			if (CipFmlSkipToken(pW) != EAJCTK_EOF) {		// EOF以外？
				//	演算パーツ（定数や変数値）取得
				n = part(pW, pL);
				if (pW->ErrCod == AJCIP_ERR_OK) {			// 「part」エラーなし？
					//	単項演算「-~!」
					switch (s) {
						case EAJCTK_DLM_MINUS:	n = ~n + 1;	break;
						case EAJCTK_DLM_NOT:	n = ~n;		break;
						case EAJCTK_DLM_LNOT:	n = !n;		break;
					}
				}
			}
		}

		//	●定数（整数）
		else if (pW->pCur->tkn == EAJCTK_CIP_INTEGER) {
			//	定数値設定
			n = (TCALC)pW->pCur->u.val;
			//	次の語句ポインタ設定（定数をスキップ）
			pW->pFml = pW->pCur++;
		}

		//	●定数（実数）
		else if (pW->pCur->tkn == EAJCTK_CIP_REAL) {
			//	定数値設定
			n = (TCALC)pW->pCur->u.dbl;
			//	次の語句ポインタ設定（定数をスキップ）
			pW->pFml = pW->pCur++;
		}

		//	●＋＋ユーザ定義名（プリ・インクリメント）
		else if (pW->pCur->tkn == EAJCTK_DLM_PLUS2	&& IS_NUM_SYM(pNxt)) {
			pW->pFml = pW->pCur++;										//	'++' スキップ
			n = CipFmlGetVarInfoInt64(pW, &LsPart.vLs, pL->fFmlCalc);	//	変数値取得
			//	変数値インクリメント（戻り値もインクリメントした値）
			if (pW->ErrCod == AJCIP_ERR_OK) {
				n++;
				SetVarVal(pW, pL, LsPart.vLs.hVmg, LsPart.vLs.pSym, LsPart.vLs.ix, n);	//	CALCULATION時のみ変数設定
			}
		}

		//	●－－ユーザ定義名（プリ・デクリメント）
		else if (pW->pCur->tkn == EAJCTK_DLM_MINUS2 && IS_NUM_SYM(pNxt)) {
			pW->pFml = pW->pCur++;										//	'--' スキップ
			n = CipFmlGetVarInfoInt64(pW, &LsPart.vLs, pL->fFmlCalc);	//	変数値取得
			//	変数値デクリメント（戻り値もデクリメントした値）
			if (pW->ErrCod == AJCIP_ERR_OK) {
				n--;
				SetVarVal(pW, pL, LsPart.vLs.hVmg, LsPart.vLs.pSym, LsPart.vLs.ix, n);	//	CALCULATION時のみ変数設定
			}
		}

		//	●ユーザ定義名（・・）--- 関数
		else if (IS_NUM_SYM(pW->pCur) && pNxt->tkn == EAJCTK_DLM_LSPART) { // 'Sym(' ？
			AJCCIPVARINFO	vInf;
			vInf.hVmg = pW->sFni[0].hVmg;
			vInf.pSym = "@rc";
			vInf.ix   = 0;
			AjcVmgSetIntegerA(vInf.hVmg, "@rc", 0, 0);
			if (CipFmlCallFunc(pW, &vInf, pL->fFmlCalc)) {
				AjcVmgGetInt64A(vInf.hVmg, "@rc", 0, &n);
			}
		}
		//	●ユーザ定義名（数値変数）
		else if (IS_NUM_SYM(pW->pCur)) {
			//	変数値取得
			n = CipFmlGetVarInfoInt64(pW, &LsPart.vLs, pL->fFmlCalc);
			if (pW->ErrCod == AJCIP_ERR_OK) {
				//	○ '変数＋＋'（ポスト・インクリメント）
				if		(pW->pCur->tkn == EAJCTK_DLM_PLUS2) {
					pW->pFml = pW->pCur++;		//	'++' スキップ
					//	変数値インクリメント（戻り値はインクリメントする前の値）
					SetVarVal(pW, pL, LsPart.vLs.hVmg, LsPart.vLs.pSym, LsPart.vLs.ix, n + 1);	//	CALCULATION時のみ変数設定
				}
				//	○ '変数－－'（ポスト・デクリメント）
				else if (pW->pCur->tkn == EAJCTK_DLM_MINUS2) {
					pW->pFml = pW->pCur++;		//	'--' スキップ
					//	変数値デクリメント（戻り値はデクリメントする前の値）
					SetVarVal(pW, pL, LsPart.vLs.hVmg, LsPart.vLs.pSym, LsPart.vLs.ix, n - 1);	//	CALCULATION時のみ変数設定
				}
				//	○ 変数のみ
				else {
					//	直前の語句群が変数であることを示すフラグ設定
					LsPart.fLsVar = TRUE;
				}
			}
		}
		//	●文字変数／文字列
		else if (IS_STR_SYM(pW->pCur) || pW->pCur->tkn == EAJCTK_CIP_STRING) {
			SLL	cond;
			CipFmlStr(pW, &cond, pL->fFmlCalc);
			n = (TCALC)cond;
		}
		//	●その他（エラー）
		else {
			pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno,
								LNGSEL("式の表現に誤りがあります。<%s>",
									   "Invalid formula expression <%s>."), CipGsrTknStr(pW->pCur));
		}
	}

	//	最終PART情報設定（最終PARTが変数化否かの情報）
	if (!(AjcFQueEnqTop(pL->hFmlLsPart, (C_VOP)&LsPart))) {
		pW->ErrCod = CipGsrNtcErr(pW, AJCIP_ERR_CALC, pW->pCur->lno,
							LNGSEL("式の評価用のメモリが不足しています。",
								   "Insufficient formula expression work."));
	}

	return (pL->fFmlCalc ? n : 1);
}

//--------------------------------------------------------------------------------------------------------------//
//	変数値設定																									//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SetVarVal(HAJCCIP pW, PLWORK pL, HAJCVMGA hVmg, BCP pName, UI ix, TCALC n)
{
	BOOL	rc = TRUE;

	if (pL->fFmlCalc) {
		rc = AjcVmgSetIntegerA(hVmg, pName, ix, n);
	}
	return rc;
}
