﻿#include	"AjcInternal.h"
#include	"AjcCtrl3dGraphicDef.h"
//**************************************************************************************************************//
//																												//
//	カスタムコントロール（３Ｄグラフィック）	ＡＰＩ															//
//																												//
//**************************************************************************************************************//

#define	IS_MY_INST(P)	(P != NULL && P->InstID == G3D_INST_ID)

//--------------------------------------------------------------------------------------------------------------//
//	共通変数																									//
//--------------------------------------------------------------------------------------------------------------//
ATOM	G3dClassMain	= 0;
ATOM	G3dClassFilter	= 0;
ATOM	G3dClassForMSIL	= 0;
HFONT	G3dhMyFont 		= NULL;

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	CbAg3sMetaBall(PWRK3DGRAPH pW, PGI_ELPS pElps, UX id);

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcCtrl3dGraphInit(VO)
{
	BOOL		rc = FALSE;
	WNDCLASS	wndclass;
	LOGFONT		lf;

	do {
		//----- フォント --------------------------------------//
		lf.lfHeight			= -9;	lf.lfStrikeOut		= 0;
		lf.lfWidth			= 0;	lf.lfCharSet		= DEFAULT_CHARSET;
		lf.lfEscapement		= 0;	lf.lfOutPrecision	= 0;
		lf.lfOrientation	= 0;	lf.lfClipPrecision	= 0;
		lf.lfWeight			= 0;	lf.lfQuality		= 0;
		lf.lfItalic			= 0;	lf.lfPitchAndFamily	= VARIABLE_PITCH;
		lf.lfUnderline		= 0;	wcscpy(lf.lfFaceName, L"Arial");
		if ((G3dhMyFont = CreateFontIndirect(&lf)) == NULL) break;

		//----- メインウインドクラス --------------------------//
		wndclass.style			= CS_DBLCLKS | CS_GLOBALCLASS;
		wndclass.lpfnWndProc	= G3dWndProcMain();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(PWRK3DGRAPH);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"AjcCtrl3dGraph";
		if ((G3dClassMain = RegisterClass(&wndclass)) == 0) break;

		//----- フィルタ・ウインドクラス ----------------------//
		wndclass.style			= 0;
		wndclass.lpfnWndProc	= G3dWndProcFilt();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(PWRK3DGRAPH);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(LTGRAY_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"Ajc3dGraphFilterClass";
		if ((G3dClassFilter = RegisterClass(&wndclass)) == 0) break;

		//----- ＭＳＩＬコードサポート用 ---------------------//
		wndclass.style			= 0;
		wndclass.lpfnWndProc	= G3dWndProcMSIL();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(VOP);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"AjcG3dForMSIL";
		if ((G3dClassForMSIL = RegisterClass(&wndclass)) == 0) break;

		rc = TRUE;
	} while(0);

	if (rc == FALSE) {
		AjcCtrl3dGraphEnd();
	}

	return rc;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
VO		AjcCtrl3dGraphEnd (VO)
{
	HINSTANCE	hModInst = GetModuleHandle(NULL);

	if (G3dClassMain	!= 0	) {UnregisterClass((WCP)G3dClassMain	, hDllInst); G3dClassMain	 = 0;	}
	if (G3dClassFilter	!= 0	) {UnregisterClass((WCP)G3dClassFilter	, hDllInst); G3dClassFilter  = 0;	}
	if (G3dClassForMSIL != 0 	) {UnregisterClass((WCP)G3dClassForMSIL	, hDllInst); G3dClassForMSIL = 0;	}
	if (G3dhMyFont 		!= NULL ) {DeleteObject   (G3dhMyFont					  ); G3dhMyFont		 = NULL;}

}
//**************************************************************************************************************//
//																												//
//		３Ｄファンクション																						//
//																												//
//**************************************************************************************************************//
//==============================================================================================================//
//	３Ｄ : 初期化																								//
//																												//
//	引　数	：	x1, x2		- Ｘ軸のレンジ																		//
//				y1, y2		- Ｙ軸のレンジ																		//
//				z1, z2		- Ｚ軸のレンジ																		//
//				style		- コントロールのスタイル															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dInit		   (HWND hwnd, double x1, double y1, double z1,
															   double x2, double y2, double z2, UI style)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			double		xmin = __min(x1, x2);
			double		xmax = __max(x1, x2);
			double		ymin = __min(y1, y2);
			double		ymax = __max(y1, y2);
			double		zmin = __min(z1, z2);
			double		zmax = __max(z1, z2);
			AJC3DGPROP	prop;
	
			//	プロパティ設定
			memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
			prop.Rot.x	= 60.0;
			prop.Rot.y	= 10.0;
			prop.Rot.z	= 45.0;
			prop.Cent.x = (xmin + xmax) / 2.0;
			prop.Cent.y = (ymin + ymax) / 2.0;
			prop.Cent.z = (zmin + zmax) / 2.0;
			prop.xr		= (xmax - xmin) / 2.0;
			prop.yr		= (ymax - ymin) / 2.0;
			prop.zr		= (zmax - zmin) / 2.0;
		//	prop.ratio	= 0.7;
			if (prop.xr > 0.0  &&	prop.yr > 0.0  &&	prop.zr > 0.0) {
				G3dSetCtrlProp(pW, &prop);
			}
			//	３Ｄモード設定
			G3dSet3dMode(pW);
			//	スタイル設定
			if (style != -1) {
				UI	sty;
				sty = (UI)MAjcGetWindowLong(hwnd, GWL_STYLE) & 0xFFFF0000;
				sty |= (style & 0x0000FFFF);
				MAjcSetWindowLong(hwnd, GWL_STYLE, sty);
			}
		}
	}
	return	rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dInitV		   (HWND hwnd, PCAJC3DVEC pLo, PAJC3DVEC pHi, PAJC3DVEC pRot, UI style)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			double		xmin;
			double		xmax;
			double		ymin;
			double		ymax;
			double		zmin;
			double		zmax;
			AJC3DGPROP	prop;
	
			//	プロパティ設定
			if ((pLo != NULL && pHi != NULL) || pRot != NULL) {
				memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
				if (pRot != NULL) {
					prop.Rot.x	= pRot->x;
					prop.Rot.y	= pRot->y;
					prop.Rot.z	= pRot->z;
				}
				if (pLo != NULL && pHi != NULL) {
					xmin = __min(pLo->x, pHi->x);
					xmax = __max(pLo->x, pHi->x);
					ymin = __min(pLo->y, pHi->y);
					ymax = __max(pLo->y, pHi->y);
					zmin = __min(pLo->z, pHi->z);
					zmax = __max(pLo->z, pHi->z);
					prop.Cent.x = (xmin + xmax) / 2.0;
					prop.Cent.y = (ymin + ymax) / 2.0;
					prop.Cent.z = (zmin + zmax) / 2.0;
					prop.xr		= (xmax - xmin) / 2.0;
					prop.yr		= (ymax - ymin) / 2.0;
					prop.zr		= (zmax - zmin) / 2.0;
				}
				//	prop.ratio	= 0.7;
				if (prop.xr > 0.0  &&	prop.yr > 0.0  &&	prop.zr > 0.0) {
					G3dSetCtrlProp(pW, &prop);
				}
			}
			//	３Ｄモード設定
			G3dSet3dMode(pW);
			//	スタイル設定
			if (style != -1) {
				UI	sty;
				sty = (UI)MAjcGetWindowLong(hwnd, GWL_STYLE) & 0xFFFF0000;
				sty |= (style & 0x0000FFFF);
				MAjcSetWindowLong(hwnd, GWL_STYLE, sty);
			}
			rc = TRUE;
		}
	}
	return	rc;
}

//==============================================================================================================//
//	３Ｄ : プロパティ設定																						//
//																												//
//	引　数	：	pProp		- 設定するプロパティデータのアドレス												//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetProp	   (HWND hwnd, PCAJC3DGPROP pProp)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProp != NULL) {
			rc = G3dSetCtrlProp(pW, pProp);
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : プロパティ取得																						//
//																												//
//	引　数	：	pBuf		- 取得したプロパティデータを格納するバッファのアドレス								//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dGetProp	   (HWND hwnd, PAJC3DGPROP	pBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pBuf != NULL) {
			memcpy(pBuf, &pW->prop, sizeof(AJC3DGPROP));
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	３Ｄ : レンジ自動調整																						//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
#define	AVOSET_MINMAX(XXX)	xmin = __min(xmin, XXX.x); xmax = __max(xmax, XXX.x);	\
							ymin = __min(ymin, XXX.y); ymax = __max(ymax, XXX.y);	\
							zmin = __min(zmin, XXX.z); zmax = __max(zmax, XXX.z);	\
							cnt++

AJCEXPORT	BOOL	WINAPI		AjcG3dAdjustRange  (HWND hwnd)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			UI			i, j, msk, cnt=0;
			double		xmin, ymin, zmin, xmax, ymax, zmax, r1, r2;
			AJC3DGPROP	prop;
	
			memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
			xmin = ymin = zmin =  DBL_MAX;
			xmax = ymax = zmax = -DBL_MAX;
	
			//----- データ最小値，最大値設定 -----------------------------------//
			for (i = 0, msk = 0x01; i < AJC3DG_MAXITEM; i++, msk <<= 1) {
				PGI_INFO	p;
				if ((pW->MskFilt & msk) && (p = AjcVQueTopNode(pW->DatItem[i].hQueDat, NULL))) {
					do {
						switch (p->h.gic) {
							case GIC_POINT:
								AVOSET_MINMAX(p->poi.p.v3);
								break;
	
							case GIC_LINE:
								AVOSET_MINMAX(p->line.p1.v3);
								AVOSET_MINMAX(p->line.p2.v3);
								break;
	
							case GIC_TRIAN:
								AVOSET_MINMAX(p->tri.p1.v3);
								AVOSET_MINMAX(p->tri.p2.v3);
								AVOSET_MINMAX(p->tri.p3.v3);
								break;
	
							case GIC_SQUA:
								AVOSET_MINMAX(p->squa.p1.v3);
								AVOSET_MINMAX(p->squa.p2.v3);
								AVOSET_MINMAX(p->squa.p3.v3);
								AVOSET_MINMAX(p->squa.p4.v3);
								break;
	
							case GIC_ELPS:
								for (j = 0; j < G3D_ELPSPOINTS; j++) {
									AVOSET_MINMAX(p->elps.p[j].v3);
								}
								break;
	
							case GIC_FILL:
								AVOSET_MINMAX(p->fill.p.v3);
								break;
	
						}
					} while (p = AjcVQueNextNode(pW->DatItem[i].hQueDat, p, NULL));
				}
			}
			//----- プロットデータ最小値，最大値設定 ---------------------------//
			for (i = 0, msk = 0x01; i < AJC3DG_MAXITEM; i++, msk <<= 1) {
				PGI_VEC	p = pW->DatItem[i].plot.pBuf;
				if (pW->MskFilt & msk) {
					for (j = 0; j < pW->DatItem[i].plot.cnt; j++) {
						AVOSET_MINMAX(p->v3);
					}
				}
			}
			//----- ボリュームサイズ設定 ---------------------------------------//
			if (cnt != 0) {
				if (xmin == DBL_MAX) xmin = xmax = 0.0;
				if (ymin == DBL_MAX) ymin = ymax = 0.0;
				if (zmin == DBL_MAX) zmin = zmax = 0.0;
				if (xmin == xmax) {xmin--; xmax++;}
				if (ymin == ymax) {ymin--; ymax++;}
				if (zmin == zmax) {zmin--; zmax++;}
				r1 = fabs(prop.Cent.x - xmin); r2 = fabs(prop.Cent.x - xmax); prop.xr = __max(r1, r2) * 1.1;
				r1 = fabs(prop.Cent.y - ymin); r2 = fabs(prop.Cent.y - ymax); prop.yr = __max(r1, r2) * 1.1;
				r1 = fabs(prop.Cent.z - zmin); r2 = fabs(prop.Cent.z - zmax); prop.zr = __max(r1, r2) * 1.1;
				rc = G3dSetCtrlProp(pW, &prop);
			}
			//----- スケール情報計算 -------------------------------//
			G3dSetScaleInfo(pW);
			//----- ポイント情報再計算 -----------------------------//
			G3dCnvAllPoints(pW);
			//----- アスペクト固定処理 -----------------------------//
			if (pW->prop.fAspect1) {
				AjcG3dSetSameRangeWidth(hwnd);
			}
			//----- 再描画 -----------------------------------------//
			InvalidateRect(hwnd, NULL, FALSE);
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : レンジ設定																							//
//																												//
//	引　数	：	x1, x2		- Ｘ軸のレンジ																		//
//				y1, y2		- Ｙ軸のレンジ																		//
//				z1, z2		- Ｚ軸のレンジ																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetRange	   (HWND hwnd, double x1, double y1, double z1,
															   double x2, double y2, double z2)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			double		xmin = __min(x1, x2);
			double		xmax = __max(x1, x2);
			double		ymin = __min(y1, y2);
			double		ymax = __max(y1, y2);
			double		zmin = __min(z1, z2);
			double		zmax = __max(z1, z2);
			AJC3DGPROP	prop;

			memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
			prop.Cent.x = (xmin + xmax) / 2.0;
			prop.Cent.y = (ymin + ymax) / 2.0;
			prop.Cent.z = (zmin + zmax) / 2.0;
			prop.xr		= (xmax - xmin) / 2.0;
			prop.yr		= (ymax - ymin) / 2.0;
			prop.zr		= (zmax - zmin) / 2.0;

			if (prop.xr > 0.0  &&	prop.yr > 0.0  &&	prop.zr > 0.0) {
				rc = G3dSetCtrlProp(pW, &prop);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 中心位置設定																							//
//																												//
//	引　数	：	xc, yc, zc	- 各軸の中心位置																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetCenter	   (HWND hwnd, double xc, double yc, double zc)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;

			memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
			prop.Cent.x = xc;
			prop.Cent.y = yc;
			prop.Cent.z = zc;
			rc = G3dSetCtrlProp(pW, &prop);
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 各軸の幅（半径）設定																					//
//																												//
//	引　数	：	xr, yr, zr	- 各軸の半径																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetWidth	   (HWND hwnd, double xr, double yr, double zr)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;

			if (xr != 0.0 && yr != 0.0 && zr != 0.0) {
				memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
				prop.xr = xr;
				prop.yr = yr;
				prop.zr = zr;
				rc = G3dSetCtrlProp(pW, &prop);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 各軸のレンジ幅を同一サイズに揃える																	//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetSameRangeWidth(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;
			double		rmax;

			memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
			if (pW->f2dMode) {
				rmax = __max(prop.xr, prop.yr);
				prop.xr = prop.yr = rmax;
			}
			else {
				rmax = __max(prop.xr, prop.yr);
				rmax = __max(prop.zr, rmax);
				prop.xr = prop.yr = prop.zr = rmax;
			}
			rc = G3dSetCtrlProp(pW, &prop);
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : レンジの表示域をアスペクト１にする（レンジの表示域を立方体にする）									//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetFixedAspect   (HWND hwnd, BOOL fFixed)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;
	
			memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
			prop.fAspect1 = fFixed;
			rc = G3dSetCtrlProp(pW, &prop);
		}
	}
	return rc;
}

//==============================================================================================================//
//	３Ｄ : 表示色設定																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				rgbP		- 中心より手前側の表示色															//
//				rgbN		- 中心より奥側の表示色																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetColor	   (HWND hwnd, UI id, COLORREF rgbP, COLORREF rgbN)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;

			if (id < AJC3DG_MAXITEM) {
				memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
				prop.Item[id].rgbP = rgbP;
				if (rgbN != -1) prop.Item[id].rgbN = rgbN;
				else			prop.Item[id].rgbN = rgbP;
				rc = G3dSetCtrlProp(pW, &prop);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : プロットデータ数設定																					//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				PlotNumber	- プロットデータ数																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetPlotNumber(HWND hwnd, UI id, UI PlotNumber)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;
	
			if (id < AJC3DG_MAXITEM) {
				memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
				prop.Item[id].MaxPlot = PlotNumber;
				rc = G3dSetCtrlProp(pW, &prop);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : プロット点のサイズ設定																				//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				PixelSize	- プロット点のサイズ																//
//				PixelSizeE	- 終点のプロット点サイズ															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetPlotSize  (HWND hwnd, UI id, UI PixelSize, UI PixelSizeE)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;
	
			if (id < AJC3DG_MAXITEM) {
				memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
				prop.PlotSize  = PixelSize;
				prop.PlotSizeE = PixelSizeE;
				rc = G3dSetCtrlProp(pW, &prop);
			}
		}
	}
	return	rc;
}

//==============================================================================================================//
//	３Ｄ : プロットデータ投与																					//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y, z		- プロットデータ																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dPutPlotData  (HWND hwnd, UI id, double x, double y, double z)
{
	AJC3DVEC	pt;

	pt.x = x; pt.y = y; pt.z = z;
	return AjcG3dPutPlotDataV(hwnd, id, &pt);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dPutPlotDataV (HWND hwnd, UI id, PCAJC3DVEC pPoint)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pPoint != NULL) {
			if (id < AJC3DG_MAXITEM	 &&	 pW->prop.Item[id].MaxPlot != 0) {
				memcpy(&pW->DatItem[id].plot.pBuf[pW->DatItem[id].plot.ix].v3, pPoint, sizeof(AJC3DVEC));
				G3dCnv3dTo2d(pW, &pW->DatItem[id].plot.pBuf[pW->DatItem[id].plot.ix].v3, &pW->DatItem[id].plot.pBuf[pW->DatItem[id].plot.ix].v2);
				pW->DatItem[id].plot.ix = ((pW->DatItem[id].plot.ix + 1) % pW->prop.Item[id].MaxPlot);
				if (pW->DatItem[id].plot.cnt < pW->prop.Item[id].MaxPlot) pW->DatItem[id].plot.cnt++;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
				rc = TRUE;
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : ピクセル描画																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y, z		- 描画位置																			//
//				PixelSize	- 描画ピクセルのサイズ																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dPixel		   (HWND hwnd, UI id, double x , double y , double z , UI PixelSize)
{
	AJC3DVEC	pt;

	pt.x = x; pt.y = y; pt.z = z;
	return AjcG3dPixelV(hwnd, id, &pt, PixelSize);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dPixelV	   (HWND hwnd, UI id, PCAJC3DVEC pPoint, UI PixelSize)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pPoint != NULL) {
			GI_POINT	poi;
			VOP			pNode;
	
			if (id < AJC3DG_MAXITEM) {
				poi.h.gic = GIC_POINT;
				poi.h.id  = id;
				poi.h.seq = pW->DatSeq++;
				memcpy(&poi.p.v3, pPoint, sizeof(AJC3DVEC));
				G3dCnv3dTo2d(pW, &poi.p.v3, &poi.p.v2);
				poi.siz = PixelSize;
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &poi, sizeof poi)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, poi.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : ライン描画																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x1, y1, z1	- 始点位置																			//
//				x2, y2, z2	- 終点位置																			//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
static	BOOL	SubG3dLineV(HWND hwnd, UI id, PCAJC3DVEC p1, PCAJC3DVEC p2, BOOL fArrow);
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dLine		   (HWND hwnd, UI id, double x1, double y1, double z1,
																	  double x2, double y2, double z2)
{
	AJC3DVEC	p1, p2;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	return SubG3dLineV(hwnd, id, &p1, &p2, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dLineV		   (HWND hwnd, UI id, PCAJC3DVEC p1, PCAJC3DVEC p2)
{
	return SubG3dLineV(hwnd, id,  p1,  p2, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dArrow		   (HWND hwnd, UI id, double x1, double y1, double z1,
																	  double x2, double y2, double z2)
{
	AJC3DVEC	p1, p2;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	return SubG3dLineV(hwnd, id, &p1, &p2, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dArrowV		   (HWND hwnd, UI id, PCAJC3DVEC p1, PCAJC3DVEC p2)
{
	return SubG3dLineV(hwnd, id,  p1,  p2, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubG3dLineV(HWND hwnd, UI id, PCAJC3DVEC p1, PCAJC3DVEC p2, BOOL fArrow)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && p1 != NULL && p2 != NULL) {
			GI_LINE		line;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				line.h.gic = GIC_LINE;
				line.h.id  = id;
				line.h.seq = pW->DatSeq++;
				memcpy(&line.p1.v3, p1, sizeof(AJC3DVEC));
				memcpy(&line.p2.v3, p2, sizeof(AJC3DVEC));
				G3dCnv3dTo2d(pW, &line.p1.v3, &line.p1.v2);
				G3dCnv3dTo2d(pW, &line.p2.v3, &line.p2.v2);
				line.fArrow = fArrow;
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &line, sizeof line)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, line.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : ライン始点設定																						//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y, z		- 始点位置																			//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dMoveTo		   (HWND hwnd, UI id, double x, double y, double z)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (id < AJC3DG_MAXITEM) {
				pW->V3dMoveTo[id].x = x;
				pW->V3dMoveTo[id].y = y;
				pW->V3dMoveTo[id].z = z;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dMoveToV		   (HWND hwnd, UI id, PCAJC3DVEC ps)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && ps != NULL) {
			if (id < AJC3DG_MAXITEM) {
				pW->V3dMoveTo[id].x = ps->x;
				pW->V3dMoveTo[id].y = ps->y;
				pW->V3dMoveTo[id].z = ps->z;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : ライン終点設定（ライン描画）																			//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y, z		- 始点位置																			//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
static	BOOL	SubG3dLineToV		   (HWND hwnd, UI id, PCAJC3DVEC pe, BOOL fArrow);
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dLineTo		   (HWND hwnd, UI id, double x, double y, double z)
{
	AJC3DVEC	pe;

	pe.x = x;
	pe.y = y;
	pe.z = z;
	return SubG3dLineToV(hwnd, id, &pe, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dLineToV		   (HWND hwnd, UI id, PCAJC3DVEC pe)
{
	return SubG3dLineToV(hwnd, id,	pe, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dArrowTo		   (HWND hwnd, UI id, double x, double y, double z)
{
	AJC3DVEC	pe;

	pe.x = x;
	pe.y = y;
	pe.z = z;
	return SubG3dLineToV(hwnd, id, &pe, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dArrowToV		   (HWND hwnd, UI id, PCAJC3DVEC pe)
{
	return SubG3dLineToV(hwnd, id,	pe, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubG3dLineToV		   (HWND hwnd, UI id, PCAJC3DVEC pe, BOOL fArrow)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pe != NULL) {
			if (id < AJC3DG_MAXITEM) {
				if (rc = SubG3dLineV(hwnd, id, &pW->V3dMoveTo[id], pe, fArrow)) {
					pW->V3dMoveTo[id].x = pe->x;
					pW->V3dMoveTo[id].y = pe->y;
					pW->V3dMoveTo[id].z = pe->z;
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 三角形描画																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xn, yn, zn	- 各頂点位置																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dTriangle	   (HWND hwnd, UI id, double x1, double y1, double z1,
																	  double x2, double y2, double z2,
																	  double x3, double y3, double z3)
{
	AJC3DVEC	p1, p2, p3;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	p3.x = x3; p3.y = y3; p3.z = z3;
	return AjcG3dTriangleV(hwnd, id, &p1, &p2, &p3);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dTriangleV	   (HWND hwnd, UI id, PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && p1 != NULL && p2 != NULL && p3 != NULL) {
			GI_TRIAN	tri;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				tri.h.gic = GIC_TRIAN;
				tri.h.id  = id;
				tri.h.seq = pW->DatSeq++;
				memcpy(&tri.p1.v3, p1, sizeof(AJC3DVEC));
				memcpy(&tri.p2.v3, p2, sizeof(AJC3DVEC));
				memcpy(&tri.p3.v3, p3, sizeof(AJC3DVEC));
				G3dCnv3dTo2d(pW, &tri.p1.v3, &tri.p1.v2);
				G3dCnv3dTo2d(pW, &tri.p2.v3, &tri.p2.v2);
				G3dCnv3dTo2d(pW, &tri.p3.v3, &tri.p3.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &tri, sizeof tri)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, tri.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 四角形描画																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xn, yn, zn	- 各頂点位置																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSquare	   (HWND hwnd, UI id, double x1, double y1, double z1,
																	  double x2, double y2, double z2,
																	  double x3, double y3, double z3,
																	  double x4, double y4, double z4)
{
	AJC3DVEC	p1, p2, p3, p4;
	p1.x = x1; p1.y = y1; p1.z = z1;
	p2.x = x2; p2.y = y2; p2.z = z2;
	p3.x = x3; p3.y = y3; p3.z = z3;
	p4.x = x4; p4.y = y4; p4.z = z4;
	return AjcG3dSquareV(hwnd, id, &p1, &p2, &p3, &p4);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dSquareV	   (HWND hwnd, UI id, PCAJC3DVEC p1, PCAJC3DVEC p2, PCAJC3DVEC p3, PCAJC3DVEC p4)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && p1 != NULL && p2 != NULL && p3 != NULL && p4 != NULL) {
			GI_SQUA		squa;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				squa.h.gic = GIC_SQUA;
				squa.h.id  = id;
				squa.h.seq = pW->DatSeq++;
				memcpy(&squa.p1.v3, p1, sizeof(AJC3DVEC));
				memcpy(&squa.p2.v3, p2, sizeof(AJC3DVEC));
				memcpy(&squa.p3.v3, p3, sizeof(AJC3DVEC));
				memcpy(&squa.p4.v3, p4, sizeof(AJC3DVEC));
				G3dCnv3dTo2d(pW, &squa.p1.v3, &squa.p1.v2);
				G3dCnv3dTo2d(pW, &squa.p2.v3, &squa.p2.v2);
				G3dCnv3dTo2d(pW, &squa.p3.v3, &squa.p3.v2);
				G3dCnv3dTo2d(pW, &squa.p4.v3, &squa.p4.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &squa, sizeof squa)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, squa.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 長方体描画																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xc, yc, zc	- 中心位置																			//
//				xr, yr, zr	- 半径																				//
//				division	- 分割数（１～３２）																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dCube		   (HWND hwnd, UI id, double xc, double yc, double zc,
																	  double xr, double yr, double zr, UI division)
{
	AJC3DVEC	pc;
	pc.x = xc; pc.y = yc; pc.z = zc;
	return AjcG3dCubeV(hwnd, id, &pc, xr, yr, zr, division);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dCubeV		   (HWND hwnd, UI id, PCAJC3DVEC pCent, double xr, double yr, double zr, UI division)
{
	BOOL		rc = TRUE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pCent != NULL) {
			AJC3DVECU	c, r;
			AJC3DVECU	p[4];
			GI_SQUA		squa;
			VOP			pNode;
			UI			i, j, k, n;
			double		v, step;
			static const double	s0[4] = {-1.0, +1.0, +1.0, -1.0};
			static const double	s1[4] = {+1.0, +1.0, -1.0, -1.0};
	
			if (id < AJC3DG_MAXITEM && division > 0) {
				division   = __min(division, 32);
				division   = __max(division,	1);
				squa.h.gic = GIC_SQUA;
				squa.h.id  = id;
				squa.h.seq = pW->DatSeq++;
				c.arr[0] = pCent->x; c.arr[1] = pCent->y; c.arr[2] = pCent->z; 
				r.arr[0] = xr;		 r.arr[1] = yr;		  r.arr[2] = zr;
				for (i = 0; i < AJC3DG_MAXAXIS; i++) {
					j = (i + 1) % AJC3DG_MAXAXIS;
					k = (i + 2) % AJC3DG_MAXAXIS;
					//----- -rの四角 -----------------//
					squa.h.seq = pW->DatSeq++;
					for (n = 0; n < 4; n++) {
						p[n].arr[i] = c.arr[i] + r.arr[i] * s0[n];
						p[n].arr[j] = c.arr[j] + r.arr[j] * s1[n];
						p[n].arr[k] = c.arr[k] - r.arr[k];
					}
					memcpy(&squa.p1.v3, &p[0], sizeof(AJC3DVEC));
					memcpy(&squa.p2.v3, &p[1], sizeof(AJC3DVEC));
					memcpy(&squa.p3.v3, &p[2], sizeof(AJC3DVEC));
					memcpy(&squa.p4.v3, &p[3], sizeof(AJC3DVEC));
					G3dCnv3dTo2d(pW, &squa.p1.v3, &squa.p1.v2);
					G3dCnv3dTo2d(pW, &squa.p2.v3, &squa.p2.v2);
					G3dCnv3dTo2d(pW, &squa.p3.v3, &squa.p3.v2);
					G3dCnv3dTo2d(pW, &squa.p4.v3, &squa.p4.v2);
					if (rc &= ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &squa, sizeof squa)) != NULL)) {
						rc &= AjcAvlInsNode(pW->hAvlDat, squa.h.seq, &pNode, sizeof(VOP));
					}
					//----- +rの四角 -----------------//
					squa.h.seq = pW->DatSeq++;
					for (n = 0; n < 4; n++) {
						p[n].arr[k] = c.arr[k] + r.arr[k];
					}
					memcpy(&squa.p1.v3, &p[0], sizeof(AJC3DVEC));
					memcpy(&squa.p2.v3, &p[1], sizeof(AJC3DVEC));
					memcpy(&squa.p3.v3, &p[2], sizeof(AJC3DVEC));
					memcpy(&squa.p4.v3, &p[3], sizeof(AJC3DVEC));
					G3dCnv3dTo2d(pW, &squa.p1.v3, &squa.p1.v2);
					G3dCnv3dTo2d(pW, &squa.p2.v3, &squa.p2.v2);
					G3dCnv3dTo2d(pW, &squa.p3.v3, &squa.p3.v2);
					G3dCnv3dTo2d(pW, &squa.p4.v3, &squa.p4.v2);
					if (rc &= ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &squa, sizeof squa)) != NULL)) {
						rc &= AjcAvlInsNode(pW->hAvlDat, squa.h.seq, &pNode, sizeof(VOP));
					}
					//----- 中間の四角 ---------------//
					step = r.arr[k] * 2.0 / (double)division;
					for (v = -r.arr[k] + step; v < r.arr[k] - (step / 2.0); v += step) {
						squa.h.seq = pW->DatSeq++;
						for (n = 0; n < 4; n++) {
							p[n].arr[k] = c.arr[k] + v;
						}
						memcpy(&squa.p1.v3, &p[0], sizeof(AJC3DVEC));
						memcpy(&squa.p2.v3, &p[1], sizeof(AJC3DVEC));
						memcpy(&squa.p3.v3, &p[2], sizeof(AJC3DVEC));
						memcpy(&squa.p4.v3, &p[3], sizeof(AJC3DVEC));
						G3dCnv3dTo2d(pW, &squa.p1.v3, &squa.p1.v2);
						G3dCnv3dTo2d(pW, &squa.p2.v3, &squa.p2.v2);
						G3dCnv3dTo2d(pW, &squa.p3.v3, &squa.p3.v2);
						G3dCnv3dTo2d(pW, &squa.p4.v3, &squa.p4.v2);
						if (rc &= ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &squa, sizeof squa)) != NULL)) {
							rc &= AjcAvlInsNode(pW->hAvlDat, squa.h.seq, &pNode, sizeof(VOP));
						}
					}
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
			else rc = FALSE;
		}
		else rc = FALSE;
	}
	else rc = FALSE;

	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 楕球体描画																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xc, yc, zc	- 中心位置																			//
//				xr, yr, zr	- 半径																				//
//				slice		- 水平分割数（２～３２）															//
//				stack		- 垂直分割数																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSphere	   (HWND hwnd, UI id, double xc, double yc, double zc,
																	  double xr, double yr, double zr, UI slice, UI stack)
{
	AJC3DVEC	pc;
	pc.x = xc; pc.y = yc; pc.z = zc;
	return AjcG3dSphereV(hwnd, id, &pc, xr, yr, zr, slice, stack);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dSphereV	   (HWND hwnd, UI id, PCAJC3DVEC pCent, double rx, double ry, double rz, UI slice, UI stack)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pCent != NULL) {
			if (id < AJC3DG_MAXITEM) {
				slice = __min(slice, 32); slice = __max(slice, 1);
				stack = __min(stack, 32); stack = __max(stack, 1);
				rc = G3dMakeMetaBall(hwnd, pCent, rx, ry, rz, slice, stack, (UX)id, CbAg3sMetaBall);
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	CbAg3sMetaBall(PWRK3DGRAPH pW, PGI_ELPS pElps, UX id)
{
	BOOL	rc = FALSE;
	VOP		pNode;

	pElps->h.gic = GIC_ELPS;
	pElps->h.id  = (UW)id;
	pElps->h.seq = pW->DatSeq++;
	if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, pElps, sizeof(GI_ELPS))) != NULL)) {
		rc = AjcAvlInsNode(pW->hAvlDat, pElps->h.seq, &pNode, sizeof(VOP));
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 平面定義																								//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xc, yc, zc	- 平面の原点位置																	//
//				xv, yv, zv	- 平面の法線																		//
//				xo, yo, zo	- 平面上のＸ軸ベクトル																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dDefPlane	   (HWND hwnd, UI id, double xc, double yc, double zc,
																	  double xv, double yv, double zv,
																	  double xo, double yo, double zo)
{
	AJC3DLVEC	lv;
	AJC3DVEC	vb;

	lv.p.x = xc, lv.p.y = yc; lv.p.z = zc;
	lv.v.x = xv, lv.v.y = yv; lv.v.z = zv;
	vb.x   = xo; vb.y	= yo; vb.z	 = zo;
	return AjcG3dDefPlaneV(hwnd, id, &lv, &vb);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dDefPlaneV	   (HWND hwnd, UI id, PCAJC3DLVEC pLVec, PCAJC3DVEC pVOrg)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pLVec != NULL) {
			AJC3DVEC	v1, v2;
			UI			idL, idH;
			if (id < AJC3DG_MAXITEM || id == -1) {
				if (pLVec->v.x != 0.0 || pLVec->v.y != 0.0 || pLVec->v.z != 0.0) {
					if (id == -1) {
						idL = 0;
						idH = 15;
					}
					else {
						idL = idH = id;
					}
					for (id = idL; id <= idH; id++) {
						memcpy(&pW->DatItem[id].PlnLVec, pLVec, sizeof(AJC3DLVEC));
						if (pVOrg == NULL || (pVOrg->x == 0.0 && pVOrg->y == 0.0 && pVOrg->z == 0.0)) {
							AjcV3dAnyOrthoVec(&pLVec->v, &pW->DatItem[id].PlnOrg);
						}
						else {
							AjcV3dAdd(&pLVec->p, pVOrg, &v1);
							AjcV3dCrossP2F(pLVec, &v1, &v2);
							AjcV3dSub(&v2, &pLVec->p, &pW->DatItem[id].PlnOrg);
						}
						AjcV3dNormal(&pW->DatItem[id].PlnOrg, &pW->DatItem[id].PlnOrg);
					}
					if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
					rc = TRUE;
				}
			}
		}
	}
	return	rc;
}

//==============================================================================================================//
//	３Ｄ : 視点設定																								//
//																												//
//	引　数	：	rtx, rty, rtz	- 各軸の回転角度[度]															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetAngle	   (HWND hwnd, double rtx, double rty, double rtz)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AJC3DGPROP	prop;

			memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
			prop.Rot.x = fmod(rtx, 360.0);
			prop.Rot.y = fmod(rty, 360.0);
			prop.Rot.z = fmod(rtz, 360.0);
			rc = G3dSetCtrlProp(pW, &prop);
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 視点設定（法線ベクトル指定）																			//
//																												//
//	引　数	：	pV	- 法線ベクトル																				//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetAngleV	   (HWND hwnd, PCAJC3DVEC pV)
{
	BOOL		rc = FALSE;
	double		tx, ty;
	AJC3DVEC	v;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pV != NULL && AjcV3dLength(pV) != 0) {
			memcpy(&v, pV, sizeof v);
			tx = ty = 0.0;
			//	Ｙ軸，Ｘ軸回転角度算出
			if (v.x != 0.0 || v.z != 0.0) {
				ty = AjcATan2(v.x, v.z);
				AjcV3dRotateY(&v, -ty, &v);
			}
			if (v.y != 0.0 || v.z != 0.0) {
				tx = -AjcATan2(v.y, v.z);
			}
			//	回転角度設定
			AjcG3dSetAngle(hwnd, tx, ty, 0.0);
			rc = TRUE;
		}
	}
	return	rc;
}

//==============================================================================================================//
//	３Ｄ : Ｘ－Ｙ平面アングル設定																				//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetAngleXY   (HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			G3dSetAngleXY(pW);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : Ｘ－Ｚ平面アングル設定																				//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetAngleXZ   (HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			G3dSetAngleXZ(pW);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : Ｙ－Ｚ平面アングル設定																				//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetAngleYZ   (HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			G3dSetAngleYZ(pW);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : ３Ｄアングル設定																						//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetAngle3D   (HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			G3dSetAngle3D(pW);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 平面アングル設定																						//
//																												//
//	引　数	：	xDir		- Ｘ方向の軸の種別と昇順／降順														//
//				yDir		- Ｙ方向の軸の種別と昇順／降順														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetPlane	 (HWND hwnd, AJCG3DAXIS_DIR xDir, AJCG3DAXIS_DIR yDir)
{
	BOOL	rc = FALSE;
	PCROTPLANE	p;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (p = G3dRpSearchAngle(xDir, yDir)) {
				AJC3DGPROP	prop;
				AjcG3dGetProp(hwnd, &prop);
				prop.Rot.x = p->u.s.x;
				prop.Rot.y = p->u.s.y;
				prop.Rot.z = p->u.s.z;
				AjcG3dSetProp(hwnd, &prop);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : ビットマップ取得																						//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//																												//
//	戻り値	：	≠NULL：ビットマップハンドル																	//
//				＝NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT	HBITMAP		WINAPI	AjcG3dGetBitmap		(HWND hwnd)
{
	HBITMAP	hBm = NULL;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			hBm = CopyImage(pW->hBmpImg, IMAGE_BITMAP, pW->ww, pW->wh, LR_COPYRETURNORG);
		}
	}
	return hBm;
}

//==============================================================================================================//
//	３Ｄ : プロファイルからプロパティ読み出し																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//				pDefProp	 - デフォルトプロパティ値（現在値をデフォルトとする場合はNULL)						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dLoadPropA		(HWND hwnd, C_BCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	if (pProfileSect != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAllocW(len))) {
			MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
			rc = AjcG3dLoadPropW(hwnd, pTmp, pDefProp);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dLoadPropW		(HWND hwnd, C_WCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	BOOL		rc = TRUE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProfileSect != NULL) {
			AJC3DGPROP	prop;
	
			if (pDefProp != NULL) memcpy(&prop, pDefProp, sizeof prop);
			else				  rc = AjcG3dGetProp(hwnd, &prop);
	
			if (rc) {
				prop.Rot.x			  = AjcGetProfileRealW(pProfileSect, L"RotX"	 , prop.Rot.x		);
				prop.Rot.y			  = AjcGetProfileRealW(pProfileSect, L"RotY"	 , prop.Rot.y		);
				prop.Rot.z			  = AjcGetProfileRealW(pProfileSect, L"RotZ"	 , prop.Rot.z		);
				prop.Cent.x			  = AjcGetProfileRealW(pProfileSect, L"CentX"	 , prop.Cent.x		);
				prop.Cent.y			  = AjcGetProfileRealW(pProfileSect, L"CentY"	 , prop.Cent.y		);
				prop.Cent.z			  = AjcGetProfileRealW(pProfileSect, L"CentZ"	 , prop.Cent.z		);
				prop.xr				  = AjcGetProfileRealW(pProfileSect, L"xr"		 , prop.xr			);
				prop.yr				  = AjcGetProfileRealW(pProfileSect, L"yr"		 , prop.yr			);
				prop.zr				  = AjcGetProfileRealW(pProfileSect, L"zr"		 , prop.zr			);
				prop.ratio			  = AjcGetProfileRealW(pProfileSect, L"ratio"	 , prop.ratio		);
				prop.PlotSize		  = AjcGetProfileUIntW(pProfileSect, L"PlotSize" , prop.PlotSize	);
				prop.PlotSizeE		  = AjcGetProfileUIntW(pProfileSect, L"PlotSizeE", prop.PlotSizeE	);
				prop.fPlotLine		  = AjcGetProfileUIntW(pProfileSect, L"fPlotLine", prop.fPlotLine	);
				prop.fAspect1		  = AjcGetProfileUIntW(pProfileSect, L"fAspect1" , prop.fAspect1	);
	
				prop.axis[0].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPX"	 , prop.axis[0].rgbP);
				prop.axis[0].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNX"	 , prop.axis[0].rgbN);
				prop.axis[1].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPY"	 , prop.axis[1].rgbP);
				prop.axis[1].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNY"	 , prop.axis[1].rgbN);
				prop.axis[2].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPZ"	 , prop.axis[2].rgbP);
				prop.axis[2].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNZ"	 , prop.axis[2].rgbN);

				prop.Item[ 0].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot0" , prop.Item[ 0].MaxPlot);
				prop.Item[ 0].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP0"	 , prop.Item[ 0].rgbP	);
				prop.Item[ 0].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN0"	 , prop.Item[ 0].rgbN	);
				prop.Item[ 1].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot1" , prop.Item[ 1].MaxPlot);
				prop.Item[ 1].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP1"	 , prop.Item[ 1].rgbP	);
				prop.Item[ 1].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN1"	 , prop.Item[ 1].rgbN	);
				prop.Item[ 2].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot2" , prop.Item[ 2].MaxPlot);
				prop.Item[ 2].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP2"	 , prop.Item[ 2].rgbP	);
				prop.Item[ 2].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN2"	 , prop.Item[ 2].rgbN	);
				prop.Item[ 3].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot3" , prop.Item[ 3].MaxPlot);
				prop.Item[ 3].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP3"	 , prop.Item[ 3].rgbP	);
				prop.Item[ 3].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN3"	 , prop.Item[ 3].rgbN	);
				prop.Item[ 4].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot4" , prop.Item[ 4].MaxPlot);
				prop.Item[ 4].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP4"	 , prop.Item[ 4].rgbP	);
				prop.Item[ 4].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN4"	 , prop.Item[ 4].rgbN	);
				prop.Item[ 5].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot5" , prop.Item[ 5].MaxPlot);
				prop.Item[ 5].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP5"	 , prop.Item[ 5].rgbP	);
				prop.Item[ 5].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN5"	 , prop.Item[ 5].rgbN	);
				prop.Item[ 6].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot6" , prop.Item[ 6].MaxPlot);
				prop.Item[ 6].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP6"	 , prop.Item[ 6].rgbP	);
				prop.Item[ 6].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN6"	 , prop.Item[ 6].rgbN	);
				prop.Item[ 7].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot7" , prop.Item[ 7].MaxPlot);
				prop.Item[ 7].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP7"	 , prop.Item[ 7].rgbP	);
				prop.Item[ 7].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN7"	 , prop.Item[ 7].rgbN	);
				prop.Item[ 8].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot8" , prop.Item[ 8].MaxPlot);
				prop.Item[ 8].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP8"	 , prop.Item[ 8].rgbP	);
				prop.Item[ 8].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN8"	 , prop.Item[ 8].rgbN	);
				prop.Item[ 9].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlot9" , prop.Item[ 9].MaxPlot);
				prop.Item[ 9].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbP9"	 , prop.Item[ 9].rgbP	);
				prop.Item[ 9].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbN9"	 , prop.Item[ 9].rgbN	);
				prop.Item[10].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlotA" , prop.Item[10].MaxPlot);
				prop.Item[10].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPA"	 , prop.Item[10].rgbP	);
				prop.Item[10].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNA"	 , prop.Item[10].rgbN	);
				prop.Item[11].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlotB" , prop.Item[11].MaxPlot);
				prop.Item[11].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPB"	 , prop.Item[11].rgbP	);
				prop.Item[11].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNB"	 , prop.Item[11].rgbN	);
				prop.Item[12].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlotC" , prop.Item[12].MaxPlot);
				prop.Item[12].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPC"	 , prop.Item[12].rgbP	);
				prop.Item[12].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNC"	 , prop.Item[12].rgbN	);
				prop.Item[13].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlotD" , prop.Item[13].MaxPlot);
				prop.Item[13].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPD"	 , prop.Item[13].rgbP	);
				prop.Item[13].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbND"	 , prop.Item[13].rgbN	);
				prop.Item[14].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlotE" , prop.Item[14].MaxPlot);
				prop.Item[14].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPE"	 , prop.Item[14].rgbP	);
				prop.Item[14].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNE"	 , prop.Item[14].rgbN	);
				prop.Item[15].MaxPlot = AjcGetProfileUIntW(pProfileSect, L"MaxPlotF" , prop.Item[15].MaxPlot);
				prop.Item[15].rgbP	  = AjcGetProfileUIntW(pProfileSect, L"rgbPF"	 , prop.Item[15].rgbP	);
				prop.Item[15].rgbN	  = AjcGetProfileUIntW(pProfileSect, L"rgbNF"	 , prop.Item[15].rgbN	);
				rc = AjcG3dSetProp(hwnd, &prop);
			}
		}
		else rc = FALSE;
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	３Ｄ : プロファイルへプロパティ書き込み																		//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dSavePropA		(HWND hwnd, C_BCP pProfileSect)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	if (pProfileSect != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAllocW(len))) {
			MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
			rc = AjcG3dSavePropW(hwnd, pTmp);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dSavePropW		(HWND hwnd, C_WCP pProfileSect)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProfileSect != NULL) {
			AJC3DGPROP	prop;

			if (rc = AjcG3dGetProp(hwnd, &prop)) {
				AjcPutProfileRealW(pProfileSect, L"RotX"		, prop.Rot.x	  );
				AjcPutProfileRealW(pProfileSect, L"RotY"		, prop.Rot.y	  );
				AjcPutProfileRealW(pProfileSect, L"RotZ"		, prop.Rot.z	  );
				AjcPutProfileRealW(pProfileSect, L"CentX"		, prop.Cent.x	  );
				AjcPutProfileRealW(pProfileSect, L"CentY"		, prop.Cent.y	  );
				AjcPutProfileRealW(pProfileSect, L"CentZ"		, prop.Cent.z	  );
				AjcPutProfileRealW(pProfileSect, L"xr"			, prop.xr		  );
				AjcPutProfileRealW(pProfileSect, L"yr"			, prop.yr		  );
				AjcPutProfileRealW(pProfileSect, L"zr"			, prop.zr		  );
				AjcPutProfileRealW(pProfileSect, L"ratio"		, prop.ratio	  );
				AjcPutProfileUIntW(pProfileSect, L"PlotSize"	, prop.PlotSize	  );
				AjcPutProfileUIntW(pProfileSect, L"PlotSizeE" 	, prop.PlotSizeE  );
				AjcPutProfileUIntW(pProfileSect, L"fPlotLine" 	, prop.fPlotLine  );
				AjcPutProfileUIntW(pProfileSect, L"fAspect1"	, prop.fAspect1	  );

				AjcPutProfileUIntW(pProfileSect, L"rgbPX"		, prop.axis[0].rgbP);
				AjcPutProfileUIntW(pProfileSect, L"rgbNX"		, prop.axis[0].rgbN);
				AjcPutProfileUIntW(pProfileSect, L"rgbPY"		, prop.axis[1].rgbP);
				AjcPutProfileUIntW(pProfileSect, L"rgbNY"		, prop.axis[1].rgbN);
				AjcPutProfileUIntW(pProfileSect, L"rgbPZ"		, prop.axis[2].rgbP);
				AjcPutProfileUIntW(pProfileSect, L"rgbNZ"		, prop.axis[2].rgbN);

				AjcPutProfileUIntW(pProfileSect, L"MaxPlot0"	, prop.Item[ 0].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP0"		, prop.Item[ 0].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN0"		, prop.Item[ 0].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot1"	, prop.Item[ 1].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP1"		, prop.Item[ 1].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN1"		, prop.Item[ 1].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot2"	, prop.Item[ 2].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP2"		, prop.Item[ 2].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN2"		, prop.Item[ 2].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot3"	, prop.Item[ 3].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP3"		, prop.Item[ 3].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN3"		, prop.Item[ 3].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot4"	, prop.Item[ 4].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP4"		, prop.Item[ 4].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN4"		, prop.Item[ 4].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot5"	, prop.Item[ 5].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP5"		, prop.Item[ 5].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN5"		, prop.Item[ 5].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot6"	, prop.Item[ 6].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP6"		, prop.Item[ 6].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN6"		, prop.Item[ 6].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot7"	, prop.Item[ 7].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP7"		, prop.Item[ 7].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN7"		, prop.Item[ 7].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot8"	, prop.Item[ 8].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP8"		, prop.Item[ 8].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN8"		, prop.Item[ 8].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlot9"	, prop.Item[ 9].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbP9"		, prop.Item[ 9].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbN9"		, prop.Item[ 9].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlotA"	, prop.Item[10].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbPA"		, prop.Item[10].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbNA"		, prop.Item[10].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlotB"	, prop.Item[11].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbPB"		, prop.Item[11].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbNB"		, prop.Item[11].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlotC"	, prop.Item[12].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbPC"		, prop.Item[12].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbNC"		, prop.Item[12].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlotD"	, prop.Item[13].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbPD"		, prop.Item[13].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbND"		, prop.Item[13].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlotE"	, prop.Item[14].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbPE"		, prop.Item[14].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbNE"		, prop.Item[14].rgbN   );
				AjcPutProfileUIntW(pProfileSect, L"MaxPlotF"	, prop.Item[15].MaxPlot);
				AjcPutProfileUIntW(pProfileSect, L"rgbPF"		, prop.Item[15].rgbP   );
				AjcPutProfileUIntW(pProfileSect, L"rgbNF"		, prop.Item[15].rgbN   );
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : プロファイルからプロパティ，フィルタ設定，ウインドスタイル読み出し									//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//				pDefProp	 - デフォルトプロパティ値（現在値をデフォルトとする場合はNULL)						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dLoadPropExA	(HWND hwnd, C_BCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	if (pProfileSect != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAllocW(len))) {
			MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
			rc = AjcG3dLoadPropExW(hwnd, pTmp, pDefProp);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dLoadPropExW	(HWND hwnd, C_WCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProfileSect != NULL) {
			int			i, msk, flt, styS, styU;
			if (rc = AjcG3dLoadPropW(hwnd, pProfileSect, pDefProp)) {
				//----- 現在のフィルタ値取得 ---------------------------------------------------//
				flt = 0;
				for (i = 0, msk = 0x0001; i < AJC3DG_MAXITEM; i++, msk <<= 1) {
					if (AjcG3dGetFilter(hwnd, i)) flt |= msk;
				}
				//----- プロファイルからフィルタ値読み出し -------------------------------------//
				flt = AjcGetProfileUIntW(pProfileSect, L"FilterValue", flt);
				//----- フィルタ値設定 ---------------------------------------------------------//
				for (i = 0, msk = 0x0001; i < AJC3DG_MAXITEM; i++, msk <<= 1) {
					AjcG3dSetFilter(hwnd, i, (flt & msk) != 0); 
				}
				//----- 現在のスタイル値読み出し -----------------------------------------------//
				styS  = (int)MAjcGetWindowLong(hwnd, GWL_STYLE);
				styU  = (styS & 0xFFFF);
				//----- プロファイルからスタイル値読み出し -------------------------------------//
				styU  = AjcGetProfileUIntW(pProfileSect, L"WndStyle", styU);
				styU &= 0xFFFF;
				//----- スタイル設定 -----------------------------------------------------------//
				styS &= ~0xFFFF;
				styS |= styU;
				MAjcSetWindowLong(hwnd, GWL_STYLE, styS);
				//----- プロファイルから平面アングル読み出し -----------------------------------//
				pW->PlaneAxisH = (AJCG3DAXIS_DIR)AjcGetProfileUIntW(pProfileSect, L"PlaneAxisH", (UI)pW->PlaneAxisH);
				pW->PlaneAxisV = (AJCG3DAXIS_DIR)AjcGetProfileUIntW(pProfileSect, L"PlaneAxisV", (UI)pW->PlaneAxisV);
				//----- ２Ｄモード時は、平面アングル設定 ---------------------------------------//
				if (pW->f2dMode) {
					AjcG2dSetPlane(hwnd, pW->PlaneAxisH, pW->PlaneAxisV);
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : プロファイルへプロパティ，フィルタ設定，ウインドスタイル書き込み										//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dSavePropExA	(HWND hwnd, C_BCP pProfileSect)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
	if (len != 0 && (pTmp = AjcTAllocW(len))) {
		MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
		rc = AjcG3dSavePropExW(hwnd, pTmp);
	}
	if (pTmp != NULL) free(pTmp);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG3dSavePropExW	(HWND hwnd, C_WCP pProfileSect)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int			i, msk, flt, sty;

			if (rc = AjcG3dSavePropW(hwnd, pProfileSect)) {
				//----- 現在のフィルタ値取得 ---------------------------------------------------//
				flt = 0;
				for (i = 0, msk = 0x0001; i < AJC3DG_MAXITEM; i++, msk <<= 1) {
					if (AjcG3dGetFilter(hwnd, i)) flt |= msk;
				}
				//----- プロファイルへフィルタ値書き込み ---------------------------------------//
				AjcPutProfileUIntW(pProfileSect, L"FilterValue", flt);
				//----- 現在のスタイル値読み出し -----------------------------------------------//
				sty	 = (int)MAjcGetWindowLong(hwnd, GWL_STYLE);
				sty &= 0xFFFF;
				//----- プロファイルへスタイル値書き込み ---------------------------------------//
				AjcPutProfileUIntW(pProfileSect, L"WndStyle", sty);
				//----- プロファイルへ平面アングル書き込み -------------------------------------//
				AjcPutProfileUIntW(pProfileSect, L"PlaneAxisH", (UI)pW->PlaneAxisH);
				AjcPutProfileUIntW(pProfileSect, L"PlaneAxisV", (UI)pW->PlaneAxisV);
			}
		}
	}
	return rc;
}


//==============================================================================================================//
//	３Ｄ : テキスト描画フォント設定																				//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				hFont		 - フォントハンドル																	//
//																												//
//	戻り値	：	≠NULL : 前回のフォントハンドル																	//
//				＝NULL : 失敗																					//
//==============================================================================================================//
AJCEXPORT	HFONT	WINAPI		AjcG3dSetTextFont	(HWND hwnd, HFONT hFont)
{
	HFONT	rc = NULL;
	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && hFont != NULL) {
			rc = pW->hFontTxo;
			pW->hFontTxo = hFont;
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : テキスト描画（ピクセル位置指定）																		//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				x, y		 - 描画位置																			//
//				pTxt		 - 描画テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dTextOutA		(HWND hwnd, int x, int y, C_BCP pTxt)
{
	UI		rc = 0;
	UI		len;
	WCP		pTmp = NULL;

	len = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
	if (len != 0 && (pTmp = AjcTAllocW(len))) {
		MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, len);
		rc = AjcG3dTextOutW(hwnd, x, y, pTmp);
	}
	if (pTmp != NULL) free(pTmp);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dTextOutW		(HWND hwnd, int x, int y, C_WCP pTxt)
{
	UI		rc = 0;
	PG3DTXO	pNode;
	UI		stl, len;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (stl = (UI)wcslen(pTxt)) {
				len = sizeof(G3DTXO) + stl * 2;
				if (pNode = malloc(len)) {
					memset(pNode, 0, sizeof(G3DTXO));
					pNode->fPix  = TRUE;
					pNode->u.pt.x = x;
					pNode->u.pt.y = y;
					wcscpy(pNode->txt, pTxt);
					if (AjcAvlInsNode(pW->hAvlTxo, pW->kAvlTxo, (C_VOP)pNode, len)) {
						rc = pW->kAvlTxo++;
						InvalidateRect(pW->hMain, NULL, FALSE);
					}
					free(pNode);
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : テキスト描画（３Ｄ座標指定）																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pV			 - 描画位置																			//
//				md			 - テキスト描画方法																	//
//				pTxt		 - 描画テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dTextOutVA		(HWND hwnd, AJCG3DTXOMD md, PCAJC3DVEC pV, C_BCP pTxt)
{
	UI		rc = 0;
	UI		len;
	WCP		pTmp = NULL;

	len = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
	if (len != 0 && (pTmp = AjcTAllocW(len))) {
		MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, len);
		rc = AjcG3dTextOutVW(hwnd, md, pV, pTmp);
	}
	if (pTmp != NULL) free(pTmp);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dTextOutVW		(HWND hwnd, AJCG3DTXOMD md, PCAJC3DVEC pV, C_WCP pTxt)
{
	UI		rc = 0;
	PG3DTXO	pNode;
	UI		stl, len;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (stl = (UI)wcslen(pTxt)) {
				len = sizeof(G3DTXO) + stl * 2;
				if (pNode = malloc(len)) {
					memset(pNode, 0, sizeof(G3DTXO));
					pNode->md	 = md;
					pNode->fPix  = FALSE;
					memcpy(&pNode->u.v, pV, sizeof(AJC3DVEC));
					wcscpy(pNode->txt, pTxt);
					if (AjcAvlInsNode(pW->hAvlTxo, pW->kAvlTxo, (C_VOP)pNode, len)) {
						rc = pW->kAvlTxo++;
						InvalidateRect(pW->hMain, NULL, FALSE);
					}
					free(pNode);
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 書式テキスト描画（ピクセル位置指定）																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				x, y		 - 描画位置																			//
//				pFmt		 - 書式テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dPrintFA		(HWND hwnd, int x, int y, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFA(buf, 2048, pFmt, vls);
		va_end	(vls);
		rc = AjcG3dTextOutA(hwnd, x, y, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dPrintFW		(HWND hwnd, int x, int y, C_WCP pFmt, ...)
{
	UI		rc = 0;
	va_list	vls;
	WC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFW(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcG3dTextOutW(hwnd, x, y, buf);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 書式テキスト描画（３Ｄ座標指定）																		//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pV			 - 描画位置																			//
//				md			 - テキスト描画方法																	//
//				pFmt		 - 書式テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dPrintFVA		(HWND hwnd, AJCG3DTXOMD md, PCAJC3DVEC pV, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFA(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcG3dTextOutVA(hwnd, md, pV, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dPrintFVW		(HWND hwnd, AJCG3DTXOMD md, PCAJC3DVEC pV, C_WCP pFmt, ...)
{
	UI		rc = 0;
	va_list	vls;
	WC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFW(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcG3dTextOutVW(hwnd, md, pV, buf);
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 描画テキスト取得																						//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				key			 - テキストキー																		//
//				pBuf		 - テキスト格納バッファへのポインタ（不要時はNULL）									//
//				lBuft		 - テキスト格納バッファのバイト数／文字数											//
//																												//
//	戻り値	：	≠0 : 描画テキストのバイト数／文字数															//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dGetTextA		(HWND hwnd, UI key, BCP pBuf, UI lBuf)
{
	UI		rc = 0;
	PG3DTXO	pNode;
	UI		lNode;
	UI		stl;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (pNode = (PG3DTXO)AjcAvlGetNodePtr(pW->hAvlTxo, key, &lNode)) {
				stl = (UI)wcslen(pNode->txt);
				rc = WideCharToMultiByte(CP_ACP, 0, pNode->txt, stl, NULL, 0, NULL, NULL);
				if (pBuf != NULL && lBuf != 0) {
					WideCharToMultiByte(CP_ACP, 0, pNode->txt, -1, pBuf, lBuf, NULL, NULL);
					pBuf[lBuf - 1] = 0;
				}
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dGetTextW		(HWND hwnd, UI key, WCP pBuf, UI lBuf)
{
	UI		rc = 0;
	PG3DTXO	pNode;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (pNode = AjcAvlGetNodePtr(pW->hAvlTxo, key, NULL)) {
				rc = (UI)wcslen(pNode->txt);
				if (pBuf != NULL && lBuf != 0) {
					wcsncpy(pBuf, pNode->txt, lBuf);
					pBuf[lBuf - 1] = 0;
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 図形描画データクリアー																				//
//																												//
//	引　数	：	id			- データ項目番号																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dClearShape   (HWND hwnd, UI id)
{
	BOOL		rc = FALSE;
	GI_INFO		giBuf;;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (id < AJC3DG_MAXITEM) {
				//----- 全描画データ取り出し＆ＡＶＬノード削除 ---//
				while (AjcVQueDeque(pW->DatItem[id].hQueDat, &giBuf, sizeof giBuf) != 0) {
					AjcAvlDelNode(pW->hAvlDat, giBuf.h.seq);
				}
				//----- 再表示 -----------------------------------//
				pW->fReDrawData = TRUE;
				InvalidateRect(hwnd, NULL, FALSE);
				//----- クリアー通知 -----------------------------//
				pW->NtcClrFact = AJC3DG_CLEAR_DATA;
				SendMessage(GetParent(pW->hMain), WM_COMMAND,
						MAKELONG(MAjcGetWindowLong(pW->hMain, GWL_ID), AJC3DGN_CLEAR),
						fCmdWithHdl ? (LPARAM)pW->hMain : (LPARAM)pW->NtcClrFact);
				rc = TRUE;
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 全ての図形描画データクリアー																			//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dClearAllShape	   (HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			UI			i;
			//----- データクリアー ----------------------//
			for (i = 0; i < AJC3DG_MAXITEM; i++) {
				//	描画データキュー消去
				AjcVQuePurge(pW->DatItem[i].hQueDat);
			}
			//----- 描画順序ＡＶＬデータ消去 ------------//
			AjcAvlDelAllNodes(pW->hAvlDat);
			//----- 描画順序番号クリアー ----------------//
			pW->DatSeq = 0;
			//----- 再表示 ------------------------------//
			pW->fReDrawData = TRUE;
			InvalidateRect(hwnd, NULL, FALSE);
			//----- クリアー通知 ------------------------//
			pW->NtcClrFact = (AJC3DG_CLEAR_PLOT | AJC3DG_CLEAR_DATA);
			SendMessage(GetParent(pW->hMain), WM_COMMAND,
					MAKELONG(MAjcGetWindowLong(pW->hMain, GWL_ID), AJC3DGN_CLEAR),
					fCmdWithHdl ? (LPARAM)pW->hMain : (LPARAM)pW->NtcClrFact);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : プロットデータクリアー																				//
//																												//
//	引　数	：	id			- データ項目番号																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dClearPlot	   (HWND hwnd, UI id)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			//----- プロットデータクリアー，再表示 ------//
			if (id < AJC3DG_MAXITEM) {
				pW->DatItem[id].plot.ix	 = 0;
				pW->DatItem[id].plot.cnt = 0;
				InvalidateRect(hwnd, NULL, FALSE);
				//----- クリアー通知 --------------------//
				pW->NtcClrFact = AJC3DG_CLEAR_PLOT;
				SendMessage(GetParent(pW->hMain), WM_COMMAND,
						MAKELONG(MAjcGetWindowLong(pW->hMain, GWL_ID), AJC3DGN_CLEAR),
						fCmdWithHdl ? (LPARAM)pW->hMain : (LPARAM)pW->NtcClrFact);
				rc = TRUE;
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	３Ｄ : 全てのプロットデータクリアー																			//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dClearAllPlot	   (HWND hwnd)
{
	BOOL	rc = FALSE;
	UI		id;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			//----- プロットデータクリアー，再表示 ------//
			for (id = 0; id < AJC3DG_MAXITEM; id++) {
				pW->DatItem[id].plot.ix	 = 0;
				pW->DatItem[id].plot.cnt = 0;
			}
			InvalidateRect(hwnd, NULL, FALSE);
			//----- クリアー通知 --------------------//
			pW->NtcClrFact = AJC3DG_CLEAR_PLOT;
			SendMessage(GetParent(pW->hMain), WM_COMMAND,
					MAKELONG(MAjcGetWindowLong(pW->hMain, GWL_ID), AJC3DGN_CLEAR),
					fCmdWithHdl ? (LPARAM)pW->hMain : (LPARAM)pW->NtcClrFact);
			rc = TRUE;
		}
	}
	return	rc;
}

//==============================================================================================================//
//	３Ｄ : 描画テキストクリアー																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				key			 - テキストキー																		//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI		AjcG3dClearText		(HWND hwnd, UI key)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = AjcAvlDelNode(pW->hAvlTxo, key);
			InvalidateRect(pW->hMain, NULL, FALSE);
		}
	}
	return rc;
}

//==============================================================================================================//
//	３Ｄ : 全ての描画テキストクリアー																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI		AjcG3dClearAllText		(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AjcAvlDelAllNodes(pW->hAvlTxo);
			pW->kAvlTxo = 1;
			InvalidateRect(pW->hMain, NULL, FALSE);
		}
	}
	return rc;
}
//==============================================================================================================//
//	３Ｄ : 描画データ（描画データ，プロットデータ）クリアー														//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dClearData		(HWND hwnd, UI id)
{
	BOOL	rc = TRUE;

	rc &= AjcG2dClearShape(hwnd, id);
	rc &= AjcG2dClearPlot (hwnd, id);

	return rc;
}
//==============================================================================================================//
//	３Ｄ : 全ての描画データ（描画データ，プロットデータ）クリアー												//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				id			- データ項目番号																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dClearAllData		(HWND hwnd)
{
	BOOL	rc = TRUE;

	rc &= AjcG3dClearAllShape(hwnd);
	rc &= AjcG3dClearAllPlot (hwnd);

	return rc;
}
//==============================================================================================================//
//	３Ｄ : 全てのデータ（図形，プロット，テキスト）クリアー														//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dClear				(HWND hwnd)
{
	BOOL	rc = TRUE;

	rc &= AjcG3dClearAllShape(hwnd);
	rc &= AjcG3dClearAllPlot (hwnd);
	rc &= AjcG3dClearAllText (hwnd);

	return rc;
}


//**************************************************************************************************************//
//																												//
//		２Ｄファンクション																						//
//																												//
//**************************************************************************************************************//
//==============================================================================================================//
//	２Ｄ : 初期化																								//
//																												//
//	引　数	：	x1, x2		- Ｘ軸のレンジ																		//
//				y1, y2		- Ｙ軸のレンジ																		//
//				style		- コントロールのスタイル															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dInit		   (HWND hwnd, double x1, double y1, double x2, double y2, UI style)
{
	BOOL		rc = FALSE;
	AJC2DVEC	vLo, vHi;

	vLo.x = x1;
	vLo.y = y1;
	vHi.x = x2;
	vHi.y = y2;

	rc = AjcG2dInitV(hwnd, &vLo, &vHi, style);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dInitV		   (HWND hwnd, PCAJC2DVEC pLo, PCAJC2DVEC pHi, UI style)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			UI			sty;
			double		xmin;
			double		xmax;
			double		ymin;
			double		ymax;
			AJC3DGPROP	prop;
	
			if (pLo != NULL && pHi != NULL) {
				xmin = __min(pLo->x, pHi->x);
				xmax = __max(pLo->x, pHi->x);
				ymin = __min(pLo->y, pHi->y);
				ymax = __max(pLo->y, pHi->y);
				memcpy(&prop, &pW->prop, sizeof(AJC3DGPROP));
				prop.Rot.x	= 0.0;
				prop.Rot.y	= 0.0;
				prop.Rot.z	= 0.0;
				prop.Cent.x = (xmin + xmax) / 2.0;
				prop.Cent.y = (ymin + ymax) / 2.0;
				prop.Cent.z = 0.0;
				prop.xr		= (xmax - xmin) / 2.0;
				prop.yr		= (ymax - ymin) / 2.0;
				prop.zr		= 1.0;
				prop.ratio	= 1.0;
				//	プロパティ設定
				if (prop.xr > 0.0  &&	prop.yr > 0.0) {
					G3dSetCtrlProp(pW, &prop);
				}
			}
			//	２Ｄモード設定
			G3dSet2dMode(pW);
			//	スタイル設定
			sty = (UI)MAjcGetWindowLong(pW->hMain, GWL_STYLE) & 0xFFFF0000;
			if (style != -1) {
				sty |= ((style & 0xFFFF));
			}
			MAjcSetWindowLong(pW->hMain, GWL_STYLE, sty | (AJC3DGS_NODEPTHCTRL | AJC3DGS_NOANGLE));
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 平面アングル設定																						//
//																												//
//	引　数	：	xDir		- Ｘ方向の軸の種別と昇順／降順														//
//				yDir		- Ｙ方向の軸の種別と昇順／降順														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetPlane	 (HWND hwnd, AJCG3DAXIS_DIR xDir, AJCG3DAXIS_DIR yDir)
{
	BOOL	rc;
	rc = AjcG3dSetPlane(hwnd, xDir, yDir);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : プロパティ設定																						//
//																												//
//	引　数	：	pProp		- 設定するプロパティデータのアドレス												//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetProp	   (HWND hwnd, PCAJC2DGPROP pProp)
{
	BOOL	rc;
	rc = AjcG3dSetProp(hwnd, pProp);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : プロパティ取得																						//
//																												//
//	引　数	：	pBuf		- 取得したプロパティデータを格納するバッファのアドレス								//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dGetProp	   (HWND hwnd, PAJC2DGPROP	pBuf)
{
	BOOL	rc;
	rc = AjcG3dGetProp(hwnd, pBuf);
	return rc;
}

//==============================================================================================================//
//	２Ｄ : レンジ自動調整																						//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dAdjustRange  (HWND hwnd)
{
	BOOL	rc;
	rc = AjcG3dAdjustRange(hwnd);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : レンジ設定																							//
//																												//
//	引　数	：	x1, x2		- Ｘ軸のレンジ																		//
//				y1, y2		- Ｙ軸のレンジ																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetRange	   (HWND hwnd, double x1, double y1, double x2, double y2)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = AjcG3dSetRange(hwnd, x1, y1, -1.0, x2, y2, +1.0)) {
				G3dSet2dMode(pW);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 中心位置設定																							//
//																												//
//	引　数	：	xc, yc		- 各軸の中心位置																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetCenter	   (HWND hwnd, double xc, double yc)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = AjcG3dSetCenter(hwnd, xc, yc, 0.0)) {
				G3dSet2dMode(pW);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 各軸の幅（半径）設定																					//
//																												//
//	引　数	：	xr, yr		- 各軸の半径																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetWidth	   (HWND hwnd, double xr, double yr)
{
	BOOL	rc;
	rc = AjcG3dSetWidth(hwnd, xr, yr, 1.0);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 各軸のレンジ幅を同一サイズに揃える																	//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG2dSetSameRangeWidth(HWND hwnd)
{
	return AjcG3dSetSameRangeWidth(hwnd);
}
//==============================================================================================================//
//	２Ｄ : レンジの表示域をアスペクト１にする（レンジの表示域を正方形にする）									//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG2dSetFixedAspect   (HWND hwnd, BOOL fFixed)
{
	return AjcG3dSetFixedAspect(hwnd, fFixed);
}
//==============================================================================================================//
//	２Ｄ : 表示色設定																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				rgb			- 表示色																			//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetColor	   (HWND hwnd, UI id, COLORREF rgb)
{
	BOOL	rc;
	rc = AjcG3dSetColor(hwnd, id, rgb, rgb);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : プロットデータ数設定																					//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				PlotNumber	- プロットデータ数																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetPlotNumber(HWND hwnd, UI id, UI PlotNumber)
{
	BOOL	rc;
	rc = AjcG3dSetPlotNumber(hwnd, id, PlotNumber);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : プロット点のサイズ設定																				//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				PixelSize	- プロット点のサイズ																//
//				PixelSizeE	- 終点のプロット点サイズ															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSetPlotSize  (HWND hwnd, UI id, UI PixelSize, UI PixelSizeE)
{
	BOOL	rc;
	rc = AjcG3dSetPlotSize(hwnd, id, PixelSize, PixelSizeE);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : プロットデータ投与																					//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y		- プロットデータ																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dPutPlotData  (HWND hwnd, UI id, double x, double y)
{
	BOOL	rc;
	rc = AjcG3dPutPlotData(hwnd, id, x, y, 0.0);
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dPutPlotDataV (HWND hwnd, UI id, PCAJC2DVEC pPoint)
{
	BOOL	rc;
	rc = AjcG3dPutPlotData(hwnd, id, pPoint->x, pPoint->y, 0.0);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : ビットマップ取得																						//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//																												//
//	戻り値	：	≠NULL：ビットマップハンドル																	//
//				＝NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT	HBITMAP		WINAPI	AjcG2dGetBitmap		(HWND hwnd)
{
	return AjcG3dGetBitmap(hwnd);
}

//==============================================================================================================//
//	２Ｄ : プロファイルからプロパティ読み出し																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//				pDefProp	 - デフォルトプロパティ値（現在値をデフォルトとする場合はNULL)						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dLoadPropA		(HWND hwnd, C_BCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	return AjcG3dLoadPropA(hwnd, pProfileSect, pDefProp);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dLoadPropW		(HWND hwnd, C_WCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	return AjcG3dLoadPropW(hwnd, pProfileSect, pDefProp);
}
//==============================================================================================================//
//	２Ｄ : プロファイルへプロパティ書き込み																		//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dSavePropA		(HWND hwnd, C_BCP pProfileSect)
{
	return AjcG3dSavePropA(hwnd, pProfileSect);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dSavePropW		(HWND hwnd, C_WCP pProfileSect)
{
	return AjcG3dSavePropW(hwnd, pProfileSect);
}
//==============================================================================================================//
//	２Ｄ : プロファイルからプロパティ，フィルタ設定，ウインドスタイル読み出し									//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//				pDefProp	 - デフォルトプロパティ値（現在値をデフォルトとする場合はNULL)						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dLoadPropExA	(HWND hwnd, C_BCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	return AjcG3dLoadPropExA(hwnd, pProfileSect, pDefProp);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dLoadPropExW	(HWND hwnd, C_WCP pProfileSect, PCAJC3DGPROP pDefProp)
{
	return AjcG3dLoadPropExW(hwnd, pProfileSect, pDefProp);
}
//==============================================================================================================//
//	２Ｄ : プロファイルへプロパティ，フィルタ設定，ウインドスタイル書き込み										//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dSavePropExA	(HWND hwnd, C_BCP pProfileSect)
{
	return AjcG3dSavePropExA(hwnd, pProfileSect);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcG2dSavePropExW	(HWND hwnd, C_WCP pProfileSect)
{
	return AjcG3dSavePropExW(hwnd, pProfileSect);
}
//==============================================================================================================//
//	２Ｄ : テキスト描画フォント設定																				//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				hFont		 - フォントハンドル																	//
//																												//
//	戻り値	：	≠NULL : 前回のフォントハンドル																	//
//				＝NULL : 失敗																					//
//==============================================================================================================//
AJCEXPORT	HFONT	WINAPI		AjcG2dSetTextFont	(HWND hwnd, HFONT hFont)
{
	return AjcG3dSetTextFont(hwnd, hFont);
}

//==============================================================================================================//
//	２Ｄ : テキスト描画（ピクセル位置指定）																		//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				x, y		 - 描画位置																			//
//				pTxt		 - 描画テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dTextOutA		(HWND hwnd, int x, int y, C_BCP pTxt)
{
	return AjcG3dTextOutA(hwnd, x, y, pTxt);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dTextOutW		(HWND hwnd, int x, int y, C_WCP pTxt)
{
	return AjcG3dTextOutW(hwnd, x, y, pTxt);
}

//==============================================================================================================//
//	２Ｄ : テキスト描画（２Ｄ座標指定）																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				md			 - テキスト描画方法																	//
//				pV			 - 描画位置																			//
//				pTxt		 - 描画テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dTextOutVA		(HWND hwnd, AJCG2DTXOMD md, PCAJC2DVEC pV, C_BCP pTxt)
{
	AJC3DVEC	v3;

	v3.x = pV->x;
	v3.y = pV->y;
	v3.z = 0;
	return AjcG3dTextOutVA(hwnd, md, &v3, pTxt);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dTextOutVW		(HWND hwnd, AJCG2DTXOMD md, PCAJC2DVEC pV, C_WCP pTxt)
{
	AJC3DVEC	v3;

	v3.x = pV->x;
	v3.y = pV->y;
	v3.z = 0;
	return AjcG3dTextOutVW(hwnd, md, &v3, pTxt);
}

//==============================================================================================================//
//	２Ｄ : 書式テキスト描画（ピクセル位置指定）																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				x, y		 - 描画位置																			//
//				pFmt		 - 書式テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dPrintFA		(HWND hwnd, int x, int y, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcG3dTextOutA(hwnd, x, y, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dPrintFW		(HWND hwnd, int x, int y, C_WCP pFmt, ...)
{
	UI		rc = 0;
	va_list	vls;
	WC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		_vsnwprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcG3dTextOutW(hwnd, x, y, buf);
	}
	return rc;
}

//==============================================================================================================//
//	２Ｄ : 書式テキスト描画（２Ｄ座標指定）																		//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				md			 - テキスト描画方法																	//
//				pV			 - 描画位置																			//
//				pFmt		 - 書式テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dPrintFVA		(HWND hwnd, AJCG2DTXOMD md, PCAJC2DVEC pV, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];
	AJC3DVEC	v3;

	if (pFmt != NULL) {
		v3.x = pV->x;
		v3.y = pV->y;
		v3.z = 0;
		va_start(vls, pFmt);
		_vsnprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcG3dTextOutVA(hwnd, md, &v3, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dPrintFVW		(HWND hwnd, AJCG2DTXOMD md, PCAJC2DVEC pV, C_WCP pFmt, ...)
{
	UI		rc = 0;
	va_list	vls;
	WC		buf[2048];
	AJC3DVEC	v3;

	if (pFmt != NULL) {
		v3.x = pV->x;
		v3.y = pV->y;
		v3.z = 0;
		va_start(vls, pFmt);
		_vsnwprintf(buf, 1024, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcG3dTextOutVW(hwnd, md, &v3, buf);
	}
	return rc;

}

//==============================================================================================================//
//	２Ｄ : 描画テキスト取得																						//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				key			 - テキストキー																		//
//				pBuf		 - テキスト格納バッファへのポインタ（不要時はNULL）									//
//				lBuft		 - テキスト格納バッファのバイト数／文字数											//
//																												//
//	戻り値	：	≠0 : 描画テキストのバイト数／文字数															//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dGetTextA		(HWND hwnd, UI key, BCP pBuf, UI lBuf)
{
	return AjcG3dGetTextA(hwnd, key, pBuf, lBuf);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG2dGetTextW		(HWND hwnd, UI key, WCP pBuf, UI lBuf)
{
	return AjcG3dGetTextW(hwnd, key, pBuf, lBuf);
}

//==============================================================================================================//
//	２Ｄ : 描画テキストクリアー																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				key			 - テキストキー																		//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearText		(HWND hwnd, UI key)
{
	return AjcG3dClearText(hwnd, key);
}

//==============================================================================================================//
//	２Ｄ : 全ての描画テキストクリアー																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearAllText		(HWND hwnd)
{
	return AjcG3dClearAllText(hwnd);
}
//==============================================================================================================//
//	２Ｄ : 図形描画データクリアー																				//
//																												//
//	引　数	：	id			- データ項目番号																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearShape	   (HWND hwnd, UI id)
{
	BOOL	rc;
	rc = AjcG3dClearShape(hwnd, id);
	return rc;
}

//==============================================================================================================//
//	２Ｄ : 全ての図形描画データクリアー																			//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearAllShape	   (HWND hwnd)
{
	BOOL	rc;
	rc = AjcG3dClearAllShape(hwnd);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : プロットデータクリアー																				//
//																												//
//	引　数	：	id			- データ項目番号																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearPlot	   (HWND hwnd, UI id)
{
	BOOL	rc;
	rc = AjcG3dClearPlot(hwnd, id);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 全てのプロットデータクリアー																			//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearAllPlot	   (HWND hwnd)
{
	BOOL	rc;
	rc = AjcG3dClearAllPlot(hwnd);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 描画データ（描画データ，プロットデータ）クリアー														//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				id			- データ項目番号																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearData		(HWND hwnd, UI id)
{
	BOOL	rc;
	rc = AjcG3dClearData(hwnd, id);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 全ての描画データ（描画データ，プロットデータ）クリアー												//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClearAllData		(HWND hwnd)
{
	BOOL	rc;
	rc = AjcG3dClearAllData(hwnd);
	return rc;
}
//==============================================================================================================//
//	２Ｄ : 全てのデータ（図形，プロット，テキスト）クリアー														//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dClear				(HWND hwnd)
{
	return AjcG3dClear(hwnd);
}

//**************************************************************************************************************//
//																												//
//		平面描画ファンクション（２Ｄ／３Ｄ共通）																//
//																												//
//**************************************************************************************************************//
//==============================================================================================================//
//	平面描画 : ピクセル描画																						//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y		- 描画位置																			//
//				PixelSize	- 描画ピクセルのサイズ																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dPixel		   (HWND hwnd, UI id, double x, double y, UI PixelSize)
{
	AJC2DVEC	pt;
	pt.x = x; pt.y = y;
	return AjcG2dPixelV(hwnd, id, &pt, PixelSize);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dPixelV	   (HWND hwnd, UI id, PCAJC2DVEC pPoint, UI PixelSize)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pPoint != NULL) {
			AJC2DVEC	v;
			GI_POINT	poi;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				poi.h.gic = GIC_POINT;
				poi.h.id  = id;
				poi.h.seq = pW->DatSeq++;
				poi.siz   = PixelSize;
				memcpy(&v, pPoint, sizeof v);
				if (pW->f2dMode) {
					v.x -= pW->prop.Cent.x;
					v.y -= pW->prop.Cent.y;
				}
				G3dPlanePointTo3dPoint(pW, &v, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &poi.p.v3);
				G3dCnv3dTo2d(pW, &poi.p.v3, &poi.p.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &poi, sizeof poi)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, poi.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : ライン描画																						//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x1, y1		- 始点位置																			//
//				x2, y2		- 終点位置																			//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
static	BOOL	SubG2dLineV(HWND hwnd, UI id, PCAJC2DVEC p1, PCAJC2DVEC p2, BOOL fArrow);
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dLine		   (HWND hwnd, UI id, double x1, double y1, double x2, double y2)
{
	AJC2DVEC	p1, p2;
	p1.x = x1; p1.y = y1;
	p2.x = x2; p2.y = y2;
	return SubG2dLineV(hwnd, id, &p1, &p2, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dLineV		   (HWND hwnd, UI id, PCAJC2DVEC p1, PCAJC2DVEC p2)
{
	return SubG2dLineV(hwnd, id,  p1,  p2, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dArrow		   (HWND hwnd, UI id, double x1, double y1, double x2, double y2)
{
	AJC2DVEC	p1, p2;
	p1.x = x1; p1.y = y1;
	p2.x = x2; p2.y = y2;
	return SubG2dLineV(hwnd, id, &p1, &p2, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dArrowV		   (HWND hwnd, UI id, PCAJC2DVEC p1, PCAJC2DVEC p2)
{
	return SubG2dLineV(hwnd, id,  p1,  p2, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubG2dLineV(HWND hwnd, UI id, PCAJC2DVEC p1, PCAJC2DVEC p2, BOOL fArrow)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && p1 != NULL && p2 != NULL) {
			AJC2DVEC	v1, v2;
			GI_LINE	line;
			VOP		pNode;

			if (id < AJC3DG_MAXITEM) {
				line.h.gic = GIC_LINE;
				line.h.id  = id;
				line.h.seq = pW->DatSeq++;
				memcpy(&v1, p1, sizeof v1);
				memcpy(&v2, p2, sizeof v2);
				if (pW->f2dMode) {
					v1.x -= pW->prop.Cent.x; v2.x -= pW->prop.Cent.x;
					v1.y -= pW->prop.Cent.y; v2.y -= pW->prop.Cent.y;
				}
				G3dPlanePointTo3dPoint(pW, &v1, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &line.p1.v3);
				G3dPlanePointTo3dPoint(pW, &v2, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &line.p2.v3);
				G3dCnv3dTo2d(pW, &line.p1.v3, &line.p1.v2);
				G3dCnv3dTo2d(pW, &line.p2.v3, &line.p2.v2);
				line.fArrow = fArrow;
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &line, sizeof line)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, line.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : ライン始点設定																					//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y		- 始点位置																			//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dMoveTo		   (HWND hwnd, UI id, double x, double y)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (id < AJC3DG_MAXITEM) {
				pW->V2dMoveTo[id].x = x;
				pW->V2dMoveTo[id].y = y;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dMoveToV		   (HWND hwnd, UI id, PCAJC2DVEC ps)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && ps != NULL) {
			if (id < AJC3DG_MAXITEM) {
				pW->V2dMoveTo[id].x = ps->x;
				pW->V2dMoveTo[id].y = ps->y;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	平面描画 : ライン終点設定（ライン描画）																		//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y		- 始点位置																			//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
static	BOOL	SubG2dLineToV	(HWND hwnd, UI id, PCAJC2DVEC pe, BOOL fArrow);
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dLineTo		   (HWND hwnd, UI id, double x, double y)
{
	AJC2DVEC	pe;

	pe.x = x;
	pe.y = y;
	return SubG2dLineToV(hwnd, id, &pe, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dLineToV		   (HWND hwnd, UI id, PCAJC2DVEC pe)
{
	return SubG2dLineToV(hwnd, id,	pe, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dArrowTo		   (HWND hwnd, UI id, double x, double y)
{
	AJC2DVEC	pe;

	pe.x = x;
	pe.y = y;
	return SubG2dLineToV(hwnd, id, &pe, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dArrowToV		   (HWND hwnd, UI id, PCAJC2DVEC pe)
{
	return SubG2dLineToV(hwnd, id,	pe, TRUE);
}
//--------------------------------------------------------------------------------------------------------------//
static	BOOL	SubG2dLineToV	(HWND hwnd, UI id, PCAJC2DVEC pe, BOOL fArrow)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pe != NULL) {
			if (id < AJC3DG_MAXITEM) {
				if (rc = SubG2dLineV(hwnd, id, &pW->V2dMoveTo[id], pe, fArrow)) {
					pW->V2dMoveTo[id].x = pe->x;
					pW->V2dMoveTo[id].y = pe->y;
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	平面描画 : 三角形描画																						//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xn, yn		- 各頂点位置																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dTriangle	   (HWND hwnd, UI id, double x1, double y1,
																	  double x2, double y2,
																	  double x3, double y3)
{
	AJC2DVEC	p1, p2, p3;
	p1.x = x1; p1.y = y1;;
	p2.x = x2; p2.y = y2;;
	p3.x = x3; p3.y = y3;;
	return AjcG2dTriangleV(hwnd, id, &p1, &p2, &p3);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dTriangleV	   (HWND hwnd, UI id, PCAJC2DVEC p1, PCAJC2DVEC p2, PCAJC2DVEC p3)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && p1 != NULL && p2 != NULL && p3 != NULL) {
			AJC2DVEC	v1, v2, v3;
			GI_TRIAN	tri;
			VOP			pNode;
	
			if (id < AJC3DG_MAXITEM) {
				tri.h.gic = GIC_TRIAN;
				tri.h.id  = id;
				tri.h.seq = pW->DatSeq++;
				memcpy(&v1, p1, sizeof v1);
				memcpy(&v2, p2, sizeof v2);
				memcpy(&v3, p3, sizeof v3);
				if (pW->f2dMode) {
					v1.x -= pW->prop.Cent.x; v2.x -= pW->prop.Cent.x; v3.x -= pW->prop.Cent.x;
					v1.y -= pW->prop.Cent.y; v2.y -= pW->prop.Cent.y; v3.y -= pW->prop.Cent.y;
				}
				G3dPlanePointTo3dPoint(pW, &v1, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &tri.p1.v3);
				G3dPlanePointTo3dPoint(pW, &v2, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &tri.p2.v3);
				G3dPlanePointTo3dPoint(pW, &v3, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &tri.p3.v3);
				G3dCnv3dTo2d (pW, &tri.p1.v3, &tri.p1.v2);
				G3dCnv3dTo2d (pW, &tri.p2.v3, &tri.p2.v2);
				G3dCnv3dTo2d (pW, &tri.p3.v3, &tri.p3.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &tri, sizeof tri)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, tri.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : 四角形描画																						//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xn, yn, zn	- 各頂点位置																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dSquare	   (HWND hwnd, UI id, double x1, double y1,
																	  double x2, double y2,
																	  double x3, double y3,
																	  double x4, double y4)
{
	AJC2DVEC	p1, p2, p3, p4;
	p1.x = x1; p1.y = y1;
	p2.x = x2; p2.y = y2;
	p3.x = x3; p3.y = y3;
	p4.x = x4; p4.y = y4;
	return AjcG2dSquareV(hwnd, id, &p1, &p2, &p3, &p4);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dSquareV	   (HWND hwnd, UI id, PCAJC2DVEC p1, PCAJC2DVEC p2, PCAJC2DVEC p3, PCAJC2DVEC p4)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && p1 != NULL && p2 != NULL && p3 != NULL && p4 != NULL) {
			AJC2DVEC	v1, v2, v3, v4;
			GI_SQUA		squa;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				squa.h.gic = GIC_SQUA;
				squa.h.id  = id;
				squa.h.seq = pW->DatSeq++;
				memcpy(&v1, p1, sizeof v1);
				memcpy(&v2, p2, sizeof v2);
				memcpy(&v3, p3, sizeof v3);
				memcpy(&v4, p4, sizeof v4);
				if (pW->f2dMode) {
					v1.x -= pW->prop.Cent.x; v2.x -= pW->prop.Cent.x; v3.x -= pW->prop.Cent.x; v4.x -= pW->prop.Cent.x;
					v1.y -= pW->prop.Cent.y; v2.y -= pW->prop.Cent.y; v3.y -= pW->prop.Cent.y; v4.y -= pW->prop.Cent.y;
				}
				G3dPlanePointTo3dPoint(pW, &v1, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p1.v3);
				G3dPlanePointTo3dPoint(pW, &v2, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p2.v3);
				G3dPlanePointTo3dPoint(pW, &v3, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p3.v3);
				G3dPlanePointTo3dPoint(pW, &v4, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p4.v3);
				G3dCnv3dTo2d(pW, &squa.p1.v3, &squa.p1.v2);
				G3dCnv3dTo2d(pW, &squa.p2.v3, &squa.p2.v2);
				G3dCnv3dTo2d(pW, &squa.p3.v3, &squa.p3.v2);
				G3dCnv3dTo2d(pW, &squa.p4.v3, &squa.p4.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &squa, sizeof squa)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, squa.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : 長方形描画																						//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x1, y1		- 対角位置																			//
//				x2, y2		- 〃																				//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dRectangle	   (HWND hwnd, UI id, double x1, double y1, double x2, double y2)
{
	AJC2DVEC	p1, p2;
	p1.x = x1; p1.y = y1;
	p2.x = x2; p2.y = y2;
	return AjcG2dRectangleV(hwnd, id, &p1, &p2);
}
//---------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dRectangleV   (HWND hwnd, UI id, PCAJC2DVEC p1, PCAJC2DVEC p2)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && p1 != NULL && p2 != NULL) {
			AJC2DVEC	v1, v2, v3, v4;
			GI_SQUA		squa;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				squa.h.gic = GIC_SQUA;
				squa.h.id  = id;
				squa.h.seq = pW->DatSeq++;
				v1.x = p1->x, v1.y = p1->y;
				v2.x = p1->x, v2.y = p2->y;
				v3.x = p2->x, v3.y = p2->y;
				v4.x = p2->x, v4.y = p1->y;
				if (pW->f2dMode) {
					v1.x -= pW->prop.Cent.x; v2.x -= pW->prop.Cent.x; v3.x -= pW->prop.Cent.x; v4.x -= pW->prop.Cent.x;
					v1.y -= pW->prop.Cent.y; v2.y -= pW->prop.Cent.y; v3.y -= pW->prop.Cent.y; v4.y -= pW->prop.Cent.y;
				}
				G3dPlanePointTo3dPoint(pW, &v1, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p1.v3);
				G3dPlanePointTo3dPoint(pW, &v2, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p2.v3);
				G3dPlanePointTo3dPoint(pW, &v3, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p3.v3);
				G3dPlanePointTo3dPoint(pW, &v4, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &squa.p4.v3);
				G3dCnv3dTo2d(pW, &squa.p1.v3, &squa.p1.v2);
				G3dCnv3dTo2d(pW, &squa.p2.v3, &squa.p2.v2);
				G3dCnv3dTo2d(pW, &squa.p3.v3, &squa.p3.v2);
				G3dCnv3dTo2d(pW, &squa.p4.v3, &squa.p4.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &squa, sizeof squa)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, squa.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : 楕円描画																							//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xc, yc		- 中心位置																			//
//				rx, ry		- 半径																				//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dEllipse	   (HWND hwnd, UI id, double xc, double yc, double rx, double ry)
{
	AJC2DVEC	pc;
	pc.x = xc; pc.y = yc;
	return AjcG2dEllipseV(hwnd, id, &pc, rx, ry);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dEllipseV	   (HWND hwnd, UI id, PCAJC2DVEC pCent, double rx, double ry)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pCent != NULL) {
			AJC2DVEC	vc;
			GI_ELPS		elps;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				elps.h.gic = GIC_ELPS;
				elps.h.id  = id;
				elps.h.seq = pW->DatSeq++;
				memcpy(&vc, pCent, sizeof vc);
				if (pW->f2dMode) {
					vc.x -= pW->prop.Cent.x;
					vc.y -= pW->prop.Cent.y;
				}
				G3dMakeEllipseData(pW, &vc, rx, ry, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &elps);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &elps, sizeof elps)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, elps.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : 星形																								//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				xc, yc		- 中心位置																			//
//				r			- 半径																				//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dStar	   (HWND hwnd, UI id, double xc, double yc, double r)
{
	AJC2DVEC	pc;
	pc.x = xc; pc.y = yc;
	return AjcG2dStarEx(hwnd, id, xc, yc, r, 0.0, 5, 0.0, FALSE);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dStarV	   (HWND hwnd, UI id, PCAJC2DVEC pCent, double r)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pCent != NULL) {
			AjcG2dStarVEx(hwnd, id, pCent, r, 0.0, 5, 0.0, FALSE);
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : 星形（拡張）																						//
//																												//
//	引　数	：	id				- データ項目番号																//
//				xc, yc			- 中心位置																		//
//				r1				- 外円半径																		//
//				r2				- 内円半径																		//
//				nVertex			- 頂点の数（５以上の奇数）														//
//				rot				- 回転角度［度」																//
//				fInscribedLine	- 内接Ｎ角形描画																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dStarEx   (HWND hwnd, UI id, double xc, double yc, double r1, double r2, UI nVertex, double rot, BOOL fInscribedLine)
{
	AJC2DVEC	pc;
	pc.x = xc; pc.y = yc;
	return AjcG2dStarVEx(hwnd, id, &pc, r1, r2, nVertex, rot, fInscribedLine);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dStarVEx	   (HWND hwnd, UI id, PCAJC2DVEC pCent, double r1, double r2, UI nVertex, double rot, BOOL fInscribedLine)
{
	BOOL		rc = FALSE;
	UI			i;
	double		t, step;
	UI			nBuf;
	PAJC2DVEC	pt = NULL;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pCent != NULL && r1 > 0.0 && r2 >= 0.0 && nVertex >= 5 && (nVertex & 1) != 0) {
			nBuf = nVertex * 2;
			step = 360.0 / nVertex;
			if (pt = (PAJC2DVEC)malloc(sizeof(AJC2DVEC) * nBuf)) {
				//	外円上の５点算出
				for (i = 0, t = 90.0 + rot;  i < nBuf; t += step, i += 2) {
					pt[i].x = r1 * AjcCos(t);
					pt[i].y = r1 * AjcSin(t);
					AjcV2dAdd(&pt[i], pCent, &pt[i]);
				}
				if (r2 == 0.0 && fInscribedLine) {
					//	星形描画
					AjcG2dMoveToV(hwnd, id, &pt[0]);
					for (i = 0; i < nVertex; i++) {
						AjcG2dLineToV(hwnd, id, &pt[((i + 1) * (nVertex / 2 * 2)) % (nBuf)]);
					}
				}
				else {
					//	内円の半径算出
					if (r2 == 0.0) {
						r2 = (r1 * AjcSin(step * 0.25)) * (1.0 / AjcSin(step * 0.75));
					}
					//	内円上の５点算出
					for (i = 1, t = 90.0 + (step * 0.5) + rot;	i < nBuf; t += step, i += 2) {
						pt[i].x = r2 * AjcCos(t);
						pt[i].y = r2 * AjcSin(t);
						AjcV2dAdd(&pt[i], pCent, &pt[i]);
					}
					//	星形描画
					AjcG2dMoveToV(hwnd, id, &pt[0]);
					for (i = 1; i < nBuf; i++) {
						AjcG2dLineToV(hwnd, id, &pt[i]);
					}
					AjcG2dLineToV(hwnd, id, &pt[0]);
					//	内接五角形描画
					if (fInscribedLine) {
						AjcG2dMoveToV(hwnd, id, &pt[1]);
						for (i = 3; i < nBuf; i += 2) {
							AjcG2dLineToV(hwnd, id, &pt[i]);
						}
						AjcG2dLineToV(hwnd, id, &pt[1]);
					}
				}
				free(pt);
				rc = TRUE;
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : 塗りつぶし（境界色内の塗りつぶし）																//
//																												//
//	引　数	：	idFill		- 塗りつぶし色のID																	//
//				idBorder	- 境界色のID																		//
//				x, y		- 塗りつぶしポイント																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dFillB  	   (HWND hwnd, UI idFill, UI idBorder, double x, double y)
{
	AJC2DVEC	pc;
	pc.x = x; pc.y = y;
	return AjcG2dFillBV(hwnd, idFill, idBorder, &pc);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dFillBV  	   (HWND hwnd, UI idFill, UI idBorder, PCAJC2DVEC pPoint)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pPoint != NULL) {
			UI			id = idFill;
			AJC2DVEC	v;
			GI_FILL		fil;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				fil.h.gic 	 = GIC_FILL;
				fil.h.id	 = id;
				fil.h.seq	 = pW->DatSeq++;
				fil.idBorder = idBorder;
				memcpy(&v, pPoint, sizeof v);
				if (pW->f2dMode) {
					v.x -= pW->prop.Cent.x;
					v.y -= pW->prop.Cent.y;
				}
				G3dPlanePointTo3dPoint(pW, &v, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &fil.p.v3);
				G3dCnv3dTo2d(pW, &fil.p.v3, &fil.p.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &fil, sizeof fil)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, fil.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : 塗りつぶし（白い部分の塗りつぶし）																//
//																												//
//	引　数	：	idFill		- 塗りつぶし色のID																	//
//				x y			- 塗りつぶしポイント																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG2dFillS  	   (HWND hwnd, UI idFill, double x, double y)
{
	AJC2DVEC	pc;
	pc.x = x; pc.y = y;
	return AjcG2dFillSV(hwnd, idFill, &pc);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG2dFillSV  	   (HWND hwnd, UI idFill, PCAJC2DVEC pPoint)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pPoint != NULL) {
			UI			id = idFill;
			AJC2DVEC	v;
			GI_FILL		fil;
			VOP			pNode;

			if (id < AJC3DG_MAXITEM) {
				fil.h.gic 	 = GIC_FILL;
				fil.h.id	 = id;
				fil.h.seq	 = pW->DatSeq++;
				fil.idBorder = -1;
				memcpy(&v, pPoint, sizeof v);
				if (pW->f2dMode) {
					v.x -= pW->prop.Cent.x;
					v.y -= pW->prop.Cent.y;
				}
				G3dPlanePointTo3dPoint(pW, &v, &pW->DatItem[id].PlnLVec, &pW->DatItem[id].PlnOrg, &fil.p.v3);
				G3dCnv3dTo2d(pW, &fil.p.v3, &fil.p.v2);
				if (rc = ((pNode = AjcVQueEnqueEx(pW->DatItem[id].hQueDat, &fil, sizeof fil)) != NULL)) {
					rc = AjcAvlInsNode(pW->hAvlDat, fil.h.seq, &pNode, sizeof(VOP));
				}
				pW->fReDrawData = TRUE;
				if (!pW->fPause) InvalidateRect(hwnd, NULL, FALSE);
			}
		}
	}
	return	rc;
}
//==============================================================================================================//
//	平面描画 : ピクセルの表示色取得																				//
//																												//
//	引　数	：	id			- データ項目番号																	//
//				x, y		- 位置																				//
//																												//
//	戻り値	：	≠－１：ピクセルの色																			//
//				＝－１：エラー																					//
//==============================================================================================================//
AJCEXPORT	COLORREF WINAPI		AjcG2dGetPixel		(HWND hwnd, double x, double y)
{
	AJC2DVEC	pc;
	pc.x = x; pc.y = y;
	return AjcG2dGetPixelV(hwnd, &pc);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	COLORREF WINAPI		AjcG2dGetPixelV		(HWND hwnd, PCAJC2DVEC pPoint)
{
	COLORREF	rc = -1;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pPoint != NULL) {
			AJC3DVEC	v3;
			AJC2DVEC	v2;
			POINT		pt;
			HDC			hdc;
	
			if (pW->f2dMode) {
				if (pPoint->x >= (pW->prop.Cent.x - pW->prop.xr) && pPoint->x <= (pW->prop.Cent.x + pW->prop.xr) && 
					pPoint->y >= (pW->prop.Cent.x - pW->prop.yr) && pPoint->y <= (pW->prop.Cent.x + pW->prop.yr)) {
					memcpy(&v2, pPoint, sizeof v2);
					v2.x -= pW->prop.Cent.x;
					v2.y -= pW->prop.Cent.y;
					G3dPlanePointTo3dPoint(pW, &v2, &pW->DatItem[0].PlnLVec, &pW->DatItem[0].PlnOrg, &v3);
					G3dCnv3dTo2d(pW, &v3, &pt);
					hdc = GetDC(hwnd);
					rc = GetPixel(hdc, pt.x, pt.y);
					ReleaseDC(hwnd, hdc);
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	右クリックによるポップアップメニューの許可／禁止															//
//																												//
//	引　数	：	hwnd		  - ウインドハンドル																//
//				fEnable		  - ポップアップメニューの許可／禁止												//
//									TRUE  : 許可（右クリック非通知，Shuft/Ctrl+右クリック通知）					//
//									FALSE : 禁止（全ての右クリック通知）										//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dEnablePopupMenu(HWND hwnd, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AjcG3dSetNtcRClk(hwnd, !fEnable, WM_RBUTTONDOWN, WM_RBUTTONUP);
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	右クリック通知設定																							//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fNtcRClk	 - TRUE  : 右ボタンDOWN/UP通知	（ポップアップは禁止）								//
//							   FALSE : 右ボタンDOWN/UP非通知（ポップアップを許可）								//
//				MsgRBDown	 - 右ボタン押下時の通知メッセージコード　（０の場合は非通知）						//
//				MSgRBUp		 - 右ボタン離され時の通知メッセージコード（０の場合は非通知）						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//																												//
//	備　考	：						<ポップアップ>	<Shift/Ctrl+右>		<右-DOWN>	<右-UP>						//
//				NtcRClk	= FALSE	 	 許可		 	 通知				 非通知		 非通知						//
//				 		= TRUE		 禁止			 非通知				 MsgRBDown	 MsgRBUp					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetNtcRClk	(HWND hwnd, BOOL fNtcRClk, UI MsgRBDown, UI MsgRBUp)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fNtcRClk  = fNtcRClk;
			pW->MsgRBDown = MsgRBDown;
			pW->MsgRBUp	  = MsgRBUp;
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ツールチップ・テキスト設定																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pTxt		 - ツールチップ・テキスト文字列のアドレス											//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetTipTextA (HWND hwnd, C_BCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = SubSetTipTextA(&pW->pToolTipTxt, 0, pTxt)) {
				AjcTipTextAddW		   (hwnd	 , pW->pToolTipTxt);
				AjcTipTextSetShowAlways(hwnd	 , pW->fShowAlways);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetTipTextW (HWND hwnd, C_WCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = SubSetTipTextW(&pW->pToolTipTxt, 0, pTxt)) {
				AjcTipTextAddW		   (hwnd	 , pW->pToolTipTxt);
				AjcTipTextSetShowAlways(hwnd	 , pW->fShowAlways);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ツールチップ・テキスト取得																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pBuf		 - ツールチップ・テキスト文字列を格納するバッファのアドレス							//
//				lBuf		 - ツールチップ・テキスト文字列を格納するバッファの文字数							//
//																												//
//	戻り値	：	チップテキストのバイト数／文字数（pBuf=NULLの場合は、文字列終端を含む）							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcG3dGetTipTextA (HWND hwnd, BCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = SubGetTipTextA(&pW->pToolTipTxt, 0, pBuf, lBuf);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI	AjcG3dGetTipTextW (HWND hwnd, WCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = SubGetTipTextW(&pW->pToolTipTxt, 0, pBuf, lBuf);
		}
	}
	return rc;
}

//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト設定													//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//				pTxt		 - ツールチップ・テキスト文字列のアドレス											//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetChkBoxTipTextA(HWND hwnd, UI n, C_BCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM) {
				if (rc = SubSetTipTextA(pW->pChkBoxTool, n, pTxt)) {
					AjcTipTextAddW		   (pW->hChkFilt[n], pW->pChkBoxTool[n]);
					AjcTipTextSetShowAlways(pW->hChkFilt[n], pW->fChkBoxShowAlways[n]);
				}
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcG3dSetChkBoxTipTextW(HWND hwnd, UI n, C_WCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM) {
				if (rc = SubSetTipTextW(pW->pChkBoxTool, n, pTxt)) {
					AjcTipTextAddW		   (pW->hChkFilt[n], pW->pChkBoxTool[n]);
					AjcTipTextSetShowAlways(pW->hChkFilt[n], pW->fChkBoxShowAlways[n]);
				}
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト取得													//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//				pBuf		 - ツールチップ・テキスト文字列を格納するバッファのアドレス							//
//				lBuf		 - ツールチップ・テキスト文字列を格納するバッファの文字数							//
//																												//
//	戻り値	：	チップテキストのバイト数／文字数（pBuf=NULLの場合は文字列終端を含む）							//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI	WINAPI		AjcG3dGetChkBoxTipTextA(HWND hwnd, UI n, BCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM) {
				rc = SubGetTipTextA(pW->pChkBoxTool, n, pBuf, lBuf);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcG3dGetChkBoxTipTextW(HWND hwnd, UI n, WCP pBuf, UI lBuf)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM) {
				rc = SubGetTipTextW(pW->pChkBoxTool, n, pBuf, lBuf);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ツールチップ・テキスト表示条件設定																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fShowAlways	 - ツールチップ・テキスト表示条件													//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetTipShowAlways(HWND hwnd, BOOL fShowAlways)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fShowAlways = fShowAlways;
			AjcTipTextSetShowAlways(hwnd, pW->fShowAlways);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ツールチップ・テキスト表示条件取得																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	ツールチップ・テキスト表示条件																	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dGetTipShowAlways(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = pW->fShowAlways;
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト表示条件設定											//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//				fShowAlways	 - ツールチップ・テキスト表示条件													//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetChkBoxTipShowAlways(HWND hwnd, UI n, BOOL fShowAlways)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM) {
				pW->fChkBoxShowAlways[n] = fShowAlways;
				AjcTipTextSetShowAlways(pW->hChkFilt[n], pW->fChkBoxShowAlways[n]);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト表示条件取得											//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//																												//
//	戻り値	：	ツールチップ・テキスト表示条件																	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dGetChkBoxTipShowAlways(HWND hwnd, UI n)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM) {
				rc = pW->fChkBoxShowAlways[n];
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	全てのツールチップ・テキスト表示条件設定																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fShowAlways	 - ツールチップ・テキスト表示条件													//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dSetTipShowAlwaysAll(HWND hwnd, BOOL fShowAlways)
{
	BOOL	rc = FALSE;
	int		i;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fShowAlways = fShowAlways;
			AjcTipTextSetShowAlways(hwnd, pW->fShowAlways);
			for (i = 0; i < AJCTC_MAXITEM; i++) {
				pW->fChkBoxShowAlways[i] = fShowAlways;
				AjcTipTextSetShowAlways(pW->hChkFilt[i], pW->fChkBoxShowAlways[i]);
			}
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	フィルタの設定																								//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//				state		 - 設定する状態(FALSE:非表示，TRUE:表示)											//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcG3dSetFilter		(HWND hwnd, UI n, BOOL state)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM	&&	pW->hChkFilt[n] != NULL) {
				SendMessage(pW->hChkFilt[n], BM_SETCHECK, state, 0);
				SendMessage(pW->hFilter, WM_COMMAND, MAKELONG(G3D_IDC_CHK_BASE + n, BN_CLICKED), 0);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィルタの取得																								//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//																												//
//	戻り値	：	フィルタの設定状態																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcG3dGetFilter		(HWND hwnd, UI n)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH	pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJC3DG_MAXITEM	&&	pW->hChkFilt[n] != NULL) {
				rc = (BOOL)SendMessage(pW->hChkFilt[n], BM_GETCHECK, 0, 0);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたファイル名取得																				//
//																												//
//	引　数	：	pW	  - インスタンスワークアドレス																//
//				buf	  - ファイル名を格納するバッファのアドレス													//
//																												//
//	戻り値	：	TRUE:ファイル名あり，FALSE:ファイル名なし														//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcG3dGetDroppedFileA (HWND hwnd, BC buf[MAX_PATH])
{
	BOOL	rc = FALSE;
	WC		tmp[MAX_PATH];

	if (buf != NULL) {
		if (rc = AjcG3dGetDroppedFileW(hwnd, tmp)) {
			WideCharToMultiByte(CP_ACP, 0, tmp, -1, buf, MAX_PATH, NULL, NULL);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcG3dGetDroppedFileW (HWND hwnd, WC buf[MAX_PATH])
{
	BOOL		rc = FALSE;
	UI			len;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && buf != NULL) {
			if (AjcRngGetData(pW->hRngFile, (VOP)&len, sizeof len) == sizeof len) {
				rc = (AjcRngGetData(pW->hRngFile, (VOP)buf, len) == len);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたディレクトリ名取得																			//
//																												//
//	引　数	：	pW	  - インスタンスワークアドレス																//
//				buf	  - ディレクトリ名を格納するバッファのアドレス												//
//																												//
//	戻り値	：	TRUE:ディレクトリ名あり，FALSE:ディレクトリ名なし												//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcG3dGetDroppedDirA  (HWND hwnd, BC buf[MAX_PATH])
{
	BOOL	rc = FALSE;
	WC		tmp[MAX_PATH];

	if (buf != NULL) {
		if (rc = AjcG3dGetDroppedDirW(hwnd, tmp)) {
			WideCharToMultiByte(CP_ACP, 0, tmp, -1, buf, MAX_PATH, NULL, NULL);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcG3dGetDroppedDirW  (HWND hwnd, WC buf[MAX_PATH])
{
	BOOL		rc = FALSE;
	UI			len;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && buf != NULL) {
			if (AjcRngGetData(pW->hRngDir, (PWRK3DGRAPH)&len, sizeof len) == sizeof len) {
				rc = (AjcRngGetData(pW->hRngDir, (VOP)buf, len) == len);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたディレクトリ名取得（末尾の「￥」付加指定付き）												//
//																												//
//	引　数	：	pW	  			 - インスタンスワークアドレス													//
//				buf	  			 - ディレクトリ名を格納するバッファのアドレス									//
//				fTailIsDelimiter - TRUE  : ディレクトリパスの末尾に「\」を付ける								//
//																												//
//	戻り値	：	TRUE:ディレクトリ名あり，FALSE:ディレクトリ名なし												//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcG3dGetDroppedDirExA	(HWND hwnd, BC buf[MAX_PATH], BOOL fTailIsDelimiter)
{
	BOOL	rc = FALSE;
	WC		tmp[MAX_PATH];

	if (buf != NULL) {
		if (rc = AjcG3dGetDroppedDirExW(hwnd, tmp, fTailIsDelimiter)) {
			WideCharToMultiByte(CP_ACP, 0, tmp, -1, buf, MAX_PATH, NULL, NULL);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcG3dGetDroppedDirExW	(HWND hwnd, WC buf[MAX_PATH], BOOL fTailIsDelimiter)
{
	BOOL		rc = FALSE;

	if (rc = AjcG3dGetDroppedDirW(hwnd, buf)) {
		if (fTailIsDelimiter) AjcPathCatW(buf, L"", MAX_PATH);
		else				  PathRemoveBackslashW(buf);
	}
	return rc;
}
//==============================================================================================================//
//	タイトル文字列の設定																						//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pTitleText	- タイトルテキストへのポインタ（NULL指定時は、タイトル非表示）						//
//				TextColor	- テキスト描画色（－１指定時は前回設定値，デフォルト色＝白）						//
//				BackColor	- テキスト背景色（－１指定時は前回設定値，デフォルト色＝グレー）					//
//				hFont		- タイトルテキストのフォント（NULL指定時は前回設定値／デフォルトのフォント）		//
//																												//
//	戻り値	：	ウインド先頭行の位置（０～）																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	   WINAPI AjcG3dSetTitleTextA(HWND hwnd, C_BCP pTitleText, COLORREF TextColor, COLORREF BackColor, HFONT hFont)
{
	BOOL		rc	 = FALSE;
	WCP			pTmp = NULL;
	UL			len;

	if (IsWindow(hwnd)) {
		do {
			if (pTitleText != NULL) {
				len = MultiByteToWideChar(CP_ACP, 0, pTitleText, -1, NULL, 0);
				if (len != 0  &&  (pTmp = AjcTAllocW(len))) {
					MultiByteToWideChar(CP_ACP, 0, pTitleText, -1, pTmp, len);
				}
				else break;
			}
			else pTmp = NULL;
			rc = AjcG3dSetTitleTextW(hwnd, pTmp, TextColor, BackColor, hFont);
		} while(0);
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	   WINAPI AjcG3dSetTitleTextW(HWND hwnd, C_WCP pTitleText, COLORREF TextColor, COLORREF BackColor, HFONT hFont)
{
	BOOL	rc	 = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			//	タイトルテキスト・ビットマップ生成
			SubTtlBmpMake(&pW->TtlInfo, pW->hMain, pTitleText, TextColor, BackColor, hFont);
			//	戻り値＝正常
			rc = TRUE;
			//	再描画
			InvalidateRect(hwnd, NULL, TRUE);
		}
	}
	return rc;
}
//==============================================================================================================//
//	表示の停止／再開																							//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				fPause		- 表示停止フラグ（TRUE:表示停止，FALSE:表示開始）									//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcG3dPause(HWND hwnd, BOOL fPause)
{
	BOOL		rc	 = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (!(pW->fPause = fPause)) {						//	表示再開？
				InvalidateRect(hwnd, NULL, FALSE);				//		ＷＮＤ無効化（描画）
			}
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	描画時間計測情報の許可／禁止																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				fEnable		- 許可／禁止フラグ（TRUE:許可，FALSE:禁止）											//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcG3dEnableMesDraw(HWND hwnd, BOOL fEnable)
{
	BOOL		rc	 = FALSE;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fMesDrawEnable = fEnable;
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	メインウインドのハンドル取得																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pBkWnd		- ウインド背景色を格納するバッファへのポインタ（不要時はNULL）						//
//																												//
//	戻り値	：	≠NULL : メインウインドのハンドル																//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT	HWND	WINAPI		AjcG3dGetMainWindow(HWND hwnd, COLORREF *pBkGnd)
{
	HWND	rc = NULL;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (pBkGnd != NULL) {
				*pBkGnd = RGB(255, 255, 255);
			}
			rc = pW->hMain;
		}
	}
	return rc;
}
//==============================================================================================================//
//																												//
//	通知情報取得																								//
//																												//
//==============================================================================================================//
//	右クリック通知情報
AJCEXPORT	PAJC3DGRCLK		WINAPI		AjcG3dGetNtcRClk	(HWND hwnd)
{
	PAJC3DGRCLK	rc = NULL;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		rc = &pW->NtcRClk;
	}
	return rc;
}
//	ドロップしたディレクトリ数
AJCEXPORT	UI				WINAPI		AjcG3dGetNtcDirs	(HWND hwnd)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		rc = pW->NtcDirs;
	}
	return rc;
}
//	ドロップしたファイル数
AJCEXPORT	UI				WINAPI		AjcG3dGetNtcFiles	(HWND hwnd)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		rc = pW->NtcFiles;
	}
	return rc;
}
//	プロットリスト情報
AJCEXPORT	PAJC3DGPLOTLIST	WINAPI		AjcG3dGetNtcList	(HWND hwnd)
{
	PAJC3DGPLOTLIST	rc = NULL;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		rc = &pW->PlotList;
	}
	return rc;
}
//	視点ベクトル
AJCEXPORT	PAJC3DVEC	WINAPI		AjcG3dGetNtcRotTheta(HWND hwnd)
{
	PAJC3DVEC	rc = NULL;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		rc = &pW->NtcRotTheta;
	}
	return rc;
}


//	クリアー要因
AJCEXPORT	UI	WINAPI		AjcG3dGetNtcClrFact(HWND hwnd)
{
	UI	rc = 0;

	if (IsWindow(hwnd)) {
		PWRK3DGRAPH pW = (PWRK3DGRAPH)MAjcGetWindowLong(hwnd, 0);
		rc = pW->NtcClrFact;
	}
	return rc;
}
