﻿#include	"AjcInternal.h"
#include	"AjcCtrlTimeChartDef.h"
//**************************************************************************************************************//
//																												//
//	カスタムコントロール（タイムチャート）				ＡＰＩ													//
//																												//
//**************************************************************************************************************//

#define	IS_MY_INST(P)	(P != NULL && P->InstID == TCH_INST_ID)

//--------------------------------------------------------------------------------------------------------------//
//	グローバルワーク																							//
//--------------------------------------------------------------------------------------------------------------//
ATOM	TchClassBack	= 0;			//	バックウインドクラス
ATOM	TchClassMain	= 0;			//	メインウインドクラス
ATOM	TchClassFilter	= 0;			//	フィルタウインドクラス
ATOM	TchClassMeasure	= 0;			//	メジャーウインドクラス
ATOM	TchClassForMSIL	= 0;			//	ＭＳＩＬサポート
HFONT	TchMyFont 		= NULL;			//	フォントハンドル（スケール値表示用）

ULL		TchPfmFreq		= 0;			//	パフォーマンスカウンタの周波数（０：パフォーマンスカウンタ不可）
HPEN	TchPenGauBar	= NULL;			//	時間測定ゲージバーのペン

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//
static	ULL		GetUsCount		  (VO);

//==============================================================================================================//
//	起動時初期設定																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
BOOL	AjcCtrlTmChartInit(VO)
{
	BOOL		rc = FALSE;
	WNDCLASS	wndclass;
	LOGFONT		lf;
	LARGE_INTEGER freq;

	do {
		//----- パフォーマンスカウンタ周波数取得 --------------//
		if (QueryPerformanceFrequency(&freq) != 0) {
			TchPfmFreq = freq.QuadPart;
		}
		//----- フォント --------------------------------------//
		lf.lfHeight			= -9;	lf.lfStrikeOut		= 0;
		lf.lfWidth			= 0;	lf.lfCharSet		= DEFAULT_CHARSET;
		lf.lfEscapement		= 0;	lf.lfOutPrecision	= 0;
		lf.lfOrientation	= 0;	lf.lfClipPrecision	= 0;
		lf.lfWeight			= 0;	lf.lfQuality		= 0;
		lf.lfItalic			= 0;	lf.lfPitchAndFamily	= VARIABLE_PITCH;
		lf.lfUnderline		= 0;	wcscpy(lf.lfFaceName, L"Arial");
		if ((TchMyFont = CreateFontIndirect(&lf)) == NULL) break;

		//----- 時間測定用ゲージバー ペン生成 ----------------//
		if ((TchPenGauBar = CreatePen(PS_DOT, 1, TCH_GAUGE_COLOR)) == NULL) break;

		//----- カスタムコントロール・クラス -----------------//
		wndclass.style			= CS_HREDRAW | CS_VREDRAW | CS_GLOBALCLASS;
		wndclass.lpfnWndProc	= TchWndProcBack();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(PWRKTMCHART);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"AjcCtrlTmChart";
		if ((TchClassBack = RegisterClass(&wndclass)) == 0) break;

		//----- メイン描画ウインドクラス ---------------------//
		wndclass.style			= CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
		wndclass.lpfnWndProc	= TchWndProcMain();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(PWRKTMCHART);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);;
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"AjcTmChartMainClass";
		if ((TchClassMain = RegisterClass(&wndclass)) == 0) break;

		//----- フィルタ・ウインドクラス ----------------------//
		wndclass.style			= 0;
		wndclass.lpfnWndProc	= TchWndProcFilt();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(PWRKTMCHART);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);;
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(LTGRAY_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"AjcTmChartFilterClass";
		if ((TchClassFilter = RegisterClass(&wndclass)) == 0) break;

		//----- ＭＳＩＬコードサポート用 ---------------------//
		wndclass.style			= 0;
		wndclass.lpfnWndProc	= TchWndProcMSIL();
		wndclass.cbClsExtra		= 0;
		wndclass.cbWndExtra		= sizeof(VOP);
		wndclass.hInstance		= hDllInst;
		wndclass.hIcon			= NULL;
		wndclass.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wndclass.hbrBackground	= (HBRUSH)GetStockObject(WHITE_BRUSH);
		wndclass.lpszMenuName	= NULL;
		wndclass.lpszClassName	= L"AjcTchForMSIL";
		if ((TchClassForMSIL = RegisterClass(&wndclass)) == 0) break;


		rc = TRUE;
	} while(0);

	if (rc == FALSE) {
		AjcCtrlTmChartEnd();
	}

	return rc;
}
//==============================================================================================================//
//	終了時後処理																								//
//																												//
//	引　数	：	なし																							//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - エラー																					//
//==============================================================================================================//
VO		AjcCtrlTmChartEnd (VO)
{
	HINSTANCE	hModInst = GetModuleHandle(NULL);

	if (TchClassBack	!= 0	) {UnregisterClass((WCP)TchClassBack   , hDllInst); TchClassBack	= 0;   }
	if (TchClassMain	!= 0	) {UnregisterClass((WCP)TchClassMain   , hDllInst); TchClassMain	= 0;   }
	if (TchClassFilter	!= 0	) {UnregisterClass((WCP)TchClassFilter , hDllInst); TchClassFilter	= 0;   }
	if (TchClassMeasure	!= 0	) {UnregisterClass((WCP)TchClassMeasure, hDllInst); TchClassMeasure	= 0;   }
	if (TchClassForMSIL != 0	) {UnregisterClass((WCP)TchClassForMSIL, hDllInst); TchClassForMSIL = 0;   }
	if (TchMyFont 		!= NULL ) {DeleteObject   (TchMyFont					 ); TchMyFont 		= NULL;}
	if (TchPenGauBar 	!= NULL ) {DeleteObject   (TchPenGauBar					 ); TchPenGauBar	= NULL;}
}


//==============================================================================================================//
//	停止																										//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchStop			(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fStop = TRUE;		//	プロット停止
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	開始																										//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchStart			(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fStop = FALSE;		//	プロット開始
			TchMesPeriReset(pW);	//	プロット周期計測リセット
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロットデータ消去																							//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchPurge	(HWND hwnd)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			//----- バッファクリアー ------------//
			pW->ixView = 0;
			pW->nView  = 0;
			pW->nBuf   = 0;
			pW->nTotal = 0;
			pW->ixBuf  = 0;
			//----- 再表示 ----------------------//
			InvalidateRect(pW->hMain, NULL, FALSE);
			//----- バッファクリアー通知 --------//
			SendMessage(GetParent(pW->hBack), WM_COMMAND, MAKELONG(MAjcGetWindowLong(pW->hBack, GWL_ID), AJCTCN_CLEAR), (LPARAM)pW->hBack);
		}
	}
	return rc;
}
//==============================================================================================================//
//	実数データ設定																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				dat		- データ配列のアドレス																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchPutRealData	(HWND hwnd, double dat[])
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && dat != NULL) {
			double			*pDat = dat;
			double			*pBuf = pW->pBuf + (pW->ixBuf * pW->prop.MaxItem);
			PTCH_VLINEBUF	 pVLi = pW->pVLi + (pW->ixBuf					);
			SCROLLINFO		si;
			UI				i, j, ix, nAve;

			//----- プロット周期計測 ----------------------//
			if (pW->MesPeriCnt == 0) {
				pW->MesPeriSrt = GetUsCount();
			}
			else {
				ULL		tick	= GetUsCount();
				ULL		elapsed = tick - pW->MesPeriSrt;
				pW->MesPeriAve	= (UI)(elapsed / pW->MesPeriCnt);
			}
			pW->MesPeriCnt++;		//	プロット数カウンタ

			//----- プロット周期平均値を反映 --------------//
			if (pW->MesPeriCnt == pW->MesPeriSetAveS) {
				pW->MesPeriAveS = pW->MesPeriAve;
			}

			//----- 移動平均処理 --------------------------//
			//	移動平均無し（平均化個数＝１）
			if (pW->prop.AveNum <= 1) {
				for (i = 0; i < pW->prop.MaxItem; i++) {
					pW->AveDat[i] = *pDat++;
				}
			}
			//	移動平均有り
			else {
				//	移動平均する個数設定
				if (pW->nBuf < pW->prop.MaxBuf) nAve = pW->nBuf + 1;
				else							nAve = pW->nBuf;
				nAve = __min(nAve, pW->prop.AveNum);
				//	加算バッファクリアー
				memset(pW->AveDat, 0, sizeof pW->AveDat);
				//	直前の平均個数－１のデータを加算
				if (nAve >= 2) {
					for (i = 0; i < pW->prop.MaxItem; i++) {
						for (j = 0, ix = (pW->ixBuf == 0 ? pW->prop.MaxBuf - 1 : pW->ixBuf - 1);
							 j < nAve - 1;
							 j++  , ix = (ix		== 0 ? pW->prop.MaxBuf - 1 : ix 	   - 1)) {
							pW->AveDat[i] += *(pW->pBuf + ((ix * pW->prop.MaxItem) + i));
						}
					}
				}
				//	現データを加算＆平均化
				for (i = 0; i < pW->prop.MaxItem; i++) {
					pW->AveDat[i] += *pDat++;
					pW->AveDat[i] /= (double)nAve;
				}
			}
			//----- データ設定 ----------------------------//
			if (!pW->fStop) {													//	停止中以外？
				pW->nTotal++;													//		通算データ数更新
				memcpy(pBuf, pW->AveDat, sizeof(double) * pW->prop.MaxItem);	//		バッファへデータ設定
				memset(pVLi, 0, sizeof(TCH_VLINEBUF));							//		縦ライン情報クリアー
				if (pW->nBuf < pW->prop.MaxBuf) pW->nBuf++;						//		バッファデータ数更新
				pW->ixBuf = (pW->ixBuf + 1) % pW->prop.MaxBuf;					//		バッファインデクス更新

				if (pW->fPause) {												//		表示停止中？
					TchSetViewIndex(pW);										//			表示開始位置設定
					TchSetViewWidth(pW);										//			表示データ数設定
				}
				else {															//		表示停止中以外？
					si.cbSize = sizeof(SCROLLINFO);								//				
					si.fMask  = SIF_POS;										//				
					if (!pW->fScrBar) {											//			スクロールバー非操作中？
						TchSetViewIndex(pW);									//				表示開始位置設定
						si.nPos = TchBixToDix(pW, pW->ixView);					//				スクロールバー更新
						SetScrollInfo(pW->hMain, SB_HORZ, &si, TRUE);			//				・
					}
					else {														//			スクロールバー操作中？
						GetScrollInfo(pW->hMain, SB_HORZ, &si);					//				表示開始位置を、
						pW->ixView = TchDixToBix(pW, si.nPos);					//				スクロールバー位置とする
					}
					TchSetViewWidth(pW);										//			表示データ数設定
					InvalidateRect(pW->hMain, NULL, FALSE);						//			再描画
				}
			}
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	整数データ設定																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				dat		- データ配列のアドレス																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchPutIntData   (HWND hwnd, int dat[])
{
	BOOL		rc = FALSE;
	UI			i;
	double		rdat[AJCTC_MAXITEM];

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && dat != NULL) {
			for (i=0; i<pW->prop.MaxItem; i++) {
				rdat[i] = (double)dat[i];
			}
			rc = AjcTchPutRealData(hwnd, rdat);
		}
	}
	return rc;
}

//==============================================================================================================//
//	外枠表示																									//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				fShow	- TRUE : 表示，FALSE : 非表示															//
//				rgb		- 表示色（－１：変更なし）																//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchShowBorder	(HWND hwnd, BOOL fShow, COLORREF rgb)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int		sty;
			if (rgb != -1) {
				TchSetBorderColor(pW, rgb);
			}
			sty = (int)MAjcGetWindowLong(hwnd, GWL_STYLE);
			if (fShow) sty &= ~AJCTCS_NOBORDER;
			else	   sty |=  AJCTCS_NOBORDER;
			MAjcSetWindowLong(hwnd, GWL_STYLE, sty);
			InvalidateRect(hwnd, NULL, FALSE);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィルタ表示																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				fShow	- TRUE : 表示，FALSE : 非表示															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchShowFilter	(HWND hwnd, BOOL fShow)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int		sty;
			sty = (int)MAjcGetWindowLong(hwnd, GWL_STYLE);
			if (fShow) sty &= ~AJCTCS_NOFILTER;
			else	   sty |=  AJCTCS_NOFILTER;
			MAjcSetWindowLong(hwnd, GWL_STYLE, sty);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	スケール表示																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				fLine	- TRUE : スケールライン表示，FALSE : 非表示												//
//				fValue	- TRUE : スケール値表示	   ，FALSE : 非表示												//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchShowScale(HWND hwnd, BOOL fLine, BOOL fValue)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int		sty;
			sty = (int)MAjcGetWindowLong(hwnd, GWL_STYLE);
			if (fLine)	sty &= ~AJCTCS_NOSCALELINE;
			else		sty |=	AJCTCS_NOSCALELINE;
			if (fValue) sty &= ~AJCTCS_NOSCALEVALUE;
			else		sty |=	AJCTCS_NOSCALEVALUE;
			MAjcSetWindowLong(hwnd, GWL_STYLE, sty);
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	プロパティ取得																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				pBuf	- 取得したプロパティを格納するバッファのアドレス										//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchGetProp		(HWND hwnd, PAJCTCPROP	pBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pBuf != NULL) {
			rc = TchGetProp(pW, pBuf);
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロパティ設定																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				pProp	- 設定するプロパティデータのアドレス													//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetProp		(HWND hwnd, PCAJCTCPROP pProp)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProp != NULL) {
			rc = TchSetProp(pW, pProp);
		}
	}
	return rc;
}

//==============================================================================================================//
//	実数レンジ取得																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				low		- グラフ下部レンジを格納するバッファのアドレス											//
//				high	- グラフ上部レンジを格納するバッファのアドレス											//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchGetRealRange	(HWND hwnd, double *low, double *high)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = TchGetProp(pW, &prop)) {
				if (low  != NULL) *low	= prop.RngL;
				if (high != NULL) *high = prop.RngH;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	実数レンジ設定																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				low		- グラフ下部レンジ																		//
//				high	- グラフ上部レンジ																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetRealRange	(HWND hwnd, double low, double high)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			memcpy(&prop, &pW->prop, sizeof(AJCTCPROP));
			prop.RngL = low;
			prop.RngH = high;
			rc = TchSetProp(pW, &prop);
		}
	}
	return rc;
}
//==============================================================================================================//
//	整数レンジ取得																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				low		- グラフ下部レンジを格納するバッファのアドレス											//
//				high	- グラフ上部レンジを格納するバッファのアドレス											//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchGetIntRange  (HWND hwnd, int *low, int *high)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = TchGetProp(pW, &prop)) {
				if (low  != NULL) *low	= (int)prop.RngL;
				if (high != NULL) *high = (int)prop.RngL;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	整数レンジ設定																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				low		- グラフ下部レンジ																		//
//				high	- グラフ上部レンジ																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetIntRange	(HWND hwnd, int	   low, int	   high)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			double	l = (double)low;
			double	h = (double)high;
			rc = AjcTchSetRealRange(hwnd, l, h);
		}
	}
	return rc;
}
//==============================================================================================================//
//	レンジ自動調整																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchAdjustRange	(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			UI				n = pW->nBuf * pW->prop.MaxItem;
			double			min =  DBL_MAX;
			double			max = -DBL_MAX;
			double			ofs;
			double			*p;
			AJCTCPROP		prop;
			UI				i, msk;
			double			SvRngL = pW->prop.RngL;
			double			SvRngH = pW->prop.RngH;

			memcpy(&prop, &pW->prop, sizeof(AJCTCPROP));

			if (pW->nBuf != 0  &&  (pW->MskFilt & ((1 << pW->prop.MaxItem) - 1)) != 0) {
				//----- 最小値／最大値検索 ------------//
				for (i=0, msk=0x01, p=pW->pBuf; i<n; i++, msk<<=1, p++) {
					if (msk & (1 << pW->prop.MaxItem)) {
						msk = 0x01;
					}
					if (pW->MskFilt & msk) {
						ofs = pW->prop.Item[i % pW->prop.MaxItem].ofs;
						min = __min(min, (*p + ofs));
						max = __max(max, (*p + ofs));
					}
				}
				//----- マージン設定 ------------------//
				if (min == max) {
					min--;
					max++;
				}
				else {
					min -= ((max - min) * 0.05);
					max += ((max - min) * 0.05);
				}
				//----- レンジ設定 --------------------//
				if (prop.RngL <= prop.RngH) {
					prop.RngL = min;
					prop.RngH = max;
				}
				else {
					prop.RngL = max;
					prop.RngH = min;
				}
				//----- レンジ変化通知 -----------------//
				if (SvRngL != pW->prop.RngL || SvRngH != pW->prop.RngH) {
					pW->NtcRng.RngL = pW->prop.RngL;
					pW->NtcRng.RngH = pW->prop.RngH;
					SendMessage(GetParent(pW->hBack), WM_COMMAND,
								MAKELONG(MAjcGetWindowLong(pW->hBack, GWL_ID), AJCTCN_RANGE),
								fCmdWithHdl ? (LPARAM)pW->hBack : (LPARAM)&pW->NtcRng);
				}
				rc = TchSetProp(pW, &prop);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	バッファサイズ設定																							//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				n		- バッファのサイズを格納するデータ数で指定												//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetBufSize	(HWND hwnd, int n)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			memcpy(&prop, &pW->prop, sizeof(AJCTCPROP));
			prop.MaxBuf = n;
			rc = TchSetProp(pW, &prop);
		}
	}
	return rc;
}
//==============================================================================================================//
//	データ項目数設定																							//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				n		- 設定するデータ項目数																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetItemNumber (HWND hwnd, int n)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			memcpy(&prop, &pW->prop, sizeof(AJCTCPROP));
			prop.MaxItem = n;
			rc = TchSetProp(pW, &prop);
		}
	}
	return rc;
}
//==============================================================================================================//
//	平均化個数設定																								//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				n		- 設定するデータ項目数																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetAveNumber (HWND hwnd, int n)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			memcpy(&prop, &pW->prop, sizeof(AJCTCPROP));
			prop.AveNum = n;
			rc = TchSetProp(pW, &prop);
		}
	}
	return rc;
}
//==============================================================================================================//
//	タイムスケール幅設定																						//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//				n		- 設定するタイムスケール幅（０はタイムスケール非表示）									//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetTimeScale(HWND hwnd, int n)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			memcpy(&prop, &pW->prop, sizeof(AJCTCPROP));
			prop.TmScale = n;
			rc = TchSetProp(pW, &prop);
		}
	}
	return rc;
}
//==============================================================================================================//
//	ビットマップ取得																							//
//																												//
//	引　数	：	hwnd	- ウインドハンドル																		//
//																												//
//	戻り値	：	≠NULL：ビットマップハンドル																	//
//				＝NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT	HBITMAP		WINAPI	AjcTchGetBitmap		(HWND hwnd)
{
	HBITMAP	hBm = NULL;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			hBm = CopyImage(pW->hBitmap, IMAGE_BITMAP, pW->ww, pW->wh, LR_COPYRETURNORG);
		}
	}
	return hBm;
}
//==============================================================================================================//
//	プロファイルから永続情報読み出し（標準）																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPermInfoA		(HWND hwnd, C_BCP pProfileSect, C_BCP pKeyPrefix)
{
	BOOL		rc = TRUE;

	rc = AjcTchLoadPermInfoExA(hwnd, pProfileSect, pKeyPrefix, AJCTCH_PERM_FILTER);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPermInfoW		(HWND hwnd, C_WCP pProfileSect, C_WCP pKeyPrefix)
{
	BOOL		rc = TRUE;

	rc = AjcTchLoadPermInfoExW(hwnd, pProfileSect, pKeyPrefix, AJCTCH_PERM_FILTER);

	return rc;
}
//==============================================================================================================//
//	プロファイルから永続情報読み出し（項目指定）																//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名  （不要時はNULL)										//
//				pKeyPrefix	 - キー名称先頭部分へのポインタ（不要時はNULL)										//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
#define			MKKEY(TAIL)		MkKey(pKeyPrefix, TAIL, szKey, AJCTSIZE(szKey))
static	C_WCP	MkKey(C_WCP pKeyPrefix, WCP pTail, WCP pKey, UI lKey)
{
	AjcSnPrintF(pKey, lKey, L"%s_%s", pKeyPrefix, pTail);
	return	pKey;
}
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPermInfoExA		(HWND hwnd, C_BCP pProfileSect, C_BCP pKeyPrefix, UI PermItem)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmpSec = NULL;
	WCP		pTmpKey = NULL;

	do {
		if (pProfileSect != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
			if (len != 0 && (pTmpSec = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmpSec, len);
			}
			else break;
		}
		if (pKeyPrefix != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pKeyPrefix, -1, NULL, 0);
			if (len != 0 && (pTmpKey = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pKeyPrefix, -1, pTmpKey, len);
			}
			else break;
		}
		rc = AjcTchLoadPermInfoExW(hwnd, pTmpSec, pTmpKey, PermItem);
	} while(0);

	if (pTmpSec != NULL) free(pTmpSec);
	if (pTmpKey != NULL) free(pTmpKey);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPermInfoExW		(HWND hwnd, C_WCP pProfileSect, C_WCP pKeyPrefix, UI PermItem)
{
	BOOL		rc = FALSE;
	BOOL		fNeedUpdateProp = FALSE;
	AJCTCPROP	prop;
	WC			szKey[256];

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProfileSect != NULL && pKeyPrefix != NULL) {
			//----- デフォルトプロパティ値設定 -----------------------------------------------------------------//
			memcpy(&prop, &pW->prop, sizeof prop);
			//----- セクション名，永続化項目退避 ---------------------------------------------------------------//
			//	プロファイルセクション名，キー接頭語退避
			AjcTchSetPermAttW(hwnd, pProfileSect, pKeyPrefix);
			pProfileSect = pW->PermSect;
			pKeyPrefix	 = pW->PermKey;
			//	永続化項目退避
			pW->PermItem = PermItem;
			//----- 常時永続化項目 -----------------------------------------------------------------------------//
			pW->GauBarL		= AjcGetProfileUInt(pProfileSect, MKKEY(L"GauBarL"	  ), pW->GauBarL	);	//	ゲージ低位置
			pW->GauBarH		= AjcGetProfileUInt(pProfileSect, MKKEY(L"GauBarH"	  ), pW->GauBarH	);	//	ゲージ高位置
			pW->fGauUseMes	= AjcGetProfileUInt(pProfileSect, MKKEY(L"fGauUseMes" ), pW->fGauUseMes	);	//	単位時間＝計測値を示すフラグ
			pW->GauSetUTime	= AjcGetProfileUInt(pProfileSect, MKKEY(L"GauSetUTime"), pW->GauSetUTime);	//	単位時間（設定値）[us]
			//----- フィルタ選択状態 ---------------------------------------------------------------------------//
			if (pW->PermItem & AJCTCH_PERM_FILTER) {
				UI	MskFilt = AjcGetProfileUInt(pProfileSect, MKKEY(L"MskFilt"), pW->MskFilt	);	//	フィルタマスク値
				UI	i, msk;
				for (i = 0, msk = 0x01; i < AJCTC_MAXITEM; i++, msk <<= 1) {
					AjcTchSetFilter(hwnd, i, (MskFilt & msk) != 0);
				}
			}
			//----- レンジ値 -----------------------------------------------------------------------------------//
			if (pW->PermItem & AJCTCH_PERM_RANGE) {
				prop.RngL = AjcGetProfileReal(pProfileSect, MKKEY(L"RngL")	, prop.RngL);	//	レンジ低位置
				prop.RngH = AjcGetProfileReal(pProfileSect, MKKEY(L"RngH")	, prop.RngH);	//	レンジ高位置
				//	プロパティ更新の旨、フラグ設定
				fNeedUpdateProp = TRUE;
			}
			//----- その他 -------------------------------------------------------------------------------------//
			if (pW->PermItem & AJCTCH_PERM_OTHER) {
				prop.MaxBuf = AjcGetProfileUInt(pProfileSect, MKKEY(L"MaxBuf")	, prop.MaxBuf);		//	最大データ数
				prop.AveNum = AjcGetProfileUInt(pProfileSect, MKKEY(L"AveNum")	, prop.AveNum);		//	平均化個数
				//	プロパティ更新の旨、フラグ設定
				fNeedUpdateProp = TRUE;
			}
			//----- プロパティ更新 -----------------------------------------------------------------------------//
			rc = TRUE;
			if (fNeedUpdateProp) {
				rc = AjcTchSetProp(hwnd, &prop);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロファイルへ永続情報書き込み																				//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSavePermInfo		(HWND hwnd)
{
	return AjcTchSavePermInfoExW(hwnd, NULL, NULL, AJCTCH_PERM_BYLOAD);
}
//==============================================================================================================//
//	プロファイルへ永続情報書き込み																				//
//																												//
//	引　数	：	hwnd			- ウインドハンドル																//
//				pProfileSect	- セクション名へのポインタ	（不要時はNULL)										//
//				pKeyPrefix		- キー名先頭部分へのポインタ（不要時はNULL)										//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSavePermInfoExA		(HWND hwnd, C_BCP pProfileSect, C_BCP pKeyPrefix, UI PermItem)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmpSec = NULL;
	WCP		pTmpKey = NULL;

	do {
		if (pProfileSect != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
			if (len != 0 && (pTmpSec = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmpSec, len);
			}
			else break;
		}
		if (pKeyPrefix != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pKeyPrefix, -1, NULL, 0);
			if (len != 0 && (pTmpKey = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pKeyPrefix, -1, pTmpKey, len);
			}
			else break;
		}
		rc = AjcTchSavePermInfoExW(hwnd, pTmpSec, pTmpKey, PermItem);
	} while(0);

	if (pTmpSec != NULL) free(pTmpSec);
	if (pTmpKey != NULL) free(pTmpKey);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSavePermInfoExW		(HWND hwnd, C_WCP pProfileSect, C_WCP pKeyPrefix, UI PermItem)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;
	WC			szKey[256];

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			//----- セクション，キー，永続化項目 指定無しならば、ロード時に設定した名称とする ------------------//
			if (pProfileSect == NULL		 ) pProfileSect = pW->PermSect;
			if (pKeyPrefix	 == NULL		 ) pKeyPrefix	= pW->PermKey;
			if (PermItem & AJCTCH_PERM_BYLOAD) PermItem 	= pW->PermItem;
			//----- プロファイルセーブ -------------------------------------------------------------------------//
			if (*pProfileSect != 0) {
				//	現プロパティ値設定
				memcpy(&prop, &pW->prop, sizeof prop);
				//	常時永続化項目
				AjcPutProfileUInt(pProfileSect, MKKEY(L"GauBarL"	), pW->GauBarL		);	//	ゲージ低位置
				AjcPutProfileUInt(pProfileSect, MKKEY(L"GauBarH"	), pW->GauBarH		);	//	ゲージ高位置
				AjcPutProfileUInt(pProfileSect, MKKEY(L"fGauUseMes" ), pW->fGauUseMes	);	//	単位時間＝計測値を示すフラグ
				AjcPutProfileUInt(pProfileSect, MKKEY(L"GauSetUTime"), pW->GauSetUTime	);	//	単位時間（設定値）[us]
				//	フィルタ選択状態
				if (PermItem & AJCTCH_PERM_FILTER) {
					AjcPutProfileUInt(pProfileSect, MKKEY(L"MskFilt"), pW->MskFilt	);		//	フィルタマスク値
				}
				else {
					AjcDelProfileKey (pProfileSect, MKKEY(L"MskFilt"));
				}
				//	レンジ値
				if (PermItem & AJCTCH_PERM_RANGE) {
					AjcPutProfileReal(pProfileSect, MKKEY(L"RngL"	 ), prop.RngL	);		//	レンジ低位置
					AjcPutProfileReal(pProfileSect, MKKEY(L"RngH"	 ), prop.RngH	);		//	レンジ高位置
				}
				else {
					AjcDelProfileKey (pProfileSect, MKKEY(L"RngL"	 ));
					AjcDelProfileKey (pProfileSect, MKKEY(L"RngH"	 ));
				}
				//	その他
				if (PermItem & AJCTCH_PERM_OTHER) {
					AjcPutProfileUInt(pProfileSect, MKKEY(L"MaxBuf" ), prop.MaxBuf	);		//	最大データ数
					AjcPutProfileUInt(pProfileSect, MKKEY(L"AveNum" ), prop.AveNum	);		//	平均化個数
				}
				else {
					AjcDelProfileKey (pProfileSect, MKKEY(L"MaxBuf" ));
					AjcDelProfileKey (pProfileSect, MKKEY(L"AveNum" ));
				}
				rc = TRUE;
			}
		}
	}

	return rc;
}
//==============================================================================================================//
//	永続化の属性（セクション名，キー接頭語）設定																//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE:OK，FALSE:エラー																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSetPermAttA	(HWND hwnd, C_BCP pProfileSect, C_BCP pKeyPrefix)
{
	return AjcTchSetPermAttExA(hwnd, pProfileSect, pKeyPrefix, AJCTCH_PERM_FILTER);
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSetPermAttW	(HWND hwnd, C_WCP pProfileSect, C_WCP pKeyPrefix)
{
	return AjcTchSetPermAttExW(hwnd, pProfileSect, pKeyPrefix, AJCTCH_PERM_FILTER);
}

//==============================================================================================================//
//	永続化の属性（セクション名，キー接頭語）設定	（項目指定）												//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE:OK，FALSE:エラー																			//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSetPermAttExA	(HWND hwnd, C_BCP pProfileSect, C_BCP pKeyPrefix, UI PermItem)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmpSec = NULL;
	WCP		pTmpKey = NULL;

	do {
		if (pProfileSect != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
			if (len != 0 && (pTmpSec = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmpSec, len);
			}
			else break;
		}
		if (pKeyPrefix != NULL) {
			len = MultiByteToWideChar(CP_ACP, 0, pKeyPrefix, -1, NULL, 0);
			if (len != 0 && (pTmpKey = AjcTAlloc(len))) {
				MultiByteToWideChar(CP_ACP, 0, pKeyPrefix, -1, pTmpKey, len);
			}
			else break;
		}
		rc = AjcTchSetPermAttExW(hwnd, pTmpSec, pTmpKey, PermItem);
	} while(0);

	if (pTmpSec != NULL) free(pTmpSec);
	if (pTmpKey != NULL) free(pTmpKey);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSetPermAttExW	(HWND hwnd, C_WCP pProfileSect, C_WCP pKeyPrefix, UI PermItem)
{
	BOOL		rc = FALSE;
	WC			szid[16];

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			//----- ＩＤ文字列設定 -----------------------------------------------------------------------------//
			AjcSnPrintFW(szid, AJCTSIZE(szid), L"%06u", (UW)MAjcGetWindowLong(pW->hBack, GWL_ID));
			//----- セクション名設定 ---------------------------------------------------------------------------//
			if (pProfileSect != NULL) AjcSnPrintFW(pW->PermSect, AJCTSIZE(pW->PermSect) 	, L"%s"    , pProfileSect);
			else					  AjcSnPrintFW(pW->PermSect, AJCTSIZE(pW->PermSect) 	, L"STch%s", szid		 );
			//----- キー接頭語設定 -----------------------------------------------------------------------------//
			if (pKeyPrefix	 != NULL) AjcSnPrintFW(pW->PermKey , AJCTSIZE(pW->PermKey ) - 32, L"%s"    , pKeyPrefix);	// 32:後続キー名称の最大値
			else					  AjcSnPrintFW(pW->PermKey , AJCTSIZE(pW->PermKey ) - 32, L"KTch%s", szid	   );
			//----- 永続化項目設定 -----------------------------------------------------------------------------//
			pW->PermItem = PermItem;
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	プロファイルからプロパティ読み出し																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//				pDefProp	 - デフォルトプロパティ値（現在値をデフォルトとする場合はNULL)						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPropA		(HWND hwnd, C_BCP pProfileSect, PCAJCTCPROP pDefProp)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	if (pProfileSect != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAlloc(len))) {
			MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
			rc = AjcTchLoadPropW(hwnd, pTmp, pDefProp);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPropW		(HWND hwnd, C_WCP pProfileSect, PCAJCTCPROP pDefProp)
{
	BOOL		rc = TRUE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProfileSect != NULL) {

			if (pDefProp != NULL) memcpy(&prop, pDefProp, sizeof prop);
			else				  rc = AjcTchGetProp(hwnd, &prop);

			if (rc) {
				prop.RngL		  = AjcGetProfileReal(pProfileSect, L"RngL"	  , prop.RngL	   );
				prop.RngH		  = AjcGetProfileReal(pProfileSect, L"RngH"	  , prop.RngH	   );
				prop.TmScale	  = AjcGetProfileUInt(pProfileSect, L"TmScale", prop.TmScale   );
				prop.MaxBuf		  = AjcGetProfileUInt(pProfileSect, L"MaxBuf" , prop.MaxBuf	   );
				prop.MaxItem	  = AjcGetProfileUInt(pProfileSect, L"MaxItem", prop.MaxItem   );
				prop.AveNum		  = AjcGetProfileUInt(pProfileSect, L"AveNum" , prop.AveNum	   );

				prop.Item[0].ofs  = AjcGetProfileReal(pProfileSect, L"ofs0"	 , prop.Item[0].ofs);
				prop.Item[0].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb0"	 , prop.Item[0].rgb);
				prop.Item[1].ofs  = AjcGetProfileReal(pProfileSect, L"ofs1"	 , prop.Item[1].ofs);
				prop.Item[1].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb1"	 , prop.Item[1].rgb);
				prop.Item[2].ofs  = AjcGetProfileReal(pProfileSect, L"ofs2"	 , prop.Item[2].ofs);
				prop.Item[2].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb2"	 , prop.Item[2].rgb);
				prop.Item[3].ofs  = AjcGetProfileReal(pProfileSect, L"ofs3"	 , prop.Item[3].ofs);
				prop.Item[3].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb3"	 , prop.Item[3].rgb);
				prop.Item[4].ofs  = AjcGetProfileReal(pProfileSect, L"ofs4"	 , prop.Item[4].ofs);
				prop.Item[4].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb4"	 , prop.Item[4].rgb);
				prop.Item[5].ofs  = AjcGetProfileReal(pProfileSect, L"ofs5"	 , prop.Item[5].ofs);
				prop.Item[5].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb5"	 , prop.Item[5].rgb);
				prop.Item[6].ofs  = AjcGetProfileReal(pProfileSect, L"ofs6"	 , prop.Item[6].ofs);
				prop.Item[6].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb6"	 , prop.Item[6].rgb);
				prop.Item[7].ofs  = AjcGetProfileReal(pProfileSect, L"ofs7"	 , prop.Item[7].ofs);
				prop.Item[7].rgb  = AjcGetProfileUInt(pProfileSect, L"rgb7"	 , prop.Item[7].rgb);

				rc = AjcTchSetProp(hwnd, &prop);
			}
		}
		else rc = FALSE;
	}
	else rc = FALSE;

	return rc;
}
//==============================================================================================================//
//	プロファイルへプロパティ書き込み																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSavePropA		(HWND hwnd, C_BCP pProfileSect)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	if (pProfileSect != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAlloc(len))) {
			MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
			rc = AjcTchSavePropW(hwnd, pTmp);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSavePropW		(HWND hwnd, C_WCP pProfileSect)
{
	BOOL		rc = FALSE;
	AJCTCPROP	prop;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pProfileSect != NULL) {
			if (rc = AjcTchGetProp(hwnd, &prop)) {
				AjcPutProfileReal(pProfileSect, L"RngL"	 , prop.RngL		);
				AjcPutProfileReal(pProfileSect, L"RngH"	 , prop.RngH		);
				AjcPutProfileUInt(pProfileSect, L"TmScale", prop.TmScale	);
				AjcPutProfileUInt(pProfileSect, L"MaxBuf" , prop.MaxBuf		);
				AjcPutProfileUInt(pProfileSect, L"MaxItem", prop.MaxItem	);
				AjcPutProfileUInt(pProfileSect, L"AveNum" , prop.AveNum		);

				AjcPutProfileReal(pProfileSect, L"ofs0"	 , prop.Item[0].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb0"	 , prop.Item[0].rgb);
				AjcPutProfileReal(pProfileSect, L"ofs1"	 , prop.Item[1].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb1"	 , prop.Item[1].rgb);
				AjcPutProfileReal(pProfileSect, L"ofs2"	 , prop.Item[2].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb2"	 , prop.Item[2].rgb);
				AjcPutProfileReal(pProfileSect, L"ofs3"	 , prop.Item[3].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb3"	 , prop.Item[3].rgb);
				AjcPutProfileReal(pProfileSect, L"ofs4"	 , prop.Item[4].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb4"	 , prop.Item[4].rgb);
				AjcPutProfileReal(pProfileSect, L"ofs5"	 , prop.Item[5].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb5"	 , prop.Item[5].rgb);
				AjcPutProfileReal(pProfileSect, L"ofs6"	 , prop.Item[6].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb6"	 , prop.Item[6].rgb);
				AjcPutProfileReal(pProfileSect, L"ofs7"	 , prop.Item[7].ofs);
				AjcPutProfileUInt(pProfileSect, L"rgb7"	 , prop.Item[7].rgb);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロファイルからプロパティ，フィルタ設定，ウインドスタイル読み出し											//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//				pDefProp	 - デフォルトプロパティ値（現在値をデフォルトとする場合はNULL)						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPropExA	(HWND hwnd, C_BCP pProfileSect, PCAJCTCPROP pDefProp)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
	if (len != 0 && (pTmp = AjcTAlloc(len))) {
		MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
		rc = AjcTchLoadPropExW(hwnd, pTmp, pDefProp);
	}
	if (pTmp != NULL) free(pTmp);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchLoadPropExW	(HWND hwnd, C_WCP pProfileSect, PCAJCTCPROP pDefProp)
{
	BOOL		rc = FALSE;
	AJCTCIPINFO	IpInf;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int		i, msk, flt, styS, styU;
			if (rc = AjcTchLoadPropW(hwnd, pProfileSect, pDefProp)) {
				//----- 現在のフィルタ値取得 ---------------------------------------------------//
				flt = 0;
				for (i = 0, msk = 0x0001; i < AJCTC_MAXITEM; i++, msk <<= 1) {
					if (AjcTchGetFilter(hwnd, i)) flt |= msk;
				}
				//----- プロファイルからフィルタ値読み出し -------------------------------------//
				flt = AjcGetProfileUInt(pProfileSect, L"FilterValue", flt);
				//----- フィルタ値設定 ---------------------------------------------------------//
				for (i = 0, msk = 0x0001; i < AJCTC_MAXITEM; i++, msk <<= 1) {
					AjcTchSetFilter(hwnd, i, (flt & msk) != 0); 
				}
				//----- 現在のスタイル値読み出し -----------------------------------------------//
				styS = (int)MAjcGetWindowLong(hwnd, GWL_STYLE);
				styU = (styS & 0xFFFF);
				//----- プロファイルからスタイル値読み出し -------------------------------------//
				styU  = AjcGetProfileUInt(pProfileSect, L"WndStyle", styU);
				styU &= 0xFFFF;
				//----- スタイル設定 -----------------------------------------------------------//
				styS &= ~0xFFFF;
				styS |= styU;
				MAjcSetWindowLong(hwnd, GWL_STYLE, styS);
				//----- プロファイルから補間情報読み出し ---------------------------------------//
				AjcTchGetIpInfo(hwnd, &IpInf);
				IpInf.IpKnd   = (AJCTCIPKND)AjcGetProfileUInt(pProfileSect, L"IpInf.IpKnd"	,(UI)IpInf.IpKnd  );
				IpInf.IpNum   = 			AjcGetProfileUInt(pProfileSect, L"IpInf.IpNum"	,	 IpInf.IpNum  );
				IpInf.IpWidth = 			AjcGetProfileUInt(pProfileSect, L"IpInf.IpWidth",	 IpInf.IpWidth);
				//----- 補間情報設定 -----------------------------------------------------------//
				AjcTchGetIpInfo(hwnd, &IpInf);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロファイルへプロパティ，フィルタ設定，ウインドスタイル書き込み											//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pProfileSect - プロファイルのセクション名														//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSavePropExA	(HWND hwnd, C_BCP pProfileSect)
{
	BOOL	rc = FALSE;
	UI		len;
	WCP		pTmp = NULL;

	if (pProfileSect != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAlloc(len))) {
			MultiByteToWideChar(CP_ACP, 0, pProfileSect, -1, pTmp, len);
			rc = AjcTchSavePropExW(hwnd, pTmp);
		}
		if (pTmp != NULL) free(pTmp);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL		WINAPI	AjcTchSavePropExW	(HWND hwnd, C_WCP pProfileSect)
{
	BOOL	rc = FALSE;
	AJCTCIPINFO	IpInf;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int		i, msk, flt, sty;
			if (rc = AjcTchSaveProp(hwnd, pProfileSect)) {
				//----- 現在のフィルタ値取得 ---------------------------------------------------//
				flt = 0;
				for (i = 0, msk = 0x0001; i < AJCTC_MAXITEM; i++, msk <<= 1) {
					if (AjcTchGetFilter(hwnd, i)) flt |= msk;
				}
				//----- プロファイルへフィルタ値書き込み ---------------------------------------//
				AjcPutProfileUInt(pProfileSect, L"FilterValue", flt);
				//----- 現在のスタイル値読み出し -----------------------------------------------//
				sty	 = (int)MAjcGetWindowLong(hwnd, GWL_STYLE);
				sty &= 0xFFFF;
				//----- プロファイルへスタイル値書き込み ---------------------------------------//
				sty = AjcPutProfileUInt(pProfileSect, L"WndStyle", sty);
				//----- 補間情報読み出し -------------------------------------------------------//
				AjcTchGetIpInfo(hwnd, &IpInf);
				//----- プロファイルへ補間情報書き込み -----------------------------------------//
				AjcPutProfileUInt(pProfileSect, L"IpInf.IpKnd"	,(UI)IpInf.IpKnd  );
				AjcPutProfileUInt(pProfileSect, L"IpInf.IpNum"	,	 IpInf.IpNum  );
				AjcPutProfileUInt(pProfileSect, L"IpInf.IpWidth",	 IpInf.IpWidth);
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	右クリックによるポップアップメニューの許可／禁止															//
//																												//
//	引　数	：	hwnd		  - ウインドハンドル																//
//				fEnable		  - ポップアップメニューの許可／禁止												//
//									TRUE  : 許可（右クリック非通知，Shuft/Ctrl+右クリック通知）					//
//									FALSE : 禁止（全ての右クリック通知）										//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchEnablePopupMenu(HWND hwnd, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AjcTchSetNtcRClk(hwnd, !fEnable, WM_RBUTTONDOWN, WM_RBUTTONUP);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	右クリック通知設定																							//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fNtcRClk	 - TRUE  : 右ボタンDOWN/UP通知	（ポップアップは禁止）								//
//							   FALSE : 右ボタンDOWN/UP非通知（ポップアップを許可）								//
//				MsgRBDown	 - 右ボタン押下時の通知メッセージコード　（０の場合は非通知）						//
//				MSgRBUp		 - 右ボタン離され時の通知メッセージコード（０の場合は非通知）						//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//																												//
//	備　考	：						<ポップアップ>	<Shift/Ctrl+右>		<右-DOWN>	<右-UP>						//
//				NtcRClk	= FALSE	 	 許可		 	 通知				 非通知		 非通知						//
//				 		= TRUE		 禁止			 非通知				 MsgRBDown	 MsgRBUp					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchSetNtcRClk	(HWND hwnd, BOOL fNtcRClk, UI MsgRBDown, UI MsgRBUp)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fNtcRClk  = fNtcRClk;
			pW->MsgRBDown = MsgRBDown;
			pW->MsgRBUp	  = MsgRBUp;
			rc = TRUE;
		}
	}
	return rc;
}


//==============================================================================================================//
//	ツールチップ・テキスト設定																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pTxt		 - ツールチップ・テキスト文字列のアドレス											//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTchSetTipTextA (HWND hwnd, C_BCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = SubSetTipTextA(&pW->pToolTipTxt, 0, pTxt)) {
				AjcTipTextAddW		   (pW->hBack, pW->pToolTipTxt);
				AjcTipTextSetShowAlways(pW->hBack, pW->fShowAlways);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTchSetTipTextW (HWND hwnd, C_WCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (rc = SubSetTipTextW(&pW->pToolTipTxt, 0, pTxt)) {
				AjcTipTextAddW		   (pW->hBack, pW->pToolTipTxt);
				AjcTipTextSetShowAlways(pW->hBack, pW->fShowAlways);
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	ツールチップ・テキスト取得																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pBuf		 - ツールチップ・テキスト文字列を格納するバッファのアドレス							//
//				lBuf		 - ツールチップ・テキスト文字列を格納するバッファの文字数							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTchGetTipTextA (HWND hwnd, BCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = SubGetTipTextA(&pW->pToolTipTxt, 0, pBuf, lBuf);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTchGetTipTextW (HWND hwnd, WCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = SubGetTipTextW(&pW->pToolTipTxt, 0, pBuf, lBuf);
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト設定													//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～７）											//
//				pTxt		 - ツールチップ・テキスト文字列のアドレス											//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcTchSetChkBoxTipTextA(HWND hwnd, UI n, C_BCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM) {
				if (rc = SubSetTipTextA(pW->pChkBoxTool, n, pTxt)) {
					AjcTipTextAddW		   (pW->hChkFilt[n], pW->pChkBoxTool[n]);
					AjcTipTextSetShowAlways(pW->hChkFilt[n], pW->fChkBoxShowAlways[n]);
				}
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcTchSetChkBoxTipTextW(HWND hwnd, UI n, C_WCP pTxt)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM) {
				if (rc = SubSetTipTextW(pW->pChkBoxTool, n, pTxt)) {
					AjcTipTextAddW		   (pW->hChkFilt[n], pW->pChkBoxTool[n]);
					AjcTipTextSetShowAlways(pW->hChkFilt[n], pW->fChkBoxShowAlways[n]);
				}
			}
		}
	}
	return rc;
}

//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト取得													//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～７）											//
//				pBuf		 - ツールチップ・テキスト文字列を格納するバッファのアドレス							//
//				lBuf		 - ツールチップ・テキスト文字列を格納するバッファの文字数							//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcTchGetChkBoxTipTextA(HWND hwnd, UI n, BCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM) {
				rc = SubGetTipTextA(pW->pChkBoxTool, n, pBuf, lBuf);
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI		AjcTchGetChkBoxTipTextW(HWND hwnd, UI n, WCP pBuf, UI lBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM) {
				rc = SubGetTipTextW(pW->pChkBoxTool, n, pBuf, lBuf);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ツールチップ・テキスト表示条件設定																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fShowAlways	 - ツールチップ・テキスト表示条件													//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchSetTipShowAlways(HWND hwnd, BOOL fShowAlways)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fShowAlways = fShowAlways;
			AjcTipTextSetShowAlways(hwnd, pW->fShowAlways);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ツールチップ・テキスト表示条件取得																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	ツールチップ・テキスト表示条件																	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchGetTipShowAlways(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = pW->fShowAlways;
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト表示条件設定											//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//				fShowAlways	 - ツールチップ・テキスト表示条件													//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchSetChkBoxTipShowAlways(HWND hwnd, UI n, BOOL fShowAlways)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM) {
				pW->fChkBoxShowAlways[n] = fShowAlways;
				AjcTipTextSetShowAlways(pW->hChkFilt[n], pW->fChkBoxShowAlways[n]);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィィルタ（チェックボックス）のツールチップ・テキスト表示条件取得											//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～１５）											//
//																												//
//	戻り値	：	ツールチップ・テキスト表示条件																	//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchGetChkBoxTipShowAlways(HWND hwnd, UI n)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM) {
				rc = pW->fChkBoxShowAlways[n];
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	全てのツールチップ・テキスト表示条件設定																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fShowAlways	 - ツールチップ・テキスト表示条件													//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchSetTipShowAlwaysAll(HWND hwnd, BOOL fShowAlways)
{
	BOOL	rc = FALSE;
	int		i;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fShowAlways = fShowAlways;
			for (i = 0; i < AJCTC_MAXITEM; i++) {
				pW->fChkBoxShowAlways[i] = fShowAlways;
			}
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	最大結線長設定																								//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				dist		 - 最大結線長（０：全結線）															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetMaxLineDist(HWND hwnd, double MaxLineDist)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->MaxLineDist = MaxLineDist;
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	最大結線長取得																								//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	最大結線長（０：全結線）																		//
//==============================================================================================================//
AJCEXPORT	double		WINAPI	AjcTchGetMaxLineDist(HWND hwnd)
{
	double	rc = 0.0;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = pW->MaxLineDist;
		}
	}
	return rc;
}

//==============================================================================================================//
//	スクロール位置の取得																						//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	現在のスクロール位置（０～）																	//
//==============================================================================================================//
AJCEXPORT	int			WINAPI	AjcTchGetScrollPos	(HWND hwnd)
{
	SCROLLINFO	si;

	si.nPos = 0;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			si.cbSize = sizeof(SCROLLINFO);
			si.fMask  = SIF_POS;
			GetScrollInfo(pW->hMain, SB_HORZ, &si);
		}
	}
	return si.nPos;
}

//==============================================================================================================//
//	スクロール位置の設定																						//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				pos			 - 設定するスクロール位置															//
//																												//
//	戻り値	：	TRUE - OK, FALSE - Error																		//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetScrollPos	(HWND hwnd, int pos)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			SCROLLINFO	si;

			si.cbSize = sizeof(SCROLLINFO);
			si.fMask  = SIF_POS;
			si.nPos	  = pos;
			SetScrollInfo(pW->hMain, SB_HORZ, &si, TRUE);
			GetScrollInfo(pW->hMain, SB_HORZ, &si);

			pW->ixView = TchDixToBix(pW, si.nPos);
			TchSetViewWidth(pW);
			InvalidateRect(pW->hMain, NULL, FALSE);
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	フィルタの設定																								//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～７）											//
//				state		 - 設定する状態(FALSE:非表示，TRUE:表示)											//
//																												//
//	戻り値	：	なし																							//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetFilter		(HWND hwnd, UI n, BOOL state)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM  &&  pW->hChkFilt[n] != NULL) {
				SendMessage(pW->hChkFilt[n], BM_SETCHECK, state, 0);
				SendMessage(pW->hFilter, WM_COMMAND, MAKELONG(TCH_IDC_CHK_BASE + n, BN_CLICKED), 0);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	フィルタの取得																								//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				n			 - チェックボックス・インデクス（０～７）											//
//																												//
//	戻り値	：	フィルタの設定状態																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchGetFilter		(HWND hwnd, UI n)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (n < AJCTC_MAXITEM  &&  pW->hChkFilt[n] != NULL) {
				rc = (BOOL)SendMessage(pW->hChkFilt[n], BM_GETCHECK, 0, 0);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	横線（属性の設定）																							//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				id			 - 横線識別（０～７）																//
//				color		 - 描画色																			//
//				width		 - 線の太さ																			//
//				style		 - 横線種別	(AJCTCH_SOLID / DASH / DOT / DASHDOT / DASHDOTDOT / NULL)				//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetHLineAtt (HWND hwnd, UI id, COLORREF color, int width, int style)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			HPEN		hPen;
			if (id < TCH_MAX_HLINE	&&	style >= AJCTCH_SOLID && style <= AJCTCH_INSIDEFRAME) {
				pW->HLineTbl[id].color = color;
				pW->HLineTbl[id].width = width;
				pW->HLineTbl[id].style = style;
				if (hPen = CreatePen(style, width, color)) {
					if (pW->HLineTbl[id].hPen != NULL) {
						DeleteObject(pW->HLineTbl[id].hPen);
					}
					pW->HLineTbl[id].hPen = hPen;
					InvalidateRect(pW->hMain, NULL, FALSE);
					rc = TRUE;
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	横線（描画位置の設定）																						//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				id			 - 横線識別（０～７）																//
//				pos			 - 描画位置																			//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchSetHLinePos (HWND hwnd, UI id, double pos)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (id < TCH_MAX_HLINE) {
				pW->HLineTbl[id].pos = pos;
				InvalidateRect(pW->hMain, NULL, FALSE);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	横線（描画の許可／禁止）																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				id			 - 横線識別（０～７）																//
//				fEnable		 - TRUE	 : 描画																		//
//							   FALSE : 非描画																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchEnableHLine (HWND hwnd, UI id, BOOL fEnable)
{
	BOOL		rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (id < TCH_MAX_HLINE) {
				pW->HLineTbl[id].fEnable = fEnable;
				InvalidateRect(pW->hMain, NULL, FALSE);
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	縦線描画（最後に格納したデータの位置に縦線を描画する）														//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				color		 - 描画色																			//
//				width		 - 線の太さ																			//
//				style		 - 横線種別	(AJCTCH_SOLID / DASH / DOT / DASHDOT / DASHDOTDOT / NULL)				//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchSetVLine	 (HWND hwnd, COLORREF color, int width, int style)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int				ix;
			PTCH_VLINEBUF	pVLi;

			if (pW->nBuf != 0) {
				if (pW->ixBuf == 0) ix = pW->prop.MaxBuf - 1;
				else				ix = pW->ixBuf - 1;

				pVLi = pW->pVLi + ix;
				pVLi->flag	 |= TCH_VLIF_VLINE;
				pVLi->width	  = (UB)width;
				pVLi->style	  = (UB)style;
				pVLi->rgb	  = color;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	縦線描画の許可／禁止																						//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fEnable		 - TRUE	 : 描画																		//
//							   FALSE : 非描画																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchEnableVLine(HWND hwnd, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fVLiEnable = fEnable;
			InvalidateRect(pW->hMain, NULL, FALSE);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロット点描画（最後に格納したデータの位置にプロット点（円）を描画する）									//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				r			 - プロット点の半径																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchSetPoint (HWND hwnd, int r)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			int				ix;
			PTCH_VLINEBUF	pVLi;

			if (r > 0 && pW->nBuf != 0) {
				if (pW->ixBuf == 0) ix = pW->prop.MaxBuf - 1;
				else				ix = pW->ixBuf - 1;

				pVLi = pW->pVLi + ix;
				pVLi->flag |= TCH_VLIF_PLOT;
				pVLi->r 	= (UB)r;
				rc = TRUE;
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロット点描画の許可／禁止																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				fEnable		 - TRUE	 : 描画																		//
//							   FALSE : 非描画																	//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI	AjcTchEnablePoint(HWND hwnd, BOOL fEnable)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fPointEnable = fEnable;
			InvalidateRect(pW->hMain, NULL, FALSE);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたファイル名取得																				//
//																												//
//	引　数	：	pW	  - インスタンスワークアドレス																//
//				buf	  - ファイル名を格納するバッファのアドレス													//
//																												//
//	戻り値	：	TRUE:ファイル名あり，FALSE:ファイル名なし														//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTchGetDroppedFileA (HWND hwnd, BC buf[MAX_PATH])
{
	BOOL	rc = FALSE;
	WC		tmp[MAX_PATH];

	if (buf != NULL) {
		if (rc = AjcTchGetDroppedFileW(hwnd, tmp)) {
			WideCharToMultiByte(CP_ACP, 0, tmp, -1, buf, MAX_PATH, NULL, NULL);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTchGetDroppedFileW (HWND hwnd, WC buf[MAX_PATH])
{
	BOOL		rc = FALSE;
	UI			bytes;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && buf != NULL) {
			if (AjcRngGetData(pW->hRngFile, (VOP)&bytes, sizeof bytes) == sizeof bytes) {
				rc = (AjcRngGetData(pW->hRngFile, (VOP)buf, bytes) == bytes);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたディレクトリ名取得																			//
//																												//
//	引　数	：	pW	  - インスタンスワークアドレス																//
//				buf	  - ディレクトリ名を格納するバッファのアドレス												//
//																												//
//	戻り値	：	TRUE:ディレクトリ名あり，FALSE:ディレクトリ名なし												//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTchGetDroppedDirA  (HWND hwnd, BC buf[MAX_PATH])
{
	BOOL	rc = FALSE;
	WC		tmp[MAX_PATH];

	if (buf != NULL) {
		if (rc = AjcTchGetDroppedDirW(hwnd, tmp)) {
			WideCharToMultiByte(CP_ACP, 0, tmp, -1, buf, MAX_PATH, NULL, NULL);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTchGetDroppedDirW  (HWND hwnd, WC buf[MAX_PATH])
{
	BOOL		rc = FALSE;
	UI			bytes;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && buf != NULL) {
			if (AjcRngGetData(pW->hRngDir, (VOP)&bytes, sizeof bytes) == sizeof bytes) {
				rc = (AjcRngGetData(pW->hRngDir, (VOP)buf, bytes) == bytes);
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	ドロップされたディレクトリ名取得（末尾の「￥」付加指定付き）												//
//																												//
//	引　数	：	pW	  			 - インスタンスワークアドレス													//
//				buf	  			 - ディレクトリ名を格納するバッファのアドレス									//
//				fTailIsDelimiter - TRUE  : ディレクトリパスの末尾に「\」を付ける								//
//																												//
//	戻り値	：	TRUE:ディレクトリ名あり，FALSE:ディレクトリ名なし												//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTchGetDroppedDirExA	(HWND hwnd, BC buf[MAX_PATH], BOOL fTailIsDelimiter)
{
	BOOL	rc = FALSE;
	WC		tmp[MAX_PATH];

	if (rc = AjcTchGetDroppedDirExW(hwnd, tmp, fTailIsDelimiter)) {
		WideCharToMultiByte(CP_ACP, 0, tmp, -1, buf, MAX_PATH, NULL, NULL);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	 WINAPI AjcTchGetDroppedDirExW	(HWND hwnd, WC buf[MAX_PATH], BOOL fTailIsDelimiter)
{
	BOOL		rc = FALSE;

	if (rc = AjcTchGetDroppedDirW(hwnd, buf)) {
		if (fTailIsDelimiter) AjcPathCat(buf, L"", MAX_PATH);
		else				  PathRemoveBackslashW(buf);
	}
	return rc;
}
//==============================================================================================================//
//	タイトル文字列の設定																						//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pTitleText	- タイトルテキストへのポインタ（NULL指定時は、タイトル非表示）						//
//				TextColor	- テキスト描画色（－１指定時は前回設定値，デフォルト色＝白）						//
//				BackColor	- テキスト背景色（－１指定時は前回設定値，デフォルト色＝グレー）					//
//				hFont		- タイトルテキストのフォント（NULL指定時は前回設定値／デフォルトのフォント）		//
//																												//
//	戻り値	：	ウインド先頭行の位置（０～）																	//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	   WINAPI AjcTchSetTitleTextA(HWND hwnd, C_BCP pTitleText, COLORREF TextColor, COLORREF BackColor, HFONT hFont)
{
	BOOL		rc	 = FALSE;
	WCP			pTmp = NULL;
	UL			len;

	if (IsWindow(hwnd)) {
		do {
			if (pTitleText != NULL) {
				len = MultiByteToWideChar(CP_ACP, 0, pTitleText, -1, NULL, 0);
				if (len != 0  &&  (pTmp = AjcTAlloc(len + 1))) {
					MultiByteToWideChar(CP_ACP, 0, pTitleText, -1, pTmp, len);
				}
				else break;
			}
			rc = AjcTchSetTitleTextW(hwnd, pTmp, TextColor, BackColor, hFont);
			if (pTmp != NULL) free(pTmp);
		} while(0);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT BOOL	   WINAPI AjcTchSetTitleTextW(HWND hwnd, C_WCP pTitleText, COLORREF TextColor, COLORREF BackColor, HFONT hFont)
{
	BOOL	rc	 = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			//	タイトルテキスト・ビットマップ生成
			SubTtlBmpMake(&pW->TtlInfo, pW->hBack, pTitleText, TextColor, BackColor, hFont);
			//	戻り値＝正常
			rc = TRUE;
			//	再描画
			InvalidateRect(hwnd 	, NULL, TRUE);
			InvalidateRect(pW->hMain, NULL, TRUE);
		}
	}
	return rc;
}
//==============================================================================================================//
//	波形補間表示情報の設定																						//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pInfo		- 波形補間表示情報のアドレス														//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL	   WINAPI AjcTchSetIpInfo(HWND hwnd, PCAJCTCIPINFO pInfo)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd) 				&&
		pInfo != NULL				&&
		AJCTCIS_IPKND(pInfo->IpKnd)	&&
		pInfo->IpNum >= 10			&&
		pInfo->IpWidth >= AJCTCMIN_IPWID && pInfo->IpWidth <= AJCTCMAX_IPWID) {

		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);

		if (IS_MY_INST(pW)) {
			memcpy(&pW->IpInfo, pInfo,	sizeof(AJCTCIPINFO));
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	波形補間表示情報の取得																						//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pBuf		- 波形補間表示情報を格納するバッファのアドレス										//
//																												//
//	戻り値	：	TRUE  - OK																						//
//				FALSE - Error																					//
//==============================================================================================================//
AJCEXPORT BOOL	   WINAPI AjcTchGetIpInfo(HWND hwnd, PAJCTCIPINFO pBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pBuf != NULL) {
			memcpy(pBuf, &pW->IpInfo, sizeof(AJCTCIPINFO));
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	表示の停止／再開																							//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				fPause		- 表示停止フラグ（TRUE:表示停止，FALSE:表示開始）									//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTchPause(HWND hwnd, BOOL fPause)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (!(pW->fPause = fPause)) {						//	表示再開？
				//	スクロールバー再表示とウインド再描画
				TchSetScrollBar(pW);
				//	ＷＮＤ再描画
				InvalidateRect(pW->hMain, NULL, FALSE);
			}
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	描画時間計測情報の許可／禁止																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				fEnable		- 許可／禁止フラグ（TRUE:許可，FALSE:禁止）											//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI AjcTchEnableMesDraw(HWND hwnd, BOOL fEnable)
{
	BOOL		rc	 = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			pW->fMesDrawEnable = fEnable;
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロット周期計測情報表示／非表示の設定																		//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				fShow		- 表示フラグ																		//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI AjcTchMesPeriShow(HWND hwnd, BOOL fShow)
{
	BOOL		rc	 = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (pW->fMesPeriShow = (fShow == TRUE)) {
				pW->MesPeriAveS = pW->MesPeriAve;
				SetTimer(pW->hMain, TCH_TID_MESPERISHOW, 5000, NULL);
			}
			InvalidateRect(pW->hMain, NULL, FALSE);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロット周期計測情報リセット																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI AjcTchMesPeriReset(HWND hwnd)
{
	BOOL		rc	 = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			TchMesPeriReset(pW);
			InvalidateRect(pW->hMain, NULL, FALSE);
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	プロット周期取得																							//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//																												//
//	戻り値	：	プロットデータの投与周期[us]																	//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI AjcTchMesPeriGet(HWND hwnd)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = pW->MesPeriAve;
		}
	}
	return rc;
}
//==============================================================================================================//
//	時間計測ゲージ情報取得																						//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pBuf		- 時間計測ゲージ情報を格納するバッファのアドレス									//
//																												//
//	戻り値	：	TRUE  : OK																						//
//				FALSE : - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI AjcTchGetGauInfo(HWND hwnd, PAJCTCGAUINFO pBuf)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && pBuf != NULL) {
			pBuf->fGauEnable	= pW->fGauEnable;
			pBuf->GauBarL		= pW->GauBarL;
			pBuf->GauBarH		= pW->GauBarH;
			pBuf->fGauUseMes	= pW->fGauUseMes;
			pBuf->GauSetUTime	= pW->GauSetUTime;
			rc = TRUE;
		}
	}
	return rc;
}
//==============================================================================================================//
//	テキスト描画フォント設定																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				hFont		 - フォントハンドル																	//
//																												//
//	戻り値	：	≠NULL : 前回のフォントハンドル																	//
//				＝NULL : 失敗																					//
//==============================================================================================================//
AJCEXPORT	HFONT	WINAPI		AjcTchSetTextFont	(HWND hwnd, HFONT hFont)
{
	HFONT	rc = NULL;
	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW) && hFont != NULL) {
			rc = pW->hFontTxo;
			pW->hFontTxo = hFont;
		}
	}
	return rc;
}
//==============================================================================================================//
//	テキスト描画（ピクセル位置指定）																			//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				x, y		 - 描画位置																			//
//				pTxt		 - 描画テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcTchTextOutA		(HWND hwnd, int x, int y, C_BCP pTxt)
{
	UI		rc = 0;
	UI		len;
	WCP		pTmp = NULL;

	len = MultiByteToWideChar(CP_ACP, 0, pTxt, -1, NULL, 0);
	if (len != 0 && (pTmp = AjcTAllocW(len))) {
		MultiByteToWideChar(CP_ACP, 0, pTxt, -1, pTmp, len);
		rc = AjcTchTextOutW(hwnd, x, y, pTmp);
	}
	if (pTmp != NULL) free(pTmp);

	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcTchTextOutW		(HWND hwnd, int x, int y, C_WCP pTxt)
{
	UI		rc = 0;
	PTCHTXO	pNode;
	UI		stl, len;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (stl = (UI)wcslen(pTxt)) {
				len = sizeof(TCHTXO) + stl * 2;
				if (pNode = malloc(len)) {
					memset(pNode, 0, sizeof(TCHTXO));
					pNode->pt.x = x;
					pNode->pt.y = y;
					wcscpy(pNode->txt, pTxt);
					if (AjcAvlInsNode(pW->hAvlTxo, pW->kAvlTxo, (C_VOP)pNode, len)) {
						rc = pW->kAvlTxo++;
						InvalidateRect(pW->hMain, NULL, FALSE);
					}
					free(pNode);
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	書式テキスト描画（ピクセル位置指定）																		//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				x, y		 - 描画位置																			//
//				pFmt		 - 書式テキスト																		//
//																												//
//	戻り値	：	≠0 : テキストキー																				//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcTchPrintFA		(HWND hwnd, int x, int y, C_BCP pFmt, ...)
{
	BOOL	rc = FALSE;
	va_list vls;
	BC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFA(buf, 2048, pFmt, vls);
		va_end	(vls);
		rc = AjcTchTextOutA(hwnd, x, y, buf);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcTchPrintFW		(HWND hwnd, int x, int y, C_WCP pFmt, ...)
{
	UI		rc = 0;
	va_list	vls;
	WC		buf[2048];

	if (pFmt != NULL) {
		va_start(vls, pFmt);
		AjcVSnPrintFW(buf, 2048, pFmt, vls);
		buf[2047] = 0;
		va_end	(vls);
		rc = AjcTchTextOutW(hwnd, x, y, buf);
	}
	return rc;
}
//==============================================================================================================//
//	描画テキスト取得																							//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				key			 - テキストキー																		//
//				pBuf		 - テキスト格納バッファへのポインタ（不要時はNULL）									//
//				lBuft		 - テキスト格納バッファのバイト数／文字数											//
//																												//
//	戻り値	：	≠0 : 描画テキストのバイト数／文字数															//
//				＝0 : 失敗																						//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcTchGetTextA		(HWND hwnd, UI key, BCP pBuf, UI lBuf)
{
	UI		rc = 0;
	PTCHTXO	pNode;
	UI		lNode;
	UI		stl;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (pNode = (PTCHTXO)AjcAvlGetNodePtr(pW->hAvlTxo, key, &lNode)) {
				stl = (UI)wcslen(pNode->txt);
				rc = WideCharToMultiByte(CP_ACP, 0, pNode->txt, stl, NULL, 0, NULL, NULL);
				if (pBuf != NULL && lBuf != 0) {
					WideCharToMultiByte(CP_ACP, 0, pNode->txt, -1, pBuf, lBuf, NULL, NULL);
					pBuf[lBuf - 1] = 0;
				}
			}
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	UI		WINAPI		AjcTchGetTextW		(HWND hwnd, UI key, WCP pBuf, UI lBuf)
{
	UI		rc = 0;
	PTCHTXO	pNode;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (pNode = AjcAvlGetNodePtr(pW->hAvlTxo, key, NULL)) {
				rc = (UI)wcslen(pNode->txt);
				if (pBuf != NULL && lBuf != 0) {
					wcsncpy(pBuf, pNode->txt, lBuf);
					pBuf[lBuf - 1] = 0;
				}
			}
		}
	}
	return rc;
}
//==============================================================================================================//
//	描画テキストクリアー																						//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//				key			 - テキストキー																		//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI		AjcTchClearText		(HWND hwnd, UI key)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			rc = AjcAvlDelNode(pW->hAvlTxo, key);
			InvalidateRect(pW->hMain, NULL, FALSE);
		}
	}
	return rc;
}

//==============================================================================================================//
//	全ての描画テキストクリアー																					//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL		WINAPI		AjcTchClearAllText		(HWND hwnd)
{
	BOOL	rc = FALSE;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			AjcAvlDelAllNodes(pW->hAvlTxo);
			pW->kAvlTxo = 1;
			InvalidateRect(pW->hMain, NULL, FALSE);
		}
	}
	return rc;
}
//==============================================================================================================//
//	全てデータ（プロット，テキスト）クリアー																	//
//																												//
//	引　数	：	hwnd		 - ウインドハンドル																	//
//																												//
//	戻り値	：	TRUE  : 成功																					//
//				FALSE : 失敗																					//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI		AjcTchClear				(HWND hwnd)
{
	BOOL	rc = TRUE;

	rc &= AjcTchPurge		 (hwnd);
	rc &= AjcTchClearAllText (hwnd);

	return rc;
}

//==============================================================================================================//
//	メインウインドのハンドル取得																				//
//																												//
//	引　数	：	hwnd		- ウインドハンドル																	//
//				pBkWnd		- ウインド背景色を格納するバッファへのポインタ（不要時はNULL）						//
//																												//
//	戻り値	：	≠NULL : メインウインドのハンドル																//
//				＝NULL : エラー																					//
//==============================================================================================================//
AJCEXPORT	HWND	WINAPI		AjcTchGetMainWindow(HWND hwnd, COLORREF *pBkGnd)
{
	HWND	rc = NULL;

	if (IsWindow(hwnd)) {
		PWRKTMCHART	pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		if (IS_MY_INST(pW)) {
			if (pBkGnd != NULL) {
				*pBkGnd = RGB(255, 255, 255);
			}
			rc = pW->hMain;
		}
	}
	return rc;
}

//==============================================================================================================//
//																												//
//	通知情報取得																								//
//																												//
//==============================================================================================================//
//	右クリック通知情報
AJCEXPORT	PAJCTCRCLK			WINAPI AjcTchGetNtcRClk		(HWND hwnd)
{
	PAJCTCRCLK	rc = NULL;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		rc = &pW->NtcRClk;
	}
	return rc;
}
//	ドロップしたディレクトリ数
AJCEXPORT	UI					WINAPI AjcTchGetNtcDirs		(HWND hwnd)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		rc = pW->NtcDirs;
	}
	return rc;
}
//	ドロップしたファイル数
AJCEXPORT	UI					WINAPI AjcTchGetNtcFiles	(HWND hwnd)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		rc = pW->NtcFiles;
	}
	return rc;
}
//	グラフレンジ情報
AJCEXPORT	PAJCTC_NTC_RANGE	WINAPI AjcTchGetNtcRng		(HWND hwnd)
{
	PAJCTC_NTC_RANGE	rc = NULL;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		rc = &pW->NtcRng;
	}
	return rc;
}
//	スクロール位置
AJCEXPORT	UI					WINAPI AjcTchGetNtcScrPos	(HWND hwnd)
{
	UI		rc = 0;

	if (IsWindow(hwnd)) {
		PWRKTMCHART pW = (PWRKTMCHART)MAjcGetWindowLong(hwnd, 0);
		rc = pW->NtcScrPos;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
//																												//
//	チックカウンタ取得（μｓカウンタ）																			//
//																												//
//--------------------------------------------------------------------------------------------------------------//
static	ULL		GetUsCount(VO)
{
	ULL				rc = 0;
	LARGE_INTEGER	tick;

	if (TchPfmFreq != 0) {
		QueryPerformanceCounter(&tick);
		rc = tick.QuadPart * (ULL)1000000 / TchPfmFreq;
	}
	else {
		rc = (ULL)timeGetTime() * (ULL)1000;
	}
	return rc;
}
