﻿#include	"AjcInternal.h"

//==============================================================================================================//
//	Function	:	1970/1/1 00:00:00からの通算秒をシステムタイムに変換											//
//																												//
//	Argument	:	Time1970 - 1970/1/1 00:00:00からの通算秒													//
//					pStime	 - 変換したシステムタイムを格納するバッファのアドレス								//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - 失敗																				//
//==============================================================================================================//
#define	Y65535_12_31_23_59_59	0x01d30e53c8ffULL	//	1970/1/1 00:00:00 ～ 65535/12/31 23:59:59 の秒数
//----- 各月の日数テーブル -------------------------------------------------------------------------------------//
//									  1	  2	  3	  4	  5	  6	  7	  8	  9	  10  11  12							//
static	const UB DaysTable[13] = {0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
//----- うるう年判定マクロ -------------------------------------------------------------------------------------//
#define		ISURU(Y)	(((Y % 4) == 0 && (Y % 100) != 0) || (Y % 400) == 0)
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTime1970ToSysTime (UI  Time1970, LPSYSTEMTIME pStime)
{
	return AjcTime1970ToSysTimeL(Time1970, pStime);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcTime1970ToSysTimeL(ULL Time1970, LPSYSTEMTIME pStime)
{
	BOOL	rc = FALSE;
	ULL		days = Time1970 / 86400;
	ULL		y = 1970;
	BOOL	u = ISURU(y);
	UI		m;

	if (Time1970 < Y65535_12_31_23_59_59 && pStime != NULL) {
		pStime->wDayOfWeek = (UB)((days + 4) % 7);

		while (days >= (365 + u)) {
			days -= (365 + u);
			y++;
			u = ISURU(y);
		}
		for (m = 1;
			days >= (UL)(DaysTable[m] + (m == 2 ? u : 0)) && m < 12;
			days -= (DaysTable[m] + (m == 2 ? u : 0)), m++);

		pStime->wYear		  = (UW)y;
		pStime->wMonth		  = (UB)m;
		pStime->wDay		  = (UB)(days + 1);
		pStime->wDayOfWeek	  = (((Time1970 / (24 * 60 * 60)) + 4) % 7);
		pStime->wHour		  = (UB)((Time1970 % 86400) / 3600);
		pStime->wMinute		  = (UB)((Time1970 %  3600) / 60);
		pStime->wSecond		  = (UB)( Time1970 %	60);
		pStime->wMilliseconds = 0;
		rc = TRUE;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	システムタイムを1970/1/1 00:00:00からの通算秒に変換											//
//																												//
//	Argument	:	pStime		- システムタイム																//
//																												//
//	Return		:	≠-1 - 1970/1/1 00:00:00からの通算秒														//
//					＝-1 - エラー																				//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcSysTimeToTime1970(const SYSTEMTIME *pStime)
{
	return (UI)AjcSysTimeToTime1970L(pStime);
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcSysTimeToTime1970L(const SYSTEMTIME *pStime)
{
	ULL		ftime = -1;
	UI		m;
	ULL		y400 = pStime->wYear - (pStime->wYear % 400);
	ULL		y100 = pStime->wYear - (pStime->wYear % 100);

	if (pStime != NULL) {
		//	前年大晦日までの秒数
		ftime = (((((ULL)pStime->wYear - 1970ULL) * 365ULL) + (((ULL)pStime->wYear - 1969ULL) / 4ULL)) * 86400ULL);
		//	うるう年以外の年の日数分を減算
		ftime -= (((((ULL)pStime->wYear - (ULL)y400) / 100ULL) - (((ULL)pStime->wYear - (ULL)y400) / 400ULL)) * 86400ULL);
		//	今年分の秒数加算
		for (m = 1; m < pStime->wMonth; m++) 
			ftime += ((DaysTable[m] + (m == 2 ? ISURU(pStime->wYear) : 0)) * 86400);
		ftime += (pStime->wDay - 1) * 86400;
		ftime += (pStime->wHour * 3600);
		ftime += (pStime->wMinute * 60);
		ftime += pStime->wSecond;
	}
	return ftime;
}
//==============================================================================================================//
//	Function	:	システムタイムをローカルタイムに変換														//
//																												//
//	Argument	:	pStime	 - システムタイム																	//
//					pLTime	 - ローカルタイムを格納するバッファのアドレス										//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - 失敗																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcSysTimeToLocalTime(const SYSTEMTIME *pStime, LPSYSTEMTIME pLtime)
{
	BOOL		rc = FALSE;
	FILETIME	sft, lft;

	if (pStime != NULL && pLtime != NULL) {
		if (SystemTimeToFileTime(pStime, &sft)) {
			FileTimeToLocalFileTime(&sft, &lft);
			FileTimeToSystemTime(&lft, pLtime);
			rc = TRUE;
		}
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	指定秒数進めた（あるいは戻した）時刻を求める												//
//																												//
//	Argument	:	pStime	 - 更新対象時刻																		//
//					pLTime	 - 更新後の時刻を格納するバッファのアドレス											//
//					Second	 - 更新秒数（戻す場合は負数）														//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - 失敗																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcTimeUpdate(const SYSTEMTIME *pStime, LPSYSTEMTIME pUpdated, int Second)
{
	BOOL	rc = FALSE;
	union {FILETIME ft;	 __int64 tm;} u;

	if (pStime != NULL && pUpdated != NULL) {
		SystemTimeToFileTime(pStime, &u.ft);
		u.tm += ((__int64)Second * (__int64)10000000);
		FileTimeToSystemTime(&u.ft, pUpdated);
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	日付文字列取得																				//
//																												//
//	Argument	:	pStime	 - 日付／時刻情報																	//
//																												//
//	Return		:	≠NULL - 日付文字列のアドレス																//
//					＝NULL - エラー																				//
//==============================================================================================================//
__declspec(thread) static BC DateStrA[64] = {0};
__declspec(thread) static WC DateStrW[64] = {0};
__declspec(thread) static BC TimeStrA[64] = {0};
__declspec(thread) static WC TimeStrW[64] = {0};

//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcDateStrA(const SYSTEMTIME *pStime)
{
	BCP		rc = NULL;
	BC		sd[16];
	BC		id[16];

	if (pStime != NULL) {
		if (GetLocaleInfoA(LOCALE_USER_DEFAULT, LOCALE_SDATE, sd, sizeof sd) == 0) strcpy(sd, "/");
		if (GetLocaleInfoA(LOCALE_USER_DEFAULT, LOCALE_IDATE, id, sizeof id) == 0) id[0] = 2;

		switch (id[0]) {
			case '0':  AjcSnPrintFA(DateStrA, 64, "%02d%s%02d%s%4d", pStime->wMonth, sd, pStime->wDay  , sd, pStime->wYear); break;
			case '1':  AjcSnPrintFA(DateStrA, 64, "%02d%s%02d%s%4d", pStime->wDay  , sd, pStime->wMonth, sd, pStime->wYear); break;
			default:   AjcSnPrintFA(DateStrA, 64, "%4d%s%02d%s%02d", pStime->wYear , sd, pStime->wMonth, sd, pStime->wDay ); break;
		}
		rc = DateStrA;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcDateStrW(const SYSTEMTIME *pStime)
{
	WCP		rc = NULL;
	WC		sd[16];
	WC		id[16];

	if (pStime != NULL) {
		if (GetLocaleInfoW(LOCALE_USER_DEFAULT, LOCALE_SDATE, sd, sizeof sd) == 0) wcscpy(sd, L"/");
		if (GetLocaleInfoW(LOCALE_USER_DEFAULT, LOCALE_IDATE, id, sizeof id) == 0) id[0] = 2;

		switch (id[0]) {
			case L'0':	AjcSnPrintFW(DateStrW, 64, L"%02d%s%02d%s%4d", pStime->wMonth, sd, pStime->wDay	 , sd, pStime->wYear); break;
			case L'1':	AjcSnPrintFW(DateStrW, 64, L"%02d%s%02d%s%4d", pStime->wDay	 , sd, pStime->wMonth, sd, pStime->wYear); break;
			default:	AjcSnPrintFW(DateStrW, 64, L"%4d%s%02d%s%02d", pStime->wYear , sd, pStime->wMonth, sd, pStime->wDay ); break;
		}
		rc = DateStrW;
	}
	return rc;
}

//==============================================================================================================//
//	Function	:	時刻文字列取得																				//
//																												//
//	Argument	:	pStime	 - 日付／時刻情報																	//
//																												//
//	Return		:	≠NULL - 時刻文字列のアドレス																//
//					＝NULL - エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BCP		WINAPI	AjcTimeStrA(const SYSTEMTIME *pStime, BOOL fMilliseconds)
{
	BCP		rc = NULL;
	BC		st[16];

	if (pStime != NULL) {
		if (GetLocaleInfoA(LOCALE_USER_DEFAULT, LOCALE_STIME, st, sizeof st) == 0) strcpy(st, ":");

		if (fMilliseconds) {
			AjcSnPrintFA(TimeStrA, 64, "%02d%s%02d%s%02d.%03d", pStime->wHour, st, pStime->wMinute, st, pStime->wSecond, pStime->wMilliseconds);
		}
		else {
			AjcSnPrintFA(TimeStrA, 64, "%02d%s%02d%s%02d", pStime->wHour, st, pStime->wMinute, st, pStime->wSecond);
		}
		rc = TimeStrA;
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	WCP		WINAPI	AjcTimeStrW(const SYSTEMTIME *pStime, BOOL fMilliseconds)
{
	WCP		rc = NULL;
	WC		st[16];

	if (pStime != NULL) {
		if (GetLocaleInfoW(LOCALE_USER_DEFAULT, LOCALE_STIME, st, sizeof st) == 0) wcscpy(st, L":");

		if (fMilliseconds) {
			AjcSnPrintFW(TimeStrW, 64, L"%02d%s%02d%s%02d.%03d", pStime->wHour, st, pStime->wMinute, st, pStime->wSecond, pStime->wMilliseconds);
		}
		else {
			AjcSnPrintFW(TimeStrW, 64, L"%02d%s%02d%s%02d", pStime->wHour, st, pStime->wMinute, st, pStime->wSecond);
		}
		rc = TimeStrW;
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	日付文字列からSYSTEMTIME構造体の日付情報を設定												//
//																												//
//	Argument	:	pDateText	 - 日付テキストのアドレス（ex. “2022/12/23”）									//
//					pSt 		 - 日付情報を設定するSYSTEMTIME構造体のアドレス									//
//					pNext		 - 日付テキストの直後のアドレスを格納するポインタ変数のアドレス（不要時はNULL）	//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateFromTextA			(C_BCP pDateText, LPSYSTEMTIME pSt, BCP *pNext)
{
	BOOL	rc = FALSE;
	rc = AjcSetDateFromTextExA(pDateText, pSt, pNext, AJCOPT4(AJCDATEFLAG_, YYMMDD, ADJUST, SETDEF, SETDOW), '/');
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateFromTextW			(C_WCP pDateText, LPSYSTEMTIME pSt, WCP *pNext)
{
	BOOL	rc = FALSE;
	rc = AjcSetDateFromTextExW(pDateText, pSt, pNext, AJCOPT4(AJCDATEFLAG_, YYMMDD, ADJUST, SETDEF, SETDOW), L'/');
	return rc;
}

//==============================================================================================================//
//	Function	:	日付文字列からSYSTEMTIME構造体の日付情報を設定（拡張版）									//
//																												//
//	Argument	:	pDateText	- 日付テキストのアドレス（ex. “2022/12/23”）									//
//					pSt 		- 日付情報を設定するSYSTEMTIME構造体のアドレス									//
//					pNext		- 日付テキストの直後のアドレスを格納するポインタ変数のアドレス（不要時はNULL）	//
//					flag		- フラグ																		//
//					dlm			- 年月日の区切り文字（０指定時は、'/' を仮定）									//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - エラー																				//
//==============================================================================================================//
//----- ワーク -------------------------------------------------------------------------------------------------//
typedef struct {
	C_WCP			pTxt;					//	入力テキストポインタ
	UI				flag;					//	フラグ
	WC				dlm;					//	日付分離記号
	UW				yy, mm, dd, dow;		//	出力情報（年月日，曜日）
} WKSETDATE, *PWKSETDATE;
//----- 区切り文字スキップ -------------------------------------------------------------------------------------//
static BOOL SubSkipDlmD(PWKSETDATE pW)
{
	BOOL	rc = FALSE;

	//	区切り文字ならばスキップ
	if (*pW->pTxt == pW->dlm) {
		pW->pTxt++;
		rc = TRUE;
	}
	//	空白／終端ならば何もしない
	else if (MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		rc = TRUE;
	}
	return rc;
}
//----- 年設定 -------------------------------------------------------------------------------------------------//
static BOOL SubSetYear(PWKSETDATE pW)
{
	BOOL	rc = FALSE;

	//	● 数字ならば「年」設定
	if (MAjcIsDigitW(*pW->pTxt)) {
		int		stl  = 0;
		C_WCP	pTmp = pW->pTxt;
		//	桁数チェック
		while (MAjcIsDigitW(*pTmp)) {
			pTmp++;
			stl++;
		}
		if (stl == 2 || stl == 4) {
			//	「年」設定
			pW->yy = _wtoi(pW->pTxt);
			if		(pW->yy >=	0 && pW->yy <= 69) pW->yy += 2000;
			else if (pW->yy >= 70 && pW->yy <= 99) pW->yy += 1900;
			//	数字スキップ
			pW->pTxt += stl;
			rc = TRUE;
		}
	}
	//	● "****" ならば、何もしない（構造体へ設定しない）
	else if (wcsncmp(pW->pTxt, L"****", 4) == 0 && *(pW->pTxt + 4) != L'*') {
		pW->pTxt += 4;
		rc = TRUE;
	}
	//	● "**" ならば、何もしない（構造体へ設定しない）
	else if (wcsncmp(pW->pTxt, L"**"  , 2) == 0 && *(pW->pTxt + 2) != L'*') {
		pW->pTxt += 2;
		rc = TRUE;
	}
	//	● 空白／終端 ＆ AJCDATEFLAG_SETDEF指定ならばデフォルト値設定
	else if (MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		if (pW->flag & AJCDATEFLAG_SETDEF) {
			pW->yy = 2000;
		}
		rc = TRUE;
	}
	return rc;
}
//----- 月設定 -------------------------------------------------------------------------------------------------//
static BOOL SubSetMonth(PWKSETDATE pW)
{
	BOOL	rc = FALSE;

	//	● 数字ならば「月」設定
	if (MAjcIsDigitW(*pW->pTxt)) {
		int		stl  = 0;
		C_WCP		pTmp = pW->pTxt;
		//	桁数チェック
		while (MAjcIsDigitW(*pTmp)) {
			pTmp++;
			stl++;
		}
		if (stl == 2) {
			//	「月」設定
			pW->mm = _wtoi(pW->pTxt);
			//	数字スキップ＆桁数チェック
			pW->pTxt += stl;
			rc = TRUE;
		}
	}
	//	● "**" ならば、何もしない（構造体へ設定しない）
	else if (wcsncmp(pW->pTxt, L"**", 2) == 0) {
		pW->pTxt += 2;
		rc = TRUE;
	}
	//	● 空白／終端 ＆ AJCDATEFLAG_SETDEF指定ならばデフォルト値設定
	else if (MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		if (pW->flag & AJCDATEFLAG_SETDEF) {
			pW->mm = 1;
		}
		rc = TRUE;
	}
	return rc;
}
//----- 日設定 -------------------------------------------------------------------------------------------------//
static BOOL SubSetDay(PWKSETDATE pW)
{
	BOOL	rc = FALSE;

	//	● 数字ならば「日」設定
	if (MAjcIsDigitW(*pW->pTxt)) {
		int		stl  = 0;
		C_WCP	pTmp = pW->pTxt;
		//	桁数チェック
		while (MAjcIsDigitW(*pTmp)) {
			pTmp++;
			stl++;
		}
		if (stl == 2) {
			//	「日」設定
			pW->dd = _wtoi(pW->pTxt);
			//	数字スキップ＆桁数チェック
			pW->pTxt += stl;
			rc = TRUE;
		}
	}
	//	● "**" ならば、何もしない（構造体へ設定しない）
	else if (wcsncmp(pW->pTxt, L"**", 2) == 0) {
		pW->pTxt += 2;
		rc = TRUE;
	}
	//	● 空白／終端 ＆ AJCDATEFLAG_SETDEF指定ならばデフォルト値設定
	else if (MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		if (pW->flag & AJCDATEFLAG_SETDEF) {
			pW->dd = 1;
		}
		rc = TRUE;
	}
	return rc;
}
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateFromTextExA			(C_BCP pDateText, LPSYSTEMTIME pSt, BCP *pNext, UI flag, BC dlm)
{
	BOOL	rc = FALSE;
	WCP		pTxtW = NULL;
	int		lTxtW;
	WCP		pNxtW;

	if (pDateText != NULL && pSt != NULL) {
		lTxtW = AjcByteCharToWideChar(pDateText, NULL, 0);
		if (pTxtW = AjcTAlloc(lTxtW)) {
			AjcByteCharToWideChar(pDateText, pTxtW, lTxtW);
			rc = AjcSetDateFromTextExW(pTxtW, pSt, &pNxtW, flag, dlm);
			if (pNext != NULL) {
				UX	wcl = ((UX)pNxtW - (UX)pTxtW) / 2;
				UI	bcl = WideCharToMultiByte(CP_ACP, 0, pTxtW, (UI)wcl, NULL, 0, NULL, NULL);
				*pNext = (BCP)(pDateText + bcl);
			}
		}
		if (pTxtW != NULL) free(pTxtW);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateFromTextExW			(C_WCP pDateText, LPSYSTEMTIME pSt, WCP *pNext, UI flag, WC dlm)
{
	BOOL		rc = FALSE;
	WKSETDATE	wrk;

	if (pDateText != NULL && pSt != NULL && !MAjcIsAlNum(dlm) && !MAjcIsBlank(dlm) && dlm != L'*') {
		//	デフォルト区切り文字設定
		if (dlm == 0) dlm = L'/';
		//	ワーク設定
		wrk.pTxt = pDateText;
		wrk.flag = flag;
		wrk.dlm  = dlm;
		wrk.yy	 = pSt->wYear;
		wrk.mm	 = pSt->wMonth;
		wrk.dd	 = pSt->wDay;
		wrk.dow  = pSt->wDayOfWeek;
		//	先頭空白スキップ
		while (MAjcIsBlankW(*wrk.pTxt)) {
			wrk.pTxt++;
		}
		//	年月日設定
		switch (flag& AJCDATEFLAG_FORM) {
			case AJCDATEFLAG_YYMMDD:
				if (!(rc = SubSetYear (&wrk))) break;		//	「年」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetMonth(&wrk))) break;		//	「月」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetDay  (&wrk))) break;		//	「日」設定
				break;

			case AJCDATEFLAG_MMDDYY:
				if (!(rc = SubSetMonth(&wrk))) break;		//	「月」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetDay  (&wrk))) break;		//	「日」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetYear (&wrk))) break;		//	「年」設定
				break;

			case AJCDATEFLAG_DDMMYY:
				if (!(rc = SubSetDay  (&wrk))) break;		//	「日」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetMonth(&wrk))) break;		//	「月」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetYear (&wrk))) break;		//	「年」設定
				break;

			case AJCDATEFLAG_MMYYDD:
				if (!(rc = SubSetMonth(&wrk))) break;		//	「月」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetYear (&wrk))) break;		//	「年」設定
				if (!(rc = SubSkipDlmD(&wrk))) break;		//	 区切り文字スキップ
				if (!(rc = SubSetDay  (&wrk))) break;		//	「日」設定
				break;

			default:
				rc = FALSE;
				break;
		}
	}

	if (rc) {
		//	年月日補正
		if (wrk.flag & AJCDATEFLAG_ADJUST) {
			if (wrk.yy < 1970) wrk.yy = 1970;
			if (wrk.mm <	1) wrk.mm = 1;
			if (wrk.mm >   12) wrk.mm = 12;
			if (wrk.dd <	1) wrk.dd = 1;
			if (wrk.dd >   31) wrk.dd = 31;
			if (wrk.mm == 2) {
				if (ISURU(wrk.yy)) {
					if (wrk.dd > 29) wrk.dd = 29;
				}
				else {
					if (wrk.dd > 28) wrk.dd = 28;
				}
			}
			else if (wrk.mm == 4 || wrk.mm == 6 || wrk.mm == 9 || wrk.mm == 11) {
				if (wrk.dd > 30) wrk.dd = 30;
			}
		}
		//	曜日設定
		if (wrk.flag & AJCDATEFLAG_SETDOW) {
			ULL			sec1970;
			SYSTEMTIME	st = {0};
			st.wYear  = wrk.yy;
			st.wMonth = wrk.mm;
			st.wDay   = wrk.dd;
			sec1970 = AjcSysTimeToTime1970L(&st);
			sec1970 /= (ULL)(24 * 60 * 60);
			wrk.dow = ((sec1970 + 4ULL) % 7ULL);
		}
		//	戻りSYSTEMTIME構造体設定
		pSt->wYear		= wrk.yy;
		pSt->wMonth 	= wrk.mm;
		pSt->wDay		= wrk.dd;
		pSt->wDayOfWeek = wrk.dow;
	}
	//	戻りポインタ設定
	if (pNext != NULL) {
		*pNext = (WCP)wrk.pTxt;
	}

	return rc;
}

//==============================================================================================================//
//	Function	:	時刻文字列からSYSTEMTIME構造体の時刻情報を設定												//
//																												//
//	Argument	:	pTimeText	- 時刻テキストのアドレス（ex. “13:23:51.345”）								//
//					pSt 		- 時刻情報を設定するSYSTEMTIME構造体のアドレス									//
//					pNext		- 時刻テキストの直後のアドレスを格納するポインタ変数のアドレス（不要時はNULL）	//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetTimeFromTextA			(C_BCP pTimeText, LPSYSTEMTIME pSt, BCP *pNext)
{
	BOOL	rc = FALSE;
	rc = AjcSetTimeFromTextExA(pTimeText, pSt, pNext, AJCOPT2(AJCTIMEFLAG_, ADJUST, SETDEF), ':', '.');
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetTimeFromTextW			(C_WCP pTimeText, LPSYSTEMTIME pSt, WCP *pNext)
{
	BOOL	rc = FALSE;
	rc = AjcSetTimeFromTextExW(pTimeText, pSt, pNext, AJCOPT2(AJCTIMEFLAG_, ADJUST, SETDEF), L':', L'.');
	return rc;
}

//==============================================================================================================//
//	Function	:	時刻文字列からSYSTEMTIME構造体の時刻情報を設定（拡張版）									//
//																												//
//	Argument	:	pTimeText	- 時刻テキストのアドレス（ex. “13:23:51.345”）								//
//					pSt 		- 時刻情報を設定するSYSTEMTIME構造体のアドレス									//
//					pNext		- 時刻テキストの直後のアドレスを格納するポインタ変数のアドレス（不要時はNULL）	//
//					dlm			- 時分秒の区切り文字（０指定時は、':' を仮定）									//
//					dms			- 時分秒の区切り文字（０指定時は、'.' を仮定）									//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - エラー																				//
//==============================================================================================================//
//----- ワーク -------------------------------------------------------------------------------------------------//
typedef struct {
	C_WCP			pTxt;					//	入力テキストポインタ
	UI				flag;					//	フラグ
	WC				dlm;					//	時刻分離記号
	WC				dms;					//	ミリ秒分離記号
	UW				hh, mm, ss, ms;			//	出力情報（時分秒，ミリ秒）
} WKSETTIME, *PWKSETTIME;
//----- 区切り文字(:)スキップ ----------------------------------------------------------------------------------//
static BOOL SubSkipDlmT(PWKSETTIME pW)
{
	BOOL	rc = FALSE;

	//	区切り文字ならばスキップ
	if (*pW->pTxt == pW->dlm) {
		pW->pTxt++;
		rc = TRUE;
	}
	//	ミリ秒区切り文字／空白／終端／ならば何もしない
	else if (*pW->pTxt == pW->dms || MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		rc = TRUE;
	}
	return rc;
}
//----- 区切り文字(.)スキップ ----------------------------------------------------------------------------------//
static BOOL SubSkipDlmP(PWKSETTIME pW)
{
	BOOL	rc = FALSE;

	//	区切り文字ならばスキップ
	if (*pW->pTxt == pW->dms) {
		pW->pTxt++;
		rc = TRUE;
	}
	//	空白／終端／ならば何もしない
	else if (MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		rc = TRUE;
	}
	return rc;
}
//----- 時/分/秒設定 -------------------------------------------------------------------------------------------//
static BOOL SubSetHHMMSS(PWKSETTIME pW, UWP pVal)
{
	BOOL	rc = FALSE;

	//	● 数字ならば「時/分/秒」設定
	if (MAjcIsDigitW(*pW->pTxt)) {
		int		stl  = 0;
		C_WCP	pTmp = pW->pTxt;
		//	桁数チェック
		while (MAjcIsDigitW(*pTmp)) {
			pTmp++;
			stl++;
		}
		if (stl == 1 || stl == 2) {
			//	「時/分/秒」設定
			*pVal = _wtoi(pW->pTxt);
			//	数字スキップ
			pW->pTxt += stl;
			rc = TRUE;
		}
	}
	//	● "**" ならば、何もしない（構造体へ設定しない）
	else if (wcsncmp(pW->pTxt, L"**"  , 2) == 0 && *(pW->pTxt + 2) != L'*') {
		pW->pTxt += 2;
		rc = TRUE;
	}
	//	● ミリ秒区切り文字／空白／終端 ＆ AJCDATEFLAG_SETDEF指定ならばデフォルト値設定
	else if (*pW->pTxt == pW->dms || MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		if (pW->flag & AJCTIMEFLAG_SETDEF) {
			*pVal = 0;
		}
		rc = TRUE;
	}
	return rc;
}
//----- ミリ秒設定 ---------------------------------------------------------------------------------------------//
static BOOL SubSetMS(PWKSETTIME pW)
{
	BOOL	rc = FALSE;

	//	● 数字ならば「ミリ秒」設定
	if (MAjcIsDigitW(*pW->pTxt)) {
		int		stl  = 0;
		C_WCP	pTmp = pW->pTxt;
		//	桁数チェック
		while (MAjcIsDigitW(*pTmp)) {
			pTmp++;
			stl++;
		}
		if (stl == 3) {
			//	「ミリ秒」設定
			pW->ms = _wtoi(pW->pTxt);
			//	数字スキップ
			pW->pTxt += stl;
			rc = TRUE;
		}
	}
	//	● "***" ならば、何もしない（構造体へ設定しない）
	else if (wcsncmp(pW->pTxt, L"**+"  , 3) == 0 && *(pW->pTxt + 3) != L'*') {
		pW->pTxt += 3;
		rc = TRUE;
	}
	//	● 空白／終端 ＆ AJCDATEFLAG_SETDEF指定ならばデフォルト値設定
	else if (MAjcIsBlankW(*pW->pTxt) || *pW->pTxt == 0) {
		if (pW->flag & AJCTIMEFLAG_SETDEF) {
			pW->ms = 0;
		}
		rc = TRUE;
	}
	return rc;
}
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetTimeFromTextExA			(C_BCP pTimeText, LPSYSTEMTIME pSt, BCP *pNext, UI flag, BC dlm, BC dms)
{
	BOOL	rc = FALSE;
	WCP		pTxtW = NULL;
	int		lTxtW;
	WCP		pNxtW;

	if (pTimeText != NULL && pSt != NULL) {
		lTxtW = AjcByteCharToWideChar(pTimeText, NULL, 0);
		if (pTxtW = AjcTAlloc(lTxtW)) {
			AjcByteCharToWideChar(pTimeText, pTxtW, lTxtW);
			rc = AjcSetTimeFromTextExW(pTxtW, pSt, &pNxtW, flag, dlm, dms);
			if (pNext != NULL) {
				UX	wcl = ((UX)pNxtW - (UX)pTxtW) / 2;
				UI	bcl = WideCharToMultiByte(CP_ACP, 0, pTxtW, (UI)wcl, NULL, 0, NULL, NULL);
				*pNext = (BCP)(pTimeText + bcl);
			}
		}
		if (pTxtW != NULL) free(pTxtW);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetTimeFromTextExW			(C_WCP pTimeText, LPSYSTEMTIME pSt, WCP *pNext, UI flag, WC dlm, WC dms)
{
	BOOL		rc = FALSE;
	WKSETTIME	wrk;

	if (pTimeText != NULL && pSt != NULL && !MAjcIsAlNum(dlm) && !MAjcIsBlank(dlm) && dlm != L'*') {
		//	デフォルト区切り文字設定
		if (dlm == 0) dlm = L':';
		if (dms == 0) dms = L'.';
		//	ワーク設定
		wrk.pTxt = pTimeText;
		wrk.flag = flag;
		wrk.dlm  = dlm;
		wrk.dms  = dms;
		wrk.hh	 = pSt->wHour;
		wrk.mm	 = pSt->wMinute;
		wrk.ss	 = pSt->wSecond;
		wrk.ms	 = pSt->wMilliseconds;
		//	先頭空白スキップ
		while (MAjcIsBlankW(*wrk.pTxt)) {
			wrk.pTxt++;
		}
		//	時分秒，ミリ秒設定
		do {
			if (!(rc = SubSetHHMMSS(&wrk, &wrk.hh))) break;		//	「時」設定
			if (!(rc = SubSkipDlmT (&wrk		 ))) break;		//	 区切り文字(:)スキップ
			if (!(rc = SubSetHHMMSS(&wrk, &wrk.mm))) break;		//	「分」設定
			if (!(rc = SubSkipDlmT (&wrk		 ))) break;		//	 区切り文字(:)スキップ
			if (!(rc = SubSetHHMMSS(&wrk, &wrk.ss))) break;		//	「秒」設定
			if (!(rc = SubSkipDlmP (&wrk		 ))) break;		//	 区切り文字(.)スキップ
			if (!(rc = SubSetMS    (&wrk		 ))) break;		//	「ミリ秒」設定
		} while (0);
	}
	if (rc) {
		//	時分秒，ミリ秒補正
		if (wrk.flag & AJCTIMEFLAG_ADJUST) {
			if (wrk.hh <   0) wrk.hh = 1;
			if (wrk.hh >  23) wrk.hh = 12;
			if (wrk.mm <   0) wrk.mm = 0;
			if (wrk.mm >  59) wrk.mm = 59;
			if (wrk.ss <   0) wrk.ss = 0;
			if (wrk.ss >  59) wrk.ss = 59;
			if (wrk.ms <   0) wrk.ss = 0;
			if (wrk.ms > 999) wrk.ss = 999;
		}
		//	戻りSYSTEMTIME構造体設定
		pSt->wHour		   = wrk.hh;
		pSt->wMinute 	   = wrk.mm;
		pSt->wSecond	   = wrk.ss;
		pSt->wMilliseconds = wrk.ms;
	}
	//	戻りポインタ設定
	if (pNext != NULL) {
		*pNext = (WCP)wrk.pTxt;
	}

	return rc;
}

//==============================================================================================================//
//	Function	:	日時文字列からSYSTEMTIME構造体の日時情報を設定												//
//																												//
//	Argument	:	pText		- 日時テキストのアドレス（ex. “2022/12/23	13:23:51”）						//
//					pSt 		- 日時情報を設定するSYSTEMTIME構造体のアドレス									//
//					pNext		- 日時テキストの直後のアドレスを格納するポインタ変数のアドレス（不要時はNULL）	//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateAndTimeFromTextA 	(C_BCP pText, LPSYSTEMTIME pSt, BCP *pNext)
{
	BOOL	rc = FALSE;
	rc = AjcSetDateAndTimeFromTextExA(pText, pSt, pNext, AJCOPT4(AJCDATEFLAG_, YYMMDD, ADJUST, SETDEF, SETDOW) | AJCOPT2(AJCTIMEFLAG_, ADJUST, SETDEF), '/', ':', '.');
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateAndTimeFromTextW 	(C_WCP pText, LPSYSTEMTIME pSt, WCP *pNext)
{
	BOOL	rc = FALSE;
	rc = AjcSetDateAndTimeFromTextExW(pText, pSt, pNext, AJCOPT4(AJCDATEFLAG_, YYMMDD, ADJUST, SETDEF, SETDOW) | AJCOPT2(AJCTIMEFLAG_, ADJUST, SETDEF), L'/', L':', L'.');
	return rc;
}

//==============================================================================================================//
//	Function	:	日時文字列からSYSTEMTIME構造体の日時情報を設定（拡張版）									//
//																												//
//	Argument	:	pText		- 日時テキストのアドレス（ex. “2022/12/23	13:23:51”）						//
//					pSt 		- 日時情報を設定するSYSTEMTIME構造体のアドレス									//
//					pNext		- 日時テキストの直後のアドレスを格納するポインタ変数のアドレス（不要時はNULL）	//
//					flag		- フラグ																		//
//					dlm_d		- 年月日の区切り文字（０指定時は、'/' を仮定）									//
//					dlm_t		- 時分秒の区切り文字（０指定時は、':' を仮定）									//
//					dms			- ミリ秒直前の区切り記号（０指定時は、'.' を仮定）								//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - エラー																				//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateAndTimeFromTextExA	(C_BCP pText, LPSYSTEMTIME pSt, BCP *pNext, UI flag, BC dlm_d, BC dlm_t, BC dms)
{
	BOOL	rc = FALSE;
	WCP		pTxtW = NULL;
	int		lTxtW;
	WCP		pNxtW;

	if (pText != NULL && pSt != NULL) {
		lTxtW = AjcByteCharToWideChar(pText, NULL, 0);
		if (pTxtW = AjcTAlloc(lTxtW)) {
			AjcByteCharToWideChar(pText, pTxtW, lTxtW);
			rc = AjcSetDateAndTimeFromTextExW(pTxtW, pSt, &pNxtW, flag, dlm_d, dlm_t, dms);
			if (pNext != NULL) {
				UX	wcl = ((UX)pNxtW - (UX)pTxtW) / 2;
				UI	bcl = WideCharToMultiByte(CP_ACP, 0, pTxtW, (UI)wcl, NULL, 0, NULL, NULL);
				*pNext = (BCP)(pText + bcl);
			}
		}
		if (pTxtW != NULL) free(pTxtW);
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL WINAPI	AjcSetDateAndTimeFromTextExW	(C_WCP pText, LPSYSTEMTIME pSt, WCP *pNext, UI flag, WC dlm_d, WC dlm_t, WC dms)
{
	BOOL	rc = FALSE;
	WCP		pTmp;

	if (rc = AjcSetDateFromTextExW(pText, pSt, &pTmp, flag, dlm_d)) {
		rc = AjcSetTimeFromTextExW(pTmp , pSt, pNext, flag, dlm_t, dms);
	}
	else {
		if (pNext != NULL) {
			*pNext = pTmp;
		}
	}
	return rc;
}
//==============================================================================================================//
//	Function	:	ランダムな日付と時刻を取得																	//
//																												//
//	Argument	:	pSt	- ランダムな日時情報を設定するSYSTEMTIME構造体のアドレス								//
//																												//
//	Return		:	TRUE  - 成功																				//
//					FALSE - エラー																				//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcGetRandomDateAndTime (LPSYSTEMTIME pSt, LPSYSTEMTIME pStSrt, LPSYSTEMTIME pStEnd)
{
	BOOL		rc = FALSE;
	SYSTEMTIME	st, RndSrt, RndEnd;
	ULL			rndmax = (ULL)RAND_MAX * (ULL)RAND_MAX;
	ULL			rnd, trg;
	ULL			sec, secSrt, secEnd;
	ULL			ms , msSrt , msEnd;

	if (pSt != NULL) {
		//	開始，終了日時設定
		if (pStEnd != NULL) memcpy(&RndEnd, pStEnd, sizeof(SYSTEMTIME));
		else				{
			GetLocalTime(&RndEnd);
			RndEnd.wHour		 = 23;
			RndEnd.wMinute		 = 59;
			RndEnd.wSecond		 = 59;
			RndEnd.wMilliseconds = 999;
		}
		if (pStSrt != NULL) memcpy(&RndSrt, pStSrt, sizeof(SYSTEMTIME));
		else {
			sec  = AjcSysTimeToTime1970(&RndEnd);
			sec -= (365 * 24 * 60 * 60);
			AjcTime1970ToSysTime((UI)sec, &RndSrt);
			RndSrt.wHour		 = 0;
			RndSrt.wMinute		 = 0;
			RndSrt.wSecond		 = 0;
			RndSrt.wMilliseconds = 0;
		}
		//	範囲チェック
		secSrt = AjcSysTimeToTime1970(&RndSrt);
		secEnd = AjcSysTimeToTime1970(&RndEnd);
		if (secSrt <= secEnd) {
			//	開始日（秒）設定
			memset(&st, 0, sizeof st);
			st.wYear  = RndSrt.wYear;
			st.wMonth = RndSrt.wMonth;
			st.wDay   = RndSrt.wDay;
			secSrt	  = AjcSysTimeToTime1970(&st);
			//	終了日（秒）設定
			memset(&st, 0, sizeof st);
			st.wYear   = RndEnd.wYear;
			st.wMonth  = RndEnd.wMonth;
			st.wDay    = RndEnd.wDay;
			st.wHour   = 23;
			st.wMinute = 59;
			st.wSecond = 59;
			secEnd	  = AjcSysTimeToTime1970(&st);
			//	開始～終了の秒数算出
			sec = secEnd - secSrt;
			//	ランダムな日付算出
			rnd = (UX)rand() * (UX)rand();
			trg = sec * rnd / rndmax;
			trg += secSrt;
			AjcTime1970ToSysTime((UI)trg, &st);
			pSt->wYear		= st.wYear;
			pSt->wMonth 	= st.wMonth;
			pSt->wDay		= st.wDay;
			pSt->wDayOfWeek = st.wDayOfWeek;
			//	開始時刻(ms)設定
			msSrt = ((ULL)RndSrt.wHour * 60 * 60 * 1000) + ((ULL)RndSrt.wMinute * 60 * 1000) + ((ULL)RndSrt.wSecond * 1000) + (ULL)RndSrt.wMilliseconds;
			msEnd = ((ULL)RndEnd.wHour * 60 * 60 * 1000) + ((ULL)RndEnd.wMinute * 60 * 1000) + ((ULL)RndEnd.wSecond * 1000) + (ULL)RndEnd.wMilliseconds;
			//	開始～終了のミリ秒算出
			msSrt %= (24 * 60 * 60 * 1000);
			msEnd %= (24 * 60 * 60 * 1000);
			ms = msEnd - msSrt;
			//	ランダムな時刻算出
			rnd  = (ULL)rand() * (ULL)rand();
			trg  = ms * rnd / rndmax;
			trg += msSrt;
			pSt->wHour		   = (UW)(trg / (60 * 60 * 1000));	trg %= (60 * 60 * 1000);
			pSt->wMinute	   = (UW)(trg / (	  60 * 1000));	trg %= (	 60 * 1000);
			pSt->wSecond	   = (UW)(trg / (		   1000));	trg %= (		  1000);
			pSt->wMilliseconds = (UW)(trg					);
		}
	}
	return rc;
}
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//
//																												//
//	ＭＳＩＬ用サポートコード																					//
//																												//
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++//

//==============================================================================================================//
//	1970/1/1 00:00:00からの通算秒を日時に変換	（戻り値＝FALSE : エラー）										//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcMsilTime1970ToSysTime(UI Time1970, UIP pYear, UIP pMonth, UIP pDay, UIP pHour, UIP pMinute, UIP pSecond)
{
	BOOL		rc = FALSE;
	SYSTEMTIME	st;

	if (rc = AjcTime1970ToSysTime(Time1970, &st)) {
		*pYear	 = st.wYear;
		*pMonth	 = st.wMonth;
		*pDay	 = st.wDay;
		*pHour	 = st.wHour;
		*pMinute = st.wMinute;
		*pSecond = st.wSecond;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	BOOL	WINAPI	AjcMsilTime1970ToSysTimeL(ULL Time1970, UIP pYear, UIP pMonth, UIP pDay, UIP pHour, UIP pMinute, UIP pSecond)
{
	BOOL		rc = FALSE;
	SYSTEMTIME	st;

	if (rc = AjcTime1970ToSysTimeL(Time1970, &st)) {
		*pYear	 = st.wYear;
		*pMonth	 = st.wMonth;
		*pDay	 = st.wDay;
		*pHour	 = st.wHour;
		*pMinute = st.wMinute;
		*pSecond = st.wSecond;
	}
	return rc;
}
//==============================================================================================================//
//	日時を1970/1/1 00:00:00からの通算秒に変換	（戻り値＝-1 : エラー）											//
//==============================================================================================================//
AJCEXPORT	UI		WINAPI	AjcMsilSysTimeToTime1970(UI Year, UI Month, UI Day, UI Hour, UI Minute, UI Second)
{
	UI			rc = -1;
	SYSTEMTIME	st;

	memset(&st, 0, sizeof st);
	st.wYear	 = Year;
	st.wMonth	 = Month;
	st.wDay		 = Day;
	st.wHour	 = Hour;
	st.wMinute	 = Minute;
	st.wSecond	 = Second;
	rc = AjcSysTimeToTime1970(&st);

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT	ULL		WINAPI	AjcMsilSysTimeToTime1970L(UI Year, UI Month, UI Day, UI Hour, UI Minute, UI Second)
{
	ULL			rc = -1;
	SYSTEMTIME	st;

	memset(&st, 0, sizeof st);
	st.wYear	 = Year;
	st.wMonth	 = Month;
	st.wDay		 = Day;
	st.wHour	 = Hour;
	st.wMinute	 = Minute;
	st.wSecond	 = Second;
	rc = AjcSysTimeToTime1970L(&st);

	return rc;
}

//==============================================================================================================//
//	ＵＴＣ日時をローカルタイムに変換		（戻り値＝FALSE : エラー）											//
//==============================================================================================================//
AJCEXPORT	BOOL	WINAPI	AjcMsilSysTimeToLocalTime(UIP pYear, UIP pMonth, UIP pDay, UIP pHour, UIP pMinute, UIP pSecond)
{
	BOOL		rc = FALSE;
	SYSTEMTIME	st, lt;

	memset(&st, 0, sizeof st);
	memset(&lt, 0, sizeof lt);

	st.wYear	 = *pYear;
	st.wMonth	 = *pMonth;
	st.wDay		 = *pDay;
	st.wHour	 = *pHour;
	st.wMinute	 = *pMinute;
	st.wSecond	 = *pSecond;

	if (rc = AjcSysTimeToLocalTime(&st, &lt)) {
		*pYear	 = lt.wYear;
		*pMonth	 = lt.wMonth;
		*pDay	 = lt.wDay;
		*pHour	 = lt.wHour;
		*pMinute = lt.wMinute;
		*pSecond = lt.wSecond;
	}
	return rc;
}

