﻿#include	"AjcInternal.h"

//**************************************************************************************************************//
//																												//
//	汎用サブ関数																								//
//																												//
//**************************************************************************************************************//

//--------------------------------------------------------------------------------------------------------------//
//	内部サブ関数																								//
//--------------------------------------------------------------------------------------------------------------//


//==============================================================================================================//
//	ＤＩＢセクションの生成																						//
//																												//
//	引　数：	width		- ビットマップの幅　（横ピクセル数）												//
//				height		- ビットマップの高さ（縦ピクセル数）												//
//				BitCount	- イメージデータのビット数															//
//				pBuf		- 生成したＤＩＢセクションの情報を格納するバッファのアドレス（不要時はＮＵＬＬ）	//
//																												//
//	戻り値：	≠NULL：ビットマップ・ハンドル																	//
//				＝NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT HBITMAP	WINAPI AjcDibCreate		(int width, int height, int BitCount, PAJCDIBINFO pBuf)
{
	HBITMAP				rc = NULL;
	BITMAPINFOHEADER	bmih;
	UBP					pBits;
	int					lsz;

	if (width > 0  &&  height > 0  &&  (BitCount == 24 || BitCount == 32)) {
		bmih.biSize			= sizeof(BITMAPINFOHEADER);
		bmih.biWidth		= width;
		bmih.biHeight		= height;
		bmih.biPlanes		= 1;
		bmih.biBitCount		= BitCount;
		bmih.biCompression	= BI_RGB;
		bmih.biSizeImage	= 0;
		bmih.biXPelsPerMeter= 0;
		bmih.biYPelsPerMeter= 0;
		bmih.biClrUsed		= 0;
		bmih.biClrImportant	= 0;
		if (rc = CreateDIBSection(NULL, (BITMAPINFO *)&bmih, 0, &pBits, NULL, 0)) {
			lsz = ((width * BitCount + 31) & ~31) >> 3;
			if (pBuf != NULL) {
				pBuf->hBmp		= rc;
				pBuf->BitCount	= BitCount;
				pBuf->width		= width;
				pBuf->height	= height;
				pBuf->PixelSize	= BitCount / 8;
				pBuf->LineSize	= lsz;
				pBuf->pImage	= pBits;
			}
			memset(pBits, 0, lsz * height);
		}
	}
	return rc;
}

//==============================================================================================================//
//	ＤＩＢセクションを指定の色でクリアー																		//
//																												//
//	引　数：	pDibInfo	- ＤＩＢセクション情報のアドレス													//
//				rgb			- クリアーする色																	//
//																												//
//	戻り値：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcDibClear			(PAJCDIBINFO pDibInfo, COLORREF rgb)
{
	BOOL	rc = FALSE;
	int		x, y;
	UB		r, g, b;
	UBP		p;

	if (pDibInfo != NULL && pDibInfo->BitCount >= 24) {
		r = (UB)GetRValue(rgb);
		g = (UB)GetGValue(rgb);
		b = (UB)GetBValue(rgb);
		for (y = 0; y < pDibInfo->height; y++) {
			p = pDibInfo->pImage + (pDibInfo->LineSize * y);
			for (x = 0; x < pDibInfo->width; x++) {
				*(p + 0) = b;
				*(p + 1) = g;
				*(p + 2) = r;
				p += pDibInfo->PixelSize;
			}
		}
		rc = TRUE;
	}
	return rc;
}

//==============================================================================================================//
//	ＤＩＢセクションへビットマップをコピー																		//
//																												//
//	引　数：	pDibInfo					- コピー先ＤＩＢセクション情報のアドレス							//
//				dpx, dpy, dcx, dcy / ptDest	- コピー先矩形の位置とサイズ										//
//				hBitmap						- コピー元のビットマップハンドル									//
//				spx, spy, scx, scy / ptSrc	- コピー元矩形の位置とサイズ										//
//				rop							- ラスターオペレーション											//
//																												//
//	戻り値：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcDibCopyBitmap	(PAJCDIBINFO pDibInfo, int dpx, int dpy, int dcx, int dcy, HBITMAP hBitmap , int spx, int spy, int scx, int scy, int rop)
{
	BOOL		rc = FALSE;
	BITMAP		BmpInfo;
	HDC			hdc, ddc, sdc;
	HBITMAP		hBmpD, hBmpS;

	if (pDibInfo != NULL && hBitmap != NULL) {
		GetObject(hBitmap, sizeof(BITMAP), &BmpInfo);
		if (dpx >= 0 && dpx < pDibInfo->width && dpy >= 0 && dpy < pDibInfo->height &&
			spx >= 0 && spx < BmpInfo.bmWidth && spy >= 0 && spy < BmpInfo.bmHeight) {
			//----- コピー先ビットマップサイズ設定 -------------------------------------//
			if (dcx <= 0) dcx = pDibInfo->width  - dpx;
			if (dcy <= 0) dcy = pDibInfo->height - dpy;
			//----- コピー先ビットマップサイズ補正 -------------------------------------//
			if (dpx + dcx > pDibInfo->width ) dcx = pDibInfo->width  - dpx;
			if (dpy + dcy > pDibInfo->height) dcy = pDibInfo->height - dpy;
			//----- コピー元ビットマップサイズ設定 -------------------------------------//
			if (scx <= 0) scx = BmpInfo.bmWidth  - spx;
			if (scy <= 0) scy = BmpInfo.bmHeight - spy;
			//----- コピー元ビットマップサイズ補正 -------------------------------------//
			if (spx + scx > BmpInfo.bmWidth ) scx = BmpInfo.bmWidth  - spx;
			if (spy + scy > BmpInfo.bmHeight) scy = BmpInfo.bmHeight - spy;
			//----- コピー -------------------------------------------------------------//
			hdc   = GetDC(NULL);
			ddc   = CreateCompatibleDC(hdc);
			sdc   = CreateCompatibleDC(hdc);
			hBmpD = (HBITMAP)SelectObject(ddc, pDibInfo->hBmp);
			hBmpS = (HBITMAP)SelectObject(sdc, hBitmap);
			if (dcx == scx	&&	dcy == scy) {
				rc = BitBlt(ddc, dpx, dpy, dcx, dcy, sdc, spx, spy, rop);
			}
			else {
				rc = StretchBlt(ddc, dpx, dpy, dcx, dcy, sdc, spx, spy, scx, scy, rop);
			}
			SelectObject(ddc, hBmpD);
			SelectObject(sdc, hBmpS);
			DeleteDC(ddc);
			DeleteDC(sdc);
			ReleaseDC(NULL, hdc);
		}
	}

	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		WINAPI AjcDibCopyBitmapV(PAJCDIBINFO pDibInfo, const RECT *rcDest, HBITMAP hBitmap , const RECT *rcSrc, int rop)
{
	BOOL	rc = FALSE;

	if (rcDest != NULL && rcSrc != NULL) {
		rc = AjcDibCopyBitmap(pDibInfo, rcDest->left, rcDest->top, rcDest->right - rcDest->left, rcDest->bottom - rcDest->top,
							  hBitmap , rcSrc ->left, rcSrc ->top, rcSrc ->right - rcSrc ->left, rcSrc ->bottom - rcSrc ->top, rop);
	}
	return rc;
}
//==============================================================================================================//
//	ＤＩＢセクションからＤＩＢセクションへ特定の色範囲の部分だけをコピー										//
//																												//
//	引　数：	pDibDest					- コピー先ＤＩＢセクション情報のアドレス							//
//				dpx, dpy, dcx, dcy / rcDest	- コピー先矩形の位置とサイズ										//
//				pDibSrc						- コピー元ＤＩＢセクション情報のアドレス							//
//				spx, spy		   / ptSrc	- コピー元矩形の位置												//
//				rgbBase						- コピー対象とする色のベース値										//
//				distance					- コピー対象とする色の距離											//
//																												//
//	戻り値：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcDibCorrectedCopy	(PAJCDIBINFO pDibDest, int dpx, int dpy, int dcx, int dcy, PAJCDIBINFO pDibSrc , int spx, int spy, COLORREF rgbBase, double distance)
{
	BOOL	rc = FALSE;
	int		scx, scy, tcx, tcy, cx, cy;
	int		r, g, b, sr, sg, sb;
	UBP		pDL, pSL, pDB, pSB;
	double	len;

	if (pDibDest != NULL && pDibDest->BitCount >= 24 && dpx >= 0 && dpx < pDibDest->width && dpy >= 0 && dpy < pDibDest->height &&
		pDibSrc  != NULL && pDibSrc ->BitCount >= 24 && spx >= 0 && spx < pDibSrc ->width && spy >= 0 && spy < pDibSrc ->height) {
		//----- コピー先ビットマップサイズ設定 -------------------------------------//
		if (dcx <= 0) dcx = pDibDest->width  - dpx;
		if (dcy <= 0) dcy = pDibDest->height - dpy;
		//----- コピー先ビットマップサイズ補正 -------------------------------------//
		if (dpx + dcx > pDibDest->width ) dcx = pDibDest->width  - dpx;
		if (dpy + dcy > pDibDest->height) dcy = pDibDest->height - dpy;
		//----- コピー元の最大ピクセル数設定 ---------------------------------------//
		scx = pDibSrc->width  - spx;
		scy = pDibSrc->height - spy;
		//----- コピーサイズ設定 ---------------------------------------------------//
		tcx = __min(dcx, scx);
		tcy = __min(dcy, scy);
		//----- 色の選択とコピー ---------------------------------------------------//
		r = (UB)GetRValue(rgbBase);
		g = (UB)GetGValue(rgbBase);
		b = (UB)GetBValue(rgbBase);
		pDL = pDibDest->pImage + (pDibDest->LineSize * (pDibDest->height - dpy - 1));
		pSL = pDibSrc ->pImage + (pDibSrc ->LineSize * (pDibSrc ->height - spy - 1));
		for (cy = 0; cy < tcy; cy++) {
			pDB = pDL + (pDibDest->PixelSize * dpx);
			pSB = pSL + (pDibSrc ->PixelSize * spx);
			for (cx = 0; cx < tcx; cx++) {
				sb = *(pSB + 0);
				sg = *(pSB + 1);
				sr = *(pSB + 2);
				len = sqrt(pow(r - sr, 2.0) + pow(g - sg, 2.0) + pow(b - sb, 2.0));
				if (len <= distance) {
					memcpy(pDB, pSB, 3);
				}
				pDB += pDibDest->PixelSize;
				pSB += pDibSrc ->PixelSize;
			}
			pDL -= pDibDest->LineSize;
			pSL -= pDibSrc ->LineSize;
		}
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		WINAPI AjcDibCorrectedCopyV	(PAJCDIBINFO pDibDest, const RECT *rcDest, PAJCDIBINFO pDibSrc , const POINT *ptSrc, COLORREF rgbBase, double distance)
{
	BOOL	rc = FALSE;

	if (rcDest != NULL && ptSrc != NULL) {
		rc = AjcDibCorrectedCopy(pDibDest, rcDest->left, rcDest->top, rcDest->right - rcDest->left, rcDest->bottom - rcDest->top,
								 pDibSrc , ptSrc ->x   , ptSrc ->y  , rgbBase, distance);
	}
	return rc;
}
//==============================================================================================================//
//	ＤＩＢセクションからＤＩＢセクションへ特定の色範囲を除くの部分だけをコピー									//
//																												//
//	引　数：	pDibDest			- コピー先ＤＩＢセクション情報のアドレス									//
//				dpx, dpy, dcx, dcy	- コピー先矩形の位置とサイズ												//
//				pDibSrc				- コピー元ＤＩＢセクション情報のアドレス									//
//				spx, spy, scx, scy	- コピー元矩形の位置とサイズ												//
//				rgbBase				- コピー除外とする色のベース値												//
//				distance			- コピー除外とする色の距離													//
//																												//
//	戻り値：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcDibExcludedCopy 	(PAJCDIBINFO pDibDest, int dpx, int dpy, int dcx, int dcy, PAJCDIBINFO pDibSrc , int spx, int spy, COLORREF rgbBase, double distance)
{
	BOOL	rc = FALSE;
	int		scx, scy, tcx, tcy, cx, cy;
	int		r, g, b, sr, sg, sb;
	UBP		pDL, pSL, pDB, pSB;
	double	len;

	if (pDibDest != NULL && pDibDest->BitCount >= 24 && dpx >= 0 && dpx < pDibDest->width && dpy >= 0 && dpy < pDibDest->height &&
		pDibSrc  != NULL && pDibSrc ->BitCount >= 24 && spx >= 0 && spx < pDibSrc ->width && spy >= 0 && spy < pDibSrc ->height) {
		//----- コピー先ビットマップサイズ設定 -------------------------------------//
		if (dcx <= 0) dcx = pDibDest->width  - dpx;
		if (dcy <= 0) dcy = pDibDest->height - dpy;
		//----- コピー先ビットマップサイズ補正 -------------------------------------//
		if (dpx + dcx > pDibDest->width ) dcx = pDibDest->width  - dpx;
		if (dpy + dcy > pDibDest->height) dcy = pDibDest->height - dpy;
		//----- コピー元の最大ピクセル数設定 ---------------------------------------//
		scx = pDibSrc->width  - spx;
		scy = pDibSrc->height - spy;
		//----- コピーサイズ設定 ---------------------------------------------------//
		tcx = __min(dcx, scx);
		tcy = __min(dcy, scy);
		//----- 色の除外とコピー ---------------------------------------------------//
		r = (UB)GetRValue(rgbBase);
		g = (UB)GetGValue(rgbBase);
		b = (UB)GetBValue(rgbBase);
		pDL = pDibDest->pImage + (pDibDest->LineSize * (pDibDest->height - dpy - 1));
		pSL = pDibSrc ->pImage + (pDibSrc ->LineSize * (pDibSrc ->height - spy - 1));
		for (cy = 0; cy < tcy; cy++) {
			pDB = pDL + (pDibDest->PixelSize * dpx);
			pSB = pSL + (pDibSrc ->PixelSize * spx);
			for (cx = 0; cx < tcx; cx++) {
				sb = *(pSB + 0);
				sg = *(pSB + 1);
				sr = *(pSB + 2);
				len = sqrt(pow(r - sr, 2.0) + pow(g - sg, 2.0) + pow(b - sb, 2.0));
				if (len > distance) {
					memcpy(pDB, pSB, 3);
				}
				pDB += pDibDest->PixelSize;
				pSB += pDibSrc ->PixelSize;
			}
			pDL -= pDibDest->LineSize;
			pSL -= pDibSrc ->LineSize;
		}
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		WINAPI AjcDibExcludedCopyV 	(PAJCDIBINFO pDibDest, const RECT *rcDest, PAJCDIBINFO pDibSrc , const POINT *ptSrc, COLORREF rgbBase, double distance)
{
	BOOL	rc = FALSE;

	if (rcDest != NULL && ptSrc != NULL) {
		rc = AjcDibExcludedCopy(pDibDest, rcDest->left, rcDest->top, rcDest->right - rcDest->left, rcDest->bottom - rcDest->top,
								pDibSrc , ptSrc ->x   , ptSrc ->y  , rgbBase, distance);
	}
	return rc;
}

//==============================================================================================================//
//	ＤＩＢセクションからＤＩＢセクションへ複数の色範囲を選択してコピー											//
//																												//
//	引　数：	pDibDest			- コピー先ＤＩＢセクション情報のアドレス									//
//				dpx, dpy, dcx, dcy	- コピー先矩形の位置とサイズ												//
//				pDibSrc				- コピー元ＤＩＢセクション情報のアドレス									//
//				spx, spy, scx, scy	- コピー元矩形の位置とサイズ												//
//				OpeNum				- 選択／除外する色範囲の個数												//
//																												//
//	戻り値：	TRUE ：成功																						//
//				FALSE：エラー																					//
//==============================================================================================================//
typedef struct {
	int		ope;
	int		r, g, b;
	double	dist;
} DCS_NODE, *PDCS_NODE;
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		WINAPI AjcDibSelectedCopy 	(PAJCDIBINFO pDibDest, int dpx, int dpy, int dcx, int dcy, PAJCDIBINFO pDibSrc , int spx, int spy, int OpeNum, ...)
{
	BOOL	rc = FALSE;
	int		scx, scy, tcx, tcy, cx, cy;
	int		sr, sg, sb;
	UBP		pDL, pSL, pDB, pSB;
	double	len;

	HAJCFQUE	hCorrect = NULL;
	HAJCFQUE	hExclude = NULL;

	int			i, ope;
	va_list 	vls;
	COLORREF	rgb;
	DCS_NODE	node;
	PDCS_NODE	pNode;
	BOOL		fCorrect, fExclude;

	if (pDibDest != NULL && pDibDest->BitCount >= 24 && dpx >= 0 && dpx < pDibDest->width && dpy >= 0 && dpy < pDibDest->height && 
		pDibSrc  != NULL && pDibSrc ->BitCount >= 24 && spx >= 0 && spx < pDibSrc ->width && spy >= 0 && spy < pDibSrc ->height && OpeNum > 0) {
		do {
			//----- 色選択パラメタの収集 -----------------------------------------------//
			if ((hCorrect = AjcFQueCreate(sizeof(DCS_NODE), 0, NULL)) == NULL) break;
			if ((hExclude = AjcFQueCreate(sizeof(DCS_NODE), 0, NULL)) == NULL) break;
			va_start(vls, OpeNum);
			for (i = 0; i < OpeNum; i++) {
				ope  = (int 	)va_arg(vls, int);
				rgb  = (COLORREF)va_arg(vls, COLORREF);
				node.ope  = ope;
				node.r	  = (UB)GetRValue(rgb);
				node.g	  = (UB)GetGValue(rgb);
				node.b	  = (UB)GetBValue(rgb);
				node.dist = (double)va_arg(vls, double);
				switch (ope & AJCDIBSEL_MASK) {
					case AJCDIBSEL_CORRECT:
						if (!AjcFQueEnque(hCorrect, &node)) goto dsc_exit;
						break;

					case AJCDIBSEL_EXCLUDE:
						if (!AjcFQueEnque(hExclude, &node)) goto dsc_exit;
						break;

					default:
						goto dsc_exit;
				}
			}
			va_end	(vls);
			//----- コピー先ビットマップサイズ設定 -------------------------------------//
			if (dcx <= 0) dcx = pDibDest->width  - dpx;
			if (dcy <= 0) dcy = pDibDest->height - dpy;
			//----- コピー先ビットマップサイズ補正 -------------------------------------//
			if (dpx + dcx > pDibDest->width ) dcx = pDibDest->width  - dpx;
			if (dpy + dcy > pDibDest->height) dcy = pDibDest->height - dpy;
			//----- コピー元の最大ピクセル数設定 ---------------------------------------//
			scx = pDibSrc->width  - spx;
			scy = pDibSrc->height - spy;
			//----- コピーサイズ設定 ---------------------------------------------------//
			tcx = __min(dcx, scx);
			tcy = __min(dcy, scy);
			//----- 色の選択とコピー ---------------------------------------------------//
			pDL = pDibDest->pImage + (pDibDest->LineSize * (pDibDest->height - dpy - 1));
			pSL = pDibSrc ->pImage + (pDibSrc ->LineSize * (pDibSrc ->height - spy - 1));
			for (cy = 0; cy < tcy; cy++) {
				pDB = pDL + (pDibDest->PixelSize * dpx);
				pSB = pSL + (pDibSrc ->PixelSize * spx);
				for (cx = 0; cx < tcx; cx++) {
					sb = *(pSB + 0);
					sg = *(pSB + 1);
					sr = *(pSB + 2);
					// 色の選択チェック
					fCorrect = FALSE;
					if (pNode = AjcFQueTopNode(hCorrect)) {
						do {
							len = sqrt(pow(pNode->r - sr, 2.0) + pow(pNode->g - sg, 2.0) + pow(pNode->b - sb, 2.0));
							if (len <= pNode->dist) {
								ope = (pNode->ope & AJCDIBOPE_MASK);
								fCorrect = TRUE;
								break;
							}
						} while (pNode = AjcFQueNextNode(hCorrect, pNode));
					}
					//　色の除外チェック
					fExclude = FALSE;
					if (pNode = AjcFQueTopNode(hExclude)) {
						do {
							len = sqrt(pow(pNode->r - sr, 2.0) + pow(pNode->g - sg, 2.0) + pow(pNode->b - sb, 2.0));
							if (len <= pNode->dist) {
								ope = (pNode->ope & AJCDIBOPE_MASK);
								fExclude = TRUE;
								break;
							}
						} while (pNode = AjcFQueNextNode(hExclude, pNode));
					}
					//　ピクセル・コピー
					if (fCorrect  &&  !fExclude) {
						switch (ope) {
							case AJCDIBOPE_OR:
								*(pDB + 0) |= *(pSB + 0);
								*(pDB + 1) |= *(pSB + 1);
								*(pDB + 2) |= *(pSB + 2);
								break;
							case AJCDIBOPE_AND:
								*(pDB + 0) &= *(pSB + 0);
								*(pDB + 1) &= *(pSB + 1);
								*(pDB + 2) &= *(pSB + 2);
								break;
							case AJCDIBOPE_XOR:
								*(pDB + 0) ^= *(pSB + 0);
								*(pDB + 1) ^= *(pSB + 1);
								*(pDB + 2) ^= *(pSB + 2);
								break;
							case AJCDIBOPE_COPY:
							default:
							memcpy(pDB, pSB, 3);
						}
					}
					//	ピクセルポインタ更新
					pDB += pDibDest->PixelSize;
					pSB += pDibSrc ->PixelSize;
				}
				pDL -= pDibDest->LineSize;
				pSL -= pDibSrc ->LineSize;
			}
			rc = TRUE;
		} while(0);

dsc_exit:;

		//----- 色選択パラメタキューの解放 ---------------------------------------------//
		if (hCorrect != NULL) AjcFQueDelete(hCorrect);
		if (hExclude != NULL) AjcFQueDelete(hExclude);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
#define AjcDibSelectedCopyV(pDibDest, rcDest, pDibSrc , ptSrc, OpeNum, ...)												\
	AjcDibSelectedCopy(pDibDest, rcDest->left, rcDest->top, rcDest->right - rcDest->left, rcDest->bottom - rcDest->top,	\
					   pDibSrc , ptSrc ->x	 , ptSrc ->y  , OpeNum, __VA_ARGS__)

//==============================================================================================================//
//	ＤＩＢセクション内で特定の色範囲のピクセル数をカウントする													//
//																												//
//	引　数：	pDibInfo			- ＤＩＢセクション情報のアドレス											//
//				x, y				- ピクセル数をカウントする矩形の左上位置									//
//				cx, cy				- ピクセル数をカウントする矩形のサイズ										//
//				rgbBase				- ピクセル数をカウントする色のベース値										//
//				distance			- ピクセル数をカウントする色の距離											//
//																												//
//	戻り値：	>=	0 ：成功																					//
//				== -1 ：エラー																					//
//==============================================================================================================//
AJCEXPORT int		WINAPI AjcDibColorCount	(PAJCDIBINFO pDibInfo, int x, int y, int cx, int cy, COLORREF rgbBase, double distance)
{
	int		rc = -1;
	int		r, g, b, sr, sg, sb;
	UBP		pL, pB;
	int		i, j;
	double	len;

	if (pDibInfo != NULL && pDibInfo->BitCount >= 24 && x >= 0 && x < pDibInfo->width && y >= 0 && y < pDibInfo->height) {
		rc = 0;
		//----- 矩形サイズ設定 ---------------------------------------------------//
		if (cx <= 0) cx = pDibInfo->width  - x;
		if (cy <= 0) cy = pDibInfo->height - y;
		//----- 矩形サイズ補正 ---------------------------------------------------//
		if (x + cx > pDibInfo->width ) cx = pDibInfo->width  - x;
		if (y + cy > pDibInfo->height) cy = pDibInfo->height - y;
		//----- 色の選択とコピー -------------------------------------------------//
		r = (UB)GetRValue(rgbBase);
		g = (UB)GetGValue(rgbBase);
		b = (UB)GetBValue(rgbBase);
		pL = pDibInfo->pImage + (pDibInfo->LineSize * (pDibInfo->height - y - 1));
		for (i = 0; i < cy; i++) {
			pB = pL + (pDibInfo->PixelSize * x);
			for (j = 0; j < cx; j++) {
				sb = *(pB + 0);
				sg = *(pB + 1);
				sr = *(pB + 2);
				len = sqrt(pow(r - sr, 2.0) + pow(g - sg, 2.0) + pow(b - sb, 2.0));
				if (len <= distance) {
					rc++;
				}
				pB += pDibInfo->PixelSize;
			}
			pL -= pDibInfo->LineSize;
		}
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT int		WINAPI AjcDibColorCountV(PAJCDIBINFO pDibInfo, const RECT *rcArea, COLORREF rgbBase, double distance)
{
	int		rc = -1;

	if (rcArea != NULL) {
		rc = AjcDibColorCount(pDibInfo, rcArea->left, rcArea->top, rcArea->right - rcArea->left, rcArea->bottom - rcArea->top, rgbBase, distance);
	}
	return rc;
}
//==============================================================================================================//
//	ＤＩＢセクション内の矩形領域を塗りつぶす（左上座標と長さ指定）												//
//																												//
//	引　数：	pDibInfo			- ＤＩＢセクション情報のアドレス											//
//				x, y				- ピクセル数をカウントする矩形の左上位置									//
//				cx, cy				- ピクセル数をカウントする矩形のサイズ										//
//				rgb					- ピクセル数をカウントする色のベース値										//
//																												//
//	戻り値：	>=	0 ：成功																					//
//				== -1 ：エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI	AjcDibFillRect		(PAJCDIBINFO pDibInfo, int x, int y, int cx, int cy, COLORREF rgb)
{
	int			rc = FALSE;
	UBP			pL, pB;

	if (pDibInfo != NULL && pDibInfo->BitCount >= 24 && x >= 0 && x < pDibInfo->width && y >= 0 && y < pDibInfo->height) {
		//----- 矩形サイズ設定 ---------------------------------------------------//
		if (cx <= 0) cx = pDibInfo->width  - x;
		if (cy <= 0) cy = pDibInfo->height - y;
		//----- 矩形サイズ補正 ---------------------------------------------------//
		if (x + cx > pDibInfo->width ) cx = pDibInfo->width  - x;
		if (y + cy > pDibInfo->height) cy = pDibInfo->height - y;
		//----- 塗りつぶし -------------------------------------------------------//
		pL = pDibInfo->pImage + (pDibInfo->LineSize * (pDibInfo->height - y - 1));
		if (pDibInfo->PixelSize == 4) {
			COLORREF	rev = ((rgb << 16) & 0x00FF0000) | (rgb & 0x0000FF00) | ((rgb >> 16) & 0x000000FF);
			int			i;
			for (i = 0; i < cy; i++) {
				pB = pL + (pDibInfo->PixelSize * x);
				AjcMemSet32(pB, rev, cx);
				pL -= pDibInfo->LineSize;
			}
		}
		else {
			int		r = (UB)GetRValue(rgb);
			int		g = (UB)GetGValue(rgb);
			int		b = (UB)GetBValue(rgb);
			int		i, j;
			for (i = 0; i < cy; i++) {
				pB = pL + (pDibInfo->PixelSize * x);
				for (j = 0; j < cx; j++) {
					*(pB + 0) = b;
					*(pB + 1) = g;
					*(pB + 2) = r;
					pB += pDibInfo->PixelSize;
				}
				pL -= pDibInfo->LineSize;
			}
		}
		rc = TRUE;
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		WINAPI	AjcDibFillRectV		(PAJCDIBINFO pDibInfo, const RECT *pRect, COLORREF rgb)
{
	return AjcDibFillRect(pDibInfo, pRect->left, pRect->top, pRect->right - pRect->left, pRect->bottom - pRect->top, rgb);
}


//==============================================================================================================//
//	ピクセルの取得																								//
//																												//
//	引　数：	pDibInfo	- ＤＩＢセクション情報のアドレス													//
//				x			- イメージのピクセル位置（０～）													//
//				y			- イメージのライン位置　（０～）													//
//																												//
//	戻り値：	≠-1：ピクセルの色																				//
//				＝-1：エラー																					//
//==============================================================================================================//
AJCEXPORT COLORREF	WINAPI AjcDibGetPixel	(PAJCDIBINFO pDibInfo, int x, int y)
{
	COLORREF			rc = -1;
	UBP					pBit;

	if (pDibInfo != NULL && x >= 0 && x < pDibInfo->width && y >= 0 && y < pDibInfo->height) {
		pBit = pDibInfo->pImage + ((pDibInfo->LineSize * (pDibInfo->height - y - 1)) + (pDibInfo->PixelSize * x));
		rc = RGB(*(pBit + 2), *(pBit + 1), *(pBit + 0));
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT COLORREF	WINAPI AjcDibGetPixelV	(PAJCDIBINFO pDibInfo, const POINT *pt)
{
	COLORREF			rc = -1;

	if (pt != NULL) {
		rc = AjcDibGetPixel	(pDibInfo, pt->x, pt->y);
	}
	return rc;
}

//==============================================================================================================//
//	ピクセルの設定																								//
//																												//
//	引　数：	pDibInfo	- ＤＩＢセクション情報のアドレス													//
//				x			- イメージのピクセル位置（０～）													//
//				y			- イメージのライン位置　（０～）													//
//				rgb			- 設定する色																		//
//																												//
//	戻り値：	TRUE  ：成功																					//
//				FALSE ：エラー																					//
//==============================================================================================================//
AJCEXPORT BOOL		WINAPI AjcDibSetPixel	(PAJCDIBINFO pDibInfo, int x, int y, COLORREF rgb)
{
	COLORREF			rc = -1;
	UBP					pBit;

	if (pDibInfo != NULL && x >= 0 && x < pDibInfo->width && y >= 0 && y < pDibInfo->height) {
		pBit = pDibInfo->pImage + ((pDibInfo->LineSize * (pDibInfo->height - y - 1)) + (pDibInfo->PixelSize * x));
		*(pBit + 0) = GetBValue(rgb);
		*(pBit + 1) = GetGValue(rgb);
		*(pBit + 2) = GetRValue(rgb);
	}
	return rc;
}
//--------------------------------------------------------------------------------------------------------------//
AJCEXPORT BOOL		WINAPI AjcDibSetPixelV	(PAJCDIBINFO pDibInfo, const POINT *pt, COLORREF rgb)
{
	COLORREF			rc = -1;

	if (pt != NULL) {
		rc = AjcDibSetPixel	(pDibInfo, pt->x, pt->y, rgb);
	}
	return rc;
}
//==============================================================================================================//
//	ラインの先頭アドレス取得																					//
//																												//
//	引　数：	pDibInfo	- ＤＩＢセクション情報のアドレス													//
//				y			- イメージのライン位置　（０～）													//
//																												//
//	戻り値	：	≠NULL：ラインの先頭アドレス																	//
//				＝NULL：エラー																					//
//==============================================================================================================//
AJCEXPORT UBP		WINAPI AjcDibGetLinrPtr	(PAJCDIBINFO pDibInfo, int y)
{
	UBP			rc = NULL;

	if (pDibInfo != NULL && y >= 0 && y < pDibInfo->height) {
		rc = pDibInfo->pImage + (pDibInfo->LineSize * (pDibInfo->height - y - 1));
	}
	return rc;
}
//==============================================================================================================//
//	ＤＩＢセクションの生成とビットマップファイルを読み出し														//
//																												//
//	引　数	：	pFilePath - ビットマップファイルのパス名														//
//				pBuf	  - 生成したＤＩＢセクションの情報を格納するｉバッファのアドレス（不要時はＮＵＬＬ）	//
//																												//
//	戻り値	：	≠NULL：ビットマップハンドル																	//
//				＝NULL：エラー																					//
//==============================================================================================================//
//----- バイト文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT HBITMAP WINAPI	AjcDibReadFileAndCreateA(C_BCP pFilePath, PAJCDIBINFO pBuf)
{
	HBITMAP	rc = NULL;
	int		len;
	WCP		pTmp;

	if (pFilePath != NULL) {
		len = MultiByteToWideChar(CP_ACP, 0, pFilePath, -1, NULL, 0);
		if (len != 0 && (pTmp = AjcTAlloc(len))) {
			MultiByteToWideChar(CP_ACP, 0, pFilePath, -1, pTmp, len);
			rc = AjcDibReadFileAndCreateW(pTmp, pBuf);
			free(pTmp);
		}
	}
	return rc;
}
//----- ワイド文字 ---------------------------------------------------------------------------------------------//
AJCEXPORT HBITMAP WINAPI	AjcDibReadFileAndCreateW(C_WCP pFilePath, PAJCDIBINFO pBuf)
{
	HBITMAP				rc = NULL;
	UBP					pBits;
	DWORD				dwBytesRead;
	HANDLE				hFile = INVALID_HANDLE_VALUE;
	HBITMAP				hBitmap;
	BITMAPFILEHEADER	BmapFh;			//	ビットマップファイルヘッダ
	LPBITMAPINFO		pBmapIf = NULL;	//	ビットマップ情報
	UI					lBmapIf = 0;	//	ビットマップ情報のサイズ
	UI					lImage	= 0;	//	イメージデータバイト数

	do {
		if (pFilePath == NULL) break;
		//----- ビットマップファイル オープン ------------------------------------------------------------------//
		if ((hFile = CreateFile (pFilePath, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL)) == 
																						INVALID_HANDLE_VALUE) {
			break;
		}
		//----- ビットマップファイルヘッダ読み出し -------------------------------------------------------------//
		if (!ReadFile (hFile, &BmapFh, sizeof (BITMAPFILEHEADER), &dwBytesRead, NULL)) {
			break;
		}
		//----- ビットマップファイルヘッダの内容チェック -------------------------------------------------------//
		if (dwBytesRead != sizeof(BITMAPFILEHEADER) || BmapFh.bfType != *(WORD *)"BM") {
			break;
		}
		//----- ビットマップ情報読み出し -----------------------------------------------------------------------//
		lBmapIf = BmapFh.bfOffBits - sizeof (BITMAPFILEHEADER) ;
		pBmapIf = (LPBITMAPINFO)AJCMEM(lBmapIf);
		if (pBmapIf == NULL) break;
		if (!ReadFile (hFile, pBmapIf, lBmapIf, &dwBytesRead, NULL)) {
			break;
		}
		//----- ビットマップ情報読み出しサイズチェック ---------------------------------------------------------//
		if (dwBytesRead != lBmapIf) {
			break;
		}
		//----- ＤＩＢＳＥＣＴＩＯＮ作成 -----------------------------------------------------------------------//
		if ((hBitmap = CreateDIBSection (NULL, pBmapIf, DIB_RGB_COLORS, (void **)&pBits, NULL, 0)) == NULL) {
			break;
		}
		//----- ビットマップイメージ読み出し -------------------------------------------------------------------//
		lImage = BmapFh.bfSize - BmapFh.bfOffBits;
		if (!ReadFile(hFile, pBits, lImage, &dwBytesRead, NULL)) {
			DeleteObject(hBitmap);
			break;
		}
		//---- イメージ読み出しサイズチェック ------------------------------------------------------------------//
		if (dwBytesRead != lImage) {
			DeleteObject(hBitmap);
			break;
		}
		//----- ＤＩＢセクション情報設定 -----------------------------------------------------------------------//
		if (pBuf != NULL) {
			pBuf->hBmp = hBitmap;
			//-- ビットマップサイズ情報 -------------//
			if (pBmapIf->bmiHeader.biSize == sizeof (BITMAPCOREHEADER)) {
				pBuf->BitCount = ((BITMAPCOREHEADER *) pBmapIf)->bcBitCount;
				pBuf->width    = ((BITMAPCOREHEADER *) pBmapIf)->bcWidth;
				pBuf->height   = ((BITMAPCOREHEADER *) pBmapIf)->bcHeight;
			}
			else {
				pBuf->BitCount = pBmapIf->bmiHeader.biBitCount;
				pBuf->width    = pBmapIf->bmiHeader.biWidth;
				pBuf->height   = pBmapIf->bmiHeader.biHeight;
			}
			pBuf->PixelSize = pBuf->BitCount / 8;
			pBuf->LineSize	= ((pBuf->width * pBuf->BitCount + 31) & ~31) >> 3;
			pBuf->pImage	= pBits;
		}
		//----- 戻り値（ビットマップハンドル）設定 -------------------------------------------------------------//
		rc = hBitmap;
	} while(0);

	//----- リソース開放 ---------------------------------------------------------------------------------------//
	if (hFile != INVALID_HANDLE_VALUE) CloseHandle(hFile);
	if (pBmapIf != NULL) free(pBmapIf); pBmapIf = NULL;

	return rc;
}
